<?php
/**
 * AIOX AJAX Handlers Class
 * 
 * Handles AJAX requests for the enhanced AIOX system
 */

if (!defined('ABSPATH')) {
    exit;
}

class AIOX_Ajax_Handlers {
    
    /**
     * Constructor
     */
    public function __construct() {
        // Meta box AJAX handlers
        // Auto-classify removed - metadata is automatically extracted during comprehensive AI processing
        add_action('wp_ajax_aiox_preview_ingredients', array($this, 'ajax_preview_ingredients'));
        
        // Settings AJAX handlers
        add_action('wp_ajax_aiox_test_ai_connection', array($this, 'ajax_test_ai_connection'));
        add_action('wp_ajax_aiox_validate_license', array($this, 'ajax_validate_license'));
        add_action('wp_ajax_aiox_reset_queue', array($this, 'ajax_reset_queue'));
        add_action('wp_ajax_aiox_clear_logs', array($this, 'ajax_clear_logs'));
        // Note: aiox_autofix_files is handled by class-aiox-settings.php
        
        // Dashboard AJAX handlers
        add_action('wp_ajax_aiox_process_queue_now', array($this, 'ajax_process_queue_now'));
        
        // Licensing AJAX handlers
        add_action('wp_ajax_aiox_get_licensing_data', array($this, 'ajax_get_licensing_data'));
        add_action('wp_ajax_aiox_save_licensing_data', array($this, 'ajax_save_licensing_data'));
        // AI Info Page creation removed in Version 2.0.1+ (manual processing only)
        add_action('wp_ajax_aiox_get_file_status', array($this, 'ajax_get_file_status'));
        // Note: aiox_generate_manifest is handled by class-aiox-hooks.php
    }
    
    /**
     * Auto-classify content AJAX handler
     * REMOVED: Metadata is now automatically extracted during comprehensive AI processing
     * See class-aiox-comprehensive-processor.php lines 84-113
     */
    
    /**
     * Preview ingredients AJAX handler
     */
    public function ajax_preview_ingredients() {
        // Verify nonce
        if (!wp_verify_nonce($_POST['nonce'], 'aiox_meta_nonce')) {
            wp_die('Security check failed');
        }
        
        // Check permissions
        if (!current_user_can('edit_posts')) {
            wp_die('Insufficient permissions');
        }
        
        $content = sanitize_textarea_field($_POST['content'] ?? '');
        $title = sanitize_text_field($_POST['title'] ?? '');
        $post_id = intval($_POST['post_id'] ?? 0);
        
        if (empty($content) && empty($title)) {
            wp_send_json_error('No content provided for preview');
        }
        
        try {
            // Create a temporary post object
            $temp_post = (object) array(
                'ID' => $post_id ?: 999999, // Use high number for temp post
                'post_title' => $title,
                'post_content' => $content,
                'post_type' => 'post',
                'post_status' => 'draft',
                'post_author' => get_current_user_id(),
                'post_date' => current_time('mysql'),
                'post_modified' => current_time('mysql')
            );
            
            // Extract ingredients
            $ingredients = AIOX_Ingredients::extract_ingredients($temp_post);
            
            AIOX_Logger::info('Ingredients preview generated via AJAX', array(
                'post_id' => $post_id,
                'ingredients_count' => count($ingredients)
            ));
            
            wp_send_json_success($ingredients);
            
        } catch (Exception $e) {
            AIOX_Logger::error('Ingredients preview failed', array(
                'error' => $e->getMessage(),
                'post_id' => $post_id
            ));
            
            wp_send_json_error('Preview generation failed: ' . $e->getMessage());
        }
    }
    
    /**
     * Test AI connection AJAX handler
     */
    public function ajax_test_ai_connection() {
        // Verify nonce
        if (!wp_verify_nonce($_POST['nonce'], 'aiox_settings_nonce')) {
            wp_die('Security check failed');
        }
        
        // Check permissions
        if (!current_user_can('manage_options')) {
            wp_die('Insufficient permissions');
        }
        
        $endpoint = esc_url_raw($_POST['endpoint'] ?? '');
        $license_key = sanitize_text_field($_POST['license_key'] ?? '');
        
        if (empty($endpoint)) {
            wp_send_json_error('No endpoint provided');
        }
        
        try {
            // Test connection
            $response = wp_remote_post($endpoint, array(
                'timeout' => 10,
                'headers' => array(
                    'Authorization' => 'Bearer ' . $license_key,
                    'Content-Type' => 'application/json',
                    'User-Agent' => 'AIOX-Publisher-Suite-Pro/' . AIOX_VERSION
                ),
                'body' => json_encode(array(
                    'action' => 'test_connection',
                    'timestamp' => time()
                ))
            ));
            
            if (is_wp_error($response)) {
                wp_send_json_error('Connection failed: ' . $response->get_error_message());
            }
            
            $response_code = wp_remote_retrieve_response_code($response);
            $response_body = wp_remote_retrieve_body($response);
            
            if ($response_code === 200) {
                wp_send_json_success(array(
                    'message' => 'Connection successful',
                    'response' => $response_body
                ));
            } else {
                wp_send_json_error('Connection failed with status: ' . $response_code);
            }
            
        } catch (Exception $e) {
            wp_send_json_error('Connection test failed: ' . $e->getMessage());
        }
    }
    
    /**
     * Validate license AJAX handler
     */
    public function ajax_validate_license() {
        // Verify nonce
        if (!wp_verify_nonce($_POST['nonce'], 'aiox_settings_nonce')) {
            wp_die('Security check failed');
        }
        
        // Check permissions
        if (!current_user_can('manage_options')) {
            wp_die('Insufficient permissions');
        }
        
        $license_key = sanitize_text_field($_POST['license_key'] ?? '');
        
        if (empty($license_key)) {
            wp_send_json_error('No license key provided');
        }
        
        try {
            // Validate license (this would typically call an external service)
            $validation_endpoint = 'https://api.aiox.com/v1/validate-license';
            
            $response = wp_remote_post($validation_endpoint, array(
                'timeout' => 10,
                'headers' => array(
                    'Content-Type' => 'application/json',
                    'User-Agent' => 'AIOX-Publisher-Suite-Pro/' . AIOX_VERSION
                ),
                'body' => json_encode(array(
                    'license_key' => $license_key,
                    'domain' => home_url(),
                    'version' => AIOX_VERSION
                ))
            ));
            
            if (is_wp_error($response)) {
                // For alpha version, simulate successful validation
                wp_send_json_success(array(
                    'valid' => true,
                    'message' => 'License validation simulated (alpha version)',
                    'features' => array('ai_processing', 'advanced_analytics', 'priority_support')
                ));
            }
            
            $response_code = wp_remote_retrieve_response_code($response);
            $response_body = wp_remote_retrieve_body($response);
            
            if ($response_code === 200) {
                $validation_data = json_decode($response_body, true);
                wp_send_json_success($validation_data);
            } else {
                // For alpha version, simulate successful validation
                wp_send_json_success(array(
                    'valid' => true,
                    'message' => 'License validation simulated (alpha version)',
                    'features' => array('ai_processing', 'advanced_analytics', 'priority_support')
                ));
            }
            
        } catch (Exception $e) {
            // For alpha version, simulate successful validation
            wp_send_json_success(array(
                'valid' => true,
                'message' => 'License validation simulated (alpha version)',
                'features' => array('ai_processing', 'advanced_analytics', 'priority_support')
            ));
        }
    }
    
    /**
     * Reset queue AJAX handler
     */
    public function ajax_reset_queue() {
        // Verify nonce
        if (!wp_verify_nonce($_POST['nonce'], 'aiox_settings_nonce')) {
            wp_die('Security check failed');
        }
        
        // Check permissions
        if (!current_user_can('manage_options')) {
            wp_die('Insufficient permissions');
        }
        
        try {
            global $wpdb;
            
            $table_name = $wpdb->prefix . 'aiox_queue';
            $result = $wpdb->query("TRUNCATE TABLE $table_name");
            
            if ($result !== false) {
                AIOX_Logger::info('Queue reset via AJAX');
                wp_send_json_success('Queue reset successfully');
            } else {
                wp_send_json_error('Failed to reset queue');
            }
            
        } catch (Exception $e) {
            wp_send_json_error('Queue reset failed: ' . $e->getMessage());
        }
    }
    
    /**
     * Clear logs AJAX handler
     */
    public function ajax_clear_logs() {
        // Verify nonce
        if (!wp_verify_nonce($_POST['nonce'], 'aiox_settings_nonce')) {
            wp_die('Security check failed');
        }
        
        // Check permissions
        if (!current_user_can('manage_options')) {
            wp_die('Insufficient permissions');
        }
        
        try {
            global $wpdb;
            
            $table_name = $wpdb->prefix . 'aiox_logs';
            $result = $wpdb->query("TRUNCATE TABLE $table_name");
            
            if ($result !== false) {
                AIOX_Logger::info('Logs cleared via AJAX');
                wp_send_json_success('Logs cleared successfully');
            } else {
                wp_send_json_error('Failed to clear logs');
            }
            
        } catch (Exception $e) {
            wp_send_json_error('Log clearing failed: ' . $e->getMessage());
        }
    }
    
    /**
     * Auto-fix files AJAX handler
     */
    public function ajax_autofix_files() {
        // Verify nonce
        if (!wp_verify_nonce($_POST['nonce'], 'aiox_settings_nonce')) {
            wp_die('Security check failed');
        }
        
        // Check permissions
        if (!current_user_can('manage_options')) {
            wp_die('Insufficient permissions');
        }
        
        try {
            // Regenerate all AIOX files
            if (class_exists('AIOX_Hooks')) {
                $hooks = new AIOX_Hooks();
                $result = $hooks->generate_all_files();
                
                if ($result) {
                    AIOX_Logger::info('Files auto-fixed via AJAX');
                    wp_send_json_success('Files regenerated successfully');
                } else {
                    wp_send_json_error('Failed to regenerate files');
                }
            } else {
                wp_send_json_error('AIOX_Hooks class not available');
            }
            
        } catch (Exception $e) {
            wp_send_json_error('File auto-fix failed: ' . $e->getMessage());
        }
    }
    
    /**
     * Process queue now AJAX handler
     */
    public function ajax_process_queue_now() {
        // Verify nonce
        if (!wp_verify_nonce($_POST['nonce'], 'aiox_ajax_nonce')) {
            wp_die('Security check failed');
        }
        
        // Check permissions
        if (!current_user_can('manage_options')) {
            wp_die('Insufficient permissions');
        }
        
        try {
            if (class_exists('AIOX_Hooks')) {
                $hooks = new AIOX_Hooks();
                $result = $hooks->process_queue();
                
                AIOX_Logger::info('Queue processed manually via AJAX');
                wp_send_json_success('Queue processed successfully');
            } else {
                wp_send_json_error('AIOX_Hooks class not available');
            }
            
        } catch (Exception $e) {
            wp_send_json_error('Queue processing failed: ' . $e->getMessage());
        }
    }
    
    // AI Info Page AJAX handler removed in Version 2.0.1+ (manual processing only)
    
    /**
     * Get file status AJAX handler
     */
    public function ajax_get_file_status() {
        // Verify nonce
        if (!wp_verify_nonce($_POST['nonce'], 'aiox_ajax_nonce')) {
            wp_die('Security check failed');
        }
        
        // Check permissions
        if (!current_user_can('manage_options')) {
            wp_die('Insufficient permissions');
        }
        
        try {
            $files_status = array();
            
            $aiox_files = array(
                'aio.json' => array('name' => 'Main Manifest', 'description' => 'Primary AIOX manifest file'),
                'aiox.aiox' => array('name' => 'Binary Snapshot', 'description' => 'Compressed binary format')
            );
            
            foreach ($aiox_files as $filename => $info) {
                $file_path = ABSPATH . '.well-known/' . $filename;
                $files_status[$filename] = array(
                    'exists' => file_exists($file_path),
                    'size' => file_exists($file_path) ? filesize($file_path) : 0,
                    'modified' => file_exists($file_path) ? filemtime($file_path) : 0,
                    'url' => home_url('/.well-known/' . $filename)
                );
            }
            
            wp_send_json_success($files_status);
            
        } catch (Exception $e) {
            wp_send_json_error('File status check failed: ' . $e->getMessage());
        }
    }
    
    /**
     * Generate manifest AJAX handler
     */
    public function ajax_generate_manifest() {
        // Verify nonce
        if (!wp_verify_nonce($_POST['nonce'], 'aiox_ajax_nonce')) {
            wp_die('Security check failed');
        }
        
        // Check permissions
        if (!current_user_can('manage_options')) {
            wp_die('Insufficient permissions');
        }
        
        try {
            if (class_exists('AIOX_Hooks')) {
                $hooks = new AIOX_Hooks();
                $result = $hooks->generate_all_files();
                
                if ($result) {
                    AIOX_Logger::info('Manifest generated via AJAX');
                    wp_send_json_success('Manifest generated successfully');
                } else {
                    wp_send_json_error('Failed to generate manifest');
                }
            } else {
                wp_send_json_error('AIOX_Hooks class not available');
            }
            
        } catch (Exception $e) {
            wp_send_json_error('Manifest generation failed: ' . $e->getMessage());
        }
    }
    
    /**
     * Get licensing data AJAX handler
     */
    public function ajax_get_licensing_data() {
        // Verify nonce
        if (!wp_verify_nonce($_POST['nonce'], 'aiox_ajax_nonce')) {
            wp_die('Security check failed');
        }
        
        // Check permissions
        if (!current_user_can('edit_posts')) {
            wp_die('Insufficient permissions');
        }
        
        $post_id = intval($_POST['post_id'] ?? 0);
        
        if (!$post_id) {
            wp_send_json_error('Invalid post ID');
            return;
        }
        
        // Get licensing metadata
        $licensing_data = array(
            'training_allowed' => get_post_meta($post_id, '_aiox_training_allowed', true),
            'citation_required' => get_post_meta($post_id, '_aiox_citation_required', true),
            'attribution_string' => get_post_meta($post_id, '_aiox_attribution_string', true),
            'license_url' => get_post_meta($post_id, '_aiox_license_url', true),
            'expiration_date' => get_post_meta($post_id, '_aiox_expiration_date', true)
        );
        
        // Apply defaults if no data exists
        if (empty($licensing_data['attribution_string'])) {
            $licensing_data['attribution_string'] = get_option('aiox_attribution_string', get_bloginfo('name'));
        }
        
        wp_send_json_success($licensing_data);
    }
    
    /**
     * Save licensing data AJAX handler
     */
    public function ajax_save_licensing_data() {
        // Verify nonce
        if (!wp_verify_nonce($_POST['nonce'], 'aiox_ajax_nonce')) {
            wp_die('Security check failed');
        }
        
        // Check permissions
        if (!current_user_can('edit_posts')) {
            wp_die('Insufficient permissions');
        }
        
        // Parse form data
        parse_str($_POST['form_data'], $form_data);
        
        $post_id = intval($form_data['post_id'] ?? 0);
        
        if (!$post_id) {
            wp_send_json_error('Invalid post ID');
            return;
        }
        
        // Save licensing data
        update_post_meta($post_id, '_aiox_training_allowed', isset($form_data['training_allowed']) ? 1 : 0);
        update_post_meta($post_id, '_aiox_citation_required', isset($form_data['citation_required']) ? 1 : 0);
        update_post_meta($post_id, '_aiox_attribution_string', sanitize_text_field($form_data['attribution_string'] ?? ''));
        update_post_meta($post_id, '_aiox_license_url', esc_url_raw($form_data['license_url'] ?? ''));
        update_post_meta($post_id, '_aiox_expiration_date', sanitize_text_field($form_data['expiration_date'] ?? ''));
        
         wp_send_json_success('Licensing data saved successfully');
    }
}
