<?php
/**
 * AIOX Backfill Tool
 * Generates individual post files for all existing processed posts
 */

if (!defined('ABSPATH')) {
    exit;
}

class AIOX_Backfill_Tool {
    
    /**
     * Backfill individual post files for all processed posts
     */
    public static function backfill_all_posts() {
        $start_time = microtime(true);
        
        // Get all processed posts
        $posts = get_posts(array(
            'post_type' => array('post', 'page'),
            'post_status' => 'publish',
            'numberposts' => -1,
            'meta_query' => array(
                array(
                    'key' => '_aiox_processed',
                    'value' => '1',
                    'compare' => '='
                )
            )
        ));
        
        $total = count($posts);
        $success_count = 0;
        $error_count = 0;
        $errors = array();
        
        AIOX_Logger::info("Backfill Tool: Starting backfill for {$total} posts");
        
        require_once AIOX_PLUGIN_DIR . 'core/class-aiox-comprehensive-processor.php';
        $processor = new AIOX_Comprehensive_Processor();
        
        foreach ($posts as $post) {
            try {
                // Get processed data
                $processed_data = get_post_meta($post->ID, '_aiox_processed_data', true);
                
                if (empty($processed_data)) {
                    // Try to get old-style data
                    $processed_data = array(
                        'summary' => get_post_meta($post->ID, '_aiox_summary', true),
                        'key_points' => get_post_meta($post->ID, '_aiox_key_points', true),
                        'topic_analysis' => get_post_meta($post->ID, '_aiox_topics', true),
                        'qa_pairs' => get_post_meta($post->ID, '_aiox_qa_pairs', true)
                    );
                }
                
                // Generate individual post file using server_data_handler method
                // This creates: .well-known/posts/{post_id}.json
                if (class_exists('AIOX_Server_Data_Handler')) {
                    $reflection = new ReflectionClass('AIOX_Server_Data_Handler');
                    $method = $reflection->getMethod('generate_files_for_post');
                    $method->setAccessible(true);
                    $result = $method->invoke(null, $post->ID);
                } else {
                    throw new Exception('AIOX_Server_Data_Handler class not found');
                }
                
                if ($result) {
                    $success_count++;
                    AIOX_Logger::debug("Backfill Tool: Success for post {$post->ID}");
                } else {
                    $error_count++;
                    $errors[] = array(
                        'post_id' => $post->ID,
                        'error' => 'File generation returned false'
                    );
                }
                
            } catch (Exception $e) {
                $error_count++;
                $errors[] = array(
                    'post_id' => $post->ID,
                    'error' => $e->getMessage()
                );
                AIOX_Logger::error("Backfill Tool: Failed for post {$post->ID}: " . $e->getMessage());
            }
        }
        
        $elapsed_time = microtime(true) - $start_time;
        
        $result = array(
            'total' => $total,
            'success' => $success_count,
            'errors' => $error_count,
            'error_details' => $errors,
            'elapsed_time' => round($elapsed_time, 2)
        );
        
        AIOX_Logger::info("Backfill Tool: Completed", $result);
        
        return $result;
    }
    
    /**
     * AJAX handler for backfill
     */
    public static function ajax_backfill() {
        check_ajax_referer('aiox_backfill_nonce', 'nonce');
        
        if (!current_user_can('manage_options')) {
            wp_send_json_error(array('message' => 'Insufficient permissions'));
            return;
        }
        
        $result = self::backfill_all_posts();
        
        wp_send_json_success($result);
    }
    
    /**
     * Add admin menu item
     */
    public static function add_admin_menu() {
        add_submenu_page(
            'aiox-settings',
            'AIOX Backfill Tool',
            'Backfill Tool',
            'manage_options',
            'aiox-backfill',
            array(__CLASS__, 'render_page')
        );
    }
    
    /**
     * Render admin page
     */
    public static function render_page() {
        ?>
        <div class="wrap">
            <?php aiox_render_page_header(__('AIOX Backfill Tool', 'aiox-publisher-suite-pro'), __('Generate individual post files for all processed posts', 'aiox-publisher-suite-pro')); ?>
            
            <div class="card">
                <h2>Backfill Status</h2>
                <p>Click the button below to generate individual JSON files for all processed posts.</p>
                
                <button type="button" class="button button-primary" id="aiox-backfill-btn">
                    Start Backfill
                </button>
                
                <div id="aiox-backfill-progress" style="display:none; margin-top: 20px;">
                    <p><strong>Processing...</strong></p>
                    <div class="progress-bar">
                        <div class="progress-fill" style="width: 0%;"></div>
                    </div>
                </div>
                
                <div id="aiox-backfill-results" style="display:none; margin-top: 20px;"></div>
            </div>
        </div>
        
        <script>
        jQuery(document).ready(function($) {
            $('#aiox-backfill-btn').on('click', function() {
                var $btn = $(this);
                var $progress = $('#aiox-backfill-progress');
                var $results = $('#aiox-backfill-results');
                
                $btn.prop('disabled', true);
                $progress.show();
                $results.hide();
                
                $.ajax({
                    url: ajaxurl,
                    type: 'POST',
                    data: {
                        action: 'aiox_backfill',
                        nonce: '<?php echo wp_create_nonce('aiox_backfill_nonce'); ?>'
                    },
                    success: function(response) {
                        $progress.hide();
                        $btn.prop('disabled', false);
                        
                        if (response.success) {
                            var data = response.data;
                            var html = '<div class="notice notice-success"><p><strong>Backfill Complete!</strong></p>';
                            html += '<ul>';
                            html += '<li>Total posts: ' + data.total + '</li>';
                            html += '<li>Successfully processed: ' + data.success + '</li>';
                            html += '<li>Errors: ' + data.errors + '</li>';
                            html += '<li>Time elapsed: ' + data.elapsed_time + ' seconds</li>';
                            html += '</ul></div>';
                            
                            if (data.errors > 0 && data.error_details) {
                                html += '<div class="notice notice-warning"><p><strong>Errors:</strong></p><ul>';
                                data.error_details.forEach(function(error) {
                                    html += '<li>Post ID ' + error.post_id + ': ' + error.error + '</li>';
                                });
                                html += '</ul></div>';
                            }
                            
                            $results.html(html).show();
                        } else {
                            $results.html('<div class="notice notice-error"><p>' + response.data.message + '</p></div>').show();
                        }
                    },
                    error: function() {
                        $progress.hide();
                        $btn.prop('disabled', false);
                        $results.html('<div class="notice notice-error"><p>An error occurred during backfill.</p></div>').show();
                    }
                });
            });
        });
        </script>
        
        <style>
        .progress-bar {
            width: 100%;
            height: 30px;
            background: #f0f0f0;
            border-radius: 5px;
            overflow: hidden;
        }
        .progress-fill {
            height: 100%;
            background: #0073aa;
            transition: width 0.3s ease;
        }
        </style>
        <?php
    }
}

// Hook up the AJAX handler
add_action('wp_ajax_aiox_backfill', array('AIOX_Backfill_Tool', 'ajax_backfill'));

// Add admin menu
add_action('admin_menu', array('AIOX_Backfill_Tool', 'add_admin_menu'), 20);