<?php
/**
 * AIOX Capsule Meta Box Class
 */

if (!defined('ABSPATH')) {
    exit;
}

class AIOX_Capsule_Meta_Box {
    
    public function __construct() {
        
        add_action('add_meta_boxes', array($this, 'add_meta_box'));
        add_action('wp_ajax_aiox_generate_capsule_for_post', array($this, 'ajax_generate_capsule'));
        add_action('wp_ajax_aiox_delete_capsule_for_post', array($this, 'ajax_delete_capsule'));
        add_action('wp_ajax_aiox_debug_post_status', array($this, 'ajax_debug_post_status'));
        add_action('admin_footer', array($this, 'add_scripts'));
        
    }
    
    public function add_meta_box() {
        
        $post_types = array('post', 'page');
        
        foreach ($post_types as $post_type) {
            add_meta_box(
                'aiox-capsule-meta-box',
                'AIOX Processing',
                array($this, 'render_meta_box'),
                $post_type,
                'side',
                'default'
            );
        }
    }
    
    public function render_meta_box($post) {
        
        wp_nonce_field('aiox_capsule_meta_box', 'aiox_capsule_meta_box_nonce');
        
        // Check if post is processed
        $is_processed = get_post_meta($post->ID, '_aiox_processing_complete', true);
        $capsule = $this->get_capsule_for_post($post->ID);
        
        
        echo '<div id="aiox-capsule-messages"></div>';
        
        if ($is_processed && $capsule) {
            // POST IS PROCESSED - Show complete status with Regenerate and Clear buttons
            echo '<div class="aiox-processing-complete">';
            echo '<p><strong>AI Processing Status:</strong> <span class="aiox-status-ready">✅ Complete</span></p>';
            
            $created_at = isset($capsule['created_at']) ? $capsule['created_at'] : 'Unknown';
            $file_size = isset($capsule['file_size']) ? $capsule['file_size'] : 0;
            $capsule_id = isset($capsule['id']) ? $capsule['id'] : '';
            
            // Get additional processing info
            $ingredients_json = get_post_meta($post->ID, '_aiox_ingredients', true);
            $ingredients_data = !empty($ingredients_json) ? json_decode($ingredients_json, true) : array();
            $ingredients_count = is_array($ingredients_data) ? count($ingredients_data) : 0;
            
            $metadata_json = get_post_meta($post->ID, '_aiox_metadata', true);
            $metadata_data = !empty($metadata_json) ? json_decode($metadata_json, true) : array();
            $metadata_count = is_array($metadata_data) ? count($metadata_data) : 0;
            
            // Check for manifesto file using the new pagination system
            // NEW: Individual post files are now in .well-known/posts/{post_id}.json
            $manifesto_file_new = ABSPATH . '.well-known/posts/' . $post->ID . '.json';
            
            // Also check old locations for backward compatibility
            $post_slug = get_post_field('post_name', $post->ID);
            $post_title = get_the_title($post->ID);
            
            $clean_name = '';
            if (!empty($post_slug)) {
                $clean_name = $post_slug;
            } elseif (!empty($post_title)) {
                $clean_name = sanitize_title($post_title);
            } else {
                $clean_name = 'post-' . $post->ID;
            }
            
            $clean_name = substr($clean_name, 0, 50);
            $clean_name = preg_replace('/[^a-z0-9\-_]/', '', strtolower($clean_name));
            if (empty($clean_name)) {
                $clean_name = 'post-' . $post->ID;
            }
            
            $manifesto_file_old_slug = ABSPATH . '.well-known/ai-info-' . $clean_name . '.json';
            $manifesto_file_old_id = ABSPATH . '.well-known/ai-info-' . $post->ID . '.json';
            
            // Check new location first, then fall back to old locations
            $manifesto_exists = file_exists($manifesto_file_new) || file_exists($manifesto_file_old_slug) || file_exists($manifesto_file_old_id);
            $comprehensive_processed = get_post_meta($post->ID, '_aiox_comprehensive_processed', true);
            $badge_data = get_post_meta($post->ID, '_aiox_processed_data', true);
            
            echo '<p><strong>Last Processed:</strong> ' . date('M j, Y g:i A', strtotime($created_at)) . '</p>';
            echo '<p><strong>Processing Type:</strong> ' . ($comprehensive_processed ? '✅ Comprehensive' : '⚠️ Capsule Only') . '</p>';
            echo '<p><strong>Components:</strong></p>';
            echo '<ul style="margin-left: 20px; margin-top: 5px;">';
            
            // Capsule with clickable icon
            $capsule_url = home_url('/.well-known/capsules/' . $capsule_id . '.json');
            echo '<li><a href="' . esc_url($capsule_url) . '" target="_blank" title="Click to view this post\'s capsule file" style="text-decoration: none;">📦</a> Capsule: ' . $this->format_file_size($file_size) . '</li>';
            
            // Ingredients with clickable icon (links to post file which contains them)
            $post_file_url = home_url('/.well-known/posts/' . $post->ID . '.json');
            $ingredients_icon = $ingredients_count > 0 ? '<a href="' . esc_url($post_file_url) . '" target="_blank" title="Click to view this post\'s data file (contains ' . $ingredients_count . ' ingredients)" style="text-decoration: none;">🧩</a>' : '🧩';
            echo '<li>' . $ingredients_icon . ' Ingredients: ' . $ingredients_count . ' extracted' . ($ingredients_count > 0 ? '' : ' ❌') . '</li>';
            
            // Metadata
            echo '<li>🏷️ Metadata: ' . $metadata_count . ' fields' . ($metadata_count > 0 ? '' : ' ❌') . '</li>';
            
            // Manifesto/Post Data with clickable icon
            $manifesto_icon = $manifesto_exists ? '<a href="' . esc_url($post_file_url) . '" target="_blank" title="Click to view this post\'s complete data file" style="text-decoration: none;">📄</a>' : '📄';
            echo '<li>' . $manifesto_icon . ' Manifesto: ' . ($manifesto_exists ? '✅ Generated' : '❌ Missing') . '</li>';
            
            // Badge
            echo '<li>🏆 Badge: ' . ($badge_data ? '✅ Active' : '❌ Missing') . '</li>';
            echo '</ul>';
            
            // Debug info
            echo '<details style="margin-top: 10px;"><summary style="cursor: pointer; color: #666;">🔍 Debug Info</summary>';
            echo '<pre style="font-size: 11px; background: #f5f5f5; padding: 10px; margin-top: 5px;">';
            echo 'Comprehensive: ' . ($comprehensive_processed ? 'YES' : 'NO') . "\n";
            echo 'Processing Complete Flag: ' . ($is_processed ? 'SET' : 'NOT SET') . "\n";
            echo 'Ingredients data: ' . (is_array($ingredients_data) ? 'Array[' . count($ingredients_data) . ']' : 'Not array') . "\n";
            echo 'Metadata data: ' . (is_array($metadata_data) ? 'Array[' . count($metadata_data) . ']' : 'Not array') . "\n";
            echo 'Manifesto path (NEW): ' . $manifesto_file_new . "\n";
            echo 'Manifesto path (old slug): ' . $manifesto_file_old_slug . "\n";
            echo 'Manifesto path (old ID): ' . $manifesto_file_old_id . "\n";
            echo 'Manifesto exists: ' . ($manifesto_exists ? 'YES' : 'NO') . "\n";
            if (file_exists($manifesto_file_new)) {
                echo 'Found at: NEW location (posts directory)' . "\n";
            } elseif (file_exists($manifesto_file_old_slug)) {
                echo 'Found at: OLD location (slug-based)' . "\n";
            } elseif (file_exists($manifesto_file_old_id)) {
                echo 'Found at: OLD location (ID-based)' . "\n";
            }
            echo 'Clean filename: ' . $clean_name . "\n";
            echo 'Badge data: ' . ($badge_data ? 'Present' : 'Missing') . "\n";
            echo 'Capsule ID: ' . $capsule_id . "\n";
            echo '</pre></details>';
            
            // PROCESSED STATE: Show Regenerate and Clear AI Data buttons
            echo '<div class="aiox-capsule-actions" style="margin-top: 15px;">';
            echo '<button type="button" class="button button-primary aiox-regenerate-btn" data-post-id="' . $post->ID . '" style="margin-right: 5px;">🔄 Regenerate</button>';
            echo '<button type="button" class="button button-link-delete aiox-clear-data-btn" data-post-id="' . $post->ID . '" data-capsule-id="' . esc_attr($capsule_id) . '">🗑️ Clear AI Data</button>';
            echo '</div>';
            
            // TEMPORARY DEBUG BUTTON
            echo '<button type="button" class="button" id="aiox-debug-status" data-post-id="' . $post->ID . '" style="margin-top: 10px; width: 100%;">🔍 Debug Status</button>';
            echo '<div id="aiox-debug-output" style="margin-top: 10px; padding: 10px; background: #f5f5f5; display: none; font-family: monospace; font-size: 11px; max-height: 200px; overflow-y: auto;"></div>';
            
            echo '</div>';
            
        } else {
            // POST IS NOT PROCESSED - Show unprocessed status with Process button
            echo '<div class="aiox-no-processing">';
            echo '<p><strong>AI Processing Status:</strong> <span style="color: #d63638;">⏳ Not Processed</span></p>';
            echo '<p>This post has not been processed with AI enhancement yet.</p>';
            echo '<p><strong>Processing will generate:</strong></p>';
            echo '<ul style="margin-left: 20px; margin-top: 5px;">';
            echo '<li>📦 Capsule (structured content)</li>';
            echo '<li>🧩 Ingredients (semantic components)</li>';
            echo '<li>🏷️ Metadata (content classification)</li>';
            echo '<li>📄 Manifesto (public AI info)</li>';
            echo '<li>🏆 Badge (content enhancement)</li>';
            echo '</ul>';
            
            // UNPROCESSED STATE: Show Process button only
            echo '<button type="button" class="button button-primary aiox-process-btn" data-post-id="' . $post->ID . '" style="margin-top: 10px; width: 100%;">🚀 Process Post</button>';
            
            // Debug info for unprocessed state
            echo '<details style="margin-top: 10px;"><summary style="cursor: pointer; color: #666;">🔍 Debug Info</summary>';
            echo '<pre style="font-size: 11px; background: #f5f5f5; padding: 10px; margin-top: 5px;">';
            echo 'Processing Complete Flag: ' . ($is_processed ? 'SET (but no capsule)' : 'NOT SET') . "\n";
            echo 'Capsule Found: ' . ($capsule ? 'YES' : 'NO') . "\n";
            echo 'Meta _aiox_processing_complete: ' . get_post_meta($post->ID, '_aiox_processing_complete', true) . "\n";
            echo 'Meta _aiox_comprehensive_processed: ' . get_post_meta($post->ID, '_aiox_comprehensive_processed', true) . "\n";
            echo '</pre></details>';
            
            // TEMPORARY DEBUG BUTTON
            echo '<button type="button" class="button" id="aiox-debug-status" data-post-id="' . $post->ID . '" style="margin-top: 10px; width: 100%;">🔍 Debug Status</button>';
            echo '<div id="aiox-debug-output" style="margin-top: 10px; padding: 10px; background: #f5f5f5; display: none; font-family: monospace; font-size: 11px; max-height: 200px; overflow-y: auto;"></div>';
            
            echo '</div>';
        }
        
        echo '<style>
        .aiox-capsule-actions { margin-top: 10px; }
        .aiox-capsule-actions .button { margin-right: 5px; margin-bottom: 5px; }
        .aiox-status-ready { color: #46b450; font-weight: bold; }
        </style>';
    }
    
    private function get_capsule_for_post($post_id) {
        global $wpdb;
        
        $table_name = $wpdb->prefix . 'aiox_capsules';
        
        // Check if table exists
        if ($wpdb->get_var("SHOW TABLES LIKE '$table_name'") != $table_name) {
            return false;
        }
        
        // FIRST: Check _aiox_capsule_id meta key (set by Processing Manager)
        $capsule_id_from_meta = get_post_meta($post_id, '_aiox_capsule_id', true);
        if (!empty($capsule_id_from_meta)) {
            $capsule = $wpdb->get_row($wpdb->prepare(
                "SELECT * FROM $table_name WHERE id = %s LIMIT 1",
                $capsule_id_from_meta
            ), ARRAY_A);
            
            if ($capsule) {
                // Verify at least one capsule file exists
                $capsule_id = $capsule['id'];
                $upload_dir = wp_upload_dir();
                $file_path = $upload_dir['basedir'] . '/aiox-capsules/' . $capsule_id . '.json';
                $well_known_path = ABSPATH . '.well-known/capsules/' . $capsule_id . '.json';
                
                // Return capsule if any file exists, or if _aiox_processing_complete is set
                // (file might not exist yet but processing was done)
                if (file_exists($file_path) || file_exists($well_known_path) || get_post_meta($post_id, '_aiox_processing_complete', true)) {
                    return $capsule;
                }
            }
        }
        
        // SECOND: Check _aiox_processed_data exists (server-only mode may not create capsule files)
        $processed_data = get_post_meta($post_id, '_aiox_processed_data', true);
        if (!empty($processed_data)) {
            // Create a virtual capsule array for display purposes
            $data = is_string($processed_data) ? json_decode($processed_data, true) : $processed_data;
            if (is_array($data)) {
                return array(
                    'id' => $capsule_id_from_meta ?: 'virtual-' . $post_id,
                    'created_at' => get_post_meta($post_id, '_aiox_processed_at', true) ?: current_time('mysql'),
                    'file_size' => strlen(json_encode($data)),
                    'metadata' => json_encode(array('post_id' => $post_id)),
                    'status' => 'ready'
                );
            }
        }
        
        // THIRD: Fallback to pattern matching in metadata (legacy support)
        $patterns = array(
            '%"post_id":' . $post_id . '%',
            '%"post_id":"' . $post_id . '"%',
            '%"source_posts":[' . $post_id . ']%',
            '%"source_posts":["' . $post_id . '"]%'
        );
        
        foreach ($patterns as $pattern) {
            $capsule = $wpdb->get_row($wpdb->prepare(
                "SELECT * FROM $table_name WHERE metadata LIKE %s ORDER BY created_at DESC LIMIT 1",
                $pattern
            ), ARRAY_A);
            
            if ($capsule) {
                // Verify the capsule file actually exists
                $capsule_id = $capsule['id'];
                $upload_dir = wp_upload_dir();
                $file_path = $upload_dir['basedir'] . '/aiox-capsules/' . $capsule_id . '.json';
                $well_known_path = ABSPATH . '.well-known/capsules/' . $capsule_id . '.json';
                
                // If neither file exists, the capsule is orphaned - return false
                if (!file_exists($file_path) && !file_exists($well_known_path)) {
                    // Don't return this capsule since files are missing
                    continue;
                }
                
                return $capsule;
            }
        }
        
        return false;
    }
    
    private function format_file_size($bytes) {
        if ($bytes >= 1048576) {
            return number_format($bytes / 1048576, 2) . ' MB';
        } elseif ($bytes >= 1024) {
            return number_format($bytes / 1024, 2) . ' KB';
        } else {
            return $bytes . ' bytes';
        }
    }
    
    public function ajax_generate_capsule() {
        // Enhanced error handling to prevent internal server errors
        try {
            // Validate nonce
            if (!isset($_POST['nonce']) || !wp_verify_nonce($_POST['nonce'], 'aiox_capsules_nonce')) {
                wp_send_json_error(array('message' => 'Security check failed'));
                return;
            }
            
            // Check user permissions
            if (!current_user_can('edit_posts')) {
                wp_send_json_error(array('message' => 'Insufficient permissions'));
                return;
            }
            
            // Validate post ID
            $post_id = isset($_POST['post_id']) ? intval($_POST['post_id']) : 0;
            if (!$post_id) {
                wp_send_json_error(array('message' => 'Invalid post ID'));
                return;
            }
            
            // Check if post exists
            $post = get_post($post_id);
            if (!$post) {
                wp_send_json_error(array('message' => 'Post not found'));
                return;
            }
            
            // Check if required classes exist
            if (!file_exists(AIOX_PLUGIN_DIR . 'core/class-aiox-ai-only-processor.php')) {
                wp_send_json_error(array('message' => 'AI processor not found. Please reinstall the plugin.'));
                return;
            }
            
            if (!file_exists(AIOX_PLUGIN_DIR . 'core/class-aiox-capsule-manager.php')) {
                wp_send_json_error(array('message' => 'Capsule manager not found. Please reinstall the plugin.'));
                return;
            }
            
            // Load required classes safely
            require_once AIOX_PLUGIN_DIR . 'core/class-aiox-ai-only-processor.php';
            require_once AIOX_PLUGIN_DIR . 'core/class-aiox-capsule-manager.php';
            
            // Initialize AI processor
            $ai_processor = new AIOX_AI_Only_Processor();
            
            // Check if API keys are configured using the API validator
            require_once AIOX_PLUGIN_DIR . 'core/class-aiox-api-validator.php';
            $validation = AIOX_API_Validator::validate_current_provider();
            
            if (!$validation['valid']) {
                wp_send_json_error(array('message' => $validation['message'] . ' Please configure your API keys in AIOX Settings.'));
                return;
            }
            
            // Initialize capsule manager
            $capsule_manager = new AIOX_Capsule_Manager();
            
            // Generate capsule with comprehensive processing (all components)
            AIOX_Logger::log('Capsule Meta Box: Starting comprehensive processing for post ' . $post_id, AIOX_Logger::INFO);
            $comprehensive_processor = new AIOX_Comprehensive_Processor();
            $result = $comprehensive_processor->process_post_completely($post_id);
            AIOX_Logger::log('Capsule Meta Box: Comprehensive processing result type: ' . gettype($result), AIOX_Logger::INFO);
            
            if ($result && is_array($result) && isset($result['success']) && $result['success']) {
                // Comprehensive processor already created all components including capsule
                $capsule_id = isset($result['capsule_id']) ? $result['capsule_id'] : null;
                $ingredients_count = isset($result['ingredients_count']) ? $result['ingredients_count'] : 0;
                $metadata_fields = isset($result['metadata_fields']) ? $result['metadata_fields'] : 0;
                
                $message = 'Complete processing successful! ';
                $message .= 'Created: Capsule, ' . $ingredients_count . ' ingredients, ' . $metadata_fields . ' metadata fields, manifesto files.';
                
                wp_send_json_success(array(
                    'message' => $message,
                    'capsule_id' => $capsule_id,
                    'reload' => true
                ));
             } else {
                $error_message = 'AI content processing failed';
                if (is_array($result) && isset($result['message'])) {
                    $error_message .= ': ' . $result['message'];
                } elseif (is_string($result)) {
                    $error_message .= ': ' . $result;
                }
                $error_message .= '. Please check your API configuration.';
                
                wp_send_json_error(array('message' => $error_message));
            }
            
        } catch (Exception $e) {
            wp_send_json_error(array('message' => 'Error: ' . $e->getMessage()));
        } catch (Error $e) {
            wp_send_json_error(array('message' => 'Fatal Error: ' . $e->getMessage()));
        } catch (Throwable $e) {
            wp_send_json_error(array('message' => 'System Error: ' . $e->getMessage()));
        }
    }
    
    public function ajax_delete_capsule() {
        if (!wp_verify_nonce($_POST['nonce'], 'aiox_capsules_nonce')) {
            wp_send_json_error(array('message' => 'Security check failed'));
            return;
        }
        
        if (!current_user_can('edit_posts')) {
            wp_send_json_error(array('message' => 'Insufficient permissions'));
            return;
        }
        
        $post_id = intval($_POST['post_id']);
        $capsule_id = sanitize_text_field($_POST['capsule_id']);
        
        if (!$post_id || !$capsule_id) {
            wp_send_json_error(array('message' => 'Invalid parameters'));
            return;
        }
        
        try {
            // Check if AIOX_Capsule class exists
            if (!class_exists('AIOX_Capsule')) {
                wp_send_json_error(array('message' => 'AIOX_Capsule class not found'));
                return;
            }
            
            // Try to delete directly from database first
            global $wpdb;
            $table_name = $wpdb->prefix . 'aiox_capsules';
            
            $deleted = $wpdb->delete(
                $table_name,
                array('id' => $capsule_id),
                array('%s')
            );
            
            if ($deleted) {
                // Also try to delete the file
                $upload_dir = wp_upload_dir();
                $capsules_dir = $upload_dir['basedir'] . '/aiox-capsules/';
                $file_path = $capsules_dir . $capsule_id . '.json';
                
                if (file_exists($file_path)) {
                    unlink($file_path);
                }
                
                // Delete from well-known directory
                $well_known_path = ABSPATH . '.well-known/capsules/' . $capsule_id . '.json';
                if (file_exists($well_known_path)) {
                    unlink($well_known_path);
                }
                
                wp_send_json_success(array(
                    'message' => 'Capsule deleted successfully!',
                    'reload' => true
                ));
            } else {
                wp_send_json_error(array('message' => 'Failed to delete capsule from database'));
            }
            
        } catch (Exception $e) {
            wp_send_json_error(array('message' => 'Error: ' . $e->getMessage()));
        }
    }
    
    /**
     * Debug AJAX handler to check post status
     */
    public function ajax_debug_post_status() {
        
        if (!current_user_can('edit_posts')) {
            wp_send_json_error('No permission');
            return;
        }
        
        $post_id = intval($_POST['post_id'] ?? 0);
        if (!$post_id) {
            wp_send_json_error('No post ID');
            return;
        }
        
        global $wpdb;
        $table_name = $wpdb->prefix . 'aiox_capsules';
        
        // Get all meta flags
        $all_meta = get_post_meta($post_id);
        $aiox_meta = array();
        foreach ($all_meta as $key => $value) {
            if (strpos($key, '_aiox_') === 0) {
                $aiox_meta[$key] = $value[0];
            }
        }
        
        // Check for capsules in database
        $capsule_count = $wpdb->get_var($wpdb->prepare(
            "SELECT COUNT(*) FROM {$table_name} WHERE metadata LIKE %s OR metadata LIKE %s",
            '%"post_id":' . $post_id . '%',
            '%"source_posts":[' . $post_id . ']%'
        ));
        
        $capsule_records = $wpdb->get_results($wpdb->prepare(
            "SELECT id, title, status, created_at FROM {$table_name} WHERE metadata LIKE %s OR metadata LIKE %s",
            '%"post_id":' . $post_id . '%',
            '%"source_posts":[' . $post_id . ']%'
        ), ARRAY_A);
        
        $debug_info = array(
            'post_id' => $post_id,
            'post_title' => get_the_title($post_id),
            'all_aiox_meta' => $aiox_meta,
            'capsule_count_in_db' => $capsule_count,
            'capsule_records' => $capsule_records,
            'get_capsule_result' => $this->get_capsule_for_post($post_id) ? 'Found (with valid files)' : 'Not found or orphaned',
            'is_processed_check' => get_post_meta($post_id, '_aiox_processing_complete', true) ? 'TRUE' : 'FALSE'
        );
        
        
        wp_send_json_success($debug_info);
    }
    
    public function add_scripts() {
        global $post_type;
        
        if (!in_array($post_type, array('post', 'page'))) {
            return;
        }
        
        
        $nonce = wp_create_nonce('aiox_capsules_nonce');
        ?>
        <script type="text/javascript">
        console.log('AIOX Capsule Meta Box: Scripts loaded');
        
        jQuery(document).ready(function($) {
            console.log('AIOX Capsule Meta Box: jQuery ready');
            
            // Debug button
            $('#aiox-debug-status').on('click', function() {
                console.log('Debug button clicked');
                var postId = $(this).data('post-id');
                var output = $('#aiox-debug-output');
                
                output.show().text('Loading...');
                
                $.post(ajaxurl, {
                    action: 'aiox_debug_post_status',
                    post_id: postId
                }, function(response) {
                    console.log('Debug response:', response);
                    if (response.success) {
                        output.text(JSON.stringify(response.data, null, 2));
                    } else {
                        output.text('Error: ' + JSON.stringify(response));
                    }
                }).fail(function(xhr, status, error) {
                    console.error('Debug AJAX error:', error);
                    output.text('AJAX Error: ' + error);
                });
            });
            
            $('.aiox-view-btn').on('click', function() {
                var capsuleId = $(this).data('capsule-id');
                var viewUrl = '<?php echo admin_url('admin.php?page=aiox-capsules&tab=manage'); ?>&action=view&id=' + encodeURIComponent(capsuleId);
                window.open(viewUrl, '_blank');
            });
            
            // Process button - for unprocessed posts
            $('.aiox-process-btn').on('click', function() {
                var button = $(this);
                var postId = button.data('post-id');
                var originalText = button.text();
                
                button.prop('disabled', true).text('🔄 Processing...');
                $('#aiox-capsule-messages').empty();
                
                console.log('Process button clicked for post:', postId);
                
                $.ajax({
                    url: ajaxurl,
                    type: 'POST',
                    data: {
                        action: 'aiox_process_all_components',
                        post_id: postId,
                        nonce: '<?php echo $nonce; ?>'
                    },
                    success: function(response) {
                        console.log('Process response:', response);
                        if (response.success) {
                            var message = '<div class="notice notice-success"><p><strong>✅ Processing Complete!</strong><br>';
                            message += response.data.message;
                            if (response.data.components) {
                                message += '<br><br><strong>Components processed:</strong><br>';
                                if (response.data.components.capsule) message += '• Capsule: ' + response.data.components.capsule + '<br>';
                                if (response.data.components.ingredients) message += '• Ingredients: ' + response.data.components.ingredients + ' extracted<br>';
                                if (response.data.components.metadata) message += '• Metadata: ' + response.data.components.metadata + ' fields<br>';
                                if (response.data.components.manifesto) message += '• Manifesto files: Updated<br>';
                                if (response.data.components.badge) message += '• Badge data: Updated<br>';
                            }
                            message += '</p></div>';
                            $('#aiox-capsule-messages').html(message);
                            setTimeout(function() {
                                location.reload();
                            }, 2000);
                        } else {
                            $('#aiox-capsule-messages').html('<div class="notice notice-error"><p><strong>❌ Processing Failed:</strong><br>' + response.data.message + '</p></div>');
                            button.prop('disabled', false).text(originalText);
                        }
                    },
                    error: function(xhr, status, error) {
                        console.log('AJAX Error:', xhr.responseText);
                        $('#aiox-capsule-messages').html('<div class="notice notice-error"><p><strong>❌ AJAX Error:</strong> ' + error + '</p></div>');
                        button.prop('disabled', false).text(originalText);
                    }
                });
            });
            
            // Regenerate button - for processed posts
            $('.aiox-regenerate-btn').on('click', function() {
                if (!confirm('This will delete all existing AI data and regenerate everything from scratch. Continue?')) {
                    return;
                }
                
                var button = $(this);
                var postId = button.data('post-id');
                var originalText = button.text();
                
                button.prop('disabled', true).text('🔄 Regenerating...');
                $('#aiox-capsule-messages').html('<div class="notice notice-info"><p>⏳ Deleting existing data...</p></div>');
                
                console.log('Regenerate button clicked for post:', postId);
                
                // Step 1: Clear existing data
                $.ajax({
                    url: ajaxurl,
                    type: 'POST',
                    data: {
                        action: 'aiox_clear_all_ai_data',
                        post_id: postId,
                        nonce: '<?php echo $nonce; ?>'
                    },
                    success: function(clearResponse) {
                        console.log('Clear response:', clearResponse);
                        if (clearResponse.success) {
                            $('#aiox-capsule-messages').html('<div class="notice notice-info"><p>⏳ Processing post with AI...</p></div>');
                            
                            // Step 2: Reprocess the post
                            $.ajax({
                                url: ajaxurl,
                                type: 'POST',
                                data: {
                                    action: 'aiox_process_all_components',
                                    post_id: postId,
                                    nonce: '<?php echo $nonce; ?>'
                                },
                                success: function(response) {
                                    console.log('Reprocess response:', response);
                                    if (response.success) {
                                        var message = '<div class="notice notice-success"><p><strong>✅ Regeneration Complete!</strong><br>';
                                        message += response.data.message;
                                        if (response.data.components) {
                                            message += '<br><br><strong>Components regenerated:</strong><br>';
                                            if (response.data.components.capsule) message += '• Capsule: ' + response.data.components.capsule + '<br>';
                                            if (response.data.components.ingredients) message += '• Ingredients: ' + response.data.components.ingredients + ' extracted<br>';
                                            if (response.data.components.metadata) message += '• Metadata: ' + response.data.components.metadata + ' fields<br>';
                                            if (response.data.components.manifesto) message += '• Manifesto files: Updated<br>';
                                            if (response.data.components.badge) message += '• Badge data: Updated<br>';
                                        }
                                        message += '</p></div>';
                                        $('#aiox-capsule-messages').html(message);
                                        
                                        // Reload after 2 seconds
                                        setTimeout(function() {
                                            location.reload();
                                        }, 2000);
                                    } else {
                                        $('#aiox-capsule-messages').html('<div class="notice notice-error"><p><strong>❌ Processing Failed:</strong><br>' + response.data.message + '</p></div>');
                                        button.prop('disabled', false).text(originalText);
                                    }
                                },
                                error: function(xhr, status, error) {
                                    console.log('Reprocess AJAX Error:', xhr.responseText);
                                    $('#aiox-capsule-messages').html('<div class="notice notice-error"><p><strong>❌ AJAX Error:</strong> ' + error + '</p></div>');
                                    button.prop('disabled', false).text(originalText);
                                }
                            });
                        } else {
                            $('#aiox-capsule-messages').html('<div class="notice notice-error"><p><strong>❌ Clear Failed:</strong><br>' + clearResponse.data.message + '</p></div>');
                            button.prop('disabled', false).text(originalText);
                        }
                    },
                    error: function(xhr, status, error) {
                        console.log('Clear AJAX Error:', xhr.responseText);
                        $('#aiox-capsule-messages').html('<div class="notice notice-error"><p><strong>❌ AJAX Error:</strong> ' + error + '</p></div>');
                        button.prop('disabled', false).text(originalText);
                    }
                });
            });
            
            // Clear AI Data button - FIXED to properly delete everything and show Process button
            $('.aiox-clear-data-btn').on('click', function() {
                if (!confirm('Are you sure you want to clear all AI data for this post? This will remove the capsule, ingredients, metadata, and manifesto files.')) {
                    return;
                }
                
                var button = $(this);
                var postId = button.data('post-id');
                var originalText = button.text();
                
                button.prop('disabled', true).text('🗑️ Clearing...');
                $('#aiox-capsule-messages').empty();
                
                console.log('Clear AI Data button clicked for post:', postId);
                
                $.ajax({
                    url: ajaxurl,
                    type: 'POST',
                    data: {
                        action: 'aiox_clear_all_ai_data',
                        post_id: postId,
                        nonce: '<?php echo $nonce; ?>'
                    },
                    success: function(response) {
                        console.log('Clear AI Data response:', response);
                        if (response.success) {
                            var message = '<div class="notice notice-success"><p><strong>🗑️ AI Data Cleared!</strong><br>' + response.data.message;
                            if (response.data.components_cleared) {
                                message += '<br><br><strong>Components cleared:</strong><br>';
                                if (response.data.components_cleared.capsule) message += '• Capsule removed<br>';
                                if (response.data.components_cleared.ingredients) message += '• Ingredients cleared<br>';
                                if (response.data.components_cleared.metadata) message += '• Metadata cleared<br>';
                                if (response.data.components_cleared.manifesto) message += '• Manifesto files removed<br>';
                                if (response.data.components_cleared.badge) message += '• Badge data cleared<br>';
                            }
                            message += '</p></div>';
                            $('#aiox-capsule-messages').html(message);
                            
                            // Reload after 2 seconds to show Process button
                            setTimeout(function() {
                                location.reload();
                            }, 2000);
                        } else {
                            $('#aiox-capsule-messages').html('<div class="notice notice-error"><p><strong>❌ Clear Failed:</strong><br>' + response.data.message + '</p></div>');
                            button.prop('disabled', false).text(originalText);
                        }
                    },
                    error: function(xhr, status, error) {
                        console.log('Clear AI Data AJAX Error:', xhr.responseText);
                        $('#aiox-capsule-messages').html('<div class="notice notice-error"><p>AJAX error occurred: ' + error + '</p></div>');
                        button.prop('disabled', false).text(originalText);
                    }
                });
            });
        });
        </script>
        <?php
    }
}