<?php
/**
 * Additional AJAX Handlers for AIOX Capsules
 * Handles edit and duplicate operations
 */

if (!defined('ABSPATH')) {
    exit;
}

class AIOX_Capsules_Ajax_Handlers {
    
    /**
     * Constructor
     */
    public function __construct() {
        add_action('wp_ajax_aiox_edit_capsule', array($this, 'ajax_edit_capsule'));
        add_action('wp_ajax_aiox_duplicate_capsule', array($this, 'ajax_duplicate_capsule'));
        add_action('wp_ajax_aiox_update_capsule', array($this, 'ajax_update_capsule'));
    }
    
    /**
     * AJAX handler for editing capsules
     */
    public function ajax_edit_capsule() {
        check_ajax_referer('aiox_capsules_nonce', 'nonce');
        
        try {
            $capsule_id = sanitize_text_field($_POST['capsule_id'] ?? '');
            
            if (empty($capsule_id)) {
                throw new Exception('No capsule ID provided');
            }
            
            // Get capsule manager
            $capsule_manager = new AIOX_Capsule_Manager();
            $capsule = $capsule_manager->get_capsule($capsule_id);
            
            if (!$capsule) {
                throw new Exception('Capsule not found');
            }
            
            // Return capsule data for editing
            wp_send_json_success(array(
                'capsule' => array(
                    'id' => $capsule->get_id(),
                    'title' => $capsule->get_title(),
                    'description' => $capsule->get_description(),
                    'type' => $capsule->get_type(),
                    'status' => $capsule->get_status(),
                    'metadata' => $capsule->get_metadata(),
                    'content' => $capsule->get_content()
                )
            ));
            
        } catch (Exception $e) {
            wp_send_json_error($e->getMessage());
        }
    }
    
    /**
     * AJAX handler for updating capsules
     */
    public function ajax_update_capsule() {
        check_ajax_referer('aiox_capsules_nonce', 'nonce');
        
        try {
            $capsule_id = sanitize_text_field($_POST['capsule_id'] ?? '');
            $title = sanitize_text_field($_POST['title'] ?? '');
            $description = sanitize_textarea_field($_POST['description'] ?? '');
            $status = sanitize_text_field($_POST['status'] ?? '');
            
            if (empty($capsule_id)) {
                throw new Exception('No capsule ID provided');
            }
            
            if (empty($title)) {
                throw new Exception('Title is required');
            }
            
            // Get capsule manager
            $capsule_manager = new AIOX_Capsule_Manager();
            $capsule = $capsule_manager->load_capsule($capsule_id);
            
            if (!$capsule) {
                throw new Exception('Capsule not found');
            }
            
            // Update capsule
            $capsule->set_metadata('title', $title);
            $capsule->set_metadata('description', $description);
            
            if (!empty($status) && array_key_exists($status, AIOX_Capsule::STATUSES)) {
                $capsule->set_status($status);
            }
            
            // Also handle type if provided
            $type = sanitize_text_field($_POST['type'] ?? '');
            if (!empty($type) && array_key_exists($type, AIOX_Capsule::TYPES)) {
                $capsule->set_type($type);
            }
            
            $result = $capsule->save();
            
            if (!$result) {
                throw new Exception('Failed to update capsule');
            }
            
            wp_send_json_success(array(
                'message' => __('Capsule updated successfully!', 'aiox-publisher-suite-pro'),
                'capsule' => array(
                    'id' => $capsule->get_id(),
                    'title' => $capsule->get_title(),
                    'description' => $capsule->get_description(),
                    'status' => $capsule->get_status()
                )
            ));
            
        } catch (Exception $e) {
            wp_send_json_error($e->getMessage());
        }
    }
    
    /**
     * AJAX handler for duplicating capsules
     */
    public function ajax_duplicate_capsule() {
        check_ajax_referer('aiox_capsules_nonce', 'nonce');
        
        try {
            $capsule_id = sanitize_text_field($_POST['capsule_id'] ?? '');
            
            if (empty($capsule_id)) {
                throw new Exception('No capsule ID provided');
            }
            
            // Get capsule manager
            $capsule_manager = new AIOX_Capsule_Manager();
            $original_capsule = $capsule_manager->get_capsule($capsule_id);
            
            if (!$original_capsule) {
                throw new Exception('Original capsule not found');
            }
            
            // Create new capsule with copied data
            $new_capsule = new AIOX_Capsule();
            
            // Copy metadata
            $metadata = $original_capsule->get_metadata();
            $metadata['title'] = $metadata['title'] . ' (Copy)';
            $metadata['created_at'] = current_time('mysql');
            $metadata['updated_at'] = current_time('mysql');
            
            foreach ($metadata as $key => $value) {
                if ($key !== 'id') { // Don't copy the ID
                    $new_capsule->set_metadata($key, $value);
                }
            }
            
            // Copy content
            $content = $original_capsule->get_content();
            foreach ($content as $key => $value) {
                $new_capsule->set_content($key, $value);
            }
            
            // Set type and status
            $new_capsule->set_type($original_capsule->get_type());
            $new_capsule->set_status('draft'); // New copies start as draft
            
            // Save the new capsule
            $result = $new_capsule->save();
            
            if (!$result) {
                throw new Exception('Failed to duplicate capsule');
            }
            
            wp_send_json_success(array(
                'message' => __('Capsule duplicated successfully!', 'aiox-publisher-suite-pro'),
                'new_capsule_id' => $new_capsule->get_id(),
                'redirect' => admin_url('admin.php?page=aiox-capsules&tab=manage')
            ));
            
        } catch (Exception $e) {
            wp_send_json_error($e->getMessage());
        }
    }
}

// AJAX handlers are initialized by the main plugin file
