<?php
/**
 * AIOX Capsules Admin Page - Hybrid Version
 * 
 * Combines original styling with fixed workflow functionality.
 */

if (!defined('ABSPATH')) {
    exit;
}

require_once AIOX_PLUGIN_DIR . 'core/class-aiox-capsule-manager.php';
require_once AIOX_PLUGIN_DIR . 'core/class-aiox-capsule-generator.php';

class AIOX_Capsules_Hybrid {
    
    /**
     * Capsule manager instance
     */
    private $capsule_manager;
    
    /**
     * Capsule generator instance
     */
    private $capsule_generator;
    
    /**
     * Constructor
     */
    public function __construct() {
        
        try {
            $this->capsule_manager = new AIOX_Capsule_Manager();
            $this->capsule_generator = new AIOX_Capsule_Generator();
        } catch (Exception $e) {
            error_log('AIOX Capsules initialization error: ' . $e->getMessage());
        }
        
        // Register AJAX handlers only if WordPress is loaded
        if (function_exists('add_action')) {
            
            $ajax_handlers = array(
                'aiox_create_capsule' => 'ajax_create_capsule',
                'aiox_delete_capsule' => 'ajax_delete_capsule',
                'aiox_export_capsule' => 'ajax_export_capsule',
                'aiox_import_capsule' => 'ajax_import_capsule',
                'aiox_get_capsule_details' => 'ajax_get_capsule_details',
                'aiox_search_capsules' => 'ajax_search_capsules',
                'aiox_bulk_capsule_action' => 'ajax_bulk_capsule_action',
                'aiox_generate_topic_capsule' => 'ajax_generate_topic_capsule',
                'aiox_generate_author_capsule' => 'ajax_generate_author_capsule',
                'aiox_get_posts_for_selection' => 'ajax_get_posts_for_selection',
                'aiox_generate_all_missing_capsules' => 'ajax_generate_all_missing_capsules',
                'aiox_get_posts_without_capsules' => 'ajax_get_posts_without_capsules',
                'aiox_generate_capsules_for_selected_posts' => 'ajax_generate_capsules_for_selected_posts',
                'aiox_get_authors_for_selection' => 'ajax_get_authors_for_selection',
                'aiox_edit_capsule' => 'ajax_edit_capsule',
                'aiox_duplicate_capsule' => 'ajax_duplicate_capsule',
                'aiox_save_capsules_settings' => 'ajax_save_capsules_settings',
                'aiox_generate_capsule_for_post' => 'ajax_generate_capsule_for_post'
            );
            
            foreach ($ajax_handlers as $action => $method) {
                if (method_exists($this, $method)) {
                    add_action('wp_ajax_' . $action, array($this, $method));
                } else {
                }
            }
            
        } else {
        }
    }
    
    /**
     * Render capsules page
     */
    public function render_page() {
        $current_tab = isset($_GET['tab']) ? sanitize_text_field($_GET['tab']) : 'overview';
        
        ?>
        <div class="wrap aiox-capsules-page">
            <?php aiox_render_page_header(__('AIOX Capsules', 'aiox-publisher-suite-pro'), __('AI-Generated Content Capsules Management', 'aiox-publisher-suite-pro')); ?>
            
            <nav class="nav-tab-wrapper">
                <a href="?page=aiox-capsules&tab=overview" class="nav-tab <?php echo $current_tab === 'overview' ? 'nav-tab-active' : ''; ?>">
                    <?php _e('Overview', 'aiox-publisher-suite-pro'); ?>
                </a>
                <a href="?page=aiox-capsules&tab=manage" class="nav-tab <?php echo $current_tab === 'manage' ? 'nav-tab-active' : ''; ?>">
                    <?php _e('Manage Capsules', 'aiox-publisher-suite-pro'); ?>
                </a>
                <a href="?page=aiox-capsules&tab=create" class="nav-tab <?php echo $current_tab === 'create' ? 'nav-tab-active' : ''; ?>">
                    <?php _e('Create Capsule', 'aiox-publisher-suite-pro'); ?>
                </a>
                <a href="?page=aiox-capsules&tab=settings" class="nav-tab <?php echo $current_tab === 'settings' ? 'nav-tab-active' : ''; ?>">
                    <?php _e('Settings', 'aiox-publisher-suite-pro'); ?>
                </a>
            </nav>
            
            <div class="aiox-tab-content">
                <?php
                switch ($current_tab) {
                    case 'overview':
                        $this->render_overview_tab();
                        break;
                    case 'manage':
                        $this->render_manage_tab();
                        break;
                    case 'create':
                        $this->render_create_tab();
                        break;
                    case 'settings':
                        $this->render_settings_tab();
                        break;
                    default:
                        $this->render_overview_tab();
                        break;
                }
                ?>
            </div>
        </div>
        
        <?php $this->render_modals(); ?>
        <?php $this->render_scripts(); ?>
        <?php $this->render_styles(); ?>
        <?php
    }
    
    /**
     * Render overview tab
     */
    private function render_overview_tab() {
        $stats = $this->capsule_manager ? $this->capsule_manager->get_statistics() : array(
            'total' => 0,
            'by_type' => array(),
            'by_status' => array(),
            'recent' => 0
        );
        ?>
        <div class="aiox-overview-tab">
            <div class="aiox-stats-grid">
                <div class="aiox-stat-card">
                    <div class="aiox-stat-icon">📦</div>
                    <div class="aiox-stat-content">
                        <h3><?php echo esc_html($stats['total']); ?></h3>
                        <p><?php _e('Total Capsules', 'aiox-publisher-suite-pro'); ?></p>
                    </div>
                </div>
                
                <div class="aiox-stat-card">
                    <div class="aiox-stat-icon">✅</div>
                    <div class="aiox-stat-content">
                        <h3><?php echo esc_html(isset($stats['by_status']['active']) ? $stats['by_status']['active'] : 0); ?></h3>
                        <p><?php _e('Active Capsules', 'aiox-publisher-suite-pro'); ?></p>
                    </div>
                </div>
                
                <div class="aiox-stat-card">
                    <div class="aiox-stat-icon">🆕</div>
                    <div class="aiox-stat-content">
                        <h3><?php echo esc_html($stats['recent']); ?></h3>
                        <p><?php _e('Recent Capsules', 'aiox-publisher-suite-pro'); ?></p>
                    </div>
                </div>
                
                <div class="aiox-stat-card">
                    <div class="aiox-stat-icon">📊</div>
                    <div class="aiox-stat-content">
                        <h3><?php echo esc_html(count($stats['by_type'])); ?></h3>
                        <p><?php _e('Capsule Types', 'aiox-publisher-suite-pro'); ?></p>
                    </div>
                </div>
            </div>
            
            <div class="aiox-quick-actions">
                <h2><?php _e('Quick Actions', 'aiox-publisher-suite-pro'); ?></h2>
                <div class="aiox-action-grid">
                    <div class="aiox-action-card">
                        <div class="aiox-action-icon">📝</div>
                        <h3><?php _e('Single Post Capsule', 'aiox-publisher-suite-pro'); ?></h3>
                        <p><?php _e('Create a capsule from one post or page', 'aiox-publisher-suite-pro'); ?></p>
                        <button class="button button-primary aiox-quick-create" data-type="post">
                            <?php _e('Create Now', 'aiox-publisher-suite-pro'); ?>
                        </button>
                    </div>
                    
                    <div class="aiox-action-card">
                        <div class="aiox-action-icon">📚</div>
                        <h3><?php _e('Collection Capsule', 'aiox-publisher-suite-pro'); ?></h3>
                        <p><?php _e('Bundle multiple related posts together', 'aiox-publisher-suite-pro'); ?></p>
                        <button class="button button-primary aiox-quick-create" data-type="collection">
                            <?php _e('Create Now', 'aiox-publisher-suite-pro'); ?>
                        </button>
                    </div>
                    
                    <div class="aiox-action-card">
                        <div class="aiox-action-icon">👤</div>
                        <h3><?php _e('Author Profile', 'aiox-publisher-suite-pro'); ?></h3>
                        <p><?php _e('Create a comprehensive author profile capsule', 'aiox-publisher-suite-pro'); ?></p>
                        <button class="button button-primary aiox-quick-create" data-type="author">
                            <?php _e('Create Now', 'aiox-publisher-suite-pro'); ?>
                        </button>
                    </div>
                    
                    <div class="aiox-action-card">
                        <div class="aiox-action-icon">🏷️</div>
                        <h3><?php _e('Topic Capsule', 'aiox-publisher-suite-pro'); ?></h3>
                        <p><?php _e('Generate capsules based on topics or keywords', 'aiox-publisher-suite-pro'); ?></p>
                        <button class="button button-primary aiox-quick-create" data-type="topic">
                            <?php _e('Create Now', 'aiox-publisher-suite-pro'); ?>
                        </button>
                    </div>
                </div>
            </div>
        </div>
        <?php
    }
    
    /**
     * Render manage tab with error handling
     */
    private function render_manage_tab() {
        try {
            $capsules = array();
            if ($this->capsule_manager) {
                $capsules = $this->capsule_manager->get_all_capsules();
            }
        } catch (Exception $e) {
            $capsules = array();
            error_log('AIOX Capsules manage tab error: ' . $e->getMessage());
        }
        ?>
        <div class="aiox-manage-tab">
            <div class="aiox-capsules-header">
                <h2><?php _e('Manage Capsules', 'aiox-publisher-suite-pro'); ?></h2>
                <div class="aiox-capsules-actions">
                    <button type="button" class="button button-primary" id="aiox-generate-all-capsules">
                        <?php _e('Generate All Missing Capsules', 'aiox-publisher-suite-pro'); ?>
                    </button>
                    <button type="button" class="button button-secondary" id="aiox-toggle-post-selection">
                        <?php _e('Select Posts for Capsules', 'aiox-publisher-suite-pro'); ?>
                    </button>
                    <input type="search" id="aiox-capsules-search" placeholder="<?php _e('Search capsules...', 'aiox-publisher-suite-pro'); ?>" />
                    <select id="aiox-capsules-filter">
                        <option value=""><?php _e('All Types', 'aiox-publisher-suite-pro'); ?></option>
                        <option value="post"><?php _e('Post', 'aiox-publisher-suite-pro'); ?></option>
                        <option value="collection"><?php _e('Collection', 'aiox-publisher-suite-pro'); ?></option>
                        <option value="author"><?php _e('Author', 'aiox-publisher-suite-pro'); ?></option>
                        <option value="topic"><?php _e('Topic', 'aiox-publisher-suite-pro'); ?></option>
                    </select>
                    <a href="?page=aiox-capsules&tab=create" class="button button-primary">
                        <?php _e('Create New Capsule', 'aiox-publisher-suite-pro'); ?>
                    </a>
                </div>
            </div>
            
            <!-- Inline Post Selection Section -->
            <div id="aiox-post-selection-section" style="display: none; background: #f9f9f9; border: 1px solid #ddd; border-radius: 4px; padding: 20px; margin: 20px 0;">
                <h3><?php _e('Select Posts to Generate Capsules For', 'aiox-publisher-suite-pro'); ?></h3>
                <div id="aiox-post-selection-content">
                    <p><?php _e('Loading posts...', 'aiox-publisher-suite-pro'); ?></p>
                </div>
            </div>
            
            <div class="aiox-capsules-table-wrapper">
                <table class="wp-list-table widefat fixed striped">
                    <thead>
                        <tr>
                            <th class="check-column">
                                <input type="checkbox" id="aiox-select-all-capsules" />
                            </th>
                            <th><?php _e('Title', 'aiox-publisher-suite-pro'); ?></th>
                            <th><?php _e('Type', 'aiox-publisher-suite-pro'); ?></th>
                            <th><?php _e('Status', 'aiox-publisher-suite-pro'); ?></th>
                            <th><?php _e('Created', 'aiox-publisher-suite-pro'); ?></th>
                            <th><?php _e('Actions', 'aiox-publisher-suite-pro'); ?></th>
                        </tr>
                    </thead>
                    <tbody id="aiox-capsules-table-body">
                        <?php if (empty($capsules)): ?>
                            <tr>
                                <td colspan="6" class="aiox-no-capsules">
                                    <div class="aiox-empty-state">
                                        <div class="aiox-empty-icon">📦</div>
                                        <h3><?php _e('No capsules found', 'aiox-publisher-suite-pro'); ?></h3>
                                        <p><?php _e('Create your first capsule to get started!', 'aiox-publisher-suite-pro'); ?></p>
                                        <a href="?page=aiox-capsules&tab=create" class="button button-primary">
                                            <?php _e('Create Capsule', 'aiox-publisher-suite-pro'); ?>
                                        </a>
                                    </div>
                                </td>
                            </tr>
                        <?php else: ?>
                            <?php foreach ($capsules as $capsule): ?>
                                <?php
                                try {
                                    $title = method_exists($capsule, 'get_title') ? $capsule->get_title() : 'Untitled';
                                    $title = !empty($title) ? $title : 'AI Indexing Information - Capsule';
                                    $type = method_exists($capsule, 'get_type') ? $capsule->get_type() : 'unknown';
                                    $status = method_exists($capsule, 'get_status') ? $capsule->get_status() : 'active';
                                    $created = method_exists($capsule, 'get_created_date') ? $capsule->get_created_date() : 'Unknown';
                                    $id = method_exists($capsule, 'get_id') ? $capsule->get_id() : 'unknown';
                                    $description = method_exists($capsule, 'get_description') ? $capsule->get_description() : '';
                                } catch (Exception $e) {
                                    $title = 'Error loading capsule';
                                    $type = 'unknown';
                                    $status = 'error';
                                    $created = 'Unknown';
                                    $id = 'error';
                                    $description = '';
                                }
                                ?>
                                <tr class="aiox-capsule-row" data-type="<?php echo esc_attr($type); ?>" data-status="<?php echo esc_attr($status); ?>">
                                    <th class="check-column">
                                        <input type="checkbox" class="aiox-capsule-checkbox" value="<?php echo esc_attr($id); ?>" />
                                    </th>
                                    <td class="aiox-capsule-title">
                                        <strong><?php echo esc_html($title); ?></strong>
                                        <?php if ($description): ?>
                                        <div class="aiox-capsule-description">
                                            <?php echo esc_html(wp_trim_words($description, 15)); ?>
                                        </div>
                                        <?php endif; ?>
                                    </td>
                                    <td>
                                        <span class="aiox-capsule-type aiox-type-<?php echo esc_attr($type); ?>">
                                            <?php echo esc_html(ucfirst($type)); ?>
                                        </span>
                                    </td>
                                    <td>
                                        <span class="aiox-status aiox-status-<?php echo esc_attr($status); ?>">
                                            <?php echo esc_html(ucfirst($status)); ?>
                                        </span>
                                    </td>
                                    <td><?php echo esc_html($created); ?></td>
                                    <td class="aiox-capsule-actions">
                                        <button class="button button-small aiox-view-capsule" data-id="<?php echo esc_attr($id); ?>" title="<?php _e('View Details', 'aiox-publisher-suite-pro'); ?>">
                                            <?php _e('View', 'aiox-publisher-suite-pro'); ?>
                                        </button>
                                        <?php 
                                        $post_id = '';
                                        try {
                                            $post_id = $this->get_post_id_from_capsule($capsule);
                                        } catch (Exception $e) {
                                            $post_id = '';
                                        }
                                        ?>
                                        <button class="button button-small aiox-regenerate-capsule" data-id="<?php echo esc_attr($id); ?>" data-post-id="<?php echo esc_attr($post_id); ?>" title="<?php _e('Regenerate Capsule', 'aiox-publisher-suite-pro'); ?>">
                                            <?php _e('Regenerate', 'aiox-publisher-suite-pro'); ?>
                                        </button>
                                        <button class="button button-small aiox-export-capsule" data-id="<?php echo esc_attr($id); ?>" title="<?php _e('Export Capsule', 'aiox-publisher-suite-pro'); ?>">
                                            <?php _e('Export', 'aiox-publisher-suite-pro'); ?>
                                        </button>
                                        <button class="button button-small aiox-delete-capsule" data-id="<?php echo esc_attr($id); ?>" title="<?php _e('Delete Capsule', 'aiox-publisher-suite-pro'); ?>">
                                            <?php _e('Delete', 'aiox-publisher-suite-pro'); ?>
                                        </button>
                                    </td>
                                </tr>
                            <?php endforeach; ?>
                        <?php endif; ?>
                    </tbody>
                </table>
            </div>
            
            <?php if (!empty($capsules)): ?>
            <div class="aiox-bulk-actions">
                <select id="aiox-bulk-action">
                    <option value=""><?php _e('Bulk Actions', 'aiox-publisher-suite-pro'); ?></option>
                    <option value="delete"><?php _e('Delete', 'aiox-publisher-suite-pro'); ?></option>
                    <option value="export"><?php _e('Export', 'aiox-publisher-suite-pro'); ?></option>
                    <option value="activate"><?php _e('Activate', 'aiox-publisher-suite-pro'); ?></option>
                    <option value="deactivate"><?php _e('Deactivate', 'aiox-publisher-suite-pro'); ?></option>
                </select>
                <button class="button" id="aiox-apply-bulk-action"><?php _e('Apply', 'aiox-publisher-suite-pro'); ?></button>
            </div>
            <?php endif; ?>
            

        </div>
        <?php
    }
    
    /**
     * Render create tab with improved workflow
     */
    private function render_create_tab() {
        ?>
        <div class="aiox-create-tab">
            <div class="aiox-create-header">
                <h2><?php _e('Create New Capsule', 'aiox-publisher-suite-pro'); ?></h2>
                <p><?php _e('Choose a capsule type and follow the guided workflow to create your content capsule.', 'aiox-publisher-suite-pro'); ?></p>
            </div>
            
            <div class="aiox-create-types">
                <div class="aiox-type-card" data-type="post">
                    <div class="aiox-type-icon">📝</div>
                    <h3><?php _e('Single Post Capsule', 'aiox-publisher-suite-pro'); ?></h3>
                    <p><?php _e('Create a comprehensive capsule from a single post or page with enhanced metadata and AI analysis.', 'aiox-publisher-suite-pro'); ?></p>
                    <button class="button button-primary aiox-select-create-type" data-type="post">
                        <?php _e('Create Post Capsule', 'aiox-publisher-suite-pro'); ?>
                    </button>
                </div>
                
                <div class="aiox-type-card" data-type="collection">
                    <div class="aiox-type-icon">📚</div>
                    <h3><?php _e('Collection Capsule', 'aiox-publisher-suite-pro'); ?></h3>
                    <p><?php _e('Bundle multiple related posts together into a curated collection with unified metadata.', 'aiox-publisher-suite-pro'); ?></p>
                    <button class="button button-primary aiox-select-create-type" data-type="collection">
                        <?php _e('Create Collection', 'aiox-publisher-suite-pro'); ?>
                    </button>
                </div>
                
                <div class="aiox-type-card" data-type="author">
                    <div class="aiox-type-icon">👤</div>
                    <h3><?php _e('Author Profile Capsule', 'aiox-publisher-suite-pro'); ?></h3>
                    <p><?php _e('Generate a comprehensive author profile including all their content and biographical information.', 'aiox-publisher-suite-pro'); ?></p>
                    <button class="button button-primary aiox-select-create-type" data-type="author">
                        <?php _e('Create Author Profile', 'aiox-publisher-suite-pro'); ?>
                    </button>
                </div>
                
                <div class="aiox-type-card" data-type="topic">
                    <div class="aiox-type-icon">🏷️</div>
                    <h3><?php _e('Topic-Based Capsule', 'aiox-publisher-suite-pro'); ?></h3>
                    <p><?php _e('Create capsules based on specific topics or keywords, automatically discovering related content.', 'aiox-publisher-suite-pro'); ?></p>
                    <button class="button button-primary aiox-select-create-type" data-type="topic">
                        <?php _e('Create Topic Capsule', 'aiox-publisher-suite-pro'); ?>
                    </button>
                </div>
            </div>
            
            <!-- Content Selection Modal -->
            <div id="aiox-content-selection-modal" class="aiox-modal" style="display: none;">
                <div class="aiox-modal-content">
                    <div class="aiox-modal-header">
                        <h3 id="aiox-modal-title"><?php _e('Select Content', 'aiox-publisher-suite-pro'); ?></h3>
                        <button class="aiox-modal-close">&times;</button>
                    </div>
                    <div class="aiox-modal-body" id="aiox-modal-body">
                        <!-- Dynamic content will be loaded here -->
                    </div>
                    <div class="aiox-modal-footer">
                        <button class="button" id="aiox-modal-cancel"><?php _e('Cancel', 'aiox-publisher-suite-pro'); ?></button>
                        <button class="button button-primary" id="aiox-modal-create"><?php _e('Create Capsule', 'aiox-publisher-suite-pro'); ?></button>
                    </div>
                </div>
            </div>
        </div>
        <?php
    }
    
    /**
     * Render settings tab
     */
    private function render_settings_tab() {
        $settings = get_option('aiox_capsules_settings', array(
            'auto_generate' => false,
            'default_status' => 'active',
            'ai_enhancement' => true,
            'export_format' => 'json',
            'retention_days' => 365
        ));
        ?>
        <div class="aiox-settings-tab">
            <form method="post" action="" id="aiox-capsules-settings-form">
                <?php wp_nonce_field('aiox_capsules_settings', 'aiox_capsules_settings_nonce'); ?>
                
                <table class="form-table">
                    <tr>
                        <th scope="row"><?php _e('Auto-Generate Capsules', 'aiox-publisher-suite-pro'); ?></th>
                        <td>
                            <label>
                                <input type="checkbox" name="auto_generate" value="1" <?php checked($settings['auto_generate'], true); ?> />
                                <?php _e('Automatically generate capsules when posts are published or updated', 'aiox-publisher-suite-pro'); ?>
                            </label>
                        </td>
                    </tr>
                    
                    <tr>
                        <th scope="row"><?php _e('Default Status', 'aiox-publisher-suite-pro'); ?></th>
                        <td>
                            <select name="default_status">
                                <option value="active" <?php selected($settings['default_status'], 'active'); ?>><?php _e('Active', 'aiox-publisher-suite-pro'); ?></option>
                                <option value="draft" <?php selected($settings['default_status'], 'draft'); ?>><?php _e('Draft', 'aiox-publisher-suite-pro'); ?></option>
                                <option value="archived" <?php selected($settings['default_status'], 'archived'); ?>><?php _e('Archived', 'aiox-publisher-suite-pro'); ?></option>
                            </select>
                        </td>
                    </tr>
                    
                    <tr>
                        <th scope="row"><?php _e('AI Enhancement', 'aiox-publisher-suite-pro'); ?></th>
                        <td>
                            <label>
                                <input type="checkbox" name="ai_enhancement" value="1" <?php checked($settings['ai_enhancement'], true); ?> />
                                <?php _e('Enable AI-powered content analysis and enhancement for capsules', 'aiox-publisher-suite-pro'); ?>
                            </label>
                        </td>
                    </tr>
                    
                    <tr>
                        <th scope="row"><?php _e('Export Format', 'aiox-publisher-suite-pro'); ?></th>
                        <td>
                            <select name="export_format">
                                <option value="json" <?php selected($settings['export_format'], 'json'); ?>><?php _e('JSON', 'aiox-publisher-suite-pro'); ?></option>
                                <option value="xml" <?php selected($settings['export_format'], 'xml'); ?>><?php _e('XML', 'aiox-publisher-suite-pro'); ?></option>
                                <option value="csv" <?php selected($settings['export_format'], 'csv'); ?>><?php _e('CSV', 'aiox-publisher-suite-pro'); ?></option>
                            </select>
                        </td>
                    </tr>
                    
                    <tr>
                        <th scope="row"><?php _e('Retention Period', 'aiox-publisher-suite-pro'); ?></th>
                        <td>
                            <input type="number" name="retention_days" value="<?php echo esc_attr($settings['retention_days']); ?>" min="1" max="3650" />
                            <p class="description"><?php _e('Number of days to keep capsules before automatic cleanup (0 = never delete)', 'aiox-publisher-suite-pro'); ?></p>
                        </td>
                    </tr>
                </table>
                
                <p class="submit">
                    <button type="submit" class="button button-primary" id="aiox-save-capsules-settings">
                        <?php _e('Save Changes', 'aiox-publisher-suite-pro'); ?>
                    </button>
                </p>
            </form>
        </div>
        <?php
    }
    
    /**
     * AJAX handler for creating capsules
     */
    public function ajax_create_capsule() {
        
        check_ajax_referer('aiox_capsules_nonce', 'nonce');
        
        try {
            $type = sanitize_text_field($_POST['type'] ?? 'post');
            $title = sanitize_text_field($_POST['title'] ?? '');
            $description = sanitize_textarea_field($_POST['description'] ?? '');
            $post_ids = isset($_POST['post_ids']) ? array_map('intval', $_POST['post_ids']) : array();
            $author_id = isset($_POST['author_id']) ? intval($_POST['author_id']) : 0;
            $keywords = sanitize_text_field($_POST['keywords'] ?? '');
            
            
            if (!$this->capsule_manager) {
                throw new Exception('Capsule manager not available');
            }
            
            
            // Create capsule based on type
            switch ($type) {
                case 'post':
                    if (empty($post_ids)) {
                        throw new Exception('No post selected');
                    }
                    $options = array(
                        'post_id' => $post_ids[0],
                        'title' => $title ?: get_the_title($post_ids[0]),
                        'description' => $description
                    );
                    $capsule = $this->capsule_manager->create_capsule($type, $options);
                    break;
                    
                case 'collection':
                    if (empty($post_ids)) {
                        throw new Exception('No posts selected');
                    }
                    $options = array(
                        'post_ids' => $post_ids,
                        'title' => $title ?: 'Collection Capsule',
                        'description' => $description
                    );
                    $capsule = $this->capsule_manager->create_capsule($type, $options);
                    break;
                    
                case 'author':
                    if (empty($author_id)) {
                        throw new Exception('No author selected');
                    }
                    $author = get_userdata($author_id);
                    $options = array(
                        'author_id' => $author_id,
                        'title' => $title ?: ($author ? $author->display_name . ' Profile' : 'Author Profile'),
                        'description' => $description
                    );
                    $capsule = $this->capsule_manager->create_capsule($type, $options);
                    break;
                    
                case 'topic':
                    if (empty($keywords)) {
                        throw new Exception('No keywords provided');
                    }
                    $options = array(
                        'keywords' => $keywords,
                        'title' => $title ?: 'Topic: ' . $keywords,
                        'description' => $description
                    );
                    $capsule = $this->capsule_manager->create_capsule($type, $options);
                    break;
                    
                default:
                    // Create empty capsule for other types
                    $options = array(
                        'title' => $title ?: 'New Capsule',
                        'description' => $description
                    );
                    $capsule = $this->capsule_manager->create_capsule($type, $options);
                    break;
            }
            
            
            if (!$capsule) {
                throw new Exception('Failed to create capsule');
            }
            
            $capsule_id = method_exists($capsule, 'get_id') ? $capsule->get_id() : 'unknown';
            
            $response = array(
                'message' => __('Capsule created successfully!', 'aiox-publisher-suite-pro'),
                'capsule_id' => $capsule_id,
                'redirect' => admin_url('admin.php?page=aiox-capsules&tab=manage')
            );
            
            wp_send_json_success($response);
            
        } catch (Exception $e) {
            
            wp_send_json_error(array(
                'message' => sprintf(__('Error: %s', 'aiox-publisher-suite-pro'), $e->getMessage())
            ));
        }
    }
    
    /**
     * AJAX handler for deleting capsules
     */
    public function ajax_delete_capsule() {
        check_ajax_referer('aiox_capsules_nonce', 'nonce');
        
        try {
            $capsule_id = sanitize_text_field($_POST['capsule_id'] ?? '');
            
            if (empty($capsule_id)) {
                throw new Exception('No capsule ID provided');
            }
            
            if (!$this->capsule_manager) {
                throw new Exception('Capsule manager not available');
            }
            
            
            // Check if capsule exists before attempting deletion
            $capsule = $this->capsule_manager->load_capsule($capsule_id);
            if (!$capsule) {
                throw new Exception("Capsule not found: {$capsule_id}");
            }
            
            // Perform the deletion
            $result = $this->capsule_manager->delete_capsule($capsule_id);
            
            
            if (!$result) {
                // Get more specific error information
                $upload_dir = wp_upload_dir();
                $file_path = $upload_dir['basedir'] . '/aiox-capsules/' . $capsule_id . '.json';
                $file_exists_after = file_exists($file_path);
                
                // Check database
                global $wpdb;
                $table_name = $wpdb->prefix . 'aiox_capsules';
                $db_record_exists = $wpdb->get_var($wpdb->prepare("SELECT COUNT(*) FROM {$table_name} WHERE id = %s OR capsule_id = %s", $capsule_id, $capsule_id));
                
                $error_details = array();
                if ($file_exists_after) {
                    $error_details[] = 'File still exists';
                }
                if ($db_record_exists > 0) {
                    $error_details[] = 'Database record still exists';
                }
                
                $error_message = 'Failed to delete capsule completely';
                if (!empty($error_details)) {
                    $error_message .= ': ' . implode(', ', $error_details);
                }
                
                throw new Exception($error_message);
            }
            
            // Verify deletion was successful
            $upload_dir = wp_upload_dir();
            $file_path = $upload_dir['basedir'] . '/aiox-capsules/' . $capsule_id . '.json';
            $file_still_exists = file_exists($file_path);
            
            if ($file_still_exists) {
                throw new Exception('Capsule file could not be deleted - check file permissions');
            }
            
            
            wp_send_json_success(array(
                'message' => __('Capsule deleted successfully!', 'aiox-publisher-suite-pro'),
                'capsule_id' => $capsule_id
            ));
            
        } catch (Exception $e) {
            wp_send_json_error(array(
                'message' => sprintf(__('Error: %s', 'aiox-publisher-suite-pro'), $e->getMessage())
            ));
        } catch (Throwable $e) {
            wp_send_json_error(array(
                'message' => __('Internal server error occurred during deletion', 'aiox-publisher-suite-pro')
            ));
        }
    }
    
    /**
     * AJAX handler for getting posts for selection
     */
    public function ajax_get_posts_for_selection() {
        check_ajax_referer('aiox_capsules_nonce', 'nonce');
        
        try {
            $multiple = isset($_POST['multiple']) && $_POST['multiple'] === 'true';
            
            $posts = get_posts(array(
                'post_type' => array('post', 'page'),
                'post_status' => 'publish',
                'numberposts' => 50,
                'orderby' => 'date',
                'order' => 'DESC'
            ));
            
            // Generate HTML for post selection
            $html = '<div class="aiox-post-selection">';
            
            if (empty($posts)) {
                $html .= '<p>' . __('No posts found.', 'aiox-publisher-suite-pro') . '</p>';
            } else {
                $html .= '<div class="aiox-posts-list">';
                
                foreach ($posts as $post) {
                    $post_date = date('M j, Y', strtotime($post->post_date));
                    $post_excerpt = wp_trim_words($post->post_excerpt ?: $post->post_content, 20);
                    $input_type = $multiple ? 'checkbox' : 'radio';
                    $input_name = $multiple ? 'selected_posts[]' : 'selected_post';
                    
                    $html .= '<div class="aiox-post-item">';
                    $html .= '<label>';
                    $html .= '<input type="' . $input_type . '" name="' . $input_name . '" value="' . $post->ID . '" />';
                    $html .= '<div class="aiox-post-info">';
                    $html .= '<h4>' . esc_html($post->post_title) . '</h4>';
                    $html .= '<div class="aiox-post-meta">';
                    $html .= '<span class="post-type">' . ucfirst($post->post_type) . '</span>';
                    $html .= '<span class="post-date">' . $post_date . '</span>';
                    $html .= '</div>';
                    if ($post_excerpt) {
                        $html .= '<p class="post-excerpt">' . esc_html($post_excerpt) . '</p>';
                    }
                    $html .= '</div>';
                    $html .= '</label>';
                    $html .= '</div>';
                }
                
                $html .= '</div>';
            }
            
            $html .= '</div>';
            
            wp_send_json_success(array('html' => $html));
            
        } catch (Exception $e) {
            wp_send_json_error(array('message' => $e->getMessage()));
        }
    }
    
    /**
     * AJAX handler for getting authors for selection
     */
    public function ajax_get_authors_for_selection() {
        check_ajax_referer('aiox_capsules_nonce', 'nonce');
        
        try {
            $authors = get_users(array(
                'who' => 'authors',
                'orderby' => 'display_name',
                'order' => 'ASC'
            ));
            
            // Generate HTML for author selection
            $html = '<div class="aiox-author-selection">';
            
            if (empty($authors)) {
                $html .= '<p>' . __('No authors found.', 'aiox-publisher-suite-pro') . '</p>';
            } else {
                $html .= '<div class="aiox-authors-list">';
                
                foreach ($authors as $author) {
                    $post_count = count_user_posts($author->ID);
                    $avatar = get_avatar($author->ID, 48);
                    
                    $html .= '<div class="aiox-author-item">';
                    $html .= '<label>';
                    $html .= '<input type="radio" name="selected_author" value="' . $author->ID . '" />';
                    $html .= '<div class="aiox-author-info">';
                    $html .= '<div class="aiox-author-avatar">' . $avatar . '</div>';
                    $html .= '<div class="aiox-author-details">';
                    $html .= '<h4>' . esc_html($author->display_name) . '</h4>';
                    $html .= '<div class="aiox-author-meta">';
                    $html .= '<span class="author-login">@' . esc_html($author->user_login) . '</span>';
                    $html .= '<span class="author-posts">' . sprintf(_n('%d post', '%d posts', $post_count, 'aiox-publisher-suite-pro'), $post_count) . '</span>';
                    $html .= '</div>';
                    if ($author->user_email) {
                        $html .= '<p class="author-email">' . esc_html($author->user_email) . '</p>';
                    }
                    $html .= '</div>';
                    $html .= '</div>';
                    $html .= '</label>';
                    $html .= '</div>';
                }
                
                $html .= '</div>';
            }
            
            $html .= '</div>';
            
            wp_send_json_success(array('html' => $html));
            
        } catch (Exception $e) {
            wp_send_json_error(array('message' => $e->getMessage()));
        }
    }
    
    /**
     * AJAX handler for getting capsule details
     */
    public function ajax_get_capsule_details() {
        check_ajax_referer('aiox_capsules_nonce', 'nonce');
        
        try {
            $capsule_id = sanitize_text_field($_POST['capsule_id'] ?? '');
            
            if (empty($capsule_id)) {
                throw new Exception('No capsule ID provided');
            }
            
            if (!$this->capsule_manager) {
                throw new Exception('Capsule manager not available');
            }
            
            $capsule = $this->capsule_manager->load_capsule($capsule_id);
            
            if (!$capsule) {
                throw new Exception('Capsule not found');
            }
            
            // Get complete capsule data including rich content
            $metadata = $capsule->get_metadata();
            $content = $capsule->get_content();
            
            $capsule_data = array(
                'id' => $capsule->get_id() ?: $capsule_id,
                'title' => $capsule->get_title() ?: 'Untitled Capsule',
                'description' => $capsule->get_description() ?: 'No description available',
                'type' => $capsule->get_type() ?: 'unknown',
                'status' => $capsule->get_status() ?: 'draft',
                'created_at' => $capsule->get_created_date() ?: date('Y-m-d H:i:s'),
                'file_size' => 0,
                'metadata' => $metadata ?: array(),
                'content' => $content ?: array()
            );
            
            // Try to get file size from multiple sources
            try {
                // First try from metadata
                if (isset($metadata['file_size']) && $metadata['file_size'] > 0) {
                    $capsule_data['file_size'] = $metadata['file_size'];
                } else {
                    // Try to get from actual file
                    $upload_dir = wp_upload_dir();
                    $file_path = $upload_dir['basedir'] . '/aiox-capsules/' . $capsule_id . '.json';
                    if (file_exists($file_path)) {
                        $capsule_data['file_size'] = filesize($file_path);
                    }
                }
            } catch (Exception $e) {
                // Ignore file size errors
            }
            
            // Add rich content summary for display
            if (is_array($content)) {
                $capsule_data['content_summary'] = array(
                    'has_summary' => !empty($content['summary']),
                    'key_points_count' => is_array($content['key_points']) ? count($content['key_points']) : 0,
                    'topics_count' => is_array($content['topics']) ? count($content['topics']) : 0,
                    'qa_pairs_count' => is_array($content['qa_pairs']) ? count($content['qa_pairs']) : 0,
                    'entities_count' => is_array($content['entities']) ? count($content['entities']) : 0
                );
            }
            
            
            if (is_array($content)) {
            }
            
            wp_send_json_success(array(
                'capsule' => $capsule_data
            ));
            
        } catch (Exception $e) {
            error_log('AIOX Capsule Details Error: ' . $e->getMessage());
            wp_send_json_error('Error loading capsule details: ' . $e->getMessage());
        } catch (Throwable $e) {
            error_log('AIOX Capsule Details Fatal Error: ' . $e->getMessage());
            wp_send_json_error('Internal server error occurred');
        }
    }
    
    /**
     * AJAX handler for exporting capsules
     */
    public function ajax_export_capsule() {
        check_ajax_referer('aiox_capsules_nonce', 'nonce');
        
        try {
            $capsule_id = sanitize_text_field($_POST['capsule_id'] ?? '');
            
            if (empty($capsule_id)) {
                throw new Exception('No capsule ID provided');
            }
            
            if (!$this->capsule_manager) {
                throw new Exception('Capsule manager not available');
            }
            
            $export_data = $this->capsule_manager->export_capsule($capsule_id);
            
            if (!$export_data) {
                throw new Exception('Failed to export capsule');
            }
            
            wp_send_json_success(array(
                'export_data' => $export_data,
                'filename' => 'capsule_' . $capsule_id . '_' . date('Y-m-d') . '.json'
            ));
            
        } catch (Exception $e) {
            wp_send_json_error(array(
                'message' => sprintf(__('Error: %s', 'aiox-publisher-suite-pro'), $e->getMessage())
            ));
        }
    }
    
    /**
     * AJAX handler for importing capsules
     */
    public function ajax_import_capsule() {
        check_ajax_referer('aiox_capsules_nonce', 'nonce');
        
        try {
            if (!isset($_FILES['capsule_file'])) {
                throw new Exception('No file uploaded');
            }
            
            $file = $_FILES['capsule_file'];
            
            if ($file['error'] !== UPLOAD_ERR_OK) {
                throw new Exception('File upload error');
            }
            
            $content = file_get_contents($file['tmp_name']);
            $data = json_decode($content, true);
            
            if (!$data) {
                throw new Exception('Invalid file format');
            }
            
            if (!$this->capsule_manager) {
                throw new Exception('Capsule manager not available');
            }
            
            $capsule = $this->capsule_manager->import_capsule($data);
            
            if (!$capsule) {
                throw new Exception('Failed to import capsule');
            }
            
            wp_send_json_success(array(
                'message' => __('Capsule imported successfully!', 'aiox-publisher-suite-pro'),
                'capsule_id' => $capsule->get_id()
            ));
            
        } catch (Exception $e) {
            wp_send_json_error(array(
                'message' => sprintf(__('Error: %s', 'aiox-publisher-suite-pro'), $e->getMessage())
            ));
        }
    }
    
    /**
     * AJAX handler for searching capsules
     */
    public function ajax_search_capsules() {
        check_ajax_referer('aiox_capsules_nonce', 'nonce');
        
        try {
            $search_term = sanitize_text_field($_POST['search'] ?? '');
            $type_filter = sanitize_text_field($_POST['type'] ?? '');
            
            if (!$this->capsule_manager) {
                throw new Exception('Capsule manager not available');
            }
            
            $capsules = $this->capsule_manager->search_capsules($search_term, $type_filter);
            
            wp_send_json_success(array(
                'capsules' => $capsules
            ));
            
        } catch (Exception $e) {
            wp_send_json_error(array(
                'message' => sprintf(__('Error: %s', 'aiox-publisher-suite-pro'), $e->getMessage())
            ));
        }
    }
    
    /**
     * AJAX handler for bulk actions
     */
    public function ajax_bulk_capsule_action() {
        check_ajax_referer('aiox_capsules_nonce', 'nonce');
        
        try {
            $action = sanitize_text_field($_POST['bulk_action'] ?? '');
            $capsule_ids = isset($_POST['capsule_ids']) ? array_map('sanitize_text_field', $_POST['capsule_ids']) : array();
            
            if (empty($action) || empty($capsule_ids)) {
                throw new Exception('Invalid action or no capsules selected');
            }
            
            if (!$this->capsule_manager) {
                throw new Exception('Capsule manager not available');
            }
            
            $processed = 0;
            $errors = array();
            
            foreach ($capsule_ids as $capsule_id) {
                try {
                    switch ($action) {
                        case 'delete':
                            $result = $this->capsule_manager->delete_capsule($capsule_id);
                            if (!$result) {
                                throw new Exception("Failed to delete capsule {$capsule_id}");
                            }
                            break;
                        case 'activate':
                            $capsule = $this->capsule_manager->load_capsule($capsule_id);
                            if ($capsule) {
                                $capsule->set_status('ready');
                                $capsule->save();
                            }
                            break;
                        case 'deactivate':
                            $capsule = $this->capsule_manager->load_capsule($capsule_id);
                            if ($capsule) {
                                $capsule->set_status('draft');
                                $capsule->save();
                            }
                            break;
                        case 'export':
                            // For export, we'll just count it as processed
                            // The actual export would need to be handled differently
                            break;
                    }
                    $processed++;
                } catch (Exception $e) {
                    $errors[] = "Error processing capsule {$capsule_id}: " . $e->getMessage();
                }
            }
            
            $message = sprintf(__('Bulk action completed. %d capsules processed.', 'aiox-publisher-suite-pro'), $processed);
            if (!empty($errors)) {
                $message .= ' ' . sprintf(__('%d errors occurred.', 'aiox-publisher-suite-pro'), count($errors));
            }
            
            wp_send_json_success(array(
                'message' => $message,
                'processed' => $processed,
                'errors' => $errors
            ));
            
        } catch (Exception $e) {
            wp_send_json_error($e->getMessage());
        }
    }
    
    /**
     * AJAX handler for generating topic capsules
     */
    public function ajax_generate_topic_capsule() {
        check_ajax_referer('aiox_capsules_nonce', 'nonce');
        
        try {
            $keywords = sanitize_text_field($_POST['keywords'] ?? '');
            
            if (empty($keywords)) {
                throw new Exception('No keywords provided');
            }
            
            if (!$this->capsule_generator) {
                throw new Exception('Capsule generator not available');
            }
            
            $capsule = $this->capsule_generator->generate_topic_capsule($keywords);
            
            if (!$capsule) {
                throw new Exception('Failed to generate topic capsule');
            }
            
            wp_send_json_success(array(
                'message' => __('Topic capsule generated successfully!', 'aiox-publisher-suite-pro'),
                'capsule_id' => $capsule->get_id()
            ));
            
        } catch (Exception $e) {
            wp_send_json_error(array(
                'message' => sprintf(__('Error: %s', 'aiox-publisher-suite-pro'), $e->getMessage())
            ));
        }
    }
    
    /**
     * AJAX handler for generating author capsules
     */
    public function ajax_generate_author_capsule() {
        check_ajax_referer('aiox_capsules_nonce', 'nonce');
        
        try {
            $author_id = intval($_POST['author_id'] ?? 0);
            
            if (empty($author_id)) {
                throw new Exception('No author selected');
            }
            
            if (!$this->capsule_generator) {
                throw new Exception('Capsule generator not available');
            }
            
            $capsule = $this->capsule_generator->generate_author_capsule($author_id);
            
            if (!$capsule) {
                throw new Exception('Failed to generate author capsule');
            }
            
            wp_send_json_success(array(
                'message' => __('Author capsule generated successfully!', 'aiox-publisher-suite-pro'),
                'capsule_id' => $capsule->get_id()
            ));
            
        } catch (Exception $e) {
            wp_send_json_error(array(
                'message' => sprintf(__('Error: %s', 'aiox-publisher-suite-pro'), $e->getMessage())
            ));
        }
    }


    
    /**
     * AJAX handler for editing capsule
     */
    public function ajax_edit_capsule() {
        // Log that the function was called
        
        // Simple test response
        wp_send_json_success(array(
            'message' => 'Edit handler is working',
            'test' => true
        ));
    }
    
    /**
     * AJAX handler for duplicating capsule
     */
    public function ajax_duplicate_capsule() {
        check_ajax_referer('aiox_capsules_nonce', 'nonce');
        
        try {
            $capsule_id = sanitize_text_field($_POST['capsule_id'] ?? '');
            
            if (empty($capsule_id)) {
                throw new Exception('No capsule ID provided');
            }
            
            if (!$this->capsule_manager) {
                throw new Exception('Capsule manager not available');
            }
            
            // Get original capsule
            $original_capsule = $this->capsule_manager->load_capsule($capsule_id);
            if (!$original_capsule) {
                throw new Exception('Original capsule not found');
            }
            
            // Create duplicate with new ID
            $duplicate_data = $original_capsule->get_data();
            $duplicate_data['title'] = $duplicate_data['title'] . ' (Copy)';
            
            $duplicate_capsule = $this->capsule_manager->create_capsule('custom', $duplicate_data);
            
            if (!$duplicate_capsule) {
                throw new Exception('Failed to create duplicate capsule');
            }
            
            wp_send_json_success(array(
                'message' => __('Capsule duplicated successfully!', 'aiox-publisher-suite-pro'),
                'capsule_id' => $duplicate_capsule->get_id()
            ));
            
        } catch (Exception $e) {
            wp_send_json_error(array(
                'message' => sprintf(__('Error: %s', 'aiox-publisher-suite-pro'), $e->getMessage())
            ));
        }
    }
    
    /**
     * AJAX handler for generating all missing capsules
     */
    public function ajax_generate_all_missing_capsules() {
        check_ajax_referer('aiox_capsules_nonce', 'nonce');
        
        try {
            if (!current_user_can('edit_posts')) {
                throw new Exception('Insufficient permissions');
            }
            
            // Get all published posts/pages without capsules
            $posts_without_capsules = $this->get_posts_without_capsules();
            
            if (empty($posts_without_capsules)) {
                wp_send_json_success(array(
                    'message' => 'All posts already have capsules',
                    'generated' => 0
                ));
            }
            
            $generated_count = 0;
            $errors = array();
            
            foreach ($posts_without_capsules as $post) {
                try {
                    $capsule = $this->capsule_generator->generate_post_capsule($post->ID);
                    if ($capsule) {
                        $generated_count++;
                    }
                } catch (Exception $e) {
                    $errors[] = "Post {$post->ID}: " . $e->getMessage();
                }
            }
            
            wp_send_json_success(array(
                'message' => "Generated {$generated_count} capsules",
                'generated' => $generated_count,
                'errors' => $errors
            ));
            
        } catch (Exception $e) {
            wp_send_json_error($e->getMessage());
        }
    }
    
    /**
     * AJAX handler for getting posts without capsules
     */
    public function ajax_get_posts_without_capsules() {
        check_ajax_referer('aiox_capsules_nonce', 'nonce');
        
        try {
            $posts_without_capsules = $this->get_posts_without_capsules();
            
            $posts_data = array();
            foreach ($posts_without_capsules as $post) {
                $posts_data[] = array(
                    'id' => $post->ID,
                    'title' => $post->post_title,
                    'type' => $post->post_type,
                    'date' => $post->post_date,
                    'status' => $post->post_status
                );
            }
            
            wp_send_json_success($posts_data);
            
        } catch (Exception $e) {
            wp_send_json_error($e->getMessage());
        }
    }
    
    /**
     * AJAX handler for generating capsules for selected posts
     */
    public function ajax_generate_capsules_for_selected_posts() {
        check_ajax_referer('aiox_capsules_nonce', 'nonce');
        
        try {
            if (!current_user_can('edit_posts')) {
                throw new Exception('Insufficient permissions');
            }
            
            $post_ids = array_map('intval', $_POST['post_ids'] ?? array());
            
            if (empty($post_ids)) {
                throw new Exception('No posts selected');
            }
            
            $generated_count = 0;
            $errors = array();
            
            foreach ($post_ids as $post_id) {
                try {
                    $capsule = $this->capsule_generator->generate_post_capsule($post_id);
                    if ($capsule) {
                        $generated_count++;
                    }
                } catch (Exception $e) {
                    $errors[] = "Post {$post_id}: " . $e->getMessage();
                }
            }
            
            wp_send_json_success(array(
                'message' => "Generated {$generated_count} capsules",
                'generated' => $generated_count,
                'errors' => $errors
            ));
            
        } catch (Exception $e) {
            wp_send_json_error($e->getMessage());
        }
    }
    
    /**
     * Get posts without capsules
     */
    private function get_posts_without_capsules() {
        global $wpdb;
        
        $posts = get_posts(array(
            'post_type' => array('post', 'page'),
            'post_status' => 'publish',
            'numberposts' => -1,
            'fields' => 'ids'
        ));
        
        if (empty($posts)) {
            return array();
        }
        
        // Get posts that already have capsules
        $table_name = $wpdb->prefix . 'aiox_capsules';
        $posts_with_capsules = array();
        
        if ($wpdb->get_var("SHOW TABLES LIKE '{$table_name}'") == $table_name) {
            foreach ($posts as $post_id) {
                // Check both metadata field and source_posts field for post ID
                $exists = $wpdb->get_var($wpdb->prepare(
                    "SELECT COUNT(*) FROM {$table_name} WHERE metadata LIKE %s OR metadata LIKE %s",
                    '%"post_id":' . $post_id . '%',
                    '%"source_posts":[' . $post_id . ']%'
                ));
                
                if ($exists > 0) {
                    $posts_with_capsules[] = $post_id;
                }
            }
        }
        
        // Get posts without capsules
        $posts_without_capsules_ids = array_diff($posts, $posts_with_capsules);
        
        if (empty($posts_without_capsules_ids)) {
            return array();
        }
        
        return get_posts(array(
            'post__in' => $posts_without_capsules_ids,
            'post_type' => array('post', 'page'),
            'post_status' => 'publish',
            'numberposts' => -1
        ));
    }
    
    /**
     * AJAX handler for saving capsules settings
     */
    public function ajax_save_capsules_settings() {
        check_ajax_referer('aiox_capsules_nonce', 'nonce');
        
        try {
            if (!current_user_can('manage_options')) {
                throw new Exception('Insufficient permissions');
            }
            
            $settings = array(
                'auto_generate' => isset($_POST['auto_generate']) ? (bool) $_POST['auto_generate'] : false,
                'default_status' => sanitize_text_field($_POST['default_status'] ?? 'active'),
                'ai_enhancement' => isset($_POST['ai_enhancement']) ? (bool) $_POST['ai_enhancement'] : false,
                'export_format' => sanitize_text_field($_POST['export_format'] ?? 'json'),
                'retention_days' => intval($_POST['retention_days'] ?? 365)
            );
            
            // Validate settings
            $valid_statuses = array('active', 'draft', 'archived');
            if (!in_array($settings['default_status'], $valid_statuses)) {
                $settings['default_status'] = 'active';
            }
            
            $valid_formats = array('json', 'xml', 'csv');
            if (!in_array($settings['export_format'], $valid_formats)) {
                $settings['export_format'] = 'json';
            }
            
            if ($settings['retention_days'] < 0) {
                $settings['retention_days'] = 365;
            }
            
            // Save settings
            $result = update_option('aiox_capsules_settings', $settings);
            
            wp_send_json_success(array(
                'message' => __('Settings saved successfully!', 'aiox-publisher-suite-pro'),
                'settings' => $settings
            ));
            
        } catch (Exception $e) {
            wp_send_json_error($e->getMessage());
        }
    }
    
    /**
     * Render modals
     */
    private function render_modals() {
        ?>
        <!-- Capsule Details Modal -->
        <div id="aiox-capsule-details-modal" class="aiox-modal" style="display: none;">
            <div class="aiox-modal-content aiox-modal-large">
                <div class="aiox-modal-header">
                    <h3><?php _e('Capsule Details', 'aiox-publisher-suite-pro'); ?></h3>
                    <button class="aiox-modal-close">&times;</button>
                </div>
                <div class="aiox-modal-body" id="aiox-capsule-details-content">
                    <!-- Content will be loaded dynamically -->
                </div>
                <div class="aiox-modal-footer">
                    <button class="button" id="aiox-close-details-modal"><?php _e('Close', 'aiox-publisher-suite-pro'); ?></button>
                </div>
            </div>
        </div>
        
        <!-- Confirmation Modal -->
        <div id="aiox-confirm-modal" class="aiox-modal" style="display: none;">
            <div class="aiox-modal-content">
                <div class="aiox-modal-header">
                    <h3 id="aiox-confirm-title"><?php _e('Confirm Action', 'aiox-publisher-suite-pro'); ?></h3>
                    <button class="aiox-modal-close">&times;</button>
                </div>
                <div class="aiox-modal-body">
                    <p id="aiox-confirm-message"><?php _e('Are you sure you want to perform this action?', 'aiox-publisher-suite-pro'); ?></p>
                </div>
                <div class="aiox-modal-footer">
                    <button class="button" id="aiox-confirm-cancel"><?php _e('Cancel', 'aiox-publisher-suite-pro'); ?></button>
                    <button class="button button-primary" id="aiox-confirm-ok"><?php _e('Confirm', 'aiox-publisher-suite-pro'); ?></button>
                </div>
            </div>
        </div>
        <?php
    }
    
    /**
     * Render JavaScript
     */
    private function render_scripts() {
        ?>
        <script type="text/javascript">
        jQuery(document).ready(function($) {
            // Debug: Check if variables are available
            console.log('AIOX DEBUG: Inline script loaded');
            console.log('AIOX DEBUG: aiox_capsules_ajax available:', typeof aiox_capsules_ajax !== 'undefined');
            
            // Ensure variables are available with fallbacks
            if (typeof aiox_capsules_ajax === 'undefined') {
                console.error('AIOX DEBUG: aiox_capsules_ajax not defined, using fallbacks');
                window.aiox_capsules_ajax = {
                    ajax_url: '<?php echo admin_url('admin-ajax.php'); ?>',
                    nonce: '<?php echo wp_create_nonce('aiox_capsules_nonce'); ?>'
                };
            }
            
            // Use the same variables as the external script
            var ajaxurl = aiox_capsules_ajax.ajax_url;
            var nonce = aiox_capsules_ajax.nonce;
            
            console.log('AIOX DEBUG: Variables set - ajaxurl:', ajaxurl, 'nonce:', nonce);
            
            // Bulk generation buttons
            $('#aiox-generate-all-capsules').on('click', function(e) {
                e.preventDefault();
                generateAllMissingCapsules();
            });
            
            $('#aiox-toggle-post-selection').on('click', function(e) {
                e.preventDefault();
                var section = $('#aiox-post-selection-section');
                if (section.is(':visible')) {
                    section.hide();
                    $(this).text('<?php _e('Select Posts for Capsules', 'aiox-publisher-suite-pro'); ?>');
                } else {
                    section.show();
                    $(this).text('<?php _e('Hide Post Selection', 'aiox-publisher-suite-pro'); ?>');
                    loadPostsForInlineSelection();
                }
            });
            
            // Quick create buttons
            $('.aiox-quick-create, .aiox-select-create-type').on('click', function(e) {
                e.preventDefault();
                var type = $(this).data('type');
                showContentSelectionModal(type);
            });
            
            // Content selection modal
            function showContentSelectionModal(type) {
                var modal = $('#aiox-content-selection-modal');
                var modalBody = $('#aiox-modal-body');
                var modalTitle = $('#aiox-modal-title');
                
                // Set title based on type
                var titles = {
                    'post': '<?php _e('Select Post', 'aiox-publisher-suite-pro'); ?>',
                    'collection': '<?php _e('Select Posts for Collection', 'aiox-publisher-suite-pro'); ?>',
                    'author': '<?php _e('Select Author', 'aiox-publisher-suite-pro'); ?>',
                    'topic': '<?php _e('Enter Keywords', 'aiox-publisher-suite-pro'); ?>'
                };
                
                modalTitle.text(titles[type] || '<?php _e('Select Content', 'aiox-publisher-suite-pro'); ?>');
                
                // Load content based on type
                loadContentSelection(type, modalBody);
                
                // Store type for later use
                modal.data('type', type);
                modal.show();
            }
            
            function loadContentSelection(type, container) {
                container.html('<div class="aiox-loading"><?php _e('Loading...', 'aiox-publisher-suite-pro'); ?></div>');
                
                switch(type) {
                    case 'post':
                        loadPostSelection(container, false);
                        break;
                    case 'collection':
                        loadPostSelection(container, true);
                        break;
                    case 'author':
                        loadAuthorSelection(container);
                        break;
                    case 'topic':
                        loadTopicSelection(container);
                        break;
                }
            }
            
            function loadPostSelection(container, multiple) {
                $.post(ajaxurl, {
                    action: 'aiox_get_posts_for_selection',
                    nonce: nonce,
                    multiple: multiple
                }, function(response) {
                    if (response.success) {
                        container.html(response.data.html);
                    } else {
                        container.html('<p class="error">' + response.data.message + '</p>');
                    }
                });
            }
            
            function loadAuthorSelection(container) {
                $.post(ajaxurl, {
                    action: 'aiox_get_authors_for_selection',
                    nonce: nonce
                }, function(response) {
                    if (response.success) {
                        container.html(response.data.html);
                    } else {
                        container.html('<p class="error">' + response.data.message + '</p>');
                    }
                });
            }
            
            function loadTopicSelection(container) {
                var html = '<div class="aiox-topic-selection">';
                html += '<label for="aiox-keywords"><?php _e('Keywords (comma-separated):', 'aiox-publisher-suite-pro'); ?></label>';
                html += '<input type="text" id="aiox-keywords" name="keywords" placeholder="<?php _e('Enter keywords...', 'aiox-publisher-suite-pro'); ?>" />';
                html += '<label for="aiox-capsule-title"><?php _e('Capsule Title:', 'aiox-publisher-suite-pro'); ?></label>';
                html += '<input type="text" id="aiox-capsule-title" name="title" placeholder="<?php _e('Enter title...', 'aiox-publisher-suite-pro'); ?>" />';
                html += '<label for="aiox-capsule-description"><?php _e('Description:', 'aiox-publisher-suite-pro'); ?></label>';
                html += '<textarea id="aiox-capsule-description" name="description" placeholder="<?php _e('Enter description...', 'aiox-publisher-suite-pro'); ?>"></textarea>';
                html += '</div>';
                container.html(html);
            }
            
            // Modal create button
            $('#aiox-modal-create').on('click', function(e) {
                e.preventDefault();
                var modal = $('#aiox-content-selection-modal');
                var type = modal.data('type');
                createCapsule(type);
            });
            
            function createCapsule(type) {
                var data = {
                    action: 'aiox_create_capsule',
                    nonce: nonce,
                    type: type
                };
                
                // Collect data based on type
                switch(type) {
                    case 'post':
                        var selectedPost = $('input[name="selected_post"]:checked').val();
                        if (!selectedPost) {
                            alert('<?php _e('Please select a post.', 'aiox-publisher-suite-pro'); ?>');
                            return;
                        }
                        data.post_ids = [selectedPost];
                        break;
                        
                    case 'collection':
                        var selectedPosts = $('input[name="selected_posts[]"]:checked').map(function() {
                            return this.value;
                        }).get();
                        if (selectedPosts.length === 0) {
                            alert('<?php _e('Please select at least one post.', 'aiox-publisher-suite-pro'); ?>');
                            return;
                        }
                        data.post_ids = selectedPosts;
                        break;
                        
                    case 'author':
                        var selectedAuthor = $('input[name="selected_author"]:checked').val();
                        if (!selectedAuthor) {
                            alert('<?php _e('Please select an author.', 'aiox-publisher-suite-pro'); ?>');
                            return;
                        }
                        data.author_id = selectedAuthor;
                        break;
                        
                    case 'topic':
                        var keywords = $('#aiox-keywords').val();
                        if (!keywords) {
                            alert('<?php _e('Please enter keywords.', 'aiox-publisher-suite-pro'); ?>');
                            return;
                        }
                        data.keywords = keywords;
                        data.title = $('#aiox-capsule-title').val();
                        data.description = $('#aiox-capsule-description').val();
                        break;
                }
                
                // Show loading
                $('#aiox-modal-create').prop('disabled', true).text('<?php _e('Creating...', 'aiox-publisher-suite-pro'); ?>');
                
                $.post(ajaxurl, data, function(response) {
                    if (response.success) {
                        alert(response.data.message);
                        $('#aiox-content-selection-modal').hide();
                        if (response.data.redirect) {
                            window.location.href = response.data.redirect;
                        } else {
                            location.reload();
                        }
                    } else {
                        alert(response.data.message);
                    }
                }).always(function() {
                    $('#aiox-modal-create').prop('disabled', false).text('<?php _e('Create Capsule', 'aiox-publisher-suite-pro'); ?>');
                });
            }
            
            // Modal close handlers
            $('.aiox-modal-close, #aiox-modal-cancel, #aiox-close-details-modal').on('click', function() {
                $('.aiox-modal').hide();
            });
            
            // View capsule details
            $(document).on('click', '.aiox-view-capsule', function(e) {
                e.preventDefault();
                var capsuleId = $(this).data('id');
                viewCapsuleDetails(capsuleId);
            });
            
            function viewCapsuleDetails(capsuleId) {
                console.log('DEBUG: viewCapsuleDetails called with ID:', capsuleId);
                
                $.ajax({
                    url: ajaxurl,
                    type: 'POST',
                    data: {
                        action: 'aiox_get_capsule_details',
                        nonce: nonce,
                        capsule_id: capsuleId
                    },
                    success: function(response) {
                        console.log('DEBUG: viewCapsuleDetails response:', response);
                        
                        if (response.success && response.data && response.data.capsule) {
                            var capsule = response.data.capsule;
                            var html = '<div class="aiox-capsule-details">';
                            html += '<h4>' + (capsule.title || 'Untitled Capsule') + '</h4>';
                            
                            // Basic Information
                            html += '<div class="aiox-details-section">';
                            html += '<h5><?php _e('Basic Information', 'aiox-publisher-suite-pro'); ?></h5>';
                            html += '<p><strong><?php _e('Type:', 'aiox-publisher-suite-pro'); ?></strong> ' + (capsule.type || 'Unknown') + '</p>';
                            html += '<p><strong><?php _e('Status:', 'aiox-publisher-suite-pro'); ?></strong> ' + (capsule.status || 'Unknown') + '</p>';
                            html += '<p><strong><?php _e('Created:', 'aiox-publisher-suite-pro'); ?></strong> ' + (capsule.created_at || 'Unknown') + '</p>';
                            if (capsule.description && capsule.description !== 'No description available') {
                                html += '<p><strong><?php _e('Description:', 'aiox-publisher-suite-pro'); ?></strong> ' + capsule.description + '</p>';
                            }
                            
                            // File Information
                            var actualFileSize = (capsule.metadata && capsule.metadata.file_size) ? capsule.metadata.file_size : (capsule.file_size || 0);
                            html += '<p><strong><?php _e('File Size:', 'aiox-publisher-suite-pro'); ?></strong> ' + actualFileSize + ' bytes</p>';
                            html += '</div>';
                            
                            // Content Information
                            if (capsule.content) {
                                html += '<div class="aiox-details-section">';
                                html += '<h5><?php _e('Content Information', 'aiox-publisher-suite-pro'); ?></h5>';
                                
                                if (capsule.content.summary) {
                                    html += '<p><strong><?php _e('Summary:', 'aiox-publisher-suite-pro'); ?></strong> ' + capsule.content.summary.substring(0, 200) + (capsule.content.summary.length > 200 ? '...' : '') + '</p>';
                                }
                                
                                if (capsule.content.topics && capsule.content.topics.length > 0) {
                                    html += '<p><strong><?php _e('Topics:', 'aiox-publisher-suite-pro'); ?></strong> ' + capsule.content.topics.slice(0, 5).join(', ') + (capsule.content.topics.length > 5 ? '...' : '') + '</p>';
                                }
                                
                                if (capsule.content.qa_pairs && capsule.content.qa_pairs.length > 0) {
                                    html += '<p><strong><?php _e('Q&A Pairs:', 'aiox-publisher-suite-pro'); ?></strong> ' + capsule.content.qa_pairs.length + ' pairs</p>';
                                }
                                
                                html += '</div>';
                            }
                            
                            // Metadata Information
                            if (capsule.metadata) {
                                html += '<div class="aiox-details-section">';
                                html += '<h5><?php _e('Metadata', 'aiox-publisher-suite-pro'); ?></h5>';
                                
                                if (capsule.metadata.source_posts && capsule.metadata.source_posts.length > 0) {
                                    html += '<p><strong><?php _e('Source Posts:', 'aiox-publisher-suite-pro'); ?></strong> ' + capsule.metadata.source_posts.length + ' posts</p>';
                                }
                                
                                if (capsule.metadata.version) {
                                    html += '<p><strong><?php _e('Version:', 'aiox-publisher-suite-pro'); ?></strong> ' + capsule.metadata.version + '</p>';
                                }
                                
                                if (capsule.metadata.content_hash) {
                                    html += '<p><strong><?php _e('Content Hash:', 'aiox-publisher-suite-pro'); ?></strong> ' + capsule.metadata.content_hash.substring(0, 16) + '...</p>';
                                }
                                
                                html += '</div>';
                            }
                            
                            html += '</div>';
                            
                            $('#aiox-capsule-details-content').html(html);
                            $('#aiox-capsule-details-modal').show();
                            console.log('DEBUG: Modal should be visible now');
                        } else {
                            var errorMsg = 'Failed to load capsule details';
                            if (response.data && response.data.message) {
                                errorMsg = response.data.message;
                            }
                            alert('Error: ' + errorMsg);
                            console.error('DEBUG: viewCapsuleDetails error:', response);
                        }
                    },
                    error: function(xhr, status, error) {
                        alert('AJAX Error: ' + error);
                        console.error('DEBUG: viewCapsuleDetails AJAX error:', xhr, status, error);
                    }
                });
            }
            
            // Regenerate capsule
            $(document).on('click', '.aiox-regenerate-capsule', function(e) {
                e.preventDefault();
                var button = $(this);
                var capsuleId = button.data('id');
                var postId = button.data('post-id');
                var originalText = button.text();
                
                if (!postId) {
                    alert('Post ID not found for this capsule');
                    return;
                }
                
                button.text('Regenerating...').prop('disabled', true);
                
                $.ajax({
                    url: ajaxurl,
                    type: 'POST',
                    data: {
                        action: 'aiox_generate_capsule_for_post',
                        post_id: postId,
                        nonce: nonce
                    },
                    success: function(response) {
                        if (response.success) {
                            alert('Capsule regenerated successfully!');
                            location.reload();
                        } else {
                            alert('Error: ' + (response.data ? response.data.message || response.data : 'Unknown error'));
                        }
                    },
                    error: function(xhr, status, error) {
                        alert('AJAX error occurred: ' + error);
                    },
                    complete: function() {
                        button.text(originalText).prop('disabled', false);
                    }
                });
            });
            
            function editCapsule(capsuleId) {
                // Construct the edit URL using the current page URL
                var currentUrl = window.location.href;
                var baseUrl = currentUrl.split('&action=')[0].split('&id=')[0].split('&edit=')[0];
                var editUrl = baseUrl + '&action=view&id=' + encodeURIComponent(capsuleId);
                window.location.href = editUrl;
            }
            
            function showEditModal(capsule) {
                var html = '<div class="aiox-edit-capsule">';
                html += '<h3>Edit Capsule</h3>';
                html += '<form id="aiox-edit-form">';
                html += '<p><label for="edit-title">Title:</label>';
                html += '<input type="text" id="edit-title" value="' + (capsule.title || '') + '" style="width: 100%; margin-top: 5px;"></p>';
                html += '<p><label for="edit-description">Description:</label>';
                html += '<textarea id="edit-description" rows="3" style="width: 100%; margin-top: 5px;">' + (capsule.description || '') + '</textarea></p>';
                html += '<p><label for="edit-status">Status:</label>';
                html += '<select id="edit-status" style="width: 100%; margin-top: 5px;">';
                html += '<option value="draft"' + (capsule.status === 'draft' ? ' selected' : '') + '>Draft</option>';
                html += '<option value="ready"' + (capsule.status === 'ready' ? ' selected' : '') + '>Ready</option>';
                html += '<option value="published"' + (capsule.status === 'published' ? ' selected' : '') + '>Published</option>';
                html += '</select></p>';
                html += '<div class="aiox-edit-actions">';
                html += '<button type="button" class="button button-primary" id="aiox-save-edit">Save Changes</button>';
                html += '<button type="button" class="button" id="aiox-cancel-edit">Cancel</button>';
                html += '</div>';
                html += '</form>';
                html += '</div>';
                
                // Create modal with inline styles
                var backdrop = $('<div></div>').css({
                    'position': 'fixed',
                    'top': '0',
                    'left': '0',
                    'width': '100%',
                    'height': '100%',
                    'background': 'rgba(0,0,0,0.5)',
                    'z-index': '999999'
                });
                
                var modal = $('<div></div>').css({
                    'position': 'fixed',
                    'top': '50%',
                    'left': '50%',
                    'transform': 'translate(-50%, -50%)',
                    'background': 'white',
                    'border': '1px solid #ccc',
                    'border-radius': '4px',
                    'box-shadow': '0 4px 20px rgba(0,0,0,0.3)',
                    'z-index': '1000000',
                    'max-width': '500px',
                    'width': '90%',
                    'max-height': '80vh',
                    'overflow': 'auto',
                    'padding': '20px',
                    'box-sizing': 'border-box'
                }).html(html);
                
                $('body').append(backdrop).append(modal);
                
                // Handle actions
                modal.find('#aiox-cancel-edit').on('click', function() {
                    backdrop.remove();
                    modal.remove();
                });
                
                backdrop.on('click', function() {
                    backdrop.remove();
                    modal.remove();
                });
                
                modal.find('#aiox-save-edit').on('click', function() {
                    var title = modal.find('#edit-title').val();
                    var description = modal.find('#edit-description').val();
                    var status = modal.find('#edit-status').val();
                    
                    $.post(ajaxurl, {
                        action: 'aiox_edit_capsule',
                        nonce: nonce,
                        capsule_id: capsule.id,
                        save_edit: true,
                        title: title,
                        description: description,
                        status: status
                    }, function(response) {
                        if (response.success) {
                            backdrop.remove();
                            modal.remove();
                            if (response.data.reload) {
                                location.reload();
                            } else {
                                alert(response.data.message);
                            }
                        } else {
                            alert('Error saving: ' + response.data);
                        }
                    }).fail(function() {
                        alert('Save failed');
                    });
                });
            }
            
            // Duplicate capsule
            $(document).on('click', '.aiox-duplicate-capsule', function(e) {
                e.preventDefault();
                var capsuleId = $(this).data('id');
                duplicateCapsule(capsuleId);
            });
            
            function duplicateCapsule(capsuleId) {
                $.post(ajaxurl, {
                    action: 'aiox_duplicate_capsule',
                    nonce: nonce,
                    capsule_id: capsuleId
                }, function(response) {
                    if (response.success) {
                        alert(response.data.message);
                        location.reload();
                    } else {
                        alert(response.data.message);
                    }
                });
            }
            
            // Delete capsule
            $(document).on('click', '.aiox-delete-capsule', function(e) {
                e.preventDefault();
                var capsuleId = $(this).data('id');
                
                if (confirm('<?php _e('Are you sure you want to delete this capsule?', 'aiox-publisher-suite-pro'); ?>')) {
                    deleteCapsule(capsuleId);
                }
            });
            
            function deleteCapsule(capsuleId) {
                console.log('DEBUG: deleteCapsule called with ID:', capsuleId);
                $.ajax({
                    url: aiox_capsules_ajax.ajax_url,
                    type: 'POST',
                    data: {
                        action: 'aiox_delete_capsule',
                        nonce: aiox_capsules_ajax.nonce,
                        capsule_id: capsuleId
                    },
                    success: function(response) {
                        console.log('DEBUG: deleteCapsule response:', response);
                        
                        if (response.success) {
                            // Show success message briefly, then reload
                            alert(response.data.message || 'Capsule deleted successfully!');
                            location.reload();
                        } else {
                            // Show error message
                            var errorMsg = 'Failed to delete capsule';
                            if (response.data && response.data.message) {
                                errorMsg = response.data.message;
                            }
                            alert('Error: ' + errorMsg);
                            console.error('DEBUG: deleteCapsule error:', response);
                        }
                    },
                    error: function(xhr, status, error) {
                        console.error('DEBUG: deleteCapsule AJAX error:', xhr, status, error);
                        alert('AJAX Error: ' + error);
                    }
                });
            }
            
            // Export capsule
            $(document).on('click', '.aiox-export-capsule', function(e) {
                e.preventDefault();
                var capsuleId = $(this).data('id');
                exportCapsule(capsuleId);
            });
            
            function exportCapsule(capsuleId) {
                console.log('DEBUG: exportCapsule called with ID:', capsuleId);
                
                $.ajax({
                    url: ajaxurl,
                    type: 'POST',
                    data: {
                        action: 'aiox_export_capsule',
                        nonce: nonce,
                        capsule_id: capsuleId
                    },
                    success: function(response) {
                        console.log('DEBUG: exportCapsule response:', response);
                        
                        if (response.success && response.data && response.data.export_data) {
                            try {
                                // Create download link
                                var jsonString = JSON.stringify(response.data.export_data, null, 2);
                                var blob = new Blob([jsonString], {type: 'application/json'});
                                var url = window.URL.createObjectURL(blob);
                                var a = document.createElement('a');
                                a.href = url;
                                a.download = response.data.filename || ('capsule_' + capsuleId + '.json');
                                document.body.appendChild(a);
                                a.click();
                                document.body.removeChild(a);
                                window.URL.revokeObjectURL(url);
                                
                                console.log('DEBUG: Export download initiated');
                            } catch (e) {
                                alert('Error creating download: ' + e.message);
                                console.error('DEBUG: Export error:', e);
                            }
                        } else {
                            var errorMsg = 'Failed to export capsule';
                            if (response.data && response.data.message) {
                                errorMsg = response.data.message;
                            }
                            alert('Error: ' + errorMsg);
                            console.error('DEBUG: exportCapsule error:', response);
                        }
                    },
                    error: function(xhr, status, error) {
                        alert('AJAX Error: ' + error);
                        console.error('DEBUG: exportCapsule AJAX error:', xhr, status, error);
                    }
                });
            }
            
            // Search functionality
            $('#aiox-capsules-search').on('input', function() {
                var searchTerm = $(this).val();
                var typeFilter = $('#aiox-capsules-filter').val();
                searchCapsules(searchTerm, typeFilter);
            });
            
            $('#aiox-capsules-filter').on('change', function() {
                var searchTerm = $('#aiox-capsules-search').val();
                var typeFilter = $(this).val();
                searchCapsules(searchTerm, typeFilter);
            });
            
            function searchCapsules(searchTerm, typeFilter) {
                // Simple client-side filtering for now
                $('.aiox-capsule-row').each(function() {
                    var row = $(this);
                    var title = row.find('.aiox-capsule-title strong').text().toLowerCase();
                    var type = row.data('type');
                    
                    var matchesSearch = !searchTerm || title.includes(searchTerm.toLowerCase());
                    var matchesType = !typeFilter || type === typeFilter;
                    
                    if (matchesSearch && matchesType) {
                        row.show();
                    } else {
                        row.hide();
                    }
                });
            }
            
            // Bulk actions
            $('#aiox-select-all-capsules').on('change', function() {
                $('.aiox-capsule-checkbox').prop('checked', this.checked);
            });
            
            $('#aiox-apply-bulk-action').on('click', function(e) {
                e.preventDefault();
                var action = $('#aiox-bulk-action').val();
                var selectedCapsules = $('.aiox-capsule-checkbox:checked').map(function() {
                    return this.value;
                }).get();
                
                if (!action) {
                    alert('<?php _e('Please select an action.', 'aiox-publisher-suite-pro'); ?>');
                    return;
                }
                
                if (selectedCapsules.length === 0) {
                    alert('<?php _e('Please select at least one capsule.', 'aiox-publisher-suite-pro'); ?>');
                    return;
                }
                
                if (confirm('<?php _e('Are you sure you want to perform this bulk action?', 'aiox-publisher-suite-pro'); ?>')) {
                    performBulkAction(action, selectedCapsules);
                }
            });
            
            function performBulkAction(action, capsuleIds) {
                $.ajax({
                    url: ajaxurl,
                    type: 'POST',
                    data: {
                        action: 'aiox_bulk_capsule_action',
                        nonce: nonce,
                        bulk_action: action, // Fixed parameter name
                        capsule_ids: capsuleIds
                    },
                    success: function(response) {
                        if (response.success) {
                            alert(response.data.message);
                            location.reload();
                        } else {
                            alert('Error: ' + (response.data.message || 'Unknown error occurred'));
                        }
                    },
                    error: function(xhr, status, error) {
                        alert('AJAX Error: ' + error);
                    }
                });
            }
            
            // Settings form
            $('#aiox-save-capsules-settings').on('click', function(e) {
                e.preventDefault();
                
                // Get form values individually
                var form = $('#aiox-capsules-settings-form');
                var postData = {
                    action: 'aiox_save_capsules_settings',
                    nonce: nonce,
                    auto_generate: form.find('input[name="auto_generate"]').is(':checked') ? 1 : 0,
                    default_status: form.find('select[name="default_status"]').val(),
                    ai_enhancement: form.find('input[name="ai_enhancement"]').is(':checked') ? 1 : 0,
                    export_format: form.find('select[name="export_format"]').val(),
                    retention_days: form.find('input[name="retention_days"]').val()
                };
                
                $.post(ajaxurl, postData, function(response) {
                    if (response.success) {
                        alert('<?php _e('Settings saved successfully!', 'aiox-publisher-suite-pro'); ?>');
                    } else {
                        alert(response.data || 'Error saving settings');
                    }
                }).fail(function() {
                    alert('AJAX request failed');
                });
            });
            
            // Bulk generation functions
            function generateAllMissingCapsules() {
                if (!confirm('This will generate capsules for all posts that don\'t have them. Continue?')) {
                    return;
                }
                
                var button = $('#aiox-generate-all-capsules');
                var originalText = button.text();
                button.prop('disabled', true).text('Generating...');
                
                $.post(ajaxurl, {
                    action: 'aiox_generate_all_missing_capsules',
                    nonce: nonce
                }, function(response) {
                    if (response.success) {
                        alert(response.data.message);
                        if (response.data.generated > 0) {
                            location.reload();
                        }
                    } else {
                        alert('Error: ' + response.data);
                    }
                }).fail(function() {
                    alert('AJAX request failed');
                }).always(function() {
                    button.prop('disabled', false).text(originalText);
                });
            }
            
            function loadPostsForInlineSelection() {
                var contentDiv = $('#aiox-post-selection-content');
                contentDiv.html('<p><?php _e('Loading posts...', 'aiox-publisher-suite-pro'); ?></p>');
                // Get posts without capsules
                $.post(ajaxurl, {
                    action: 'aiox_get_posts_without_capsules',
                    nonce: nonce
                }, function(response) {
                    if (response.success) {
                        if (response.data.length === 0) {
                            contentDiv.html('<p><?php _e('All posts already have capsules!', 'aiox-publisher-suite-pro'); ?></p>');
                            return;
                        }
                        
                        var html = '<div style="max-height: 400px; overflow-y: auto; border: 1px solid #ddd; border-radius: 4px; padding: 10px; margin: 15px 0;">';
                        
                        response.data.forEach(function(post) {
                            html += '<label style="display: block; padding: 8px 10px; border-bottom: 1px solid #eee; cursor: pointer; margin: 0; word-wrap: break-word; box-sizing: border-box;">';
                            html += '<input type="checkbox" class="aiox-post-checkbox" style="margin-right: 10px; vertical-align: top;" value="' + post.id + '">';
                            html += '<span style="font-weight: bold; word-wrap: break-word;">' + post.title + '</span><br>';
                            html += '<span style="color: #666; font-size: 12px; word-wrap: break-word;">(' + post.type + ' - ' + post.date + ')</span>';
                            html += '</label>';
                        });
                        
                        html += '</div>';
                        html += '<div style="text-align: center; padding: 15px 0; border-top: 1px solid #ddd;">';
                        html += '<button type="button" class="button button-primary" id="aiox-generate-selected-inline" style="margin: 0 5px;">Generate Selected</button>';
                        html += '<button type="button" class="button" id="aiox-select-all-posts-inline" style="margin: 0 5px;">Select All</button>';
                        html += '<button type="button" class="button" id="aiox-clear-selection-inline" style="margin: 0 5px;">Clear All</button>';
                        html += '</div>';
                        
                        contentDiv.html(html);
                        
                        // Handle inline actions
                        $('#aiox-select-all-posts-inline').on('click', function() {
                            $('.aiox-post-checkbox').prop('checked', true);
                        });
                        
                        $('#aiox-clear-selection-inline').on('click', function() {
                            $('.aiox-post-checkbox').prop('checked', false);
                        });
                        
                        $('#aiox-generate-selected-inline').on('click', function() {
                            var selectedPosts = [];
                            $('.aiox-post-checkbox:checked').each(function() {
                                selectedPosts.push($(this).val());
                            });
                            
                            if (selectedPosts.length === 0) {
                                alert('Please select at least one post');
                                return;
                            }
                            
                            generateCapsulesForSelectedPosts(selectedPosts);
                            $('#aiox-post-selection-section').hide();
                            $('#aiox-toggle-post-selection').text('<?php _e('Select Posts for Capsules', 'aiox-publisher-suite-pro'); ?>');
                        });
                        
                    } else {
                        contentDiv.html('<p style="color: red;"><?php _e('Error loading posts: ', 'aiox-publisher-suite-pro'); ?>' + (response.data || 'Unknown error') + '</p>');
                    }
                }).fail(function() {
                    contentDiv.html('<p style="color: red;"><?php _e('Failed to load posts. Please try again.', 'aiox-publisher-suite-pro'); ?></p>');
                });
            }
            
            function generateCapsulesForSelectedPosts(postIds) {
                var button = $('#aiox-toggle-post-selection');
                var originalText = button.text();
                button.prop('disabled', true).text('Generating...');
                
                $.post(ajaxurl, {
                    action: 'aiox_generate_capsules_for_selected_posts',
                    nonce: nonce,
                    post_ids: postIds
                }, function(response) {
                    if (response.success) {
                        alert(response.data.message);
                        if (response.data.generated > 0) {
                            location.reload();
                        }
                    } else {
                        alert('Error: ' + response.data);
                    }
                }).fail(function() {
                    alert('AJAX request failed');
                }).always(function() {
                    button.prop('disabled', false).text(originalText);
                });
            }
        });
        </script>
        <?php
    }
    
    /**
 * Render CSS styles
 */
private function render_styles() {
    ?>
    <style type="text/css">
    /* Modern Dark Theme for AIOX Capsules */
    .wrap.aiox-capsules-page {
        background: radial-gradient(circle at top left, #1b1b3a, #0f0f1f);
        padding: 30px;
        border-radius: 12px;
        color: #e0e0ff;
        font-family: "Segoe UI", Tahoma, Geneva, Verdana, sans-serif;
        margin: 20px 20px 20px 0;
        min-height: 100vh;
    }
        
        /* Modern Dark Theme for AIOX Capsules */
.wrap.aiox-capsules-page {
    background: radial-gradient(circle at top left, #1b1b3a, #0f0f1f);
    padding: 30px;
    border-radius: 12px;
    color: #e0e0ff;
    font-family: "Segoe UI", Tahoma, Geneva, Verdana, sans-serif;
    margin: 20px 20px 20px 0;
    min-height: 100vh;
}

/* Page Title */
.aiox-capsules-page > h1 {
    font-size: 28px;
    background: linear-gradient(90deg, #6a11cb, #2575fc);
    -webkit-background-clip: text;
    -webkit-text-fill-color: transparent;
    background-clip: text;
    margin-bottom: 20px;
    font-weight: 600;
}

/* Tab Navigation */
.nav-tab-wrapper {
    background: linear-gradient(145deg, #15152e 0%, #1a1a35 100%);
    border: 1px solid #2e2e5f;
    border-radius: 12px 12px 0 0;
    padding: 0;
    margin-bottom: 0;
}

.nav-tab {
    background: transparent;
    border: none;
    color: #9fa9ff;
    padding: 15px 25px;
    font-weight: 500;
    transition: all 0.3s ease;
}

.nav-tab:hover {
    background: rgba(37, 117, 252, 0.1);
    color: #e0e0ff;
}

.nav-tab-active {
    background: linear-gradient(90deg, #6a11cb, #2575fc);
    color: #fff !important;
    border-radius: 8px 8px 0 0;
}

/* Tab Content */
.aiox-tab-content {
    background: linear-gradient(145deg, #15152e 0%, #1a1a35 100%);
    border: 1px solid #2e2e5f;
    border-top: none;
    border-radius: 0 0 12px 12px;
    padding: 25px;
    min-height: 500px;
}

/* Stats Grid - Overview Tab */
.aiox-stats-grid {
    display: grid;
    grid-template-columns: repeat(auto-fit, minmax(250px, 1fr));
    gap: 15px;
    margin-bottom: 30px;
}

.aiox-stat-card {
    background: linear-gradient(135deg, #1e1e3f 0%, #252550 100%);
    border: 1px solid #3c3c7a;
    border-radius: 10px;
    padding: 20px;
    display: flex;
    align-items: center;
    gap: 15px;
    transition: all 0.3s ease;
}

.aiox-stat-card:hover {
    transform: translateY(-3px);
    box-shadow: 0 6px 20px rgba(37, 117, 252, 0.3);
    border-color: #2575fc;
}

.aiox-stat-icon {
    width: 50px;
    height: 50px;
    border-radius: 10px;
    background: linear-gradient(135deg, #6a11cb, #2575fc);
    display: flex;
    align-items: center;
    justify-content: center;
    font-size: 24px;
}

.aiox-stat-content h3 {
    margin: 0;
    font-size: 28px;
    font-weight: 700;
    color: #fff;
}

.aiox-stat-content p {
    margin: 5px 0 0 0;
    color: #9fa9ff;
    font-size: 13px;
    text-transform: uppercase;
    letter-spacing: 0.5px;
}

/* Quick Actions Section */
.aiox-quick-actions h2 {
    color: #e0e0ff;
    font-size: 20px;
    margin-bottom: 20px;
    padding-bottom: 15px;
    border-bottom: 1px solid #2e2e5f;
}

.aiox-action-grid {
    display: grid;
    grid-template-columns: repeat(auto-fit, minmax(280px, 1fr));
    gap: 20px;
}

.aiox-action-card {
    background: #1e1e3f;
    border: 1px solid #3c3c7a;
    border-radius: 10px;
    padding: 25px;
    text-align: center;
    transition: all 0.3s ease;
}

.aiox-action-card:hover {
    transform: translateY(-3px);
    box-shadow: 0 6px 20px rgba(37, 117, 252, 0.3);
    border-color: #2575fc;
}

.aiox-action-icon {
    font-size: 48px;
    margin-bottom: 15px;
}

.aiox-action-card h3 {
    margin: 10px 0;
    color: #e0e0ff;
    font-size: 16px;
    font-weight: 600;
}

.aiox-action-card p {
    color: #9fa9ff;
    margin-bottom: 20px;
    font-size: 14px;
    line-height: 1.5;
}

.aiox-action-card .button-primary {
    background: linear-gradient(90deg, #6a11cb, #2575fc);
    border: none;
    color: #fff;
    padding: 10px 20px;
    border-radius: 6px;
    font-weight: 600;
    transition: all 0.3s ease;
}

.aiox-action-card .button-primary:hover {
    background: linear-gradient(90deg, #7a21db, #3585fc);
    transform: translateY(-2px);
}

/* Manage Tab Header */
.aiox-capsules-header {
    display: flex;
    justify-content: space-between;
    align-items: center;
    margin-bottom: 25px;
    flex-wrap: wrap;
    gap: 15px;
    padding-bottom: 20px;
    border-bottom: 1px solid #2e2e5f;
}

.aiox-capsules-header h2 {
    color: #e0e0ff;
    margin: 0;
    font-size: 20px;
}

.aiox-capsules-actions {
    display: flex;
    gap: 10px;
    align-items: center;
    flex-wrap: wrap;
}

.aiox-capsules-actions input[type="search"],
.aiox-capsules-actions select {
    padding: 8px 12px;
    border: 1px solid #3c3c7a;
    border-radius: 6px;
    background: #1e1e3f;
    color: #e0e0ff;
}

.aiox-capsules-actions input[type="search"]:focus,
.aiox-capsules-actions select:focus {
    outline: none;
    border-color: #2575fc;
    box-shadow: 0 0 0 2px rgba(37, 117, 252, 0.2);
}

.aiox-capsules-actions .button {
    background: #1e1e3f;
    border: 1px solid #3c3c7a;
    color: #cfdcff;
    padding: 8px 16px;
    border-radius: 6px;
    transition: all 0.3s ease;
}

.aiox-capsules-actions .button:hover {
    background: #2e2e5f;
    border-color: #2575fc;
    color: #e0e0ff;
}

.aiox-capsules-actions .button-primary {
    background: linear-gradient(90deg, #6a11cb, #2575fc);
    border: none;
    color: #fff;
}

.aiox-capsules-actions .button-primary:hover {
    background: linear-gradient(90deg, #7a21db, #3585fc);
}

/* Capsules Table */
.aiox-capsules-table-wrapper {
    background: #1e1e3f;
    border-radius: 10px;
    overflow: hidden;
    margin-bottom: 20px;
}

.wp-list-table {
    background: transparent;
    border: none;
}

.wp-list-table thead tr {
    background: #252550;
}

.wp-list-table thead th {
    color: #9fa9ff;
    font-weight: 600;
    text-transform: uppercase;
    font-size: 12px;
    letter-spacing: 0.5px;
    padding: 15px;
    border: none;
}

.wp-list-table tbody tr {
    background: #1e1e3f;
    border-bottom: 1px solid #2e2e5f;
    transition: all 0.3s ease;
}

.wp-list-table tbody tr:hover {
    background: #252550;
}

.wp-list-table tbody td {
    color: #cfdcff;
    padding: 15px;
    border: none;
}

.aiox-capsule-title strong {
    color: #fff;
    font-weight: 600;
}

.aiox-capsule-description {
    font-size: 13px;
    color: #9fa9ff;
    margin-top: 5px;
}

/* Capsule Type Badges */
.aiox-capsule-type {
    display: inline-block;
    padding: 4px 12px;
    border-radius: 12px;
    font-size: 11px;
    font-weight: 600;
    text-transform: uppercase;
    letter-spacing: 0.5px;
}

.aiox-type-post {
    background: rgba(33, 150, 243, 0.2);
    color: #42a5f5;
}

.aiox-type-collection {
    background: rgba(156, 39, 176, 0.2);
    color: #ba68c8;
}

.aiox-type-author {
    background: rgba(76, 175, 80, 0.2);
    color: #81c784;
}

.aiox-type-topic {
    background: rgba(255, 152, 0, 0.2);
    color: #ffb74d;
}

/* Status Badges */
.aiox-status {
    display: inline-block;
    padding: 4px 12px;
    border-radius: 12px;
    font-size: 11px;
    font-weight: 600;
    text-transform: uppercase;
    letter-spacing: 0.5px;
}

.aiox-status-active {
    background: rgba(76, 175, 80, 0.2);
    color: #81c784;
}

.aiox-status-draft {
    background: rgba(255, 152, 0, 0.2);
    color: #ffb74d;
}

.aiox-status-archived {
    background: rgba(158, 158, 158, 0.2);
    color: #bdbdbd;
}

.aiox-status-error {
    background: rgba(244, 67, 54, 0.2);
    color: #e57373;
}

/* Action Buttons */
.aiox-capsule-actions .button {
    background: #252550;
    border: 1px solid #3c3c7a;
    color: #cfdcff;
    padding: 6px 12px;
    border-radius: 6px;
    font-size: 12px;
    transition: all 0.3s ease;
    margin-right: 5px;
}

.aiox-capsule-actions .button:hover {
    background: linear-gradient(90deg, #6a11cb, #2575fc);
    border-color: #2575fc;
    color: #fff;
    transform: translateY(-1px);
}

/* Empty State */
.aiox-empty-state {
    text-align: center;
    padding: 60px 20px;
}

.aiox-empty-icon {
    font-size: 64px;
    margin-bottom: 20px;
    opacity: 0.5;
}

.aiox-empty-state h3 {
    margin: 0 0 10px 0;
    color: #e0e0ff;
    font-size: 20px;
}

.aiox-empty-state p {
    margin: 0 0 25px 0;
    color: #9fa9ff;
    font-size: 14px;
}

/* Bulk Actions */
.aiox-bulk-actions {
    margin-top: 20px;
    padding-top: 20px;
    border-top: 1px solid #2e2e5f;
    display: flex;
    gap: 10px;
    align-items: center;
}

.aiox-bulk-actions select {
    padding: 8px 12px;
    border: 1px solid #3c3c7a;
    border-radius: 6px;
    background: #1e1e3f;
    color: #e0e0ff;
}

.aiox-bulk-actions .button {
    background: #1e1e3f;
    border: 1px solid #3c3c7a;
    color: #cfdcff;
    padding: 8px 16px;
    border-radius: 6px;
}

.aiox-bulk-actions .button:hover {
    background: #2e2e5f;
    border-color: #2575fc;
}

/* Create Tab */
.aiox-create-header {
    text-align: center;
    margin-bottom: 40px;
    padding-bottom: 25px;
    border-bottom: 1px solid #2e2e5f;
}

.aiox-create-header h2 {
    color: #e0e0ff;
    margin-bottom: 10px;
    font-size: 24px;
}

.aiox-create-header p {
    color: #9fa9ff;
    font-size: 15px;
}

.aiox-create-types {
    display: grid;
    grid-template-columns: repeat(auto-fit, minmax(280px, 1fr));
    gap: 20px;
}

.aiox-type-card {
    background: #1e1e3f;
    border: 2px solid #3c3c7a;
    border-radius: 10px;
    padding: 30px;
    text-align: center;
    transition: all 0.3s ease;
    cursor: pointer;
}

.aiox-type-card:hover {
    border-color: #2575fc;
    transform: translateY(-5px);
    box-shadow: 0 8px 25px rgba(37, 117, 252, 0.3);
}

.aiox-type-icon {
    font-size: 56px;
    margin-bottom: 20px;
}

.aiox-type-card h3 {
    margin: 0 0 15px 0;
    color: #e0e0ff;
    font-size: 18px;
}

.aiox-type-card p {
    color: #9fa9ff;
    margin-bottom: 25px;
    line-height: 1.6;
    font-size: 14px;
}

/* Modal Styles */
.aiox-modal {
    position: fixed;
    top: 0;
    left: 0;
    width: 100%;
    height: 100%;
    background: rgba(0, 0, 0, 0.7);
    z-index: 100000;
    display: flex;
    align-items: center;
    justify-content: center;
}

.aiox-modal-content {
    background: linear-gradient(145deg, #15152e 0%, #1a1a35 100%);
    border: 1px solid #2e2e5f;
    border-radius: 12px;
    max-width: 600px;
    width: 90%;
    max-height: 80vh;
    overflow-y: auto;
    box-shadow: 0 10px 40px rgba(0, 0, 0, 0.5);
}

.aiox-modal-large {
    max-width: 800px;
}

.aiox-modal-header {
    padding: 25px;
    border-bottom: 1px solid #2e2e5f;
    display: flex;
    justify-content: space-between;
    align-items: center;
}

.aiox-modal-header h3 {
    margin: 0;
    color: #e0e0ff;
    font-size: 20px;
}

.aiox-modal-close {
    background: none;
    border: none;
    font-size: 28px;
    cursor: pointer;
    color: #9fa9ff;
    transition: all 0.3s ease;
    line-height: 1;
    padding: 0;
    width: 28px;
    height: 28px;
}

.aiox-modal-close:hover {
    color: #e0e0ff;
    transform: rotate(90deg);
}

.aiox-modal-body {
    padding: 25px;
    color: #cfdcff;
}

.aiox-modal-footer {
    padding: 20px 25px;
    border-top: 1px solid #2e2e5f;
    text-align: right;
    display: flex;
    gap: 10px;
    justify-content: flex-end;
}

.aiox-modal-footer .button {
    background: #1e1e3f;
    border: 1px solid #3c3c7a;
    color: #cfdcff;
}

.aiox-modal-footer .button:hover {
    background: #2e2e5f;
    border-color: #2575fc;
}

.aiox-modal-footer .button-primary {
    background: linear-gradient(90deg, #6a11cb, #2575fc);
    border: none;
    color: #fff;
}

.aiox-modal-footer .button-primary:hover {
    background: linear-gradient(90deg, #7a21db, #3585fc);
}

/* Post/Author Selection */
.aiox-post-selection,
.aiox-author-selection {
    max-height: 400px;
    overflow-y: auto;
}

.aiox-posts-list,
.aiox-authors-list {
    display: flex;
    flex-direction: column;
    gap: 10px;
}

.aiox-post-item,
.aiox-author-item {
    background: #1e1e3f;
    border: 1px solid #3c3c7a;
    border-radius: 8px;
    padding: 0;
    transition: all 0.3s ease;
}

.aiox-post-item:hover,
.aiox-author-item:hover {
    border-color: #2575fc;
    box-shadow: 0 2px 8px rgba(37, 117, 252, 0.2);
}

.aiox-post-item label,
.aiox-author-item label {
    display: flex;
    align-items: flex-start;
    padding: 15px;
    margin: 0;
    cursor: pointer;
    gap: 12px;
}

.aiox-post-info h4,
.aiox-author-details h4 {
    margin: 0 0 8px 0;
    font-size: 14px;
    font-weight: 600;
    color: #e0e0ff;
}

.aiox-post-meta,
.aiox-author-meta {
    display: flex;
    gap: 15px;
    margin-bottom: 8px;
    font-size: 12px;
    color: #9fa9ff;
}

/* Topic Selection */
.aiox-topic-selection label {
    display: block;
    margin-bottom: 8px;
    font-weight: 600;
    color: #e0e0ff;
}

.aiox-topic-selection input,
.aiox-topic-selection textarea {
    width: 100%;
    padding: 10px 12px;
    border: 1px solid #3c3c7a;
    border-radius: 6px;
    background: #1e1e3f;
    color: #e0e0ff;
    margin-bottom: 15px;
}

.aiox-topic-selection input:focus,
.aiox-topic-selection textarea:focus {
    outline: none;
    border-color: #2575fc;
    box-shadow: 0 0 0 2px rgba(37, 117, 252, 0.2);
}

.aiox-topic-selection textarea {
    height: 80px;
    resize: vertical;
}

/* Capsule Details */
.aiox-capsule-details h4 {
    margin-top: 0;
    color: #e0e0ff;
    border-bottom: 1px solid #2e2e5f;
    padding-bottom: 15px;
    font-size: 20px;
}

.aiox-details-section {
    margin-bottom: 25px;
    padding: 20px;
    background: #1e1e3f;
    border-radius: 8px;
    border-left: 4px solid #2575fc;
}

.aiox-details-section h5 {
    margin: 0 0 15px 0;
    color: #e0e0ff;
    font-size: 16px;
    font-weight: 600;
}

.aiox-details-section p {
    margin: 10px 0;
    line-height: 1.6;
    color: #cfdcff;
}

.aiox-details-section strong {
    color: #9fa9ff;
    font-weight: 600;
}

/* Settings Tab */
.aiox-settings-tab .form-table {
    background: transparent;
}

.aiox-settings-tab .form-table th {
    color: #9fa9ff;
    font-weight: 600;
    padding-left: 0;
}

.aiox-settings-tab .form-table td {
    color: #cfdcff;
}

.aiox-settings-tab input[type="checkbox"] {
    margin-right: 8px;
}

.aiox-settings-tab select,
.aiox-settings-tab input[type="number"] {
    padding: 8px 12px;
    border: 1px solid #3c3c7a;
    border-radius: 6px;
    background: #1e1e3f;
    color: #e0e0ff;
}

.aiox-settings-tab .description {
    font-style: italic;
    color: #9fa9ff;
    font-size: 13px;
}

/* Loading State */
.aiox-loading {
    text-align: center;
    padding: 50px;
    color: #9fa9ff;
}

.aiox-loading:before {
    content: "⏳";
    font-size: 48px;
    display: block;
    margin-bottom: 15px;
    animation: pulse 1.5s infinite;
}

@keyframes pulse {
    0%, 100% { opacity: 1; }
    50% { opacity: 0.5; }
}

/* Inline Post Selection Section */
#aiox-post-selection-section {
    background: #1e1e3f;
    border: 1px solid #3c3c7a;
    border-radius: 10px;
    padding: 25px;
    margin: 20px 0;
}

#aiox-post-selection-section h3 {
    color: #e0e0ff;
    margin-top: 0;
    margin-bottom: 20px;
}

/* Responsive Design */
@media (max-width: 782px) {
    .wrap.aiox-capsules-page {
        padding: 20px;
        margin: 10px 10px 10px 0;
    }
    
    .aiox-capsules-header {
        flex-direction: column;
        align-items: stretch;
    }
    
    .aiox-capsules-actions {
        flex-direction: column;
        align-items: stretch;
    }
    
    .aiox-capsules-actions input,
    .aiox-capsules-actions select,
    .aiox-capsules-actions .button {
        width: 100%;
    }
    
    .aiox-stats-grid,
    .aiox-action-grid,
    .aiox-create-types {
        grid-template-columns: 1fr;
    }
    
    .aiox-capsule-actions .button {
        display: block;
        width: 100%;
        margin: 5px 0;
    }
}

/* Scrollbar Styling */
.aiox-tab-content::-webkit-scrollbar,
.aiox-modal-content::-webkit-scrollbar,
.aiox-post-selection::-webkit-scrollbar {
    width: 8px;
}

.aiox-tab-content::-webkit-scrollbar-track,
.aiox-modal-content::-webkit-scrollbar-track,
.aiox-post-selection::-webkit-scrollbar-track {
    background: #1e1e3f;
}

.aiox-tab-content::-webkit-scrollbar-thumb,
.aiox-modal-content::-webkit-scrollbar-thumb,
.aiox-post-selection::-webkit-scrollbar-thumb {
    background: #2575fc;
    border-radius: 4px;
}

.aiox-tab-content::-webkit-scrollbar-thumb:hover,
.aiox-modal-content::-webkit-scrollbar-thumb:hover,
.aiox-post-selection::-webkit-scrollbar-thumb:hover {
    background: #3585fc;
}
        </style>
        
        <!-- Enhanced Modal Styles -->
        <link rel="stylesheet" href="<?php echo plugin_dir_url(__FILE__) . '../assets/css/capsules-admin-modal-styles.css'; ?>" />
        <link rel="stylesheet" href="<?php echo plugin_dir_url(__FILE__) . '../assets/css/capsules-admin-enhanced.css'; ?>" />
        <link rel="stylesheet" href="<?php echo plugin_dir_url(__FILE__) . '../assets/css/capsules-edit-form.css'; ?>" />
        <?php
    }
    
    /**
     * AJAX handler for generating capsule for a specific post
     */
    public function ajax_generate_capsule_for_post() {
        if (!wp_verify_nonce($_POST['nonce'], 'aiox_capsules_nonce')) {
            wp_send_json_error(array('message' => 'Security check failed'));
            return;
        }
        
        if (!current_user_can('edit_posts')) {
            wp_send_json_error(array('message' => 'Insufficient permissions'));
            return;
        }
        
        $post_id = intval($_POST['post_id']);
        
        if (!$post_id) {
            wp_send_json_error(array('message' => 'Invalid post ID'));
            return;
        }
        
        try {
            if (class_exists('AIOX_Capsule_Manager')) {
                $manager = new AIOX_Capsule_Manager();
                
                // Use the regenerate method instead of create to replace existing capsules
                if (method_exists($manager, 'regenerate_capsule_from_post')) {
                    $result = $manager->regenerate_capsule_from_post($post_id);
                } else {
                    // Fallback to create method if regenerate method doesn't exist
                    $result = $manager->create_capsule_from_post($post_id);
                }
                
                if ($result) {
                    wp_send_json_success(array(
                        'message' => 'Capsule regenerated successfully!',
                        'reload' => true
                    ));
                } else {
                    wp_send_json_error(array('message' => 'Failed to regenerate capsule'));
                }
            } else {
                wp_send_json_error(array('message' => 'Capsule manager not available'));
            }
        } catch (Exception $e) {
            wp_send_json_error(array('message' => 'Error: ' . $e->getMessage()));
        }
    }
    
    private function get_post_id_from_capsule($capsule) {
        try {
            // Handle both object and array inputs
            if (is_object($capsule)) {
                // If it's a capsule object, get metadata using methods
                if (method_exists($capsule, 'get_metadata')) {
                    $metadata = $capsule->get_metadata();
                    
                    // Check for source_posts array (primary method for objects)
                    if (isset($metadata['source_posts']) && is_array($metadata['source_posts']) && !empty($metadata['source_posts'])) {
                        $first_post_id = $metadata['source_posts'][0];
                        if (is_numeric($first_post_id)) {
                            return intval($first_post_id);
                        }
                    }
                    
                    // Check for post_id in metadata (direct field)
                    if (isset($metadata['post_id']) && is_numeric($metadata['post_id'])) {
                        return intval($metadata['post_id']);
                    }
                }
                
                // Try to get content and extract from there
                if (method_exists($capsule, 'get_content')) {
                    $content = $capsule->get_content();
                    if (isset($content['metadata']) && is_array($content['metadata'])) {
                        if (isset($content['metadata']['post_id']) && is_numeric($content['metadata']['post_id'])) {
                            return intval($content['metadata']['post_id']);
                        }
                    }
                }
                
                // Fallback: try to find post by title for objects
                if (method_exists($capsule, 'get_title')) {
                    $title = $capsule->get_title();
                    if (!empty($title)) {
                        // Remove " - Capsule" suffix if present
                        $search_title = str_replace(' - Capsule', '', $title);
                        
                        $post = get_page_by_title($search_title, OBJECT, array('post', 'page'));
                        if ($post) {
                            return $post->ID;
                        }
                    }
                }
                
            } else if (is_array($capsule)) {
                // Handle array input (legacy support)
                
                // Try to extract post ID from metadata
                if (isset($capsule['metadata'])) {
                    $metadata = is_string($capsule['metadata']) ? json_decode($capsule['metadata'], true) : $capsule['metadata'];
                    
                    // Handle JSON decode errors
                    if (json_last_error() !== JSON_ERROR_NONE && is_string($capsule['metadata'])) {
                        return null;
                    }
                    
                    // Check for post_id in metadata (direct field)
                    if (isset($metadata['post_id']) && is_numeric($metadata['post_id'])) {
                        return intval($metadata['post_id']);
                    }
                    
                    // Check for source_posts array (primary method)
                    if (isset($metadata['source_posts']) && is_array($metadata['source_posts']) && !empty($metadata['source_posts'])) {
                        $first_post_id = $metadata['source_posts'][0];
                        if (is_numeric($first_post_id)) {
                            return intval($first_post_id);
                        }
                    }
                }
                
                // Try to extract from content metadata (alternative location)
                if (isset($capsule['content'])) {
                    $content = is_string($capsule['content']) ? json_decode($capsule['content'], true) : $capsule['content'];
                    
                    if (isset($content['metadata']) && is_array($content['metadata'])) {
                        if (isset($content['metadata']['post_id']) && is_numeric($content['metadata']['post_id'])) {
                            return intval($content['metadata']['post_id']);
                        }
                    }
                }
                
                // Fallback: try to find post by title (less reliable)
                if (isset($capsule['title']) && !empty($capsule['title'])) {
                    // Remove " - Capsule" suffix if present
                    $search_title = str_replace(' - Capsule', '', $capsule['title']);
                    
                    $post = get_page_by_title($search_title, OBJECT, array('post', 'page'));
                    if ($post) {
                        return $post->ID;
                    }
                }
            }
            
        } catch (Exception $e) {
        }
        
        // Log when no post ID can be found
        $capsule_id = 'unknown';
        if (is_object($capsule) && method_exists($capsule, 'get_id')) {
            $capsule_id = $capsule->get_id();
        } else if (is_array($capsule) && isset($capsule['id'])) {
            $capsule_id = $capsule['id'];
        }
        
        return null;
    }
}
