<?php
/**
 * AIOX Processing Comparison Tool
 * 
 * Compare basic extraction vs AI enhanced results
 * STYLED VERSION: Modern dark theme applied
 */

if (!defined('ABSPATH')) {
    exit;
}

class AIOX_Comparison {
    
    /**
     * Constructor
     */
    public function __construct() {
        add_action('wp_ajax_aiox_compare_processing', array($this, 'ajax_compare_processing'));
    }
    
    /**
     * Render comparison page
     */
    public function render_comparison_page() {
        ?>
        
        <div class="wrap aiox-comparison-wrap">
            <?php aiox_render_page_header(__('AIOX Processing Comparison', 'aiox-publisher-suite-pro'), __('Compare Basic vs AI Enhanced Processing', 'aiox-publisher-suite-pro')); ?>
            
            <div class="aiox-comparison-container">
                <div class="aiox-comparison-controls">
                    <h2><?php _e('Compare Processing Methods', 'aiox-publisher-suite-pro'); ?></h2>
                    <p><?php _e('Select a post to compare basic extraction vs AI enhanced processing results.', 'aiox-publisher-suite-pro'); ?></p>
                    
                    <table class="form-table">
                        <tr>
                            <th scope="row"><?php _e('Select Post', 'aiox-publisher-suite-pro'); ?></th>
                            <td>
                                <select id="comparison-post-select" class="regular-text">
                                    <option value=""><?php _e('Choose a post...', 'aiox-publisher-suite-pro'); ?></option>
                                    <?php
                                    $posts = get_posts(array(
                                        'post_type' => array('post', 'page'),
                                        'post_status' => 'publish',
                                        'numberposts' => 50,
                                        'orderby' => 'date',
                                        'order' => 'DESC'
                                    ));
                                    
                                    foreach ($posts as $post) {
                                        $processing_method = get_post_meta($post->ID, '_aiox_processing_method', true);
                                        $method_label = $processing_method ? " ({$processing_method})" : '';
                                        echo '<option value="' . $post->ID . '">' . esc_html($post->post_title) . $method_label . '</option>';
                                    }
                                    ?>
                                </select>
                                <button type="button" id="run-comparison" class="button button-primary">
                                    <?php _e('🚀 Run Comparison', 'aiox-publisher-suite-pro'); ?>
                                </button>
                            </td>
                        </tr>
                    </table>
                </div>
                
                <div id="comparison-results" style="display: none;">
                    <div class="aiox-comparison-grid">
                        <div class="aiox-comparison-column">
                            <h3><?php _e('📋 Basic Extraction', 'aiox-publisher-suite-pro'); ?></h3>
                            <div id="basic-results" class="aiox-results-container">
                                <!-- Basic results will be loaded here -->
                            </div>
                        </div>
                        
                        <div class="aiox-comparison-column">
                            <h3><?php _e('🤖 AI Enhanced', 'aiox-publisher-suite-pro'); ?></h3>
                            <div id="ai-results" class="aiox-results-container">
                                <!-- AI results will be loaded here -->
                            </div>
                        </div>
                    </div>
                </div>
                
                <div id="comparison-loading" style="display: none;">
                    <p><?php _e('⚙️ Processing comparison... This may take a moment.', 'aiox-publisher-suite-pro'); ?></p>
                </div>
            </div>
        </div>
        
        <script type="text/javascript">
        jQuery(document).ready(function($) {
            $('#run-comparison').on('click', function() {
                var postId = $('#comparison-post-select').val();
                
                if (!postId) {
                    alert('Please select a post first.');
                    return;
                }
                
                var button = $(this);
                button.prop('disabled', true).text('Processing...');
                
                $('#comparison-results').hide();
                $('#comparison-loading').show();
                
                $.post(ajaxurl, {
                    action: 'aiox_compare_processing',
                    post_id: postId,
                    nonce: '<?php echo wp_create_nonce('aiox_compare_processing'); ?>'
                }, function(response) {
                    $('#comparison-loading').hide();
                    button.prop('disabled', false).html('🚀 Run Comparison');
                    
                    if (response.success) {
                        $('#basic-results').html(response.data.basic_html);
                        $('#ai-results').html(response.data.ai_html);
                        $('#comparison-results').show();
                    } else {
                        alert('Error: ' + response.data.message);
                    }
                }).fail(function() {
                    $('#comparison-loading').hide();
                    button.prop('disabled', false).html('🚀 Run Comparison');
                    alert('Request failed. Please try again.');
                });
            });
        });
        </script>
        <?php
    }
    
    /**
     * AJAX: Compare processing methods
     */
    public function ajax_compare_processing() {
        // Add debugging
        
        // Check nonce
        if (!isset($_POST['nonce']) || !wp_verify_nonce($_POST['nonce'], 'aiox_compare_processing')) {
            error_log('AIOX Comparison: Nonce verification failed');
            wp_send_json_error(array('message' => __('Security check failed', 'aiox-publisher-suite-pro')));
        }
        
        if (!current_user_can('manage_options')) {
            error_log('AIOX Comparison: Insufficient permissions');
            wp_send_json_error(array('message' => __('Insufficient permissions', 'aiox-publisher-suite-pro')));
        }
        
        $post_id = intval($_POST['post_id']);
        
        if (!$post_id) {
            wp_send_json_error(array('message' => __('Invalid post ID', 'aiox-publisher-suite-pro')));
        }
        
        try {
            
            // Get both processing results
            $basic_results = $this->get_basic_processing_results($post_id);
            $ai_results = $this->get_ai_processing_results($post_id);
            
            // Generate HTML for comparison
            $basic_html = $this->format_results_html($basic_results, 'basic');
            $ai_html = $this->format_results_html($ai_results, 'ai');
            
            
            wp_send_json_success(array(
                'basic_html' => $basic_html,
                'ai_html' => $ai_html
            ));
            
        } catch (Exception $e) {
            error_log('AIOX Comparison Error: ' . $e->getMessage());
            wp_send_json_error(array('message' => $e->getMessage()));
        }
    }
    
    /**
     * Get basic processing results
     */
    private function get_basic_processing_results($post_id) {
        // Use dedicated basic extractor for comparison purposes only
        require_once AIOX_PLUGIN_DIR . 'core/class-aiox-basic-extractor-comparison.php';
        
        if (!AIOX_Basic_Extractor_Comparison::should_use_for_processing()) {
            // This confirms basic extraction is disabled for actual processing
        }
        
        $results = AIOX_Basic_Extractor_Comparison::extract_for_comparison($post_id);
        
        return $results;
    }
    
    /**
     * Get AI processing results
     */
    private function get_ai_processing_results($post_id) {
        // First try to get stored AI results
        $stored_results = get_post_meta($post_id, '_aiox_processed_data', true);
        
        if ($stored_results && isset($stored_results['processing_method']) && $stored_results['processing_method'] !== 'basic') {
            return $stored_results;
        }
        
        // If no stored AI results, process with current AI provider
        require_once AIOX_PLUGIN_DIR . 'core/class-aiox-ai-only-processor.php';
        
        $processor = new AIOX_AI_Only_Processor();
        
        // Validate that AI processing is available
        if (!$processor->is_ai_available()) {
            throw new Exception('AI provider not configured. Please set up OpenAI or Gemini in settings.');
        }
        
        return $processor->process_post_content($post_id);
    }
    
    /**
     * Format results as HTML
     */
    private function format_results_html($results, $type) {
        $html = '';
        
        // Processing method
        $method = isset($results['processing_method']) ? $results['processing_method'] : 'unknown';
        $enhancement_badge = ($type === 'ai' && $method !== 'basic') ? '<span class="aiox-enhancement-badge">✨ AI Enhanced</span>' : '';
        
        $html .= '<div class="aiox-result-section">';
        $html .= '<h4>Processing Method' . $enhancement_badge . '</h4>';
        $html .= '<div class="aiox-result-content">' . esc_html(ucfirst($method)) . '</div>';
        $html .= '</div>';
        
        // Summary
        if (isset($results['summary'])) {
            $summary = html_entity_decode($results['summary'], ENT_QUOTES | ENT_HTML5, 'UTF-8');
            $summary = html_entity_decode($summary, ENT_QUOTES | ENT_HTML5, 'UTF-8');
            $html .= '<div class="aiox-result-section">';
            $html .= '<h4>📝 Summary</h4>';
            $html .= '<div class="aiox-result-content">' . esc_html($summary) . '</div>';
            $html .= '</div>';
        }
        
        // Key Points
        if (isset($results['key_points']) && is_array($results['key_points'])) {
            $html .= '<div class="aiox-result-section">';
            $html .= '<h4>🎯 Key Points (' . count($results['key_points']) . ')</h4>';
            $html .= '<ul class="aiox-result-list">';
            foreach (array_slice($results['key_points'], 0, 5) as $point) {
                $point = html_entity_decode($point, ENT_QUOTES | ENT_HTML5, 'UTF-8');
                $point = html_entity_decode($point, ENT_QUOTES | ENT_HTML5, 'UTF-8');
                $html .= '<li>' . esc_html($point) . '</li>';
            }
            $html .= '</ul>';
            $html .= '</div>';
        }
        
        // Topics
        if (isset($results['topics']) && is_array($results['topics'])) {
            $html .= '<div class="aiox-result-section">';
            $html .= '<h4>🏷️ Topics (' . count($results['topics']) . ')</h4>';
            $html .= '<div class="aiox-result-content">' . esc_html(implode(', ', array_slice($results['topics'], 0, 10))) . '</div>';
            $html .= '</div>';
        }
        
        // Entities (AI only)
        if (isset($results['entities']) && is_array($results['entities'])) {
            $html .= '<div class="aiox-result-section">';
            $html .= '<h4>🔍 Entities</h4>';
            $html .= '<div class="aiox-result-content">';
            foreach ($results['entities'] as $type => $entities) {
                if (!empty($entities)) {
                    $html .= '<strong>' . ucfirst($type) . ':</strong> ' . esc_html(implode(', ', array_slice($entities, 0, 5))) . '<br>';
                }
            }
            $html .= '</div>';
            $html .= '</div>';
        }
        
        // Sentiment & Metadata (AI only)
        if (isset($results['sentiment']) || isset($results['content_type']) || isset($results['complexity_level'])) {
            $html .= '<div class="aiox-result-section">';
            $html .= '<h4>📊 Content Analysis</h4>';
            $html .= '<div class="aiox-result-content">';
            if (isset($results['sentiment'])) {
                $html .= '<strong>Sentiment:</strong> ' . esc_html(ucfirst($results['sentiment'])) . '<br>';
            }
            if (isset($results['content_type'])) {
                $html .= '<strong>Type:</strong> ' . esc_html(ucfirst($results['content_type'])) . '<br>';
            }
            if (isset($results['complexity_level'])) {
                $html .= '<strong>Complexity:</strong> ' . esc_html(ucfirst($results['complexity_level'])) . '<br>';
            }
            if (isset($results['reading_time_minutes'])) {
                $html .= '<strong>Reading Time:</strong> ' . esc_html($results['reading_time_minutes']) . ' minutes<br>';
            }
            $html .= '</div>';
            $html .= '</div>';
        }
        
        // Q&A Pairs
        if (isset($results['qa_pairs']) && is_array($results['qa_pairs'])) {
            $html .= '<div class="aiox-result-section">';
            $html .= '<h4>❓ Q&A Pairs (' . count($results['qa_pairs']) . ')</h4>';
            foreach (array_slice($results['qa_pairs'], 0, 3) as $qa) {
                // Properly decode HTML entities to avoid double-encoding
                $question = html_entity_decode($qa['question'], ENT_QUOTES | ENT_HTML5, 'UTF-8');
                $answer = html_entity_decode($qa['answer'], ENT_QUOTES | ENT_HTML5, 'UTF-8');
                // Decode again in case of double-encoding
                $question = html_entity_decode($question, ENT_QUOTES | ENT_HTML5, 'UTF-8');
                $answer = html_entity_decode($answer, ENT_QUOTES | ENT_HTML5, 'UTF-8');
                
                $html .= '<div class="aiox-qa-pair">';
                $html .= '<div class="aiox-qa-question">Q: ' . esc_html($question) . '</div>';
                $html .= '<div class="aiox-qa-answer">A: ' . esc_html($answer) . '</div>';
                $html .= '</div>';
            }
            $html .= '</div>';
        }
        
        return $html;
    }
}