<?php
/**
 * AIOX Dashboard Widget for WordPress Dashboard
 * 
 * Native WordPress styling - displays in the standard dashboard widget area
 */

if (!defined('ABSPATH')) {
    exit;
}

class AIOX_Dashboard_Widget {
    
    /**
     * Initialize the dashboard widget
     */
    public static function init() {
        add_action('wp_dashboard_setup', array(__CLASS__, 'register_dashboard_widget'));
    }
    
    /**
     * Register the dashboard widget
     */
    public static function register_dashboard_widget() {
        if (!current_user_can('manage_options')) {
            return;
        }
        
        wp_add_dashboard_widget(
            'aiox_dashboard_widget',
            __('AIOX AI Processing Status', 'aiox-publisher-suite-pro'),
            array(__CLASS__, 'render_widget_content')
        );
    }
    
    /**
     * Render the widget content
     */
    public static function render_widget_content() {
        require_once AIOX_PLUGIN_DIR . 'core/class-aiox-api-validator.php';
        
        // Get AI status
        $status = AIOX_API_Validator::get_dashboard_status();
        $provider = $status['provider'];
        $provider_name = ucfirst($provider);
        
        // Get performance stats
        require_once AIOX_PLUGIN_DIR . 'admin/class-aiox-performance-monitor.php';
        $perf_stats = AIOX_Performance_Monitor::get_stats();
        
        ?>
        <style>
            .aiox-widget-grid {
                display: grid;
                grid-template-columns: 1fr 1fr;
                gap: 15px;
                margin-bottom: 15px;
            }
            .aiox-widget-section h4 {
                margin: 0 0 10px 0;
                padding-bottom: 8px;
                border-bottom: 1px solid #c3c4c7;
                font-size: 13px;
                display: flex;
                align-items: center;
                gap: 5px;
            }
            .aiox-widget-section h4 .dashicons {
                color: #2271b1;
                font-size: 16px;
                width: 16px;
                height: 16px;
            }
            .aiox-widget-table {
                width: 100%;
                border-collapse: collapse;
            }
            .aiox-widget-table td {
                padding: 6px 0;
                font-size: 13px;
                border-bottom: 1px solid #f0f0f1;
            }
            .aiox-widget-table tr:last-child td {
                border-bottom: none;
            }
            .aiox-widget-table td:first-child {
                color: #646970;
            }
            .aiox-widget-table td:last-child {
                text-align: right;
                font-weight: 500;
            }
            .aiox-status-success { color: #00a32a; }
            .aiox-status-warning { color: #dba617; }
            .aiox-status-error { color: #d63638; }
            .aiox-widget-actions {
                display: flex;
                gap: 8px;
                flex-wrap: wrap;
                padding-top: 15px;
                border-top: 1px solid #c3c4c7;
            }
            .aiox-widget-actions .button {
                display: inline-flex;
                align-items: center;
                gap: 5px;
            }
            .aiox-widget-actions .button .dashicons {
                font-size: 16px;
                width: 16px;
                height: 16px;
            }
            @media (max-width: 782px) {
                .aiox-widget-grid {
                    grid-template-columns: 1fr;
                }
            }
        </style>
        
        <div class="aiox-widget-grid">
            <!-- AI Configuration -->
            <div class="aiox-widget-section">
                <h4><span class="dashicons dashicons-cloud"></span> <?php _e('AI Configuration', 'aiox-publisher-suite-pro'); ?></h4>
                <table class="aiox-widget-table">
                    <tr>
                        <td><?php _e('Provider', 'aiox-publisher-suite-pro'); ?></td>
                        <td><?php echo esc_html($provider_name); ?></td>
                    </tr>
                    <tr>
                        <td><?php _e('Status', 'aiox-publisher-suite-pro'); ?></td>
                        <td class="aiox-status-<?php echo esc_attr($status['status_class']); ?>">
                            <?php echo esc_html($status['message']); ?>
                        </td>
                    </tr>
                    <?php if (isset($status['connection_valid'])): ?>
                    <tr>
                        <td><?php _e('Connection', 'aiox-publisher-suite-pro'); ?></td>
                        <td class="aiox-status-<?php echo $status['connection_valid'] ? 'success' : 'error'; ?>">
                            <?php echo esc_html($status['connection_message']); ?>
                        </td>
                    </tr>
                    <?php endif; ?>
                </table>
            </div>
            
            <!-- Performance -->
            <div class="aiox-widget-section">
                <h4><span class="dashicons dashicons-chart-bar"></span> <?php _e('Performance', 'aiox-publisher-suite-pro'); ?></h4>
                <table class="aiox-widget-table">
                    <tr>
                        <td><?php _e('Manifest', 'aiox-publisher-suite-pro'); ?></td>
                        <td><?php echo esc_html($perf_stats['manifest_size_human']); ?></td>
                    </tr>
                    <tr>
                        <td><?php _e('Score', 'aiox-publisher-suite-pro'); ?></td>
                        <td class="aiox-status-<?php echo $perf_stats['performance_score'] >= 80 ? 'success' : ($perf_stats['performance_score'] >= 50 ? 'warning' : 'error'); ?>">
                            <?php echo esc_html($perf_stats['performance_score']); ?>/100
                        </td>
                    </tr>
                    <tr>
                        <td><?php _e('Processed', 'aiox-publisher-suite-pro'); ?></td>
                        <td><?php echo number_format($perf_stats['processed_posts']); ?>/<?php echo number_format($perf_stats['total_posts']); ?></td>
                    </tr>
                </table>
            </div>
        </div>
        
        <!-- Quick Actions -->
        <div class="aiox-widget-actions">
            <a href="<?php echo esc_url(admin_url('admin.php?page=aiox-dashboard')); ?>" class="button button-primary">
                <span class="dashicons dashicons-dashboard"></span>
                <?php _e('Dashboard', 'aiox-publisher-suite-pro'); ?>
            </a>
            <a href="<?php echo esc_url(admin_url('admin.php?page=aiox-capsules')); ?>" class="button">
                <span class="dashicons dashicons-editor-code"></span>
                <?php _e('Process Content', 'aiox-publisher-suite-pro'); ?>
            </a>
            <a href="<?php echo esc_url(admin_url('admin.php?page=aiox-settings')); ?>" class="button">
                <span class="dashicons dashicons-admin-generic"></span>
                <?php _e('Settings', 'aiox-publisher-suite-pro'); ?>
            </a>
        </div>
        <?php
    }
}
