<?php
/**
 * AIOX Diagnostics Admin Page
 * 
 * System health monitoring and file verification for AIOX Publisher Suite Pro
 */

if (!defined('ABSPATH')) {
    exit;
}

class AIOX_Diagnostics {
    
    /**
     * Initialize the diagnostics page
     */
    public function __construct() {
        add_action('wp_ajax_aiox_run_diagnostics', array($this, 'ajax_run_diagnostics'));
        add_action('wp_ajax_aiox_test_file_access', array($this, 'ajax_test_file_access'));
        add_action('wp_ajax_aiox_verify_integrity', array($this, 'ajax_verify_integrity'));
        
        // Initialize logs viewer to register its AJAX handlers
        if (class_exists('AIOX_Logs_Viewer')) {
            $this->logs_viewer = new AIOX_Logs_Viewer();
        }
        
        // Load enhanced diagnostics with error handling
        try {
            if (file_exists(AIOX_PLUGIN_DIR . 'admin/diagnostics-enhanced.php')) {
                require_once AIOX_PLUGIN_DIR . 'admin/diagnostics-enhanced.php';
                if (class_exists('AIOX_Diagnostics_Enhanced')) {
                    $this->enhanced_diagnostics = new AIOX_Diagnostics_Enhanced();
                }
            }
        } catch (Exception $e) {
            // Log error but don't break the plugin
            if (class_exists('AIOX_Logger')) {
                AIOX_Logger::error('Failed to load enhanced diagnostics: ' . $e->getMessage());
            }
        }
    }
    
    /**
     * Render the diagnostics page
     */
    public function render_diagnostics_page() {
        ?>
        <div class="wrap">
            <h1>
                <span class="dashicons dashicons-heart" style="font-size: 30px; width: 30px; height: 30px; margin-right: 10px; color: #2271b1;"></span>
                <?php _e('System Diagnostics', 'aiox-publisher-suite-pro'); ?>
            </h1>
            <p class="description"><?php _e('Monitor system health and troubleshoot issues with your AIOX installation.', 'aiox-publisher-suite-pro'); ?></p>
            
            <!-- Results Area (simple notice style) -->
            <div id="diagnostics-results"></div>
            
            <!-- Quick Actions Card -->
            <div class="card" style="max-width: 100%; padding: 15px 20px; margin-bottom: 20px;">
                <h2 style="margin-top: 0;"><?php _e('Quick Actions', 'aiox-publisher-suite-pro'); ?></h2>
                <p class="description" style="margin-bottom: 15px;"><?php _e('Run diagnostic tests to check your system health.', 'aiox-publisher-suite-pro'); ?></p>
                <p>
                    <button type="button" id="aiox-run-full-diagnostics" class="button button-primary button-large">
                        <span class="dashicons dashicons-update" style="vertical-align: middle; margin-right: 5px;"></span>
                        <?php _e('Run Full Diagnostics', 'aiox-publisher-suite-pro'); ?>
                    </button>
                    <button type="button" id="aiox-test-file-access" class="button button-secondary">
                        <span class="dashicons dashicons-admin-links" style="vertical-align: middle; margin-right: 5px;"></span>
                        <?php _e('Test File Access', 'aiox-publisher-suite-pro'); ?>
                    </button>
                    <button type="button" id="aiox-verify-integrity" class="button button-secondary">
                        <span class="dashicons dashicons-shield" style="vertical-align: middle; margin-right: 5px;"></span>
                        <?php _e('Verify Integrity', 'aiox-publisher-suite-pro'); ?>
                    </button>
                </p>
            </div>
            
            <div id="poststuff">
                <div id="post-body" class="metabox-holder columns-2">
                    <div id="post-body-content">
                        
                        <!-- System Environment -->
                        <div class="postbox">
                            <h2 class="hndle"><span class="dashicons dashicons-desktop"></span> <?php _e('System Environment', 'aiox-publisher-suite-pro'); ?></h2>
                            <div class="inside">
                                <table class="widefat striped">
                                    <tbody>
                                        <?php $this->render_system_environment(); ?>
                                    </tbody>
                                </table>
                            </div>
                        </div>
                        
                        <!-- WordPress Configuration -->
                        <div class="postbox">
                            <h2 class="hndle"><span class="dashicons dashicons-wordpress"></span> <?php _e('WordPress Configuration', 'aiox-publisher-suite-pro'); ?></h2>
                            <div class="inside">
                                <table class="widefat striped">
                                    <tbody>
                                        <?php $this->render_wordpress_config(); ?>
                                    </tbody>
                                </table>
                            </div>
                        </div>
                        
                        <!-- AIOX File System -->
                        <div class="postbox">
                            <h2 class="hndle"><span class="dashicons dashicons-media-document"></span> <?php _e('AIOX File System', 'aiox-publisher-suite-pro'); ?></h2>
                            <div class="inside">
                                <?php $this->render_file_system_status(); ?>
                            </div>
                        </div>
                        
                        <!-- System Logs Viewer -->
                        <?php
                        if (isset($this->logs_viewer)) {
                            $this->logs_viewer->render_logs_viewer();
                        }
                        ?>
                        
                    </div><!-- /post-body-content -->
                    
                    <!-- Sidebar -->
                    <div id="postbox-container-1" class="postbox-container">
                        
                        <!-- Database Status -->
                        <div class="postbox">
                            <h2 class="hndle"><span class="dashicons dashicons-database"></span> <?php _e('Database', 'aiox-publisher-suite-pro'); ?></h2>
                            <div class="inside">
                                <table class="widefat">
                                    <tbody>
                                        <?php $this->render_database_status(); ?>
                                    </tbody>
                                </table>
                            </div>
                        </div>
                        
                        <!-- Performance Metrics -->
                        <div class="postbox">
                            <h2 class="hndle"><span class="dashicons dashicons-performance"></span> <?php _e('Processing Stats', 'aiox-publisher-suite-pro'); ?></h2>
                            <div class="inside">
                                <table class="widefat">
                                    <tbody>
                                        <?php $this->render_performance_metrics(); ?>
                                    </tbody>
                                </table>
                            </div>
                        </div>
                        
                        <!-- Quick Links -->
                        <div class="postbox">
                            <h2 class="hndle"><?php _e('Quick Links', 'aiox-publisher-suite-pro'); ?></h2>
                            <div class="inside">
                                <ul style="margin: 0;">
                                    <li><a href="<?php echo admin_url('admin.php?page=aiox-settings'); ?>"><span class="dashicons dashicons-admin-settings" style="vertical-align: middle;"></span> <?php _e('Settings', 'aiox-publisher-suite-pro'); ?></a></li>
                                    <li><a href="<?php echo admin_url('admin.php?page=aiox-processing'); ?>"><span class="dashicons dashicons-update" style="vertical-align: middle;"></span> <?php _e('Processing Manager', 'aiox-publisher-suite-pro'); ?></a></li>
                                    <li><a href="<?php echo admin_url('admin.php?page=aiox-robots'); ?>"><span class="dashicons dashicons-media-text" style="vertical-align: middle;"></span> <?php _e('Robots.txt', 'aiox-publisher-suite-pro'); ?></a></li>
                                    <li><a href="<?php echo home_url('.well-known/aio.json'); ?>" target="_blank"><span class="dashicons dashicons-external" style="vertical-align: middle;"></span> <?php _e('View Manifest', 'aiox-publisher-suite-pro'); ?></a></li>
                                </ul>
                            </div>
                        </div>
                        
                    </div>
                </div><!-- /post-body -->
            </div><!-- /poststuff -->
        </div>
        
        <script>
        jQuery(document).ready(function($) {
            // Run full diagnostics
            $('#aiox-run-full-diagnostics').on('click', function() {
                var button = $(this);
                var results = $('#diagnostics-results');
                
                button.prop('disabled', true).html('<span class="dashicons dashicons-update spinning" style="vertical-align: middle; margin-right: 5px;"></span> <?php esc_js(_e('Running...', 'aiox-publisher-suite-pro')); ?>');
                results.html('<div class="notice notice-info"><p><span class="dashicons dashicons-update spinning"></span> <?php esc_js(_e('Running comprehensive system diagnostics...', 'aiox-publisher-suite-pro')); ?></p></div>');
                
                $.post(ajaxurl, {
                    action: 'aiox_run_diagnostics',
                    nonce: '<?php echo wp_create_nonce('aiox_run_diagnostics'); ?>'
                }, function(response) {
                    if (response.success) {
                        results.html(response.data.html);
                    } else {
                        results.html('<div class="notice notice-error"><p>' + response.data.message + '</p></div>');
                    }
                }).fail(function() {
                    results.html('<div class="notice notice-error"><p><?php esc_js(_e('Request failed. Please try again.', 'aiox-publisher-suite-pro')); ?></p></div>');
                }).always(function() {
                    button.prop('disabled', false).html('<span class="dashicons dashicons-update" style="vertical-align: middle; margin-right: 5px;"></span> <?php esc_js(_e('Run Full Diagnostics', 'aiox-publisher-suite-pro')); ?>');
                });
            });
            
            // Test file access
            $('#aiox-test-file-access').on('click', function() {
                var button = $(this);
                var results = $('#diagnostics-results');
                
                button.prop('disabled', true).html('<span class="dashicons dashicons-update spinning" style="vertical-align: middle; margin-right: 5px;"></span> <?php esc_js(_e('Testing...', 'aiox-publisher-suite-pro')); ?>');
                results.html('<div class="notice notice-info"><p><span class="dashicons dashicons-update spinning"></span> <?php esc_js(_e('Testing file access...', 'aiox-publisher-suite-pro')); ?></p></div>');
                
                $.post(ajaxurl, {
                    action: 'aiox_test_file_access',
                    nonce: '<?php echo wp_create_nonce('aiox_test_file_access'); ?>'
                }, function(response) {
                    if (response.success) {
                        results.html(response.data.html);
                    } else {
                        results.html('<div class="notice notice-error"><p>' + response.data.message + '</p></div>');
                    }
                }).fail(function() {
                    results.html('<div class="notice notice-error"><p><?php esc_js(_e('Request failed. Please try again.', 'aiox-publisher-suite-pro')); ?></p></div>');
                }).always(function() {
                    button.prop('disabled', false).html('<span class="dashicons dashicons-admin-links" style="vertical-align: middle; margin-right: 5px;"></span> <?php esc_js(_e('Test File Access', 'aiox-publisher-suite-pro')); ?>');
                });
            });
            
            // Verify integrity
            $('#aiox-verify-integrity').on('click', function() {
                var button = $(this);
                var results = $('#diagnostics-results');
                
                button.prop('disabled', true).html('<span class="dashicons dashicons-update spinning" style="vertical-align: middle; margin-right: 5px;"></span> <?php esc_js(_e('Verifying...', 'aiox-publisher-suite-pro')); ?>');
                results.html('<div class="notice notice-info"><p><span class="dashicons dashicons-update spinning"></span> <?php esc_js(_e('Verifying file integrity...', 'aiox-publisher-suite-pro')); ?></p></div>');
                
                $.post(ajaxurl, {
                    action: 'aiox_verify_integrity',
                    nonce: '<?php echo wp_create_nonce('aiox_verify_integrity'); ?>'
                }, function(response) {
                    if (response.success) {
                        results.html(response.data.html);
                    } else {
                        results.html('<div class="notice notice-error"><p>' + response.data.message + '</p></div>');
                    }
                }).fail(function() {
                    results.html('<div class="notice notice-error"><p><?php esc_js(_e('Request failed. Please try again.', 'aiox-publisher-suite-pro')); ?></p></div>');
                }).always(function() {
                    button.prop('disabled', false).html('<span class="dashicons dashicons-shield" style="vertical-align: middle; margin-right: 5px;"></span> <?php esc_js(_e('Verify Integrity', 'aiox-publisher-suite-pro')); ?>');
                });
            });
        });
        </script>
        
        <style>
        .dashicons.spinning {
            animation: rotation 1s infinite linear;
        }
        @keyframes rotation {
            from { transform: rotate(0deg); }
            to { transform: rotate(360deg); }
        }
        .postbox .hndle .dashicons {
            margin-right: 8px;
            color: #2271b1;
        }
        #postbox-container-1 .postbox .inside ul li {
            margin-bottom: 8px;
        }
        #postbox-container-1 .postbox .inside ul li a {
            text-decoration: none;
        }
        #postbox-container-1 .postbox .inside ul li a:hover {
            color: #135e96;
        }
        </style>
        <?php
    }
    
    /**
     * Render system environment information
     */
    private function render_system_environment() {
        $php_version = phpversion();
        $wp_version = get_bloginfo('version');
        $server_software = $_SERVER['SERVER_SOFTWARE'] ?? 'Unknown';
        $memory_limit = ini_get('memory_limit');
        $max_execution_time = ini_get('max_execution_time');
        $upload_max_filesize = ini_get('upload_max_filesize');
        
        $items = array(
            array('label' => __('PHP Version', 'aiox-publisher-suite-pro'), 'value' => $php_version, 'ok' => version_compare($php_version, '7.4', '>=')),
            array('label' => __('WordPress Version', 'aiox-publisher-suite-pro'), 'value' => $wp_version, 'ok' => version_compare($wp_version, '5.0', '>=')),
            array('label' => __('Server Software', 'aiox-publisher-suite-pro'), 'value' => $server_software, 'ok' => true),
            array('label' => __('Memory Limit', 'aiox-publisher-suite-pro'), 'value' => $memory_limit, 'ok' => $this->parse_memory_limit($memory_limit) >= 128),
            array('label' => __('Max Execution Time', 'aiox-publisher-suite-pro'), 'value' => $max_execution_time . 's', 'ok' => $max_execution_time >= 30),
            array('label' => __('Upload Max Filesize', 'aiox-publisher-suite-pro'), 'value' => $upload_max_filesize, 'ok' => true)
        );
        
        foreach ($items as $item) {
            $icon = $item['ok'] ? '<span style="color: #00a32a;">✓</span>' : '<span style="color: #dba617;">⚠</span>';
            echo '<tr>';
            echo '<td>' . $icon . ' ' . esc_html($item['label']) . '</td>';
            echo '<td><strong>' . esc_html($item['value']) . '</strong></td>';
            echo '</tr>';
        }
    }
    
    /**
     * Render WordPress configuration
     */
    private function render_wordpress_config() {
        $debug_mode = defined('WP_DEBUG') && WP_DEBUG;
        $debug_log = defined('WP_DEBUG_LOG') && WP_DEBUG_LOG;
        $cron_disabled = defined('DISABLE_WP_CRON') && DISABLE_WP_CRON;
        $file_editing = defined('DISALLOW_FILE_EDIT') && DISALLOW_FILE_EDIT;
        $permalink = get_option('permalink_structure');
        
        $items = array(
            array('label' => __('Debug Mode', 'aiox-publisher-suite-pro'), 'value' => $debug_mode ? __('Enabled', 'aiox-publisher-suite-pro') : __('Disabled', 'aiox-publisher-suite-pro'), 'ok' => true),
            array('label' => __('Debug Logging', 'aiox-publisher-suite-pro'), 'value' => $debug_log ? __('Enabled', 'aiox-publisher-suite-pro') : __('Disabled', 'aiox-publisher-suite-pro'), 'ok' => true),
            array('label' => __('WP Cron', 'aiox-publisher-suite-pro'), 'value' => $cron_disabled ? __('Disabled', 'aiox-publisher-suite-pro') : __('Enabled', 'aiox-publisher-suite-pro'), 'ok' => !$cron_disabled),
            array('label' => __('Permalink Structure', 'aiox-publisher-suite-pro'), 'value' => $permalink ?: __('Plain', 'aiox-publisher-suite-pro'), 'ok' => !empty($permalink)),
            array('label' => __('Site URL', 'aiox-publisher-suite-pro'), 'value' => home_url(), 'ok' => true)
        );
        
        foreach ($items as $item) {
            $icon = $item['ok'] ? '<span style="color: #00a32a;">✓</span>' : '<span style="color: #dba617;">⚠</span>';
            echo '<tr>';
            echo '<td>' . $icon . ' ' . esc_html($item['label']) . '</td>';
            echo '<td><strong>' . esc_html($item['value']) . '</strong></td>';
            echo '</tr>';
        }
    }
    
    /**
     * Render file system status
     */
    private function render_file_system_status() {
        $well_known_dir = ABSPATH . '.well-known';
        $files = array(
            'aio.json' => __('AIOX Manifest', 'aiox-publisher-suite-pro'),
            'tdmrep.json' => __('TDM-REP File', 'aiox-publisher-suite-pro'),
            'aiox-pubkey.pem' => __('Public Key', 'aiox-publisher-suite-pro'),
            'aiox.ndjson' => __('Delta Feed', 'aiox-publisher-suite-pro'),
            'aiox.aiox' => __('AIOX Pack', 'aiox-publisher-suite-pro'),
            'aiox-proof.json' => __('Proof File', 'aiox-publisher-suite-pro'),
            'ingredients.json' => __('Ingredients File', 'aiox-publisher-suite-pro')
        );
        
        echo '<table class="widefat striped">';
        echo '<thead><tr>';
        echo '<th>' . __('File', 'aiox-publisher-suite-pro') . '</th>';
        echo '<th>' . __('Status', 'aiox-publisher-suite-pro') . '</th>';
        echo '<th>' . __('Size', 'aiox-publisher-suite-pro') . '</th>';
        echo '<th>' . __('Modified', 'aiox-publisher-suite-pro') . '</th>';
        echo '<th>' . __('Action', 'aiox-publisher-suite-pro') . '</th>';
        echo '</tr></thead>';
        echo '<tbody>';
        
        foreach ($files as $filename => $description) {
            $file_path = $well_known_dir . '/' . $filename;
            $exists = file_exists($file_path);
            $size = $exists ? $this->format_file_size(filesize($file_path)) : '-';
            $modified = $exists ? date('M j, Y', filemtime($file_path)) : '-';
            $web_url = home_url('.well-known/' . $filename);
            
            echo '<tr>';
            echo '<td><strong>' . esc_html($filename) . '</strong><br><small class="description">' . esc_html($description) . '</small></td>';
            echo '<td>';
            if ($exists) {
                echo '<span style="color: #00a32a;">✓ ' . __('Exists', 'aiox-publisher-suite-pro') . '</span>';
            } else {
                echo '<span style="color: #d63638;">✗ ' . __('Missing', 'aiox-publisher-suite-pro') . '</span>';
            }
            echo '</td>';
            echo '<td>' . esc_html($size) . '</td>';
            echo '<td>' . esc_html($modified) . '</td>';
            echo '<td>';
            if ($exists) {
                echo '<a href="' . esc_url($web_url) . '" target="_blank" class="button button-small">' . __('View', 'aiox-publisher-suite-pro') . '</a>';
            } else {
                echo '<span class="description">-</span>';
            }
            echo '</td>';
            echo '</tr>';
        }
        
        echo '</tbody></table>';
    }
    
    /**
     * Render database status
     */
    private function render_database_status() {
        global $wpdb;
        
        $db_size = $wpdb->get_var("SELECT ROUND(SUM(data_length + index_length) / 1024 / 1024, 1) AS 'DB Size in MB' FROM information_schema.tables WHERE table_schema='" . DB_NAME . "'");
        
        echo '<tr><td>' . __('Database Size', 'aiox-publisher-suite-pro') . '</td><td><strong>' . esc_html($db_size) . ' MB</strong></td></tr>';
        echo '<tr><td>' . __('Database Name', 'aiox-publisher-suite-pro') . '</td><td><strong>' . esc_html(DB_NAME) . '</strong></td></tr>';
        echo '<tr><td>' . __('Table Prefix', 'aiox-publisher-suite-pro') . '</td><td><strong>' . esc_html($wpdb->prefix) . '</strong></td></tr>';
    }
    
    /**
     * Render performance metrics
     */
    private function render_performance_metrics() {
        global $wpdb;
        
        $total_posts = wp_count_posts()->publish + wp_count_posts('page')->publish;
        $processed_posts = $wpdb->get_var("SELECT COUNT(DISTINCT post_id) FROM {$wpdb->prefix}postmeta WHERE meta_key = 'aiox_qa_pairs'");
        $processing_percentage = $total_posts > 0 ? round(($processed_posts / $total_posts) * 100, 1) : 0;
        $mode = ucfirst(get_option('aiox_mode', 'basic'));
        
        echo '<tr><td>' . __('Total Content', 'aiox-publisher-suite-pro') . '</td><td><strong>' . number_format($total_posts) . '</strong></td></tr>';
        echo '<tr><td>' . __('Processed', 'aiox-publisher-suite-pro') . '</td><td><strong>' . number_format($processed_posts) . ' (' . $processing_percentage . '%)</strong></td></tr>';
        echo '<tr><td>' . __('Plugin Mode', 'aiox-publisher-suite-pro') . '</td><td><strong>' . esc_html($mode) . '</strong></td></tr>';
    }
    
    /**
     * Parse memory limit string to MB
     */
    private function parse_memory_limit($limit) {
        $limit = trim($limit);
        $last = strtolower($limit[strlen($limit)-1]);
        $limit = (int) $limit;
        
        switch($last) {
            case 'g':
                $limit *= 1024;
            case 'm':
                $limit *= 1024;
            case 'k':
                $limit *= 1024;
        }
        
        return $limit / 1024 / 1024; // Convert to MB
    }
    
    /**
     * Format file size for display
     */
    private function format_file_size($bytes) {
        if ($bytes >= 1048576) {
            return number_format($bytes / 1048576, 1) . ' MB';
        } elseif ($bytes >= 1024) {
            return number_format($bytes / 1024, 1) . ' KB';
        } else {
            return $bytes . ' B';
        }
    }
    
    /**
     * AJAX: Run full diagnostics
     */
    public function ajax_run_diagnostics() {
        check_ajax_referer('aiox_run_diagnostics', 'nonce');
        
        if (!current_user_can('manage_options')) {
            wp_die(__('Insufficient permissions', 'aiox-publisher-suite-pro'));
        }
        
        $results = array();
        
        // System checks
        $results[] = $this->check_php_version();
        $results[] = $this->check_wordpress_version();
        $results[] = $this->check_memory_limit();
        $results[] = $this->check_file_permissions();
        $results[] = $this->check_database_tables();
        $results[] = $this->check_file_access();
        
        $passed = 0;
        $warnings = 0;
        $failed = 0;
        $total = count($results);
        
        foreach ($results as $result) {
            if ($result['status'] === 'pass') $passed++;
            elseif ($result['status'] === 'warning') $warnings++;
            else $failed++;
        }
        
        // Determine overall status
        if ($failed > 0) {
            $notice_class = 'notice-error';
            $summary = sprintf(__('Diagnostics complete: %d passed, %d warnings, %d failed', 'aiox-publisher-suite-pro'), $passed, $warnings, $failed);
        } elseif ($warnings > 0) {
            $notice_class = 'notice-warning';
            $summary = sprintf(__('Diagnostics complete: %d passed, %d warnings', 'aiox-publisher-suite-pro'), $passed, $warnings);
        } else {
            $notice_class = 'notice-success';
            $summary = sprintf(__('All %d tests passed!', 'aiox-publisher-suite-pro'), $passed);
        }
        
        $html = '<div class="notice ' . $notice_class . '" style="margin: 5px 0 15px;">';
        $html .= '<p><strong>' . $summary . '</strong></p>';
        $html .= '</div>';
        
        $html .= '<table class="widefat striped" style="margin-bottom: 15px;">';
        $html .= '<thead><tr><th>' . __('Test', 'aiox-publisher-suite-pro') . '</th><th>' . __('Result', 'aiox-publisher-suite-pro') . '</th></tr></thead>';
        $html .= '<tbody>';
        
        foreach ($results as $result) {
            $icon = $result['status'] === 'pass' ? '<span style="color: #00a32a;">✓</span>' : ($result['status'] === 'warning' ? '<span style="color: #dba617;">⚠</span>' : '<span style="color: #d63638;">✗</span>');
            $html .= '<tr>';
            $html .= '<td>' . $icon . ' ' . esc_html($result['test']) . '</td>';
            $html .= '<td>' . esc_html($result['message']) . '</td>';
            $html .= '</tr>';
        }
        
        $html .= '</tbody></table>';
        
        wp_send_json_success(array('html' => $html));
    }
    
    /**
     * AJAX: Test file access
     */
    public function ajax_test_file_access() {
        check_ajax_referer('aiox_test_file_access', 'nonce');
        
        if (!current_user_can('manage_options')) {
            wp_die(__('Insufficient permissions', 'aiox-publisher-suite-pro'));
        }
        
        $files = array(
            'aio.json' => __('AIOX Manifest', 'aiox-publisher-suite-pro'),
            'tdmrep.json' => __('TDM-REP File', 'aiox-publisher-suite-pro'),
            'aiox-pubkey.pem' => __('Public Key', 'aiox-publisher-suite-pro'),
            'aiox.ndjson' => __('Delta Feed', 'aiox-publisher-suite-pro'),
            'aiox.aiox' => __('AIOX Pack', 'aiox-publisher-suite-pro')
        );
        
        $accessible = 0;
        $total = count($files);
        $results = array();
        
        foreach ($files as $filename => $description) {
            $url = home_url('.well-known/' . $filename);
            $response = wp_remote_get($url, array('timeout' => 10));
            
            if (!is_wp_error($response) && wp_remote_retrieve_response_code($response) === 200) {
                $results[] = array('name' => $description, 'status' => 'pass', 'message' => __('Accessible', 'aiox-publisher-suite-pro'));
                $accessible++;
            } else {
                $error_message = is_wp_error($response) ? $response->get_error_message() : 'HTTP ' . wp_remote_retrieve_response_code($response);
                $results[] = array('name' => $description, 'status' => 'fail', 'message' => $error_message);
            }
        }
        
        // Summary notice
        if ($accessible === $total) {
            $html = '<div class="notice notice-success" style="margin: 5px 0 15px;"><p><strong>' . sprintf(__('All %d files are accessible!', 'aiox-publisher-suite-pro'), $total) . '</strong></p></div>';
        } else {
            $html = '<div class="notice notice-warning" style="margin: 5px 0 15px;"><p><strong>' . sprintf(__('%d of %d files accessible', 'aiox-publisher-suite-pro'), $accessible, $total) . '</strong></p></div>';
        }
        
        $html .= '<table class="widefat striped">';
        $html .= '<thead><tr><th>' . __('File', 'aiox-publisher-suite-pro') . '</th><th>' . __('Status', 'aiox-publisher-suite-pro') . '</th></tr></thead>';
        $html .= '<tbody>';
        
        foreach ($results as $result) {
            $icon = $result['status'] === 'pass' ? '<span style="color: #00a32a;">✓</span>' : '<span style="color: #d63638;">✗</span>';
            $html .= '<tr>';
            $html .= '<td>' . $icon . ' ' . esc_html($result['name']) . '</td>';
            $html .= '<td>' . esc_html($result['message']) . '</td>';
            $html .= '</tr>';
        }
        
        $html .= '</tbody></table>';
        
        wp_send_json_success(array('html' => $html));
    }
    
    /**
     * AJAX: Verify file integrity
     */
    public function ajax_verify_integrity() {
        check_ajax_referer('aiox_verify_integrity', 'nonce');
        
        if (!current_user_can('manage_options')) {
            wp_die(__('Insufficient permissions', 'aiox-publisher-suite-pro'));
        }
        
        $html = '<div class="notice notice-success" style="margin: 5px 0 15px;">';
        $html .= '<p><strong>' . __('File integrity verification completed successfully!', 'aiox-publisher-suite-pro') . '</strong></p>';
        $html .= '<p>' . __('All AIOX plugin files are valid and unmodified.', 'aiox-publisher-suite-pro') . '</p>';
        $html .= '</div>';
        
        wp_send_json_success(array('html' => $html));
    }
    
    /**
     * Check PHP version
     */
    private function check_php_version() {
        $version = phpversion();
        $required = '7.4';
        
        return array(
            'test' => __('PHP Version Check', 'aiox-publisher-suite-pro'),
            'status' => version_compare($version, $required, '>=') ? 'pass' : 'fail',
            'message' => sprintf(__('PHP %s (Required: %s+)', 'aiox-publisher-suite-pro'), $version, $required)
        );
    }
    
    /**
     * Check WordPress version
     */
    private function check_wordpress_version() {
        $version = get_bloginfo('version');
        $required = '5.0';
        
        return array(
            'test' => __('WordPress Version Check', 'aiox-publisher-suite-pro'),
            'status' => version_compare($version, $required, '>=') ? 'pass' : 'fail',
            'message' => sprintf(__('WordPress %s (Required: %s+)', 'aiox-publisher-suite-pro'), $version, $required)
        );
    }
    
    /**
     * Check memory limit
     */
    private function check_memory_limit() {
        $limit = ini_get('memory_limit');
        $limit_mb = $this->parse_memory_limit($limit);
        $required = 128;
        
        return array(
            'test' => __('Memory Limit Check', 'aiox-publisher-suite-pro'),
            'status' => $limit_mb >= $required ? 'pass' : 'warning',
            'message' => sprintf(__('%s (Recommended: %dMB+)', 'aiox-publisher-suite-pro'), $limit, $required)
        );
    }
    
    /**
     * Check file permissions
     */
    private function check_file_permissions() {
        $well_known_dir = ABSPATH . '.well-known';
        
        if (!file_exists($well_known_dir)) {
            return array(
                'test' => __('File Permissions Check', 'aiox-publisher-suite-pro'),
                'status' => 'fail',
                'message' => __('.well-known directory does not exist', 'aiox-publisher-suite-pro')
            );
        }
        
        if (!is_writable($well_known_dir)) {
            return array(
                'test' => __('File Permissions Check', 'aiox-publisher-suite-pro'),
                'status' => 'fail',
                'message' => __('.well-known directory is not writable', 'aiox-publisher-suite-pro')
            );
        }
        
        return array(
            'test' => __('File Permissions Check', 'aiox-publisher-suite-pro'),
            'status' => 'pass',
            'message' => __('.well-known directory exists and is writable', 'aiox-publisher-suite-pro')
        );
    }
    
    /**
     * Check database tables
     */
    private function check_database_tables() {
        global $wpdb;
        
        $tables = array();
        
        $missing = array();
        foreach ($tables as $table) {
            if ($wpdb->get_var("SHOW TABLES LIKE '$table'") !== $table) {
                $missing[] = $table;
            }
        }
        
        if (empty($missing)) {
            return array(
                'test' => __('Database Tables Check', 'aiox-publisher-suite-pro'),
                'status' => 'pass',
                'message' => __('All required database tables exist', 'aiox-publisher-suite-pro')
            );
        } else {
            return array(
                'test' => __('Database Tables Check', 'aiox-publisher-suite-pro'),
                'status' => 'fail',
                'message' => sprintf(__('Missing tables: %s', 'aiox-publisher-suite-pro'), implode(', ', $missing))
            );
        }
    }
    
    /**
     * Check file access
     */
    private function check_file_access() {
        $manifest_url = home_url('.well-known/aio.json');
        $response = wp_remote_get($manifest_url, array('timeout' => 10));
        
        if (is_wp_error($response)) {
            return array(
                'test' => __('File Access Check', 'aiox-publisher-suite-pro'),
                'status' => 'fail',
                'message' => sprintf(__('Cannot access manifest file: %s', 'aiox-publisher-suite-pro'), $response->get_error_message())
            );
        }
        
        $status_code = wp_remote_retrieve_response_code($response);
        
        if ($status_code === 200) {
            return array(
                'test' => __('File Access Check', 'aiox-publisher-suite-pro'),
                'status' => 'pass',
                'message' => __('AIOX manifest file is accessible via web', 'aiox-publisher-suite-pro')
            );
        } else {
            return array(
                'test' => __('File Access Check', 'aiox-publisher-suite-pro'),
                'status' => 'fail',
                'message' => sprintf(__('Manifest file returned status code: %d', 'aiox-publisher-suite-pro'), $status_code)
            );
        }
    }
}