<?php
/**
 * AIOX AI License & Monetization Admin Page
 * 
 * Dedicated page for managing AI licensing, monetization, and enforcement settings
 */

if (!defined('ABSPATH')) {
    exit;
}

class AIOX_Licensing_Page {
    
    /**
     * Initialize the licensing page
     */
    public function __construct() {
        add_action('admin_menu', array($this, 'add_menu_page'));
        add_action('admin_init', array($this, 'register_settings'));
        add_action('wp_ajax_aiox_regenerate_all_manifests', array($this, 'ajax_regenerate_all_manifests'));
        
        // Auto-regenerate manifests when license/monetization presets change
        add_action('update_option_aiox_license_preset', array($this, 'on_preset_change'), 10, 3);
        add_action('update_option_aiox_monetization_preset', array($this, 'on_preset_change'), 10, 3);
        add_action('update_option_aiox_enforcement_mode', array($this, 'on_preset_change'), 10, 3);
        add_action('update_option_aiox_tdm_training_allowed', array($this, 'on_preset_change'), 10, 3);
        add_action('update_option_aiox_tdm_citation_required', array($this, 'on_preset_change'), 10, 3);
        add_action('update_option_aiox_license_custom_terms', array($this, 'on_preset_change'), 10, 3);
        add_action('update_option_aiox_license_terms_url', array($this, 'on_preset_change'), 10, 3);
        add_action('update_option_aiox_monetization_license_api', array($this, 'on_preset_change'), 10, 3);
    }
    
    /**
     * Add submenu page under AIOX
     */
    public function add_menu_page() {
        add_submenu_page(
            'aiox-publisher-suite',
            __('AI License & Monetization', 'aiox-publisher-suite-pro'),
            __('AI License & Monetization', 'aiox-publisher-suite-pro'),
            'manage_options',
            'aiox-licensing',
            array($this, 'render_page')
        );
    }
    
    /**
     * Register settings
     */
    public function register_settings() {
        // License preset
        register_setting('aiox-licensing', 'aiox_license_preset', array(
            'type' => 'string',
            'sanitize_callback' => array($this, 'sanitize_license_preset'),
            'default' => 'attribution_required'
        ));
        
        // Monetization preset
        register_setting('aiox-licensing', 'aiox_monetization_preset', array(
            'type' => 'string',
            'sanitize_callback' => array($this, 'sanitize_monetization_preset'),
            'default' => 'free'
        ));
        
        // Enforcement mode
        register_setting('aiox-licensing', 'aiox_enforcement_mode', array(
            'type' => 'string',
            'sanitize_callback' => array($this, 'sanitize_enforcement_mode'),
            'default' => 'flexible'
        ));
        
        // Custom terms
        register_setting('aiox-licensing', 'aiox_license_custom_terms', array(
            'type' => 'string',
            'sanitize_callback' => 'sanitize_textarea_field',
            'default' => ''
        ));
        
        // Contact email
        register_setting('aiox-licensing', 'aiox_license_contact_email', array(
            'type' => 'string',
            'sanitize_callback' => 'sanitize_email',
            'default' => ''
        ));
        
        // Terms URL
        register_setting('aiox-licensing', 'aiox_license_terms_url', array(
            'type' => 'string',
            'sanitize_callback' => 'esc_url_raw',
            'default' => ''
        ));
        
        // License API endpoint
        register_setting('aiox-licensing', 'aiox_monetization_license_api', array(
            'type' => 'string',
            'sanitize_callback' => 'esc_url_raw',
            'default' => ''
        ));
    }
    
    /**
     * Sanitize license preset
     */
    public function sanitize_license_preset($value) {
        $valid = array('open_access', 'attribution_required', 'commercial_license');
        return in_array($value, $valid) ? $value : 'attribution_required';
    }
    
    /**
     * Sanitize monetization preset
     */
    public function sanitize_monetization_preset($value) {
        $valid = array('free', 'freemium', 'paid');
        return in_array($value, $valid) ? $value : 'free';
    }
    
    /**
     * Sanitize enforcement mode
     */
    public function sanitize_enforcement_mode($value) {
        $valid = array('strict', 'flexible');
        return in_array($value, $valid) ? $value : 'flexible';
    }
    
    /**
     * Handle preset/licensing setting changes - regenerate manifests
     */
    public function on_preset_change($old_value, $new_value, $option) {
        if ($old_value === $new_value) {
            return;
        }
        
        // Schedule regeneration (use transient to batch multiple changes)
        if (!get_transient('aiox_preset_change_pending')) {
            set_transient('aiox_preset_change_pending', true, 30);
            add_action('shutdown', array($this, 'regenerate_all_on_save'));
        }
        
        if (class_exists('AIOX_Logger')) {
            AIOX_Logger::info('Licensing setting changed, manifests will regenerate', array(
                'option' => $option,
                'old_value' => $old_value,
                'new_value' => $new_value
            ));
        }
    }
    
    /**
     * Regenerate all manifest files after preset changes
     */
    public function regenerate_all_on_save() {
        delete_transient('aiox_preset_change_pending');
        
        // Regenerate main manifest, TDM-REP, and delta feed
        if (class_exists('AIOX_Hooks')) {
            $hooks = new AIOX_Hooks();
            
            if (method_exists($hooks, 'generate_manifest')) {
                $hooks->generate_manifest();
            }
            if (method_exists($hooks, 'generate_tdmrep')) {
                $hooks->generate_tdmrep();
            }
            if (method_exists($hooks, 'generate_delta_feed')) {
                $hooks->generate_delta_feed();
            }
            if (method_exists($hooks, 'generate_robots_txt')) {
                $hooks->generate_robots_txt();
            }
        }
        
        // Regenerate ALL individual post manifest files (NO API calls)
        $this->regenerate_all_post_manifests();
        
        if (class_exists('AIOX_Logger')) {
            AIOX_Logger::info('All manifests regenerated after licensing settings change');
        }
        
        // Set a transient to show success message
        set_transient('aiox_manifests_regenerated', true, 30);
    }
    
    /**
     * AJAX handler to regenerate all manifest files (no API calls)
     */
    public function ajax_regenerate_all_manifests() {
        check_ajax_referer('aiox_regenerate_manifests', 'nonce');
        
        if (!current_user_can('manage_options')) {
            wp_send_json_error(array('message' => 'Permission denied'));
            return;
        }
        
        $start_time = microtime(true);
        
        // Regenerate main manifest files
        if (class_exists('AIOX_Hooks')) {
            $hooks = new AIOX_Hooks();
            
            if (method_exists($hooks, 'generate_manifest')) {
                $hooks->generate_manifest();
            }
            if (method_exists($hooks, 'generate_tdmrep')) {
                $hooks->generate_tdmrep();
            }
            if (method_exists($hooks, 'generate_delta_feed')) {
                $hooks->generate_delta_feed();
            }
        }
        
        // Regenerate all post manifest files
        $stats = $this->regenerate_all_post_manifests();
        
        $elapsed = round(microtime(true) - $start_time, 2);
        
        wp_send_json_success(array(
            'message' => sprintf(
                __('Regenerated %d post manifests in %s seconds. Updated: %d, Failed: %d, Skipped: %d', 'aiox-publisher-suite-pro'),
                $stats['total'],
                $elapsed,
                $stats['updated'],
                $stats['failed'],
                $stats['skipped']
            ),
            'stats' => $stats,
            'elapsed' => $elapsed
        ));
    }
    
    /**
     * Regenerate all individual post manifest files with current settings
     */
    public function regenerate_all_post_manifests() {
        $well_known_dir = ABSPATH . '.well-known';
        $posts_dir = $well_known_dir . '/posts';
        
        if (!file_exists($posts_dir)) {
            wp_mkdir_p($posts_dir);
        }
        
        $args = array(
            'post_type' => array('post', 'page'),
            'post_status' => 'publish',
            'posts_per_page' => -1,
            'meta_query' => array(
                array(
                    'key' => '_aiox_processed',
                    'value' => '1',
                    'compare' => '='
                )
            ),
            'fields' => 'ids'
        );
        
        $post_ids = get_posts($args);
        $stats = array(
            'total' => count($post_ids),
            'updated' => 0,
            'failed' => 0,
            'skipped' => 0
        );
        
        if (empty($post_ids)) {
            return $stats;
        }
        
        $license_config = $this->get_license_config();
        $monetization_config = $this->get_monetization_config();
        $enforcement_config = $this->get_enforcement_config();
        
        foreach ($post_ids as $post_id) {
            $result = $this->update_post_manifest($post_id, $license_config, $monetization_config, $enforcement_config, $posts_dir);
            
            if ($result === true) {
                $stats['updated']++;
            } elseif ($result === false) {
                $stats['failed']++;
            } else {
                $stats['skipped']++;
            }
        }
        
        if (class_exists('AIOX_Logger')) {
            AIOX_Logger::info('Post manifests regenerated with new settings', $stats);
        }
        
        return $stats;
    }
    
    /**
     * Update a single post's manifest file
     */
    private function update_post_manifest($post_id, $license_config, $monetization_config, $enforcement_config, $posts_dir) {
        $file_path = $posts_dir . '/' . $post_id . '.json';
        $home_url = home_url();
        
        if (file_exists($file_path)) {
            $existing_data = json_decode(file_get_contents($file_path), true);
            
            if (!$existing_data) {
                return false;
            }
            
            // Update only the licensing/monetization sections
            $existing_data['licensing'] = array(
                'preset' => $license_config['preset'],
                'license_code' => $license_config['license_code'],
                'training_allowed' => $license_config['training_allowed'],
                'citation_required' => $license_config['citation_required'],
                'attribution_required' => $license_config['attribution_required'],
                'commercial_use' => $license_config['commercial_use'],
                'attribution' => get_option('aiox_attribution_string', get_bloginfo('name')),
                'inherited_from' => 'site'
            );
            
            $existing_data['monetization'] = array(
                'preset' => $monetization_config['preset'],
                'model' => $monetization_config['pricing_model'],
                'subscription_required' => $monetization_config['subscription_required'],
                'pay_per_use' => $monetization_config['pay_per_use'],
                'inherited_from' => 'site'
            );
            
            $existing_data['enforcement'] = array(
                'mode' => $enforcement_config['mode'],
                'compliance_level' => $enforcement_config['compliance_level'],
                'legal_binding' => $enforcement_config['legal_binding']
            );
            
            $existing_data['settings_updated_at'] = current_time('c');
            
            if (!isset($existing_data['links']['tdmrep'])) {
                $existing_data['links']['tdmrep'] = $home_url . '/.well-known/tdmrep.json';
            }
            
            $result = file_put_contents($file_path, json_encode($existing_data, JSON_PRETTY_PRINT | JSON_UNESCAPED_SLASHES));
            
            return $result !== false;
        } else {
            // Generate fresh manifest
            $manifest_data = $this->generate_post_manifest($post_id, $license_config, $monetization_config, $enforcement_config);
            
            if ($manifest_data) {
                $result = file_put_contents($file_path, json_encode($manifest_data, JSON_PRETTY_PRINT | JSON_UNESCAPED_SLASHES));
                return $result !== false;
            }
            
            return null;
        }
    }
    
    /**
     * Generate a complete post manifest
     */
    private function generate_post_manifest($post_id, $license_config, $monetization_config, $enforcement_config) {
        $post = get_post($post_id);
        
        if (!$post || $post->post_status !== 'publish') {
            return null;
        }
        
        $home_url = home_url();
        
        $ingredients = array();
        if (class_exists('AIOX_Ingredients')) {
            $ingredients = AIOX_Ingredients::get_post_ingredients($post_id);
        }
        
        $metadata = array();
        if (class_exists('AIOX_Metadata')) {
            $metadata = AIOX_Metadata::get_post_metadata($post_id);
        }
        
        $capsule_data = array();
        if (class_exists('AIOX_Capsule_Manager')) {
            $capsule_manager = new AIOX_Capsule_Manager();
            $capsule = $capsule_manager->get_capsule_by_post_id($post_id);
            if ($capsule) {
                $capsule_data = array(
                    'id' => $capsule->get_id(),
                    'status' => $capsule->get_status(),
                    'type' => $capsule->get_type()
                );
            }
        }
        
        return array(
            'version' => '2.0',
            'generator' => 'AIOX Publisher Suite Pro',
            'generated_at' => get_post_meta($post_id, '_aiox_processed_at', true) ?: current_time('c'),
            'settings_updated_at' => current_time('c'),
            'post' => array(
                'id' => $post_id,
                'title' => $post->post_title,
                'url' => get_permalink($post_id),
                'published' => get_the_date('c', $post_id),
                'modified' => get_the_modified_date('c', $post_id),
                'type' => $post->post_type,
                'author' => get_the_author_meta('display_name', $post->post_author),
                'excerpt' => wp_trim_words(get_the_excerpt($post_id), 50, '...'),
                'word_count' => str_word_count(strip_tags($post->post_content)),
                'categories' => wp_get_post_categories($post_id, array('fields' => 'names')),
                'tags' => wp_get_post_tags($post_id, array('fields' => 'names'))
            ),
            'licensing' => array(
                'preset' => $license_config['preset'],
                'license_code' => $license_config['license_code'],
                'training_allowed' => $license_config['training_allowed'],
                'citation_required' => $license_config['citation_required'],
                'attribution_required' => $license_config['attribution_required'],
                'commercial_use' => $license_config['commercial_use'],
                'attribution' => get_option('aiox_attribution_string', get_bloginfo('name')),
                'inherited_from' => 'site'
            ),
            'monetization' => array(
                'preset' => $monetization_config['preset'],
                'model' => $monetization_config['pricing_model'],
                'subscription_required' => $monetization_config['subscription_required'],
                'pay_per_use' => $monetization_config['pay_per_use'],
                'inherited_from' => 'site'
            ),
            'enforcement' => array(
                'mode' => $enforcement_config['mode'],
                'compliance_level' => $enforcement_config['compliance_level'],
                'legal_binding' => $enforcement_config['legal_binding']
            ),
            'ingredients' => array(
                'count' => count($ingredients),
                'items' => $ingredients
            ),
            'capsule' => $capsule_data,
            'metadata' => $metadata,
            'processing' => array(
                'processed' => true,
                'last_processed' => get_post_meta($post_id, '_aiox_processed_at', true),
                'server_processed' => (bool) get_post_meta($post_id, '_aiox_server_processed', true)
            ),
            'links' => array(
                'self' => $home_url . '/.well-known/posts/' . $post_id . '.json',
                'post' => get_permalink($post_id),
                'manifest' => $home_url . '/.well-known/aio.json',
                'tdmrep' => $home_url . '/.well-known/tdmrep.json'
            )
        );
    }
    
    /**
     * Get license preset configuration
     */
    public static function get_license_config() {
        $preset = get_option('aiox_license_preset', 'attribution_required');
        
        $presets = array(
            'open_access' => array(
                'preset' => 'open_access',
                'name' => 'Open Access',
                'training_allowed' => true,
                'citation_required' => true,
                'attribution_required' => true,
                'commercial_use' => true,
                'derivatives_allowed' => true,
                'share_alike' => false,
                'license_code' => 'AIOX-OA-1.0',
                'description' => 'Free to use, train, and cite with attribution. Ideal for open-source content.',
                'similar_to' => 'CC BY 4.0'
            ),
            'attribution_required' => array(
                'preset' => 'attribution_required',
                'name' => 'Attribution Required',
                'training_allowed' => false,
                'citation_required' => true,
                'attribution_required' => true,
                'commercial_use' => true,
                'derivatives_allowed' => true,
                'share_alike' => true,
                'license_code' => 'AIOX-AR-1.0',
                'description' => 'Content can be cited and referenced, but AI training requires permission.',
                'similar_to' => 'CC BY-SA 4.0'
            ),
            'commercial_license' => array(
                'preset' => 'commercial_license',
                'name' => 'Commercial License',
                'training_allowed' => false,
                'citation_required' => true,
                'attribution_required' => true,
                'commercial_use' => false,
                'derivatives_allowed' => false,
                'share_alike' => false,
                'license_code' => 'AIOX-CL-1.0',
                'description' => 'All AI usage requires a licensing agreement. For premium content.',
                'similar_to' => 'All Rights Reserved'
            )
        );
        
        return $presets[$preset] ?? $presets['attribution_required'];
    }
    
    /**
     * Get monetization preset configuration
     */
    public static function get_monetization_config() {
        $preset = get_option('aiox_monetization_preset', 'free');
        
        $presets = array(
            'free' => array(
                'preset' => 'free',
                'name' => 'Free Access',
                'subscription_required' => false,
                'pay_per_use' => false,
                'api_access' => true,
                'rate_limited' => false,
                'pricing_model' => 'free',
                'description' => 'No cost for AI systems to access your content. Maximum reach.'
            ),
            'freemium' => array(
                'preset' => 'freemium',
                'name' => 'Freemium',
                'subscription_required' => false,
                'pay_per_use' => true,
                'api_access' => true,
                'rate_limited' => true,
                'pricing_model' => 'freemium',
                'description' => 'Basic access is free, premium features require payment.'
            ),
            'paid' => array(
                'preset' => 'paid',
                'name' => 'Paid Access',
                'subscription_required' => true,
                'pay_per_use' => true,
                'api_access' => true,
                'rate_limited' => true,
                'pricing_model' => 'paid',
                'description' => 'All AI access requires payment - subscription or pay-per-use.'
            )
        );
        
        return $presets[$preset] ?? $presets['free'];
    }
    
    /**
     * Get enforcement mode configuration
     */
    public static function get_enforcement_config() {
        $mode = get_option('aiox_enforcement_mode', 'flexible');
        
        $modes = array(
            'strict' => array(
                'mode' => 'strict',
                'name' => 'Strict Enforcement',
                'compliance_level' => 'mandatory',
                'violation_response' => 'block',
                'legal_binding' => true,
                'grace_period' => false,
                'description' => 'These are requirements, not suggestions. Non-compliant AI systems may face legal action.'
            ),
            'flexible' => array(
                'mode' => 'flexible',
                'name' => 'Flexible Enforcement',
                'compliance_level' => 'recommended',
                'violation_response' => 'warn',
                'legal_binding' => false,
                'grace_period' => true,
                'description' => 'Strong preferences. AI systems may proceed in good faith during protocol adoption.'
            )
        );
        
        return $modes[$mode] ?? $modes['flexible'];
    }
    
    /**
     * Render the licensing page
     */

    public function render_page() {
        // Check for success message
        $regenerated = get_transient('aiox_manifests_regenerated');
        if ($regenerated) {
            delete_transient('aiox_manifests_regenerated');
        }
        
        // Get current settings
        $license_preset = get_option('aiox_license_preset', 'attribution_required');
        $monetization_preset = get_option('aiox_monetization_preset', 'free');
        $enforcement_mode = get_option('aiox_enforcement_mode', 'flexible');
        
        // Get preset configs for display
        $license_presets = array(
            'open_access' => array(
                'name' => 'Open Access',
                'description' => 'Free to use, train, and cite with attribution.',
                'similar_to' => 'CC BY 4.0',
                'icon' => 'unlock'
            ),
            'attribution_required' => array(
                'name' => 'Attribution Required',
                'description' => 'Content can be cited, but AI training requires permission.',
                'similar_to' => 'CC BY-SA 4.0',
                'icon' => 'id-alt'
            ),
            'commercial_license' => array(
                'name' => 'Commercial License',
                'description' => 'All AI usage requires a licensing agreement.',
                'similar_to' => 'All Rights Reserved',
                'icon' => 'money-alt'
            )
        );
        
        $monetization_presets = array(
            'free' => array('name' => 'Free Access', 'description' => 'No cost for AI systems.', 'icon' => 'heart'),
            'freemium' => array('name' => 'Freemium', 'description' => 'Basic free, premium paid.', 'icon' => 'star-half'),
            'paid' => array('name' => 'Paid Access', 'description' => 'All access requires payment.', 'icon' => 'money-alt')
        );
        
        $enforcement_modes = array(
            'strict' => array('name' => 'Strict Enforcement', 'description' => 'Legal requirements.', 'icon' => 'shield-alt'),
            'flexible' => array('name' => 'Flexible Enforcement', 'description' => 'Strong preferences.', 'icon' => 'admin-generic')
        );
        
        // Get stats
        $processed_count = $this->get_processed_posts_count();
        ?>
        <style>
        .aiox-option-cards { display: grid; grid-template-columns: repeat(auto-fit, minmax(200px, 1fr)); gap: 15px; margin-top: 15px; }
        .aiox-option-card { border: 2px solid #ddd; border-radius: 8px; padding: 20px; cursor: pointer; transition: all 0.2s; background: #fff; position: relative; }
        .aiox-option-card:hover { border-color: #2271b1; background: #f0f6fc; }
        .aiox-option-card.selected { border-color: #2271b1; background: #f0f6fc; }
        .aiox-option-card input[type="radio"] { position: absolute; top: 15px; right: 15px; }
        .aiox-option-card .card-icon { font-size: 28px; margin-bottom: 10px; color: #2271b1; }
        .aiox-option-card .card-title { font-weight: 600; font-size: 14px; margin-bottom: 5px; }
        .aiox-option-card .card-desc { font-size: 12px; color: #666; line-height: 1.4; }
        .aiox-option-card .card-similar { font-size: 11px; color: #999; margin-top: 8px; font-style: italic; }
        .aiox-summary-grid { display: grid; grid-template-columns: repeat(4, 1fr); gap: 15px; }
        .aiox-summary-item { background: #f0f0f1; padding: 15px; border-radius: 4px; text-align: center; }
        .aiox-summary-item span { display: block; font-size: 12px; color: #666; margin-bottom: 5px; }
        .aiox-summary-item strong { font-size: 14px; color: #1d2327; }
        @media (max-width: 782px) { 
            .aiox-option-cards { grid-template-columns: 1fr; }
            .aiox-summary-grid { grid-template-columns: repeat(2, 1fr); }
        }
        </style>
        
        <div class="wrap">
            <h1><span class="dashicons dashicons-shield" style="font-size: 30px; width: 30px; height: 30px; margin-right: 10px; color: #2271b1;"></span><?php _e('AI License & Monetization', 'aiox-publisher-suite-pro'); ?></h1>
            <p class="description"><?php _e('Control how AI systems can use, train on, and monetize your content.', 'aiox-publisher-suite-pro'); ?></p>
            
            <?php if ($regenerated): ?>
            <div class="notice notice-success is-dismissible">
                <p><strong><?php _e('Settings Saved & Manifests Regenerated!', 'aiox-publisher-suite-pro'); ?></strong></p>
            </div>
            <?php endif; ?>
            
            <form method="post" action="options.php">
                <?php settings_fields('aiox-licensing'); ?>
                
                <div id="poststuff">
                    <div id="post-body" class="metabox-holder columns-2">
                        <div id="post-body-content">
                            
                            <!-- Enforcement Mode -->
                            <div class="postbox">
                                <h2 class="hndle"><span class="dashicons dashicons-shield-alt"></span> <?php _e('Enforcement Mode', 'aiox-publisher-suite-pro'); ?></h2>
                                <div class="inside">
                                    <p class="description"><?php _e('How strictly should AI systems follow your preferences?', 'aiox-publisher-suite-pro'); ?></p>
                                    
                                    <div class="aiox-option-cards" style="grid-template-columns: repeat(2, 1fr);">
                                        <?php foreach ($enforcement_modes as $key => $mode): 
                                            $is_selected = ($enforcement_mode === $key);
                                        ?>
                                        <label class="aiox-option-card <?php echo $is_selected ? 'selected' : ''; ?>">
                                            <input type="radio" name="aiox_enforcement_mode" value="<?php echo esc_attr($key); ?>" <?php checked($enforcement_mode, $key); ?>>
                                            <div class="card-icon"><span class="dashicons dashicons-<?php echo esc_attr($mode['icon']); ?>"></span></div>
                                            <div class="card-title"><?php echo esc_html($mode['name']); ?></div>
                                            <div class="card-desc"><?php echo esc_html($mode['description']); ?></div>
                                        </label>
                                        <?php endforeach; ?>
                                    </div>
                                </div>
                            </div>
                            
                            <!-- License Type -->
                            <div class="postbox">
                                <h2 class="hndle"><span class="dashicons dashicons-media-document"></span> <?php _e('License Type', 'aiox-publisher-suite-pro'); ?></h2>
                                <div class="inside">
                                    <p class="description"><?php _e('What can AI systems do with your content?', 'aiox-publisher-suite-pro'); ?></p>
                                    
                                    <div class="aiox-option-cards">
                                        <?php foreach ($license_presets as $key => $preset): 
                                            $is_selected = ($license_preset === $key);
                                        ?>
                                        <label class="aiox-option-card <?php echo $is_selected ? 'selected' : ''; ?>">
                                            <input type="radio" name="aiox_license_preset" value="<?php echo esc_attr($key); ?>" <?php checked($license_preset, $key); ?>>
                                            <div class="card-icon"><span class="dashicons dashicons-<?php echo esc_attr($preset['icon']); ?>"></span></div>
                                            <div class="card-title"><?php echo esc_html($preset['name']); ?></div>
                                            <div class="card-desc"><?php echo esc_html($preset['description']); ?></div>
                                            <div class="card-similar"><?php printf(__('Similar to: %s', 'aiox-publisher-suite-pro'), $preset['similar_to']); ?></div>
                                        </label>
                                        <?php endforeach; ?>
                                    </div>
                                </div>
                            </div>
                            
                            <!-- Monetization Model -->
                            <div class="postbox">
                                <h2 class="hndle"><span class="dashicons dashicons-money-alt"></span> <?php _e('Monetization Model', 'aiox-publisher-suite-pro'); ?></h2>
                                <div class="inside">
                                    <p class="description"><?php _e('How should AI systems pay for access?', 'aiox-publisher-suite-pro'); ?></p>
                                    
                                    <div class="aiox-option-cards">
                                        <?php foreach ($monetization_presets as $key => $preset): 
                                            $is_selected = ($monetization_preset === $key);
                                        ?>
                                        <label class="aiox-option-card <?php echo $is_selected ? 'selected' : ''; ?>">
                                            <input type="radio" name="aiox_monetization_preset" value="<?php echo esc_attr($key); ?>" <?php checked($monetization_preset, $key); ?>>
                                            <div class="card-icon"><span class="dashicons dashicons-<?php echo esc_attr($preset['icon']); ?>"></span></div>
                                            <div class="card-title"><?php echo esc_html($preset['name']); ?></div>
                                            <div class="card-desc"><?php echo esc_html($preset['description']); ?></div>
                                        </label>
                                        <?php endforeach; ?>
                                    </div>
                                </div>
                            </div>
                            
                            <!-- Advanced Settings -->
                            <div class="postbox">
                                <h2 class="hndle"><span class="dashicons dashicons-admin-generic"></span> <?php _e('Advanced Settings', 'aiox-publisher-suite-pro'); ?></h2>
                                <div class="inside">
                                    <table class="form-table">
                                        <tr>
                                            <th scope="row"><label for="aiox_license_contact_email"><?php _e('Licensing Contact', 'aiox-publisher-suite-pro'); ?></label></th>
                                            <td>
                                                <input type="email" id="aiox_license_contact_email" name="aiox_license_contact_email" 
                                                       value="<?php echo esc_attr(get_option('aiox_license_contact_email', get_option('admin_email'))); ?>"
                                                       class="regular-text" placeholder="licensing@yoursite.com">
                                                <p class="description"><?php _e('Email for AI companies to contact about licensing.', 'aiox-publisher-suite-pro'); ?></p>
                                            </td>
                                        </tr>
                                        <tr>
                                            <th scope="row"><label for="aiox_license_terms_url"><?php _e('Terms URL', 'aiox-publisher-suite-pro'); ?></label></th>
                                            <td>
                                                <input type="url" id="aiox_license_terms_url" name="aiox_license_terms_url" 
                                                       value="<?php echo esc_attr(get_option('aiox_license_terms_url', '')); ?>"
                                                       class="regular-text" placeholder="https://yoursite.com/ai-terms">
                                                <p class="description"><?php _e('Link to your AI licensing terms page.', 'aiox-publisher-suite-pro'); ?></p>
                                            </td>
                                        </tr>
                                        <tr>
                                            <th scope="row"><label for="aiox_monetization_license_api"><?php _e('License API', 'aiox-publisher-suite-pro'); ?></label></th>
                                            <td>
                                                <input type="url" id="aiox_monetization_license_api" name="aiox_monetization_license_api" 
                                                       value="<?php echo esc_attr(get_option('aiox_monetization_license_api', home_url('/wp-json/aiox/v1/license'))); ?>"
                                                       class="regular-text">
                                                <p class="description"><?php _e('API endpoint for programmatic license requests.', 'aiox-publisher-suite-pro'); ?></p>
                                            </td>
                                        </tr>
                                        <tr>
                                            <th scope="row"><label for="aiox_license_custom_terms"><?php _e('Custom Terms', 'aiox-publisher-suite-pro'); ?></label></th>
                                            <td>
                                                <textarea id="aiox_license_custom_terms" name="aiox_license_custom_terms" rows="6" class="large-text"
                                                          placeholder="<?php esc_attr_e('Add additional licensing terms here...', 'aiox-publisher-suite-pro'); ?>"><?php echo esc_textarea(get_option('aiox_license_custom_terms', '')); ?></textarea>
                                                <p class="description"><?php _e('Additional terms included in your manifest.', 'aiox-publisher-suite-pro'); ?></p>
                                            </td>
                                        </tr>
                                    </table>
                                </div>
                            </div>
                            
                            <!-- Summary -->
                            <div class="postbox">
                                <h2 class="hndle"><span class="dashicons dashicons-visibility"></span> <?php _e('Current Settings', 'aiox-publisher-suite-pro'); ?></h2>
                                <div class="inside">
                                    <div class="aiox-summary-grid">
                                        <div class="aiox-summary-item">
                                            <span><?php _e('License', 'aiox-publisher-suite-pro'); ?></span>
                                            <strong id="summary-license"><?php echo esc_html($license_presets[$license_preset]['name']); ?></strong>
                                        </div>
                                        <div class="aiox-summary-item">
                                            <span><?php _e('Monetization', 'aiox-publisher-suite-pro'); ?></span>
                                            <strong id="summary-monetization"><?php echo esc_html($monetization_presets[$monetization_preset]['name']); ?></strong>
                                        </div>
                                        <div class="aiox-summary-item">
                                            <span><?php _e('Enforcement', 'aiox-publisher-suite-pro'); ?></span>
                                            <strong id="summary-enforcement"><?php echo esc_html($enforcement_modes[$enforcement_mode]['name']); ?></strong>
                                        </div>
                                        <div class="aiox-summary-item">
                                            <span><?php _e('Posts', 'aiox-publisher-suite-pro'); ?></span>
                                            <strong><?php echo number_format($processed_count); ?></strong>
                                        </div>
                                    </div>
                                </div>
                            </div>
                            
                        </div><!-- /post-body-content -->
                        
                        <!-- Sidebar -->
                        <div id="postbox-container-1" class="postbox-container">
                            <div class="postbox">
                                <h2 class="hndle"><?php _e('Save Settings', 'aiox-publisher-suite-pro'); ?></h2>
                                <div class="inside">
                                    <?php submit_button(__('Save & Regenerate', 'aiox-publisher-suite-pro'), 'primary', 'submit', false); ?>
                                    <p class="description" style="margin-top: 10px;"><?php _e('Manifests auto-regenerate on save.', 'aiox-publisher-suite-pro'); ?></p>
                                </div>
                            </div>
                            
                            <div class="postbox">
                                <h2 class="hndle"><?php _e('Manual Regenerate', 'aiox-publisher-suite-pro'); ?></h2>
                                <div class="inside">
                                    <button type="button" id="aiox-regenerate-manifests" class="button" style="width: 100%;">
                                        <span class="dashicons dashicons-update" style="vertical-align: middle;"></span>
                                        <?php _e('Regenerate Now', 'aiox-publisher-suite-pro'); ?>
                                    </button>
                                    <span id="regenerate-status" style="display: block; margin-top: 10px;"></span>
                                </div>
                            </div>
                            
                            <div class="postbox">
                                <h2 class="hndle"><?php _e('Quick Links', 'aiox-publisher-suite-pro'); ?></h2>
                                <div class="inside">
                                    <ul style="margin: 0;">
                                        <li><a href="<?php echo admin_url('admin.php?page=aiox-settings'); ?>"><?php _e('Settings', 'aiox-publisher-suite-pro'); ?></a></li>
                                        <li><a href="<?php echo home_url('/.well-known/aio.json'); ?>" target="_blank"><?php _e('View Manifest', 'aiox-publisher-suite-pro'); ?> ↗</a></li>
                                    </ul>
                                </div>
                            </div>
                        </div>
                        
                    </div><!-- /post-body -->
                </div><!-- /poststuff -->
            </form>
        </div>
        
        <script type="text/javascript">
        jQuery(document).ready(function($) {
            var presetNames = {
                'aiox_enforcement_mode': {
                    'strict': '<?php _e('Strict Enforcement', 'aiox-publisher-suite-pro'); ?>',
                    'flexible': '<?php _e('Flexible Enforcement', 'aiox-publisher-suite-pro'); ?>'
                },
                'aiox_license_preset': {
                    'open_access': '<?php _e('Open Access', 'aiox-publisher-suite-pro'); ?>',
                    'attribution_required': '<?php _e('Attribution Required', 'aiox-publisher-suite-pro'); ?>',
                    'commercial_license': '<?php _e('Commercial License', 'aiox-publisher-suite-pro'); ?>'
                },
                'aiox_monetization_preset': {
                    'free': '<?php _e('Free Access', 'aiox-publisher-suite-pro'); ?>',
                    'freemium': '<?php _e('Freemium', 'aiox-publisher-suite-pro'); ?>',
                    'paid': '<?php _e('Paid Access', 'aiox-publisher-suite-pro'); ?>'
                }
            };
            
            // Option card selection
            $('.aiox-option-card').on('click', function() {
                var $card = $(this);
                var $radio = $card.find('input[type="radio"]');
                var radioName = $radio.attr('name');
                
                // Reset all cards in this group
                $('input[name="' + radioName + '"]').closest('.aiox-option-card').removeClass('selected');
                
                // Select this card
                $radio.prop('checked', true);
                $card.addClass('selected');
                
                // Update summary
                updateSummary();
            });
            
            function updateSummary() {
                $('#summary-license').text(presetNames['aiox_license_preset'][$('input[name="aiox_license_preset"]:checked').val()]);
                $('#summary-monetization').text(presetNames['aiox_monetization_preset'][$('input[name="aiox_monetization_preset"]:checked').val()]);
                $('#summary-enforcement').text(presetNames['aiox_enforcement_mode'][$('input[name="aiox_enforcement_mode"]:checked').val()]);
            }
            
            // Manual regeneration button
            $('#aiox-regenerate-manifests').on('click', function() {
                var $btn = $(this);
                var $status = $('#regenerate-status');
                var originalHtml = $btn.html();
                
                $btn.prop('disabled', true).html('<span class="dashicons dashicons-update spin" style="vertical-align:middle;"></span> <?php _e('Working...', 'aiox-publisher-suite-pro'); ?>');
                $status.html('<span style="color: #2271b1;"><?php _e('Processing...', 'aiox-publisher-suite-pro'); ?></span>');
                
                $.ajax({
                    url: ajaxurl,
                    type: 'POST',
                    data: {
                        action: 'aiox_regenerate_all_manifests',
                        nonce: '<?php echo wp_create_nonce('aiox_regenerate_manifests'); ?>'
                    },
                    success: function(response) {
                        if (response.success) {
                            $status.html('<span style="color: #00a32a;">✓ ' + response.data.message + '</span>');
                        } else {
                            $status.html('<span style="color: #d63638;">✗ ' + (response.data.message || 'Failed') + '</span>');
                        }
                    },
                    error: function(xhr, status, error) {
                        $status.html('<span style="color: #d63638;">✗ Error: ' + error + '</span>');
                    },
                    complete: function() {
                        $btn.prop('disabled', false).html(originalHtml);
                    }
                });
            });
        });
        </script>
        <?php
    }
    
    /**
     * Get count of processed posts
     */
    private function get_processed_posts_count() {
        global $wpdb;
        
        return (int) $wpdb->get_var(
            "SELECT COUNT(DISTINCT post_id) FROM {$wpdb->postmeta} WHERE meta_key = '_aiox_processed' AND meta_value = '1'"
        );
    }
}
