<?php
/**
 * AIOX Meta Boxes Class
 * 
 * Handles admin meta boxes for enhanced metadata fields
 */

if (!defined('ABSPATH')) {
    exit;
}

class AIOX_Meta_Boxes {
    
    /**
     * Constructor
     */
    public function __construct() {
        add_action('add_meta_boxes', array($this, 'add_meta_boxes'));
        add_action('save_post', array($this, 'save_meta_boxes'), 10, 2);
        add_action('admin_enqueue_scripts', array($this, 'enqueue_admin_scripts'));
    }
    
    /**
     * Add meta boxes
     */
    public function add_meta_boxes() {
        $post_types = array('post', 'page');
        
        foreach ($post_types as $post_type) {
            add_meta_box(
                'aiox_metadata',
                __('AIOX Content Metadata', 'aiox-publisher-suite-pro'),
                array($this, 'render_metadata_meta_box'),
                $post_type,
                'normal',
                'default'
            );
            
            add_meta_box(
                'aiox_licensing',
                __('AIOX Licensing & Rights', 'aiox-publisher-suite-pro'),
                array($this, 'render_licensing_meta_box'),
                $post_type,
                'side',
                'default'
            );
            
            add_meta_box(
                'aiox_ingredients_preview',
                __('AIOX Ingredients Preview', 'aiox-publisher-suite-pro'),
                array($this, 'render_ingredients_preview_meta_box'),
                $post_type,
                'normal',
                'low'
            );
        }
    }
    
    /**
     * Enqueue admin scripts
     */
    public function enqueue_admin_scripts($hook) {
        if (!in_array($hook, array('post.php', 'post-new.php'))) {
            return;
        }
        
        wp_enqueue_script(
            'aiox-meta-boxes',
            AIOX_PLUGIN_URL . 'assets/js/meta-boxes.js',
            array('jquery'),
            AIOX_VERSION,
            true
        );
        
        wp_enqueue_style(
            'aiox-meta-boxes',
            AIOX_PLUGIN_URL . 'assets/css/meta-boxes.css',
            array(),
            AIOX_VERSION
        );
        
        wp_localize_script('aiox-meta-boxes', 'aiox_meta', array(
            'ajax_url' => admin_url('admin-ajax.php'),
            'nonce' => wp_create_nonce('aiox_meta_nonce'),
            'strings' => array(
                'auto_classify' => __('Auto-Classify', 'aiox-publisher-suite-pro'),
                'classifying' => __('Classifying...', 'aiox-publisher-suite-pro'),
                'preview_ingredients' => __('Preview Ingredients', 'aiox-publisher-suite-pro'),
                'generating_preview' => __('Generating Preview...', 'aiox-publisher-suite-pro')
            )
        ));
    }
    
    /**
     * Render metadata meta box
     */
    public function render_metadata_meta_box($post) {
        wp_nonce_field('aiox_metadata_meta_box', 'aiox_metadata_nonce');
        
        $metadata_fields = AIOX_Metadata::get_metadata_fields();
        $current_metadata = AIOX_Metadata::get_post_metadata($post->ID);
        
        ?>
        <div class="aiox-metadata-container">
            <div class="aiox-metadata-header">
                <p><?php _e('Configure how AI systems should understand and categorize this content. These fields are automatically populated when you process the post with AI.', 'aiox-publisher-suite-pro'); ?></p>
                <p class="description" style="margin-top: 10px;">
                    <?php _e('💡 <strong>Tip:</strong> Use the <strong>"Process"</strong> button in the "AIOX Processing" box to automatically classify this content using AI. You can also manually configure the fields below.', 'aiox-publisher-suite-pro'); ?>
                </p>
            </div>
            
            <div class="aiox-metadata-grid">
                <?php foreach ($metadata_fields as $field_name => $field_config): ?>
                    <div class="aiox-metadata-field">
                        <label for="aiox_<?php echo esc_attr($field_name); ?>">
                            <?php echo esc_html($field_config['label']); ?>
                            <?php if (isset($field_config['description'])): ?>
                                <span class="aiox-help-icon" title="<?php echo esc_attr($field_config['description']); ?>">?</span>
                            <?php endif; ?>
                        </label>
                        
                        <?php if ($field_config['type'] === 'select'): ?>
                            <select name="aiox_<?php echo esc_attr($field_name); ?>" id="aiox_<?php echo esc_attr($field_name); ?>">
                                <?php foreach ($field_config['options'] as $value => $label): ?>
                                    <option value="<?php echo esc_attr($value); ?>" <?php selected($current_metadata[$field_name], $value); ?>>
                                        <?php echo esc_html($label); ?>
                                    </option>
                                <?php endforeach; ?>
                            </select>
                            
                        <?php elseif ($field_config['type'] === 'multiselect'): ?>
                            <div class="aiox-multiselect">
                                <?php 
                                $current_values = is_array($current_metadata[$field_name]) ? $current_metadata[$field_name] : array();
                                foreach ($field_config['options'] as $value => $label): 
                                ?>
                                    <label class="aiox-checkbox-label">
                                        <input type="checkbox" 
                                               name="aiox_<?php echo esc_attr($field_name); ?>[]" 
                                               value="<?php echo esc_attr($value); ?>"
                                               <?php checked(in_array($value, $current_values)); ?> />
                                        <?php echo esc_html($label); ?>
                                    </label>
                                <?php endforeach; ?>
                            </div>
                            
                        <?php elseif ($field_config['type'] === 'text'): ?>
                            <input type="text" 
                                   name="aiox_<?php echo esc_attr($field_name); ?>" 
                                   id="aiox_<?php echo esc_attr($field_name); ?>"
                                   value="<?php echo esc_attr($current_metadata[$field_name]); ?>"
                                   placeholder="<?php echo esc_attr($field_config['placeholder'] ?? ''); ?>" />
                        <?php endif; ?>
                    </div>
                <?php endforeach; ?>
            </div>
        </div>
        
        <style>
        .aiox-metadata-container { margin: 15px 0; }
        .aiox-metadata-header { margin-bottom: 20px; padding-bottom: 15px; border-bottom: 1px solid #ddd; }
        .aiox-metadata-grid { display: grid; grid-template-columns: repeat(auto-fit, minmax(300px, 1fr)); gap: 20px; }
        .aiox-metadata-field label { display: block; font-weight: 600; margin-bottom: 5px; }
        .aiox-metadata-field select, .aiox-metadata-field input[type="text"] { width: 100%; }
        .aiox-multiselect { max-height: 150px; overflow-y: auto; border: 1px solid #ddd; padding: 10px; }
        .aiox-checkbox-label { display: block; margin-bottom: 5px; font-weight: normal; }
        .aiox-help-icon { display: inline-block; width: 16px; height: 16px; background: #666; color: white; border-radius: 50%; text-align: center; font-size: 12px; line-height: 16px; margin-left: 5px; cursor: help; }
        </style>
        <?php
    }
    
    /**
     * Render licensing meta box
     */
    public function render_licensing_meta_box($post) {
        wp_nonce_field('aiox_licensing_meta_box', 'aiox_licensing_nonce');
        
        $licensing_metadata = AIOX_Metadata::get_licensing_metadata($post->ID);
        
        ?>
        <div class="aiox-licensing-container">
            <p><strong><?php _e('AI Training & Usage Rights', 'aiox-publisher-suite-pro'); ?></strong></p>
            
            <div class="aiox-licensing-field">
                <label>
                    <input type="checkbox" 
                           name="aiox_training_allowed" 
                           value="1" 
                           <?php checked($licensing_metadata['training_allowed'], true); ?> />
                    <?php _e('Allow AI training on this content', 'aiox-publisher-suite-pro'); ?>
                </label>
            </div>
            
            <div class="aiox-licensing-field">
                <label>
                    <input type="checkbox" 
                           name="aiox_citation_required" 
                           value="1" 
                           <?php checked($licensing_metadata['citation_required'], true); ?> />
                    <?php _e('Require citation when used', 'aiox-publisher-suite-pro'); ?>
                </label>
            </div>
            
            <div class="aiox-licensing-field">
                <label for="aiox_attribution_string"><?php _e('Attribution String:', 'aiox-publisher-suite-pro'); ?></label>
                <input type="text" 
                       name="aiox_attribution_string" 
                       id="aiox_attribution_string"
                       value="<?php echo esc_attr($licensing_metadata['attribution_string']); ?>"
                       placeholder="<?php echo esc_attr(get_bloginfo('name')); ?>" />
            </div>
            
            <div class="aiox-licensing-field">
                <label for="aiox_license_url"><?php _e('License URL:', 'aiox-publisher-suite-pro'); ?></label>
                <input type="url" 
                       name="aiox_license_url" 
                       id="aiox_license_url"
                       value="<?php echo esc_attr($licensing_metadata['license_url']); ?>"
                       placeholder="https://creativecommons.org/licenses/..." />
            </div>
            
            <div class="aiox-licensing-field">
                <label for="aiox_expiration_date"><?php _e('Content Expiration:', 'aiox-publisher-suite-pro'); ?></label>
                <input type="date" 
                       name="aiox_expiration_date" 
                       id="aiox_expiration_date"
                       value="<?php echo esc_attr($licensing_metadata['expiration_date']); ?>" />
                <small><?php _e('Leave blank for no expiration', 'aiox-publisher-suite-pro'); ?></small>
            </div>
        </div>
        
        <style>
        .aiox-licensing-container { margin: 15px 0; }
        .aiox-licensing-field { margin-bottom: 15px; }
        .aiox-licensing-field label { display: block; font-weight: 600; margin-bottom: 5px; }
        .aiox-licensing-field input[type="text"], .aiox-licensing-field input[type="url"], .aiox-licensing-field input[type="date"] { width: 100%; }
        .aiox-licensing-field small { color: #666; font-style: italic; }
        </style>
        <?php
    }
    
    /**
     * Render ingredients preview meta box
     */
    public function render_ingredients_preview_meta_box($post) {
        $ingredients_summary = AIOX_Ingredients::get_post_ingredients_summary($post->ID);
        
        ?>
        <div class="aiox-ingredients-preview">
            <?php if (!empty($ingredients_summary)): ?>
                <div class="aiox-ingredients-stats">
                    <div class="aiox-stat">
                        <span class="aiox-stat-number"><?php echo esc_html($ingredients_summary['total_count'] ?? 0); ?></span>
                        <span class="aiox-stat-label"><?php _e('Total Ingredients', 'aiox-publisher-suite-pro'); ?></span>
                    </div>
                    <div class="aiox-stat">
                        <span class="aiox-stat-number"><?php echo esc_html($ingredients_summary['total_words'] ?? 0); ?></span>
                        <span class="aiox-stat-label"><?php _e('Total Words', 'aiox-publisher-suite-pro'); ?></span>
                    </div>
                    <div class="aiox-stat">
                        <span class="aiox-stat-number"><?php echo esc_html(count($ingredients_summary['types'] ?? array())); ?></span>
                        <span class="aiox-stat-label"><?php _e('Ingredient Types', 'aiox-publisher-suite-pro'); ?></span>
                    </div>
                </div>
                
                <?php if (!empty($ingredients_summary['types'])): ?>
                    <div class="aiox-ingredients-breakdown">
                        <h4><?php _e('Ingredient Breakdown:', 'aiox-publisher-suite-pro'); ?></h4>
                        <ul>
                            <?php foreach ($ingredients_summary['types'] as $type => $count): ?>
                                <li>
                                    <strong><?php echo esc_html(ucfirst(str_replace('_', ' ', $type))); ?>:</strong> 
                                    <?php echo esc_html($count); ?>
                                </li>
                            <?php endforeach; ?>
                        </ul>
                    </div>
                <?php endif; ?>
                
                <p class="aiox-last-updated">
                    <em><?php printf(__('Last updated: %s', 'aiox-publisher-suite-pro'), esc_html($ingredients_summary['last_updated'] ?? 'Never')); ?></em>
                </p>
            <?php else: ?>
                <p><?php _e('No ingredients generated yet. Save or update this post to generate ingredients.', 'aiox-publisher-suite-pro'); ?></p>
            <?php endif; ?>
            
            <button type="button" id="aiox-preview-ingredients" class="button button-secondary">
                <?php _e('Preview Ingredients', 'aiox-publisher-suite-pro'); ?>
            </button>
        </div>
        
        <style>
        .aiox-ingredients-preview { margin: 15px 0; }
        .aiox-ingredients-stats { display: flex; gap: 20px; margin-bottom: 20px; }
        .aiox-stat { text-align: center; }
        .aiox-stat-number { display: block; font-size: 24px; font-weight: bold; color: #0073aa; }
        .aiox-stat-label { display: block; font-size: 12px; color: #666; }
        .aiox-ingredients-breakdown ul { margin: 10px 0; }
        .aiox-ingredients-breakdown li { margin-bottom: 5px; }
        .aiox-last-updated { color: #666; font-size: 12px; margin-top: 15px; }
        </style>
        <?php
    }
    
    /**
     * Save meta box data
     */
    public function save_meta_boxes($post_id, $post) {
        // Check if this is an autosave
        if (defined('DOING_AUTOSAVE') && DOING_AUTOSAVE) {
            return;
        }
        
        // Check user permissions
        if (!current_user_can('edit_post', $post_id)) {
            return;
        }
        
        // Save metadata
        if (isset($_POST['aiox_metadata_nonce']) && wp_verify_nonce($_POST['aiox_metadata_nonce'], 'aiox_metadata_meta_box')) {
            $metadata_fields = AIOX_Metadata::get_metadata_fields();
            $metadata = array();
            
            foreach ($metadata_fields as $field_name => $field_config) {
                if (isset($_POST['aiox_' . $field_name])) {
                    if ($field_config['type'] === 'multiselect') {
                        $metadata[$field_name] = array_map('sanitize_text_field', $_POST['aiox_' . $field_name]);
                    } else {
                        $metadata[$field_name] = sanitize_text_field($_POST['aiox_' . $field_name]);
                    }
                } else {
                    $metadata[$field_name] = $field_config['default'];
                }
            }
            
            AIOX_Metadata::save_post_metadata($post_id, $metadata);
        }
        
        // Save licensing data
        if (isset($_POST['aiox_licensing_nonce']) && wp_verify_nonce($_POST['aiox_licensing_nonce'], 'aiox_licensing_meta_box')) {
            update_post_meta($post_id, '_aiox_training_allowed', isset($_POST['aiox_training_allowed']) ? 1 : 0);
            update_post_meta($post_id, '_aiox_citation_required', isset($_POST['aiox_citation_required']) ? 1 : 0);
            update_post_meta($post_id, '_aiox_attribution_string', sanitize_text_field($_POST['aiox_attribution_string'] ?? ''));
            update_post_meta($post_id, '_aiox_license_url', esc_url_raw($_POST['aiox_license_url'] ?? ''));
            update_post_meta($post_id, '_aiox_expiration_date', sanitize_text_field($_POST['aiox_expiration_date'] ?? ''));
        }
    }
}
