<?php
/**
 * AIOX Enhanced Modern Analytics Admin
 * Beautiful blue/purple interface with comprehensive AI tracking
 * 
 * @package AIOX_Publisher_Suite_Pro
 * @version 5.4.0
 */

if (!defined('ABSPATH')) {
    exit;
}

class AIOX_Modern_Analytics_Admin {
    private static $instance = null;
    private static $hooks_added = false;

    public static function get_instance() {
        if (self::$instance === null) {
            self::$instance = new self();
        }
        return self::$instance;
    }

    private function __construct() {
        if (!self::$hooks_added) {
            add_action('admin_menu', [$this, 'add_admin_menu'], 99);
            add_action('admin_enqueue_scripts', [$this, 'enqueue_assets']);
            add_action('wp_ajax_aiox_modern_analytics_data', [$this, 'ajax_get_data']);
            add_action('wp_ajax_aiox_modern_analytics_export', [$this, 'ajax_export']);
            add_action('wp_ajax_aiox_modern_analytics_reset', [$this, 'ajax_reset']);
            add_action('wp_ajax_aiox_get_visitor_logs', [$this, 'ajax_get_visitor_logs']);
            self::$hooks_added = true;
        }
    }

    private function __clone() {}
    
    public function __wakeup() {
        throw new Exception("Cannot unserialize singleton");
    }

    public function enqueue_assets($hook) {
        if (strpos($hook, 'aiox-modern-analytics') === false) {
            return;
        }

        wp_enqueue_script('chartjs', 'https://cdn.jsdelivr.net/npm/chart.js@4.4.0/dist/chart.umd.js', [], '4.4.0', true);
        
        wp_enqueue_script(
            'aiox-modern-analytics',
            AIOX_PLUGIN_URL . 'assets/js/modern-analytics.js',
            ['jquery', 'chartjs'],
            AIOX_VERSION,
            true
        );

        wp_localize_script('aiox-modern-analytics', 'aioxModernAnalytics', [
            'ajaxUrl' => admin_url('admin-ajax.php'),
            'nonce' => wp_create_nonce('aiox_modern_analytics_nonce')
        ]);

        wp_enqueue_style(
            'aiox-modern-analytics',
            AIOX_PLUGIN_URL . 'assets/css/analytics-native.css',
            [],
            AIOX_VERSION
        );
    }

    public function add_admin_menu() {
        global $submenu;
        if (isset($submenu['aiox-dashboard'])) {
            foreach ($submenu['aiox-dashboard'] as $key => $item) {
                if ($item[2] === 'aiox-modern-analytics') {
                    unset($submenu['aiox-dashboard'][$key]);
                }
            }
        }

        add_submenu_page(
            'aiox-dashboard',
            __('AIOX Analytics', 'aiox-publisher-suite-pro'),
            __('📊 Analytics', 'aiox-publisher-suite-pro'),
            'manage_options',
            'aiox-modern-analytics',
            [$this, 'render_admin_page']
        );
    }

    public function ajax_get_data() {
        check_ajax_referer('aiox_modern_analytics_nonce', 'nonce');

        if (!current_user_can('manage_options')) {
            wp_send_json_error(['message' => 'Unauthorized']);
        }

        $tab = sanitize_text_field($_POST['tab'] ?? 'overview');
        $days = intval($_POST['days'] ?? 30);

        $analytics = AIOX_Modern_Analytics::get_instance();
        $data = [];

        switch ($tab) {
            case 'overview':
                $data = $analytics->get_overview_data($days);
                break;

            case 'crawlers':
                $data = [
                    'type_distribution' => $analytics->get_crawler_types($days),
                    'top_crawlers' => $analytics->get_top_crawlers($days)
                ];
                break;

            case 'content':
                $data = [
                    'content_performance' => $analytics->get_content_performance($days),
                    'top_posts' => $analytics->get_top_posts($days)
                ];
                break;

            case 'activity':
                $data = [
                    'hourly_activity' => $analytics->get_hourly_activity($days),
                    'top_pages' => $analytics->get_top_pages($days)
                ];
                break;
                
            case 'badges':
                $data = $analytics->get_badge_performance($days);
                break;
                
            case 'wellknown':
                $data = $analytics->get_wellknown_performance($days);
                break;

            default:
                wp_send_json_error(['message' => 'Invalid tab']);
                return;
        }

        wp_send_json_success($data);
    }

    public function ajax_export() {
        check_ajax_referer('aiox_modern_analytics_nonce', 'nonce');

        if (!current_user_can('manage_options')) {
            wp_send_json_error(['message' => 'Unauthorized']);
        }

        $format = sanitize_text_field($_POST['format'] ?? 'csv');
        $days = intval($_POST['days'] ?? 30);

        $analytics = AIOX_Modern_Analytics::get_instance();
        $data = $analytics->get_overview_data($days);

        if ($format === 'json') {
            $output = wp_json_encode($data, JSON_PRETTY_PRINT);
            $mime_type = 'application/json';
            $filename = 'aiox-analytics-' . date('Y-m-d') . '.json';
        } else {
            $output = "Date,Total Visits,AI Visits,Search Visits,AIOX File Visits,Badge Views\n";
            foreach ($data['daily_activity'] as $day) {
                $output .= sprintf(
                    "%s,%d,%d,%d,%d,%d\n",
                    $day['date'],
                    $day['total_visits'],
                    $day['ai_visits'],
                    $day['search_visits'],
                    $day['well_known_visits'],
                    $day['badge_views']
                );
            }
            $mime_type = 'text/csv';
            $filename = 'aiox-analytics-' . date('Y-m-d') . '.csv';
        }

        wp_send_json_success([
            'data' => base64_encode($output),
            'filename' => $filename,
            'mime_type' => $mime_type
        ]);
    }

    /**
     * AJAX handler to reset all analytics data
     */
    public function ajax_reset() {
        check_ajax_referer('aiox_modern_analytics_nonce', 'nonce');

        if (!current_user_can('manage_options')) {
            wp_send_json_error(['message' => 'Unauthorized']);
        }

        global $wpdb;
        
        // Truncate all analytics tables
        $tables = [
            $wpdb->prefix . 'aiox_server_analytics',
            $wpdb->prefix . 'aiox_server_daily_stats',
            $wpdb->prefix . 'aiox_server_crawlers',
            $wpdb->prefix . 'aiox_badge_views',
            $wpdb->prefix . 'aiox_wellknown_hits'
        ];
        
        $errors = [];
        $reset_count = 0;
        
        foreach ($tables as $table) {
            // Check if table exists first
            $table_exists = $wpdb->get_var("SHOW TABLES LIKE '{$table}'") === $table;
            
            if ($table_exists) {
                $result = $wpdb->query("TRUNCATE TABLE {$table}");
                if ($result === false) {
                    $errors[] = basename($table) . ' (truncate failed)';
                } else {
                    $reset_count++;
                }
            } else {
                // Table doesn't exist - not an error, just skip
                $reset_count++;
            }
        }
        
        if (!empty($errors)) {
            wp_send_json_error(['message' => 'Failed to reset some tables: ' . implode(', ', $errors)]);
        }
        
        wp_send_json_success(['message' => "All analytics data has been reset successfully. ({$reset_count} tables cleared)"]);
    }
    
    /**
     * AJAX handler to get visitor logs from database
     */
    public function ajax_get_visitor_logs() {
        check_ajax_referer('aiox_modern_analytics_nonce', 'nonce');

        if (!current_user_can('manage_options')) {
            wp_send_json_error(['message' => 'Unauthorized']);
        }

        global $wpdb;
        
        $page = isset($_POST['page']) ? intval($_POST['page']) : 1;
        $per_page = isset($_POST['per_page']) ? intval($_POST['per_page']) : 50;
        $type_filter = isset($_POST['type']) ? sanitize_text_field($_POST['type']) : '';
        
        $offset = ($page - 1) * $per_page;
        $table = $wpdb->prefix . 'aiox_server_analytics';
        
        // Check if table exists
        if ($wpdb->get_var("SHOW TABLES LIKE '{$table}'") !== $table) {
            wp_send_json_success([
                'logs' => [],
                'total' => 0,
                'page' => $page,
                'per_page' => $per_page,
                'message' => 'Analytics table not found'
            ]);
            return;
        }
        
        // Build query
        $where = "1=1";
        if (!empty($type_filter)) {
            // We need to search in the serialized request_data
            $where .= $wpdb->prepare(" AND request_data LIKE %s", '%"crawler_type";s:' . strlen($type_filter) . ':"' . $type_filter . '"%');
        }
        
        // Get total count
        $total = $wpdb->get_var("SELECT COUNT(*) FROM {$table} WHERE {$where}");
        
        // Get logs
        $logs = $wpdb->get_results(
            "SELECT id, request_time, endpoint, request_data 
             FROM {$table} 
             WHERE {$where}
             ORDER BY request_time DESC 
             LIMIT {$per_page} OFFSET {$offset}"
        );
        
        // Parse logs
        $parsed_logs = [];
        foreach ($logs as $log) {
            $data = maybe_unserialize($log->request_data);
            $parsed_logs[] = [
                'id' => $log->id,
                'time' => $log->request_time,
                'endpoint' => $log->endpoint,
                'ip' => isset($data['ip']) ? substr($data['ip'], 0, 20) : 'N/A',
                'user_agent' => isset($data['agent']) ? substr($data['agent'], 0, 150) : 'N/A',
                'type' => isset($data['crawler_type']) ? $data['crawler_type'] : 'unknown',
                'name' => isset($data['crawler_name']) ? $data['crawler_name'] : 'Unknown',
                'referrer' => isset($data['referrer']) ? $data['referrer'] : ''
            ];
        }
        
        wp_send_json_success([
            'logs' => $parsed_logs,
            'total' => intval($total),
            'page' => $page,
            'per_page' => $per_page,
            'total_pages' => ceil($total / $per_page)
        ]);
    }

    public function render_admin_page() {
        ?>
        <div class="wrap aiox-analytics-enhanced">
            <!-- Header -->
            <div class="aiox-analytics-header">
                <div class="aiox-header-content">
                    <img src="<?php echo esc_url(AIOX_PLUGIN_URL . 'assets/images/aiox-logo.png'); ?>" alt="AIOX" style="width: 40px; height: 40px; margin-right: 12px;">
                    <div>
                        <h1 style="margin: 0;">
                            <?php _e('AI Analytics Dashboard', 'aiox-publisher-suite-pro'); ?>
                        </h1>
                        <p class="aiox-subtitle" style="margin: 4px 0 0 0;">
                            <?php _e('Comprehensive tracking of AI crawlers, badges, and AIOX system files', 'aiox-publisher-suite-pro'); ?>
                        </p>
                    </div>
                </div>
                
                <div class="aiox-header-actions">
                    <div class="aiox-period-selector">
                        <label for="aiox-analytics-period">
                            <span class="dashicons dashicons-calendar-alt"></span>
                            <?php _e('Period:', 'aiox-publisher-suite-pro'); ?>
                        </label>
                        <select id="aiox-analytics-period" class="aiox-select">
                            <option value="1"><?php _e('Last 24 hours', 'aiox-publisher-suite-pro'); ?></option>
                            <option value="7"><?php _e('Last 7 days', 'aiox-publisher-suite-pro'); ?></option>
                            <option value="30" selected><?php _e('Last 30 days', 'aiox-publisher-suite-pro'); ?></option>
                            <option value="90"><?php _e('Last 90 days', 'aiox-publisher-suite-pro'); ?></option>
                        </select>
                    </div>
                    <button type="button" id="aiox-analytics-refresh" class="aiox-btn aiox-btn-primary">
                        <span class="dashicons dashicons-update"></span>
                        <?php _e('Refresh', 'aiox-publisher-suite-pro'); ?>
                    </button>
                    <button type="button" id="aiox-analytics-export" class="aiox-btn aiox-btn-secondary">
                        <span class="dashicons dashicons-download"></span>
                        <?php _e('Export', 'aiox-publisher-suite-pro'); ?>
                    </button>
                    <button type="button" id="aiox-analytics-reset" class="aiox-btn aiox-btn-danger" style="background: #d63638; border-color: #d63638;">
                        <span class="dashicons dashicons-trash"></span>
                        <?php _e('Reset', 'aiox-publisher-suite-pro'); ?>
                    </button>
                </div>
            </div>

            <!-- Navigation Tabs -->
            <nav class="aiox-analytics-tabs">
                <a href="#overview" class="aiox-tab active" data-tab="overview">
                    <span class="dashicons dashicons-dashboard"></span>
                    <span><?php _e('Overview', 'aiox-publisher-suite-pro'); ?></span>
                </a>
                <a href="#crawlers" class="aiox-tab" data-tab="crawlers">
                    <span class="dashicons dashicons-admin-site-alt3"></span>
                    <span><?php _e('AI Crawlers', 'aiox-publisher-suite-pro'); ?></span>
                </a>
                <a href="#badges" class="aiox-tab" data-tab="badges">
                    <span class="dashicons dashicons-awards"></span>
                    <span><?php _e('Badge Performance', 'aiox-publisher-suite-pro'); ?></span>
                </a>
                <a href="#wellknown" class="aiox-tab" data-tab="wellknown">
                    <span class="dashicons dashicons-admin-links"></span>
                    <span><?php _e('AIOX System Files', 'aiox-publisher-suite-pro'); ?></span>
                </a>
                <a href="#content" class="aiox-tab" data-tab="content">
                    <span class="dashicons dashicons-admin-page"></span>
                    <span><?php _e('Content', 'aiox-publisher-suite-pro'); ?></span>
                </a>
                <a href="#activity" class="aiox-tab" data-tab="activity">
                    <span class="dashicons dashicons-clock"></span>
                    <span><?php _e('Activity', 'aiox-publisher-suite-pro'); ?></span>
                </a>
                <a href="#visitorlogs" class="aiox-tab" data-tab="visitorlogs">
                    <span class="dashicons dashicons-list-view"></span>
                    <span><?php _e('Visitor Logs', 'aiox-publisher-suite-pro'); ?></span>
                </a>
            </nav>

            <!-- Loading Overlay -->
            <div id="aiox-analytics-loading" class="aiox-loading-overlay" style="display:none;">
                <div class="aiox-spinner-wrapper">
                    <div class="aiox-spinner"></div>
                    <p><?php _e('Loading analytics...', 'aiox-publisher-suite-pro'); ?></p>
                </div>
            </div>

            <!-- Overview Tab -->
            <div id="tab-overview" class="aiox-tab-content active">
                <!-- Key Metrics -->
                <div class="aiox-metrics-grid">
                    <div class="aiox-metric-card gradient-blue">
                        <div class="aiox-metric-icon">
                            <span class="dashicons dashicons-desktop"></span>
                        </div>
                        <div class="aiox-metric-content">
                            <div class="aiox-metric-value" id="human-visits">-</div>
                            <div class="aiox-metric-label"><?php _e('Browser Visits', 'aiox-publisher-suite-pro'); ?>
                                <span class="aiox-info-tooltip" title="<?php esc_attr_e('Visits from standard web browsers (Chrome, Firefox, Safari, Edge, etc.) with valid browser signatures. May include sophisticated bots that mimic browser behavior.', 'aiox-publisher-suite-pro'); ?>">
                                    <span class="dashicons dashicons-info-outline"></span>
                                </span>
                            </div>
                        </div>
                    </div>

                    <div class="aiox-metric-card gradient-purple">
                        <div class="aiox-metric-icon">
                            <span class="dashicons dashicons-admin-site-alt3"></span>
                        </div>
                        <div class="aiox-metric-content">
                            <div class="aiox-metric-value" id="ai-visits">-</div>
                            <div class="aiox-metric-label"><?php _e('AI Crawler Visits', 'aiox-publisher-suite-pro'); ?></div>
                        </div>
                    </div>

                    <div class="aiox-metric-card gradient-green">
                        <div class="aiox-metric-icon">
                            <span class="dashicons dashicons-search"></span>
                        </div>
                        <div class="aiox-metric-content">
                            <div class="aiox-metric-value" id="search-visits">-</div>
                            <div class="aiox-metric-label"><?php _e('Search Engine Visits', 'aiox-publisher-suite-pro'); ?></div>
                        </div>
                    </div>

                    <div class="aiox-metric-card gradient-orange">
                        <div class="aiox-metric-icon">
                            <span class="dashicons dashicons-networking"></span>
                        </div>
                        <div class="aiox-metric-content">
                            <div class="aiox-metric-value" id="other-bot-visits">-</div>
                            <div class="aiox-metric-label"><?php _e('Other Bots', 'aiox-publisher-suite-pro'); ?>
                                <span class="aiox-info-tooltip" title="<?php esc_attr_e('Social media crawlers, SEO tools, monitoring services, feed readers, security scanners, and other identified bots. Note: This breakdown is new - historical data will show in Unidentified.', 'aiox-publisher-suite-pro'); ?>">
                                    <span class="dashicons dashicons-info-outline"></span>
                                </span>
                            </div>
                        </div>
                    </div>

                    <div class="aiox-metric-card gradient-gray">
                        <div class="aiox-metric-icon">
                            <span class="dashicons dashicons-warning"></span>
                        </div>
                        <div class="aiox-metric-content">
                            <div class="aiox-metric-value" id="unidentified-visits">-</div>
                            <div class="aiox-metric-label"><?php _e('Unidentified', 'aiox-publisher-suite-pro'); ?>
                                <span class="aiox-info-tooltip" title="<?php esc_attr_e('Traffic not matching known patterns. For historical data (before this update), this includes all non-AI/non-Search traffic. Going forward, this only shows truly unidentifiable requests. Unlike Google Analytics, server-side tracking captures ALL HTTP requests including bots.', 'aiox-publisher-suite-pro'); ?>">
                                    <span class="dashicons dashicons-info-outline"></span>
                                </span>
                            </div>
                        </div>
                    </div>

                    <div class="aiox-metric-card gradient-teal">
                        <div class="aiox-metric-icon">
                            <span class="dashicons dashicons-chart-area"></span>
                        </div>
                        <div class="aiox-metric-content">
                            <div class="aiox-metric-value" id="ai-percentage">-</div>
                            <div class="aiox-metric-label"><?php _e('AI Traffic %', 'aiox-publisher-suite-pro'); ?></div>
                        </div>
                    </div>
                </div>
                
                <!-- Secondary Metrics Row -->
                <div class="aiox-metrics-grid aiox-metrics-secondary" style="margin-top: 15px;">
                    <div class="aiox-metric-card-small">
                        <div class="aiox-metric-icon-small">
                            <span class="dashicons dashicons-visibility"></span>
                        </div>
                        <div class="aiox-metric-content">
                            <div class="aiox-metric-value-small" id="total-visits">-</div>
                            <div class="aiox-metric-label-small"><?php _e('Total Requests', 'aiox-publisher-suite-pro'); ?></div>
                        </div>
                    </div>
                    
                    <div class="aiox-metric-card-small">
                        <div class="aiox-metric-icon-small">
                            <span class="dashicons dashicons-admin-links"></span>
                        </div>
                        <div class="aiox-metric-content">
                            <div class="aiox-metric-value-small" id="well-known-requests">-</div>
                            <div class="aiox-metric-label-small"><?php _e('AIOX File Requests', 'aiox-publisher-suite-pro'); ?></div>
                        </div>
                    </div>
                    
                    <div class="aiox-metric-card-small">
                        <div class="aiox-metric-icon-small">
                            <span class="dashicons dashicons-awards"></span>
                        </div>
                        <div class="aiox-metric-content">
                            <div class="aiox-metric-value-small" id="badge-views">-</div>
                            <div class="aiox-metric-label-small"><?php _e('Badge Views', 'aiox-publisher-suite-pro'); ?></div>
                        </div>
                    </div>
                </div>

                <!-- Charts -->
                <div class="aiox-charts-grid">
                    <div class="aiox-chart-card">
                        <h3 id="activity-chart-title">
                            <span class="dashicons dashicons-chart-line"></span>
                            <span id="activity-chart-title-text"><?php _e('Activity Trends', 'aiox-publisher-suite-pro'); ?></span>
                        </h3>
                        <div class="aiox-chart-container">
                            <canvas id="daily-activity-chart"></canvas>
                        </div>
                    </div>

                    <div class="aiox-chart-card">
                        <h3>
                            <span class="dashicons dashicons-chart-pie"></span>
                            <?php _e('Top AI Crawlers', 'aiox-publisher-suite-pro'); ?>
                        </h3>
                        <div class="aiox-chart-container">
                            <canvas id="top-crawlers-chart"></canvas>
                        </div>
                    </div>
                </div>
            </div>

            <!-- Crawlers Tab -->
            <div id="tab-crawlers" class="aiox-tab-content">
                <div class="aiox-charts-grid">
                    <div class="aiox-chart-card">
                        <h3>
                            <span class="dashicons dashicons-chart-bar"></span>
                            <?php _e('Crawler Types Distribution', 'aiox-publisher-suite-pro'); ?>
                        </h3>
                        <div class="aiox-chart-container">
                            <canvas id="crawler-types-chart"></canvas>
                        </div>
                    </div>
                </div>

                <div class="aiox-table-card">
                    <h3>
                        <span class="dashicons dashicons-list-view"></span>
                        <?php _e('Detailed Crawler Statistics', 'aiox-publisher-suite-pro'); ?>
                    </h3>
                    <div class="aiox-table-wrapper">
                        <table id="crawlers-table" class="aiox-table">
                            <thead>
                                <tr>
                                    <th><?php _e('Crawler Name', 'aiox-publisher-suite-pro'); ?></th>
                                    <th><?php _e('Type', 'aiox-publisher-suite-pro'); ?></th>
                                    <th><?php _e('Total Visits', 'aiox-publisher-suite-pro'); ?></th>
                                    <th><?php _e('Last Seen', 'aiox-publisher-suite-pro'); ?></th>
                                </tr>
                            </thead>
                            <tbody>
                                <tr>
                                    <td colspan="4" class="aiox-loading-cell">
                                        <div class="aiox-spinner-small"></div>
                                        <?php _e('Loading...', 'aiox-publisher-suite-pro'); ?>
                                    </td>
                                </tr>
                            </tbody>
                        </table>
                    </div>
                </div>
            </div>

            <!-- Badge Performance Tab -->
            <div id="tab-badges" class="aiox-tab-content">
                <div class="aiox-info-banner">
                    <span class="dashicons dashicons-info"></span>
                    <div>
                        <strong><?php _e('Badge Performance Tracking', 'aiox-publisher-suite-pro'); ?></strong>
                        <p><?php _e('Monitor how AI crawlers and browsers interact with your AI-optimized badges on content pages.', 'aiox-publisher-suite-pro'); ?></p>
                    </div>
                </div>

                <div class="aiox-charts-grid">
                    <div class="aiox-chart-card">
                        <h3>
                            <span class="dashicons dashicons-chart-pie"></span>
                            <?php _e('Badge Views by Viewer Type', 'aiox-publisher-suite-pro'); ?>
                        </h3>
                        <div class="aiox-chart-container">
                            <canvas id="badge-viewer-types-chart"></canvas>
                        </div>
                    </div>

                    <div class="aiox-chart-card">
                        <h3>
                            <span class="dashicons dashicons-chart-line"></span>
                            <?php _e('Daily Badge Views', 'aiox-publisher-suite-pro'); ?>
                        </h3>
                        <div class="aiox-chart-container">
                            <canvas id="badge-daily-chart"></canvas>
                        </div>
                    </div>
                </div>

                <div class="aiox-table-card">
                    <h3>
                        <span class="dashicons dashicons-star-filled"></span>
                        <?php _e('Top Content by Badge Views', 'aiox-publisher-suite-pro'); ?>
                    </h3>
                    <div class="aiox-table-wrapper">
                        <table id="badge-posts-table" class="aiox-table">
                            <thead>
                                <tr>
                                    <th><?php _e('Content', 'aiox-publisher-suite-pro'); ?></th>
                                    <th><?php _e('Total Views', 'aiox-publisher-suite-pro'); ?></th>
                                    <th><?php _e('AI Views', 'aiox-publisher-suite-pro'); ?></th>
                                    <th><?php _e('Browser Views', 'aiox-publisher-suite-pro'); ?></th>
                                </tr>
                            </thead>
                            <tbody>
                                <tr>
                                    <td colspan="4" class="aiox-loading-cell">
                                        <div class="aiox-spinner-small"></div>
                                        <?php _e('Loading...', 'aiox-publisher-suite-pro'); ?>
                                    </td>
                                </tr>
                            </tbody>
                        </table>
                    </div>
                </div>
            </div>

            <!-- AIOX System Files Tab -->
            <div id="tab-wellknown" class="aiox-tab-content">
                <div class="aiox-info-banner">
                    <span class="dashicons dashicons-info"></span>
                    <div>
                        <strong><?php _e('AIOX System Files Performance', 'aiox-publisher-suite-pro'); ?></strong>
                        <p><?php _e('Track how AI crawlers access your AIOX system files (aio.json, aiox.ndjson, tdmrep.json, capsules, etc.)', 'aiox-publisher-suite-pro'); ?></p>
                    </div>
                </div>

                <div class="aiox-charts-grid">
                    <div class="aiox-chart-card">
                        <h3>
                            <span class="dashicons dashicons-chart-bar"></span>
                            <?php _e('Top AIOX Files', 'aiox-publisher-suite-pro'); ?>
                        </h3>
                        <div class="aiox-chart-container">
                            <canvas id="wellknown-files-chart"></canvas>
                        </div>
                    </div>

                    <div class="aiox-chart-card">
                        <h3>
                            <span class="dashicons dashicons-chart-line"></span>
                            <?php _e('Daily AIOX File Requests', 'aiox-publisher-suite-pro'); ?>
                        </h3>
                        <div class="aiox-chart-container">
                            <canvas id="wellknown-daily-chart"></canvas>
                        </div>
                    </div>
                </div>

                <div class="aiox-table-card">
                    <h3>
                        <span class="dashicons dashicons-admin-site-alt3"></span>
                        <?php _e('Crawlers Accessing AIOX Files', 'aiox-publisher-suite-pro'); ?>
                    </h3>
                    <div class="aiox-table-wrapper">
                        <table id="wellknown-crawlers-table" class="aiox-table">
                            <thead>
                                <tr>
                                    <th><?php _e('Crawler Name', 'aiox-publisher-suite-pro'); ?></th>
                                    <th><?php _e('Type', 'aiox-publisher-suite-pro'); ?></th>
                                    <th><?php _e('Total Hits', 'aiox-publisher-suite-pro'); ?></th>
                                </tr>
                            </thead>
                            <tbody>
                                <tr>
                                    <td colspan="3" class="aiox-loading-cell">
                                        <div class="aiox-spinner-small"></div>
                                        <?php _e('Loading...', 'aiox-publisher-suite-pro'); ?>
                                    </td>
                                </tr>
                            </tbody>
                        </table>
                    </div>
                </div>
            </div>

            <!-- Content Tab -->
            <div id="tab-content" class="aiox-tab-content">
                <div class="aiox-charts-grid">
                    <div class="aiox-chart-card">
                        <h3>
                            <span class="dashicons dashicons-chart-pie"></span>
                            <?php _e('Content Performance by Type', 'aiox-publisher-suite-pro'); ?>
                        </h3>
                        <div class="aiox-chart-container">
                            <canvas id="content-performance-chart"></canvas>
                        </div>
                    </div>
                </div>

                <div class="aiox-table-card">
                    <h3>
                        <span class="dashicons dashicons-edit-page"></span>
                        <?php _e('Top Content by Crawler Visits', 'aiox-publisher-suite-pro'); ?>
                    </h3>
                    <div class="aiox-table-wrapper">
                        <table id="top-content-table" class="aiox-table">
                            <thead>
                                <tr>
                                    <th><?php _e('Content', 'aiox-publisher-suite-pro'); ?></th>
                                    <th><?php _e('Type', 'aiox-publisher-suite-pro'); ?></th>
                                    <th><?php _e('Crawler Visits', 'aiox-publisher-suite-pro'); ?></th>
                                    <th><?php _e('Actions', 'aiox-publisher-suite-pro'); ?></th>
                                </tr>
                            </thead>
                            <tbody>
                                <tr>
                                    <td colspan="4" class="aiox-loading-cell">
                                        <div class="aiox-spinner-small"></div>
                                        <?php _e('Loading...', 'aiox-publisher-suite-pro'); ?>
                                    </td>
                                </tr>
                            </tbody>
                        </table>
                    </div>
                </div>
            </div>

            <!-- Activity Tab -->
            <div id="tab-activity" class="aiox-tab-content">
                <div class="aiox-charts-grid">
                    <div class="aiox-chart-card">
                        <h3>
                            <span class="dashicons dashicons-clock"></span>
                            <?php _e('Hourly Activity Pattern', 'aiox-publisher-suite-pro'); ?>
                        </h3>
                        <div class="aiox-chart-container">
                            <canvas id="hourly-activity-chart"></canvas>
                        </div>
                    </div>
                </div>

                <div class="aiox-table-card">
                    <h3>
                        <span class="dashicons dashicons-admin-links"></span>
                        <?php _e('Most Visited Pages', 'aiox-publisher-suite-pro'); ?>
                    </h3>
                    <div class="aiox-table-wrapper">
                        <table id="top-pages-table" class="aiox-table">
                            <thead>
                                <tr>
                                    <th><?php _e('Page URL', 'aiox-publisher-suite-pro'); ?></th>
                                    <th><?php _e('Visits', 'aiox-publisher-suite-pro'); ?></th>
                                    <th><?php _e('Type', 'aiox-publisher-suite-pro'); ?></th>
                                </tr>
                            </thead>
                            <tbody>
                                <tr>
                                    <td colspan="3" class="aiox-loading-cell">
                                        <div class="aiox-spinner-small"></div>
                                        <?php _e('Loading...', 'aiox-publisher-suite-pro'); ?>
                                    </td>
                                </tr>
                            </tbody>
                        </table>
                    </div>
                </div>
            </div>

            <!-- Visitor Logs Tab -->
            <div id="tab-visitorlogs" class="aiox-tab-content">
                <div class="aiox-table-card">
                    <div class="aiox-card-header" style="display: flex; justify-content: space-between; align-items: center; margin-bottom: 15px;">
                        <h3 style="margin: 0;">
                            <span class="dashicons dashicons-list-view"></span>
                            <?php _e('Recent Visitor Detection Logs', 'aiox-publisher-suite-pro'); ?>
                        </h3>
                        <div style="display: flex; gap: 10px; align-items: center;">
                            <select id="visitor-type-filter" class="aiox-select" style="min-width: 150px;">
                                <option value=""><?php _e('All Types', 'aiox-publisher-suite-pro'); ?></option>
                                <option value="human"><?php _e('Browser', 'aiox-publisher-suite-pro'); ?></option>
                                <option value="ai"><?php _e('AI', 'aiox-publisher-suite-pro'); ?></option>
                                <option value="search"><?php _e('Search', 'aiox-publisher-suite-pro'); ?></option>
                                <option value="bot"><?php _e('Bot', 'aiox-publisher-suite-pro'); ?></option>
                                <option value="unidentified"><?php _e('Unidentified', 'aiox-publisher-suite-pro'); ?></option>
                            </select>
                            <button type="button" id="refresh-visitor-logs" class="aiox-btn aiox-btn-secondary">
                                <span class="dashicons dashicons-update"></span>
                                <?php _e('Refresh', 'aiox-publisher-suite-pro'); ?>
                            </button>
                        </div>
                    </div>
                    
                    <p class="aiox-info-text" style="background: #f0f6fc; padding: 10px 15px; border-radius: 4px; margin-bottom: 15px; border-left: 4px solid #2271b1;">
                        <span class="dashicons dashicons-info" style="color: #2271b1;"></span>
                        <?php _e('To enable detailed visitor logging, add <code>define(\'AIOX_DEBUG_VISITORS\', true);</code> to your wp-config.php file. Logs are stored in <code>wp-content/aiox-logs/</code>.', 'aiox-publisher-suite-pro'); ?>
                    </p>
                    
                    <div class="aiox-table-wrapper">
                        <table id="visitor-logs-table" class="aiox-table">
                            <thead>
                                <tr>
                                    <th style="width: 140px;"><?php _e('Time', 'aiox-publisher-suite-pro'); ?></th>
                                    <th style="width: 100px;"><?php _e('Type', 'aiox-publisher-suite-pro'); ?></th>
                                    <th style="width: 120px;"><?php _e('Name', 'aiox-publisher-suite-pro'); ?></th>
                                    <th style="width: 100px;"><?php _e('IP', 'aiox-publisher-suite-pro'); ?></th>
                                    <th><?php _e('User Agent', 'aiox-publisher-suite-pro'); ?></th>
                                    <th style="width: 200px;"><?php _e('Page', 'aiox-publisher-suite-pro'); ?></th>
                                </tr>
                            </thead>
                            <tbody id="visitor-logs-body">
                                <tr>
                                    <td colspan="6" class="aiox-loading-cell">
                                        <div class="aiox-spinner-small"></div>
                                        <?php _e('Loading visitor logs...', 'aiox-publisher-suite-pro'); ?>
                                    </td>
                                </tr>
                            </tbody>
                        </table>
                    </div>
                    
                    <div id="visitor-logs-pagination" style="margin-top: 15px; display: flex; justify-content: center; gap: 10px;">
                        <!-- Pagination will be added via JS -->
                    </div>
                </div>
                
                <!-- Log File Viewer -->
                <div class="aiox-table-card" style="margin-top: 20px;">
                    <h3>
                        <span class="dashicons dashicons-media-text"></span>
                        <?php _e('Debug Log File', 'aiox-publisher-suite-pro'); ?>
                    </h3>
                    <div id="debug-log-content" style="background: #1e1e1e; color: #d4d4d4; padding: 15px; border-radius: 4px; font-family: monospace; font-size: 12px; max-height: 400px; overflow-y: auto; white-space: pre-wrap;">
                        <?php
                        $log_file = WP_CONTENT_DIR . '/aiox-logs/visitor-detection-' . date('Y-m-d') . '.log';
                        if (file_exists($log_file)) {
                            $content = file_get_contents($log_file);
                            // Get last 100 lines
                            $lines = explode("\n", $content);
                            $last_lines = array_slice($lines, -100);
                            echo esc_html(implode("\n", $last_lines));
                        } else {
                            _e('No log file found for today. Enable AIOX_DEBUG_VISITORS to start logging.', 'aiox-publisher-suite-pro');
                        }
                        ?>
                    </div>
                </div>
            </div>

            <!-- Export Modal -->
            <div id="aiox-export-modal" class="aiox-modal" style="display:none;">
                <div class="aiox-modal-backdrop"></div>
                <div class="aiox-modal-content">
                    <div class="aiox-modal-header">
                        <h2><?php _e('Export Analytics Data', 'aiox-publisher-suite-pro'); ?></h2>
                        <button class="aiox-modal-close">
                            <span class="dashicons dashicons-no-alt"></span>
                        </button>
                    </div>
                    
                    <div class="aiox-modal-body">
                        <div class="aiox-export-options">
                            <label class="aiox-radio-label">
                                <input type="radio" name="export-format" value="csv" checked>
                                <span class="aiox-radio-text">
                                    <strong>CSV Format</strong>
                                    <small><?php _e('Excel compatible spreadsheet', 'aiox-publisher-suite-pro'); ?></small>
                                </span>
                            </label>
                            <label class="aiox-radio-label">
                                <input type="radio" name="export-format" value="json">
                                <span class="aiox-radio-text">
                                    <strong>JSON Format</strong>
                                    <small><?php _e('Machine-readable data format', 'aiox-publisher-suite-pro'); ?></small>
                                </span>
                            </label>
                        </div>

                        <div class="aiox-export-period">
                            <label for="export-period"><?php _e('Time Period:', 'aiox-publisher-suite-pro'); ?></label>
                            <select id="export-period" class="aiox-select">
                                <option value="7"><?php _e('Last 7 days', 'aiox-publisher-suite-pro'); ?></option>
                                <option value="30" selected><?php _e('Last 30 days', 'aiox-publisher-suite-pro'); ?></option>
                                <option value="90"><?php _e('Last 90 days', 'aiox-publisher-suite-pro'); ?></option>
                            </select>
                        </div>
                    </div>

                    <div class="aiox-modal-footer">
                        <button id="aiox-export-download" class="aiox-btn aiox-btn-primary">
                            <span class="dashicons dashicons-download"></span>
                            <?php _e('Download Export', 'aiox-publisher-suite-pro'); ?>
                        </button>
                    </div>
                </div>
            </div>
        </div>
        <?php
    }
}

AIOX_Modern_Analytics_Admin::get_instance();