<?php
/**
 * AIOX Monetization Page
 * 
 * Dedicated page for AI License & Monetization settings
 * Settings are automatically applied to all manifest files when saved
 */

if (!defined('ABSPATH')) {
    exit;
}

class AIOX_Monetization_Page {
    
    /**
     * Singleton instance
     */
    private static $instance = null;
    
    /**
     * Get singleton instance
     */
    public static function get_instance() {
        if (null === self::$instance) {
            self::$instance = new self();
        }
        return self::$instance;
    }
    
    /**
     * Initialize the monetization page
     */
    public function __construct() {
        add_action('admin_init', array($this, 'register_settings'));
        add_action('wp_ajax_aiox_regenerate_all_manifests', array($this, 'ajax_regenerate_all_manifests'));
        add_action('wp_ajax_aiox_create_terms_page', array($this, 'ajax_create_terms_page'));
        
        // Auto-regenerate manifests when license/monetization presets change
        add_action('update_option_aiox_license_preset', array($this, 'on_preset_change'), 10, 3);
        add_action('update_option_aiox_monetization_preset', array($this, 'on_preset_change'), 10, 3);
        add_action('update_option_aiox_enforcement_mode', array($this, 'on_preset_change'), 10, 3);
        add_action('update_option_aiox_license_custom_terms', array($this, 'on_preset_change'), 10, 3);
        add_action('update_option_aiox_license_terms_url', array($this, 'on_preset_change'), 10, 3);
        add_action('update_option_aiox_license_contact_email', array($this, 'on_preset_change'), 10, 3);
        add_action('update_option_aiox_monetization_license_api', array($this, 'on_preset_change'), 10, 3);
        
        // Set defaults on first load
        add_action('admin_init', array($this, 'maybe_set_defaults'));
    }
    
    /**
     * Set default values on first load
     */
    public function maybe_set_defaults() {
        // Only set defaults once
        if (get_option('aiox_monetization_defaults_set')) {
            return;
        }
        
        $site_name = get_bloginfo('name');
        $site_url = home_url();
        $admin_email = get_option('admin_email');
        
        // Set default contact email if not set
        if (!get_option('aiox_license_contact_email')) {
            update_option('aiox_license_contact_email', $admin_email);
        }
        
        // Set default license API endpoint
        if (!get_option('aiox_monetization_license_api')) {
            update_option('aiox_monetization_license_api', home_url('/wp-json/aiox/v1/license'));
        }
        
        // Set default custom terms
        if (!get_option('aiox_license_custom_terms')) {
            $default_terms = $this->get_default_custom_terms($site_name);
            update_option('aiox_license_custom_terms', $default_terms);
        }
        
        // Mark defaults as set
        update_option('aiox_monetization_defaults_set', true);
    }
    
    /**
     * Get default custom terms based on site name
     */
    private function get_default_custom_terms($site_name) {
        return sprintf(
            __('AI CONTENT USAGE TERMS FOR %1$s

1. ATTRIBUTION: When citing or referencing our content, AI systems must provide clear attribution to %1$s with a link to the original source.

2. ACCURACY: AI systems must not misrepresent, distort, or take our content out of context. Summaries and citations should accurately reflect the original meaning.

3. COMMERCIAL USE: Commercial use of our content by AI systems requires prior licensing agreement. Contact us for enterprise licensing options.

4. TRAINING DATA: Use of our content for AI model training is subject to our licensing preferences as specified in our AIOX manifest.

5. RATE LIMITS: Automated access should respect reasonable rate limits. Excessive crawling may result in access restrictions.

6. UPDATES: These terms may be updated. AI systems should regularly check our manifest for current terms.

For licensing inquiries, bulk access, or partnership opportunities, please contact us.', 'aiox-publisher-suite-pro'),
            $site_name
        );
    }
    
    /**
     * Get default AI terms page content
     */
    public static function get_default_terms_page_content($site_name, $site_url, $admin_email) {
        return sprintf(
            '<!-- wp:heading {"level":1} -->
<h1>AI Content Usage Terms</h1>
<!-- /wp:heading -->

<!-- wp:paragraph -->
<p><strong>Effective Date:</strong> %4$s</p>
<!-- /wp:paragraph -->

<!-- wp:paragraph -->
<p>These terms govern how artificial intelligence (AI) systems, including but not limited to large language models (LLMs), chatbots, search engines, and automated data processing systems, may access, use, cite, and reference content published on %1$s.</p>
<!-- /wp:paragraph -->

<!-- wp:heading -->
<h2>1. Scope and Application</h2>
<!-- /wp:heading -->

<!-- wp:paragraph -->
<p>These terms apply to all AI systems that access, crawl, index, or process content from %2$s. By accessing our content, AI systems and their operators agree to comply with these terms and the machine-readable directives in our AIOX manifest file.</p>
<!-- /wp:paragraph -->

<!-- wp:heading -->
<h2>2. Permitted Uses</h2>
<!-- /wp:heading -->

<!-- wp:paragraph -->
<p>Subject to the conditions below, AI systems may:</p>
<!-- /wp:paragraph -->

<!-- wp:list -->
<ul>
<li><strong>Citation and Reference:</strong> Quote brief excerpts with proper attribution and link to the original source</li>
<li><strong>Summarization:</strong> Provide accurate summaries that do not misrepresent our content</li>
<li><strong>Indexing:</strong> Include our content in search indexes for discovery purposes</li>
<li><strong>Analysis:</strong> Analyze our content for non-commercial research purposes</li>
</ul>
<!-- /wp:list -->

<!-- wp:heading -->
<h2>3. Attribution Requirements</h2>
<!-- /wp:heading -->

<!-- wp:paragraph -->
<p>When citing or referencing our content, AI systems must:</p>
<!-- /wp:paragraph -->

<!-- wp:list -->
<ul>
<li>Clearly identify %1$s as the source</li>
<li>Provide a direct link to the original content when possible</li>
<li>Include the publication date when available</li>
<li>Not imply endorsement or affiliation without permission</li>
</ul>
<!-- /wp:list -->

<!-- wp:heading -->
<h2>4. Prohibited Uses</h2>
<!-- /wp:heading -->

<!-- wp:paragraph -->
<p>Without explicit written permission, AI systems may not:</p>
<!-- /wp:paragraph -->

<!-- wp:list -->
<ul>
<li>Use our content for AI model training or fine-tuning (unless permitted in our AIOX manifest)</li>
<li>Reproduce substantial portions of our content without authorization</li>
<li>Misrepresent, distort, or take content out of context</li>
<li>Remove or obscure attribution or copyright notices</li>
<li>Use content in ways that compete directly with our services</li>
<li>Circumvent access controls or technical protection measures</li>
</ul>
<!-- /wp:list -->

<!-- wp:heading -->
<h2>5. AI Training and Machine Learning</h2>
<!-- /wp:heading -->

<!-- wp:paragraph -->
<p>Our current policy on AI training is specified in our AIOX manifest file at <code>/.well-known/aio.json</code>. AI developers and operators must check this file for our current training permissions. We reserve the right to modify these permissions at any time.</p>
<!-- /wp:paragraph -->

<!-- wp:heading -->
<h2>6. Commercial Licensing</h2>
<!-- /wp:heading -->

<!-- wp:paragraph -->
<p>Commercial use of our content by AI systems, including but not limited to:</p>
<!-- /wp:paragraph -->

<!-- wp:list -->
<ul>
<li>Integration into commercial AI products</li>
<li>Use in paid AI services</li>
<li>Bulk data access for commercial purposes</li>
<li>Training commercial AI models</li>
</ul>
<!-- /wp:list -->

<!-- wp:paragraph -->
<p>requires a separate licensing agreement. Contact us at <a href="mailto:%3$s">%3$s</a> for licensing inquiries.</p>
<!-- /wp:paragraph -->

<!-- wp:heading -->
<h2>7. Technical Compliance</h2>
<!-- /wp:heading -->

<!-- wp:paragraph -->
<p>AI systems accessing our content must:</p>
<!-- /wp:paragraph -->

<!-- wp:list -->
<ul>
<li>Respect our robots.txt directives</li>
<li>Honor our AIOX manifest settings</li>
<li>Comply with TDM-REP (Text and Data Mining Reservation Protocol) headers</li>
<li>Implement reasonable rate limiting (recommended: 1 request per second)</li>
<li>Identify themselves via User-Agent strings</li>
</ul>
<!-- /wp:list -->

<!-- wp:heading -->
<h2>8. Enforcement</h2>
<!-- /wp:heading -->

<!-- wp:paragraph -->
<p>We reserve the right to:</p>
<!-- /wp:paragraph -->

<!-- wp:list -->
<ul>
<li>Block access to AI systems that violate these terms</li>
<li>Pursue legal remedies for unauthorized use</li>
<li>Report violations to relevant authorities and industry bodies</li>
<li>Modify access permissions without prior notice</li>
</ul>
<!-- /wp:list -->

<!-- wp:heading -->
<h2>9. Updates to These Terms</h2>
<!-- /wp:heading -->

<!-- wp:paragraph -->
<p>We may update these terms at any time. Changes become effective upon posting. AI systems should regularly check both this page and our AIOX manifest for current terms.</p>
<!-- /wp:paragraph -->

<!-- wp:heading -->
<h2>10. Contact Information</h2>
<!-- /wp:heading -->

<!-- wp:paragraph -->
<p>For questions about these terms, licensing inquiries, or to report violations:</p>
<!-- /wp:paragraph -->

<!-- wp:list -->
<ul>
<li><strong>Email:</strong> <a href="mailto:%3$s">%3$s</a></li>
<li><strong>Website:</strong> <a href="%2$s">%2$s</a></li>
</ul>
<!-- /wp:list -->

<!-- wp:separator -->
<hr class="wp-block-separator"/>
<!-- /wp:separator -->

<!-- wp:paragraph {"fontSize":"small"} -->
<p class="has-small-font-size"><em>This page is designed to be read by both humans and AI systems. Our machine-readable licensing preferences are also available in our AIOX manifest at <code>/.well-known/aio.json</code>.</em></p>
<!-- /wp:paragraph -->',
            $site_name,
            $site_url,
            $admin_email,
            date_i18n(get_option('date_format'))
        );
    }
    
    /**
     * Register monetization settings
     */
    public function register_settings() {
        // Add settings section
        add_settings_section(
            'aiox_monetization_section',
            '',
            '__return_false',
            'aiox-monetization'
        );
        
        // License preset
        register_setting('aiox-monetization', 'aiox_license_preset', array(
            'type' => 'string',
            'sanitize_callback' => array($this, 'sanitize_license_preset'),
            'default' => 'attribution_required'
        ));
        
        // Monetization preset
        register_setting('aiox-monetization', 'aiox_monetization_preset', array(
            'type' => 'string',
            'sanitize_callback' => array($this, 'sanitize_monetization_preset'),
            'default' => 'free'
        ));
        
        // Enforcement mode
        register_setting('aiox-monetization', 'aiox_enforcement_mode', array(
            'type' => 'string',
            'sanitize_callback' => array($this, 'sanitize_enforcement_mode'),
            'default' => 'flexible'
        ));
        
        // Custom terms
        register_setting('aiox-monetization', 'aiox_license_custom_terms', array(
            'type' => 'string',
            'sanitize_callback' => 'sanitize_textarea_field',
            'default' => ''
        ));
        
        // Contact email
        register_setting('aiox-monetization', 'aiox_license_contact_email', array(
            'type' => 'string',
            'sanitize_callback' => 'sanitize_email',
            'default' => ''
        ));
        
        // Terms URL
        register_setting('aiox-monetization', 'aiox_license_terms_url', array(
            'type' => 'string',
            'sanitize_callback' => 'esc_url_raw',
            'default' => ''
        ));
        
        // License API endpoint
        register_setting('aiox-monetization', 'aiox_monetization_license_api', array(
            'type' => 'string',
            'sanitize_callback' => 'esc_url_raw',
            'default' => ''
        ));
    }
    
    /**
     * Sanitize license preset
     */
    public function sanitize_license_preset($value) {
        $valid = array('open_access', 'attribution_required', 'commercial_license');
        return in_array($value, $valid) ? $value : 'attribution_required';
    }
    
    /**
     * Sanitize monetization preset
     */
    public function sanitize_monetization_preset($value) {
        $valid = array('free', 'freemium', 'paid');
        return in_array($value, $valid) ? $value : 'free';
    }
    
    /**
     * Sanitize enforcement mode
     */
    public function sanitize_enforcement_mode($value) {
        $valid = array('strict', 'flexible');
        return in_array($value, $valid) ? $value : 'flexible';
    }
    
    /**
     * Handle preset/licensing setting changes - regenerate manifests
     */
    public function on_preset_change($old_value, $new_value, $option) {
        // Only regenerate if value actually changed
        if ($old_value === $new_value) {
            return;
        }
        
        // Schedule regeneration (use transient to batch multiple changes)
        if (!get_transient('aiox_preset_change_pending')) {
            set_transient('aiox_preset_change_pending', true, 30);
            add_action('shutdown', array($this, 'regenerate_all_on_save'));
        }
        
        // Log the change
        if (class_exists('AIOX_Logger')) {
            AIOX_Logger::info('Monetization setting changed, manifests will regenerate', array(
                'option' => $option,
                'old_value' => $old_value,
                'new_value' => $new_value
            ));
        }
    }
    
    /**
     * Regenerate all manifest files after settings are saved
     */
    public function regenerate_all_on_save() {
        delete_transient('aiox_preset_change_pending');
        
        // Regenerate main manifest, TDM-REP, and delta feed
        if (class_exists('AIOX_Hooks')) {
            $hooks = new AIOX_Hooks();
            
            if (method_exists($hooks, 'generate_manifest')) {
                $hooks->generate_manifest();
            }
            if (method_exists($hooks, 'generate_tdmrep')) {
                $hooks->generate_tdmrep();
            }
            if (method_exists($hooks, 'generate_delta_feed')) {
                $hooks->generate_delta_feed();
            }
            if (method_exists($hooks, 'generate_robots_txt')) {
                $hooks->generate_robots_txt();
            }
        }
        
        // Regenerate ALL individual post manifest files (NO API calls needed)
        $this->regenerate_all_post_manifests();
        
        if (class_exists('AIOX_Logger')) {
            AIOX_Logger::info('All manifests regenerated after monetization settings change');
        }
    }
    
    /**
     * AJAX handler to regenerate all manifest files (no API calls)
     */
    public function ajax_regenerate_all_manifests() {
        check_ajax_referer('aiox_regenerate_manifests', 'nonce');
        
        if (!current_user_can('manage_options')) {
            wp_send_json_error(array('message' => 'Permission denied'));
            return;
        }
        
        $start_time = microtime(true);
        
        // Regenerate main manifest files
        if (class_exists('AIOX_Hooks')) {
            $hooks = new AIOX_Hooks();
            
            if (method_exists($hooks, 'generate_manifest')) {
                $hooks->generate_manifest();
            }
            if (method_exists($hooks, 'generate_tdmrep')) {
                $hooks->generate_tdmrep();
            }
            if (method_exists($hooks, 'generate_delta_feed')) {
                $hooks->generate_delta_feed();
            }
        }
        
        // Regenerate all post manifest files
        $stats = $this->regenerate_all_post_manifests();
        
        $elapsed = round(microtime(true) - $start_time, 2);
        
        wp_send_json_success(array(
            'message' => sprintf(
                __('Regenerated %d post manifests in %s seconds. Updated: %d, Failed: %d, Skipped: %d', 'aiox-publisher-suite-pro'),
                $stats['total'],
                $elapsed,
                $stats['updated'],
                $stats['failed'],
                $stats['skipped']
            ),
            'stats' => $stats,
            'elapsed' => $elapsed
        ));
    }
    
    /**
     * AJAX handler to create AI Terms & Conditions page
     */
    public function ajax_create_terms_page() {
        check_ajax_referer('aiox_create_terms_page', 'nonce');
        
        if (!current_user_can('manage_options')) {
            wp_send_json_error(array('message' => 'Permission denied'));
            return;
        }
        
        $site_name = get_bloginfo('name');
        $site_url = home_url();
        $admin_email = get_option('admin_email');
        
        // Check if page already exists
        $existing = get_posts(array(
            'post_type' => 'page',
            'post_status' => array('publish', 'draft', 'private'),
            'name' => 'ai-content-usage-terms',
            'posts_per_page' => 1
        ));
        
        if (!empty($existing)) {
            $page = $existing[0];
            wp_send_json_success(array(
                'message' => __('AI Terms page already exists!', 'aiox-publisher-suite-pro'),
                'page_id' => $page->ID,
                'page_url' => get_permalink($page->ID),
                'edit_url' => admin_url('post.php?post=' . $page->ID . '&action=edit'),
                'already_exists' => true
            ));
            return;
        }
        
        // Create the page
        $page_content = self::get_default_terms_page_content($site_name, $site_url, $admin_email);
        
        $page_id = wp_insert_post(array(
            'post_title' => __('AI Content Usage Terms', 'aiox-publisher-suite-pro'),
            'post_name' => 'ai-content-usage-terms',
            'post_content' => $page_content,
            'post_status' => 'publish',
            'post_type' => 'page',
            'post_author' => get_current_user_id(),
            'comment_status' => 'closed',
            'ping_status' => 'closed'
        ));
        
        if (is_wp_error($page_id)) {
            wp_send_json_error(array('message' => $page_id->get_error_message()));
            return;
        }
        
        // Update the terms URL option
        $page_url = get_permalink($page_id);
        update_option('aiox_license_terms_url', $page_url);
        
        wp_send_json_success(array(
            'message' => __('AI Terms page created successfully!', 'aiox-publisher-suite-pro'),
            'page_id' => $page_id,
            'page_url' => $page_url,
            'edit_url' => admin_url('post.php?post=' . $page_id . '&action=edit'),
            'already_exists' => false
        ));
    }
    
    /**
     * Regenerate all individual post manifest files with current settings
     * This does NOT call the AI API - it only updates licensing/monetization in existing files
     */
    public function regenerate_all_post_manifests() {
        $well_known_dir = ABSPATH . '.well-known';
        $posts_dir = $well_known_dir . '/posts';
        
        if (!file_exists($posts_dir)) {
            wp_mkdir_p($posts_dir);
        }
        
        $args = array(
            'post_type' => array('post', 'page'),
            'post_status' => 'publish',
            'posts_per_page' => -1,
            'meta_query' => array(
                array(
                    'key' => '_aiox_processed',
                    'value' => '1',
                    'compare' => '='
                )
            ),
            'fields' => 'ids'
        );
        
        $post_ids = get_posts($args);
        $stats = array(
            'total' => count($post_ids),
            'updated' => 0,
            'failed' => 0,
            'skipped' => 0
        );
        
        if (empty($post_ids)) {
            return $stats;
        }
        
        $license_config = $this->get_license_config();
        $monetization_config = $this->get_monetization_config();
        $enforcement_config = $this->get_enforcement_config();
        
        foreach ($post_ids as $post_id) {
            $result = $this->update_post_manifest($post_id, $license_config, $monetization_config, $enforcement_config, $posts_dir);
            
            if ($result === true) {
                $stats['updated']++;
            } elseif ($result === false) {
                $stats['failed']++;
            } else {
                $stats['skipped']++;
            }
        }
        
        if (class_exists('AIOX_Logger')) {
            AIOX_Logger::info('Post manifests regenerated with new settings', $stats);
        }
        
        return $stats;
    }
    
    /**
     * Update a single post's manifest file with new settings
     */
    private function update_post_manifest($post_id, $license_config, $monetization_config, $enforcement_config, $posts_dir) {
        $file_path = $posts_dir . '/' . $post_id . '.json';
        $home_url = home_url();
        
        if (file_exists($file_path)) {
            $existing_data = json_decode(file_get_contents($file_path), true);
            
            if (!$existing_data) {
                return false;
            }
            
            // Update only the licensing/monetization sections
            $existing_data['licensing'] = array(
                'preset' => $license_config['preset'],
                'license_code' => $license_config['license_code'],
                'training_allowed' => $license_config['training_allowed'],
                'citation_required' => $license_config['citation_required'],
                'attribution_required' => $license_config['attribution_required'],
                'commercial_use' => $license_config['commercial_use'],
                'attribution' => get_option('aiox_attribution_string', get_bloginfo('name')),
                'inherited_from' => 'site'
            );
            
            $existing_data['monetization'] = array(
                'preset' => $monetization_config['preset'],
                'model' => $monetization_config['pricing_model'],
                'subscription_required' => $monetization_config['subscription_required'],
                'pay_per_use' => $monetization_config['pay_per_use'],
                'inherited_from' => 'site'
            );
            
            $existing_data['enforcement'] = array(
                'mode' => $enforcement_config['mode'],
                'compliance_level' => $enforcement_config['compliance_level'],
                'legal_binding' => $enforcement_config['legal_binding']
            );
            
            $existing_data['settings_updated_at'] = current_time('c');
            
            if (!isset($existing_data['links']['tdmrep'])) {
                $existing_data['links']['tdmrep'] = $home_url . '/.well-known/tdmrep.json';
            }
            
            $result = file_put_contents($file_path, json_encode($existing_data, JSON_PRETTY_PRINT | JSON_UNESCAPED_SLASHES));
            
            return $result !== false;
        } else {
            // Generate new file for processed post without static file
            $post = get_post($post_id);
            if (!$post || $post->post_status !== 'publish') {
                return null;
            }
            
            $manifest_data = $this->generate_post_manifest($post_id, $license_config, $monetization_config, $enforcement_config);
            
            if ($manifest_data) {
                $result = file_put_contents($file_path, json_encode($manifest_data, JSON_PRETTY_PRINT | JSON_UNESCAPED_SLASHES));
                return $result !== false;
            }
            
            return null;
        }
    }
    
    /**
     * Generate a complete post manifest with current settings
     */
    private function generate_post_manifest($post_id, $license_config, $monetization_config, $enforcement_config) {
        $post = get_post($post_id);
        
        if (!$post || $post->post_status !== 'publish') {
            return null;
        }
        
        $home_url = home_url();
        
        $ingredients = array();
        if (class_exists('AIOX_Ingredients')) {
            $ingredients = AIOX_Ingredients::get_post_ingredients($post_id);
        }
        
        $metadata = array();
        if (class_exists('AIOX_Metadata')) {
            $metadata = AIOX_Metadata::get_post_metadata($post_id);
        }
        
        $capsule_data = array();
        if (class_exists('AIOX_Capsule_Manager')) {
            $capsule_manager = new AIOX_Capsule_Manager();
            $capsule = $capsule_manager->get_capsule_by_post_id($post_id);
            if ($capsule) {
                $capsule_data = array(
                    'id' => $capsule->get_id(),
                    'status' => $capsule->get_status(),
                    'type' => $capsule->get_type()
                );
            }
        }
        
        return array(
            'version' => '2.0',
            'generator' => 'AIOX Publisher Suite Pro',
            'generated_at' => get_post_meta($post_id, '_aiox_processed_at', true) ?: current_time('c'),
            'settings_updated_at' => current_time('c'),
            'post' => array(
                'id' => $post_id,
                'title' => $post->post_title,
                'url' => get_permalink($post_id),
                'published' => get_the_date('c', $post_id),
                'modified' => get_the_modified_date('c', $post_id),
                'type' => $post->post_type,
                'author' => get_the_author_meta('display_name', $post->post_author),
                'excerpt' => wp_trim_words(get_the_excerpt($post_id), 50, '...'),
                'word_count' => str_word_count(strip_tags($post->post_content)),
                'categories' => wp_get_post_categories($post_id, array('fields' => 'names')),
                'tags' => wp_get_post_tags($post_id, array('fields' => 'names'))
            ),
            'licensing' => array(
                'preset' => $license_config['preset'],
                'license_code' => $license_config['license_code'],
                'training_allowed' => $license_config['training_allowed'],
                'citation_required' => $license_config['citation_required'],
                'attribution_required' => $license_config['attribution_required'],
                'commercial_use' => $license_config['commercial_use'],
                'attribution' => get_option('aiox_attribution_string', get_bloginfo('name')),
                'inherited_from' => 'site'
            ),
            'monetization' => array(
                'preset' => $monetization_config['preset'],
                'model' => $monetization_config['pricing_model'],
                'subscription_required' => $monetization_config['subscription_required'],
                'pay_per_use' => $monetization_config['pay_per_use'],
                'inherited_from' => 'site'
            ),
            'enforcement' => array(
                'mode' => $enforcement_config['mode'],
                'compliance_level' => $enforcement_config['compliance_level'],
                'legal_binding' => $enforcement_config['legal_binding']
            ),
            'ingredients' => array(
                'count' => count($ingredients),
                'items' => $ingredients
            ),
            'capsule' => $capsule_data,
            'metadata' => $metadata,
            'processing' => array(
                'processed' => true,
                'last_processed' => get_post_meta($post_id, '_aiox_processed_at', true),
                'server_processed' => (bool) get_post_meta($post_id, '_aiox_server_processed', true)
            ),
            'links' => array(
                'self' => $home_url . '/.well-known/posts/' . $post_id . '.json',
                'post' => get_permalink($post_id),
                'manifest' => $home_url . '/.well-known/aio.json',
                'tdmrep' => $home_url . '/.well-known/tdmrep.json'
            )
        );
    }
    
    /**
     * Get license preset configurations
     */
    public static function get_license_presets() {
        return array(
            'open_access' => array(
                'name' => __('Open Access', 'aiox-publisher-suite-pro'),
                'description' => __('Free to use, train, and cite with attribution. Ideal for open-source content and maximum visibility.', 'aiox-publisher-suite-pro'),
                'similar_to' => 'CC BY 4.0',
                'icon' => 'unlock',
                'color' => '#10b981',
                'training_allowed' => true,
                'citation_required' => true,
                'attribution_required' => true,
                'commercial_use' => true,
                'derivatives_allowed' => true,
                'license_code' => 'AIOX-OA-1.0'
            ),
            'attribution_required' => array(
                'name' => __('Attribution Required', 'aiox-publisher-suite-pro'),
                'description' => __('Content can be cited, but AI training requires permission. Good balance between visibility and control.', 'aiox-publisher-suite-pro'),
                'similar_to' => 'CC BY-SA 4.0',
                'icon' => 'id-alt',
                'color' => '#3b82f6',
                'training_allowed' => false,
                'citation_required' => true,
                'attribution_required' => true,
                'commercial_use' => true,
                'derivatives_allowed' => true,
                'license_code' => 'AIOX-AR-1.0'
            ),
            'commercial_license' => array(
                'name' => __('Commercial License', 'aiox-publisher-suite-pro'),
                'description' => __('All AI usage requires a licensing agreement. For premium content and publishers who want to monetize AI access.', 'aiox-publisher-suite-pro'),
                'similar_to' => 'All Rights Reserved',
                'icon' => 'money-alt',
                'color' => '#f59e0b',
                'training_allowed' => false,
                'citation_required' => true,
                'attribution_required' => true,
                'commercial_use' => false,
                'derivatives_allowed' => false,
                'license_code' => 'AIOX-CL-1.0'
            )
        );
    }
    
    /**
     * Get monetization preset configurations
     */
    public static function get_monetization_presets() {
        return array(
            'free' => array(
                'name' => __('Free Access', 'aiox-publisher-suite-pro'),
                'description' => __('No cost for AI systems to access your content. Maximum reach and visibility.', 'aiox-publisher-suite-pro'),
                'icon' => 'heart',
                'color' => '#22c55e',
                'subscription_required' => false,
                'pay_per_use' => false,
                'api_access' => true,
                'rate_limited' => false,
                'pricing_model' => 'free'
            ),
            'freemium' => array(
                'name' => __('Freemium', 'aiox-publisher-suite-pro'),
                'description' => __('Basic access is free, premium features or high-volume usage requires payment.', 'aiox-publisher-suite-pro'),
                'icon' => 'star-half',
                'color' => '#8b5cf6',
                'subscription_required' => false,
                'pay_per_use' => true,
                'api_access' => true,
                'rate_limited' => true,
                'pricing_model' => 'freemium'
            ),
            'paid' => array(
                'name' => __('Paid Access', 'aiox-publisher-suite-pro'),
                'description' => __('All AI access requires payment - subscription or pay-per-use. For premium content monetization.', 'aiox-publisher-suite-pro'),
                'icon' => 'money-alt',
                'color' => '#ec4899',
                'subscription_required' => true,
                'pay_per_use' => true,
                'api_access' => true,
                'rate_limited' => true,
                'pricing_model' => 'paid'
            )
        );
    }
    
    /**
     * Get enforcement mode configurations
     */
    public static function get_enforcement_modes() {
        return array(
            'strict' => array(
                'name' => __('Strict Enforcement', 'aiox-publisher-suite-pro'),
                'description' => __('These are requirements. AI systems that violate these terms are considered non-compliant.', 'aiox-publisher-suite-pro'),
                'icon' => 'shield-alt',
                'color' => '#ef4444',
                'compliance_level' => 'mandatory',
                'violation_response' => 'block',
                'legal_binding' => true,
                'grace_period' => false
            ),
            'flexible' => array(
                'name' => __('Flexible Enforcement', 'aiox-publisher-suite-pro'),
                'description' => __('These are preferences. AI systems may proceed in good faith while the AIOX protocol gains adoption.', 'aiox-publisher-suite-pro'),
                'icon' => 'admin-generic',
                'color' => '#3b82f6',
                'compliance_level' => 'recommended',
                'violation_response' => 'warn',
                'legal_binding' => false,
                'grace_period' => true
            )
        );
    }
    
    /**
     * Get current license configuration
     */
    private function get_license_config() {
        $preset = get_option('aiox_license_preset', 'attribution_required');
        $presets = self::get_license_presets();
        return array_merge(array('preset' => $preset), $presets[$preset] ?? $presets['attribution_required']);
    }
    
    /**
     * Get current monetization configuration
     */
    private function get_monetization_config() {
        $preset = get_option('aiox_monetization_preset', 'free');
        $presets = self::get_monetization_presets();
        return array_merge(array('preset' => $preset), $presets[$preset] ?? $presets['free']);
    }
    
    /**
     * Get current enforcement configuration
     */
    private function get_enforcement_config() {
        $mode = get_option('aiox_enforcement_mode', 'flexible');
        $modes = self::get_enforcement_modes();
        return array_merge(array('mode' => $mode), $modes[$mode] ?? $modes['flexible']);
    }
    
    /**
     * Render the monetization page
     */
    public function render_page() {
        $license_presets = self::get_license_presets();
        $monetization_presets = self::get_monetization_presets();
        $enforcement_modes = self::get_enforcement_modes();
        
        $license_preset = get_option('aiox_license_preset', 'attribution_required');
        $monetization_preset = get_option('aiox_monetization_preset', 'free');
        $enforcement_mode = get_option('aiox_enforcement_mode', 'flexible');
        
        // Count processed posts
        $processed_count = (int) $this->count_processed_posts();
        ?>
        <style>
        .aiox-option-cards { display: grid; grid-template-columns: repeat(auto-fit, minmax(200px, 1fr)); gap: 15px; margin-top: 15px; }
        .aiox-option-card { border: 2px solid #ddd; border-radius: 8px; padding: 20px; cursor: pointer; transition: all 0.2s; background: #fff; position: relative; }
        .aiox-option-card:hover { border-color: #2271b1; background: #f0f6fc; }
        .aiox-option-card.selected { border-color: #2271b1; background: #f0f6fc; }
        .aiox-option-card input[type="radio"] { position: absolute; top: 15px; right: 15px; }
        .aiox-option-card .card-icon { font-size: 28px; margin-bottom: 10px; color: #2271b1; }
        .aiox-option-card .card-title { font-weight: 600; font-size: 14px; margin-bottom: 5px; }
        .aiox-option-card .card-desc { font-size: 12px; color: #666; line-height: 1.4; }
        .aiox-option-card .card-similar { font-size: 11px; color: #999; margin-top: 8px; font-style: italic; }
        .aiox-summary-grid { display: grid; grid-template-columns: repeat(4, 1fr); gap: 15px; }
        .aiox-summary-item { background: #f0f0f1; padding: 15px; border-radius: 4px; text-align: center; }
        .aiox-summary-item span { display: block; font-size: 12px; color: #666; margin-bottom: 5px; }
        .aiox-summary-item strong { font-size: 14px; color: #1d2327; }
        @media (max-width: 782px) { 
            .aiox-option-cards { grid-template-columns: 1fr; }
            .aiox-summary-grid { grid-template-columns: repeat(2, 1fr); }
        }
        </style>
        
        <div class="wrap">
            <h1><span class="dashicons dashicons-money-alt" style="font-size: 30px; width: 30px; height: 30px; margin-right: 10px; color: #dba617;"></span><?php _e('AI Monetization & Licensing', 'aiox-publisher-suite-pro'); ?></h1>
            <p class="description"><?php _e('Control how AI systems can use your content. Settings apply automatically to all your content when saved.', 'aiox-publisher-suite-pro'); ?></p>
            
            <form method="post" action="options.php" id="aiox-monetization-form">
                <?php settings_fields('aiox-monetization'); ?>
                
                <div id="poststuff">
                    <div id="post-body" class="metabox-holder columns-2">
                        <div id="post-body-content">
                            
                            <!-- Enforcement Mode -->
                            <div class="postbox">
                                <h2 class="hndle"><span class="dashicons dashicons-shield"></span> <?php _e('Enforcement Mode', 'aiox-publisher-suite-pro'); ?></h2>
                                <div class="inside">
                                    <p class="description"><?php _e('How strictly should AI systems follow your preferences?', 'aiox-publisher-suite-pro'); ?></p>
                                    
                                    <div class="aiox-option-cards">
                                        <?php foreach ($enforcement_modes as $key => $mode): 
                                            $is_selected = ($enforcement_mode === $key);
                                        ?>
                                        <label class="aiox-option-card <?php echo $is_selected ? 'selected' : ''; ?>">
                                            <input type="radio" name="aiox_enforcement_mode" value="<?php echo esc_attr($key); ?>" <?php checked($enforcement_mode, $key); ?>>
                                            <div class="card-icon"><span class="dashicons dashicons-<?php echo esc_attr($mode['icon']); ?>"></span></div>
                                            <div class="card-title"><?php echo esc_html($mode['name']); ?></div>
                                            <div class="card-desc"><?php echo esc_html($mode['description']); ?></div>
                                        </label>
                                        <?php endforeach; ?>
                                    </div>
                                </div>
                            </div>
                            
                            <!-- License Type -->
                            <div class="postbox">
                                <h2 class="hndle"><span class="dashicons dashicons-admin-network"></span> <?php _e('License Type', 'aiox-publisher-suite-pro'); ?></h2>
                                <div class="inside">
                                    <p class="description"><?php _e('What can AI systems do with your content?', 'aiox-publisher-suite-pro'); ?></p>
                                    
                                    <div class="aiox-option-cards">
                                        <?php foreach ($license_presets as $key => $preset): 
                                            $is_selected = ($license_preset === $key);
                                        ?>
                                        <label class="aiox-option-card <?php echo $is_selected ? 'selected' : ''; ?>">
                                            <input type="radio" name="aiox_license_preset" value="<?php echo esc_attr($key); ?>" <?php checked($license_preset, $key); ?>>
                                            <div class="card-icon"><span class="dashicons dashicons-<?php echo esc_attr($preset['icon']); ?>"></span></div>
                                            <div class="card-title"><?php echo esc_html($preset['name']); ?></div>
                                            <div class="card-desc"><?php echo esc_html($preset['description']); ?></div>
                                            <div class="card-similar"><?php printf(__('Similar to: %s', 'aiox-publisher-suite-pro'), $preset['similar_to']); ?></div>
                                        </label>
                                        <?php endforeach; ?>
                                    </div>
                                </div>
                            </div>
                            
                            <!-- Monetization Model -->
                            <div class="postbox">
                                <h2 class="hndle"><span class="dashicons dashicons-chart-area"></span> <?php _e('Monetization Model', 'aiox-publisher-suite-pro'); ?></h2>
                                <div class="inside">
                                    <p class="description"><?php _e('How should AI systems pay for access to your content?', 'aiox-publisher-suite-pro'); ?></p>
                                    
                                    <div class="aiox-option-cards">
                                        <?php foreach ($monetization_presets as $key => $preset): 
                                            $is_selected = ($monetization_preset === $key);
                                        ?>
                                        <label class="aiox-option-card <?php echo $is_selected ? 'selected' : ''; ?>">
                                            <input type="radio" name="aiox_monetization_preset" value="<?php echo esc_attr($key); ?>" <?php checked($monetization_preset, $key); ?>>
                                            <div class="card-icon"><span class="dashicons dashicons-<?php echo esc_attr($preset['icon']); ?>"></span></div>
                                            <div class="card-title"><?php echo esc_html($preset['name']); ?></div>
                                            <div class="card-desc"><?php echo esc_html($preset['description']); ?></div>
                                        </label>
                                        <?php endforeach; ?>
                                    </div>
                                </div>
                            </div>
                            
                            <!-- Terms & Contact Settings -->
                            <div class="postbox">
                                <h2 class="hndle"><span class="dashicons dashicons-admin-generic"></span> <?php _e('Terms & Contact Settings', 'aiox-publisher-suite-pro'); ?></h2>
                                <div class="inside">
                                    
                                    <!-- Create Terms Page -->
                                    <div class="notice notice-success inline" style="margin: 0 0 20px 0; padding: 15px;">
                                        <div style="display: flex; align-items: center; justify-content: space-between; flex-wrap: wrap; gap: 15px;">
                                            <div>
                                                <strong><span class="dashicons dashicons-media-document" style="vertical-align: middle;"></span> <?php _e('AI Terms & Conditions Page', 'aiox-publisher-suite-pro'); ?></strong>
                                                <p class="description" style="margin: 5px 0 0 0;"><?php _e('Create a comprehensive AI usage terms page with pre-written content.', 'aiox-publisher-suite-pro'); ?></p>
                                                <?php
                                                $terms_url = get_option('aiox_license_terms_url', '');
                                                if ($terms_url) {
                                                    echo '<p style="margin: 5px 0 0 0;"><span class="dashicons dashicons-yes-alt" style="color: #00a32a;"></span> ' . 
                                                        sprintf(__('Current: <a href="%s" target="_blank">%s</a>', 'aiox-publisher-suite-pro'), esc_url($terms_url), esc_html($terms_url)) . '</p>';
                                                }
                                                ?>
                                            </div>
                                            <div>
                                                <button type="button" id="aiox-create-terms-page" class="button button-primary">
                                                    <span class="dashicons dashicons-welcome-add-page" style="vertical-align: middle;"></span>
                                                    <?php _e('Create AI Terms Page', 'aiox-publisher-suite-pro'); ?>
                                                </button>
                                                <span id="create-terms-status" style="display: block; margin-top: 5px;"></span>
                                            </div>
                                        </div>
                                    </div>
                                    
                                    <table class="form-table">
                                        <tr>
                                            <th scope="row">
                                                <label for="aiox_license_contact_email"><?php _e('Licensing Contact Email', 'aiox-publisher-suite-pro'); ?></label>
                                            </th>
                                            <td>
                                                <input type="email" id="aiox_license_contact_email" name="aiox_license_contact_email" 
                                                       value="<?php echo esc_attr(get_option('aiox_license_contact_email', get_option('admin_email'))); ?>" 
                                                       class="regular-text" placeholder="licensing@yoursite.com">
                                                <p class="description"><?php _e('For AI companies to reach you about licensing', 'aiox-publisher-suite-pro'); ?></p>
                                            </td>
                                        </tr>
                                        <tr>
                                            <th scope="row">
                                                <label for="aiox_license_terms_url"><?php _e('Terms & Conditions URL', 'aiox-publisher-suite-pro'); ?></label>
                                            </th>
                                            <td>
                                                <input type="url" id="aiox_license_terms_url" name="aiox_license_terms_url" 
                                                       value="<?php echo esc_attr(get_option('aiox_license_terms_url', '')); ?>" 
                                                       class="regular-text" placeholder="<?php echo esc_attr(home_url('/ai-content-usage-terms/')); ?>">
                                                <p class="description"><?php _e('Link to your AI terms page', 'aiox-publisher-suite-pro'); ?></p>
                                            </td>
                                        </tr>
                                        <tr>
                                            <th scope="row">
                                                <label for="aiox_monetization_license_api"><?php _e('License API Endpoint', 'aiox-publisher-suite-pro'); ?></label>
                                            </th>
                                            <td>
                                                <input type="url" id="aiox_monetization_license_api" name="aiox_monetization_license_api" 
                                                       value="<?php echo esc_attr(get_option('aiox_monetization_license_api', home_url('/wp-json/aiox/v1/license'))); ?>" 
                                                       class="regular-text" placeholder="<?php echo esc_attr(home_url('/wp-json/aiox/v1/license')); ?>">
                                                <p class="description"><?php _e('For automated license verification by AI systems', 'aiox-publisher-suite-pro'); ?></p>
                                            </td>
                                        </tr>
                                        <tr>
                                            <th scope="row">
                                                <label for="aiox_license_custom_terms"><?php _e('Custom License Terms', 'aiox-publisher-suite-pro'); ?></label>
                                            </th>
                                            <td>
                                                <textarea id="aiox_license_custom_terms" name="aiox_license_custom_terms" 
                                                          rows="10" class="large-text code"
                                                          placeholder="<?php esc_attr_e('Your custom terms will appear here...', 'aiox-publisher-suite-pro'); ?>"><?php echo esc_textarea(get_option('aiox_license_custom_terms', '')); ?></textarea>
                                                <p class="description"><?php _e('These terms are embedded in your AIOX manifest file for AI systems to read.', 'aiox-publisher-suite-pro'); ?></p>
                                            </td>
                                        </tr>
                                    </table>
                                </div>
                            </div>
                            
                            <!-- Summary -->
                            <div class="postbox">
                                <h2 class="hndle"><span class="dashicons dashicons-visibility"></span> <?php _e('Current Settings Summary', 'aiox-publisher-suite-pro'); ?></h2>
                                <div class="inside">
                                    <div class="aiox-summary-grid">
                                        <div class="aiox-summary-item">
                                            <span><?php _e('License', 'aiox-publisher-suite-pro'); ?></span>
                                            <strong id="summary-license"><?php echo esc_html($license_presets[$license_preset]['name']); ?></strong>
                                        </div>
                                        <div class="aiox-summary-item">
                                            <span><?php _e('Monetization', 'aiox-publisher-suite-pro'); ?></span>
                                            <strong id="summary-monetization"><?php echo esc_html($monetization_presets[$monetization_preset]['name']); ?></strong>
                                        </div>
                                        <div class="aiox-summary-item">
                                            <span><?php _e('Enforcement', 'aiox-publisher-suite-pro'); ?></span>
                                            <strong id="summary-enforcement"><?php echo esc_html($enforcement_modes[$enforcement_mode]['name']); ?></strong>
                                        </div>
                                        <div class="aiox-summary-item">
                                            <span><?php _e('Posts Affected', 'aiox-publisher-suite-pro'); ?></span>
                                            <strong><?php echo number_format($processed_count); ?></strong>
                                        </div>
                                    </div>
                                    <p class="description" style="margin-top: 15px;"><?php _e('These settings are published in your AIOX manifest and TDM-REP files. Compliant AI systems will respect them.', 'aiox-publisher-suite-pro'); ?></p>
                                </div>
                            </div>
                            
                            <!-- Info & Regenerate -->
                            <div class="notice notice-info inline" style="margin: 0 0 20px 0;">
                                <p><strong><?php _e('Automatic Updates:', 'aiox-publisher-suite-pro'); ?></strong></p>
                                <ul style="margin: 5px 0 0 20px; list-style: disc;">
                                    <li><?php _e('When you save, ALL manifest files are regenerated automatically.', 'aiox-publisher-suite-pro'); ?></li>
                                    <li><?php _e('Settings apply to all existing processed posts immediately.', 'aiox-publisher-suite-pro'); ?></li>
                                    <li><?php _e('Future posts will inherit these settings when processed.', 'aiox-publisher-suite-pro'); ?></li>
                                </ul>
                                <p style="margin-top: 15px;">
                                    <button type="button" id="aiox-regenerate-manifests" class="button">
                                        <span class="dashicons dashicons-update" style="vertical-align: middle;"></span>
                                        <?php _e('Regenerate All Manifests Now', 'aiox-publisher-suite-pro'); ?>
                                    </button>
                                    <span id="regenerate-status" style="margin-left: 10px;"></span>
                                </p>
                            </div>
                            
                        </div><!-- /post-body-content -->
                        
                        <!-- Sidebar -->
                        <div id="postbox-container-1" class="postbox-container">
                            <div class="postbox">
                                <h2 class="hndle"><?php _e('Save Settings', 'aiox-publisher-suite-pro'); ?></h2>
                                <div class="inside">
                                    <?php submit_button(__('Save Monetization Settings', 'aiox-publisher-suite-pro'), 'primary', 'submit', false); ?>
                                    <p class="description" style="margin-top: 10px;"><?php _e('Settings apply to all content automatically.', 'aiox-publisher-suite-pro'); ?></p>
                                </div>
                            </div>
                            
                            <div class="postbox">
                                <h2 class="hndle"><?php _e('Quick Links', 'aiox-publisher-suite-pro'); ?></h2>
                                <div class="inside">
                                    <ul style="margin: 0;">
                                        <li><a href="<?php echo admin_url('admin.php?page=aiox-settings'); ?>">← <?php _e('Back to Settings', 'aiox-publisher-suite-pro'); ?></a></li>
                                        <li><a href="<?php echo admin_url('admin.php?page=aiox-dashboard'); ?>"><?php _e('Dashboard', 'aiox-publisher-suite-pro'); ?></a></li>
                                        <li><a href="<?php echo home_url('/.well-known/aio.json'); ?>" target="_blank"><?php _e('View Manifest', 'aiox-publisher-suite-pro'); ?> ↗</a></li>
                                    </ul>
                                </div>
                            </div>
                        </div>
                        
                    </div><!-- /post-body -->
                </div><!-- /poststuff -->
            </form>
        </div>
        
        <script type="text/javascript">
        jQuery(document).ready(function($) {
            var presetNames = <?php echo json_encode(array(
                'aiox_enforcement_mode' => array_map(function($m) { return $m['name']; }, $enforcement_modes),
                'aiox_license_preset' => array_map(function($p) { return $p['name']; }, $license_presets),
                'aiox_monetization_preset' => array_map(function($p) { return $p['name']; }, $monetization_presets)
            )); ?>;
            
            // Handle option card clicks
            $('.aiox-option-card').on('click', function() {
                var $card = $(this);
                var $radio = $card.find('input[type="radio"]');
                var radioName = $radio.attr('name');
                
                // Reset all cards in this group
                $('input[name="' + radioName + '"]').closest('.aiox-option-card').removeClass('selected');
                
                // Select this card
                $radio.prop('checked', true);
                $card.addClass('selected');
                
                // Update summary
                updateSummary();
            });
            
            function updateSummary() {
                var license = $('input[name="aiox_license_preset"]:checked').val();
                var monetization = $('input[name="aiox_monetization_preset"]:checked').val();
                var enforcement = $('input[name="aiox_enforcement_mode"]:checked').val();
                
                $('#summary-license').text(presetNames['aiox_license_preset'][license] || license);
                $('#summary-monetization').text(presetNames['aiox_monetization_preset'][monetization] || monetization);
                $('#summary-enforcement').text(presetNames['aiox_enforcement_mode'][enforcement] || enforcement);
            }
            
            // Create Terms Page button
            $('#aiox-create-terms-page').on('click', function() {
                var button = $(this);
                var status = $('#create-terms-status');
                var originalHtml = button.html();
                
                button.prop('disabled', true).html('<span class="dashicons dashicons-update spin" style="vertical-align:middle;"></span> <?php _e('Creating...', 'aiox-publisher-suite-pro'); ?>');
                status.html('<span style="color: #2271b1;"><?php _e('Creating page...', 'aiox-publisher-suite-pro'); ?></span>');
                
                $.ajax({
                    url: ajaxurl,
                    type: 'POST',
                    data: {
                        action: 'aiox_create_terms_page',
                        nonce: '<?php echo wp_create_nonce('aiox_create_terms_page'); ?>'
                    },
                    success: function(response) {
                        if (response.success) {
                            var msg = response.data.message;
                            if (response.data.page_url) {
                                msg += ' <a href="' + response.data.page_url + '" target="_blank"><?php _e('View Page', 'aiox-publisher-suite-pro'); ?></a>';
                                msg += ' | <a href="' + response.data.edit_url + '" target="_blank"><?php _e('Edit Page', 'aiox-publisher-suite-pro'); ?></a>';
                                if (!response.data.already_exists) {
                                    $('#aiox_license_terms_url').val(response.data.page_url);
                                }
                            }
                            status.html('<span style="color: #00a32a;">✓ ' + msg + '</span>');
                        } else {
                            status.html('<span style="color: #d63638;">✗ ' + (response.data.message || '<?php _e('Failed to create page', 'aiox-publisher-suite-pro'); ?>') + '</span>');
                        }
                    },
                    error: function(xhr, status, error) {
                        $('#create-terms-status').html('<span style="color: #d63638;">✗ <?php _e('Error:', 'aiox-publisher-suite-pro'); ?> ' + error + '</span>');
                    },
                    complete: function() {
                        button.prop('disabled', false).html(originalHtml);
                    }
                });
            });
            
            // Regenerate manifests button
            $('#aiox-regenerate-manifests').on('click', function() {
                var button = $(this);
                var status = $('#regenerate-status');
                var originalHtml = button.html();
                
                button.prop('disabled', true).html('<span class="dashicons dashicons-update spin" style="vertical-align:middle;"></span> <?php _e('Regenerating...', 'aiox-publisher-suite-pro'); ?>');
                status.html('<span style="color: #2271b1;"><?php _e('Processing...', 'aiox-publisher-suite-pro'); ?></span>');
                
                $.ajax({
                    url: ajaxurl,
                    type: 'POST',
                    data: {
                        action: 'aiox_regenerate_all_manifests',
                        nonce: '<?php echo wp_create_nonce('aiox_regenerate_manifests'); ?>'
                    },
                    success: function(response) {
                        if (response.success) {
                            status.html('<span style="color: #00a32a;">✓ ' + response.data.message + '</span>');
                        } else {
                            status.html('<span style="color: #d63638;">✗ ' + (response.data.message || '<?php _e('Failed to regenerate', 'aiox-publisher-suite-pro'); ?>') + '</span>');
                        }
                    },
                    error: function(xhr, status, error) {
                        $('#regenerate-status').html('<span style="color: #d63638;">✗ <?php _e('Error:', 'aiox-publisher-suite-pro'); ?> ' + error + '</span>');
                    },
                    complete: function() {
                        button.prop('disabled', false).html(originalHtml);
                    }
                });
            });
        });
        </script>
        <?php
    }
    
    /**
     * Count processed posts
     */
    private function count_processed_posts() {
        global $wpdb;
        return $wpdb->get_var("SELECT COUNT(*) FROM {$wpdb->postmeta} WHERE meta_key = '_aiox_processed' AND meta_value = '1'");
    }
    
    /**
     * Darken a hex color
     */
    private function darken_color($hex, $percent = 20) {
        $hex = str_replace('#', '', $hex);
        $r = max(0, hexdec(substr($hex, 0, 2)) - $percent);
        $g = max(0, hexdec(substr($hex, 2, 2)) - $percent);
        $b = max(0, hexdec(substr($hex, 4, 2)) - $percent);
        return sprintf('#%02x%02x%02x', $r, $g, $b);
    }
}

// Initialize
new AIOX_Monetization_Page();
