<?php
/**
 * AIOX Performance Monitor
 * Central class for tracking and displaying performance metrics
 */

if (!defined('ABSPATH')) {
    exit;
}

class AIOX_Performance_Monitor {
    
    /**
     * Get comprehensive performance statistics
     */
    public static function get_stats() {
        $manifest_file = ABSPATH . '.well-known/aio.json';
        $posts_dir = ABSPATH . '.well-known/posts';
        
        $stats = array(
            // File sizes
            'manifest_size' => 0,
            'manifest_size_human' => 'N/A',
            'posts_total_size' => 0,
            'posts_total_size_human' => 'N/A',
            'total_storage' => 0,
            'total_storage_human' => 'N/A',
            
            // Architecture
            'architecture_type' => 'Not Configured',
            'uses_individual_files' => false,
            
            // Content counts
            'total_posts' => 0,
            'processed_posts' => 0,
            'individual_files' => 0,
            'missing_files' => 0,
            
            // Per-post metrics
            'avg_post_size' => 0,
            'avg_post_size_human' => 'N/A',
            'min_post_size' => 0,
            'min_post_size_human' => 'N/A',
            'max_post_size' => 0,
            'max_post_size_human' => 'N/A',
            
            // Performance metrics
            'estimated_load_time_3g' => 'N/A',
            'estimated_load_time_4g' => 'N/A',
            'estimated_load_time_wifi' => 'N/A',
            'performance_rating' => 'Unknown',
            'performance_score' => 0,
            
            // Health indicators
            'manifest_status' => 'unknown',
            'architecture_status' => 'unknown',
            'overall_status' => 'unknown',
            
            // Recommendations
            'show_recommendation' => false,
            'recommendation' => '',
            'show_upgrade_link' => false,
            
            // Timestamps
            'manifest_last_modified' => 'N/A',
            'last_post_generated' => 'N/A'
        );
        
        // Get manifest info
        if (file_exists($manifest_file)) {
            $stats['manifest_size'] = filesize($manifest_file);
            $stats['manifest_size_human'] = size_format($stats['manifest_size']);
            $stats['manifest_last_modified'] = date('Y-m-d H:i:s', filemtime($manifest_file));
            
            // Determine architecture
            $manifest_content = file_get_contents($manifest_file);
            $manifest_data = json_decode($manifest_content, true);
            
            if (isset($manifest_data['manifest_type']) && $manifest_data['manifest_type'] === 'lightweight_index') {
                $stats['architecture_type'] = 'Light Manifest + Individual Files';
                $stats['uses_individual_files'] = true;
            } else if (isset($manifest_data['features']['individual_post_files']) && $manifest_data['features']['individual_post_files']) {
                $stats['architecture_type'] = 'Light Manifest + Individual Files';
                $stats['uses_individual_files'] = true;
            } else {
                $stats['architecture_type'] = 'All-in-One (Legacy)';
                $stats['uses_individual_files'] = false;
            }
        }
        
        // Get post counts
        $all_posts = wp_count_posts('post');
        $all_pages = wp_count_posts('page');
        $stats['total_posts'] = $all_posts->publish + $all_pages->publish;
        
        $processed = get_posts(array(
            'post_type' => array('post', 'page'),
            'post_status' => 'publish',
            'numberposts' => -1,
            'fields' => 'ids',
            'meta_query' => array(
                array(
                    'key' => '_aiox_processed',
                    'value' => '1',
                    'compare' => '='
                )
            )
        ));
        $stats['processed_posts'] = count($processed);
        
        // Get individual files info
        if (is_dir($posts_dir)) {
            $files = glob($posts_dir . '/*.json');
            $stats['individual_files'] = count($files);
            
            if ($stats['individual_files'] > 0) {
                $sizes = array();
                $total_size = 0;
                $latest_mtime = 0;
                
                foreach ($files as $file) {
                    $size = filesize($file);
                    $sizes[] = $size;
                    $total_size += $size;
                    
                    $mtime = filemtime($file);
                    if ($mtime > $latest_mtime) {
                        $latest_mtime = $mtime;
                    }
                }
                
                $stats['posts_total_size'] = $total_size;
                $stats['posts_total_size_human'] = size_format($total_size);
                $stats['total_storage'] = $total_size + $stats['manifest_size'];
                $stats['total_storage_human'] = size_format($stats['total_storage']);
                
                $stats['avg_post_size'] = $total_size / $stats['individual_files'];
                $stats['avg_post_size_human'] = size_format($stats['avg_post_size']);
                
                $stats['min_post_size'] = min($sizes);
                $stats['min_post_size_human'] = size_format($stats['min_post_size']);
                
                $stats['max_post_size'] = max($sizes);
                $stats['max_post_size_human'] = size_format($stats['max_post_size']);
                
                if ($latest_mtime > 0) {
                    $stats['last_post_generated'] = date('Y-m-d H:i:s', $latest_mtime);
                }
            }
            
            // Check for missing files
            $stats['missing_files'] = $stats['processed_posts'] - $stats['individual_files'];
        } else {
            $stats['missing_files'] = $stats['processed_posts'];
        }
        
        // Calculate load times (rough estimates)
        if ($stats['manifest_size'] > 0) {
            // 3G: ~750 Kbps = ~94 KB/s
            $stats['estimated_load_time_3g'] = round($stats['manifest_size'] / (94 * 1024), 1) . 's';
            
            // 4G: ~10 Mbps = ~1.25 MB/s
            $stats['estimated_load_time_4g'] = round($stats['manifest_size'] / (1.25 * 1024 * 1024), 2) . 's';
            
            // WiFi: ~50 Mbps = ~6.25 MB/s
            $stats['estimated_load_time_wifi'] = round($stats['manifest_size'] / (6.25 * 1024 * 1024), 2) . 's';
        }
        
        // Calculate performance rating and score
        $score = 100;
        
        // Deduct points for large manifest
        if ($stats['manifest_size'] > 10485760) { // > 10MB
            $score -= 50;
            $stats['performance_rating'] = 'Poor';
            $stats['manifest_status'] = 'error';
        } else if ($stats['manifest_size'] > 5242880) { // > 5MB
            $score -= 30;
            $stats['performance_rating'] = 'Fair';
            $stats['manifest_status'] = 'warning';
        } else if ($stats['manifest_size'] > 1048576) { // > 1MB
            $score -= 15;
            $stats['performance_rating'] = 'Good';
            $stats['manifest_status'] = 'fair';
        } else {
            $stats['performance_rating'] = 'Excellent';
            $stats['manifest_status'] = 'excellent';
        }
        
        // Architecture status
        if ($stats['uses_individual_files']) {
            $stats['architecture_status'] = 'excellent';
        } else if ($stats['processed_posts'] > 500) {
            $score -= 30;
            $stats['architecture_status'] = 'error';
        } else if ($stats['processed_posts'] > 100) {
            $score -= 15;
            $stats['architecture_status'] = 'warning';
        } else {
            $stats['architecture_status'] = 'good';
        }
        
        // Missing files penalty
        if ($stats['missing_files'] > 0) {
            $score -= min($stats['missing_files'] * 2, 20);
        }
        
        $stats['performance_score'] = max(0, $score);
        
        // Overall status
        if ($stats['performance_score'] >= 80) {
            $stats['overall_status'] = 'excellent';
        } else if ($stats['performance_score'] >= 60) {
            $stats['overall_status'] = 'good';
        } else if ($stats['performance_score'] >= 40) {
            $stats['overall_status'] = 'fair';
        } else {
            $stats['overall_status'] = 'poor';
        }
        
        // Generate recommendations
        if (!$stats['uses_individual_files'] && $stats['processed_posts'] > 100) {
            $stats['show_recommendation'] = true;
            $stats['show_upgrade_link'] = true;
            $stats['recommendation'] = sprintf(
                'With %d processed posts, upgrading to Light Manifest architecture would improve load times by 10-50x and reduce bandwidth by 90%%.',
                $stats['processed_posts']
            );
        } else if ($stats['manifest_size'] > 10485760) {
            $stats['show_recommendation'] = true;
            $stats['show_upgrade_link'] = true;
            $stats['recommendation'] = sprintf(
                'Your manifest is very large (%s). Immediate upgrade to Light Manifest architecture is strongly recommended.',
                $stats['manifest_size_human']
            );
        } else if ($stats['missing_files'] > 0) {
            $stats['show_recommendation'] = true;
            $stats['recommendation'] = sprintf(
                '%d processed posts are missing individual files. Run the backfill tool to generate them.',
                $stats['missing_files']
            );
        }
        
        return $stats;
    }
    
    /**
     * Get a specific stat
     */
    public static function get($key) {
        $stats = self::get_stats();
        return isset($stats[$key]) ? $stats[$key] : null;
    }
    
    /**
     * Display dashboard widget
     */
    public static function render_dashboard_widget() {
        $stats = self::get_stats();
        include AIOX_PLUGIN_DIR . 'admin/views/performance-dashboard-widget.php';
    }
    
    /**
     * Display full performance page
     */
    public static function render_performance_page() {
        $stats = self::get_stats();
        include AIOX_PLUGIN_DIR . 'admin/views/performance-page.php';
    }
    
    /**
     * Get recommendations
     */
    public static function get_recommendations() {
        $stats = self::get_stats();
        $recommendations = array();
        
        if ($stats['show_recommendation']) {
            $recommendations[] = array(
                'type' => $stats['show_upgrade_link'] ? 'critical' : 'warning',
                'message' => $stats['recommendation'],
                'action_url' => $stats['show_upgrade_link'] ? admin_url('admin.php?page=aiox-backfill') : null,
                'action_text' => $stats['show_upgrade_link'] ? 'Upgrade Now' : null
            );
        }
        
        return $recommendations;
    }
}