<?php
/**
 * AIOX Processing Manager Helper Methods
 * 
 * Contains helper methods for the Processing Manager to avoid trait compatibility issues
 * and provide fallback functionality.
 */

if (!defined('ABSPATH')) {
    exit;
}

class AIOX_Processing_Manager_Helper {
    
    /**
     * Get overall processing status
     */
    public static function get_overall_processing_status() {
        $posts = get_posts(array(
            'post_type' => array('post', 'page'),
            'post_status' => 'publish',
            'numberposts' => -1,
            'fields' => 'ids'
        ));
        
        $total_posts = count($posts);
        $processed_posts = 0;
        
        foreach ($posts as $post_id) {
            if (get_post_meta($post_id, '_aiox_processing_complete', true)) {
                $processed_posts++;
            }
        }
        
        $unprocessed_posts = $total_posts - $processed_posts;
        $processing_percentage = $total_posts > 0 ? round(($processed_posts / $total_posts) * 100, 1) : 0;
        
        return array(
            'total_posts' => $total_posts,
            'processed_posts' => $processed_posts,
            'unprocessed_posts' => $unprocessed_posts,
            'processing_percentage' => $processing_percentage
        );
    }
    
    /**
     * Get post processing status
     */
    public static function get_post_processing_status($post_id) {
        $ingredients = get_post_meta($post_id, '_aiox_ingredients', true);
        $ingredients_count = 0;
        
        if (!empty($ingredients)) {
            if (is_array($ingredients)) {
                $ingredients_count = count($ingredients);
            } elseif (is_string($ingredients)) {
                $decoded = json_decode($ingredients, true);
                if (is_array($decoded)) {
                    $ingredients_count = count($decoded);
                }
            }
        }
        
        return array(
            'has_capsule' => !empty(get_post_meta($post_id, '_aiox_capsule_id', true)),
            'has_ingredients' => !empty($ingredients),
            'ingredients_count' => $ingredients_count,
            'has_processed_data' => !empty(get_post_meta($post_id, '_aiox_processed_data', true)),
            'processing_complete' => get_post_meta($post_id, '_aiox_processing_complete', true)
        );
    }
    
    /**
     * Get unprocessed posts
     */
    public static function get_unprocessed_posts($limit = 50) {
        global $wpdb;
        
        $query = "
            SELECT p.ID, p.post_title, p.post_type, p.post_date
            FROM {$wpdb->posts} p
            LEFT JOIN {$wpdb->postmeta} pm ON p.ID = pm.post_id AND pm.meta_key = '_aiox_processing_complete'
            WHERE p.post_type IN ('post', 'page')
            AND p.post_status = 'publish'
            AND (pm.meta_value IS NULL OR pm.meta_value = '' OR pm.meta_value = '0')
            ORDER BY p.post_date DESC
            LIMIT %d
        ";
        
        $results = $wpdb->get_results($wpdb->prepare($query, $limit));
        
        $posts = array();
        foreach ($results as $result) {
            $posts[] = array(
                'id' => $result->ID,
                'title' => $result->post_title,
                'type' => $result->post_type,
                'date' => $result->post_date,
                'url' => get_permalink($result->ID)
            );
        }
        
        return $posts;
    }
    
    /**
     * Render Overview Tab
     */
    public static function render_simple_overview_tab() {
        $status = self::get_overall_processing_status();
        $comprehensive_enabled = get_option('aiox_enable_comprehensive_processing', true);
        ?>
        
        <!-- Stats Grid -->
        <div class="aiox-stats-grid">
            <div class="aiox-stat-card">
                <h3><?php _e('Total Content', 'aiox-publisher-suite-pro'); ?></h3>
                <div class="stat-value blue"><?php echo $status['total_posts']; ?></div>
            </div>
            <div class="aiox-stat-card">
                <h3><?php _e('Processed', 'aiox-publisher-suite-pro'); ?></h3>
                <div class="stat-value green"><?php echo $status['processed_posts']; ?></div>
            </div>
            <div class="aiox-stat-card">
                <h3><?php _e('Pending', 'aiox-publisher-suite-pro'); ?></h3>
                <div class="stat-value red"><?php echo $status['unprocessed_posts']; ?></div>
            </div>
            <div class="aiox-stat-card">
                <h3><?php _e('Completion', 'aiox-publisher-suite-pro'); ?></h3>
                <div class="stat-value blue"><?php echo $status['processing_percentage']; ?>%</div>
            </div>
        </div>
        
        <!-- Progress Bar -->
        <div class="aiox-progress-container">
            <div style="display: flex; justify-content: space-between; align-items: center;">
                <strong><?php _e('Processing Progress', 'aiox-publisher-suite-pro'); ?></strong>
                <span><?php echo $status['processed_posts']; ?> / <?php echo $status['total_posts']; ?></span>
            </div>
            <div class="aiox-progress-bar">
                <div class="aiox-progress-fill" style="width: <?php echo $status['processing_percentage']; ?>%;"></div>
            </div>
        </div>
        
        <div id="poststuff">
            <div id="post-body" class="metabox-holder columns-2">
                <div id="post-body-content">
                    
                    <!-- Quick Actions -->
                    <?php if ($status['unprocessed_posts'] > 0): ?>
                    <div class="postbox" style="border-left: 4px solid #2271b1;">
                        <h2 class="hndle"><span class="dashicons dashicons-controls-play"></span> <?php _e('Quick Actions', 'aiox-publisher-suite-pro'); ?></h2>
                        <div class="inside">
                            <p><?php printf(__('You have <strong>%d unprocessed posts</strong> ready for AI processing.', 'aiox-publisher-suite-pro'), $status['unprocessed_posts']); ?></p>
                            <p>
                                <button type="button" class="button button-primary button-hero" id="aiox-process-all-unprocessed">
                                    <span class="dashicons dashicons-update" style="vertical-align: middle;"></span>
                                    <?php printf(__('Process All %d Posts', 'aiox-publisher-suite-pro'), $status['unprocessed_posts']); ?>
                                </button>
                            </p>
                            <div id="aiox-overview-status" style="display: none; margin-top: 15px;"></div>
                        </div>
                    </div>
                    <?php else: ?>
                    <div class="notice notice-success" style="margin: 0 0 20px 0;">
                        <p><span class="dashicons dashicons-yes-alt" style="color: #00a32a;"></span> <strong><?php _e('All content processed!', 'aiox-publisher-suite-pro'); ?></strong> <?php _e('Your site is fully optimized for AI.', 'aiox-publisher-suite-pro'); ?></p>
                    </div>
                    <?php endif; ?>
                    
                    <!-- Processing Components Status -->
                    <div class="postbox">
                        <h2 class="hndle"><span class="dashicons dashicons-admin-plugins"></span> <?php _e('Processing Components', 'aiox-publisher-suite-pro'); ?></h2>
                        <div class="inside">
                            <table class="widefat striped">
                                <thead>
                                    <tr>
                                        <th><?php _e('Component', 'aiox-publisher-suite-pro'); ?></th>
                                        <th style="width: 100px; text-align: center;"><?php _e('Status', 'aiox-publisher-suite-pro'); ?></th>
                                    </tr>
                                </thead>
                                <tbody>
                                    <tr>
                                        <td>
                                            <strong><?php _e('Capsules', 'aiox-publisher-suite-pro'); ?></strong>
                                            <br><small class="description"><?php _e('AI-optimized content summaries', 'aiox-publisher-suite-pro'); ?></small>
                                        </td>
                                        <td style="text-align: center;"><span style="color: #00a32a;"><span class="dashicons dashicons-yes-alt"></span> <?php _e('Active', 'aiox-publisher-suite-pro'); ?></span></td>
                                    </tr>
                                    <tr>
                                        <td>
                                            <strong><?php _e('Ingredients', 'aiox-publisher-suite-pro'); ?></strong>
                                            <br><small class="description"><?php _e('Semantic content breakdown', 'aiox-publisher-suite-pro'); ?></small>
                                        </td>
                                        <td style="text-align: center;">
                                            <?php if (get_option('aiox_auto_extract_ingredients', true)): ?>
                                                <span style="color: #00a32a;"><span class="dashicons dashicons-yes-alt"></span> <?php _e('Active', 'aiox-publisher-suite-pro'); ?></span>
                                            <?php else: ?>
                                                <span style="color: #646970;"><span class="dashicons dashicons-minus"></span> <?php _e('Disabled', 'aiox-publisher-suite-pro'); ?></span>
                                            <?php endif; ?>
                                        </td>
                                    </tr>
                                    <tr>
                                        <td>
                                            <strong><?php _e('Manifesto Files', 'aiox-publisher-suite-pro'); ?></strong>
                                            <br><small class="description"><?php _e('AI discovery JSON files', 'aiox-publisher-suite-pro'); ?></small>
                                        </td>
                                        <td style="text-align: center;">
                                            <?php if (get_option('aiox_auto_update_manifesto', true)): ?>
                                                <span style="color: #00a32a;"><span class="dashicons dashicons-yes-alt"></span> <?php _e('Active', 'aiox-publisher-suite-pro'); ?></span>
                                            <?php else: ?>
                                                <span style="color: #646970;"><span class="dashicons dashicons-minus"></span> <?php _e('Disabled', 'aiox-publisher-suite-pro'); ?></span>
                                            <?php endif; ?>
                                        </td>
                                    </tr>
                                    <tr>
                                        <td>
                                            <strong><?php _e('Badge Data', 'aiox-publisher-suite-pro'); ?></strong>
                                            <br><small class="description"><?php _e('Frontend AI badge content', 'aiox-publisher-suite-pro'); ?></small>
                                        </td>
                                        <td style="text-align: center;">
                                            <?php if (get_option('aiox_auto_update_badges', true)): ?>
                                                <span style="color: #00a32a;"><span class="dashicons dashicons-yes-alt"></span> <?php _e('Active', 'aiox-publisher-suite-pro'); ?></span>
                                            <?php else: ?>
                                                <span style="color: #646970;"><span class="dashicons dashicons-minus"></span> <?php _e('Disabled', 'aiox-publisher-suite-pro'); ?></span>
                                            <?php endif; ?>
                                        </td>
                                    </tr>
                                    <tr>
                                        <td>
                                            <strong><?php _e('Metadata Classification', 'aiox-publisher-suite-pro'); ?></strong>
                                            <br><small class="description"><?php _e('AI content analysis', 'aiox-publisher-suite-pro'); ?></small>
                                        </td>
                                        <td style="text-align: center;">
                                            <?php if (get_option('aiox_auto_classify_metadata', true)): ?>
                                                <span style="color: #00a32a;"><span class="dashicons dashicons-yes-alt"></span> <?php _e('Active', 'aiox-publisher-suite-pro'); ?></span>
                                            <?php else: ?>
                                                <span style="color: #646970;"><span class="dashicons dashicons-minus"></span> <?php _e('Disabled', 'aiox-publisher-suite-pro'); ?></span>
                                            <?php endif; ?>
                                        </td>
                                    </tr>
                                </tbody>
                            </table>
                            <p style="margin-top: 15px;">
                                <a href="?page=aiox-capsules&tab=settings" class="button">
                                    <span class="dashicons dashicons-admin-settings" style="vertical-align: middle;"></span>
                                    <?php _e('Configure Components', 'aiox-publisher-suite-pro'); ?>
                                </a>
                            </p>
                        </div>
                    </div>
                    
                </div><!-- /post-body-content -->
                
                <!-- Sidebar -->
                <div id="postbox-container-1" class="postbox-container">
                    <div class="postbox">
                        <h2 class="hndle"><?php _e('Comprehensive Processing', 'aiox-publisher-suite-pro'); ?></h2>
                        <div class="inside">
                            <p>
                                <strong><?php _e('Status:', 'aiox-publisher-suite-pro'); ?></strong>
                                <?php if ($comprehensive_enabled): ?>
                                    <span style="color: #00a32a;"><span class="dashicons dashicons-yes-alt"></span> <?php _e('Enabled', 'aiox-publisher-suite-pro'); ?></span>
                                <?php else: ?>
                                    <span style="color: #dba617;"><span class="dashicons dashicons-warning"></span> <?php _e('Disabled', 'aiox-publisher-suite-pro'); ?></span>
                                <?php endif; ?>
                            </p>
                            <p class="description"><?php _e('When enabled, all components are processed automatically on save.', 'aiox-publisher-suite-pro'); ?></p>
                        </div>
                    </div>
                    
                    <div class="postbox">
                        <h2 class="hndle"><?php _e('Quick Links', 'aiox-publisher-suite-pro'); ?></h2>
                        <div class="inside">
                            <ul style="margin: 0;">
                                <li><a href="?page=aiox-capsules&tab=manage"><?php _e('Manage Content', 'aiox-publisher-suite-pro'); ?></a></li>
                                <li><a href="?page=aiox-capsules&tab=bulk"><?php _e('Bulk Processing', 'aiox-publisher-suite-pro'); ?></a></li>
                                <li><a href="?page=aiox-capsules&tab=settings"><?php _e('Settings', 'aiox-publisher-suite-pro'); ?></a></li>
                                <li><a href="<?php echo admin_url('admin.php?page=aiox-modern-analytics'); ?>"><?php _e('Analytics', 'aiox-publisher-suite-pro'); ?></a></li>
                            </ul>
                        </div>
                    </div>
                </div>
                
            </div><!-- /post-body -->
        </div><!-- /poststuff -->
        
        <script type="text/javascript">
        jQuery(document).ready(function($) {
            var nonce = '<?php echo wp_create_nonce('aiox_capsules_nonce'); ?>';
            
            $('#aiox-process-all-unprocessed').on('click', function(e) {
                e.preventDefault();
                
                var button = $(this);
                var statusDiv = $('#aiox-overview-status');
                
                if (!confirm('<?php echo esc_js(__('This will process all unprocessed posts. This may take several minutes and consume API tokens. Continue?', 'aiox-publisher-suite-pro')); ?>')) {
                    return;
                }
                
                button.prop('disabled', true).html('<span class="dashicons dashicons-update" style="vertical-align: middle; animation: rotation 1s infinite linear;"></span> <?php echo esc_js(__('Starting...', 'aiox-publisher-suite-pro')); ?>');
                statusDiv.html('<div class="notice notice-info inline"><p><?php echo esc_js(__('Getting list of unprocessed posts...', 'aiox-publisher-suite-pro')); ?></p></div>').show();
                
                $.ajax({
                    url: ajaxurl,
                    type: 'POST',
                    data: { action: 'aiox_get_unprocessed_posts', nonce: nonce },
                    success: function(response) {
                        if (response.success && response.data.posts && response.data.posts.length > 0) {
                            processPostsSequentially(response.data.posts, button, statusDiv);
                        } else {
                            statusDiv.html('<div class="notice notice-success inline"><p><?php echo esc_js(__('No unprocessed posts found!', 'aiox-publisher-suite-pro')); ?></p></div>');
                            button.prop('disabled', false).html('<span class="dashicons dashicons-update" style="vertical-align: middle;"></span> <?php echo esc_js(__('Process All', 'aiox-publisher-suite-pro')); ?>');
                        }
                    },
                    error: function() {
                        statusDiv.html('<div class="notice notice-error inline"><p><?php echo esc_js(__('Failed to get posts list.', 'aiox-publisher-suite-pro')); ?></p></div>');
                        button.prop('disabled', false).html('<span class="dashicons dashicons-update" style="vertical-align: middle;"></span> <?php echo esc_js(__('Process All', 'aiox-publisher-suite-pro')); ?>');
                    }
                });
            });
            
            function processPostsSequentially(posts, button, statusDiv) {
                var total = posts.length;
                var current = 0;
                var success = 0;
                var failed = 0;
                
                function processNext() {
                    if (current >= total) {
                        var msg = success + ' <?php echo esc_js(__('processed', 'aiox-publisher-suite-pro')); ?>';
                        if (failed > 0) msg += ', ' + failed + ' <?php echo esc_js(__('failed', 'aiox-publisher-suite-pro')); ?>';
                        
                        statusDiv.html('<div class="notice notice-success inline"><p><strong><?php echo esc_js(__('Complete!', 'aiox-publisher-suite-pro')); ?></strong> ' + msg + '</p></div>');
                        button.html('<span class="dashicons dashicons-yes" style="vertical-align: middle;"></span> <?php echo esc_js(__('Done! Reloading...', 'aiox-publisher-suite-pro')); ?>');
                        
                        setTimeout(function() { location.reload(); }, 2000);
                        return;
                    }
                    
                    var post = posts[current];
                    var percent = Math.round(((current + 1) / total) * 100);
                    
                    button.html('<span class="dashicons dashicons-update" style="vertical-align: middle; animation: rotation 1s infinite linear;"></span> ' + (current + 1) + '/' + total + ' (' + percent + '%)');
                    statusDiv.html('<div class="notice notice-info inline"><p><?php echo esc_js(__('Processing:', 'aiox-publisher-suite-pro')); ?> <strong>' + post.title + '</strong></p></div>');
                    
                    $.ajax({
                        url: ajaxurl,
                        type: 'POST',
                        timeout: 60000,
                        data: { action: 'aiox_process_all_components', post_id: post.id, nonce: nonce },
                        success: function(response) {
                            if (response.success) success++;
                            else failed++;
                        },
                        error: function() { failed++; },
                        complete: function() {
                            current++;
                            if (typeof aioxRefreshBalance === 'function') aioxRefreshBalance();
                            setTimeout(processNext, 500);
                        }
                    });
                }
                
                processNext();
            }
        });
        </script>
        
        <style>
        @keyframes rotation {
            from { transform: rotate(0deg); }
            to { transform: rotate(360deg); }
        }
        </style>
        <?php
    }
    
    /**
     * Render Manage Content Tab
     */
    public static function render_simple_manage_tab() {
        $nonce = wp_create_nonce('aiox_capsules_nonce');
        ?>
        
        <!-- Notification Area -->
        <div id="aiox-processing-notification" class="notice" style="display: none; margin: 0 0 20px 0;">
            <p id="aiox-notification-content"></p>
        </div>
        
        <!-- Controls -->
        <div class="postbox" style="margin-bottom: 20px;">
            <div class="inside" style="padding: 15px; display: flex; align-items: center; gap: 15px; flex-wrap: wrap;">
                <button type="button" class="button" id="aiox-refresh-content">
                    <span class="dashicons dashicons-update" style="vertical-align: middle;"></span>
                    <?php _e('Refresh List', 'aiox-publisher-suite-pro'); ?>
                </button>
                
                <label style="display: flex; align-items: center; gap: 5px;">
                    <input type="checkbox" id="aiox-show-unprocessed-only" />
                    <?php _e('Show Unprocessed Only', 'aiox-publisher-suite-pro'); ?>
                </label>
                
                <select id="aiox-filter-type" class="postform">
                    <option value=""><?php _e('All Types', 'aiox-publisher-suite-pro'); ?></option>
                    <option value="post"><?php _e('Posts', 'aiox-publisher-suite-pro'); ?></option>
                    <option value="page"><?php _e('Pages', 'aiox-publisher-suite-pro'); ?></option>
                </select>
            </div>
        </div>
        
        <!-- Content List -->
        <div class="postbox">
            <h2 class="hndle"><span class="dashicons dashicons-admin-page"></span> <?php _e('Content List', 'aiox-publisher-suite-pro'); ?></h2>
            <div class="inside" style="padding: 0;">
                <div id="aiox-content-list-container">
                    <table class="widefat striped">
                        <thead>
                            <tr>
                                <th style="width: 40%;"><?php _e('Title', 'aiox-publisher-suite-pro'); ?></th>
                                <th><?php _e('Type', 'aiox-publisher-suite-pro'); ?></th>
                                <th><?php _e('Status', 'aiox-publisher-suite-pro'); ?></th>
                                <th><?php _e('Components', 'aiox-publisher-suite-pro'); ?></th>
                                <th style="width: 200px;"><?php _e('Actions', 'aiox-publisher-suite-pro'); ?></th>
                            </tr>
                        </thead>
                        <tbody id="aiox-content-tbody">
                            <tr>
                                <td colspan="5" style="text-align: center; padding: 40px;">
                                    <span class="dashicons dashicons-update" style="animation: rotation 1s infinite linear; font-size: 24px;"></span>
                                    <p><?php _e('Loading content...', 'aiox-publisher-suite-pro'); ?></p>
                                </td>
                            </tr>
                        </tbody>
                    </table>
                </div>
            </div>
        </div>
        
        <script type="text/javascript">
        jQuery(document).ready(function($) {
            var nonce = '<?php echo $nonce; ?>';
            
            function showNotification(type, message) {
                var notification = $('#aiox-processing-notification');
                var content = $('#aiox-notification-content');
                
                notification.removeClass('notice-success notice-error notice-warning notice-info')
                            .addClass('notice-' + type)
                            .show();
                content.html(message);
                
                if (type === 'success') {
                    setTimeout(function() { notification.fadeOut(); }, 5000);
                }
            }
            
            function loadContentList() {
                var showUnprocessedOnly = $('#aiox-show-unprocessed-only').prop('checked');
                var filterType = $('#aiox-filter-type').val();
                
                $('#aiox-content-tbody').html('<tr><td colspan="5" style="text-align: center; padding: 40px;"><span class="dashicons dashicons-update" style="animation: rotation 1s infinite linear;"></span> <?php echo esc_js(__('Loading...', 'aiox-publisher-suite-pro')); ?></td></tr>');
                
                $.ajax({
                    url: ajaxurl,
                    type: 'POST',
                    data: {
                        action: 'aiox_get_content_list',
                        nonce: nonce,
                        unprocessed_only: showUnprocessedOnly ? 1 : 0,
                        post_type: filterType
                    },
                    success: function(response) {
                        if (response.success && response.data.posts) {
                            renderContentTable(response.data.posts);
                        } else {
                            $('#aiox-content-tbody').html('<tr><td colspan="5" style="text-align: center; padding: 40px;"><?php echo esc_js(__('No content found.', 'aiox-publisher-suite-pro')); ?></td></tr>');
                        }
                    },
                    error: function() {
                        $('#aiox-content-tbody').html('<tr><td colspan="5" style="text-align: center; padding: 40px; color: #d63638;"><?php echo esc_js(__('Failed to load content.', 'aiox-publisher-suite-pro')); ?></td></tr>');
                    }
                });
            }
            
            function renderContentTable(posts) {
                if (posts.length === 0) {
                    $('#aiox-content-tbody').html('<tr><td colspan="5" style="text-align: center; padding: 40px;"><?php echo esc_js(__('No content found.', 'aiox-publisher-suite-pro')); ?></td></tr>');
                    return;
                }
                
                var html = '';
                posts.forEach(function(post) {
                    var statusHtml = post.processing_complete 
                        ? '<span style="color: #00a32a;"><span class="dashicons dashicons-yes-alt"></span> <?php echo esc_js(__('Processed', 'aiox-publisher-suite-pro')); ?></span>'
                        : '<span style="color: #dba617;"><span class="dashicons dashicons-clock"></span> <?php echo esc_js(__('Pending', 'aiox-publisher-suite-pro')); ?></span>';
                    
                    var componentsHtml = [];
                    if (post.has_capsule) componentsHtml.push('<span title="<?php echo esc_attr__('Capsule', 'aiox-publisher-suite-pro'); ?>">📦</span>');
                    if (post.has_ingredients) componentsHtml.push('<span title="<?php echo esc_attr__('Ingredients', 'aiox-publisher-suite-pro'); ?>">🧩</span>');
                    if (post.has_processed_data) componentsHtml.push('<span title="<?php echo esc_attr__('Metadata', 'aiox-publisher-suite-pro'); ?>">📊</span>');
                    
                    var actionsHtml = post.processing_complete
                        ? '<button type="button" class="button button-small aiox-regenerate-btn" data-post-id="' + post.id + '"><span class="dashicons dashicons-update" style="vertical-align: middle;"></span> <?php echo esc_js(__('Regenerate', 'aiox-publisher-suite-pro')); ?></button>'
                        : '<button type="button" class="button button-primary button-small aiox-process-btn" data-post-id="' + post.id + '"><span class="dashicons dashicons-controls-play" style="vertical-align: middle;"></span> <?php echo esc_js(__('Process', 'aiox-publisher-suite-pro')); ?></button>';
                    
                    actionsHtml += ' <a href="' + post.edit_url + '" class="button button-small" target="_blank"><?php echo esc_js(__('Edit', 'aiox-publisher-suite-pro')); ?></a>';
                    
                    html += '<tr>';
                    html += '<td><strong>' + post.title + '</strong></td>';
                    html += '<td>' + post.type + '</td>';
                    html += '<td>' + statusHtml + '</td>';
                    html += '<td>' + (componentsHtml.length > 0 ? componentsHtml.join(' ') : '<span style="color: #646970;">—</span>') + '</td>';
                    html += '<td>' + actionsHtml + '</td>';
                    html += '</tr>';
                });
                
                $('#aiox-content-tbody').html(html);
            }
            
            // Load on page load
            loadContentList();
            
            // Refresh button
            $('#aiox-refresh-content').on('click', loadContentList);
            
            // Filters
            $('#aiox-show-unprocessed-only, #aiox-filter-type').on('change', loadContentList);
            
            // Process button
            $(document).on('click', '.aiox-process-btn', function(e) {
                e.preventDefault();
                var button = $(this);
                var postId = button.data('post-id');
                
                button.prop('disabled', true).html('<span class="dashicons dashicons-update" style="vertical-align: middle; animation: rotation 1s infinite linear;"></span>');
                
                $.ajax({
                    url: ajaxurl,
                    type: 'POST',
                    data: { action: 'aiox_process_all_components', post_id: postId, nonce: nonce },
                    success: function(response) {
                        if (typeof aioxRefreshBalance === 'function') aioxRefreshBalance();
                        
                        if (response.success) {
                            showNotification('success', '<?php echo esc_js(__('Processing complete!', 'aiox-publisher-suite-pro')); ?>');
                            setTimeout(loadContentList, 1000);
                        } else {
                            var msg = response.data && response.data.message ? response.data.message : '<?php echo esc_js(__('Processing failed', 'aiox-publisher-suite-pro')); ?>';
                            showNotification('error', msg);
                            button.prop('disabled', false).html('<span class="dashicons dashicons-controls-play" style="vertical-align: middle;"></span> <?php echo esc_js(__('Process', 'aiox-publisher-suite-pro')); ?>');
                        }
                    },
                    error: function() {
                        showNotification('error', '<?php echo esc_js(__('Network error', 'aiox-publisher-suite-pro')); ?>');
                        button.prop('disabled', false).html('<span class="dashicons dashicons-controls-play" style="vertical-align: middle;"></span> <?php echo esc_js(__('Process', 'aiox-publisher-suite-pro')); ?>');
                    }
                });
            });
            
            // Regenerate button
            $(document).on('click', '.aiox-regenerate-btn', function(e) {
                e.preventDefault();
                
                if (!confirm('<?php echo esc_js(__('This will delete existing data and regenerate. Continue?', 'aiox-publisher-suite-pro')); ?>')) {
                    return;
                }
                
                var button = $(this);
                var postId = button.data('post-id');
                
                button.prop('disabled', true).html('<span class="dashicons dashicons-update" style="vertical-align: middle; animation: rotation 1s infinite linear;"></span>');
                
                // Clear then process
                $.ajax({
                    url: ajaxurl,
                    type: 'POST',
                    data: { action: 'aiox_clear_all_ai_data', post_id: postId, nonce: nonce },
                    success: function() {
                        $.ajax({
                            url: ajaxurl,
                            type: 'POST',
                            data: { action: 'aiox_process_all_components', post_id: postId, nonce: nonce },
                            success: function(response) {
                                if (typeof aioxRefreshBalance === 'function') aioxRefreshBalance();
                                
                                if (response.success) {
                                    showNotification('success', '<?php echo esc_js(__('Regeneration complete!', 'aiox-publisher-suite-pro')); ?>');
                                    setTimeout(loadContentList, 1000);
                                } else {
                                    showNotification('error', response.data && response.data.message ? response.data.message : '<?php echo esc_js(__('Failed', 'aiox-publisher-suite-pro')); ?>');
                                    button.prop('disabled', false).html('<span class="dashicons dashicons-update" style="vertical-align: middle;"></span> <?php echo esc_js(__('Regenerate', 'aiox-publisher-suite-pro')); ?>');
                                }
                            },
                            error: function() {
                                showNotification('error', '<?php echo esc_js(__('Network error', 'aiox-publisher-suite-pro')); ?>');
                                button.prop('disabled', false).html('<span class="dashicons dashicons-update" style="vertical-align: middle;"></span> <?php echo esc_js(__('Regenerate', 'aiox-publisher-suite-pro')); ?>');
                            }
                        });
                    }
                });
            });
        });
        </script>
        
        <style>
        @keyframes rotation {
            from { transform: rotate(0deg); }
            to { transform: rotate(360deg); }
        }
        </style>
        <?php
    }
    
    /**
     * Render Settings Tab
     */
    public static function render_simple_settings_tab() {
        // Handle form submission
        if (isset($_POST['save_processing_settings']) && wp_verify_nonce($_POST['aiox_processing_settings_nonce'], 'aiox_processing_settings')) {
            $settings = array(
                'aiox_enable_comprehensive_processing',
                'aiox_auto_extract_ingredients',
                'aiox_auto_update_manifesto',
                'aiox_auto_update_badges',
                'aiox_auto_classify_metadata'
            );
            
            foreach ($settings as $setting) {
                update_option($setting, isset($_POST[$setting]) && $_POST[$setting] === '1');
            }
            
            echo '<div class="notice notice-success is-dismissible"><p>' . __('Settings saved successfully!', 'aiox-publisher-suite-pro') . '</p></div>';
        }
        ?>
        
        <form method="post" action="">
            <?php wp_nonce_field('aiox_processing_settings', 'aiox_processing_settings_nonce'); ?>
            
            <div id="poststuff">
                <div id="post-body" class="metabox-holder columns-2">
                    <div id="post-body-content">
                        
                        <!-- Comprehensive Processing -->
                        <div class="postbox">
                            <h2 class="hndle"><span class="dashicons dashicons-admin-generic"></span> <?php _e('Comprehensive Processing', 'aiox-publisher-suite-pro'); ?></h2>
                            <div class="inside">
                                <p class="description"><?php _e('Enable comprehensive processing to automatically handle all AIOX components when content is saved.', 'aiox-publisher-suite-pro'); ?></p>
                                
                                <ul class="aiox-toggle-list" style="margin: 15px 0 0 0;">
                                    <li class="aiox-toggle-item" style="border-left: 4px solid #2271b1;">
                                        <div class="toggle-info">
                                            <span class="toggle-label"><?php _e('Enable Comprehensive Processing', 'aiox-publisher-suite-pro'); ?></span>
                                            <span class="toggle-desc"><?php _e('Process all components automatically when posts are saved', 'aiox-publisher-suite-pro'); ?></span>
                                        </div>
                                        <label class="aiox-switch">
                                            <input type="checkbox" name="aiox_enable_comprehensive_processing" value="1" <?php checked(get_option('aiox_enable_comprehensive_processing', true)); ?> />
                                            <span class="slider"></span>
                                        </label>
                                    </li>
                                </ul>
                            </div>
                        </div>
                        
                        <!-- Individual Components -->
                        <div class="postbox">
                            <h2 class="hndle"><span class="dashicons dashicons-admin-plugins"></span> <?php _e('Processing Components', 'aiox-publisher-suite-pro'); ?></h2>
                            <div class="inside">
                                <p class="description"><?php _e('Enable or disable individual processing components.', 'aiox-publisher-suite-pro'); ?></p>
                                
                                <ul class="aiox-toggle-list" style="margin: 15px 0 0 0;">
                                    <li class="aiox-toggle-item">
                                        <div class="toggle-info">
                                            <span class="toggle-label"><?php _e('Auto-Extract Ingredients', 'aiox-publisher-suite-pro'); ?></span>
                                            <span class="toggle-desc"><?php _e('Break content into semantic building blocks', 'aiox-publisher-suite-pro'); ?></span>
                                        </div>
                                        <label class="aiox-switch">
                                            <input type="checkbox" name="aiox_auto_extract_ingredients" value="1" <?php checked(get_option('aiox_auto_extract_ingredients', true)); ?> />
                                            <span class="slider"></span>
                                        </label>
                                    </li>
                                    <li class="aiox-toggle-item">
                                        <div class="toggle-info">
                                            <span class="toggle-label"><?php _e('Auto-Update Manifesto', 'aiox-publisher-suite-pro'); ?></span>
                                            <span class="toggle-desc"><?php _e('Update AI discovery JSON files automatically', 'aiox-publisher-suite-pro'); ?></span>
                                        </div>
                                        <label class="aiox-switch">
                                            <input type="checkbox" name="aiox_auto_update_manifesto" value="1" <?php checked(get_option('aiox_auto_update_manifesto', true)); ?> />
                                            <span class="slider"></span>
                                        </label>
                                    </li>
                                    <li class="aiox-toggle-item">
                                        <div class="toggle-info">
                                            <span class="toggle-label"><?php _e('Auto-Update Badges', 'aiox-publisher-suite-pro'); ?></span>
                                            <span class="toggle-desc"><?php _e('Update frontend badge Q&A data', 'aiox-publisher-suite-pro'); ?></span>
                                        </div>
                                        <label class="aiox-switch">
                                            <input type="checkbox" name="aiox_auto_update_badges" value="1" <?php checked(get_option('aiox_auto_update_badges', true)); ?> />
                                            <span class="slider"></span>
                                        </label>
                                    </li>
                                    <li class="aiox-toggle-item">
                                        <div class="toggle-info">
                                            <span class="toggle-label"><?php _e('Auto-Classify Metadata', 'aiox-publisher-suite-pro'); ?></span>
                                            <span class="toggle-desc"><?php _e('Extract topics, categories, and content analysis', 'aiox-publisher-suite-pro'); ?></span>
                                        </div>
                                        <label class="aiox-switch">
                                            <input type="checkbox" name="aiox_auto_classify_metadata" value="1" <?php checked(get_option('aiox_auto_classify_metadata', true)); ?> />
                                            <span class="slider"></span>
                                        </label>
                                    </li>
                                </ul>
                            </div>
                        </div>
                        
                        <p class="submit">
                            <input type="submit" name="save_processing_settings" class="button button-primary button-large" value="<?php esc_attr_e('Save Settings', 'aiox-publisher-suite-pro'); ?>" />
                        </p>
                        
                    </div><!-- /post-body-content -->
                    
                    <!-- Sidebar -->
                    <div id="postbox-container-1" class="postbox-container">
                        <div class="postbox">
                            <h2 class="hndle"><?php _e('About Components', 'aiox-publisher-suite-pro'); ?></h2>
                            <div class="inside">
                                <p><strong><?php _e('Capsules', 'aiox-publisher-suite-pro'); ?></strong><br>
                                <small><?php _e('Always active. AI-optimized content summaries.', 'aiox-publisher-suite-pro'); ?></small></p>
                                
                                <p><strong><?php _e('Ingredients', 'aiox-publisher-suite-pro'); ?></strong><br>
                                <small><?php _e('Semantic breakdown for better AI understanding.', 'aiox-publisher-suite-pro'); ?></small></p>
                                
                                <p><strong><?php _e('Manifesto', 'aiox-publisher-suite-pro'); ?></strong><br>
                                <small><?php _e('JSON files for AI crawler discovery.', 'aiox-publisher-suite-pro'); ?></small></p>
                                
                                <p><strong><?php _e('Badges', 'aiox-publisher-suite-pro'); ?></strong><br>
                                <small><?php _e('Frontend Q&A display data.', 'aiox-publisher-suite-pro'); ?></small></p>
                                
                                <p><strong><?php _e('Metadata', 'aiox-publisher-suite-pro'); ?></strong><br>
                                <small><?php _e('Content classification and analysis.', 'aiox-publisher-suite-pro'); ?></small></p>
                            </div>
                        </div>
                        
                        <div class="postbox">
                            <h2 class="hndle"><?php _e('Token Usage', 'aiox-publisher-suite-pro'); ?></h2>
                            <div class="inside">
                                <p class="description"><?php _e('Each component uses API tokens. More components = more token usage per post.', 'aiox-publisher-suite-pro'); ?></p>
                            </div>
                        </div>
                    </div>
                    
                </div><!-- /post-body -->
            </div><!-- /poststuff -->
        </form>
        <?php
    }
}
