<?php
/**
 * AIOX Processing Manager Tab Rendering Methods
 * 
 * Contains the tab rendering methods for the Processing Manager
 * to avoid making the main class file too large.
 */

if (!defined('ABSPATH')) {
    exit;
}

trait AIOX_Processing_Manager_Tabs {
    
    /**
     * Render enhanced overview tab
     */
    private function render_enhanced_overview_tab() {
        ?>
        <div class="aiox-overview-tab">
            <div class="aiox-stats-grid" style="display: grid; grid-template-columns: repeat(auto-fit, minmax(250px, 1fr)); gap: 20px; margin: 20px 0;">
                
                <!-- Processing Status Card -->
                <div class="aiox-stat-card" style="background: white; padding: 20px; border: 1px solid #ddd; border-radius: 8px;">
                    <h3 style="margin-top: 0;">📊 Processing Status</h3>
                    <?php
                    $status = $this->get_overall_processing_status();
                    ?>
                    <p><strong>Total Posts:</strong> <?php echo $status['total_posts']; ?></p>
                    <p><strong>Processed:</strong> <?php echo $status['processed_posts']; ?></p>
                    <p><strong>Unprocessed:</strong> <?php echo $status['unprocessed_posts']; ?></p>
                    <p><strong>Completion:</strong> <?php echo $status['processing_percentage']; ?>%</p>
                    
                    <?php if ($status['unprocessed_posts'] > 0): ?>
                    <button type="button" class="button button-primary" id="aiox-process-all-unprocessed">
                        Process All Unprocessed (<?php echo $status['unprocessed_posts']; ?>)
                    </button>
                    <?php endif; ?>
                </div>
                
                <!-- Comprehensive Processing Card -->
                <div class="aiox-stat-card" style="background: white; padding: 20px; border: 1px solid #ddd; border-radius: 8px;">
                    <h3 style="margin-top: 0;">🔄 Comprehensive Processing</h3>
                    <?php $comprehensive_enabled = get_option('aiox_enable_comprehensive_processing', true); ?>
                    <p><strong>Status:</strong> 
                        <span style="color: <?php echo $comprehensive_enabled ? 'green' : 'orange'; ?>;">
                            <?php echo $comprehensive_enabled ? 'Enabled' : 'Disabled'; ?>
                        </span>
                    </p>
                    <p><strong>Components:</strong></p>
                    <ul style="margin-left: 20px;">
                        <li>✅ Capsules (Always active)</li>
                        <li><?php echo get_option('aiox_auto_extract_ingredients', true) ? '✅' : '❌'; ?> Ingredients</li>
                        <li><?php echo get_option('aiox_auto_update_manifesto', true) ? '✅' : '❌'; ?> Manifesto Files</li>
                        <li><?php echo get_option('aiox_auto_update_badges', true) ? '✅' : '❌'; ?> Badge Data</li>
                        <li><?php echo get_option('aiox_auto_update_metadata', true) ? '✅' : '❌'; ?> Metadata</li>
                    </ul>
                    
                    <button type="button" class="button" id="aiox-toggle-comprehensive-processing" 
                            data-enabled="<?php echo $comprehensive_enabled ? '1' : '0'; ?>">
                        <?php echo $comprehensive_enabled ? 'Disable' : 'Enable'; ?> Comprehensive Processing
                    </button>
                </div>
                
                <!-- Recent Activity Card -->
                <div class="aiox-stat-card" style="background: white; padding: 20px; border: 1px solid #ddd; border-radius: 8px;">
                    <h3 style="margin-top: 0;">📈 Recent Activity</h3>
                    <?php
                    $hook_stats = get_option('aiox_enhanced_hook_stats', array());
                    $sync_stats = get_option('aiox_sync_stats', array());
                    ?>
                    <p><strong>Posts Processed:</strong> <?php echo isset($hook_stats['posts_processed']) ? $hook_stats['posts_processed'] : 0; ?></p>
                    <p><strong>Components Synced:</strong> <?php echo isset($sync_stats['components_synced']) ? $sync_stats['components_synced'] : 0; ?></p>
                    <p><strong>Last Activity:</strong> <?php echo isset($hook_stats['last_activity']) ? $hook_stats['last_activity'] : 'Never'; ?></p>
                    
                    <button type="button" class="button" id="aiox-reset-stats">
                        Reset Statistics
                    </button>
                </div>
                
            </div>
            
            <!-- Existing Capsules Overview (Preserved) -->
            <div class="aiox-capsules-overview" style="margin-top: 30px;">
                <h2>📦 Capsules Overview</h2>
                <?php
                // Call parent method to show existing capsules overview
                if (method_exists(get_parent_class($this), 'render_overview_tab')) {
                    parent::render_overview_tab();
                } else {
                    // Fallback capsules overview
                    $this->render_capsules_overview_fallback();
                }
                ?>
            </div>
        </div>
        
        <script type="text/javascript">
        jQuery(document).ready(function($) {
            var overviewNonce = '<?php echo wp_create_nonce('aiox_capsules_nonce'); ?>';
            
            console.log('AIOX Overview Tab: JS loaded');
            console.log('AIOX: Process All button found:', $('#aiox-process-all-unprocessed').length);
            
            // Handle Process All Unprocessed button in Overview tab
            $('#aiox-process-all-unprocessed').on('click', function(e) {
                e.preventDefault();
                console.log('AIOX: Process All Unprocessed clicked');
                
                var button = $(this);
                var originalText = button.text();
                
                if (!confirm('⚠️ This will process all unprocessed posts.\n\nThis may take several minutes and consume API credits.\n\nContinue?')) {
                    return;
                }
                
                button.prop('disabled', true).text('⏳ Starting...');
                
                // First, get the list of unprocessed posts
                $.ajax({
                    url: ajaxurl,
                    type: 'POST',
                    data: {
                        action: 'aiox_get_unprocessed_posts',
                        nonce: overviewNonce
                    },
                    success: function(response) {
                        console.log('Got unprocessed posts:', response);
                        if (response.success && response.data.posts && response.data.posts.length > 0) {
                            processPostsSequentially(response.data.posts, button, originalText);
                        } else {
                            button.prop('disabled', false).text('✅ All Done!');
                            alert('No unprocessed posts found!');
                        }
                    },
                    error: function(xhr, status, error) {
                        console.error('Failed to get posts:', error);
                        button.prop('disabled', false).text(originalText);
                        alert('Failed to get unprocessed posts: ' + error);
                    }
                });
            });
            
            function processPostsSequentially(posts, button, originalText) {
                var totalPosts = posts.length;
                var currentIndex = 0;
                var processedCount = 0;
                var failedCount = 0;
                
                function processNext() {
                    if (currentIndex >= totalPosts) {
                        // All done
                        var resultIcon = failedCount > 0 ? '⚠️' : '✅';
                        button.prop('disabled', false).text(resultIcon + ' Done! (' + processedCount + '/' + totalPosts + ')');
                        alert('Batch processing complete!\n\nProcessed: ' + processedCount + '\nFailed: ' + failedCount);
                        
                        // Reload page to update counts
                        setTimeout(function() {
                            location.reload();
                        }, 2000);
                        return;
                    }
                    
                    var post = posts[currentIndex];
                    button.text('⏳ Processing ' + (currentIndex + 1) + '/' + totalPosts + '...');
                    
                    $.ajax({
                        url: ajaxurl,
                        type: 'POST',
                        timeout: 60000,
                        data: {
                            action: 'aiox_process_all_components',
                            post_id: post.id,
                            nonce: overviewNonce
                        },
                        success: function(response) {
                            console.log('Processed post ' + post.id + ':', response);
                            if (response.success) {
                                processedCount++;
                            } else {
                                failedCount++;
                                console.error('Failed:', response.data);
                            }
                        },
                        error: function(xhr, status, error) {
                            console.error('Error processing post ' + post.id + ':', error);
                            failedCount++;
                        },
                        complete: function() {
                            currentIndex++;
                            // Small delay between requests
                            setTimeout(processNext, 500);
                        }
                    });
                }
                
                // Start processing
                processNext();
            }
            
            // Handle Reset Statistics button
            $('#aiox-reset-stats').on('click', function() {
                if (!confirm('Reset all processing statistics?')) return;
                
                $.post(ajaxurl, {
                    action: 'aiox_reset_stats',
                    nonce: overviewNonce
                }, function(response) {
                    if (response.success) {
                        location.reload();
                    } else {
                        alert('Failed to reset statistics');
                    }
                });
            });
            
            // Handle Toggle Comprehensive Processing button
            $('#aiox-toggle-comprehensive-processing').on('click', function() {
                var button = $(this);
                var currentlyEnabled = button.data('enabled') === 1 || button.data('enabled') === '1';
                
                $.post(ajaxurl, {
                    action: 'aiox_toggle_comprehensive_processing',
                    enable: currentlyEnabled ? 0 : 1,
                    nonce: overviewNonce
                }, function(response) {
                    if (response.success) {
                        location.reload();
                    } else {
                        alert('Failed to toggle comprehensive processing');
                    }
                });
            });
        });
        </script>
        <?php
    }
    
    /**
     * Render enhanced manage tab
     */
    public function render_enhanced_manage_tab() {
        ?>
        <div class="aiox-manage-tab">
            <h2>🔧 Manage Content</h2>
            
            <!-- Enhanced Content Management Table -->
            <div class="aiox-content-controls" style="margin-bottom: 15px;">
                <button type="button" class="button" id="aiox-refresh-content">
                    🔄 Refresh Content List
                </button>
                <label style="margin-left: 15px;">
                    <input type="checkbox" id="aiox-show-unprocessed-only" />
                    📋 Show Unprocessed Only
                </label>
            </div>
            
            <div id="aiox-content-list-container">
                
            </div>
        </div>
        <?php
    }
    
    /**
     * Render enhanced create tab
     */
    private function render_enhanced_create_tab() {
        ?>
        <div class="aiox-create-tab">
            <div class="aiox-create-options" style="display: grid; grid-template-columns: repeat(auto-fit, minmax(300px, 1fr)); gap: 20px; margin: 20px 0;">
                
                <!-- Comprehensive Processing Card -->
                <div class="aiox-create-card" style="background: white; padding: 20px; border: 1px solid #ddd; border-radius: 8px;">
                    <h3 style="margin-top: 0;">🔄 Comprehensive Processing</h3>
                    <p>Process all components for selected content:</p>
                    <ul style="margin-left: 20px;">
                        <li>📦 Capsules</li>
                        <li>🧪 Ingredients</li>
                        <li>📄 Manifesto Files</li>
                        <li>🏷️ Badge Data</li>
                        <li>📊 Metadata</li>
                    </ul>
                    <button type="button" class="button button-primary aiox-create-comprehensive" data-type="comprehensive">
                        Create Comprehensive Processing
                    </button>
                </div>
                
                <!-- Individual Components Cards -->
                <div class="aiox-create-card" style="background: white; padding: 20px; border: 1px solid #ddd; border-radius: 8px;">
                    <h3 style="margin-top: 0;">📦 Capsules Only</h3>
                    <p>Create capsules without other components:</p>
                    <div class="aiox-create-buttons">
                        <button type="button" class="button aiox-create-capsule" data-type="post">Post Capsule</button>
                        <button type="button" class="button aiox-create-capsule" data-type="collection">Collection Capsule</button>
                        <button type="button" class="button aiox-create-capsule" data-type="author">Author Capsule</button>
                        <button type="button" class="button aiox-create-capsule" data-type="topic">Topic Capsule</button>
                    </div>
                </div>
                
                <div class="aiox-create-card" style="background: white; padding: 20px; border: 1px solid #ddd; border-radius: 8px;">
                    <h3 style="margin-top: 0;">🧪 Ingredients Only</h3>
                    <p>Extract ingredients from content without creating capsules:</p>
                    <button type="button" class="button aiox-create-ingredients" data-type="ingredients">
                        Extract Ingredients
                    </button>
                </div>
                
                <div class="aiox-create-card" style="background: white; padding: 20px; border: 1px solid #ddd; border-radius: 8px;">
                    <h3 style="margin-top: 0;">📄 Manifesto Files</h3>
                    <p>Update manifesto files for selected content:</p>
                    <button type="button" class="button aiox-update-manifesto" data-type="manifesto">
                        Update Manifesto
                    </button>
                </div>
                
            </div>
            
            <!-- Content Creation Tools -->
            <div class="aiox-content-creation" style="margin-top: 30px;">
                <h3>📝 Content Creation Tools</h3>
                <div class="aiox-creation-options">
                    <div class="aiox-creation-card" style="border: 1px solid #ddd; padding: 20px; margin: 10px 0; border-radius: 5px;">
                        <h4>🤖 AI-Powered Content Generation</h4>
                        <p>Generate high-quality content using AI with automatic AIOX optimization.</p>
                        <button type="button" class="button button-primary" disabled>
                            Coming Soon - AI Content Generator
                        </button>
                    </div>
                    
                    <div class="aiox-creation-card" style="border: 1px solid #ddd; padding: 20px; margin: 10px 0; border-radius: 5px;">
                        <h4>📦 Manual Capsule Creation</h4>
                        <p>Create custom content capsules for specific posts or pages.</p>
                        <button type="button" class="button" disabled>
                            Coming Soon - Capsule Builder
                        </button>
                    </div>
                    
                    <div class="aiox-creation-card" style="border: 1px solid #ddd; padding: 20px; margin: 10px 0; border-radius: 5px;">
                        <h4>📊 Bulk Content Processing</h4>
                        <p>Process multiple posts at once with comprehensive AI optimization.</p>
                        <a href="<?php echo admin_url('admin.php?page=aiox-processing&tab=bulk'); ?>" class="button button-secondary">
                            Go to Bulk Processing
                        </a>
                    </div>
                </div>
            </div>
        </div>
        <?php
    }
    
    /**
     * Render bulk processing tab
     */
    private function render_bulk_processing_tab() {
        ?>
        <div class="aiox-bulk-tab">
            <h2>⚡ Bulk Processing</h2>
            
            <div class="aiox-bulk-options" style="display: grid; grid-template-columns: 1fr 1fr; gap: 30px; margin: 20px 0;">
                
                <!-- Bulk Processing Controls -->
                <div class="aiox-bulk-controls" style="background: white; padding: 20px; border: 1px solid #ddd; border-radius: 8px;">
                    <h3 style="margin-top: 0;">🎯 Processing Options</h3>
                    
                    <div class="aiox-component-selection" style="margin: 15px 0;">
                        <h4>Select Components to Process:</h4>
                        <label><input type="checkbox" name="process_capsules" value="1" checked disabled> 📦 Capsules (Always included)</label><br>
                        <label><input type="checkbox" name="process_ingredients" value="1"> 🧪 Ingredients</label><br>
                        <label><input type="checkbox" name="process_manifesto" value="1"> 📄 Manifesto Files</label><br>
                        <label><input type="checkbox" name="process_badges" value="1"> 🏷️ Badge Data</label><br>
                        <label><input type="checkbox" name="process_metadata" value="1"> 📊 Metadata</label><br>
                    </div>
                    
                    <div class="aiox-batch-settings" style="margin: 15px 0;">
                        <h4>Batch Settings:</h4>
                        <label>Batch Size: <input type="number" name="batch_size" value="5" min="1" max="20" style="width: 60px;"></label><br>
                        <label>Delay Between Batches: <input type="number" name="batch_delay" value="2" min="1" max="10" style="width: 60px;"> seconds</label>
                    </div>
                    
                    <div class="aiox-bulk-actions" style="margin-top: 20px;">
                        <button type="button" class="button button-primary" id="aiox-start-bulk-processing">
                            🚀 Start Bulk Processing
                        </button>
                        <button type="button" class="button" id="aiox-pause-bulk-processing" disabled>
                            ⏸️ Pause
                        </button>
                        <button type="button" class="button" id="aiox-stop-bulk-processing" disabled>
                            ⏹️ Stop
                        </button>
                    </div>
                </div>
                
                <!-- Progress and Status -->
                <div class="aiox-bulk-status" style="background: white; padding: 20px; border: 1px solid #ddd; border-radius: 8px;">
                    <h3 style="margin-top: 0;">📊 Progress Status</h3>
                    
                    <div class="aiox-progress-info" id="aiox-progress-info">
                        <p><strong>Status:</strong> <span id="aiox-bulk-status">Ready</span></p>
                        <p><strong>Progress:</strong> <span id="aiox-bulk-progress">0 / 0</span></p>
                        <p><strong>Estimated Time:</strong> <span id="aiox-bulk-eta">--</span></p>
                    </div>
                    
                    <div class="aiox-progress-bar" style="background: #f0f0f0; height: 20px; border-radius: 10px; margin: 15px 0;">
                        <div id="aiox-progress-fill" style="background: #0073aa; height: 100%; border-radius: 10px; width: 0%; transition: width 0.3s;"></div>
                    </div>
                    
                    <div class="aiox-bulk-log" style="max-height: 200px; overflow-y: auto; background: #f9f9f9; padding: 10px; border-radius: 5px; margin-top: 15px;">
                        <div id="aiox-bulk-log-content">
                            <p><em>Bulk processing log will appear here...</em></p>
                        </div>
                    </div>
                </div>
                
            </div>
            
            <!-- Unprocessed Posts List -->
            <div class="aiox-unprocessed-posts" style="margin-top: 30px;">
                <h3>📋 Unprocessed Posts</h3>
                <div id="aiox-unprocessed-posts-list">
                    <p><em>Loading unprocessed posts...</em></p>
                </div>
            </div>
        </div>
        
        <script type="text/javascript">
        jQuery(document).ready(function($) {
            var processingNonce = '<?php echo wp_create_nonce('aiox_capsules_nonce'); ?>';
            
            // Load unprocessed posts on tab load
            loadUnprocessedPosts();
            
            function loadUnprocessedPosts() {
                $.post(ajaxurl, {
                    action: 'aiox_get_unprocessed_posts',
                    nonce: processingNonce
                }, function(response) {
                    if (response.success) {
                        displayUnprocessedPosts(response.data.posts);
                    } else {
                        $('#aiox-unprocessed-posts-list').html('<p style="color: red;">Error loading unprocessed posts: ' + response.data.message + '</p>');
                    }
                });
            }
            
            function displayUnprocessedPosts(posts) {
                if (posts.length === 0) {
                    $('#aiox-unprocessed-posts-list').html('<p style="color: green;">✅ All posts are processed!</p>');
                    return;
                }
                
                var html = '<div id="aiox-tab-notification" style="display: none; padding: 15px; margin-bottom: 15px; border-radius: 8px; border-left: 4px solid;"></div>';
                html += '<table class="wp-list-table widefat fixed striped">';
                html += '<thead><tr><th>Title</th><th>Type</th><th>Date</th><th>Actions</th></tr></thead>';
                html += '<tbody>';
                
                posts.forEach(function(post) {
                    html += '<tr id="post-row-' + post.id + '">';
                    html += '<td><a href="' + post.url + '" target="_blank">' + post.title + '</a></td>';
                    html += '<td>' + post.type + '</td>';
                    html += '<td>' + post.date + '</td>';
                    html += '<td><button class="button button-small aiox-process-single" data-post-id="' + post.id + '">Process Now</button></td>';
                    html += '</tr>';
                });
                
                html += '</tbody></table>';
                $('#aiox-unprocessed-posts-list').html(html);
            }
            
            // Show notification in tab
            function showTabNotification(type, message) {
                var notification = $('#aiox-tab-notification');
                var bgColor, borderColor;
                
                if (type === 'error') {
                    bgColor = '#fef2f2';
                    borderColor = '#ef4444';
                } else if (type === 'success') {
                    bgColor = '#f0fdf4';
                    borderColor = '#22c55e';
                } else if (type === 'warning') {
                    bgColor = '#fffbeb';
                    borderColor = '#f59e0b';
                } else {
                    bgColor = '#eff6ff';
                    borderColor = '#3b82f6';
                }
                
                notification.css({
                    'background-color': bgColor,
                    'border-left-color': borderColor
                }).html(message).show();
            }
            
            // Handle process button clicks
            $(document).on('click', '.aiox-process-single', function(e) {
                e.preventDefault();
                
                var button = $(this);
                var postId = button.data('post-id');
                var originalText = button.text();
                
                button.prop('disabled', true).text('🔄 Processing...');
                $('#aiox-tab-notification').hide();
                
                $.ajax({
                    url: ajaxurl,
                    type: 'POST',
                    data: {
                        action: 'aiox_process_all_components',
                        post_id: postId,
                        nonce: processingNonce
                    },
                    success: function(response) {
                        // Refresh token balance
                        if (typeof aioxRefreshBalance === 'function') {
                            aioxRefreshBalance();
                        }
                        
                        if (response.success) {
                            button.text('✅ Done');
                            showTabNotification('success', '✅ <strong>Processing Complete!</strong> ' + (response.data.message || ''));
                            
                            // Remove row after 2 seconds and reload
                            setTimeout(function() {
                                $('#post-row-' + postId).fadeOut(function() {
                                    $(this).remove();
                                    loadUnprocessedPosts();
                                });
                            }, 2000);
                        } else {
                            button.prop('disabled', false).text(originalText);
                            
                            var errorMsg = response.data && response.data.message ? response.data.message : 'Unknown error';
                            
                            if (errorMsg.indexOf('Token') !== -1 || errorMsg.indexOf('token') !== -1 || errorMsg.indexOf('Insufficient') !== -1) {
                                showTabNotification('warning', '⚠️ <strong>Token Limit Reached</strong><br>' + errorMsg);
                            } else {
                                showTabNotification('error', '❌ <strong>Processing Failed:</strong> ' + errorMsg);
                            }
                        }
                    },
                    error: function(xhr, status, error) {
                        button.prop('disabled', false).text(originalText);
                        
                        if (typeof aioxRefreshBalance === 'function') {
                            aioxRefreshBalance();
                        }
                        
                        showTabNotification('error', '❌ <strong>Request Failed:</strong> ' + error);
                    }
                });
            });
            
            // Handle Process All Unprocessed button
            $('#aiox-process-all-unprocessed').on('click', function(e) {
                e.preventDefault();
                
                var button = $(this);
                var originalText = button.text();
                
                if (!confirm('⚠️ This will process ALL unprocessed posts.\n\nThis may take several minutes and consume API credits.\n\nContinue?')) {
                    return;
                }
                
                button.prop('disabled', true).text('🔄 Processing...');
                showTabNotification('info', '⏳ Starting batch processing...');
                
                // Get all unprocessed post rows
                var rows = $('#aiox-unprocessed-posts-list tbody tr');
                var totalPosts = rows.length;
                var processedCount = 0;
                var failedCount = 0;
                var currentIndex = 0;
                
                if (totalPosts === 0) {
                    showTabNotification('success', '✅ No unprocessed posts to process!');
                    button.prop('disabled', false).text(originalText);
                    return;
                }
                
                function processNextPost() {
                    if (currentIndex >= totalPosts) {
                        // All done
                        var resultType = failedCount > 0 ? 'warning' : 'success';
                        var resultIcon = failedCount > 0 ? '⚠️' : '✅';
                        showTabNotification(resultType, resultIcon + ' <strong>Batch Processing Complete!</strong><br>Processed: ' + processedCount + ', Failed: ' + failedCount);
                        button.prop('disabled', false).text('Process All Unprocessed (' + (totalPosts - processedCount) + ')');
                        loadUnprocessedPosts();
                        return;
                    }
                    
                    var row = $(rows[currentIndex]);
                    var postId = row.find('.aiox-process-single').data('post-id');
                    var postTitle = row.find('td:first a').text();
                    
                    showTabNotification('info', '⏳ Processing (' + (currentIndex + 1) + '/' + totalPosts + '): ' + postTitle);
                    
                    $.ajax({
                        url: ajaxurl,
                        type: 'POST',
                        timeout: 60000,
                        data: {
                            action: 'aiox_process_all_components',
                            post_id: postId,
                            nonce: processingNonce
                        },
                        success: function(response) {
                            if (response.success) {
                                processedCount++;
                                row.fadeOut();
                            } else {
                                failedCount++;
                                console.error('Failed to process post ' + postId + ':', response.data);
                            }
                        },
                        error: function() {
                            failedCount++;
                        },
                        complete: function() {
                            currentIndex++;
                            // Small delay between requests
                            setTimeout(processNextPost, 500);
                        }
                    });
                }
                
                // Start processing
                processNextPost();
            });
        });
        </script>
        <?php
    }
    
    /**
     * Render processing settings tab
     */
    private function render_processing_settings_tab() {
        ?>
        <div class="aiox-settings-tab">
            <h2>⚙️ Processing Settings</h2>
            
            <form method="post" action="" id="aiox-processing-settings-form">
                <?php wp_nonce_field('aiox_processing_settings', 'aiox_processing_settings_nonce'); ?>
                
                <div class="aiox-settings-grid" style="display: grid; grid-template-columns: 1fr; gap: 30px; margin: 20px 0;">
                    
                    <!-- Automatic Processing Settings -->
                    <div class="aiox-settings-card" style="background: white; padding: 20px; border: 1px solid #ddd; border-radius: 8px;">
                        <h3 style="margin-top: 0;">🚀 Automatic Processing</h3>
                        
                        <table class="form-table">
                            <tr>
                                <th scope="row">Auto-Process on Save</th>
                                <td>
                                    <label>
                                        <input type="checkbox" name="aiox_auto_process_on_save" value="1" 
                                               <?php checked(get_option('aiox_auto_process_on_save', false)); ?> />
                                        Automatically process posts with AI when they are created or updated
                                    </label>
                                    <p class="description">
                                        <strong>Recommended:</strong> When enabled, all published posts and pages will be automatically processed with AI analysis, ingredients extraction, and capsule generation without manual intervention.
                                    </p>
                                </td>
                            </tr>
                        </table>
                    </div>
                    
                    <!-- Comprehensive Processing Settings -->
                    <div class="aiox-settings-card" style="background: white; padding: 20px; border: 1px solid #ddd; border-radius: 8px;">
                        <h3 style="margin-top: 0;">🔄 Comprehensive Processing</h3>
                        
                        <table class="form-table">
                            <tr>
                                <th scope="row">Enable Comprehensive Processing</th>
                                <td>
                                    <label>
                                        <input type="checkbox" name="aiox_enable_comprehensive_processing" value="1" 
                                               <?php checked(get_option('aiox_enable_comprehensive_processing', true)); ?> />
                                        Process all components automatically when posts are saved
                                    </label>
                                </td>
                            </tr>
                            <tr>
                                <th scope="row">Auto-Extract Ingredients</th>
                                <td>
                                    <label>
                                        <input type="checkbox" name="aiox_auto_extract_ingredients" value="1" 
                                               <?php checked(get_option('aiox_auto_extract_ingredients', true)); ?> />
                                        Automatically extract ingredients from content
                                    </label>
                                </td>
                            </tr>
                            <tr>
                                <th scope="row">Auto-Update Manifesto</th>
                                <td>
                                    <label>
                                        <input type="checkbox" name="aiox_auto_update_manifesto" value="1" 
                                               <?php checked(get_option('aiox_auto_update_manifesto', true)); ?> />
                                        Automatically update manifesto files
                                    </label>
                                </td>
                            </tr>
                            <tr>
                                <th scope="row">Auto-Update Badges</th>
                                <td>
                                    <label>
                                        <input type="checkbox" name="aiox_auto_update_badges" value="1" 
                                               <?php checked(get_option('aiox_auto_update_badges', true)); ?> />
                                        Automatically update badge data
                                    </label>
                                </td>
                            </tr>
                            <tr>
                                <th scope="row">Auto-Classify Content Metadata</th>
                                <td>
                                    <label>
                                        <input type="checkbox" name="aiox_auto_classify_metadata" value="1" 
                                               <?php checked(get_option('aiox_auto_classify_metadata', true)); ?> />
                                        Automatically extract and classify content metadata from AI processing results
                                    </label>
                                    <p class="description">
                                        When enabled, metadata fields (Intent, Tone, Audience, Market Segments, etc.) are automatically extracted from the AI server's analysis and saved to the post. This happens during comprehensive processing.
                                    </p>
                                </td>
                            </tr>
                            <tr>
                                <th scope="row">Auto-Update Metadata</th>
                                <td>
                                    <label>
                                        <input type="checkbox" name="aiox_auto_update_metadata" value="1" 
                                               <?php checked(get_option('aiox_auto_update_metadata', true)); ?> />
                                        Automatically update metadata classification
                                    </label>
                                </td>
                            </tr>
                        </table>
                    </div>
                    
                    <!-- Processing Performance Settings -->
                    <div class="aiox-settings-card" style="background: white; padding: 20px; border: 1px solid #ddd; border-radius: 8px;">
                        <h3 style="margin-top: 0;">⚡ Performance Settings</h3>
                        
                        <table class="form-table">
                            <tr>
                                <th scope="row">Background Processing</th>
                                <td>
                                    <label>
                                        <input type="checkbox" name="aiox_enable_background_processing" value="1" 
                                               <?php checked(get_option('aiox_enable_background_processing', true)); ?> />
                                        Process content in background to avoid blocking page saves
                                    </label>
                                </td>
                            </tr>
                            <tr>
                                <th scope="row">Processing Delay</th>
                                <td>
                                    <input type="number" name="aiox_processing_delay" value="<?php echo get_option('aiox_processing_delay', 15); ?>" 
                                           min="5" max="300" style="width: 80px;" /> seconds
                                    <p class="description">Delay before starting background processing</p>
                                </td>
                            </tr>
                            <tr>
                                <th scope="row">Batch Size</th>
                                <td>
                                    <input type="number" name="aiox_batch_size" value="<?php echo get_option('aiox_batch_size', 5); ?>" 
                                           min="1" max="20" style="width: 80px;" /> posts
                                    <p class="description">Number of posts to process in each batch</p>
                                </td>
                            </tr>
                        </table>
                    </div>
                    
                    <!-- Default Licensing & Rights Settings -->
                    <div class="aiox-settings-card" style="background: white; padding: 20px; border: 1px solid #ddd; border-radius: 8px;">
                        <h3 style="margin-top: 0;">⚖️ Default Licensing & Rights</h3>
                        <p class="description" style="margin-bottom: 15px;">
                            These settings will be applied as defaults to new content when Auto-Process is enabled. Users can override these settings on individual posts.
                        </p>
                        
                        <table class="form-table">
                            <tr>
                                <th scope="row">AI Training Permission</th>
                                <td>
                                    <label>
                                        <input type="checkbox" name="aiox_tdm_training_allowed" value="1" 
                                               <?php checked(get_option('aiox_tdm_training_allowed', false)); ?> />
                                        Allow AI systems to train on content by default
                                    </label>
                                    <p class="description">
                                        When enabled, new content will allow AI training unless specifically overridden on individual posts.
                                    </p>
                                </td>
                            </tr>
                            <tr>
                                <th scope="row">Citation Requirement</th>
                                <td>
                                    <label>
                                        <input type="checkbox" name="aiox_tdm_citation_required" value="1" 
                                               <?php checked(get_option('aiox_tdm_citation_required', true)); ?> />
                                        Require citation when content is used by default
                                    </label>
                                    <p class="description">
                                        When enabled, new content will require attribution when used by AI systems or other applications.
                                    </p>
                                </td>
                            </tr>
                            <tr>
                                <th scope="row">Default Attribution String</th>
                                <td>
                                    <input type="text" name="aiox_attribution_string" 
                                           value="<?php echo esc_attr(get_option('aiox_attribution_string', get_bloginfo('name'))); ?>" 
                                           style="width: 100%; max-width: 400px;" 
                                           placeholder="<?php echo esc_attr(get_bloginfo('name')); ?>" />
                                    <p class="description">
                                        Default attribution text for new content. Leave blank to use site name.
                                    </p>
                                </td>
                            </tr>
                        </table>
                    </div>
                    
                </div>
                
                <p class="submit">
                    <input type="submit" name="save_processing_settings" class="button-primary" value="Save Processing Settings" />
                    <button type="button" class="button" id="aiox-reset-processing-settings">Reset to Defaults</button>
                </p>
            </form>
            
            <!-- Statistics and Maintenance -->
            <div class="aiox-maintenance-section" style="margin-top: 40px;">
                <h3>🔧 Maintenance</h3>
                
                <div class="aiox-maintenance-actions" style="display: grid; grid-template-columns: repeat(auto-fit, minmax(250px, 1fr)); gap: 20px;">
                    
                    <div class="aiox-maintenance-card" style="background: white; padding: 20px; border: 1px solid #ddd; border-radius: 8px;">
                        <h4>📊 Statistics</h4>
                        <button type="button" class="button" id="aiox-reset-all-stats">Reset All Statistics</button>
                        <button type="button" class="button" id="aiox-export-stats">Export Statistics</button>
                    </div>
                    
                    <div class="aiox-maintenance-card" style="background: white; padding: 20px; border: 1px solid #ddd; border-radius: 8px;">
                        <h4>🧹 Cleanup</h4>
                        <button type="button" class="button" id="aiox-cleanup-orphaned-data">Cleanup Orphaned Data</button>
                        <button type="button" class="button" id="aiox-rebuild-manifesto">Rebuild All Manifesto Files</button>
                    </div>
                    
                    <!-- Default Licensing & Rights Settings -->
                    <div class="aiox-settings-card" style="background: white; padding: 20px; border: 1px solid #ddd; border-radius: 8px;">
                        <h3 style="margin-top: 0;">⚖️ Default Licensing & Rights</h3>
                        <p class="description" style="margin-bottom: 15px;">
                            These settings will be applied as defaults to new content when Auto-Process is enabled. Users can override these settings on individual posts.
                        </p>
                        
                        <table class="form-table">
                            <tr>
                                <th scope="row">AI Training Permission</th>
                                <td>
                                    <label>
                                        <input type="checkbox" name="aiox_tdm_training_allowed" value="1" 
                                               <?php checked(get_option('aiox_tdm_training_allowed', false)); ?> />
                                        Allow AI systems to train on content by default
                                    </label>
                                    <p class="description">
                                        When enabled, new content will allow AI training unless specifically overridden on individual posts.
                                    </p>
                                </td>
                            </tr>
                            <tr>
                                <th scope="row">Citation Requirement</th>
                                <td>
                                    <label>
                                        <input type="checkbox" name="aiox_tdm_citation_required" value="1" 
                                               <?php checked(get_option('aiox_tdm_citation_required', true)); ?> />
                                        Require citation when content is used by default
                                    </label>
                                    <p class="description">
                                        When enabled, new content will require attribution when used by AI systems or other applications.
                                    </p>
                                </td>
                            </tr>
                            <tr>
                                <th scope="row">Default Attribution String</th>
                                <td>
                                    <input type="text" name="aiox_attribution_string" 
                                           value="<?php echo esc_attr(get_option('aiox_attribution_string', get_bloginfo('name'))); ?>" 
                                           style="width: 100%; max-width: 400px;" 
                                           placeholder="<?php echo esc_attr(get_bloginfo('name')); ?>" />
                                    <p class="description">
                                        Default attribution text for new content. Leave blank to use site name.
                                    </p>
                                </td>
                            </tr>
                        </table>
                    </div>
                    
                </div>
            </div>
        </div>
        
        <?php
        // Handle form submission
        if (isset($_POST['save_processing_settings']) && wp_verify_nonce($_POST['aiox_processing_settings_nonce'], 'aiox_processing_settings')) {
            $this->save_processing_settings();
        }
    }
    
    /**
     * Render enhanced content table
     */
    public function render_enhanced_content_table() {
        // Get posts with processing status
        $posts = get_posts(array(
            'post_type' => array('post', 'page'),
            'post_status' => 'publish',
            'numberposts' => 50,
            'orderby' => 'date',
            'order' => 'DESC'
        ));
        
        ?>
        <table class="wp-list-table widefat fixed striped aiox-content-table">
            <thead>
                <tr>
                    <td class="manage-column column-cb check-column">
                        <input type="checkbox" id="aiox-select-all" />
                    </td>
                    <th class="manage-column">Title</th>
                    <th class="manage-column">Type</th>
                    <th class="manage-column">📦 Capsule</th>
                    <th class="manage-column">🧪 Ingredients</th>
                    <th class="manage-column">📄 Manifesto</th>
                    <th class="manage-column">🏷️ Badge</th>
                    <th class="manage-column">Actions</th>
                </tr>
            </thead>
            <tbody>
                <?php foreach ($posts as $post): ?>
                    <?php
                    $status = $this->get_post_processing_status($post->ID);
                    ?>
                    <tr>
                        <th class="check-column">
                            <input type="checkbox" name="post_ids[]" value="<?php echo $post->ID; ?>" />
                        </th>
                        <td>
                            <strong><a href="<?php echo get_permalink($post->ID); ?>" target="_blank"><?php echo esc_html($post->post_title); ?></a></strong>
                            <div class="row-actions">
                                <span class="edit"><a href="<?php echo get_edit_post_link($post->ID); ?>">Edit</a></span>
                            </div>
                        </td>
                        <td><?php echo $post->post_type; ?></td>
                        <td>
                            <?php 
                            if ($status['has_capsule']) {
                                $capsule_id = get_post_meta($post->ID, '_aiox_capsule_id', true);
                                if ($capsule_id) {
                                    $capsule_url = home_url('/.well-known/capsules/' . $capsule_id . '.json');
                                    echo '<a href="' . esc_url($capsule_url) . '" target="_blank" title="View capsule file" style="text-decoration: none; font-size: 18px;">📦</a>';
                                } else {
                                    echo '✅';
                                }
                            } else {
                                echo '❌';
                            }
                            ?>
                        </td>
                        <td>
                            <?php 
                            // Clickable ingredients icon
                            if ($status['has_ingredients']) {
                                $post_file_url = home_url('/.well-known/posts/' . $post->ID . '.json');
                                $ingredients_count = $status['ingredients_count'];
                                echo '<a href="' . esc_url($post_file_url) . '" target="_blank" title="View data file (' . $ingredients_count . ' ingredients)" style="text-decoration: none; font-size: 18px;">🧩</a>';
                            } else {
                                echo '❌';
                            }
                            ?>
                        </td>
                        <td>
                            <?php 
                            // Clickable manifesto icon
                            if ($status['has_processed_data']) {
                                $post_file_url = home_url('/.well-known/posts/' . $post->ID . '.json');
                                echo '<a href="' . esc_url($post_file_url) . '" target="_blank" title="View post data file" style="text-decoration: none; font-size: 18px;">📄</a>';
                            } else {
                                echo '❌';
                            }
                            ?>
                        </td>
                        <td><?php echo $status['has_processed_data'] ? '✅' : '❌'; ?></td>
                        <td>
                            <button type="button" class="button button-small aiox-process-all-components" data-post-id="<?php echo $post->ID; ?>">
                                Process All
                            </button>
                            <button type="button" class="button button-small aiox-clear-ai-data" data-post-id="<?php echo $post->ID; ?>">
                                Clear AI Data
                            </button>
                        </td>
                    </tr>
                <?php endforeach; ?>
            </tbody>
        </table>
        <?php
    }
    
    /**
     * Fallback methods for when parent methods don't exist
     */
    private function render_capsules_overview_fallback() {
        echo '<p><em>Capsules overview will be displayed here. This is a fallback when the parent method is not available.</em></p>';
    }
    
    private function render_capsules_management_fallback() {
        echo '<p><em>Capsules management interface will be displayed here. This is a fallback when the parent method is not available.</em></p>';
    }
    
    private function render_capsules_creation_fallback() {
        echo '<p><em>Capsules creation interface will be displayed here. This is a fallback when the parent method is not available.</em></p>';
    }
    
    /**
     * Save processing settings
     */
    private function save_processing_settings() {
        $settings = array(
            'aiox_auto_process_on_save',
            'aiox_enable_comprehensive_processing',
            'aiox_auto_extract_ingredients',
            'aiox_auto_classify_metadata',
            'aiox_auto_update_manifesto',
            'aiox_auto_update_badges',
            'aiox_auto_update_metadata',
            'aiox_enable_background_processing',
            'aiox_processing_delay',
            'aiox_batch_size',
            'aiox_tdm_training_allowed',
            'aiox_tdm_citation_required'
        );
        
        // Handle text settings separately
        $text_settings = array(
            'aiox_attribution_string'
        );
        
        foreach ($settings as $setting) {
            if (isset($_POST[$setting])) {
                if (in_array($setting, array('aiox_processing_delay', 'aiox_batch_size'))) {
                    update_option($setting, intval($_POST[$setting]));
                } else {
                    update_option($setting, $_POST[$setting] === '1');
                }
            } else {
                // Checkbox not checked
                if (!in_array($setting, array('aiox_processing_delay', 'aiox_batch_size'))) {
                    update_option($setting, false);
                }
            }
        }
        
        // Handle text settings
        foreach ($text_settings as $setting) {
            if (isset($_POST[$setting])) {
                update_option($setting, sanitize_text_field($_POST[$setting]));
            }
        }
        
        echo '<div class="notice notice-success"><p>Processing settings saved successfully!</p></div>';
    }
}