<?php
/**
 * AIOX Processing Manager
 * 
 * Extends the existing AIOX_Capsules_Hybrid class to add comprehensive processing
 * capabilities while preserving all existing capsule functionality.
 * 
 * Phase 1: Foundation setup - adds new AJAX handlers and processing methods
 * without modifying any existing functionality.
 * 
 * STYLED VERSION: Modern dark theme applied
 */

if (!defined('ABSPATH')) {
    exit;
}

// Ensure the parent class is loaded
require_once AIOX_PLUGIN_DIR . 'admin/class-aiox-capsules-hybrid.php';
require_once AIOX_PLUGIN_DIR . 'core/class-aiox-comprehensive-processor.php';

// Include the helper class
require_once AIOX_PLUGIN_DIR . 'admin/class-aiox-processing-manager-helper.php';

class AIOX_Processing_Manager extends AIOX_Capsules_Hybrid {
    
    /**
     * Comprehensive processor instance
     */
    private $comprehensive_processor;
    
    /**
     * Processing statistics
     */
    private $processing_stats;
    
    /**
     * Constructor
     */
    public function __construct() {
        // CRITICAL: Call parent constructor first to preserve all existing functionality
        parent::__construct();
        
        AIOX_Logger::info('AIOX Processing Manager: Initializing Phase 1 foundation');
        
        try {
            // Initialize comprehensive processor
            $this->comprehensive_processor = new AIOX_Comprehensive_Processor();
            
            // Initialize processing stats
            $this->processing_stats = array(
                'total_posts' => 0,
                'processed_posts' => 0,
                'unprocessed_posts' => 0,
                'last_updated' => current_time('mysql')
            );
            
            // Register new AJAX handlers (additive - don't remove existing ones)
            $this->register_comprehensive_handlers();
            
            AIOX_Logger::info('AIOX Processing Manager: Phase 1 foundation initialized successfully');
            
        } catch (Exception $e) {
            AIOX_Logger::error('AIOX Processing Manager: Failed to initialize Phase 1 foundation', array(
                'error' => $e->getMessage(),
                'trace' => $e->getTraceAsString()
            ));
        }
    }
    
    /**
     * Register comprehensive processing AJAX handlers
     * These are NEW handlers that extend functionality without replacing existing ones
     */
    private function register_comprehensive_handlers() {
        if (!function_exists('add_action')) {
            return;
        }
        
        // New comprehensive processing handlers
        $new_handlers = array(
            'aiox_process_all_components' => 'ajax_process_all_components',
            'aiox_extract_ingredients_only' => 'ajax_extract_ingredients_only',
            'aiox_update_manifesto_only' => 'ajax_update_manifesto_only',
            'aiox_update_badge_data_only' => 'ajax_update_badge_data_only',
            'aiox_clear_all_ai_data' => 'ajax_clear_all_ai_data',
            'aiox_bulk_process_unprocessed' => 'ajax_bulk_process_unprocessed',
            'aiox_get_processing_status' => 'ajax_get_processing_status',
            'aiox_get_unprocessed_posts' => 'ajax_get_unprocessed_posts',
            'aiox_get_content_list' => 'ajax_get_content_list',
            'aiox_clear_post_data' => 'ajax_clear_post_data',
            'aiox_save_auto_settings' => 'ajax_save_auto_settings',
            'aiox_toggle_comprehensive_processing' => 'ajax_toggle_comprehensive_processing'
        );
        
        foreach ($new_handlers as $action => $method) {
            if (method_exists($this, $method)) {
                add_action('wp_ajax_' . $action, array($this, $method));
                AIOX_Logger::debug('AIOX Processing Manager: Registered new AJAX handler: ' . $action);
            } else {
                AIOX_Logger::warning('AIOX Processing Manager: Method does not exist: ' . $method);
            }
        }
        
        // Register balance refresh handler
        add_action('wp_ajax_aiox_refresh_balance', array($this, 'ajax_refresh_balance'));
    }
    
    /**
     * Get token balance from AIOX server
     */
    private function get_server_token_balance() {
        $api_key = get_option('aiox_custom_api_key', '');
        $endpoint = get_option('aiox_custom_endpoint', '');
        
        if (empty($api_key) || empty($endpoint)) {
            return array(
                'success' => false,
                'error' => 'API not configured'
            );
        }
        
        // Build balance endpoint URL
        $balance_url = rtrim($endpoint, '/');
        // Remove /ai/process if present and add /balance
        $balance_url = preg_replace('/\/wp-json\/aiox-server\/v1\/ai\/process$/', '', $balance_url);
        $balance_url = preg_replace('/\/wp-json\/aiox-server\/v1$/', '', $balance_url);
        $balance_url .= '/wp-json/aiox-server/v1/balance';
        
        // Check cache first (1 minute cache)
        $cache_key = 'aiox_balance_' . md5($api_key);
        $cached = get_transient($cache_key);
        if ($cached !== false) {
            return $cached;
        }
        
        $response = wp_remote_get($balance_url, array(
            'headers' => array(
                'Authorization' => 'Bearer ' . $api_key,
                'X-API-Key' => $api_key,
                'Content-Type' => 'application/json'
            ),
            'timeout' => 10
        ));
        
        if (is_wp_error($response)) {
            return array(
                'success' => false,
                'error' => 'Connection failed'
            );
        }
        
        $body = json_decode(wp_remote_retrieve_body($response), true);
        
        if (!$body || !isset($body['success']) || !$body['success']) {
            return array(
                'success' => false,
                'error' => $body['error'] ?? 'Invalid response'
            );
        }
        
        $result = array(
            'success' => true,
            'total_available' => $body['balance']['total_available'] ?? 0,
            'purchased_tokens' => $body['balance']['purchased_tokens'] ?? 0,
            'pool_remaining' => $body['balance']['pool_remaining'] ?? 0,
            'pool_limit' => $body['balance']['pool_limit'] ?? 0,
            'tokens_used' => $body['balance']['tokens_used'] ?? 0,
            'usage_type' => $body['balance']['usage_type'] ?? 'lifetime',
            'plan_name' => $body['plan']['name'] ?? 'Base Plan',
            'plan_type' => $body['plan']['type'] ?? 'base'
        );
        
        // Cache for 1 minute
        set_transient($cache_key, $result, 60);
        
        return $result;
    }
    
    /**
     * AJAX: Refresh token balance
     */
    public function ajax_refresh_balance() {
        if (!current_user_can('manage_options')) {
            wp_send_json_error(array('message' => 'Insufficient permissions'));
            return;
        }
        
        // Clear cache
        $api_key = get_option('aiox_custom_api_key', '');
        $cache_key = 'aiox_balance_' . md5($api_key);
        delete_transient($cache_key);
        
        // Get fresh balance
        $balance = $this->get_server_token_balance();
        
        wp_send_json_success($balance);
    }
    
    /**
     * AJAX: Process all components for a single post
     * This is the main comprehensive processing endpoint
     */
    public function ajax_process_all_components() {
        // Check user permissions first
        if (!current_user_can('manage_options')) {
            wp_send_json_error(array('message' => __('Insufficient permissions', 'aiox-publisher-suite-pro')));
            return;
        }
        
        // Simple nonce check - try multiple possible nonce values
        $nonce_valid = false;
        $nonce = $_POST['nonce'] ?? $_REQUEST['nonce'] ?? '';
        
        if ($nonce) {
            $nonce_valid = wp_verify_nonce($nonce, 'aiox_ajax_nonce') || 
                          wp_verify_nonce($nonce, 'aiox_capsules_nonce') ||
                          wp_verify_nonce($nonce, 'aiox_nonce');
        }
        
        if (!$nonce_valid) {
            wp_send_json_error(array('message' => 'Security check failed for reprocess - nonce: ' . substr($nonce, 0, 10) . '...'));
            return;
        }
        
        try {
            $post_id = intval(isset($_POST['post_id']) ? $_POST['post_id'] : 0);
            
            if (!$post_id) {
                throw new Exception(__('Invalid post ID', 'aiox-publisher-suite-pro'));
            }
            
            $post = get_post($post_id);
            if (!$post) {
                throw new Exception(__('Post not found', 'aiox-publisher-suite-pro'));
            }
            
            AIOX_Logger::info("Processing Manager: Starting comprehensive processing for post {$post_id}");
            
            // Force comprehensive processing for Phase 1 (always enabled)
            $comprehensive_enabled = true; // Force enable for debugging
            update_option('aiox_enable_comprehensive_processing', true); // Ensure it's saved
            
            AIOX_Logger::info("Processing Manager: FORCING comprehensive processing for post {$post_id}");
            
            if ($comprehensive_enabled) {
                // Use the comprehensive processor to handle all components
                AIOX_Logger::info("Processing Manager: Calling comprehensive processor for post {$post_id}");
                $result = $this->comprehensive_processor->process_post_completely($post_id);
                AIOX_Logger::info("Processing Manager: Comprehensive processor returned: " . json_encode($result));
            } else {
                // Fall back to capsule-only generation for backward compatibility
                if (class_exists('AIOX_Capsule_Manager')) {
                    $manager = new AIOX_Capsule_Manager();
                    $capsule = $manager->regenerate_capsule_from_post($post_id);
                    
                    if ($capsule) {
                        $result = array(
                            'success' => true,
                            'message' => 'Capsule generated successfully',
                            'capsule_id' => $capsule->get_id()
                        );
                    } else {
                        throw new Exception('Failed to generate capsule');
                    }
                } else {
                    throw new Exception('Capsule manager not available');
                }
            }
            
            if ($result && isset($result['success']) && $result['success']) {
                AIOX_Logger::info("Processing Manager: Successfully processed all components for post {$post_id}");
                
                wp_send_json_success(array(
                    'message' => __('All components processed successfully', 'aiox-publisher-suite-pro'),
                    'post_id' => $post_id,
                    'post_title' => $post->post_title,
                    'components' => array(
                        'capsule' => isset($result['capsule_id']) ? $result['capsule_id'] : null,
                        'ingredients' => isset($result['ingredients_count']) ? $result['ingredients_count'] : 0,
                        'metadata' => isset($result['metadata_fields']) ? $result['metadata_fields'] : 0,
                        'manifesto' => true,
                        'badge' => true
                    ),
                    'processing_time' => current_time('mysql')
                ));
            } else {
                throw new Exception(isset($result['message']) ? $result['message'] : __('Processing failed', 'aiox-publisher-suite-pro'));
            }
            
        } catch (Exception $e) {
            AIOX_Logger::error("Processing Manager: Failed to process all components", array(
                'post_id' => isset($post_id) ? $post_id : 0,
                'error' => $e->getMessage()
            ));
            
            wp_send_json_error(array(
                'message' => sprintf(__('Processing failed: %s', 'aiox-publisher-suite-pro'), $e->getMessage())
            ));
        }
    }
    
    /**
     * AJAX: Extract ingredients only for a post
     */
    public function ajax_extract_ingredients_only() {
        check_ajax_referer('aiox_capsules_nonce', 'nonce');
        
        if (!current_user_can('manage_options')) {
            wp_send_json_error(array('message' => __('Insufficient permissions', 'aiox-publisher-suite-pro')));
            return;
        }
        
        try {
            $post_id = intval(isset($_POST['post_id']) ? $_POST['post_id'] : 0);
            $post = get_post($post_id);
            
            if (!$post) {
                throw new Exception(__('Post not found', 'aiox-publisher-suite-pro'));
            }
            
            // Extract ingredients
            $ingredients = AIOX_Ingredients::extract_ingredients($post);
            AIOX_Ingredients::save_post_ingredients($post_id, $ingredients);
            
            wp_send_json_success(array(
                'message' => sprintf(__('Extracted %d ingredients', 'aiox-publisher-suite-pro'), count($ingredients)),
                'ingredients_count' => count($ingredients)
            ));
            
        } catch (Exception $e) {
            wp_send_json_error(array(
                'message' => sprintf(__('Ingredient extraction failed: %s', 'aiox-publisher-suite-pro'), $e->getMessage())
            ));
        }
    }
    
    /**
     * AJAX: Update manifesto files only
     */
    public function ajax_update_manifesto_only() {
        check_ajax_referer('aiox_capsules_nonce', 'nonce');
        
        if (!current_user_can('manage_options')) {
            wp_send_json_error(array('message' => __('Insufficient permissions', 'aiox-publisher-suite-pro')));
            return;
        }
        
        try {
            $post_id = intval(isset($_POST['post_id']) ? $_POST['post_id'] : 0);
            
            if ($post_id) {
                // Update manifesto for specific post
                $processed_data = get_post_meta($post_id, '_aiox_processed_data', true);
                if ($processed_data) {
                    // Regenerate all manifest files using AIOX_Hooks
                    if (class_exists('AIOX_Hooks')) {
                        $hooks = new AIOX_Hooks();
                        $hooks->generate_manifest();
                        $hooks->generate_delta_feed();
                        $hooks->generate_pack_file();
                    }
                    if (class_exists('AIOX_Ingredients')) {
                        AIOX_Ingredients::export_ingredients_file();
                    }
                    $message = __('Manifesto files updated for post', 'aiox-publisher-suite-pro');
                } else {
                    throw new Exception(__('No processed data found for post', 'aiox-publisher-suite-pro'));
                }
            } else {
                // Update all manifesto files
                $this->update_all_manifesto_files();
                $message = __('All manifesto files updated', 'aiox-publisher-suite-pro');
            }
            
            wp_send_json_success(array('message' => $message));
            
        } catch (Exception $e) {
            wp_send_json_error(array(
                'message' => sprintf(__('Manifesto update failed: %s', 'aiox-publisher-suite-pro'), $e->getMessage())
            ));
        }
    }
    
    /**
     * AJAX: Update badge data only
     */
    public function ajax_update_badge_data_only() {
        check_ajax_referer('aiox_capsules_nonce', 'nonce');
        
        if (!current_user_can('manage_options')) {
            wp_send_json_error(array('message' => __('Insufficient permissions', 'aiox-publisher-suite-pro')));
            return;
        }
        
        try {
            $post_id = intval(isset($_POST['post_id']) ? $_POST['post_id'] : 0);
            $post = get_post($post_id);
            
            if (!$post) {
                throw new Exception(__('Post not found', 'aiox-publisher-suite-pro'));
            }
            
            // Update badge data with current processed information
            $processed_data = get_post_meta($post_id, '_aiox_processed_data', true);
            if ($processed_data) {
                update_post_meta($post_id, '_aiox_badge_updated', current_time('mysql'));
                $message = __('Badge data updated', 'aiox-publisher-suite-pro');
            } else {
                $message = __('No processed data available for badge', 'aiox-publisher-suite-pro');
            }
            
            wp_send_json_success(array('message' => $message));
            
        } catch (Exception $e) {
            wp_send_json_error(array(
                'message' => sprintf(__('Badge update failed: %s', 'aiox-publisher-suite-pro'), $e->getMessage())
            ));
        }
    }
    
    /**
 * AJAX: Clear all AI data for selected posts - IMPROVED with better error handling
 */
/**
 * AJAX: Clear all AI data for selected posts - IMPROVED with better error handling
 */
public function ajax_clear_all_ai_data() {
    // More flexible permission check
    if (!current_user_can('edit_posts')) {
        wp_send_json_error(array('message' => __('Insufficient permissions', 'aiox-publisher-suite-pro')));
        return;
    }
    
    // Flexible nonce check - try multiple possible nonce values
    $nonce_valid = false;
    $nonce = $_POST['nonce'] ?? $_REQUEST['nonce'] ?? '';
    
    if ($nonce) {
        $nonce_valid = wp_verify_nonce($nonce, 'aiox_ajax_nonce') || 
                      wp_verify_nonce($nonce, 'aiox_capsules_nonce') ||
                      wp_verify_nonce($nonce, 'aiox_nonce');
    }
    
    if (!$nonce_valid) {
        wp_send_json_error(array('message' => 'Security check failed for clear data'));
        return;
    }
    
    try {
        $post_ids = array();
        
        if (isset($_POST['post_id'])) {
            // Single post
            $post_ids = array(intval($_POST['post_id']));
        } elseif (isset($_POST['post_ids']) && is_array($_POST['post_ids'])) {
            // Multiple posts
            $post_ids = array_map('intval', $_POST['post_ids']);
        } else {
            throw new Exception(__('No posts specified', 'aiox-publisher-suite-pro'));
        }
        
        $cleared_count = 0;
        $components_cleared = array();
        $errors = array();
        
        foreach ($post_ids as $post_id) {
            if ($post_id > 0) {
                AIOX_Logger::info("Clear AI Data: Starting deletion for post {$post_id}");
                
                // Call the comprehensive processor to delete everything
                $result = $this->comprehensive_processor->delete_post_components($post_id);
                
                AIOX_Logger::info("Clear AI Data: Delete result for post {$post_id}: " . print_r($result, true));
                
                if ($result && isset($result['success']) && $result['success']) {
                    $cleared_count++;
                    if (isset($result['components_cleared'])) {
                        $components_cleared = $result['components_cleared'];
                    }
                    
                    // CRITICAL: Double-check that processing flags are removed
                    delete_post_meta($post_id, '_aiox_processing_complete');
                    delete_post_meta($post_id, '_aiox_comprehensive_processed');
                    
                    AIOX_Logger::info("Clear AI Data: Successfully cleared all data for post {$post_id}");
                } else {
                    $error_msg = isset($result['message']) ? $result['message'] : 'Unknown error';
                    $errors[] = "Post {$post_id}: {$error_msg}";
                    AIOX_Logger::error("Clear AI Data: Failed for post {$post_id}: {$error_msg}");
                }
            }
        }
        
        if ($cleared_count > 0) {
            wp_send_json_success(array(
                'message' => sprintf(__('AI data cleared for %d posts', 'aiox-publisher-suite-pro'), $cleared_count),
                'cleared_count' => $cleared_count,
                'components_cleared' => $components_cleared,
                'errors' => $errors
            ));
        } else {
            wp_send_json_error(array(
                'message' => 'No posts were cleared. Errors: ' . implode(', ', $errors)
            ));
        }
        
    } catch (Exception $e) {
        AIOX_Logger::error("Clear AI Data: Exception occurred: " . $e->getMessage());
        wp_send_json_error(array(
            'message' => sprintf(__('Clear AI data failed: %s', 'aiox-publisher-suite-pro'), $e->getMessage())
        ));
    }
}
    
    /**
     * AJAX: Get processing status for posts
     */
    public function ajax_get_processing_status() {
        // Check user permissions first
        if (!current_user_can('manage_options')) {
            wp_send_json_error(array('message' => __('Insufficient permissions', 'aiox-publisher-suite-pro')));
            return;
        }
        
        // Simple nonce check - try multiple possible nonce values
        $nonce_valid = false;
        $nonce = $_POST['nonce'] ?? $_REQUEST['nonce'] ?? '';
        
        if ($nonce) {
            $nonce_valid = wp_verify_nonce($nonce, 'aiox_ajax_nonce') || 
                          wp_verify_nonce($nonce, 'aiox_capsules_nonce') ||
                          wp_verify_nonce($nonce, 'aiox_nonce');
        }
        
        if (!$nonce_valid) {
            wp_send_json_error(array('message' => 'Security check failed for get status - nonce: ' . substr($nonce, 0, 10) . '...'));
            return;
        }
        
        try {
            $post_id = intval(isset($_POST['post_id']) ? $_POST['post_id'] : 0);
            
            if ($post_id) {
                // Get status for specific post
                $status = $this->get_post_processing_status($post_id);
                
                // Add post URLs
                $post = get_post($post_id);
                if ($post) {
                    $post_data = array(
                        'id' => $post_id,
                        'title' => $post->post_title,
                        'type' => $post->post_type,
                        'url' => get_permalink($post_id),
                        'edit_url' => get_edit_post_link($post_id, 'raw')
                    );
                    
                    wp_send_json_success(array(
                        'post' => $post_data,
                        'status' => $status
                    ));
                } else {
                    wp_send_json_error(array('message' => __('Post not found', 'aiox-publisher-suite-pro')));
                }
            } else {
                // Get overall status
                $status = $this->get_overall_processing_status();
                wp_send_json_success($status);
            }
            
        } catch (Exception $e) {
            wp_send_json_error(array(
                'message' => sprintf(__('Failed to get processing status: %s', 'aiox-publisher-suite-pro'), $e->getMessage())
            ));
        }
    }
    
    /**
     * AJAX: Get list of unprocessed posts
     */
    public function ajax_get_unprocessed_posts() {
        check_ajax_referer('aiox_capsules_nonce', 'nonce');
        
        try {
            $unprocessed_posts = $this->get_unprocessed_posts();
            
            wp_send_json_success(array(
                'posts' => $unprocessed_posts,
                'count' => count($unprocessed_posts)
            ));
            
        } catch (Exception $e) {
            wp_send_json_error(array(
                'message' => sprintf(__('Failed to get unprocessed posts: %s', 'aiox-publisher-suite-pro'), $e->getMessage())
            ));
        }
    }
    
    /**
     * AJAX: Get all content list with processing status
     */
    public function ajax_get_content_list() {
        // Check user permissions first
        if (!current_user_can('manage_options')) {
            wp_send_json_error(array('message' => 'Insufficient permissions'));
            return;
        }
        
        // Simple nonce check - try multiple possible nonce values
        $nonce_valid = false;
        $nonce = $_POST['nonce'] ?? $_REQUEST['nonce'] ?? '';
        
        if ($nonce) {
            $nonce_valid = wp_verify_nonce($nonce, 'aiox_ajax_nonce') || 
                          wp_verify_nonce($nonce, 'aiox_capsules_nonce') ||
                          wp_verify_nonce($nonce, 'aiox_nonce');
        }
        
        if (!$nonce_valid) {
            wp_send_json_error(array('message' => 'Security check failed - nonce: ' . substr($nonce, 0, 10) . '...'));
            return;
        }
        
        try {
            $show_unprocessed_only = isset($_POST['show_unprocessed_only']) && $_POST['show_unprocessed_only'] === 'true';
            
            if ($show_unprocessed_only) {
                $posts = $this->get_unprocessed_posts();
                $content_list = array();
                
                foreach ($posts as $post_data) {
                    // Get actual ingredients count even for unprocessed posts
                    $status = $this->get_post_processing_status($post_data['id']);
                    
                    $content_list[] = array(
                        'id' => $post_data['id'],
                        'title' => $post_data['title'],
                        'type' => $post_data['type'],
                        'date' => $post_data['date'],
                        'url' => $post_data['url'],
                        'status' => 'unprocessed',
                        'ingredients_count' => $status['ingredients_count'],
                        'has_capsule' => false,
                        'capsule_id' => '',
                        'has_ingredients' => false,
                        'has_processed_data' => false,
                        'has_manifesto' => false,
                        'has_badge' => false
                    );
                }
            } else {
                $content_list = $this->get_all_content_with_status();
            }
            
            wp_send_json_success(array(
                'content' => $content_list,
                'count' => count($content_list),
                'show_unprocessed_only' => $show_unprocessed_only
            ));
            
        } catch (Exception $e) {
            wp_send_json_error(array(
                'message' => sprintf(__('Failed to get content list: %s', 'aiox-publisher-suite-pro'), $e->getMessage())
            ));
        }
    }
    
    /**
 * AJAX: Clear all AI data for a specific post - FIXED to use comprehensive processor
 */
public function ajax_clear_post_data() {
    // Check user permissions first
    if (!current_user_can('edit_posts')) {
        wp_send_json_error(array('message' => __('Insufficient permissions', 'aiox-publisher-suite-pro')));
        return;
    }
    
    // Simple nonce check - try multiple possible nonce values
    $nonce_valid = false;
    $nonce = $_POST['nonce'] ?? $_REQUEST['nonce'] ?? '';
    
    if ($nonce) {
        $nonce_valid = wp_verify_nonce($nonce, 'aiox_ajax_nonce') || 
                      wp_verify_nonce($nonce, 'aiox_capsules_nonce') ||
                      wp_verify_nonce($nonce, 'aiox_nonce');
    }
    
    if (!$nonce_valid) {
        wp_send_json_error(array('message' => 'Security check failed for clear data - nonce: ' . substr($nonce, 0, 10) . '...'));
        return;
    }
    
    try {
        $post_id = intval($_POST['post_id'] ?? 0);
        
        if (!$post_id) {
            throw new Exception(__('Invalid post ID', 'aiox-publisher-suite-pro'));
        }
        
        AIOX_Logger::info("Clear Post Data: Starting deletion for post {$post_id}");
        
        // Use the comprehensive processor to delete everything properly
        $result = $this->comprehensive_processor->delete_post_components($post_id);
        
        AIOX_Logger::info("Clear Post Data: Delete result for post {$post_id}: " . print_r($result, true));
        
        if ($result && isset($result['success']) && $result['success']) {
            // CRITICAL: Double-check that processing flags are removed
            delete_post_meta($post_id, '_aiox_processing_complete');
            delete_post_meta($post_id, '_aiox_comprehensive_processed');
            
            AIOX_Logger::info("Clear Post Data: Successfully cleared all data for post {$post_id}");
            
            wp_send_json_success(array(
                'message' => sprintf(__('All AI data cleared for post %d', 'aiox-publisher-suite-pro'), $post_id),
                'post_id' => $post_id,
                'components_cleared' => isset($result['components_cleared']) ? $result['components_cleared'] : array()
            ));
        } else {
            $error_msg = isset($result['message']) ? $result['message'] : 'Unknown error';
            throw new Exception($error_msg);
        }
        
    } catch (Exception $e) {
        AIOX_Logger::error("Clear Post Data: Failed for post {$post_id}: " . $e->getMessage());
        wp_send_json_error(array(
            'message' => sprintf(__('Failed to clear AI data: %s', 'aiox-publisher-suite-pro'), $e->getMessage())
        ));
    }
}
    
    /**
     * AJAX: Bulk process unprocessed posts
     */
    public function ajax_bulk_process_unprocessed() {
        check_ajax_referer('aiox_capsules_nonce', 'nonce');
        
        if (!current_user_can('manage_options')) {
            wp_send_json_error(array('message' => __('Insufficient permissions', 'aiox-publisher-suite-pro')));
            return;
        }
        
        try {
            $batch_size = intval(isset($_POST['batch_size']) ? $_POST['batch_size'] : 5);
            $offset = intval(isset($_POST['offset']) ? $_POST['offset'] : 0);
            
            $unprocessed_posts = $this->get_unprocessed_posts();
            $batch = array_slice($unprocessed_posts, $offset, $batch_size);
            
            $results = array();
            foreach ($batch as $post) {
                $result = $this->comprehensive_processor->process_post_completely($post['id']);
                $results[] = array(
                    'post_id' => $post['id'],
                    'title' => $post['title'],
                    'success' => isset($result['success']) ? $result['success'] : false,
                    'message' => isset($result['message']) ? $result['message'] : ''
                );
            }
            
            wp_send_json_success(array(
                'processed' => count($batch),
                'remaining' => count($unprocessed_posts) - ($offset + count($batch)),
                'results' => $results,
                'next_offset' => $offset + count($batch)
            ));
            
        } catch (Exception $e) {
            wp_send_json_error(array(
                'message' => sprintf(__('Bulk processing failed: %s', 'aiox-publisher-suite-pro'), $e->getMessage())
            ));
        }
    }
    
    /**
     * AJAX: Toggle comprehensive processing setting
     */
    public function ajax_toggle_comprehensive_processing() {
        check_ajax_referer('aiox_capsules_nonce', 'nonce');
        
        if (!current_user_can('manage_options')) {
            wp_send_json_error(array('message' => __('Insufficient permissions', 'aiox-publisher-suite-pro')));
            return;
        }
        
        try {
            $enabled = filter_var(isset($_POST['enabled']) ? $_POST['enabled'] : false, FILTER_VALIDATE_BOOLEAN);
            update_option('aiox_enable_comprehensive_processing', $enabled);
            
            wp_send_json_success(array(
                'message' => $enabled ? 
                    __('Comprehensive processing enabled', 'aiox-publisher-suite-pro') : 
                    __('Comprehensive processing disabled', 'aiox-publisher-suite-pro'),
                'enabled' => $enabled
            ));
            
        } catch (Exception $e) {
            wp_send_json_error(array(
                'message' => sprintf(__('Failed to toggle setting: %s', 'aiox-publisher-suite-pro'), $e->getMessage())
            ));
        }
    }
    
    /**
     * Get processing status for a specific post
     */
    private function get_post_processing_status($post_id) {
        $post = get_post($post_id);
        if (!$post) {
            throw new Exception(__('Post not found', 'aiox-publisher-suite-pro'));
        }
        
        // Get ingredients and count them
        $ingredients = get_post_meta($post_id, '_aiox_ingredients', true);
        $ingredients_count = 0;
        
        if (!empty($ingredients)) {
            if (is_array($ingredients)) {
                $ingredients_count = count($ingredients);
            } elseif (is_string($ingredients)) {
                // If it's JSON string, decode and count
                $decoded = json_decode($ingredients, true);
                if (is_array($decoded)) {
                    $ingredients_count = count($decoded);
                }
            }
        }
        
        // Get metadata and count them
        $metadata = get_post_meta($post_id, '_aiox_metadata', true);
        $metadata_count = 0;
        
        if (!empty($metadata)) {
            if (is_array($metadata)) {
                $metadata_count = count($metadata);
            } elseif (is_string($metadata)) {
                // If it's JSON string, decode and count
                $decoded = json_decode($metadata, true);
                if (is_array($decoded)) {
                    $metadata_count = count($decoded);
                }
            }
        }
        
        return array(
            'post_id' => $post_id,
            'title' => $post->post_title,
            'has_capsule' => !empty(get_post_meta($post_id, '_aiox_capsule_id', true)),
            'has_ingredients' => !empty($ingredients),
            'ingredients_count' => $ingredients_count,
            'has_processed_data' => !empty(get_post_meta($post_id, '_aiox_processed_data', true)),
            'has_metadata' => !empty($metadata),
            'metadata_count' => $metadata_count,
            'processing_complete' => get_post_meta($post_id, '_aiox_processing_complete', true) === '1',
            'last_processed' => get_post_meta($post_id, '_aiox_last_processed', true)
        );
    }
    
    /**
     * Get all content with processing status
     */
    private function get_all_content_with_status() {
        global $wpdb;
        
        $posts = get_posts(array(
            'post_type' => array('post', 'page'),
            'post_status' => 'publish',
            'numberposts' => 50,
            'orderby' => 'date',
            'order' => 'DESC'
        ));
        
        $content_list = array();
        $capsules_table = $wpdb->prefix . 'aiox_capsules';
        
        foreach ($posts as $post) {
            $status = $this->get_post_processing_status($post->ID);
            
            // Try to get capsule_id from post meta first (for newer posts)
            $capsule_id = get_post_meta($post->ID, '_aiox_capsule_id', true);
            
            // If not in meta, try to find it in the capsules table (for older posts)
            if (empty($capsule_id) && $wpdb->get_var("SHOW TABLES LIKE '$capsules_table'") == $capsules_table) {
                $patterns = array(
                    '%"post_id":' . $post->ID . '%',
                    '%"post_id":"' . $post->ID . '"%',
                    '%"source_posts":[' . $post->ID . ']%',
                    '%"source_posts":["' . $post->ID . '"]%'
                );
                
                foreach ($patterns as $pattern) {
                    $capsule = $wpdb->get_row($wpdb->prepare(
                        "SELECT id FROM $capsules_table WHERE metadata LIKE %s ORDER BY created_at DESC LIMIT 1",
                        $pattern
                    ), ARRAY_A);
                    
                    if ($capsule) {
                        $capsule_id = $capsule['id'];
                        // Save it to post meta for next time
                        update_post_meta($post->ID, '_aiox_capsule_id', $capsule_id);
                        break;
                    }
                }
            }
            
            $content_list[] = array(
                'id' => $post->ID,
                'title' => $post->post_title,
                'type' => $post->post_type,
                'date' => $post->post_date,
                'url' => get_permalink($post->ID),
                'edit_url' => get_edit_post_link($post->ID),
                'status' => $status['processing_complete'] ? 'processed' : 'unprocessed',
                'ingredients_count' => $status['ingredients_count'],
                'has_capsule' => $status['has_capsule'],
                'capsule_id' => $capsule_id ? $capsule_id : '',
                'has_ingredients' => $status['has_ingredients'],
                'has_processed_data' => $status['has_processed_data'],
                'has_manifesto' => $status['has_processed_data'],
                'has_badge' => $status['has_processed_data'],
                'has_metadata' => $status['has_metadata'],
                'metadata_count' => $status['metadata_count']
            );
        }
        
        return $content_list;
    }
    
    /**
     * Get overall processing status
     */
    private function get_overall_processing_status() {
        $total_posts = wp_count_posts('post')->publish + wp_count_posts('page')->publish;
        
        $processed_posts = get_posts(array(
            'post_type' => array('post', 'page'),
            'post_status' => 'publish',
            'meta_key' => '_aiox_processing_complete',
            'meta_value' => '1',
            'numberposts' => -1,
            'fields' => 'ids'
        ));
        
        $processed_count = count($processed_posts);
        $unprocessed_count = $total_posts - $processed_count;
        
        return array(
            'total_posts' => $total_posts,
            'processed_posts' => $processed_count,
            'unprocessed_posts' => $unprocessed_count,
            'processing_percentage' => $total_posts > 0 ? round(($processed_count / $total_posts) * 100, 2) : 0,
            'comprehensive_processing_enabled' => get_option('aiox_enable_comprehensive_processing', true),
            'last_updated' => current_time('mysql')
        );
    }
    
    /**
     * Get list of unprocessed posts
     */
    private function get_unprocessed_posts() {
        $posts = get_posts(array(
            'post_type' => array('post', 'page'),
            'post_status' => 'publish',
            'meta_query' => array(
                'relation' => 'OR',
                array(
                    'key' => '_aiox_processing_complete',
                    'compare' => 'NOT EXISTS'
                ),
                array(
                    'key' => '_aiox_processing_complete',
                    'value' => '1',
                    'compare' => '!='
                )
            ),
            'numberposts' => -1
        ));
        
        $unprocessed = array();
        foreach ($posts as $post) {
            $unprocessed[] = array(
                'id' => $post->ID,
                'title' => $post->post_title,
                'type' => $post->post_type,
                'date' => $post->post_date,
                'url' => get_permalink($post->ID)
            );
        }
        
        return $unprocessed;
    }
    
    /**
     * Update all manifesto files
     */
    private function update_all_manifesto_files() {
        // Simply regenerate all manifest files using AIOX_Hooks
        // This will automatically include all processed posts in the indexes
        if (class_exists('AIOX_Hooks')) {
            $hooks = new AIOX_Hooks();
            $hooks->generate_manifest();
            $hooks->generate_delta_feed();
            $hooks->generate_pack_file();
        }
        
        if (class_exists('AIOX_Ingredients')) {
            AIOX_Ingredients::export_ingredients_file();
        }
    }
    
    /**
     * Override parent render_page method to add comprehensive processing features
     * This preserves all existing functionality while adding new features
     */
    public function render_page() {
        $current_tab = isset($_GET['tab']) ? sanitize_text_field($_GET['tab']) : 'overview';
        
        // Get token balance from server
        $balance_data = $this->get_server_token_balance();
        
        ?>
        <div class="wrap">
            <h1>
                <span class="dashicons dashicons-admin-generic" style="font-size: 30px; width: 30px; height: 30px; margin-right: 10px; color: #2271b1;"></span>
                <?php _e('AIOX Processing Manager', 'aiox-publisher-suite-pro'); ?>
            </h1>
            <p class="description"><?php _e('AI Content Processing & Token Management', 'aiox-publisher-suite-pro'); ?></p>
            
            <!-- Token Balance Card -->
            <div class="aiox-token-card">
                <div class="aiox-token-main">
                    <div class="aiox-token-icon">
                        <span class="dashicons dashicons-database"></span>
                    </div>
                    <div class="aiox-token-info">
                        <span class="aiox-token-label"><?php _e('Token Balance', 'aiox-publisher-suite-pro'); ?></span>
                        <span class="aiox-token-value" id="aiox-token-balance">
                            <?php if ($balance_data['success']): ?>
                                <?php echo number_format($balance_data['total_available']); ?>
                            <?php else: ?>
                                <span style="color: #d63638;">--</span>
                            <?php endif; ?>
                        </span>
                    </div>
                </div>
                
                <?php if ($balance_data['success']): ?>
                <div class="aiox-token-breakdown">
                    <?php if ($balance_data['pool_remaining'] > 0): ?>
                    <div class="aiox-token-item">
                        <span class="aiox-item-label"><?php echo $balance_data['usage_type'] === 'monthly' ? 'Monthly' : 'Base'; ?></span>
                        <span class="aiox-item-value"><?php echo number_format($balance_data['pool_remaining']); ?></span>
                    </div>
                    <?php endif; ?>
                </div>
                
                <div class="aiox-token-plan">
                    <span class="aiox-plan-badge"><?php echo esc_html($balance_data['plan_name']); ?></span>
                </div>
                <?php else: ?>
                <div class="aiox-token-error">
                    <span class="dashicons dashicons-warning" style="color: #d63638;"></span>
                    <span><?php echo esc_html($balance_data['error'] ?? 'Unable to connect'); ?></span>
                </div>
                <?php endif; ?>
                
                <button type="button" class="button aiox-refresh-balance" onclick="aioxRefreshBalance()" title="<?php esc_attr_e('Refresh Balance', 'aiox-publisher-suite-pro'); ?>">
                    <span class="dashicons dashicons-update" style="vertical-align: middle;"></span>
                    <?php _e('Refresh', 'aiox-publisher-suite-pro'); ?>
                </button>
            </div>
            
            <!-- Navigation Tabs -->
            <nav class="nav-tab-wrapper" style="margin-bottom: 20px;">
                <a href="?page=aiox-capsules&tab=overview" class="nav-tab <?php echo $current_tab === 'overview' ? 'nav-tab-active' : ''; ?>">
                    <span class="dashicons dashicons-dashboard" style="vertical-align: middle; margin-right: 5px;"></span>
                    <?php _e('Overview', 'aiox-publisher-suite-pro'); ?>
                </a>
                <a href="?page=aiox-capsules&tab=manage" class="nav-tab <?php echo $current_tab === 'manage' ? 'nav-tab-active' : ''; ?>">
                    <span class="dashicons dashicons-list-view" style="vertical-align: middle; margin-right: 5px;"></span>
                    <?php _e('Manage Content', 'aiox-publisher-suite-pro'); ?>
                </a>
                <a href="?page=aiox-capsules&tab=bulk" class="nav-tab <?php echo $current_tab === 'bulk' ? 'nav-tab-active' : ''; ?>">
                    <span class="dashicons dashicons-update" style="vertical-align: middle; margin-right: 5px;"></span>
                    <?php _e('Bulk Processing', 'aiox-publisher-suite-pro'); ?>
                </a>
                <a href="?page=aiox-capsules&tab=settings" class="nav-tab <?php echo $current_tab === 'settings' ? 'nav-tab-active' : ''; ?>">
                    <span class="dashicons dashicons-admin-settings" style="vertical-align: middle; margin-right: 5px;"></span>
                    <?php _e('Settings', 'aiox-publisher-suite-pro'); ?>
                </a>
            </nav>
            
            <!-- Tab Content -->
            <div class="aiox-tab-content">
                <?php
                switch ($current_tab) {
                    case 'overview':
                        AIOX_Processing_Manager_Helper::render_simple_overview_tab();
                        break;
                    case 'manage':
                        AIOX_Processing_Manager_Helper::render_simple_manage_tab();
                        break;
                    case 'bulk':
                        $this->render_bulk_processing_tab();
                        break;
                    case 'settings':
                        AIOX_Processing_Manager_Helper::render_simple_settings_tab();
                        break;
                    default:
                        AIOX_Processing_Manager_Helper::render_simple_overview_tab();
                        break;
                }
                ?>
            </div>
        </div>
        
        <style>
        /* Token Balance Card */
        .aiox-token-card {
            display: flex;
            align-items: center;
            gap: 20px;
            background: #fff;
            border: 1px solid #c3c4c7;
            border-left: 4px solid #2271b1;
            border-radius: 4px;
            padding: 15px 20px;
            margin: 20px 0;
            flex-wrap: wrap;
        }
        .aiox-token-main {
            display: flex;
            align-items: center;
            gap: 12px;
        }
        .aiox-token-icon {
            width: 40px;
            height: 40px;
            background: #f0f6fc;
            border-radius: 50%;
            display: flex;
            align-items: center;
            justify-content: center;
        }
        .aiox-token-icon .dashicons {
            color: #2271b1;
            font-size: 20px;
        }
        .aiox-token-info {
            display: flex;
            flex-direction: column;
        }
        .aiox-token-label {
            font-size: 12px;
            color: #646970;
            text-transform: uppercase;
        }
        .aiox-token-value {
            font-size: 24px;
            font-weight: 600;
            color: #1d2327;
        }
        .aiox-token-breakdown {
            display: flex;
            gap: 20px;
            padding-left: 20px;
            border-left: 1px solid #dcdcde;
        }
        .aiox-token-item {
            display: flex;
            flex-direction: column;
        }
        .aiox-token-item .aiox-item-label {
            font-size: 11px;
            color: #646970;
        }
        .aiox-token-item .aiox-item-value {
            font-size: 16px;
            font-weight: 600;
            color: #1d2327;
        }
        .aiox-token-plan {
            margin-left: auto;
        }
        .aiox-plan-badge {
            background: #dba617;
            color: #fff;
            padding: 4px 12px;
            border-radius: 12px;
            font-size: 12px;
            font-weight: 600;
        }
        .aiox-token-error {
            display: flex;
            align-items: center;
            gap: 8px;
            color: #d63638;
        }
        .aiox-refresh-balance {
            padding: 5px 10px !important;
        }
        .aiox-refresh-balance .dashicons {
            font-size: 16px;
            width: 16px;
            height: 16px;
        }
        .aiox-refresh-balance.spinning .dashicons {
            animation: aiox-spin 1s linear infinite;
        }
        @keyframes aiox-spin {
            from { transform: rotate(0deg); }
            to { transform: rotate(360deg); }
        }
        
        /* Stats Grid */
        .aiox-stats-grid {
            display: grid;
            grid-template-columns: repeat(4, 1fr);
            gap: 15px;
            margin-bottom: 20px;
        }
        .aiox-stat-card {
            background: #fff;
            border: 1px solid #c3c4c7;
            border-radius: 4px;
            padding: 20px;
            text-align: center;
        }
        .aiox-stat-card h3 {
            margin: 0 0 10px 0;
            font-size: 14px;
            color: #646970;
        }
        .aiox-stat-card .stat-value {
            font-size: 32px;
            font-weight: 600;
            line-height: 1;
        }
        .aiox-stat-card .stat-value.green { color: #00a32a; }
        .aiox-stat-card .stat-value.red { color: #d63638; }
        .aiox-stat-card .stat-value.blue { color: #2271b1; }
        
        /* Progress Bar */
        .aiox-progress-container {
            background: #fff;
            border: 1px solid #c3c4c7;
            border-radius: 4px;
            padding: 20px;
            margin-bottom: 20px;
        }
        .aiox-progress-bar {
            height: 24px;
            background: #f0f0f1;
            border-radius: 12px;
            overflow: hidden;
            margin: 10px 0;
        }
        .aiox-progress-fill {
            height: 100%;
            background: linear-gradient(90deg, #00a32a, #2271b1);
            border-radius: 12px;
            transition: width 0.3s ease;
        }
        
        /* Responsive */
        @media screen and (max-width: 1200px) {
            .aiox-stats-grid { grid-template-columns: repeat(2, 1fr); }
        }
        @media screen and (max-width: 782px) {
            .aiox-stats-grid { grid-template-columns: 1fr; }
            .aiox-token-card { flex-direction: column; align-items: flex-start; }
            .aiox-token-breakdown { padding-left: 0; border-left: none; padding-top: 15px; border-top: 1px solid #dcdcde; width: 100%; }
            .aiox-token-plan { margin-left: 0; margin-top: 15px; }
        }
        </style>
        <?php
        
        // Add Phase 1 foundation JavaScript for new handlers
        $this->render_phase1_scripts();
    }
    
    /**
     * Render modern dark theme styles
     */
    private function render_modern_styles() {
        ?>
        <?php
    }
    
    /**
     * Render Phase 1 foundation scripts
     */
    private function render_phase1_scripts() {
        ?>
        <script type="text/javascript">
        // Token Balance Refresh Function
        function aioxRefreshBalance() {
            var btn = document.querySelector('.aiox-refresh-balance');
            var balanceEl = document.getElementById('aiox-token-balance');
            
            if (btn) btn.classList.add('spinning');
            
            jQuery.post(ajaxurl, {
                action: 'aiox_refresh_balance'
            }, function(response) {
                if (btn) btn.classList.remove('spinning');
                
                if (response.success && response.data.success) {
                    var data = response.data;
                    
                    // Update main balance
                    if (balanceEl) {
                        balanceEl.textContent = Number(data.total_available).toLocaleString();
                    }
                    
                    // Update breakdown
                    var breakdown = document.querySelector('.aiox-balance-breakdown');
                    if (breakdown) {
                        var html = '';
                        if (data.pool_remaining > 0) {
                            html += '<div class="aiox-balance-item aiox-pool">';
                            html += '<span class="aiox-item-label">' + (data.usage_type === 'monthly' ? 'Monthly' : 'Base') + '</span>';
                            html += '<span class="aiox-item-value">' + Number(data.pool_remaining).toLocaleString() + '</span>';
                            html += '</div>';
                        }
                        breakdown.innerHTML = html;
                    }
                    
                    // Update plan badge
                    var badge = document.querySelector('.aiox-plan-badge');
                    if (badge && data.plan_name) {
                        badge.textContent = data.plan_name;
                    }
                    
                    console.log('AIOX: Balance refreshed - ' + data.total_available + ' tokens');
                } else {
                    console.error('AIOX: Failed to refresh balance', response);
                }
            });
        }
        
        // Phase 1: Foundation JavaScript for comprehensive processing
        jQuery(document).ready(function($) {
            console.log('AIOX Processing Manager Phase 1: Foundation scripts loaded');
            
            // === NOTIFICATION SYSTEM ===
            // Create notification container if it doesn't exist
            if ($('#aiox-global-notification').length === 0) {
                $('body').append('<div id="aiox-global-notification" style="display: none; position: fixed; top: 50px; right: 20px; max-width: 450px; padding: 15px 20px; border-radius: 8px; border-left: 4px solid; z-index: 99999; box-shadow: 0 4px 12px rgba(0,0,0,0.3);"><div id="aiox-notification-close" style="position: absolute; top: 5px; right: 10px; cursor: pointer; font-size: 18px; color: #666;">×</div><div id="aiox-notification-content"></div></div>');
            }
            
            // Close notification on click
            $(document).on('click', '#aiox-notification-close', function() {
                $('#aiox-global-notification').fadeOut();
            });
            
            // Show notification function
            window.aioxShowNotification = function(type, title, message) {
                var notification = $('#aiox-global-notification');
                var content = $('#aiox-notification-content');
                
                var bgColor, borderColor, textColor;
                if (type === 'error') {
                    bgColor = '#fef2f2';
                    borderColor = '#ef4444';
                    textColor = '#991b1b';
                } else if (type === 'success') {
                    bgColor = '#f0fdf4';
                    borderColor = '#22c55e';
                    textColor = '#166534';
                } else if (type === 'warning') {
                    bgColor = '#fffbeb';
                    borderColor = '#f59e0b';
                    textColor = '#92400e';
                } else {
                    bgColor = '#eff6ff';
                    borderColor = '#3b82f6';
                    textColor = '#1e40af';
                }
                
                notification.css({
                    'background-color': bgColor,
                    'border-left-color': borderColor,
                    'color': textColor
                });
                
                content.html('<strong style="color: ' + textColor + '; font-size: 14px;">' + title + '</strong><br><span style="color: ' + textColor + '; font-size: 13px; line-height: 1.5;">' + message + '</span>');
                
                notification.fadeIn();
                
                // Auto-hide success after 5 seconds
                if (type === 'success') {
                    setTimeout(function() {
                        notification.fadeOut();
                    }, 5000);
                }
            };
            
            // Verify AJAX variables are available
            if (typeof aiox_capsules_ajax !== 'undefined') {
                console.log('AIOX Processing Manager: AJAX variables available');
                
                // Add foundation for comprehensive processing
                window.aiox_processing = {
                    ajax_url: aiox_capsules_ajax.ajax_url,
                    nonce: aiox_capsules_ajax.nonce,
                    
                    // Foundation method for comprehensive processing
                    processAllComponents: function(postId, callback) {
                        $.post(this.ajax_url, {
                            action: 'aiox_process_all_components',
                            nonce: this.nonce,
                            post_id: postId
                        }, function(response) {
                            if (callback) callback(response);
                        });
                    },
                    
                    // Foundation method for getting processing status
                    getProcessingStatus: function(postId, callback) {
                        $.post(this.ajax_url, {
                            action: 'aiox_get_processing_status',
                            nonce: this.nonce,
                            post_id: postId || 0
                        }, function(response) {
                            if (callback) callback(response);
                        });
                    }
                };
                
                // === CLICK HANDLERS FOR CONTENT TABLE ===
                
                // Process All Components button
                $(document).on('click', '.aiox-process-all-components, .aiox-process-single, .aiox-process-btn', function(e) {
                    e.preventDefault();
                    
                    var button = $(this);
                    var postId = button.data('post-id');
                    var originalHtml = button.html();
                    
                    if (!postId) {
                        aioxShowNotification('error', '❌ Error', 'Invalid post ID');
                        return;
                    }
                    
                    button.prop('disabled', true).html('Processing...');
                    
                    $.ajax({
                        url: aiox_capsules_ajax.ajax_url,
                        type: 'POST',
                        data: {
                            action: 'aiox_process_all_components',
                            post_id: postId,
                            nonce: aiox_capsules_ajax.nonce
                        },
                        success: function(response) {
                            // Refresh token balance
                            if (typeof aioxRefreshBalance === 'function') {
                                aioxRefreshBalance();
                            }
                            
                            if (response.success) {
                                button.html('✅ Done');
                                
                                var msg = response.data.message || 'Processing complete';
                                if (response.data.components) {
                                    msg += '<br><br><strong>Components:</strong> ';
                                    var parts = [];
                                    if (response.data.components.ingredients) parts.push(response.data.components.ingredients + ' ingredients');
                                    if (response.data.components.metadata) parts.push(response.data.components.metadata + ' metadata');
                                    if (response.data.components.capsule) parts.push('Capsule');
                                    msg += parts.join(', ');
                                }
                                
                                aioxShowNotification('success', '✅ Processing Complete!', msg);
                                
                                setTimeout(function() {
                                    location.reload();
                                }, 2500);
                            } else {
                                button.prop('disabled', false).html(originalHtml);
                                
                                var errorMsg = response.data && response.data.message ? response.data.message : 'Unknown error';
                                
                                // Check for token-related errors
                                if (errorMsg.indexOf('Token') !== -1 || errorMsg.indexOf('token') !== -1 || errorMsg.indexOf('Insufficient') !== -1 || errorMsg.indexOf('limit') !== -1) {
                                    aioxShowNotification('warning', '⚠️ Token Limit Reached', errorMsg);
                                } else {
                                    aioxShowNotification('error', '❌ Processing Failed', errorMsg);
                                }
                            }
                        },
                        error: function(xhr, status, error) {
                            button.prop('disabled', false).html(originalHtml);
                            
                            if (typeof aioxRefreshBalance === 'function') {
                                aioxRefreshBalance();
                            }
                            
                            aioxShowNotification('error', '❌ Request Failed', 'Connection error: ' + error);
                        }
                    });
                });
                
                // Clear AI Data button
                $(document).on('click', '.aiox-clear-ai-data, .aiox-clear-btn', function(e) {
                    e.preventDefault();
                    
                    if (!confirm('This will permanently delete all AI data for this post. Continue?')) {
                        return;
                    }
                    
                    var button = $(this);
                    var postId = button.data('post-id');
                    var originalHtml = button.html();
                    
                    button.prop('disabled', true).html('Clearing...');
                    
                    $.ajax({
                        url: aiox_capsules_ajax.ajax_url,
                        type: 'POST',
                        data: {
                            action: 'aiox_clear_all_ai_data',
                            post_id: postId,
                            nonce: aiox_capsules_ajax.nonce
                        },
                        success: function(response) {
                            if (response.success) {
                                aioxShowNotification('success', '✅ Data Cleared', response.data.message || 'All AI data removed');
                                setTimeout(function() {
                                    location.reload();
                                }, 1500);
                            } else {
                                button.prop('disabled', false).html(originalHtml);
                                aioxShowNotification('error', '❌ Clear Failed', response.data.message || 'Failed to clear data');
                            }
                        },
                        error: function() {
                            button.prop('disabled', false).html(originalHtml);
                            aioxShowNotification('error', '❌ Request Failed', 'Failed to clear AI data');
                        }
                    });
                });
                
                console.log('AIOX Processing Manager: Foundation methods initialized');
            } else {
                console.warn('AIOX Processing Manager: AJAX variables not available');
            }
        });
        </script>
        <?php
    }
    
    /**
     * AJAX: Save automatic processing settings
     */
    public function ajax_save_auto_settings() {
        // Check user permissions first
        if (!current_user_can('manage_options')) {
            wp_send_json_error(array('message' => __('Insufficient permissions', 'aiox-publisher-suite-pro')));
            return;
        }
        
        // Simple nonce check - try multiple possible nonce values
        $nonce_valid = false;
        $nonce = $_POST['nonce'] ?? $_REQUEST['nonce'] ?? '';
        
        if ($nonce) {
            $nonce_valid = wp_verify_nonce($nonce, 'aiox_ajax_nonce') || 
                          wp_verify_nonce($nonce, 'aiox_capsules_nonce') ||
                          wp_verify_nonce($nonce, 'aiox_nonce');
        }
        
        if (!$nonce_valid) {
            wp_send_json_error(array('message' => 'Security check failed for save settings - nonce: ' . substr($nonce, 0, 10) . '...'));
            return;
        }
        
        try {
            $auto_process_on_save = isset($_POST['auto_process_on_save']) && $_POST['auto_process_on_save'] == '1';
            
            update_option('aiox_auto_process_on_save', $auto_process_on_save);
            
            wp_send_json_success(array(
                'message' => __('Automatic processing settings saved successfully', 'aiox-publisher-suite-pro'),
                'auto_process_on_save' => $auto_process_on_save
            ));
            
        } catch (Exception $e) {
            wp_send_json_error(array(
                'message' => sprintf(__('Failed to save settings: %s', 'aiox-publisher-suite-pro'), $e->getMessage())
            ));
        }
    }
    
    /**
     * Render the bulk processing tab
     */
    private function render_bulk_processing_tab() {
        $status = AIOX_Processing_Manager_Helper::get_overall_processing_status();
        ?>
        <div id="poststuff">
            <div id="post-body" class="metabox-holder columns-2">
                <div id="post-body-content">
                    
                    <!-- Bulk Actions Card -->
                    <div class="postbox">
                        <h2 class="hndle"><span class="dashicons dashicons-update"></span> <?php _e('Bulk Processing Actions', 'aiox-publisher-suite-pro'); ?></h2>
                        <div class="inside">
                            <p class="description"><?php _e('Process multiple posts at once. This will use API tokens for each post processed.', 'aiox-publisher-suite-pro'); ?></p>
                            
                            <?php if ($status['unprocessed_posts'] > 0): ?>
                            <div class="aiox-bulk-action-card" style="background: #f0f6fc; border: 1px solid #c3c4c7; border-left: 4px solid #2271b1; padding: 20px; border-radius: 4px; margin: 15px 0;">
                                <h3 style="margin: 0 0 10px 0; display: flex; align-items: center; gap: 8px;">
                                    <span class="dashicons dashicons-clock" style="color: #2271b1;"></span>
                                    <?php _e('Unprocessed Content', 'aiox-publisher-suite-pro'); ?>
                                </h3>
                                <p style="margin: 0 0 15px 0;">
                                    <?php printf(
                                        __('You have <strong>%d posts</strong> waiting to be processed.', 'aiox-publisher-suite-pro'),
                                        $status['unprocessed_posts']
                                    ); ?>
                                </p>
                                <button type="button" class="button button-primary button-hero" id="aiox-bulk-process-all">
                                    <span class="dashicons dashicons-controls-play" style="vertical-align: middle; margin-right: 5px;"></span>
                                    <?php printf(__('Process All %d Posts', 'aiox-publisher-suite-pro'), $status['unprocessed_posts']); ?>
                                </button>
                            </div>
                            <?php else: ?>
                            <div class="notice notice-success inline" style="margin: 15px 0;">
                                <p><span class="dashicons dashicons-yes-alt" style="color: #00a32a;"></span> <?php _e('All content has been processed!', 'aiox-publisher-suite-pro'); ?></p>
                            </div>
                            <?php endif; ?>
                            
                            <!-- Progress Display -->
                            <div id="aiox-bulk-progress" style="display: none; margin: 20px 0;">
                                <div style="display: flex; justify-content: space-between; margin-bottom: 5px;">
                                    <span id="aiox-progress-text"><?php _e('Processing...', 'aiox-publisher-suite-pro'); ?></span>
                                    <span id="aiox-progress-percent">0%</span>
                                </div>
                                <div class="aiox-progress-bar">
                                    <div class="aiox-progress-fill" id="aiox-progress-fill" style="width: 0%;"></div>
                                </div>
                                <div id="aiox-progress-details" style="font-size: 12px; color: #646970; margin-top: 5px;"></div>
                            </div>
                            
                            <div id="aiox-bulk-status"></div>
                        </div>
                    </div>
                    
                    <!-- Auto Processing Settings -->
                    <div class="postbox">
                        <h2 class="hndle"><span class="dashicons dashicons-admin-settings"></span> <?php _e('Automatic Processing', 'aiox-publisher-suite-pro'); ?></h2>
                        <div class="inside">
                            <p class="description"><?php _e('Configure automatic processing for new and updated content.', 'aiox-publisher-suite-pro'); ?></p>
                            
                            <ul class="aiox-toggle-list" style="margin: 15px 0;">
                                <li class="aiox-toggle-item">
                                    <div class="toggle-info">
                                        <span class="toggle-label"><?php _e('Auto-Process on Save', 'aiox-publisher-suite-pro'); ?></span>
                                        <span class="toggle-desc"><?php _e('Automatically process content when posts/pages are saved', 'aiox-publisher-suite-pro'); ?></span>
                                    </div>
                                    <label class="aiox-switch">
                                        <input type="checkbox" name="aiox_auto_process_on_save" value="1" <?php checked(get_option('aiox_auto_process_on_save', false)); ?> />
                                        <span class="slider"></span>
                                    </label>
                                </li>
                            </ul>
                            
                            <p>
                                <button type="button" class="button button-primary" id="aiox-save-auto-settings">
                                    <?php _e('Save Settings', 'aiox-publisher-suite-pro'); ?>
                                </button>
                                <span id="aiox-settings-status" style="margin-left: 10px;"></span>
                            </p>
                        </div>
                    </div>
                    
                </div><!-- /post-body-content -->
                
                <!-- Sidebar -->
                <div id="postbox-container-1" class="postbox-container">
                    <div class="postbox">
                        <h2 class="hndle"><?php _e('Processing Tips', 'aiox-publisher-suite-pro'); ?></h2>
                        <div class="inside">
                            <ul style="margin: 0; padding-left: 20px;">
                                <li><?php _e('Processing uses API tokens', 'aiox-publisher-suite-pro'); ?></li>
                                <li><?php _e('Longer content uses more tokens', 'aiox-publisher-suite-pro'); ?></li>
                                <li><?php _e('Process in batches for large sites', 'aiox-publisher-suite-pro'); ?></li>
                                <li><?php _e('Check token balance before bulk operations', 'aiox-publisher-suite-pro'); ?></li>
                            </ul>
                        </div>
                    </div>
                    
                    <div class="postbox">
                        <h2 class="hndle"><?php _e('Quick Stats', 'aiox-publisher-suite-pro'); ?></h2>
                        <div class="inside">
                            <table class="widefat" style="border: none;">
                                <tr>
                                    <td><?php _e('Total Posts', 'aiox-publisher-suite-pro'); ?></td>
                                    <td style="text-align: right;"><strong><?php echo $status['total_posts']; ?></strong></td>
                                </tr>
                                <tr>
                                    <td><?php _e('Processed', 'aiox-publisher-suite-pro'); ?></td>
                                    <td style="text-align: right; color: #00a32a;"><strong><?php echo $status['processed_posts']; ?></strong></td>
                                </tr>
                                <tr>
                                    <td><?php _e('Pending', 'aiox-publisher-suite-pro'); ?></td>
                                    <td style="text-align: right; color: #d63638;"><strong><?php echo $status['unprocessed_posts']; ?></strong></td>
                                </tr>
                            </table>
                        </div>
                    </div>
                </div>
                
            </div><!-- /post-body -->
        </div><!-- /poststuff -->
        
        <script>
        jQuery(document).ready(function($) {
            var ajax_url = ajaxurl;
            var nonce = '<?php echo wp_create_nonce('aiox_capsules_nonce'); ?>';
            
            // Bulk process all
            $('#aiox-bulk-process-all').on('click', function() {
                if (!confirm('<?php echo esc_js(__('This will process all unprocessed posts. This may take several minutes and use API tokens. Continue?', 'aiox-publisher-suite-pro')); ?>')) {
                    return;
                }
                
                var button = $(this);
                var progressDiv = $('#aiox-bulk-progress');
                var statusDiv = $('#aiox-bulk-status');
                
                button.prop('disabled', true);
                progressDiv.show();
                statusDiv.html('');
                
                // Get unprocessed posts
                $.ajax({
                    url: ajax_url,
                    type: 'POST',
                    data: { action: 'aiox_get_unprocessed_posts', nonce: nonce },
                    success: function(response) {
                        if (response.success && response.data.posts && response.data.posts.length > 0) {
                            processPostsSequentially(response.data.posts, button, progressDiv);
                        } else {
                            progressDiv.hide();
                            statusDiv.html('<div class="notice notice-success inline"><p><?php echo esc_js(__('No unprocessed posts found!', 'aiox-publisher-suite-pro')); ?></p></div>');
                            button.prop('disabled', false);
                        }
                    },
                    error: function() {
                        progressDiv.hide();
                        statusDiv.html('<div class="notice notice-error inline"><p><?php echo esc_js(__('Failed to get posts list.', 'aiox-publisher-suite-pro')); ?></p></div>');
                        button.prop('disabled', false);
                    }
                });
            });
            
            function processPostsSequentially(posts, button, progressDiv) {
                var total = posts.length;
                var current = 0;
                var success = 0;
                var failed = 0;
                
                function processNext() {
                    if (current >= total) {
                        // Complete
                        var msg = success + ' <?php echo esc_js(__('processed', 'aiox-publisher-suite-pro')); ?>';
                        if (failed > 0) msg += ', ' + failed + ' <?php echo esc_js(__('failed', 'aiox-publisher-suite-pro')); ?>';
                        
                        $('#aiox-bulk-status').html('<div class="notice notice-success inline"><p><strong><?php echo esc_js(__('Complete!', 'aiox-publisher-suite-pro')); ?></strong> ' + msg + '</p></div>');
                        button.prop('disabled', false).html('<span class="dashicons dashicons-yes" style="vertical-align: middle;"></span> <?php echo esc_js(__('Done!', 'aiox-publisher-suite-pro')); ?>');
                        
                        setTimeout(function() { location.reload(); }, 2000);
                        return;
                    }
                    
                    var post = posts[current];
                    var percent = Math.round(((current + 1) / total) * 100);
                    
                    $('#aiox-progress-text').text('<?php echo esc_js(__('Processing:', 'aiox-publisher-suite-pro')); ?> ' + post.title);
                    $('#aiox-progress-percent').text(percent + '%');
                    $('#aiox-progress-fill').css('width', percent + '%');
                    $('#aiox-progress-details').text((current + 1) + ' / ' + total);
                    
                    $.ajax({
                        url: ajax_url,
                        type: 'POST',
                        timeout: 60000,
                        data: { action: 'aiox_process_all_components', post_id: post.id, nonce: nonce },
                        success: function(response) {
                            if (response.success) success++;
                            else failed++;
                        },
                        error: function() { failed++; },
                        complete: function() {
                            current++;
                            if (typeof aioxRefreshBalance === 'function') aioxRefreshBalance();
                            setTimeout(processNext, 500);
                        }
                    });
                }
                
                processNext();
            }
            
            // Save auto settings
            $('#aiox-save-auto-settings').on('click', function() {
                var button = $(this);
                var status = $('#aiox-settings-status');
                var autoProcess = $('input[name="aiox_auto_process_on_save"]').prop('checked');
                
                button.prop('disabled', true);
                status.html('<span style="color: #2271b1;"><?php echo esc_js(__('Saving...', 'aiox-publisher-suite-pro')); ?></span>');
                
                $.ajax({
                    url: ajax_url,
                    type: 'POST',
                    data: { action: 'aiox_save_auto_settings', nonce: nonce, auto_process_on_save: autoProcess ? 1 : 0 },
                    success: function(response) {
                        button.prop('disabled', false);
                        if (response.success) {
                            status.html('<span style="color: #00a32a;"><span class="dashicons dashicons-yes"></span> <?php echo esc_js(__('Saved!', 'aiox-publisher-suite-pro')); ?></span>');
                        } else {
                            status.html('<span style="color: #d63638;"><?php echo esc_js(__('Error saving', 'aiox-publisher-suite-pro')); ?></span>');
                        }
                        setTimeout(function() { status.html(''); }, 3000);
                    },
                    error: function() {
                        button.prop('disabled', false);
                        status.html('<span style="color: #d63638;"><?php echo esc_js(__('Network error', 'aiox-publisher-suite-pro')); ?></span>');
                    }
                });
            });
        });
        </script>
        <?php
    }
}