<?php
/**
 * AIOX Settings Admin Page - Modern Dark Style
 * 
 * Handles the comprehensive settings interface for AIOX Publisher Suite Pro
 */

if (!defined('ABSPATH')) {
    exit;
}

class AIOX_Settings {
    
    /**
     * Initialize the settings page
     */
    public function __construct() {
        add_action('admin_init', array($this, 'register_settings'));
        add_action('wp_ajax_aiox_test_ai_connection', array($this, 'ajax_test_ai_connection'));
        add_action('wp_ajax_aiox_validate_license', array($this, 'ajax_validate_license'));
        add_action('wp_ajax_aiox_clear_logs', array($this, 'ajax_clear_logs'));
        add_action('wp_ajax_aiox_autofix_files', array($this, 'ajax_autofix_files'));
        add_action('wp_ajax_aiox_test_ai_api', array($this, 'ajax_test_ai_api'));
        add_action('wp_ajax_aiox_bulk_process', array($this, 'ajax_bulk_process'));
        add_action('wp_ajax_aiox_total_reset', array($this, 'ajax_total_reset'));
        add_action('admin_enqueue_scripts', array($this, 'enqueue_settings_scripts'));
    }
    
    /**
     * Register plugin settings
     */
    public function register_settings() {
        register_setting('aiox-settings', 'aiox_license_key', array(
            'type' => 'string',
            'sanitize_callback' => 'sanitize_text_field',
            'default' => ''
        ));
        register_setting('aiox-settings', 'aiox_ai_endpoint', array(
            'type' => 'string',
            'sanitize_callback' => 'esc_url_raw',
            'default' => 'https://api.aiox.com/v1/generate'
        ));
        register_setting('aiox-settings', 'aiox_mode', array(
            'type' => 'string',
            'sanitize_callback' => 'sanitize_text_field',
            'default' => 'basic'
        ));
        register_setting('aiox-settings', 'aiox_custom_endpoint', array(
            'type' => 'string',
            'sanitize_callback' => 'esc_url_raw',
            'default' => ''
        ));
        register_setting('aiox-settings', 'aiox_custom_api_key', array(
            'type' => 'string',
            'sanitize_callback' => array($this, 'sanitize_api_key'),
            'default' => ''
        ));
        register_setting('aiox-settings', 'aiox_auto_process_on_save', array(
            'type' => 'boolean',
            'sanitize_callback' => array($this, 'sanitize_checkbox'),
            'default' => false
        ));
        register_setting('aiox-settings', 'aiox_enable_ai_badge', array(
            'type' => 'boolean',
            'sanitize_callback' => array($this, 'sanitize_checkbox'),
            'default' => true
        ));
        register_setting('aiox-settings', 'aiox_enable_faqpage_schema', array(
            'type' => 'boolean',
            'sanitize_callback' => array($this, 'sanitize_checkbox'),
            'default' => true
        ));
        register_setting('aiox-settings', 'aiox_enable_social_meta', array(
            'type' => 'boolean',
            'sanitize_callback' => array($this, 'sanitize_checkbox'),
            'default' => true
        ));
        register_setting('aiox-settings', 'aiox_enable_inline_json', array(
            'type' => 'boolean',
            'sanitize_callback' => array($this, 'sanitize_checkbox'),
            'default' => true
        ));
        register_setting('aiox-settings', 'aiox_enable_auto_processing', array(
            'type' => 'boolean',
            'sanitize_callback' => array($this, 'sanitize_checkbox'),
            'default' => false
        ));
        register_setting('aiox-settings', 'aiox_processing_limit', array(
            'type' => 'integer',
            'sanitize_callback' => array($this, 'sanitize_processing_limit'),
            'default' => 10
        ));
        register_setting('aiox-settings', 'aiox_log_level', array(
            'type' => 'string',
            'sanitize_callback' => array($this, 'sanitize_log_level'),
            'default' => 'info'
        ));
        register_setting('aiox-settings', 'aiox_tdm_training_allowed', array(
            'type' => 'boolean',
            'sanitize_callback' => array($this, 'sanitize_checkbox'),
            'default' => false
        ));
        register_setting('aiox-settings', 'aiox_tdm_citation_required', array(
            'type' => 'boolean',
            'sanitize_callback' => array($this, 'sanitize_checkbox'),
            'default' => true
        ));
        register_setting('aiox-settings', 'aiox_attribution_string', array(
            'type' => 'string',
            'sanitize_callback' => 'sanitize_text_field',
            'default' => get_bloginfo('name')
        ));
        register_setting('aiox-settings', 'aiox_enable_ingredients_system', array(
            'type' => 'boolean',
            'sanitize_callback' => array($this, 'sanitize_checkbox'),
            'default' => true
        ));
        register_setting('aiox-settings', 'aiox_enable_metadata_classification', array(
            'type' => 'boolean',
            'sanitize_callback' => array($this, 'sanitize_checkbox'),
            'default' => true
        ));
        register_setting('aiox-settings', 'aiox_ingredients_extraction_depth', array(
            'type' => 'string',
            'sanitize_callback' => array($this, 'sanitize_extraction_depth'),
            'default' => 'standard'
        ));
        register_setting('aiox-settings', 'aiox_enable_legacy_qa_compatibility', array(
            'type' => 'boolean',
            'sanitize_callback' => array($this, 'sanitize_checkbox'),
            'default' => true
        ));
        
        // Badge Q&A Count Setting
        register_setting('aiox-settings', 'aiox_badge_qa_count', array(
            'type' => 'integer',
            'sanitize_callback' => array($this, 'sanitize_qa_count'),
            'default' => 5
        ));
        
        // Monetization Settings (AIOX Protocol Compliance)
        register_setting('aiox-settings', 'aiox_monetization_pricing_tier', array(
            'type' => 'string',
            'sanitize_callback' => array($this, 'sanitize_pricing_tier'),
            'default' => 'free'
        ));
        register_setting('aiox-settings', 'aiox_monetization_subscription', array(
            'type' => 'boolean',
            'sanitize_callback' => array($this, 'sanitize_checkbox'),
            'default' => false
        ));
        register_setting('aiox-settings', 'aiox_monetization_pay_per_use', array(
            'type' => 'boolean',
            'sanitize_callback' => array($this, 'sanitize_checkbox'),
            'default' => false
        ));
        register_setting('aiox-settings', 'aiox_monetization_license_api', array(
            'type' => 'string',
            'sanitize_callback' => 'esc_url_raw',
            'default' => ''
        ));
        
        // License Preset Settings
        register_setting('aiox-settings', 'aiox_license_preset', array(
            'type' => 'string',
            'sanitize_callback' => array($this, 'sanitize_license_preset'),
            'default' => 'attribution_required'
        ));
        register_setting('aiox-settings', 'aiox_monetization_preset', array(
            'type' => 'string',
            'sanitize_callback' => array($this, 'sanitize_monetization_preset'),
            'default' => 'free'
        ));
        register_setting('aiox-settings', 'aiox_enforcement_mode', array(
            'type' => 'string',
            'sanitize_callback' => array($this, 'sanitize_enforcement_mode'),
            'default' => 'flexible'
        ));
        register_setting('aiox-settings', 'aiox_license_custom_terms', array(
            'type' => 'string',
            'sanitize_callback' => 'sanitize_textarea_field',
            'default' => ''
        ));
        register_setting('aiox-settings', 'aiox_license_contact_email', array(
            'type' => 'string',
            'sanitize_callback' => 'sanitize_email',
            'default' => ''
        ));
        register_setting('aiox-settings', 'aiox_license_terms_url', array(
            'type' => 'string',
            'sanitize_callback' => 'esc_url_raw',
            'default' => ''
        ));
        
        // Register AIOX Robots.txt settings
register_setting('aiox_robots_settings', 'aiox_robots_crawl_delay', array(
    'type' => 'integer',
    'default' => 1,
    'sanitize_callback' => 'absint'
));
    }
    
    /**
     * Sanitize checkbox values
     */
    public function sanitize_checkbox($value) {
        return !empty($value) ? 1 : 0;
    }
    
    /**
     * Sanitize API key and clear dismissed notice when key is set
     */
    public function sanitize_api_key($value) {
        $value = sanitize_text_field($value);
        
        // If API key is being set (non-empty), clear the dismissed notice
        if (!empty($value)) {
            delete_option('aiox_api_notice_dismissed_until');
        }
        
        return $value;
    }
    
    /**
     * Sanitize processing limit
     */
    public function sanitize_processing_limit($value) {
        $value = intval($value);
        return ($value >= 1 && $value <= 100) ? $value : 10;
    }
    
    /**
     * Sanitize log level
     */
    public function sanitize_log_level($value) {
        $allowed = array('debug', 'info', 'warning', 'error');
        return in_array($value, $allowed) ? $value : 'info';
    }
    
    /**
     * Sanitize extraction depth
     */
    public function sanitize_extraction_depth($value) {
        $allowed = array('basic', 'standard', 'deep');
        return in_array($value, $allowed) ? $value : 'standard';
    }
    
    /**
     * Sanitize monetization pricing tier
     */
    public function sanitize_pricing_tier($value) {
        $allowed = array('free', 'freemium', 'subscription', 'pay_per_use', 'enterprise');
        return in_array($value, $allowed) ? $value : 'free';
    }
    
    /**
     * Sanitize license preset
     */
    public function sanitize_license_preset($value) {
        $allowed = array('open_access', 'attribution_required', 'commercial_license');
        return in_array($value, $allowed) ? $value : 'attribution_required';
    }
    
    /**
     * Sanitize monetization preset
     */
    public function sanitize_monetization_preset($value) {
        $allowed = array('free', 'freemium', 'paid');
        return in_array($value, $allowed) ? $value : 'free';
    }
    
    /**
     * Sanitize enforcement mode
     */
    public function sanitize_enforcement_mode($value) {
        $allowed = array('strict', 'flexible');
        return in_array($value, $allowed) ? $value : 'flexible';
    }
    
    /**
     * Sanitize Q&A count setting
     * Ensures value is between 1 and 10
     */
    public function sanitize_qa_count($value) {
        $value = intval($value);
        if ($value < 1) return 1;
        if ($value > 10) return 10;
        return $value;
    }
    
    /**
     * Get license preset details
     * Returns full configuration for a license preset
     */
    public static function get_license_preset_config($preset = null) {
        $presets = array(
            'open_access' => array(
                'name' => __('Open Access', 'aiox-publisher-suite-pro'),
                'description' => __('Free to use, train, and cite with attribution. Ideal for open-source content, educational materials, and content creators who want maximum visibility.', 'aiox-publisher-suite-pro'),
                'icon' => 'unlock',
                'training_allowed' => true,
                'citation_required' => true,
                'attribution_required' => true,
                'commercial_use' => true,
                'derivatives_allowed' => true,
                'share_alike' => false,
                'license_code' => 'AIOX-OA-1.0',
                'comparable_to' => 'CC BY 4.0'
            ),
            'attribution_required' => array(
                'name' => __('Attribution Required', 'aiox-publisher-suite-pro'),
                'description' => __('Content can be cited and referenced, but AI training requires permission. Good balance between visibility and control.', 'aiox-publisher-suite-pro'),
                'icon' => 'id-alt',
                'training_allowed' => false,
                'citation_required' => true,
                'attribution_required' => true,
                'commercial_use' => true,
                'derivatives_allowed' => true,
                'share_alike' => true,
                'license_code' => 'AIOX-AR-1.0',
                'comparable_to' => 'CC BY-SA 4.0'
            ),
            'commercial_license' => array(
                'name' => __('Commercial License', 'aiox-publisher-suite-pro'),
                'description' => __('All AI usage requires a licensing agreement. For premium content, proprietary data, and publishers who want to monetize AI access.', 'aiox-publisher-suite-pro'),
                'icon' => 'money-alt',
                'training_allowed' => false,
                'citation_required' => true,
                'attribution_required' => true,
                'commercial_use' => false,
                'derivatives_allowed' => false,
                'share_alike' => false,
                'license_code' => 'AIOX-CL-1.0',
                'comparable_to' => 'All Rights Reserved'
            )
        );
        
        if ($preset && isset($presets[$preset])) {
            return $presets[$preset];
        }
        
        return $presets;
    }
    
    /**
     * Get monetization preset details
     */
    public static function get_monetization_preset_config($preset = null) {
        $presets = array(
            'free' => array(
                'name' => __('Free Access', 'aiox-publisher-suite-pro'),
                'description' => __('No cost for AI systems to access, index, or use your content. Maximum reach and visibility.', 'aiox-publisher-suite-pro'),
                'icon' => 'heart',
                'subscription_required' => false,
                'pay_per_use' => false,
                'api_access' => true,
                'rate_limited' => false,
                'pricing_model' => 'free'
            ),
            'freemium' => array(
                'name' => __('Freemium', 'aiox-publisher-suite-pro'),
                'description' => __('Basic access is free, but premium features or high-volume usage requires payment. Good for testing the waters.', 'aiox-publisher-suite-pro'),
                'icon' => 'star-half',
                'subscription_required' => false,
                'pay_per_use' => true,
                'api_access' => true,
                'rate_limited' => true,
                'pricing_model' => 'freemium'
            ),
            'paid' => array(
                'name' => __('Paid Access', 'aiox-publisher-suite-pro'),
                'description' => __('All AI access requires payment - either subscription or pay-per-use. For premium content monetization.', 'aiox-publisher-suite-pro'),
                'icon' => 'money-alt',
                'subscription_required' => true,
                'pay_per_use' => true,
                'api_access' => true,
                'rate_limited' => true,
                'pricing_model' => 'paid'
            )
        );
        
        if ($preset && isset($presets[$preset])) {
            return $presets[$preset];
        }
        
        return $presets;
    }
    
    /**
     * Get enforcement mode details
     */
    public static function get_enforcement_mode_config($mode = null) {
        $modes = array(
            'strict' => array(
                'name' => __('Strict Enforcement', 'aiox-publisher-suite-pro'),
                'description' => __('These are requirements, not suggestions. AI systems that violate these terms are considered non-compliant and may face legal action.', 'aiox-publisher-suite-pro'),
                'icon' => 'shield-alt',
                'compliance_level' => 'mandatory',
                'violation_response' => 'block',
                'legal_binding' => true,
                'grace_period' => false
            ),
            'flexible' => array(
                'name' => __('Flexible Enforcement', 'aiox-publisher-suite-pro'),
                'description' => __('These are strong preferences, but AI systems may proceed in good faith while the AIOX protocol gains adoption. Ideal during the protocol\'s early days.', 'aiox-publisher-suite-pro'),
                'icon' => 'admin-generic',
                'compliance_level' => 'recommended',
                'violation_response' => 'warn',
                'legal_binding' => false,
                'grace_period' => true
            )
        );
        
        if ($mode && isset($modes[$mode])) {
            return $modes[$mode];
        }
        
        return $modes;
    }
    
    /**
     * Render the settings page
     */
    public function render_settings_page() {
        // Get current settings
        $custom_endpoint = get_option('aiox_custom_endpoint', '');
        $custom_api_key = get_option('aiox_custom_api_key', '');
        $auto_process_on_save = get_option('aiox_auto_process_on_save', false);
        $enable_ai_badge = get_option('aiox_enable_ai_badge', true);
        $enable_faqpage_schema = get_option('aiox_enable_faqpage_schema', true);
        $enable_social_meta = get_option('aiox_enable_social_meta', true);
        $enable_inline_json = get_option('aiox_enable_inline_json', true);
        $processing_limit = get_option('aiox_processing_limit', 10);
        $log_level = get_option('aiox_log_level', 'error');
        $tdm_training_allowed = get_option('aiox_tdm_training_allowed', false);
        $tdm_citation_required = get_option('aiox_tdm_citation_required', true);
        $attribution_string = get_option('aiox_attribution_string', get_bloginfo('name'));
        $enable_ingredients_system = get_option('aiox_enable_ingredients_system', true);
        $enable_metadata_classification = get_option('aiox_enable_metadata_classification', true);
        $ingredients_extraction_depth = get_option('aiox_ingredients_extraction_depth', 'standard');
        $enable_legacy_qa_compatibility = get_option('aiox_enable_legacy_qa_compatibility', true);
        $badge_qa_count = get_option('aiox_badge_qa_count', 5);
        
        // Monetization settings
        $monetization_pricing_tier = get_option('aiox_monetization_pricing_tier', 'free');
        $monetization_subscription = get_option('aiox_monetization_subscription', false);
        $monetization_pay_per_use = get_option('aiox_monetization_pay_per_use', false);
        $monetization_license_api = get_option('aiox_monetization_license_api', '');
        
        ?>
        <div class="wrap">
            <h1><span class="dashicons dashicons-networking" style="font-size: 30px; width: 30px; height: 30px; margin-right: 10px; color: #2271b1;"></span><?php _e('AIOX Settings', 'aiox-publisher-suite-pro'); ?></h1>
            <p class="description"><?php _e('Configure AI Provider & Plugin Options', 'aiox-publisher-suite-pro'); ?></p>
            
            <form method="post" action="options.php" id="aiox-settings-form">
                <?php settings_fields('aiox-settings'); ?>
                
                <div id="poststuff">
                    <div id="post-body" class="metabox-holder columns-2">
                        <div id="post-body-content">
                            
                            <!-- AI Provider Configuration -->
                            <div class="postbox">
                                <h2 class="hndle"><span class="dashicons dashicons-cloud"></span> <?php _e('AI Provider Configuration', 'aiox-publisher-suite-pro'); ?></h2>
                                <div class="inside">
                                    <table class="form-table">
                                        <tr>
                                            <th scope="row">
                                                <label for="aiox_custom_endpoint"><?php _e('Custom AI Endpoint', 'aiox-publisher-suite-pro'); ?></label>
                                            </th>
                                            <td>
                                                <input type="url" name="aiox_custom_endpoint" id="aiox_custom_endpoint" value="<?php echo esc_attr($custom_endpoint); ?>" class="regular-text" placeholder="https://your-api.com/v1/chat/completions" />
                                                <p class="description"><?php _e('Enter your custom AI API endpoint URL', 'aiox-publisher-suite-pro'); ?></p>
                                            </td>
                                        </tr>
                                        <tr>
                                            <th scope="row">
                                                <label for="aiox_custom_api_key"><?php _e('API Key', 'aiox-publisher-suite-pro'); ?></label>
                                            </th>
                                            <td>
                                                <div class="aiox-input-group">
                                                    <input type="password" name="aiox_custom_api_key" id="aiox_custom_api_key" value="<?php echo esc_attr($custom_api_key); ?>" class="regular-text" placeholder="Your API Key" />
                                                    <button type="button" class="button aiox-test-api" data-provider="custom">
                                                        <span class="dashicons dashicons-yes" style="vertical-align: middle;"></span>
                                                        <?php _e('Test API', 'aiox-publisher-suite-pro'); ?>
                                                    </button>
                                                </div>
                                                <div class="aiox-api-status" id="custom-status"></div>
                                                <p class="description">
                                                    <a href="https://aioxsuite.com" target="_blank"><?php _e('Get your AIOX API', 'aiox-publisher-suite-pro'); ?> →</a>
                                                </p>
                                            </td>
                                        </tr>
                                    </table>
                                </div>
                            </div>
                            
                            <!-- Automatic Processing -->
                            <div class="postbox">
                                <h2 class="hndle"><span class="dashicons dashicons-update"></span> <?php _e('Automatic Processing', 'aiox-publisher-suite-pro'); ?></h2>
                                <div class="inside">
                                    <p class="description" style="margin-bottom: 15px;"><?php _e('Configure automatic AI processing when posts and pages are created or updated.', 'aiox-publisher-suite-pro'); ?></p>
                                    
                                    <ul class="aiox-toggle-list">
                                        <li class="aiox-toggle-item">
                                            <div class="toggle-info">
                                                <span class="toggle-label"><?php _e('Auto-Process on Save', 'aiox-publisher-suite-pro'); ?></span>
                                                <span class="toggle-desc"><?php _e('Process posts automatically when saved (recommended)', 'aiox-publisher-suite-pro'); ?></span>
                                            </div>
                                            <label class="aiox-switch">
                                                <input type="checkbox" name="aiox_auto_process_on_save" value="1" <?php checked($auto_process_on_save); ?> />
                                                <span class="slider"></span>
                                            </label>
                                        </li>
                                    </ul>
                                    
                                    <div class="notice notice-info inline" style="margin-top: 15px;">
                                        <p><strong><?php _e('Note:', 'aiox-publisher-suite-pro'); ?></strong> <?php _e('When enabled, all published posts and pages will be automatically processed with AI analysis, ingredients extraction, and capsule generation.', 'aiox-publisher-suite-pro'); ?></p>
                                    </div>
                                </div>
                            </div>
                            
                            <!-- Frontend Display Features -->
                            <div class="postbox">
                                <h2 class="hndle"><span class="dashicons dashicons-visibility"></span> <?php _e('Frontend Display', 'aiox-publisher-suite-pro'); ?></h2>
                                <div class="inside">
                                    <ul class="aiox-toggle-list">
                                        <li class="aiox-toggle-item">
                                            <div class="toggle-info">
                                                <span class="toggle-label"><?php _e('AI-Friendly Badge', 'aiox-publisher-suite-pro'); ?></span>
                                                <span class="toggle-desc"><?php _e('Display interactive AI badge with Q&A on frontend', 'aiox-publisher-suite-pro'); ?></span>
                                            </div>
                                            <label class="aiox-switch">
                                                <input type="checkbox" name="aiox_enable_ai_badge" value="1" <?php checked($enable_ai_badge); ?> />
                                                <span class="slider"></span>
                                            </label>
                                        </li>
                                    </ul>
                                    
                                    <table class="form-table" style="margin-top: 10px;">
                                        <tr>
                                            <th scope="row">
                                                <label for="aiox_badge_qa_count"><?php _e('Badge Q&A Count', 'aiox-publisher-suite-pro'); ?></label>
                                            </th>
                                            <td>
                                                <input type="number" name="aiox_badge_qa_count" id="aiox_badge_qa_count" value="<?php echo esc_attr($badge_qa_count); ?>" min="1" max="10" class="small-text" />
                                                <span class="description"><?php _e('Q&A pairs (1-10)', 'aiox-publisher-suite-pro'); ?></span>
                                            </td>
                                        </tr>
                                    </table>
                                </div>
                            </div>
                            
                            <!-- SEO & Structured Data -->
                            <div class="postbox">
                                <h2 class="hndle"><span class="dashicons dashicons-admin-site"></span> <?php _e('SEO & Structured Data', 'aiox-publisher-suite-pro'); ?></h2>
                                <div class="inside">
                                    <ul class="aiox-toggle-list">
                                        <li class="aiox-toggle-item">
                                            <div class="toggle-info">
                                                <span class="toggle-label"><?php _e('FAQPage Schema', 'aiox-publisher-suite-pro'); ?></span>
                                                <span class="toggle-desc"><?php _e('Add JSON-LD structured data for AI crawlers', 'aiox-publisher-suite-pro'); ?></span>
                                            </div>
                                            <label class="aiox-switch">
                                                <input type="checkbox" name="aiox_enable_faqpage_schema" value="1" <?php checked($enable_faqpage_schema); ?> />
                                                <span class="slider"></span>
                                            </label>
                                        </li>
                                        <li class="aiox-toggle-item">
                                            <div class="toggle-info">
                                                <span class="toggle-label"><?php _e('OpenGraph & Twitter Cards', 'aiox-publisher-suite-pro'); ?></span>
                                                <span class="toggle-desc"><?php _e('Social meta tags (auto-disabled if SEO plugin detected)', 'aiox-publisher-suite-pro'); ?></span>
                                            </div>
                                            <label class="aiox-switch">
                                                <input type="checkbox" name="aiox_enable_social_meta" value="1" <?php checked($enable_social_meta); ?> />
                                                <span class="slider"></span>
                                            </label>
                                        </li>
                                        <li class="aiox-toggle-item">
                                            <div class="toggle-info">
                                                <span class="toggle-label"><?php _e('Inline JSON Injection', 'aiox-publisher-suite-pro'); ?></span>
                                                <span class="toggle-desc"><?php _e('Embed AIOX data inline for crawler compatibility', 'aiox-publisher-suite-pro'); ?></span>
                                            </div>
                                            <label class="aiox-switch">
                                                <input type="checkbox" name="aiox_enable_inline_json" value="1" <?php checked($enable_inline_json); ?> />
                                                <span class="slider"></span>
                                            </label>
                                        </li>
                                    </ul>
                                </div>
                            </div>
                            
                            <!-- AIOX Ingredients System -->
                            <div class="postbox">
                                <h2 class="hndle"><span class="dashicons dashicons-editor-paste-text"></span> <?php _e('AIOX Ingredients System', 'aiox-publisher-suite-pro'); ?></h2>
                                <div class="inside">
                                    <p class="description" style="margin-bottom: 15px;"><?php _e('Enhanced content analysis and semantic extraction.', 'aiox-publisher-suite-pro'); ?></p>
                                    
                                    <ul class="aiox-toggle-list">
                                        <li class="aiox-toggle-item">
                                            <div class="toggle-info">
                                                <span class="toggle-label"><?php _e('Enable Ingredients System', 'aiox-publisher-suite-pro'); ?></span>
                                                <span class="toggle-desc"><?php _e('Break content into semantic building blocks', 'aiox-publisher-suite-pro'); ?></span>
                                            </div>
                                            <label class="aiox-switch">
                                                <input type="checkbox" name="aiox_enable_ingredients_system" value="1" <?php checked($enable_ingredients_system); ?> />
                                                <span class="slider"></span>
                                            </label>
                                        </li>
                                        <li class="aiox-toggle-item">
                                            <div class="toggle-info">
                                                <span class="toggle-label"><?php _e('Metadata Classification', 'aiox-publisher-suite-pro'); ?></span>
                                                <span class="toggle-desc"><?php _e('Auto-classify content intent, tone, and audience', 'aiox-publisher-suite-pro'); ?></span>
                                            </div>
                                            <label class="aiox-switch">
                                                <input type="checkbox" name="aiox_enable_metadata_classification" value="1" <?php checked($enable_metadata_classification); ?> />
                                                <span class="slider"></span>
                                            </label>
                                        </li>
                                        <li class="aiox-toggle-item">
                                            <div class="toggle-info">
                                                <span class="toggle-label"><?php _e('Legacy Q&A Compatibility', 'aiox-publisher-suite-pro'); ?></span>
                                                <span class="toggle-desc"><?php _e('Keep generating traditional Q&A pairs', 'aiox-publisher-suite-pro'); ?></span>
                                            </div>
                                            <label class="aiox-switch">
                                                <input type="checkbox" name="aiox_enable_legacy_qa_compatibility" value="1" <?php checked($enable_legacy_qa_compatibility); ?> />
                                                <span class="slider"></span>
                                            </label>
                                        </li>
                                    </ul>
                                    
                                    <table class="form-table" style="margin-top: 10px;">
                                        <tr>
                                            <th scope="row">
                                                <label for="aiox_ingredients_extraction_depth"><?php _e('Extraction Depth', 'aiox-publisher-suite-pro'); ?></label>
                                            </th>
                                            <td>
                                                <select name="aiox_ingredients_extraction_depth" id="aiox_ingredients_extraction_depth">
                                                    <option value="basic" <?php selected($ingredients_extraction_depth, 'basic'); ?>><?php _e('Basic (Fast)', 'aiox-publisher-suite-pro'); ?></option>
                                                    <option value="standard" <?php selected($ingredients_extraction_depth, 'standard'); ?>><?php _e('Standard (Balanced)', 'aiox-publisher-suite-pro'); ?></option>
                                                    <option value="deep" <?php selected($ingredients_extraction_depth, 'deep'); ?>><?php _e('Deep (Comprehensive)', 'aiox-publisher-suite-pro'); ?></option>
                                                </select>
                                            </td>
                                        </tr>
                                    </table>
                                </div>
                            </div>
                            
                            <!-- TDM & Licensing -->
                            <div class="postbox">
                                <h2 class="hndle"><span class="dashicons dashicons-shield"></span> <?php _e('Text and Data Mining (TDM)', 'aiox-publisher-suite-pro'); ?></h2>
                                <div class="inside">
                                    <ul class="aiox-toggle-list">
                                        <li class="aiox-toggle-item">
                                            <div class="toggle-info">
                                                <span class="toggle-label"><?php _e('Allow Training', 'aiox-publisher-suite-pro'); ?></span>
                                                <span class="toggle-desc"><?php _e('Permit AI training on your content', 'aiox-publisher-suite-pro'); ?></span>
                                            </div>
                                            <label class="aiox-switch">
                                                <input type="checkbox" name="aiox_tdm_training_allowed" value="1" <?php checked($tdm_training_allowed); ?> />
                                                <span class="slider"></span>
                                            </label>
                                        </li>
                                        <li class="aiox-toggle-item">
                                            <div class="toggle-info">
                                                <span class="toggle-label"><?php _e('Require Citation', 'aiox-publisher-suite-pro'); ?></span>
                                                <span class="toggle-desc"><?php _e('Mandate citation when content is referenced', 'aiox-publisher-suite-pro'); ?></span>
                                            </div>
                                            <label class="aiox-switch">
                                                <input type="checkbox" name="aiox_tdm_citation_required" value="1" <?php checked($tdm_citation_required); ?> />
                                                <span class="slider"></span>
                                            </label>
                                        </li>
                                    </ul>
                                    
                                    <table class="form-table" style="margin-top: 10px;">
                                        <tr>
                                            <th scope="row">
                                                <label for="aiox_attribution_string"><?php _e('Attribution String', 'aiox-publisher-suite-pro'); ?></label>
                                            </th>
                                            <td>
                                                <input type="text" name="aiox_attribution_string" id="aiox_attribution_string" value="<?php echo esc_attr($attribution_string); ?>" class="regular-text" />
                                                <p class="description"><?php _e('How you want to be credited in citations', 'aiox-publisher-suite-pro'); ?></p>
                                            </td>
                                        </tr>
                                    </table>
                                </div>
                            </div>
                            
                            <!-- AI Monetization Link -->
                            <div class="postbox" style="border-left: 4px solid #dba617;">
                                <div class="inside" style="display: flex; align-items: center; justify-content: space-between; flex-wrap: wrap; gap: 15px; padding: 20px;">
                                    <div>
                                        <h3 style="margin: 0 0 5px 0; display: flex; align-items: center; gap: 8px;">
                                            <span class="dashicons dashicons-money-alt" style="color: #dba617;"></span>
                                            <?php _e('AI Monetization & Licensing', 'aiox-publisher-suite-pro'); ?>
                                        </h3>
                                        <p class="description" style="margin: 0;"><?php _e('Control how AI systems can use, train on, and monetize your content.', 'aiox-publisher-suite-pro'); ?></p>
                                    </div>
                                    <a href="<?php echo admin_url('admin.php?page=aiox-monetization'); ?>" class="button button-primary">
                                        <?php _e('Configure Monetization', 'aiox-publisher-suite-pro'); ?> →
                                    </a>
                                </div>
                            </div>
                            
                            <!-- System Options -->
                            <div class="postbox">
                                <h2 class="hndle"><span class="dashicons dashicons-admin-generic"></span> <?php _e('System Options', 'aiox-publisher-suite-pro'); ?></h2>
                                <div class="inside">
                                    <table class="form-table">
                                        <tr>
                                            <th scope="row">
                                                <label for="aiox_processing_limit"><?php _e('Processing Limit', 'aiox-publisher-suite-pro'); ?></label>
                                            </th>
                                            <td>
                                                <input type="number" name="aiox_processing_limit" id="aiox_processing_limit" value="<?php echo esc_attr($processing_limit); ?>" min="1" max="100" class="small-text" />
                                                <span class="description"><?php _e('posts per batch', 'aiox-publisher-suite-pro'); ?></span>
                                            </td>
                                        </tr>
                                        <tr>
                                            <th scope="row">
                                                <label for="aiox_log_level"><?php _e('Log Level', 'aiox-publisher-suite-pro'); ?></label>
                                            </th>
                                            <td>
                                                <select name="aiox_log_level" id="aiox_log_level">
                                                    <option value="error" <?php selected($log_level, 'error'); ?>><?php _e('Error (Recommended)', 'aiox-publisher-suite-pro'); ?></option>
                                                    <option value="warning" <?php selected($log_level, 'warning'); ?>><?php _e('Warning', 'aiox-publisher-suite-pro'); ?></option>
                                                    <option value="info" <?php selected($log_level, 'info'); ?>><?php _e('Info (High Volume)', 'aiox-publisher-suite-pro'); ?></option>
                                                    <option value="debug" <?php selected($log_level, 'debug'); ?>><?php _e('Debug (Not Recommended)', 'aiox-publisher-suite-pro'); ?></option>
                                                </select>
                                                <p class="description" style="color: #d63638;"><?php _e('⚠️ Debug/Info levels generate many logs and impact performance.', 'aiox-publisher-suite-pro'); ?></p>
                                            </td>
                                        </tr>
                                    </table>
                                </div>
                            </div>
                            
                            <!-- Maintenance Tools -->
                            <div class="postbox">
                                <h2 class="hndle"><span class="dashicons dashicons-admin-tools"></span> <?php _e('Maintenance Tools', 'aiox-publisher-suite-pro'); ?></h2>
                                <div class="inside">
                                    <p class="description" style="margin-bottom: 15px;"><?php _e('Quick actions for plugin maintenance.', 'aiox-publisher-suite-pro'); ?></p>
                                    
                                    <div class="aiox-actions-row">
                                        <button type="button" id="aiox-autofix-files" class="button">
                                            <span class="dashicons dashicons-admin-tools" style="vertical-align: middle;"></span>
                                            <?php _e('AutoFix Files', 'aiox-publisher-suite-pro'); ?>
                                        </button>
                                        
                                        <button type="button" id="aiox-clear-logs" class="button">
                                            <span class="dashicons dashicons-trash" style="vertical-align: middle;"></span>
                                            <?php _e('Clear All Logs', 'aiox-publisher-suite-pro'); ?>
                                        </button>
                                        
                                        <button type="button" id="aiox-bulk-process" class="button">
                                            <span class="dashicons dashicons-update" style="vertical-align: middle;"></span>
                                            <?php _e('Process All Posts', 'aiox-publisher-suite-pro'); ?>
                                        </button>
                                    </div>
                                    
                                    <div id="maintenance-status"></div>
                                    <div id="bulk-process-status"></div>
                                </div>
                            </div>
                            
                            <!-- Danger Zone -->
                            <div class="postbox" style="border-left: 4px solid #d63638;">
                                <h2 class="hndle" style="color: #d63638;"><span class="dashicons dashicons-warning"></span> <?php _e('Danger Zone', 'aiox-publisher-suite-pro'); ?></h2>
                                <div class="inside">
                                    <p class="description" style="color: #d63638; margin-bottom: 15px;"><?php _e('These actions cannot be undone.', 'aiox-publisher-suite-pro'); ?></p>
                                    
                                    <button type="button" id="aiox-total-reset" class="button" style="color: #d63638; border-color: #d63638;">
                                        <span class="dashicons dashicons-trash" style="vertical-align: middle;"></span>
                                        <?php _e('Total Reset', 'aiox-publisher-suite-pro'); ?>
                                    </button>
                                    <p class="description"><?php _e('Remove ALL processing data. API keys and settings are preserved.', 'aiox-publisher-suite-pro'); ?></p>
                                    
                                    <div id="reset-status"></div>
                                </div>
                            </div>
                            
                            <!-- Save Button at bottom of main content -->
                            <div class="postbox">
                                <div class="inside" style="padding: 15px 20px;">
                                    <?php submit_button(__('Save Settings', 'aiox-publisher-suite-pro'), 'primary', 'submit', false); ?>
                                </div>
                            </div>
                            
                        </div><!-- /post-body-content -->
                        
                        <!-- Sidebar -->
                        <div id="postbox-container-1" class="postbox-container">
                            <div class="postbox">
                                <h2 class="hndle"><?php _e('Save Settings', 'aiox-publisher-suite-pro'); ?></h2>
                                <div class="inside">
                                    <?php submit_button(__('Save Settings', 'aiox-publisher-suite-pro'), 'primary', 'submit', false); ?>
                                </div>
                            </div>
                            
                            <div class="postbox">
                                <h2 class="hndle"><?php _e('Quick Links', 'aiox-publisher-suite-pro'); ?></h2>
                                <div class="inside">
                                    <ul style="margin: 0;">
                                        <li><a href="<?php echo admin_url('admin.php?page=aiox-dashboard'); ?>"><?php _e('Dashboard', 'aiox-publisher-suite-pro'); ?></a></li>
                                        <li><a href="<?php echo admin_url('admin.php?page=aiox-diagnostics'); ?>"><?php _e('Diagnostics', 'aiox-publisher-suite-pro'); ?></a></li>
                                        <li><a href="<?php echo admin_url('admin.php?page=aiox-modern-analytics'); ?>"><?php _e('Analytics', 'aiox-publisher-suite-pro'); ?></a></li>
                                        <li><a href="https://aioxsuite.com/docs" target="_blank"><?php _e('Documentation', 'aiox-publisher-suite-pro'); ?> ↗</a></li>
                                    </ul>
                                </div>
                            </div>
                        </div>
                        
                    </div><!-- /post-body -->
                </div><!-- /poststuff -->
            </form>
        </div>
        
        <?php
        $this->add_ai_provider_scripts();
    }
    
    /**
     * Add JavaScript for functionality
     */
    public function add_ai_provider_scripts() {
        ?>
        <script type="text/javascript">
        jQuery(document).ready(function($) {
            console.log('AIOX Settings JS: Document ready, jQuery version:', jQuery.fn.jquery);
            
            // API Testing
            $('.aiox-test-api').on('click', function() {
                var button = $(this);
                var provider = button.data('provider');
                var apiKey = $('input[name="aiox_custom_api_key"]').val();
                var endpoint = $('input[name="aiox_custom_endpoint"]').val();
                var statusDiv = $('#custom-status');
                
                if (!apiKey || apiKey.trim() === '') {
                    statusDiv.html('<span style="color: #dc3545;">Please enter an API key first</span>');
                    return;
                }
                
                if (!endpoint || endpoint.trim() === '') {
                    statusDiv.html('<span style="color: #dc3545;">Please enter an endpoint URL first</span>');
                    return;
                }
                
                button.prop('disabled', true).html('<span class="dashicons dashicons-update aiox-spin"></span> Testing...');
                statusDiv.html('<span style="color: #2575fc;">Testing connection...</span>');
                
                $.ajax({
                    url: ajaxurl,
                    type: 'POST',
                    data: {
                        action: 'aiox_test_ai_api',
                        provider: provider,
                        api_key: apiKey,
                        nonce: '<?php echo wp_create_nonce('aiox_test_ai_api'); ?>'
                    },
                    success: function(response) {
                        if (response.success) {
                            statusDiv.html('<span style="color: #28a745;">✓ ' + response.data.message + '</span>');
                        } else {
                            statusDiv.html('<span style="color: #dc3545;">✗ ' + (response.data.message || 'Connection failed') + '</span>');
                        }
                    },
                    error: function(xhr, status, error) {
                        statusDiv.html('<span style="color: #dc3545;">✗ Connection failed: ' + error + '</span>');
                    },
                    complete: function() {
                        button.prop('disabled', false).html('<span class="dashicons dashicons-yes"></span> Test API');
                    }
                });
            });
            
            // Clear Logs
            $('#aiox-clear-logs').on('click', function() {
                if (!confirm('Are you sure you want to clear all logs? This cannot be undone.')) {
                    return;
                }
                
                var button = $(this);
                var status = $('#maintenance-status');
                
                button.prop('disabled', true);
                status.html('<div class="notice notice-info"><p>Clearing logs...</p></div>').show();
                
                $.ajax({
                    url: ajaxurl,
                    type: 'POST',
                    data: {
                        action: 'aiox_clear_logs',
                        nonce: '<?php echo wp_create_nonce('aiox_clear_logs'); ?>'
                    },
                    success: function(response) {
                        if (response.success) {
                            status.html('<div class="notice notice-success"><p>✓ ' + response.data.message + '</p></div>');
                        } else {
                            status.html('<div class="notice notice-error"><p>✗ ' + (response.data.message || 'Failed to clear logs') + '</p></div>');
                        }
                    },
                    error: function() {
                        status.html('<div class="notice notice-error"><p>✗ Failed to clear logs</p></div>');
                    },
                    complete: function() {
                        button.prop('disabled', false);
                    }
                });
            });
            
            // AutoFix Files
            $('#aiox-autofix-files').on('click', function() {
                var button = $(this);
                var status = $('#maintenance-status');
                
                button.prop('disabled', true);
                status.html('<div class="notice notice-info"><p>Running AutoFix...</p></div>').show();
                
                $.ajax({
                    url: ajaxurl,
                    type: 'POST',
                    data: {
                        action: 'aiox_autofix_files',
                        nonce: '<?php echo wp_create_nonce('aiox_autofix_files'); ?>'
                    },
                    success: function(response) {
                        if (response.success) {
                            status.html('<div class="notice notice-success"><p>✓ ' + response.data.message + '</p></div>');
                        } else {
                            status.html('<div class="notice notice-error"><p>✗ ' + (response.data.message || 'AutoFix failed') + '</p></div>');
                        }
                    },
                    error: function() {
                        status.html('<div class="notice notice-error"><p>✗ AutoFix failed</p></div>');
                    },
                    complete: function() {
                        button.prop('disabled', false);
                    }
                });
            });
            
            // Bulk Processing with batch support
            console.log('AIOX: Bulk process handler registered');
            console.log('AIOX: Button found:', $('#aiox-bulk-process').length);
            
            $('#aiox-bulk-process').on('click', function(e) {
                e.preventDefault();
                console.log('AIOX: Bulk process button clicked');
                
                var button = $(this);
                var status = $('#bulk-process-status');
                
                console.log('AIOX: Status div found:', status.length);
                
                if (!confirm('⚠️ This will process ALL unprocessed posts and pages with AI.\n\nThe process will run in batches to prevent timeouts.\nThis may take several minutes and consume API credits.\n\nContinue?')) {
                    console.log('AIOX: User cancelled');
                    return;
                }
                
                console.log('AIOX: Starting bulk process...');
                button.prop('disabled', true).css('opacity', '0.6');
                status.html('<div class="notice notice-info"><p>⏳ Initializing bulk processing...</p></div>').show();
                
                var totalProcessed = 0;
                var totalFailed = 0;
                
                function processBatch(offset) {
                    $.ajax({
                        url: ajaxurl,
                        type: 'POST',
                        timeout: 120000, // 2 minute timeout per batch
                        data: {
                            action: 'aiox_bulk_process',
                            nonce: '<?php echo wp_create_nonce('aiox_bulk_process'); ?>',
                            offset: offset
                        },
                        success: function(response) {
                            console.log('Bulk process response:', response);
                            if (response.success) {
                                totalProcessed += response.data.batch_processed || 0;
                                totalFailed += response.data.batch_failed || 0;
                                
                                var progress = response.data.total > 0 ? Math.round((response.data.processed / response.data.total) * 100) : 100;
                                
                                if (response.data.complete) {
                                    // All done!
                                    var resultClass = totalFailed > 0 ? 'notice-warning' : 'notice-success';
                                    var resultIcon = totalFailed > 0 ? '⚠️' : '✓';
                                    status.html('<div class="notice ' + resultClass + '"><p>' + resultIcon + ' Bulk processing complete!<br>Total processed: ' + totalProcessed + ', Failed: ' + totalFailed + '</p></div>');
                                    button.prop('disabled', false).css('opacity', '1').html('<span class="dashicons dashicons-update"></span> <div class="aiox-action-text"><strong>Process All Existing Posts</strong><small>Bulk process all published content</small></div>');
                                } else {
                                    // More batches to process
                                    status.html('<div class="notice notice-info"><p>⏳ Progress: ' + progress + '% (' + response.data.processed + '/' + response.data.total + ')<br>Processed: ' + totalProcessed + ', Failed: ' + totalFailed + '</p></div>');
                                    
                                    // Process next batch after short delay
                                    setTimeout(function() {
                                        processBatch(response.data.next_offset);
                                    }, 1000);
                                }
                            } else {
                                console.error('Bulk process error:', response);
                                status.html('<div class="notice notice-error"><p>✗ ' + (response.data && response.data.message ? response.data.message : 'Bulk processing failed') + '</p></div>');
                                button.prop('disabled', false).css('opacity', '1').html('<span class="dashicons dashicons-update"></span> <div class="aiox-action-text"><strong>Process All Existing Posts</strong><small>Bulk process all published content</small></div>');
                            }
                        },
                        error: function(xhr, textStatus, error) {
                            console.error('AJAX error:', textStatus, error, xhr.responseText);
                            var errorMsg = error || textStatus || 'Connection failed';
                            if (xhr.responseText) {
                                try {
                                    var resp = JSON.parse(xhr.responseText);
                                    if (resp.data && resp.data.message) {
                                        errorMsg = resp.data.message;
                                    }
                                } catch(e) {
                                    // If response isn't JSON, show first 200 chars
                                    if (xhr.responseText.length > 0) {
                                        errorMsg = xhr.responseText.substring(0, 200);
                                    }
                                }
                            }
                            status.html('<div class="notice notice-error"><p>✗ Bulk processing failed: ' + errorMsg + '</p></div>');
                            button.prop('disabled', false).css('opacity', '1').html('<span class="dashicons dashicons-update"></span> <div class="aiox-action-text"><strong>Process All Existing Posts</strong><small>Bulk process all published content</small></div>');
                        }
                    });
                }
                
                // Start processing from offset 0
                processBatch(0);
            });
            
            // Total Reset
            $('#aiox-total-reset').on('click', function() {
                var button = $(this);
                var status = $('#reset-status');
                
                // Double confirmation for destructive action
                if (!confirm('⚠️ DANGER: This will permanently delete ALL AIOX processing data!\n\nThis includes:\n• All processed post/page data\n• All generated files\n• Database tables (queue, capsules, logs)\n\nAPI keys and settings will be preserved.\n\nThis action CANNOT be undone!\n\nAre you absolutely sure?')) {
                    return;
                }
                
                // Second confirmation
                var confirmText = prompt('To confirm, type "RESET" (all caps):');
                if (confirmText !== 'RESET') {
                    status.html('<div class="notice notice-info"><p>Reset cancelled.</p></div>').show();
                    return;
                }
                
                button.prop('disabled', true);
                status.html('<div class="notice notice-warning"><p>⏳ Performing total reset... Please wait...</p></div>').show();
                
                $.ajax({
                    url: ajaxurl,
                    type: 'POST',
                    data: {
                        action: 'aiox_total_reset',
                        nonce: '<?php echo wp_create_nonce('aiox_total_reset'); ?>'
                    },
                    success: function(response) {
                        if (response.success) {
                            status.html('<div class="notice notice-success"><p>✓ ' + response.data.message + '</p></div>');
                            // Reload page after 2 seconds to reflect changes
                            setTimeout(function() {
                                location.reload();
                            }, 2000);
                        } else {
                            status.html('<div class="notice notice-error"><p>✗ ' + (response.data.message || 'Total reset failed') + '</p></div>');
                        }
                    },
                    error: function() {
                        status.html('<div class="notice notice-error"><p>✗ Total reset failed</p></div>');
                    },
                    complete: function() {
                        button.prop('disabled', false);
                    }
                });
            });
        });
        </script>
        <?php
    }
    
    /**
     * Enqueue scripts and styles for settings page
     */
    public function enqueue_settings_scripts($hook) {
        if (strpos($hook, 'aiox-settings') === false && $hook !== 'aiox_page_aiox-settings') {
            return;
        }
        wp_enqueue_script('jquery');
        
        // Enqueue settings-specific native styles
        wp_enqueue_style(
            'aiox-settings-native',
            AIOX_PLUGIN_URL . 'assets/css/settings-native.css',
            array(),
            AIOX_VERSION
        );
    }
    
    /**
     * AJAX: Test AI connection
     */
    public function ajax_test_ai_connection() {
        check_ajax_referer('aiox_test_connection', 'nonce');
        
        if (!current_user_can('manage_options')) {
            wp_die(__('Insufficient permissions', 'aiox-publisher-suite-pro'));
        }
        
        $endpoint = sanitize_url($_POST['endpoint']);
        
        if (empty($endpoint)) {
            wp_send_json_error(array('message' => __('Please enter an AI endpoint URL', 'aiox-publisher-suite-pro')));
        }
        
        $response = wp_remote_post($endpoint, array(
            'timeout' => 10,
            'headers' => array('Content-Type' => 'application/json'),
            'body' => json_encode(array('test' => true, 'content' => 'Test connection'))
        ));
        
        if (is_wp_error($response)) {
            wp_send_json_error(array('message' => sprintf(__('Connection failed: %s', 'aiox-publisher-suite-pro'), $response->get_error_message())));
        }
        
        $status_code = wp_remote_retrieve_response_code($response);
        
        if ($status_code === 200) {
            wp_send_json_success(array('message' => __('Connection successful!', 'aiox-publisher-suite-pro')));
        } else {
            wp_send_json_error(array('message' => sprintf(__('Connection failed with status code: %d', 'aiox-publisher-suite-pro'), $status_code)));
        }
    }
    
    /**
     * AJAX: Validate license
     */
    public function ajax_validate_license() {
        check_ajax_referer('aiox_validate_license', 'nonce');
        
        if (!current_user_can('manage_options')) {
            wp_die(__('Insufficient permissions', 'aiox-publisher-suite-pro'));
        }
        
        $license_key = sanitize_text_field($_POST['license_key']);
        
        if (empty($license_key)) {
            wp_send_json_error(array('message' => __('Please enter a license key', 'aiox-publisher-suite-pro')));
        }
        
        $validation_endpoint = 'https://api.aiox.com/v1/validate-license';
        
        $response = wp_remote_post($validation_endpoint, array(
            'timeout' => 10,
            'headers' => array('Content-Type' => 'application/json'),
            'body' => json_encode(array(
                'license_key' => $license_key,
                'domain' => home_url()
            ))
        ));
        
        if (is_wp_error($response)) {
            wp_send_json_error(array('message' => sprintf(__('Validation failed: %s', 'aiox-publisher-suite-pro'), $response->get_error_message())));
        }
        
        $body = wp_remote_retrieve_body($response);
        $data = json_decode($body, true);
        
        if (isset($data['valid']) && $data['valid']) {
            wp_send_json_success(array('message' => __('License is valid!', 'aiox-publisher-suite-pro')));
        } else {
            wp_send_json_error(array('message' => __('Invalid license key', 'aiox-publisher-suite-pro')));
        }
    }
    
    /**
     * AJAX: Clear logs
     */
    public function ajax_clear_logs() {
        check_ajax_referer('aiox_clear_logs', 'nonce');
        
        if (!current_user_can('manage_options')) {
            wp_die(__('Insufficient permissions', 'aiox-publisher-suite-pro'));
        }
        
        global $wpdb;
        $deleted = $wpdb->query("DELETE FROM {$wpdb->prefix}aiox_logs");
        
        wp_send_json_success(array('message' => sprintf(__('Cleared %d log entries.', 'aiox-publisher-suite-pro'), $deleted)));
    }
    
    /**
     * AJAX: AutoFix files
     */
    public function ajax_autofix_files() {
        check_ajax_referer('aiox_autofix_files', 'nonce');
        
        if (!current_user_can('manage_options')) {
            wp_die(__('Insufficient permissions', 'aiox-publisher-suite-pro'));
        }
        
        $messages = array();
        
        $well_known_dir = ABSPATH . '.well-known';
        if (!file_exists($well_known_dir)) {
            if (wp_mkdir_p($well_known_dir)) {
                $messages[] = __('Created .well-known directory', 'aiox-publisher-suite-pro');
            } else {
                wp_send_json_error(array('message' => __('Failed to create .well-known directory', 'aiox-publisher-suite-pro')));
            }
        }
        
        if (chmod($well_known_dir, 0755)) {
            $messages[] = __('Set .well-known directory permissions', 'aiox-publisher-suite-pro');
        }
        
        $htaccess_content = "# AIOX Publisher Suite Pro - .well-known files\n";
        $htaccess_content .= "<Files \"*.json\">\n    Header set Content-Type \"application/json\"\n    Header set Access-Control-Allow-Origin \"*\"\n</Files>\n";
        $htaccess_content .= "<Files \"*.jsonc\">\n    Header set Content-Type \"application/json\"\n    Header set Access-Control-Allow-Origin \"*\"\n</Files>\n";
        $htaccess_content .= "<Files \"*.ndjson\">\n    Header set Content-Type \"application/x-ndjson\"\n    Header set Access-Control-Allow-Origin \"*\"\n</Files>\n";
        $htaccess_content .= "<Files \"*.pem\">\n    Header set Content-Type \"application/x-pem-file\"\n    Header set Access-Control-Allow-Origin \"*\"\n</Files>\n";
        
        $htaccess_file = $well_known_dir . '/.htaccess';
        if (file_put_contents($htaccess_file, $htaccess_content)) {
            $messages[] = __('Created .htaccess rules for .well-known files', 'aiox-publisher-suite-pro');
        }
        
        wp_send_json_success(array('message' => implode('. ', $messages)));
    }
    
    /**
     * AJAX: Total Reset - Remove ALL processing data
     * Preserves: API keys, plugin settings
     * Removes: All post meta, generated files, database tables
     */
    public function ajax_total_reset() {
        check_ajax_referer('aiox_total_reset', 'nonce');
        
        if (!current_user_can('manage_options')) {
            wp_send_json_error(array('message' => __('Insufficient permissions', 'aiox-publisher-suite-pro')));
            return;
        }
        
        global $wpdb;
        $results = array();
        
        try {
            // 1. Delete ALL _aiox_* post meta (using wildcard)
            $meta_deleted = $wpdb->query(
                "DELETE FROM {$wpdb->postmeta} WHERE meta_key LIKE '_aiox_%'"
            );
            $results[] = sprintf(__('Deleted %d post meta entries', 'aiox-publisher-suite-pro'), $meta_deleted);
            
            // 2. Delete generated files from .well-known
            $well_known_dir = ABSPATH . '.well-known/';
            $files_deleted = 0;
            
            if (is_dir($well_known_dir)) {
                // Main AIOX files
                $aiox_files = array(
                    'aio.json', 'aiox.ndjson', 'aiox.aiox', 'tdmrep.json',
                    'ingredients.json', 'aiox-pubkey.pem', 'aiox-proof.json',
                    'aiox-pack-index.json', '.htaccess'
                );
                foreach ($aiox_files as $file) {
                    if (file_exists($well_known_dir . $file)) {
                        @unlink($well_known_dir . $file);
                        $files_deleted++;
                    }
                }
                
                // AI info files
                $ai_info_files = glob($well_known_dir . 'ai-info-*.json') ?: array();
                foreach ($ai_info_files as $file) {
                    @unlink($file);
                    $files_deleted++;
                }
                
                // Capsule files
                $capsule_files = glob($well_known_dir . 'capsule_*.json') ?: array();
                foreach ($capsule_files as $file) {
                    @unlink($file);
                    $files_deleted++;
                }
                
                // Posts directory
                $posts_dir = $well_known_dir . 'posts/';
                if (is_dir($posts_dir)) {
                    $post_files = glob($posts_dir . '*.json') ?: array();
                    foreach ($post_files as $file) {
                        @unlink($file);
                        $files_deleted++;
                    }
                    @rmdir($posts_dir);
                }
                
                // Capsules directory
                $capsules_dir = $well_known_dir . 'capsules/';
                if (is_dir($capsules_dir)) {
                    $capsule_dir_files = glob($capsules_dir . '*.json') ?: array();
                    foreach ($capsule_dir_files as $file) {
                        @unlink($file);
                        $files_deleted++;
                    }
                    @rmdir($capsules_dir);
                }
                
                // Packs directory
                $packs_dir = $well_known_dir . 'packs/';
                if (is_dir($packs_dir)) {
                    $pack_files = glob($packs_dir . '*.aiox') ?: array();
                    foreach ($pack_files as $file) {
                        @unlink($file);
                        $files_deleted++;
                    }
                    @rmdir($packs_dir);
                }
            }
            
            // Root aio.json
            if (file_exists(ABSPATH . 'aio.json')) {
                @unlink(ABSPATH . 'aio.json');
                $files_deleted++;
            }
            
            $results[] = sprintf(__('Deleted %d generated files', 'aiox-publisher-suite-pro'), $files_deleted);
            
            // 3. Truncate/Drop database tables (but keep tables structure for queue)
            // Delete capsules table data
            if ($wpdb->get_var("SHOW TABLES LIKE '{$wpdb->prefix}aiox_capsules'")) {
                $wpdb->query("TRUNCATE TABLE {$wpdb->prefix}aiox_capsules");
                $results[] = __('Cleared capsules table', 'aiox-publisher-suite-pro');
            }
            
            // Delete queue table data
            if ($wpdb->get_var("SHOW TABLES LIKE '{$wpdb->prefix}aiox_queue'")) {
                $wpdb->query("TRUNCATE TABLE {$wpdb->prefix}aiox_queue");
                $results[] = __('Cleared queue table', 'aiox-publisher-suite-pro');
            }
            
            // Delete logs table data
            if ($wpdb->get_var("SHOW TABLES LIKE '{$wpdb->prefix}aiox_logs'")) {
                $wpdb->query("TRUNCATE TABLE {$wpdb->prefix}aiox_logs");
                $results[] = __('Cleared logs table', 'aiox-publisher-suite-pro');
            }
            
            // 4. Delete uploads directory files
            $upload_dir = wp_upload_dir();
            $aiox_uploads = $upload_dir['basedir'] . '/aiox/';
            if (is_dir($aiox_uploads)) {
                $upload_files = glob($aiox_uploads . '*') ?: array();
                foreach ($upload_files as $file) {
                    if (is_file($file)) {
                        @unlink($file);
                    }
                }
                @rmdir($aiox_uploads);
            }
            
            // 5. Clear WordPress cache
            wp_cache_flush();
            
            // Log the reset
            if (class_exists('AIOX_Logger')) {
                AIOX_Logger::info('Total Reset performed', array(
                    'user' => wp_get_current_user()->user_login,
                    'results' => $results
                ));
            }
            
            wp_send_json_success(array(
                'message' => __('Total Reset complete! All processing data has been removed.', 'aiox-publisher-suite-pro') . ' ' . implode('. ', $results)
            ));
            
        } catch (Exception $e) {
            wp_send_json_error(array(
                'message' => sprintf(__('Reset failed: %s', 'aiox-publisher-suite-pro'), $e->getMessage())
            ));
        }
    }
    
    /**
     * AJAX: Test AI API connection
     */
    public function ajax_test_ai_api() {
        
        if (!isset($_POST['nonce']) || !wp_verify_nonce($_POST['nonce'], 'aiox_test_ai_api')) {
            error_log('AIOX API Test: Nonce verification failed');
            wp_send_json_error(array('message' => __('Security check failed', 'aiox-publisher-suite-pro')));
        }
        
        if (!current_user_can('manage_options')) {
            error_log('AIOX API Test: Insufficient permissions');
            wp_send_json_error(array('message' => __('Insufficient permissions', 'aiox-publisher-suite-pro')));
        }
        
        $provider = sanitize_text_field($_POST['provider']);
        $api_key = sanitize_text_field($_POST['api_key']);
        
        
        if (empty($api_key)) {
            wp_send_json_error(array('message' => __('Please enter an API key', 'aiox-publisher-suite-pro')));
        }
        
        if ($provider === 'custom') {
            $custom_endpoint = get_option('aiox_custom_endpoint', '');
            
            if (empty($custom_endpoint)) {
                wp_send_json_error(array('message' => __('Please enter a custom endpoint URL', 'aiox-publisher-suite-pro')));
            }
            
            $response = wp_remote_post($custom_endpoint, array(
                'timeout' => 10,
                'headers' => array(
                    'Authorization' => 'Bearer ' . $api_key,
                    'Content-Type' => 'application/json',
                ),
                'body' => json_encode(array(
                    'contents' => array(
                        array(
                            'parts' => array(
                                array('text' => 'Test connection')
                            )
                        )
                    ),
                    'generationConfig' => array(
                        'maxOutputTokens' => 20
                    )
                ))
            ));
        } else {
            wp_send_json_error(array('message' => __('Invalid provider', 'aiox-publisher-suite-pro')));
        }
        
        if (is_wp_error($response)) {
            $error_msg = $response->get_error_message();
            error_log('AIOX API Test: WP Error - ' . $error_msg);
            wp_send_json_error(array('message' => sprintf(__('Connection failed: %s', 'aiox-publisher-suite-pro'), $error_msg)));
        }
        
        $status_code = wp_remote_retrieve_response_code($response);
        $body = wp_remote_retrieve_body($response);
        
        
        if ($status_code === 200) {
            wp_send_json_success(array('message' => __('API connection successful!', 'aiox-publisher-suite-pro')));
        } else {
            $error_data = json_decode($body, true);
            $error_message = isset($error_data['error']['message']) ? $error_data['error']['message'] : __('Unknown error', 'aiox-publisher-suite-pro');
            error_log('AIOX API Test: Error - ' . $error_message);
            wp_send_json_error(array('message' => sprintf(__('API Error (%d): %s', 'aiox-publisher-suite-pro'), $status_code, $error_message)));
        }
    }
    
    /**
     * Handle bulk processing AJAX request
     */
    public function ajax_bulk_process() {
        check_ajax_referer('aiox_bulk_process', 'nonce');
        
        if (!current_user_can('manage_options')) {
            wp_send_json_error(array('message' => 'Insufficient permissions'));
            return;
        }
        
        // Get processing limit from settings
        $batch_size = intval(get_option('aiox_processing_limit', 10));
        if ($batch_size < 1) $batch_size = 1;
        if ($batch_size > 50) $batch_size = 50;
        
        // Get offset from request for batch processing
        $offset = isset($_POST['offset']) ? intval($_POST['offset']) : 0;
        
        // Count total unprocessed posts
        $total_unprocessed = new WP_Query(array(
            'post_type' => array('post', 'page'),
            'post_status' => 'publish',
            'posts_per_page' => -1,
            'fields' => 'ids',
            'meta_query' => array(
                'relation' => 'OR',
                array(
                    'key' => '_aiox_processing_complete',
                    'compare' => 'NOT EXISTS'
                ),
                array(
                    'key' => '_aiox_processing_complete',
                    'value' => '1',
                    'compare' => '!='
                )
            )
        ));
        $total_count = $total_unprocessed->found_posts;
        
        if ($total_count === 0) {
            wp_send_json_success(array(
                'message' => 'No unprocessed posts found. All content is already processed!',
                'complete' => true,
                'processed' => 0,
                'total' => 0
            ));
            return;
        }
        
        // Get batch of unprocessed posts
        $posts = get_posts(array(
            'post_type' => array('post', 'page'),
            'post_status' => 'publish',
            'numberposts' => $batch_size,
            'offset' => $offset,
            'orderby' => 'ID',
            'order' => 'ASC',
            'meta_query' => array(
                'relation' => 'OR',
                array(
                    'key' => '_aiox_processing_complete',
                    'compare' => 'NOT EXISTS'
                ),
                array(
                    'key' => '_aiox_processing_complete',
                    'value' => '1',
                    'compare' => '!='
                )
            )
        ));
        
        if (empty($posts)) {
            wp_send_json_success(array(
                'message' => 'Bulk processing complete!',
                'complete' => true,
                'processed' => $offset,
                'total' => $total_count
            ));
            return;
        }
        
        $processed = 0;
        $failed = 0;
        $errors = array();
        
        // Check if comprehensive processor is available
        if (!class_exists('AIOX_Comprehensive_Processor')) {
            wp_send_json_error(array('message' => 'AIOX Comprehensive Processor not available'));
            return;
        }
        
        $comprehensive_processor = new AIOX_Comprehensive_Processor();
        
        foreach ($posts as $post) {
            try {
                $result = $comprehensive_processor->process_post_completely($post->ID);
                if ($result && is_array($result) && isset($result['success']) && $result['success']) {
                    $processed++;
                } else {
                    $failed++;
                    $error_msg = isset($result['message']) ? $result['message'] : 'Unknown error';
                    $errors[] = $post->post_title . ': ' . $error_msg;
                }
                // Small delay to prevent server overload
                usleep(250000); // 0.25 seconds
            } catch (Throwable $e) {
                $failed++;
                $errors[] = $post->post_title . ': ' . $e->getMessage();
                AIOX_Logger::error('Bulk processing failed for post ' . $post->ID, array(
                    'error' => $e->getMessage(),
                    'trace' => $e->getTraceAsString()
                ));
            }
        }
        
        $new_offset = $offset + count($posts);
        $is_complete = $new_offset >= $total_count;
        
        $message = sprintf(
            'Batch %d-%d of %d: Processed %d, Failed %d',
            $offset + 1,
            $new_offset,
            $total_count,
            $processed,
            $failed
        );
        
        wp_send_json_success(array(
            'message' => $message,
            'complete' => $is_complete,
            'processed' => $new_offset,
            'total' => $total_count,
            'batch_processed' => $processed,
            'batch_failed' => $failed,
            'next_offset' => $new_offset,
            'errors' => array_slice($errors, 0, 5) // Only return first 5 errors
        ));
    }
}