<?php
/**
 * AIOX AI-Only Diagnostic Tests
 * 
 * Additional diagnostic tests for Version 2.0.1+ AI-only functionality
 */

if (!defined('ABSPATH')) {
    exit;
}

class AIOX_AI_Only_Diagnostics {
    
    /**
     * Run AI-only specific tests
     */
    public static function run_ai_only_tests() {
        $tests = array(
            'api_key_validation' => self::test_api_key_validation(),
            'provider_configuration' => self::test_provider_configuration(),
            'connection_test' => self::test_api_connection(),
            'processing_capability' => self::test_processing_capability(),
            'basic_mode_disabled' => self::test_basic_mode_disabled(),
            'upgrade_status' => self::test_upgrade_status()
        );
        
        return $tests;
    }
    
    /**
     * Test API key validation
     */
    private static function test_api_key_validation() {
        require_once AIOX_PLUGIN_DIR . 'core/class-aiox-api-validator.php';
        
        $validation = AIOX_API_Validator::validate_current_provider();
        
        if ($validation['valid']) {
            return array(
                'status' => 'pass',
                'message' => 'API key validation passed: ' . $validation['message']
            );
        } else {
            return array(
                'status' => 'fail',
                'message' => 'API key validation failed: ' . $validation['message']
            );
        }
    }
    
    /**
     * Test provider configuration
     */
    private static function test_provider_configuration() {
        $provider = get_option('aiox_ai_provider', 'openai');
        
        if ($provider === 'basic') {
            return array(
                'status' => 'fail',
                'message' => 'Basic mode is no longer supported. Please select an AI provider.'
            );
        }
        
        $supported_providers = array('openai', 'gemini');
        if (!in_array($provider, $supported_providers)) {
            return array(
                'status' => 'fail',
                'message' => 'Unknown provider selected: ' . $provider
            );
        }
        
        return array(
            'status' => 'pass',
            'message' => 'Provider configured: ' . ucfirst($provider)
        );
    }
    
    /**
     * Test API connection
     */
    private static function test_api_connection() {
        require_once AIOX_PLUGIN_DIR . 'core/class-aiox-api-validator.php';
        
        try {
            $connection_test = AIOX_API_Validator::test_current_provider_connection();
            
            if ($connection_test['success'] ?? false) {
                return array(
                    'status' => 'pass',
                    'message' => 'API connection successful: ' . $connection_test['message']
                );
            } else {
                return array(
                    'status' => 'fail',
                    'message' => 'API connection failed: ' . $connection_test['message']
                );
            }
        } catch (Exception $e) {
            return array(
                'status' => 'fail',
                'message' => 'Connection test error: ' . $e->getMessage()
            );
        }
    }
    
    /**
     * Test processing capability
     */
    private static function test_processing_capability() {
        require_once AIOX_PLUGIN_DIR . 'core/class-aiox-ai-only-processor.php';
        
        try {
            $processor = new AIOX_AI_Only_Processor();
            
            if ($processor->is_ai_available()) {
                $stats = $processor->get_processing_stats();
                return array(
                    'status' => 'pass',
                    'message' => 'AI processing available with ' . $stats['provider']
                );
            } else {
                return array(
                    'status' => 'fail',
                    'message' => 'AI processing not available - check configuration'
                );
            }
        } catch (Exception $e) {
            return array(
                'status' => 'fail',
                'message' => 'Processing capability test failed: ' . $e->getMessage()
            );
        }
    }
    
    /**
     * Test that basic mode is properly disabled
     */
    private static function test_basic_mode_disabled() {
        $provider = get_option('aiox_ai_provider', 'openai');
        
        if ($provider === 'basic') {
            return array(
                'status' => 'fail',
                'message' => 'Basic mode is still enabled - should be disabled in Version 2.0.1+'
            );
        }
        
        // Check if basic processing is accessible
        if (class_exists('AIOX_AI_Processor')) {
            // This is expected - the old class may still exist but shouldn't be used
            return array(
                'status' => 'pass',
                'message' => 'Basic mode disabled, AI-only processing enforced'
            );
        }
        
        return array(
            'status' => 'pass',
            'message' => 'Basic mode properly disabled'
        );
    }
    
    /**
     * Test upgrade status
     */
    private static function test_upgrade_status() {
        $version = get_option('aiox_version', '2.0.0');
        
        if (version_compare($version, '2.0.1', '>=')) {
            return array(
                'status' => 'pass',
                'message' => 'Plugin upgraded to AI-only version: ' . $version
            );
        } else {
            return array(
                'status' => 'warning',
                'message' => 'Version may need update: ' . $version . ' (expected 2.0.1+)'
            );
        }
    }
    
    /**
     * Get AI-only requirements summary
     */
    public static function get_requirements_summary() {
        require_once AIOX_PLUGIN_DIR . 'core/class-aiox-ai-only-processor.php';
        
        $requirements = AIOX_AI_Only_Processor::get_processing_requirements();
        $providers = AIOX_AI_Only_Processor::get_supported_providers();
        
        return array(
            'version' => '2.0.1+',
            'mode' => 'AI-Only Processing',
            'api_key_required' => $requirements['api_key_required'],
            'supported_providers' => array_keys($providers),
            'fallback_available' => $requirements['fallback_available'],
            'basic_mode_supported' => $requirements['basic_mode_supported'],
            'content_limits' => array(
                'minimum' => $requirements['minimum_content_length'] . ' characters',
                'maximum' => number_format($requirements['maximum_content_length']) . ' characters'
            )
        );
    }
}
