<?php
/**
 * AIOX Diagnostics Enhanced
 * Extended system self-tests and analytics checks.
 */

if (!defined('ABSPATH')) {
    exit;
}

class AIOX_Diagnostics_Enhanced {

    public function __construct() {
        add_action('admin_menu', [$this, 'register_diagnostics_page']);
    }

    public function register_diagnostics_page() {
        add_submenu_page(
            'aiox-main',
            'Diagnostics',
            'Diagnostics',
            'manage_options',
            'aiox-diagnostics-enhanced',
            [$this, 'render_diagnostics_page']
        );
    }

    public function render_diagnostics_page() {
        echo '<div class="wrap aiox-diagnostics">';
        echo '<h1>AIOX Diagnostics &amp; System Check</h1>';
        echo '<p>Performing system self-tests to verify plugin health, analytics functionality, and crawler tracking...</p>';

        $results = [
            'database'  => $this->test_analytics_database(),
            'tracking'  => $this->test_request_tracking(),
            'endpoints' => $this->test_endpoints(),
        ];

        echo '<style>
            .aiox-diagnostics table { border-collapse: collapse; width: 100%; margin-top: 1em; }
            .aiox-diagnostics th, .aiox-diagnostics td { padding: 8px 10px; border-bottom: 1px solid #ddd; text-align: left; }
            .aiox-diagnostics th { background: #f3f3f3; font-weight: 600; }
            .aiox-diagnostics .pass { color: #2e7d32; font-weight: bold; }
            .aiox-diagnostics .fail { color: #c62828; font-weight: bold; }
            .aiox-diagnostics .notice { color: #f9a825; font-weight: bold; }
            .aiox-diagnostics-section { margin-top: 2em; }
            .aiox-diagnostics-section h2 { margin-bottom: 0.5em; }
            .aiox-diagnostics pre { background: #fafafa; border: 1px solid #eee; padding: 10px; overflow: auto; }
        </style>';

        echo '<div class="aiox-diagnostics-section">';
        echo '<h2>Core System Tests</h2>';
        echo '<table class="widefat fixed striped">';
        echo '<thead><tr><th>Test</th><th>Status</th><th>Message</th></tr></thead><tbody>';

        foreach ($results as $test => $result) {
            $status_class = $result['status'] === 'pass' ? 'pass' : ($result['status'] === 'fail' ? 'fail' : 'notice');
            echo '<tr>';
            echo '<td>' . esc_html(ucfirst($test)) . '</td>';
            echo '<td class="' . esc_attr($status_class) . '">' . strtoupper(esc_html($result['status'])) . '</td>';
            echo '<td>' . esc_html($result['message']) . '</td>';
            echo '</tr>';
        }

        echo '</tbody></table>';
        echo '</div>';

        echo '<div class="aiox-diagnostics-section">';
        echo '<h2>Environment Information</h2>';
        echo '<table class="widefat fixed striped">';
        echo '<thead><tr><th>Parameter</th><th>Value</th></tr></thead><tbody>';
        echo '<tr><td>WordPress Version</td><td>' . esc_html(get_bloginfo('version')) . '</td></tr>';
        echo '<tr><td>PHP Version</td><td>' . esc_html(PHP_VERSION) . '</td></tr>';
        echo '<tr><td>MySQL Version</td><td>' . esc_html($this->get_mysql_version()) . '</td></tr>';
        echo '<tr><td>Server Software</td><td>' . esc_html($_SERVER['SERVER_SOFTWARE'] ?? 'N/A') . '</td></tr>';
        echo '<tr><td>Site URL</td><td>' . esc_html(get_site_url()) . '</td></tr>';
        echo '<tr><td>Home URL</td><td>' . esc_html(get_home_url()) . '</td></tr>';
        echo '</tbody></table>';
        echo '</div>';

        echo '<div class="aiox-diagnostics-section">';
        echo '<h2>Recent Activity</h2>';
        echo '<p>This data shows recent analytics and crawler interactions recorded by the AIOX system.</p>';
        echo '<pre>' . esc_html($this->get_recent_activity_sample()) . '</pre>';
        echo '</div>';

        echo '</div>';
    }

    /**
     * Check analytics database table existence and content
     */
    private function test_analytics_database() {
        global $wpdb;
        $analytics_table = $wpdb->prefix . 'aiox_analytics';

        if ($wpdb->get_var("SHOW TABLES LIKE '$analytics_table'") === $analytics_table) {
            $count = (int) $wpdb->get_var("SELECT COUNT(*) FROM $analytics_table");
            return [
                'status'  => 'pass',
                'message' => 'Analytics table found with ' . $count . ' records'
            ];
        } else {
            return [
                'status'  => 'fail',
                'message' => 'Analytics table not found. Logging may be disabled or not yet initialized.'
            ];
        }
    }

    /**
     * Test tracking of crawler/badge requests
     */
    private function test_request_tracking() {
        global $wpdb;
        $analytics_table = $wpdb->prefix . 'aiox_analytics';

        $recent_count = (int) $wpdb->get_var($wpdb->prepare(
            "SELECT COUNT(*) FROM $analytics_table WHERE timestamp > %s",
            date('Y-m-d H:i:s', strtotime('-24 hours'))
        ));

        return [
            'status'  => 'pass',
            'message' => 'Logged ' . $recent_count . ' analytics events in the last 24 hours.'
        ];
    }

    /**
     * Test endpoint accessibility
     */
    private function test_endpoints() {
        $tests = [
            'Ping'      => admin_url('admin-ajax.php?action=aiox_ping'),
            'Analytics' => admin_url('admin-ajax.php?action=aiox_log_badge'),
        ];

        $messages = [];
        foreach ($tests as $name => $url) {
            $response = wp_remote_get($url, ['timeout' => 3]);
            if (is_wp_error($response)) {
                $messages[] = "$name endpoint unreachable";
            } else {
                $messages[] = "$name endpoint OK";
            }
        }

        $all_ok = !preg_grep('/unreachable/', $messages);
        return [
            'status'  => $all_ok ? 'pass' : 'fail',
            'message' => implode('; ', $messages)
        ];
    }

    /**
     * Utility: Get MySQL version
     */
    private function get_mysql_version() {
        global $wpdb;
        return $wpdb->get_var("SELECT VERSION()");
    }

    /**
     * Utility: Get recent log sample
     */
    private function get_recent_activity_sample() {
        global $wpdb;
        $analytics_table = $wpdb->prefix . 'aiox_analytics';
        $rows = $wpdb->get_results("SELECT * FROM $analytics_table ORDER BY timestamp DESC LIMIT 5", ARRAY_A);
        if (empty($rows)) {
            return 'No recent analytics events logged.';
        }
        $output = "";
        foreach ($rows as $r) {
            $output .= '[' . $r['timestamp'] . '] ' . $r['crawler_type'] . ' - ' . $r['page'] . PHP_EOL;
        }
        return $output;
    }
}

new AIOX_Diagnostics_Enhanced();
?>
