<?php
define('AIOX_PLUGIN_VERSION', '5.2.0');
?>
<?php
/**
 * Plugin Name: AIOX Suite Pro
 * Plugin URI: https://aioxsuite.com
 * Description: AI Optimization (AIO) plugin that makes WordPress sites AI-friendly by generating structured Q&A content and exposing it to AI crawlers through multiple channels.
 * Version: 1.0.0
 * Author: AIOX Team
 * Author URI: https://aioxsuite.com
 * License: Proprietary
 * License URI: https://aioxsuite.com/license
 *
 * Copyright (c) 2025 AIOX Team. All Rights Reserved.
 * This software and its source code are proprietary and confidential.
 * Unauthorized copying, modification, redistribution, or use of any part
 * of this software is strictly prohibited without written permission from
 * the copyright holder.
 */

// Prevent direct access
if (!defined('ABSPATH')) {
    exit;
}

// Plugin activation hook
register_activation_hook(__FILE__, 'aiox_analytics_activate');

function aiox_analytics_activate() {
    // Create/update tables
    aiox_analytics_create_tables();
    
    // Verify all tables were created
    aiox_analytics_verify_tables();
    
    // Initialize logging options
    aiox_analytics_init_logging_options();
    
    // Set plugin version
    update_option('aiox_analytics_db_version', '1.0.0');
    
    // Set default options
    add_option('aiox_enhanced_microdata_enabled', true);
    
    // Update rewrite rules version to ensure rules are properly registered
    // This ensures flush_rewrite_rules() is only called on activation, not on every page load
    update_option('aiox_rewrite_rules_version', '2.1');
    
    // Flush rewrite rules (safe to do during activation)
    flush_rewrite_rules();
}

// Check and update tables on plugin load
add_action('plugins_loaded', 'aiox_analytics_check_version', 1);

function aiox_analytics_check_version() {
    $current_version = get_option('aiox_analytics_db_version', '0.0.0');
    $required_version = '1.0.0';
    
    if (version_compare($current_version, $required_version, '<')) {
        aiox_analytics_create_tables();
        update_option('aiox_analytics_db_version', $required_version);
    }
}

// Check and create missing basic AIOX files on plugin load
add_action('plugins_loaded', 'aiox_check_basic_files', 2);

function aiox_check_basic_files() {
    // Only run in admin and not during AJAX
    if (!is_admin() || wp_doing_ajax()) {
        return;
    }
    
    // Ensure the class is available
    if (!class_exists('AIOX_Generator_Basic_Files')) {
        $file = plugin_dir_path(__FILE__) . 'core/class-aiox-generator-basic-files.php';
        if (file_exists($file)) {
            require_once $file;
        } else {
            return;
        }
    }
    
    // Check if all basic files exist, if not create them
    if (!AIOX_Generator_Basic_Files::basic_files_exist()) {
        AIOX_Generator_Basic_Files::create_basic_aiox_files();
    }
}

/**
 * Create all analytics tables
 */
function aiox_analytics_create_tables() {
    global $wpdb;
    
    $charset_collate = $wpdb->get_charset_collate();
    
    // Table 1: Main Analytics
    $sql_analytics = "CREATE TABLE {$wpdb->prefix}aiox_server_analytics (
        id BIGINT(20) UNSIGNED NOT NULL AUTO_INCREMENT,
        request_time DATETIME NOT NULL DEFAULT CURRENT_TIMESTAMP,
        endpoint VARCHAR(500) NOT NULL,
        request_data LONGTEXT,
        response_data LONGTEXT,
        status_code INT(11) DEFAULT 200,
        user_id BIGINT(20) UNSIGNED DEFAULT 0,
        PRIMARY KEY (id),
        KEY idx_request_time (request_time),
        KEY idx_endpoint (endpoint(255))
    ) $charset_collate;";

    // Table 2: Crawlers
    $sql_crawlers = "CREATE TABLE {$wpdb->prefix}aiox_server_crawlers (
        id BIGINT(20) UNSIGNED NOT NULL AUTO_INCREMENT,
        crawler_name VARCHAR(255) NOT NULL,
        crawler_type VARCHAR(50) DEFAULT 'unknown',
        user_agent TEXT,
        ip VARCHAR(100),
        first_seen DATETIME NOT NULL DEFAULT CURRENT_TIMESTAMP,
        last_seen DATETIME NOT NULL DEFAULT CURRENT_TIMESTAMP,
        hits INT(11) DEFAULT 0,
        PRIMARY KEY (id),
        KEY idx_crawler_name (crawler_name),
        KEY idx_crawler_type (crawler_type),
        KEY idx_last_seen (last_seen)
    ) $charset_collate;";

    // Table 3: Daily Stats
    $sql_daily = "CREATE TABLE {$wpdb->prefix}aiox_server_daily_stats (
        id BIGINT(20) UNSIGNED NOT NULL AUTO_INCREMENT,
        date DATE NOT NULL,
        total_visits INT(11) DEFAULT 0,
        total_crawlers INT(11) DEFAULT 0,
        ai_visits INT(11) DEFAULT 0,
        search_visits INT(11) DEFAULT 0,
        well_known_visits INT(11) DEFAULT 0,
        badge_views INT(11) DEFAULT 0,
        PRIMARY KEY (id),
        UNIQUE KEY idx_date (`date`)
    ) $charset_collate;";
    
    // Table 4: Badge Views
    $sql_badge_views = "CREATE TABLE {$wpdb->prefix}aiox_badge_views (
        id BIGINT(20) UNSIGNED NOT NULL AUTO_INCREMENT,
        post_id BIGINT(20) UNSIGNED NOT NULL,
        view_time DATETIME NOT NULL DEFAULT CURRENT_TIMESTAMP,
        viewer_type VARCHAR(50) DEFAULT 'human',
        user_agent TEXT,
        ip VARCHAR(100),
        referrer TEXT,
        PRIMARY KEY (id),
        KEY idx_post_id (post_id),
        KEY idx_view_time (view_time),
        KEY idx_viewer_type (viewer_type)
    ) $charset_collate;";

    // Table 5: Well-Known Hits
    $sql_wellknown_hits = "CREATE TABLE {$wpdb->prefix}aiox_wellknown_hits (
        id BIGINT(20) UNSIGNED NOT NULL AUTO_INCREMENT,
        file_path VARCHAR(255) NOT NULL,
        hit_time DATETIME NOT NULL DEFAULT CURRENT_TIMESTAMP,
        crawler_name VARCHAR(255),
        crawler_type VARCHAR(50),
        user_agent TEXT,
        ip VARCHAR(100),
        PRIMARY KEY (id),
        KEY idx_file_path (file_path),
        KEY idx_hit_time (hit_time),
        KEY idx_crawler_type (crawler_type)
    ) $charset_collate;";

    // Table 6: Logs (Enhanced with post_id, category, user_id, ip_address)
    $sql_logs = "CREATE TABLE {$wpdb->prefix}aiox_logs (
        id BIGINT(20) UNSIGNED NOT NULL AUTO_INCREMENT,
        timestamp DATETIME NOT NULL DEFAULT CURRENT_TIMESTAMP,
        level VARCHAR(20) NOT NULL DEFAULT 'info',
        category VARCHAR(50) NOT NULL DEFAULT 'general',
        message TEXT NOT NULL,
        context LONGTEXT,
        post_id BIGINT(20) UNSIGNED DEFAULT NULL,
        user_id BIGINT(20) UNSIGNED DEFAULT NULL,
        ip_address VARCHAR(45) DEFAULT NULL,
        PRIMARY KEY (id),
        KEY idx_timestamp (timestamp),
        KEY idx_level (level),
        KEY idx_category (category),
        KEY idx_post_id (post_id),
        KEY idx_timestamp_level (timestamp, level)
    ) $charset_collate;";
    
    // Table 7: Processing Queue (CRITICAL - was missing!)
    $sql_queue = "CREATE TABLE {$wpdb->prefix}aiox_queue (
        id BIGINT(20) UNSIGNED NOT NULL AUTO_INCREMENT,
        post_id BIGINT(20) UNSIGNED NOT NULL,
        status VARCHAR(20) NOT NULL DEFAULT 'pending',
        priority INT(11) DEFAULT 10,
        attempts INT(11) DEFAULT 0,
        error_message TEXT,
        created_at DATETIME NOT NULL DEFAULT CURRENT_TIMESTAMP,
        updated_at DATETIME NOT NULL DEFAULT CURRENT_TIMESTAMP,
        processed_at DATETIME DEFAULT NULL,
        PRIMARY KEY (id),
        KEY idx_post_id (post_id),
        KEY idx_status (status),
        KEY idx_priority (priority),
        KEY idx_created_at (created_at)
    ) $charset_collate;";

    // Use dbDelta for table creation
    require_once(ABSPATH . 'wp-admin/includes/upgrade.php');
    
    dbDelta($sql_analytics);
    dbDelta($sql_crawlers);
    dbDelta($sql_daily);
    dbDelta($sql_badge_views);
    dbDelta($sql_wellknown_hits);
    dbDelta($sql_logs);
    dbDelta($sql_queue);
    
    // Add missing columns to existing tables (for updates)
    aiox_analytics_add_missing_columns();
}

/**
 * Add missing columns to existing tables
 */
function aiox_analytics_add_missing_columns() {
    global $wpdb;
    
    // Fix daily stats table
    $table = $wpdb->prefix . 'aiox_server_daily_stats';
    
    // Check if table exists
    if ($wpdb->get_var("SHOW TABLES LIKE '$table'") !== $table) {
        return;
    }
    
    $columns = $wpdb->get_results("SHOW COLUMNS FROM $table");
    $column_names = array_column($columns, 'Field');
    
    $columns_to_add = [
        'ai_visits' => "ALTER TABLE $table ADD COLUMN ai_visits INT(11) DEFAULT 0 AFTER total_crawlers",
        'search_visits' => "ALTER TABLE $table ADD COLUMN search_visits INT(11) DEFAULT 0 AFTER ai_visits",
        'well_known_visits' => "ALTER TABLE $table ADD COLUMN well_known_visits INT(11) DEFAULT 0 AFTER search_visits",
        'badge_views' => "ALTER TABLE $table ADD COLUMN badge_views INT(11) DEFAULT 0 AFTER well_known_visits"
    ];
    
    foreach ($columns_to_add as $column => $sql) {
        if (!in_array($column, $column_names)) {
            $result = $wpdb->query($sql);
            if ($result === false) {
                error_log("AIOX Analytics: Failed to add column $column to $table - " . $wpdb->last_error);
            } else {
            }
        }
    }
    
    // Fix crawlers table
    $crawler_table = $wpdb->prefix . 'aiox_server_crawlers';
    if ($wpdb->get_var("SHOW TABLES LIKE '$crawler_table'") === $crawler_table) {
        $crawler_columns = $wpdb->get_results("SHOW COLUMNS FROM $crawler_table");
        $crawler_column_names = array_column($crawler_columns, 'Field');
        
        if (!in_array('crawler_type', $crawler_column_names)) {
            $wpdb->query("ALTER TABLE $crawler_table ADD COLUMN crawler_type VARCHAR(50) DEFAULT 'unknown' AFTER crawler_name");
        }
    }
}

/**
 * Verify all required tables exist and create them if missing
 */
function aiox_analytics_verify_tables() {
    global $wpdb;
    
    $required_tables = array(
        'aiox_server_analytics' => 'Main analytics table',
        'aiox_server_crawlers' => 'Crawlers tracking table',
        'aiox_server_daily_stats' => 'Daily statistics table',
        'aiox_badge_views' => 'Badge views table',
        'aiox_wellknown_hits' => 'Well-known hits table',
        'aiox_logs' => 'System logs table',
        'aiox_queue' => 'Processing queue table'
    );
    
    $missing_tables = array();
    $charset_collate = $wpdb->get_charset_collate();
    
    foreach ($required_tables as $table_name => $description) {
        $full_table_name = $wpdb->prefix . $table_name;
        $exists = $wpdb->get_var("SHOW TABLES LIKE '{$full_table_name}'") === $full_table_name;
        
        if (!$exists) {
            $missing_tables[] = $table_name;
            error_log("AIOX: Missing table detected: {$full_table_name} ({$description})");
            
            // Force create missing table using raw SQL
            aiox_analytics_force_create_table($table_name, $charset_collate);
        }
    }
    
    if (empty($missing_tables)) {
        error_log("AIOX: All required tables verified successfully");
    } else {
        error_log("AIOX: Created " . count($missing_tables) . " missing tables: " . implode(', ', $missing_tables));
    }
    
    return empty($missing_tables);
}

/**
 * Force create a specific table if dbDelta failed
 */
function aiox_analytics_force_create_table($table_name, $charset_collate) {
    global $wpdb;
    $full_table_name = $wpdb->prefix . $table_name;
    
    // Define table structures with proper formatting
    $table_schemas = array(
        'aiox_logs' => "CREATE TABLE IF NOT EXISTS {$full_table_name} (
            id BIGINT(20) UNSIGNED NOT NULL AUTO_INCREMENT,
            timestamp DATETIME NOT NULL DEFAULT CURRENT_TIMESTAMP,
            level VARCHAR(20) NOT NULL DEFAULT 'info',
            category VARCHAR(50) NOT NULL DEFAULT 'general',
            message TEXT NOT NULL,
            context LONGTEXT,
            post_id BIGINT(20) UNSIGNED DEFAULT NULL,
            user_id BIGINT(20) UNSIGNED DEFAULT NULL,
            ip_address VARCHAR(45) DEFAULT NULL,
            PRIMARY KEY (id),
            KEY idx_timestamp (timestamp),
            KEY idx_level (level),
            KEY idx_category (category),
            KEY idx_post_id (post_id),
            KEY idx_timestamp_level (timestamp, level)
        ) {$charset_collate}",
        
        'aiox_server_analytics' => "CREATE TABLE IF NOT EXISTS {$full_table_name} (
            id BIGINT(20) UNSIGNED NOT NULL AUTO_INCREMENT,
            request_time DATETIME NOT NULL DEFAULT CURRENT_TIMESTAMP,
            endpoint VARCHAR(500) NOT NULL,
            request_data LONGTEXT,
            response_data LONGTEXT,
            status_code INT(11) DEFAULT 200,
            user_id BIGINT(20) UNSIGNED DEFAULT 0,
            PRIMARY KEY (id),
            KEY idx_request_time (request_time),
            KEY idx_endpoint (endpoint(255))
        ) {$charset_collate}",
        
        'aiox_server_crawlers' => "CREATE TABLE IF NOT EXISTS {$full_table_name} (
            id BIGINT(20) UNSIGNED NOT NULL AUTO_INCREMENT,
            crawler_name VARCHAR(255) NOT NULL,
            crawler_type VARCHAR(50) DEFAULT 'unknown',
            user_agent TEXT,
            ip VARCHAR(100),
            first_seen DATETIME NOT NULL DEFAULT CURRENT_TIMESTAMP,
            last_seen DATETIME NOT NULL DEFAULT CURRENT_TIMESTAMP,
            hits INT(11) DEFAULT 0,
            PRIMARY KEY (id),
            KEY idx_crawler_name (crawler_name),
            KEY idx_crawler_type (crawler_type),
            KEY idx_last_seen (last_seen)
        ) {$charset_collate}",
        
        'aiox_server_daily_stats' => "CREATE TABLE IF NOT EXISTS {$full_table_name} (
            id BIGINT(20) UNSIGNED NOT NULL AUTO_INCREMENT,
            date DATE NOT NULL,
            total_visits INT(11) DEFAULT 0,
            total_crawlers INT(11) DEFAULT 0,
            ai_visits INT(11) DEFAULT 0,
            search_visits INT(11) DEFAULT 0,
            well_known_visits INT(11) DEFAULT 0,
            badge_views INT(11) DEFAULT 0,
            PRIMARY KEY (id),
            UNIQUE KEY idx_date (date)
        ) {$charset_collate}",
        
        'aiox_badge_views' => "CREATE TABLE IF NOT EXISTS {$full_table_name} (
            id BIGINT(20) UNSIGNED NOT NULL AUTO_INCREMENT,
            post_id BIGINT(20) UNSIGNED NOT NULL,
            view_time DATETIME NOT NULL DEFAULT CURRENT_TIMESTAMP,
            user_agent TEXT,
            ip VARCHAR(100),
            referer TEXT,
            PRIMARY KEY (id),
            KEY idx_post_id (post_id),
            KEY idx_view_time (view_time)
        ) {$charset_collate}",
        
        'aiox_wellknown_hits' => "CREATE TABLE IF NOT EXISTS {$full_table_name} (
            id BIGINT(20) UNSIGNED NOT NULL AUTO_INCREMENT,
            file_path VARCHAR(500) NOT NULL,
            hit_time DATETIME NOT NULL DEFAULT CURRENT_TIMESTAMP,
            user_agent TEXT,
            ip VARCHAR(100),
            crawler_type VARCHAR(50) DEFAULT 'unknown',
            PRIMARY KEY (id),
            KEY idx_file_path (file_path(255)),
            KEY idx_hit_time (hit_time),
            KEY idx_crawler_type (crawler_type)
        ) {$charset_collate}",
        
        'aiox_queue' => "CREATE TABLE IF NOT EXISTS {$full_table_name} (
            id BIGINT(20) UNSIGNED NOT NULL AUTO_INCREMENT,
            post_id BIGINT(20) UNSIGNED NOT NULL,
            status VARCHAR(20) NOT NULL DEFAULT 'pending',
            priority INT(11) DEFAULT 10,
            attempts INT(11) DEFAULT 0,
            error_message TEXT,
            created_at DATETIME NOT NULL DEFAULT CURRENT_TIMESTAMP,
            updated_at DATETIME NOT NULL DEFAULT CURRENT_TIMESTAMP,
            processed_at DATETIME DEFAULT NULL,
            PRIMARY KEY (id),
            KEY idx_post_id (post_id),
            KEY idx_status (status),
            KEY idx_priority (priority),
            KEY idx_created_at (created_at)
        ) {$charset_collate}"
    );
    
    if (isset($table_schemas[$table_name])) {
        $result = $wpdb->query($table_schemas[$table_name]);
        
        if ($result === false) {
            error_log("AIOX: Failed to force create table {$full_table_name}: " . $wpdb->last_error);
            return false;
        } else {
            error_log("AIOX: Successfully force created table {$full_table_name}");
            return true;
        }
    }
    
    return false;
}

/**
 * Initialize logging options if they don't exist
 */
function aiox_analytics_init_logging_options() {
    // Enable logging by default
    if (get_option('aiox_enable_logging') === false) {
        add_option('aiox_enable_logging', 1, '', 'yes');
    }
    
    // Set default log level to error (prevents database overload)
    if (get_option('aiox_log_level') === false) {
        add_option('aiox_log_level', 'error', '', 'yes');
    }
    
    // Set default retention to 30 days
    if (get_option('aiox_log_retention_days') === false) {
        add_option('aiox_log_retention_days', 30, '', 'yes');
    }
    
    error_log("AIOX: Logging options initialized - Enabled: 1, Level: error, Retention: 30 days");
}

/**
 * Plugin deactivation hook
 */
register_deactivation_hook(__FILE__, 'aiox_analytics_deactivate');

function aiox_analytics_deactivate() {
    flush_rewrite_rules();
    // Don't delete tables on deactivation - only on uninstall
}

// Ensure WordPress functions are available
if (!function_exists('plugin_dir_path')) {
    return;
}

// Define plugin constants
define('AIOX_VERSION', '5.2.0');
define('AIOX_PLUGIN_FILE', __FILE__);
define('AIOX_PLUGIN_DIR', plugin_dir_path(__FILE__));
define('AIOX_PLUGIN_URL', plugin_dir_url(__FILE__));
define('AIOX_PLUGIN_BASENAME', plugin_basename(__FILE__));

// Define paths
define('AIOX_WELL_KNOWN_PATH', ABSPATH . '.well-known/');
define('AIOX_DEFAULT_AI_ENDPOINT', 'https://aioxsuite.com/api/ai');

/**
 * Render AIOX page header with logo
 * 
 * @param string $title Page title
 * @param string $subtitle Optional subtitle
 * @param bool $echo Whether to echo or return
 * @return string|void
 */
function aiox_render_page_header($title, $subtitle = '', $echo = true) {
    // Simple WordPress native header - no custom CSS needed
    $html = '<div class="aiox-page-header" style="display: flex; align-items: center; gap: 12px; margin-bottom: 15px;">';
    $html .= '<span class="dashicons dashicons-networking" style="font-size: 36px; width: 36px; height: 36px; color: #2271b1;"></span>';
    $html .= '<div>';
    $html .= '<h1 style="margin: 0; padding: 0; font-size: 23px; font-weight: 400; line-height: 1.3;">' . esc_html($title) . '</h1>';
    if ($subtitle) {
        $html .= '<p class="description" style="margin: 2px 0 0 0;">' . esc_html($subtitle) . '</p>';
    }
    $html .= '</div>';
    $html .= '</div>';
    
    if ($echo) {
        echo $html;
    } else {
        return $html;
    }
}

/**
 * Main AIOX Suite Pro Class
 */
class AIOX_Publisher_Suite_Pro {
    
    /**
     * Single instance of the class
     */
    private static $instance = null;
    
    /**
     * Get single instance
     */
    public static function get_instance() {
        if (null === self::$instance) {
            self::$instance = new self();
        }
        return self::$instance;
    }
    
    /**
     * Constructor
     */
    private function __construct() {
        // Load dependencies
        $this->load_dependencies();
        
        // Register activation/deactivation hooks
        register_activation_hook(__FILE__, array($this, 'activate'));
        register_deactivation_hook(__FILE__, array($this, 'deactivate'));
        register_uninstall_hook(__FILE__, array('AIOX_Publisher_Suite_Pro', 'uninstall'));
        
        // Initialize plugin
        add_action('init', array($this, 'init'));
        add_action('admin_menu', array($this, 'add_admin_menu'));
        add_action('admin_enqueue_scripts', array($this, 'admin_enqueue_scripts'));
        add_action('admin_notices', array($this, 'show_regeneration_notice'));
        add_action('admin_notices', array($this, 'show_api_key_notice'));
        
        // Initialize basic hooks as fallback only if enhanced hooks not available
        // Note: AIOX_Enhanced_Hooks is initialized later with instance reference storage
        if (!class_exists('AIOX_Enhanced_Hooks') && class_exists('AIOX_Hooks')) {
            new AIOX_Hooks();
        }
        
        // Initialize AI badge
        if (class_exists('AIOX_Badge')) {
            new AIOX_Badge();
        }
        
        // Auto-processor is now handled by Enhanced Hooks
        // Keeping this for backward compatibility if Enhanced Hooks fails
        if (!class_exists('AIOX_Enhanced_Hooks') && class_exists('AIOX_Auto_Processor')) {
            new AIOX_Auto_Processor();
        }
        
        // Add scheduled action for ingredients database fix
        add_action('aiox_fix_ingredients_database', array($this, 'fix_ingredients_database'));
        
        // Initialize post deletion handler
        if (class_exists('AIOX_Post_Deletion_Handler')) {
            new AIOX_Post_Deletion_Handler();
        }
        
        // Initialize meta boxes (admin only)
        if (is_admin() && class_exists('AIOX_Meta_Boxes')) {
            new AIOX_Meta_Boxes();
        }
        
        // Initialize capsule meta box (admin only)
        if (is_admin() && file_exists(AIOX_PLUGIN_DIR . 'admin/class-aiox-capsule-meta-box.php')) {
            require_once AIOX_PLUGIN_DIR . 'admin/class-aiox-capsule-meta-box.php';
            if (class_exists('AIOX_Capsule_Meta_Box')) {
                new AIOX_Capsule_Meta_Box();
            }
        }
        
        // Initialize AJAX handlers
        if (class_exists('AIOX_Ajax_Handlers')) {
            new AIOX_Ajax_Handlers();
        }
        
        // Initialize Capsule AJAX handlers (admin only)
        if (is_admin() && class_exists('AIOX_Capsules_Ajax_Handlers')) {
            new AIOX_Capsules_Ajax_Handlers();
        }
        
        // Initialize Settings (admin only) - Must be initialized early for settings registration
        if (is_admin() && class_exists('AIOX_Settings')) {
            new AIOX_Settings();
        }
        
        // Initialize Licensing Page (admin only)
        if (is_admin() && class_exists('AIOX_Licensing_Page')) {
            new AIOX_Licensing_Page();
        }
        
        // Initialize Capsules (admin only) - Must be initialized early for AJAX handlers
        if (is_admin() && class_exists('AIOX_Capsules_Hybrid')) {
            try {
                $this->capsules_instance = new AIOX_Capsules_Hybrid();
            } catch (Exception $e) {
                AIOX_Logger::error('Failed to initialize AIOX_Capsules_Hybrid early', array('error' => $e->getMessage()));
            }
        }
        
        // Phase 1: Initialize Processing Manager (admin only) - Extends Capsules functionality
        if (is_admin() && class_exists('AIOX_Processing_Manager')) {
            try {
                $this->processing_manager_instance = new AIOX_Processing_Manager();
                AIOX_Logger::info('AIOX Phase 1: Processing Manager initialized successfully');
            } catch (Exception $e) {
                AIOX_Logger::error('AIOX Phase 1: Failed to initialize Processing Manager', array('error' => $e->getMessage()));
            }
        }
        
        // Phase 1: Initialize Enhanced Hooks - Extends existing hooks functionality
        if (class_exists('AIOX_Enhanced_Hooks')) {
            try {
                $this->enhanced_hooks_instance = new AIOX_Enhanced_Hooks();
                AIOX_Logger::info('AIOX Phase 1: Enhanced Hooks initialized successfully');
            } catch (Exception $e) {
                AIOX_Logger::error('AIOX Phase 1: Failed to initialize Enhanced Hooks', array('error' => $e->getMessage()));
            }
        }
        
        // Phase 1: Initialize Enhanced Deletion Handler
        if (class_exists('AIOX_Enhanced_Deletion')) {
            try {
                $this->enhanced_deletion_instance = new AIOX_Enhanced_Deletion();
                AIOX_Logger::info('AIOX Phase 1: Enhanced Deletion Handler initialized successfully');
            } catch (Exception $e) {
                AIOX_Logger::error('AIOX Phase 1: Failed to initialize Enhanced Deletion Handler', array('error' => $e->getMessage()));
            }
        }
        
        // Initialize Comparison (admin only) - Must be initialized early for AJAX handlers
        if (is_admin() && class_exists('AIOX_Comparison')) {
            new AIOX_Comparison();
        }
        
        // Initialize Diagnostics early to register AJAX handlers
        if (is_admin()) {
            add_action('wp_loaded', array($this, 'init_diagnostics_early'));
        }
        
        // Initialize Inline JSON (frontend only)
        if (!is_admin() && class_exists('AIOX_Inline_JSON')) {
            add_action('wp', function() {
                new AIOX_Inline_JSON();
            });
        }
        
        // Initialize Modern Analytics Admin (admin only)
        if (is_admin() && class_exists('AIOX_Modern_Analytics_Admin')) {
            AIOX_Modern_Analytics_Admin::get_instance();
        }
    }
    
    
    
    /**
     * Initialize diagnostics early to register AJAX handlers
     */
    public function init_diagnostics_early() {
        if (!class_exists('AIOX_Diagnostics')) {
            require_once AIOX_PLUGIN_DIR . 'admin/class-aiox-diagnostics.php';
        }
        
        // Create a global instance to ensure AJAX handlers are registered
        if (class_exists('AIOX_Diagnostics') && !isset($GLOBALS['aiox_diagnostics_instance'])) {
            $GLOBALS['aiox_diagnostics_instance'] = new AIOX_Diagnostics();
        }
    }
    
    /**
     * Load required dependencies
     */
    private function load_dependencies() {
        // Core classes
        if (file_exists(AIOX_PLUGIN_DIR . 'core/class-aiox-logger.php')) {
            require_once AIOX_PLUGIN_DIR . 'core/class-aiox-logger.php';
            require_once AIOX_PLUGIN_DIR . 'admin/class-aiox-performance-monitor.php';
        }
        if (file_exists(AIOX_PLUGIN_DIR . 'core/class-aiox-utils.php')) {
            require_once AIOX_PLUGIN_DIR . 'core/class-aiox-utils.php';
        }
        if (file_exists(AIOX_PLUGIN_DIR . 'core/class-aiox-crypto.php')) {
            require_once AIOX_PLUGIN_DIR . 'core/class-aiox-crypto.php';
        }
        if (file_exists(AIOX_PLUGIN_DIR . 'core/class-aiox-pack.php')) {
            require_once AIOX_PLUGIN_DIR . 'core/class-aiox-pack.php';
        }
        if (file_exists(AIOX_PLUGIN_DIR . 'core/class-aiox-metadata.php')) {
            require_once AIOX_PLUGIN_DIR . 'core/class-aiox-metadata.php';
        }
        if (file_exists(AIOX_PLUGIN_DIR . 'core/class-aiox-ingredients.php')) {
            require_once AIOX_PLUGIN_DIR . 'core/class-aiox-ingredients.php';
        }
        if (file_exists(AIOX_PLUGIN_DIR . 'core/class-aiox-generator.php')) {
            require_once AIOX_PLUGIN_DIR . 'core/class-aiox-generator.php';
        }
        if (file_exists(AIOX_PLUGIN_DIR . 'core/class-aiox-licensing.php')) {
            require_once AIOX_PLUGIN_DIR . 'core/class-aiox-licensing.php';
        }
        if (file_exists(AIOX_PLUGIN_DIR . 'core/class-aiox-ai-client.php')) {
            require_once AIOX_PLUGIN_DIR . 'core/class-aiox-ai-client.php';
        }
        if (file_exists(AIOX_PLUGIN_DIR . 'core/class-aiox-custom-client.php')) {
            require_once AIOX_PLUGIN_DIR . 'core/class-aiox-custom-client.php';
        }
        if (file_exists(AIOX_PLUGIN_DIR . 'core/class-aiox-ai-processor.php')) {
            require_once AIOX_PLUGIN_DIR . 'core/class-aiox-ai-processor.php';
        }
        if (file_exists(AIOX_PLUGIN_DIR . 'core/class-aiox-capsule.php')) {
            require_once AIOX_PLUGIN_DIR . 'core/class-aiox-capsule.php';
        }
        if (file_exists(AIOX_PLUGIN_DIR . 'core/class-aiox-capsule-manager.php')) {
            require_once AIOX_PLUGIN_DIR . 'core/class-aiox-capsule-manager.php';
        }
        
        // Load Robots.txt Manager
if (file_exists(AIOX_PLUGIN_DIR . 'core/class-aiox-robots-manager.php')) {
    require_once AIOX_PLUGIN_DIR . 'core/class-aiox-robots-manager.php';
}
        
        if (file_exists(AIOX_PLUGIN_DIR . 'core/class-aiox-capsule-generator.php')) {
            require_once AIOX_PLUGIN_DIR . 'core/class-aiox-capsule-generator.php';
        }
        if (file_exists(AIOX_PLUGIN_DIR . 'core/class-aiox-modern-analytics.php')) {
            require_once AIOX_PLUGIN_DIR . 'core/class-aiox-modern-analytics.php';
        }
        
        // Public classes
        if (file_exists(AIOX_PLUGIN_DIR . 'public/class-aiox-badge.php')) {
            require_once AIOX_PLUGIN_DIR . 'public/class-aiox-badge.php';
        }
        
        // Includes
        if (file_exists(AIOX_PLUGIN_DIR . 'includes/class-aiox-content-data.php')) {
            require_once AIOX_PLUGIN_DIR . 'includes/class-aiox-content-data.php';
        }
        if (file_exists(AIOX_PLUGIN_DIR . 'includes/class-aiox-hooks.php')) {
            require_once AIOX_PLUGIN_DIR . 'includes/class-aiox-hooks.php';
        }
        if (file_exists(AIOX_PLUGIN_DIR . 'includes/class-aiox-auto-processor.php')) {
            require_once AIOX_PLUGIN_DIR . 'includes/class-aiox-auto-processor.php';
        }
        if (file_exists(AIOX_PLUGIN_DIR . 'core/class-aiox-comprehensive-processor.php')) {
            require_once AIOX_PLUGIN_DIR . 'core/class-aiox-comprehensive-processor.php';
        }
        if (file_exists(AIOX_PLUGIN_DIR . 'includes/class-aiox-post-deletion-handler.php')) {
            require_once AIOX_PLUGIN_DIR . 'includes/class-aiox-post-deletion-handler.php';
        }
        // Load enhanced microdata (replaces basic inline JSON)
if (file_exists(AIOX_PLUGIN_DIR . 'includes/class-aiox-enhanced-microdata.php')) {
    require_once AIOX_PLUGIN_DIR . 'includes/class-aiox-enhanced-microdata.php';
}
        
        // AI-Only Enforcer for Version 5.1.8+
        if (file_exists(AIOX_PLUGIN_DIR . 'core/class-aiox-ai-only-enforcer.php')) {
            require_once AIOX_PLUGIN_DIR . 'core/class-aiox-ai-only-enforcer.php';
            AIOX_AI_Only_Enforcer::init();
        }
        
        // Dashboard widget for AI-only status
        if (file_exists(AIOX_PLUGIN_DIR . 'admin/class-aiox-dashboard-widget.php')) {
            require_once AIOX_PLUGIN_DIR . 'admin/class-aiox-dashboard-widget.php';
            AIOX_Dashboard_Widget::init();
        }
        
        // Phase 1: Foundation classes for comprehensive processing
        // Load these BEFORE admin classes to ensure they're available
        if (file_exists(AIOX_PLUGIN_DIR . 'core/class-aiox-comprehensive-sync.php')) {
            require_once AIOX_PLUGIN_DIR . 'core/class-aiox-comprehensive-sync.php';
        }
        if (file_exists(AIOX_PLUGIN_DIR . 'includes/class-aiox-enhanced-hooks.php')) {
            require_once AIOX_PLUGIN_DIR . 'includes/class-aiox-enhanced-hooks.php';
        }
        if (file_exists(AIOX_PLUGIN_DIR . 'includes/class-aiox-enhanced-deletion.php')) {
            require_once AIOX_PLUGIN_DIR . 'includes/class-aiox-enhanced-deletion.php';
        }
        
        // REST API endpoints for license verification and post data
        if (file_exists(AIOX_PLUGIN_DIR . 'includes/class-aiox-rest-api.php')) {
            require_once AIOX_PLUGIN_DIR . 'includes/class-aiox-rest-api.php';
        }
        
        if (file_exists(AIOX_PLUGIN_DIR . 'includes/class-aiox-phase1-activation.php')) {
            require_once AIOX_PLUGIN_DIR . 'includes/class-aiox-phase1-activation.php';
        }
        
        // Admin classes
        if (is_admin()) {
            if (file_exists(AIOX_PLUGIN_DIR . 'admin/class-aiox-settings.php')) {
                require_once AIOX_PLUGIN_DIR . 'admin/class-aiox-settings.php';
            }
            if (file_exists(AIOX_PLUGIN_DIR . 'admin/class-aiox-licensing-page.php')) {
                require_once AIOX_PLUGIN_DIR . 'admin/class-aiox-licensing-page.php';
            }
            // Monetization Page - needs early loading for settings registration
            if (file_exists(AIOX_PLUGIN_DIR . 'admin/class-aiox-monetization-page.php')) {
                require_once AIOX_PLUGIN_DIR . 'admin/class-aiox-monetization-page.php';
            }
            if (file_exists(AIOX_PLUGIN_DIR . 'admin/class-aiox-diagnostics.php')) {
                require_once AIOX_PLUGIN_DIR . 'admin/class-aiox-diagnostics.php';
            }
            if (file_exists(AIOX_PLUGIN_DIR . 'includes/class-aiox-logs-viewer.php')) {
                require_once AIOX_PLUGIN_DIR . 'includes/class-aiox-logs-viewer.php';
            }
            if (file_exists(AIOX_PLUGIN_DIR . 'admin/class-aiox-meta-boxes.php')) {
                require_once AIOX_PLUGIN_DIR . 'admin/class-aiox-meta-boxes.php';
            }
            if (file_exists(AIOX_PLUGIN_DIR . 'admin/class-aiox-ajax-handlers.php')) {
                require_once AIOX_PLUGIN_DIR . 'admin/class-aiox-ajax-handlers.php';
            }
            if (file_exists(AIOX_PLUGIN_DIR . 'admin/class-aiox-comparison.php')) {
                require_once AIOX_PLUGIN_DIR . 'admin/class-aiox-comparison.php';
            }
            if (file_exists(AIOX_PLUGIN_DIR . 'admin/class-aiox-capsules-hybrid.php')) {
                require_once AIOX_PLUGIN_DIR . 'admin/class-aiox-capsules-hybrid.php';
            }
            // Phase 1: Load Processing Manager after capsules-hybrid
            if (file_exists(AIOX_PLUGIN_DIR . 'admin/class-aiox-processing-manager.php')) {
                require_once AIOX_PLUGIN_DIR . 'admin/class-aiox-processing-manager.php';
            }
            // Load Processing Manager Tabs for enhanced functionality
            if (file_exists(AIOX_PLUGIN_DIR . 'admin/class-aiox-processing-manager-tabs.php')) {
                require_once AIOX_PLUGIN_DIR . 'admin/class-aiox-processing-manager-tabs.php';
            }
            if (file_exists(AIOX_PLUGIN_DIR . 'admin/class-aiox-capsules-ajax-handlers.php')) {
                require_once AIOX_PLUGIN_DIR . 'admin/class-aiox-capsules-ajax-handlers.php';
            }
            
            // Modern Analytics Admin
            if (file_exists(AIOX_PLUGIN_DIR . 'admin/class-aiox-modern-analytics-admin.php')) {
                require_once AIOX_PLUGIN_DIR . 'admin/class-aiox-modern-analytics-admin.php';
            }
            
        }
    }
    
    /**
     * Plugin activation
     */
    public function activate() {
        try {
            // Create database tables
            $this->create_database_tables();
            
            // Create modern analytics tables
            if (class_exists("AIOX_Modern_Analytics")) {
                AIOX_Modern_Analytics::maybe_create_tables();
            }

            // Create the main database tables
            $this->create_database_tables();
            
            // Create well-known directory
            $this->create_well_known_directory();
            
            // Set default options
            $this->set_default_options();
            
            // Version 2.0.1+: Create basic AIOX files on activation
            if (class_exists('AIOX_Generator')) {
                try {
                    require_once AIOX_PLUGIN_DIR . 'core/class-aiox-generator-basic-files.php';
                    AIOX_Generator_Basic_Files::create_basic_aiox_files();
                } catch (Exception $e) {
                    error_log('AIOX: Failed to create basic files during activation: ' . $e->getMessage());
                }
            }
            
            // Phase 1: Run Phase 1 specific activation tasks
            if (class_exists('AIOX_Phase1_Activation')) {
                AIOX_Phase1_Activation::activate();
            }
            
            // Fix any existing corrupted ingredients data during activation
            $this->fix_ingredients_during_activation();
            
            // Check if we have existing processed posts but missing files (reinstall scenario)
            $this->check_and_schedule_file_regeneration();
            
            // Flush rewrite rules
            flush_rewrite_rules();
            
        } catch (Exception $e) {
            // Log activation error
            error_log('AIOX: Plugin activation failed: ' . $e->getMessage());
        }
    }
    
    /**
     * Check for processed posts with missing files and schedule regeneration
     * This handles edge cases where files were manually deleted
     */
    private function check_and_schedule_file_regeneration() {
        global $wpdb;
        
        // Check if we have processed posts (by looking for aiox_qa_pairs meta)
        $processed_count = $wpdb->get_var(
            "SELECT COUNT(*) FROM {$wpdb->postmeta} WHERE meta_key = 'aiox_qa_pairs'"
        );
        
        // Only check if there are processed posts
        if ($processed_count > 0) {
            // Check if capsule files exist
            $capsules_dir = ABSPATH . '.well-known/capsules/';
            $capsule_files_count = 0;
            
            if (is_dir($capsules_dir)) {
                $files = glob($capsules_dir . '*.json');
                $capsule_files_count = $files ? count($files) : 0;
            }
            
            // Only show warning if we have processed posts but significantly fewer capsule files
            // This handles the case where files were manually deleted
            if ($capsule_files_count == 0 && $processed_count > 0) {
                update_option('aiox_needs_file_regeneration', true);
                error_log('AIOX: Detected ' . $processed_count . ' processed posts but no capsule files. File regeneration may be needed.');
            }
        }
    }
    
    /**
     * Show admin notice if file regeneration is needed
     */
    public function show_regeneration_notice() {
        if (!get_option('aiox_needs_file_regeneration')) {
            return;
        }
        
        // Check if user dismissed the notice
        if (isset($_GET['aiox_dismiss_regen_notice'])) {
            delete_option('aiox_needs_file_regeneration');
            wp_safe_redirect(remove_query_arg('aiox_dismiss_regen_notice'));
            exit;
        }
        
        $dashboard_url = admin_url('admin.php?page=aiox-dashboard');
        $dismiss_url = add_query_arg('aiox_dismiss_regen_notice', '1');
        
        echo '<div class="notice notice-warning">';
        echo '<p><strong>⚠️ ' . __('AIOX: File Regeneration Required', 'aiox-publisher-suite-pro') . '</strong></p>';
        echo '<p>' . __('Your processed content data was preserved from a previous installation, but the AIOX capsule files need to be regenerated. Please click "Re-Generate Files" on the dashboard to restore full functionality.', 'aiox-publisher-suite-pro') . '</p>';
        echo '<p><a href="' . esc_url($dashboard_url) . '" class="button button-primary">' . __('Go to Dashboard', 'aiox-publisher-suite-pro') . '</a> ';
        echo '<a href="' . esc_url($dismiss_url) . '" class="button" style="margin-left: 10px;">' . __('Dismiss', 'aiox-publisher-suite-pro') . '</a></p>';
        echo '</div>';
    }
    
    /**
     * Show API key setup notice when no API key is configured
     */
    public function show_api_key_notice() {
        // Only show on AIOX plugin pages or WordPress dashboard
        $screen = get_current_screen();
        $is_aiox_page = $screen && (
            strpos($screen->id, 'aiox') !== false || 
            $screen->id === 'dashboard' ||
            $screen->id === 'plugins'
        );
        
        if (!$is_aiox_page) {
            return;
        }
        
        // Check if API key is set
        $api_key = get_option('aiox_custom_api_key', '');
        if (!empty($api_key)) {
            return;
        }
        
        // Check if user dismissed the notice temporarily (for 7 days)
        $dismissed_until = get_option('aiox_api_notice_dismissed_until', 0);
        if ($dismissed_until > time()) {
            return;
        }
        
        // Handle dismiss action
        if (isset($_GET['aiox_dismiss_api_notice']) && wp_verify_nonce($_GET['_wpnonce'] ?? '', 'aiox_dismiss_api_notice')) {
            // Dismiss for 7 days
            update_option('aiox_api_notice_dismissed_until', time() + (7 * DAY_IN_SECONDS));
            wp_safe_redirect(remove_query_arg(['aiox_dismiss_api_notice', '_wpnonce']));
            exit;
        }
        
        $settings_url = admin_url('admin.php?page=aiox-settings');
        $dismiss_url = wp_nonce_url(add_query_arg('aiox_dismiss_api_notice', '1'), 'aiox_dismiss_api_notice');
        
        ?>
        <div class="notice notice-info aiox-api-notice" style="border-left-color: #8b5cf6; background: linear-gradient(135deg, #f8f7ff 0%, #f0ecff 100%); padding: 15px 20px; position: relative;">
            <div style="display: flex; align-items: center; gap: 15px;">
                <div style="flex-shrink: 0;">
                    <svg width="40" height="40" viewBox="0 0 24 24" fill="none" xmlns="http://www.w3.org/2000/svg">
                        <circle cx="12" cy="12" r="10" stroke="#8b5cf6" stroke-width="2"/>
                        <path d="M12 7v5l3 3" stroke="#8b5cf6" stroke-width="2" stroke-linecap="round"/>
                    </svg>
                </div>
                <div style="flex-grow: 1;">
                    <p style="margin: 0 0 8px 0; font-size: 15px; font-weight: 600; color: #1e1e2e;">
                        🚀 <?php _e('Welcome to AIOX Suite! Get Started in 2 Minutes', 'aiox-publisher-suite-pro'); ?>
                    </p>
                    <p style="margin: 0 0 12px 0; color: #555; font-size: 13px;">
                        <?php _e('To unlock AI-powered content optimization and make your site discoverable by AI crawlers, you need an AIOX API key.', 'aiox-publisher-suite-pro'); ?>
                    </p>
                    <div style="display: flex; gap: 10px; flex-wrap: wrap;">
                        <a href="https://app.aioxsuite.com/register" target="_blank" class="button button-primary" style="background: #8b5cf6; border-color: #8b5cf6; padding: 6px 16px; height: auto;">
                            <?php _e('Create Free Account', 'aiox-publisher-suite-pro'); ?> →
                        </a>
                        <a href="<?php echo esc_url($settings_url); ?>" class="button" style="padding: 6px 16px; height: auto;">
                            <?php _e('Enter API Key', 'aiox-publisher-suite-pro'); ?>
                        </a>
                        <a href="<?php echo esc_url($dismiss_url); ?>" class="button" style="padding: 6px 16px; height: auto; color: #666;">
                            <?php _e('Remind me later', 'aiox-publisher-suite-pro'); ?>
                        </a>
                    </div>
                </div>
            </div>
        </div>
        <?php
    }
    
    /**
     * Format file size for display
     */
    private function format_file_size($bytes) {
        if ($bytes >= 1048576) {
            return number_format($bytes / 1048576, 1) . ' MB';
        } elseif ($bytes >= 1024) {
            return number_format($bytes / 1024, 1) . ' KB';
        } else {
            return $bytes . ' B';
        }
    }
    
    /**
     * Plugin deactivation
     * 
     * Only cleans up cron jobs and rewrite rules.
     * All files, database tables, and settings are preserved for reactivation.
     */
    public function deactivate() {
        // Phase 1: Run Phase 1 specific deactivation tasks
        if (class_exists('AIOX_Phase1_Activation')) {
            AIOX_Phase1_Activation::deactivate();
        }
        
        // Clean up cron jobs only
        wp_clear_scheduled_hook('aiox_process_queue');
        
        // Flush rewrite rules
        flush_rewrite_rules();
        
        // NOTE: All files, database tables, and settings are preserved
        // so that reactivation restores everything to its previous state.
        // Use "Total Reset" in Settings to remove all data before uninstalling.
    }
    
    /**
     * Initialize plugin
     */
    public function init() {
        // Load text domain
        load_plugin_textdomain('aiox-publisher-suite-pro', false, dirname(AIOX_PLUGIN_BASENAME) . '/languages/');
        
        
        
        // Add REST API endpoints
        add_action('rest_api_init', array($this, 'register_rest_routes'));
        
        // Add rewrite rules for well-known files
        add_action('init', array($this, 'add_rewrite_rules'));
    }
    
    /**
     * Add admin menu
     */
    public function add_admin_menu() {
        // Add main menu page
        add_menu_page(
            __('AIOX Dashboard', 'aiox-publisher-suite-pro'),
            __('AIOX', 'aiox-publisher-suite-pro'),
            'manage_options',
            'aiox-dashboard',
            array($this, 'render_dashboard_wrapper'),
            'dashicons-networking',
            30
        );
        
        // Add Dashboard as first submenu item
        add_submenu_page(
            'aiox-dashboard',
            __('AIOX Dashboard', 'aiox-publisher-suite-pro'),
            __('📊 Dashboard', 'aiox-publisher-suite-pro'),
            'manage_options',
            'aiox-dashboard',
            array($this, 'render_dashboard_wrapper')
        );
        
        // Add Settings submenu
        add_submenu_page(
            'aiox-dashboard',
            __('AIOX Settings', 'aiox-publisher-suite-pro'),
            __('⚙️ Settings', 'aiox-publisher-suite-pro'),
            'manage_options',
            'aiox-settings',
            array($this, 'render_settings_page')
        );
        
        // Add Monetization submenu
        add_submenu_page(
            'aiox-dashboard',
            __('AI Monetization', 'aiox-publisher-suite-pro'),
            __('💰 Monetization', 'aiox-publisher-suite-pro'),
            'manage_options',
            'aiox-monetization',
            array($this, 'render_monetization_page')
        );
        
        // Add Diagnostics submenu
        add_submenu_page(
            'aiox-dashboard',
            __('AIOX Diagnostics', 'aiox-publisher-suite-pro'),
            __('🔧 Diagnostics', 'aiox-publisher-suite-pro'),
            'manage_options',
            'aiox-diagnostics',
            array($this, 'render_diagnostics_page')
        );
        
        // Add Processing Comparison submenu
        add_submenu_page(
            'aiox-dashboard',
            __('AIOX Processing Comparison', 'aiox-publisher-suite-pro'),
            __('🤖 AI Comparison', 'aiox-publisher-suite-pro'),
            'manage_options',
            'aiox-comparison',
            array($this, 'render_comparison_page')
        );
        
        // Phase 1: Add Processing submenu
        add_submenu_page(
            'aiox-dashboard',
            __('AIOX Processing', 'aiox-publisher-suite-pro'),
            __('📄 Processing', 'aiox-publisher-suite-pro'),
            'manage_options',
            'aiox-capsules',
            array($this, 'render_processing_page')
        );
        
        add_submenu_page(
        'aiox-dashboard',
        __('Badge Customization', 'aiox-publisher-suite-pro'),
        __('🎨 Badge Design', 'aiox-publisher-suite-pro'),
        'manage_options',
        'aiox-badge-design',
        array($this, 'render_badge_design_page')
    );
        
        
    
    }
    
    /**
     * Enqueue admin scripts
     */
    public function admin_enqueue_scripts($hook) {
        if (strpos($hook, 'aiox') === false) {
            return;
        }
        
        // Single lightweight native CSS file
        wp_enqueue_style(
            'aiox-admin-native',
            AIOX_PLUGIN_URL . 'assets/css/admin-native.css',
            array(),
            AIOX_VERSION
        );
        
        // Enqueue capsules-specific scripts only when needed
        if (strpos($hook, 'aiox-capsules') !== false) {
            wp_enqueue_script(
                'aiox-capsules-admin',
                AIOX_PLUGIN_URL . 'assets/js/capsules-admin.js',
                array('jquery'),
                AIOX_VERSION,
                true
            );
            
            wp_localize_script('aiox-capsules-admin', 'aiox_capsules_ajax', array(
                'ajax_url' => admin_url('admin-ajax.php'),
                'nonce' => wp_create_nonce('aiox_capsules_nonce')
            ));
        }
        
        wp_enqueue_script(
            'aiox-admin',
            AIOX_PLUGIN_URL . 'assets/js/admin.js',
            array('jquery'),
            AIOX_VERSION,
            true
        );
        
        wp_localize_script('aiox-admin', 'aiox_ajax', array(
            'ajax_url' => admin_url('admin-ajax.php'),
            'nonce' => wp_create_nonce('aiox_ajax_nonce')
        ));
        
        if ($hook === 'aiox_page_aiox-badge-design') {
        wp_enqueue_style('wp-color-picker');
        wp_enqueue_script('wp-color-picker');
    }
    }
    
    /**
     * Wrapper for dashboard rendering with aggressive output control
     */
    public function render_dashboard_wrapper() {
        // Simply call the render method - do NOT clear output buffers
        // as this destroys the WordPress admin page structure (sidebar, header, etc.)
        $this->render_dashboard();
    }
    
    /**
     * Render dashboard page - WordPress Native Style
     */
    public function render_dashboard() {
        global $wpdb;
        
        // Status checks
        $well_known_ready = is_dir(AIOX_WELL_KNOWN_PATH);
        
        // Check for queue table
        $queue_table = $wpdb->prefix . 'aiox_queue';
        $queue_exists = ($wpdb->get_var("SHOW TABLES LIKE '{$queue_table}'") === $queue_table);
        
        if (!$queue_exists) {
            aiox_analytics_create_tables();
            $queue_exists = ($wpdb->get_var("SHOW TABLES LIKE '{$queue_table}'") === $queue_table);
        }
        
        $database_ready = $queue_exists;
        $queue_count = $queue_exists ? ($wpdb->get_var("SELECT COUNT(*) FROM {$queue_table} WHERE status = 'pending'") ?: 0) : 0;
        $processed_posts = $wpdb->get_var("SELECT COUNT(*) FROM {$wpdb->prefix}postmeta WHERE meta_key = '_aiox_processing_complete' AND meta_value = '1'") ?: 0;
        $last_processed = $wpdb->get_var("SELECT meta_value FROM {$wpdb->prefix}postmeta WHERE meta_key = '_aiox_processed_at' ORDER BY meta_id DESC LIMIT 1");
        $last_activity = $last_processed ? date('Y-m-d H:i', strtotime($last_processed)) : __('Never', 'aiox-publisher-suite-pro');
        ?>
        <div class="wrap">
            <h1><span class="dashicons dashicons-networking" style="font-size: 30px; width: 30px; height: 30px; margin-right: 10px;"></span><?php _e('AIOX Dashboard', 'aiox-publisher-suite-pro'); ?></h1>
            <p class="description"><?php _e('AI Optimization Suite Control Center', 'aiox-publisher-suite-pro'); ?></p>
            
            <!-- Status Cards -->
            <div class="aiox-status-cards">
                <div class="aiox-status-card status-good">
                    <span class="dashicons dashicons-yes-alt"></span>
                    <div>
                        <div class="status-label"><?php _e('Plugin Status', 'aiox-publisher-suite-pro'); ?></div>
                        <div class="status-value"><?php _e('Active', 'aiox-publisher-suite-pro'); ?></div>
                    </div>
                </div>
                
                <div class="aiox-status-card status-good">
                    <span class="dashicons dashicons-admin-plugins"></span>
                    <div>
                        <div class="status-label"><?php _e('Mode', 'aiox-publisher-suite-pro'); ?></div>
                        <div class="status-value"><?php echo $this->is_pro_mode() ? __('Pro', 'aiox-publisher-suite-pro') : __('Free', 'aiox-publisher-suite-pro'); ?></div>
                    </div>
                </div>
                
                <div class="aiox-status-card <?php echo $well_known_ready ? 'status-good' : 'status-error'; ?>">
                    <span class="dashicons dashicons-admin-site-alt3"></span>
                    <div>
                        <div class="status-label"><?php _e('Well-Known Directory', 'aiox-publisher-suite-pro'); ?></div>
                        <div class="status-value"><?php echo $well_known_ready ? __('Ready', 'aiox-publisher-suite-pro') : __('Missing', 'aiox-publisher-suite-pro'); ?></div>
                    </div>
                </div>
                
                <div class="aiox-status-card <?php echo $database_ready ? 'status-good' : 'status-error'; ?>">
                    <span class="dashicons dashicons-database"></span>
                    <div>
                        <div class="status-label"><?php _e('Database Tables', 'aiox-publisher-suite-pro'); ?></div>
                        <div class="status-value"><?php echo $database_ready ? __('Ready', 'aiox-publisher-suite-pro') : __('Missing', 'aiox-publisher-suite-pro'); ?></div>
                    </div>
                </div>
                
                <div class="aiox-status-card status-good">
                    <span class="dashicons dashicons-edit-page"></span>
                    <div>
                        <div class="status-label"><?php _e('Processed Posts', 'aiox-publisher-suite-pro'); ?></div>
                        <div class="status-value"><?php echo esc_html($processed_posts); ?></div>
                    </div>
                </div>
                
                <div class="aiox-status-card status-good">
                    <span class="dashicons dashicons-clock"></span>
                    <div>
                        <div class="status-label"><?php _e('Last Activity', 'aiox-publisher-suite-pro'); ?></div>
                        <div class="status-value"><?php echo esc_html($last_activity); ?></div>
                    </div>
                </div>
            </div>

            <!-- Quick Actions -->
            <div class="postbox">
                <h2 class="hndle"><span class="dashicons dashicons-admin-generic"></span> <?php _e('Quick Actions', 'aiox-publisher-suite-pro'); ?></h2>
                <div class="inside">
                    <div class="aiox-actions-row">
                        <a href="<?php echo admin_url('admin.php?page=aiox-settings'); ?>" class="button button-primary">
                            <span class="dashicons dashicons-admin-settings"></span>
                            <?php _e('Configure Settings', 'aiox-publisher-suite-pro'); ?>
                        </a>
                        
                        <button type="button" id="aiox-generate-manifest" class="button button-secondary">
                            <span class="dashicons dashicons-update"></span>
                            <?php _e('Re-Generate Files', 'aiox-publisher-suite-pro'); ?>
                        </button>
                        
                        <a href="<?php echo admin_url('admin.php?page=aiox-diagnostics'); ?>" class="button button-secondary">
                            <span class="dashicons dashicons-admin-tools"></span>
                            <?php _e('Run Diagnostics', 'aiox-publisher-suite-pro'); ?>
                        </a>
                        
                        <a href="<?php echo admin_url('admin.php?page=aiox-modern-analytics'); ?>" class="button button-secondary">
                            <span class="dashicons dashicons-chart-line"></span>
                            <?php _e('View Analytics', 'aiox-publisher-suite-pro'); ?>
                        </a>
                    </div>
                </div>
            </div>

            <!-- AIOX Files Status -->
            <div class="postbox">
                <h2 class="hndle"><span class="dashicons dashicons-media-document"></span> <?php _e('AIOX Files Status', 'aiox-publisher-suite-pro'); ?></h2>
                <div class="inside">
                    <?php
                    $aiox_files = array(
                        'aio.json' => array('name' => 'AIOX Manifest', 'desc' => 'Main manifest with publisher info'),
                        'ingredients.json' => array('name' => 'Ingredients', 'desc' => 'Processed posts data'),
                        'tdmrep.json' => array('name' => 'TDM-REP', 'desc' => 'Data mining policy'),
                        'aiox-pubkey.pem' => array('name' => 'Public Key', 'desc' => 'Signature verification'),
                        'aiox.ndjson' => array('name' => 'Delta Feed', 'desc' => 'Incremental updates'),
                        'aiox.aiox' => array('name' => 'AIOX Pack', 'desc' => 'Binary content pack')
                    );
                    ?>
                    <table class="widefat striped aiox-files-table">
                        <thead>
                            <tr>
                                <th><?php _e('File', 'aiox-publisher-suite-pro'); ?></th>
                                <th><?php _e('Description', 'aiox-publisher-suite-pro'); ?></th>
                                <th class="column-status"><?php _e('Status', 'aiox-publisher-suite-pro'); ?></th>
                                <th class="column-size"><?php _e('Size', 'aiox-publisher-suite-pro'); ?></th>
                                <th class="column-actions"><?php _e('Actions', 'aiox-publisher-suite-pro'); ?></th>
                            </tr>
                        </thead>
                        <tbody>
                            <?php foreach ($aiox_files as $filename => $info): 
                                $file_path = ABSPATH . '.well-known/' . $filename;
                                $file_exists = file_exists($file_path);
                                $file_url = home_url('/.well-known/' . $filename);
                                $file_size = $file_exists ? filesize($file_path) : 0;
                            ?>
                            <tr>
                                <td>
                                    <strong><?php echo esc_html($info['name']); ?></strong><br>
                                    <code><?php echo esc_html($filename); ?></code>
                                </td>
                                <td><?php echo esc_html($info['desc']); ?></td>
                                <td>
                                    <?php if ($file_exists): ?>
                                        <span class="file-exists"><span class="dashicons dashicons-yes"></span> <?php _e('Exists', 'aiox-publisher-suite-pro'); ?></span>
                                    <?php else: ?>
                                        <span class="file-missing"><span class="dashicons dashicons-no"></span> <?php _e('Missing', 'aiox-publisher-suite-pro'); ?></span>
                                    <?php endif; ?>
                                </td>
                                <td><?php echo $file_exists ? esc_html($this->format_file_size($file_size)) : '—'; ?></td>
                                <td>
                                    <?php if ($file_exists): ?>
                                        <a href="<?php echo esc_url($file_url); ?>" target="_blank" class="button button-small"><?php _e('View', 'aiox-publisher-suite-pro'); ?></a>
                                    <?php else: ?>
                                        —
                                    <?php endif; ?>
                                </td>
                            </tr>
                            <?php endforeach; ?>
                        </tbody>
                    </table>
                    
                    <div class="notice notice-info inline" style="margin-top: 15px;">
                        <p>
                            <span class="dashicons dashicons-info"></span>
                            <?php _e('Files are generated when processing is triggered. Use "Re-Generate Files" to update all AIOX files.', 'aiox-publisher-suite-pro'); ?>
                        </p>
                    </div>
                </div>
            </div>
        </div>
        <?php
    }
    
    /**
     * Render settings page
     */
    public function render_settings_page() {
        // Load the Settings class if not already loaded
        if (!class_exists('AIOX_Settings')) {
            require_once AIOX_PLUGIN_DIR . 'admin/class-aiox-settings.php';
        }
        
        if (class_exists('AIOX_Settings')) {
            // Create a new instance just for rendering
            $settings_page = new AIOX_Settings();
            $settings_page->render_settings_page();
        } else {
            ?>
            <div class="wrap">
                <h1><?php _e('AIOX Settings', 'aiox-publisher-suite-pro'); ?></h1>
                <div class="notice notice-error">
                    <p><?php _e('Settings class not found. Please check plugin installation.', 'aiox-publisher-suite-pro'); ?></p>
                </div>
                <a href="<?php echo admin_url('admin.php?page=aiox-dashboard'); ?>" class="button button-primary">
                    <?php _e('Go to Dashboard', 'aiox-publisher-suite-pro'); ?>
                </a>
            </div>
            <?php
        }
    }
    
    /**
     * Render monetization page
     */
    public function render_monetization_page() {
        // Load the Monetization class if not already loaded
        if (!class_exists('AIOX_Monetization_Page')) {
            require_once AIOX_PLUGIN_DIR . 'admin/class-aiox-monetization-page.php';
        }
        
        if (class_exists('AIOX_Monetization_Page')) {
            $monetization_page = new AIOX_Monetization_Page();
            $monetization_page->render_page();
        } else {
            ?>
            <div class="wrap">
                <h1><?php _e('AI Monetization', 'aiox-publisher-suite-pro'); ?></h1>
                <div class="notice notice-error">
                    <p><?php _e('Monetization class not found. Please check plugin installation.', 'aiox-publisher-suite-pro'); ?></p>
                </div>
                <a href="<?php echo admin_url('admin.php?page=aiox-dashboard'); ?>" class="button button-primary">
                    <?php _e('Go to Dashboard', 'aiox-publisher-suite-pro'); ?>
                </a>
            </div>
            <?php
        }
    }
    
    /**
     * Render diagnostics page
     */
    public function render_diagnostics_page() {
        // Use the global instance if available, otherwise create a new one
        if (isset($GLOBALS['aiox_diagnostics_instance'])) {
            $diagnostics_page = $GLOBALS['aiox_diagnostics_instance'];
        } else {
            // Load the Diagnostics class if not already loaded
            if (!class_exists('AIOX_Diagnostics')) {
                require_once AIOX_PLUGIN_DIR . 'admin/class-aiox-diagnostics.php';
            }
            
            if (class_exists('AIOX_Diagnostics')) {
                $diagnostics_page = new AIOX_Diagnostics();
            } else {
                $diagnostics_page = null;
            }
        }
        
        if ($diagnostics_page) {
            $diagnostics_page->render_diagnostics_page();
        } else {
            ?>
            <div class="wrap">
                <h1><?php _e('AIOX Diagnostics', 'aiox-publisher-suite-pro'); ?></h1>
                <div class="notice notice-error">
                    <p><?php _e('Diagnostics class not found. Please check plugin installation.', 'aiox-publisher-suite-pro'); ?></p>
                </div>
                <a href="<?php echo admin_url('admin.php?page=aiox-dashboard'); ?>" class="button button-primary">
                    <?php _e('Go to Dashboard', 'aiox-publisher-suite-pro'); ?>
                </a>
            </div>
            <?php
        }
    }
    
    /**
     * Render comparison page
     */
    public function render_comparison_page() {
        if (class_exists('AIOX_Comparison')) {
            $comparison = new AIOX_Comparison();
            $comparison->render_comparison_page();
        } else {
            ?>
            <div class="wrap">
                <h1><?php _e('AIOX Processing Comparison', 'aiox-publisher-suite-pro'); ?></h1>
                <div class="notice notice-error">
                    <p><?php _e('Comparison class not found. Please check plugin installation.', 'aiox-publisher-suite-pro'); ?></p>
                </div>
                <a href="<?php echo admin_url('admin.php?page=aiox-dashboard'); ?>" class="button button-primary">
                    <?php _e('Go to Dashboard', 'aiox-publisher-suite-pro'); ?>
                </a>
            </div>
            <?php
        }
    }
    
    /**
     * Phase 1: Render processing page (enhanced capsules functionality)
     */
    public function render_processing_page() {
        try {
            // Phase 1: Use Processing Manager if available
            if (isset($this->processing_manager_instance)) {
                AIOX_Logger::debug('AIOX Phase 1: Using Processing Manager instance for rendering');
                $this->processing_manager_instance->render_page();
            } elseif (class_exists('AIOX_Processing_Manager')) {
                AIOX_Logger::debug('AIOX Phase 1: Creating new Processing Manager instance for rendering');
                $processing_manager = new AIOX_Processing_Manager();
                $processing_manager->render_page();
            } else {
                // Fallback to original capsules functionality
                AIOX_Logger::debug('AIOX Phase 1: Processing Manager not available, falling back to Capsules');
                $this->render_capsules_page_fallback();
            }
        } catch (Exception $e) {
            AIOX_Logger::error('AIOX Phase 1: Failed to render processing page', array('error' => $e->getMessage()));
            $this->render_capsules_page_fallback();
        }
    }
    
    /**
 * Render badge design page
 */
public function render_badge_design_page() {
    // Get current settings with defaults
    $enabled = get_option('aiox_badge_enabled', true);
    $position = get_option('aiox_badge_position', 'bottom-right');
    $shape = get_option('aiox_badge_shape', 'rounded');
    $size = get_option('aiox_badge_size', 'medium');
    $bg_color = get_option('aiox_badge_bg_color', '#667eea');
    $text_color = get_option('aiox_badge_text_color', '#ffffff');
    $title = get_option('aiox_badge_title', 'AI-Enhanced');
    $subtitle = get_option('aiox_badge_subtitle', 'AI Powered');
    
    // Handle form submission
    if (isset($_POST['aiox_save_badge_settings']) && check_admin_referer('aiox_badge_settings_nonce')) {
        update_option('aiox_badge_enabled', isset($_POST['aiox_badge_enabled']));
        update_option('aiox_badge_position', sanitize_text_field($_POST['aiox_badge_position']));
        update_option('aiox_badge_shape', sanitize_text_field($_POST['aiox_badge_shape']));
        update_option('aiox_badge_size', sanitize_text_field($_POST['aiox_badge_size']));
        update_option('aiox_badge_bg_color', sanitize_hex_color($_POST['aiox_badge_bg_color']));
        update_option('aiox_badge_text_color', sanitize_hex_color($_POST['aiox_badge_text_color']));
        update_option('aiox_badge_title', sanitize_text_field($_POST['aiox_badge_title']));
        update_option('aiox_badge_subtitle', sanitize_text_field($_POST['aiox_badge_subtitle']));
        
        // Refresh values
        $enabled = get_option('aiox_badge_enabled');
        $position = get_option('aiox_badge_position');
        $shape = get_option('aiox_badge_shape');
        $size = get_option('aiox_badge_size');
        $bg_color = get_option('aiox_badge_bg_color');
        $text_color = get_option('aiox_badge_text_color');
        $title = get_option('aiox_badge_title');
        $subtitle = get_option('aiox_badge_subtitle');
        
        echo '<div class="notice notice-success is-dismissible"><p><span class="dashicons dashicons-yes-alt" style="color: #00a32a;"></span> ' . __('Badge settings saved successfully!', 'aiox-publisher-suite-pro') . '</p></div>';
    }
    
    ?>
    <div class="wrap">
        <h1>
            <span class="dashicons dashicons-art" style="font-size: 30px; width: 30px; height: 30px; margin-right: 10px; color: #2271b1;"></span>
            <?php _e('Badge Customization', 'aiox-publisher-suite-pro'); ?>
        </h1>
        <p class="description"><?php _e('Customize the appearance of your AI-enhanced badge displayed on processed content.', 'aiox-publisher-suite-pro'); ?></p>
        
        <form method="post">
            <?php wp_nonce_field('aiox_badge_settings_nonce'); ?>
            
            <div id="poststuff">
                <div id="post-body" class="metabox-holder columns-2">
                    <div id="post-body-content">
                        
                        <!-- Display Settings -->
                        <div class="postbox">
                            <h2 class="hndle"><span class="dashicons dashicons-visibility"></span> <?php _e('Display Settings', 'aiox-publisher-suite-pro'); ?></h2>
                            <div class="inside">
                                <ul class="aiox-toggle-list">
                                    <li class="aiox-toggle-item">
                                        <div class="toggle-info">
                                            <span class="toggle-label"><?php _e('Enable Badge', 'aiox-publisher-suite-pro'); ?></span>
                                            <span class="toggle-desc"><?php _e('Show AI badge on processed content', 'aiox-publisher-suite-pro'); ?></span>
                                        </div>
                                        <label class="aiox-switch">
                                            <input type="checkbox" name="aiox_badge_enabled" value="1" <?php checked($enabled, true); ?> />
                                            <span class="slider"></span>
                                        </label>
                                    </li>
                                </ul>
                                
                                <table class="form-table" style="margin-top: 0;">
                                    <tr>
                                        <th scope="row"><label for="aiox_badge_position"><?php _e('Position', 'aiox-publisher-suite-pro'); ?></label></th>
                                        <td>
                                            <select name="aiox_badge_position" id="aiox_badge_position" class="regular-text">
                                                <option value="bottom-right" <?php selected($position, 'bottom-right'); ?>><?php _e('Bottom Right', 'aiox-publisher-suite-pro'); ?></option>
                                                <option value="bottom-left" <?php selected($position, 'bottom-left'); ?>><?php _e('Bottom Left', 'aiox-publisher-suite-pro'); ?></option>
                                                <option value="top-right" <?php selected($position, 'top-right'); ?>><?php _e('Top Right', 'aiox-publisher-suite-pro'); ?></option>
                                                <option value="top-left" <?php selected($position, 'top-left'); ?>><?php _e('Top Left', 'aiox-publisher-suite-pro'); ?></option>
                                            </select>
                                            <p class="description"><?php _e('Where the badge appears on the page.', 'aiox-publisher-suite-pro'); ?></p>
                                        </td>
                                    </tr>
                                </table>
                            </div>
                        </div>
                        
                        <!-- Style Settings -->
                        <div class="postbox">
                            <h2 class="hndle"><span class="dashicons dashicons-admin-customizer"></span> <?php _e('Style Settings', 'aiox-publisher-suite-pro'); ?></h2>
                            <div class="inside">
                                <table class="form-table">
                                    <tr>
                                        <th scope="row"><label for="aiox_badge_shape"><?php _e('Shape', 'aiox-publisher-suite-pro'); ?></label></th>
                                        <td>
                                            <select name="aiox_badge_shape" id="aiox_badge_shape" class="regular-text">
                                                <option value="square" <?php selected($shape, 'square'); ?>><?php _e('Square', 'aiox-publisher-suite-pro'); ?></option>
                                                <option value="rounded" <?php selected($shape, 'rounded'); ?>><?php _e('Rounded', 'aiox-publisher-suite-pro'); ?></option>
                                                <option value="pill" <?php selected($shape, 'pill'); ?>><?php _e('Pill', 'aiox-publisher-suite-pro'); ?></option>
                                                <option value="circle" <?php selected($shape, 'circle'); ?>><?php _e('Circle', 'aiox-publisher-suite-pro'); ?></option>
                                            </select>
                                        </td>
                                    </tr>
                                    <tr>
                                        <th scope="row"><label for="aiox_badge_size"><?php _e('Size', 'aiox-publisher-suite-pro'); ?></label></th>
                                        <td>
                                            <select name="aiox_badge_size" id="aiox_badge_size" class="regular-text">
                                                <option value="small" <?php selected($size, 'small'); ?>><?php _e('Small', 'aiox-publisher-suite-pro'); ?></option>
                                                <option value="medium" <?php selected($size, 'medium'); ?>><?php _e('Medium', 'aiox-publisher-suite-pro'); ?></option>
                                                <option value="large" <?php selected($size, 'large'); ?>><?php _e('Large', 'aiox-publisher-suite-pro'); ?></option>
                                            </select>
                                        </td>
                                    </tr>
                                </table>
                            </div>
                        </div>
                        
                        <!-- Colors -->
                        <div class="postbox">
                            <h2 class="hndle"><span class="dashicons dashicons-admin-appearance"></span> <?php _e('Colors', 'aiox-publisher-suite-pro'); ?></h2>
                            <div class="inside">
                                <table class="form-table">
                                    <tr>
                                        <th scope="row"><label for="aiox_badge_bg_color"><?php _e('Background Color', 'aiox-publisher-suite-pro'); ?></label></th>
                                        <td>
                                            <input type="text" name="aiox_badge_bg_color" id="aiox_badge_bg_color" value="<?php echo esc_attr($bg_color); ?>" class="aiox-color-field" data-default-color="#667eea" />
                                        </td>
                                    </tr>
                                    <tr>
                                        <th scope="row"><label for="aiox_badge_text_color"><?php _e('Text Color', 'aiox-publisher-suite-pro'); ?></label></th>
                                        <td>
                                            <input type="text" name="aiox_badge_text_color" id="aiox_badge_text_color" value="<?php echo esc_attr($text_color); ?>" class="aiox-color-field" data-default-color="#ffffff" />
                                        </td>
                                    </tr>
                                </table>
                            </div>
                        </div>
                        
                        <!-- Content -->
                        <div class="postbox">
                            <h2 class="hndle"><span class="dashicons dashicons-edit"></span> <?php _e('Badge Text', 'aiox-publisher-suite-pro'); ?></h2>
                            <div class="inside">
                                <table class="form-table">
                                    <tr>
                                        <th scope="row"><label for="aiox_badge_title"><?php _e('Title', 'aiox-publisher-suite-pro'); ?></label></th>
                                        <td>
                                            <input type="text" name="aiox_badge_title" id="aiox_badge_title" value="<?php echo esc_attr($title); ?>" class="regular-text" />
                                            <p class="description"><?php _e('Main text displayed on the badge.', 'aiox-publisher-suite-pro'); ?></p>
                                        </td>
                                    </tr>
                                    <tr>
                                        <th scope="row"><label for="aiox_badge_subtitle"><?php _e('Subtitle', 'aiox-publisher-suite-pro'); ?></label></th>
                                        <td>
                                            <input type="text" name="aiox_badge_subtitle" id="aiox_badge_subtitle" value="<?php echo esc_attr($subtitle); ?>" class="regular-text" />
                                            <p class="description"><?php _e('Secondary text displayed below the title.', 'aiox-publisher-suite-pro'); ?></p>
                                        </td>
                                    </tr>
                                </table>
                            </div>
                        </div>
                        
                        <p class="submit">
                            <button type="submit" name="aiox_save_badge_settings" class="button button-primary button-large">
                                <span class="dashicons dashicons-saved" style="vertical-align: middle; margin-right: 5px;"></span>
                                <?php _e('Save Badge Settings', 'aiox-publisher-suite-pro'); ?>
                            </button>
                        </p>
                        
                    </div><!-- /post-body-content -->
                    
                    <!-- Sidebar with Preview -->
                    <div id="postbox-container-1" class="postbox-container">
                        <div class="postbox" style="position: sticky; top: 50px;">
                            <h2 class="hndle"><span class="dashicons dashicons-visibility"></span> <?php _e('Live Preview', 'aiox-publisher-suite-pro'); ?></h2>
                            <div class="inside">
                                <div id="aiox-badge-preview-container" style="background: #f0f0f1; border-radius: 8px; padding: 40px 20px; display: flex; align-items: center; justify-content: center; min-height: 150px;">
                                    <div id="aiox-badge-demo" style="background: <?php echo esc_attr($bg_color); ?>; color: <?php echo esc_attr($text_color); ?>; border-radius: <?php echo $shape === 'circle' ? '50%' : ($shape === 'pill' ? '50px' : ($shape === 'rounded' ? '10px' : '0')); ?>; padding: <?php echo $size === 'large' ? '20px 30px' : ($size === 'small' ? '10px 15px' : '15px 20px'); ?>; text-align: center; cursor: pointer; transition: all 0.3s ease; box-shadow: 0 4px 12px rgba(0,0,0,0.15);">
                                        <div id="badge-title-preview" style="font-size: <?php echo $size === 'large' ? '18px' : ($size === 'small' ? '12px' : '14px'); ?>; font-weight: bold;"><?php echo esc_html($title); ?></div>
                                        <div id="badge-subtitle-preview" style="font-size: <?php echo $size === 'large' ? '14px' : ($size === 'small' ? '10px' : '12px'); ?>; opacity: 0.9; margin-top: 2px;"><?php echo esc_html($subtitle); ?></div>
                                    </div>
                                </div>
                                <p class="description" style="text-align: center; margin-top: 15px;"><?php _e('This is how your badge will appear on posts and pages.', 'aiox-publisher-suite-pro'); ?></p>
                            </div>
                        </div>
                        
                        <div class="postbox">
                            <h2 class="hndle"><?php _e('Quick Tips', 'aiox-publisher-suite-pro'); ?></h2>
                            <div class="inside">
                                <ul style="margin: 0; padding-left: 20px;">
                                    <li><?php _e('Use contrasting colors for better visibility', 'aiox-publisher-suite-pro'); ?></li>
                                    <li><?php _e('Keep text short and readable', 'aiox-publisher-suite-pro'); ?></li>
                                    <li><?php _e('The badge appears on processed content only', 'aiox-publisher-suite-pro'); ?></li>
                                    <li><?php _e('Users can interact with the badge to see Q&A', 'aiox-publisher-suite-pro'); ?></li>
                                </ul>
                            </div>
                        </div>
                    </div>
                    
                </div><!-- /post-body -->
            </div><!-- /poststuff -->
        </form>
    </div>
    
    <script>
    jQuery(document).ready(function($) {
        // Initialize color pickers
        if ($.fn.wpColorPicker) {
            $('.aiox-color-field').wpColorPicker({
                change: function() {
                    setTimeout(updatePreview, 100);
                },
                clear: function() {
                    setTimeout(updatePreview, 100);
                }
            });
        }
        
        // Update preview on any change
        $('select, input[type="text"], input[type="checkbox"]').on('change input', function() {
            updatePreview();
        });
        
        function updatePreview() {
            var bgColor = $('[name="aiox_badge_bg_color"]').val();
            var textColor = $('[name="aiox_badge_text_color"]').val();
            var title = $('[name="aiox_badge_title"]').val();
            var subtitle = $('[name="aiox_badge_subtitle"]').val();
            var shape = $('[name="aiox_badge_shape"]').val();
            var size = $('[name="aiox_badge_size"]').val();
            
            var borderRadius = shape === 'circle' ? '50%' : (shape === 'pill' ? '50px' : (shape === 'rounded' ? '10px' : '0'));
            var padding = size === 'large' ? '20px 30px' : (size === 'small' ? '10px 15px' : '15px 20px');
            var titleSize = size === 'large' ? '18px' : (size === 'small' ? '12px' : '14px');
            var subtitleSize = size === 'large' ? '14px' : (size === 'small' ? '10px' : '12px');
            
            $('#aiox-badge-demo').css({
                'background': bgColor,
                'color': textColor,
                'border-radius': borderRadius,
                'padding': padding
            });
            
            $('#badge-title-preview').text(title).css('font-size', titleSize);
            $('#badge-subtitle-preview').text(subtitle).css('font-size', subtitleSize);
        }
    });
    </script>
    <?php
}
    
    /**
     * Fallback method to render original capsules page
     */
    private function render_capsules_page_fallback() {
        try {
            if (isset($this->capsules_instance)) {
                AIOX_Logger::debug('AIOX Phase 1: Using existing capsules instance for fallback rendering');
                $this->capsules_instance->render_page();
            } elseif (class_exists('AIOX_Capsules_Hybrid')) {
                AIOX_Logger::debug('AIOX Phase 1: Creating new capsules instance for fallback rendering');
                $capsules = new AIOX_Capsules_Hybrid();
                $capsules->render_page();
            } else {
                // Include minimal page as final fallback
                include_once AIOX_PLUGIN_DIR . 'admin/capsules-minimal.php';
            }
        } catch (Exception $e) {
            AIOX_Logger::error('AIOX Phase 1: Failed to render capsules fallback page', array('error' => $e->getMessage()));
            if (file_exists(AIOX_PLUGIN_DIR . 'admin/capsules-minimal.php')) {
                include_once AIOX_PLUGIN_DIR . 'admin/capsules-minimal.php';
            } else {
                echo '<div class="wrap"><h1>AIOX Processing</h1><p>Processing page temporarily unavailable.</p></div>';
            }
        }
    }
    
    /**
     * Register REST API routes
     */
    public function register_rest_routes() {
        register_rest_route('aiox/v1', '/manifest', array(
            'methods' => 'GET',
            'callback' => array($this, 'get_manifest'),
            'permission_callback' => '__return_true'
        ));
    }
    
    /**
     * Add rewrite rules for well-known files
     */
    public function add_rewrite_rules() {
        add_rewrite_rule('^\.well-known/aio\.json$', 'index.php?aiox_file=manifest', 'top');
        add_rewrite_rule('^\.well-known/aiox\.aiox$', 'index.php?aiox_file=binary', 'top');
        
        add_rewrite_tag('%aiox_file%', '([^&]+)');
        add_action('template_redirect', array($this, 'handle_well_known_request'));
    }
    
    /**
     * Handle well-known file requests
     */
    public function handle_well_known_request() {
        // Ensure WordPress is fully initialized before processing
        if (!did_action('wp_loaded')) {
            return;
        }
        
        $file = get_query_var('aiox_file');
        
        if (!$file) {
            return;
        }
        
        switch ($file) {
            case 'manifest':
                $this->serve_manifest_file();
                break;
            case 'binary':
                $this->serve_binary_file();
                break;
        }
    }
    
    /**
     * Serve manifest file
     */
    private function serve_manifest_file() {
        $manifest = $this->generate_basic_manifest();
        
        header('Content-Type: application/json');
        header('Cache-Control: public, max-age=3600');
        
        echo json_encode($manifest, JSON_PRETTY_PRINT | JSON_UNESCAPED_SLASHES);
        exit;
    }
    
    /**
     * Serve binary file
     */
    private function serve_binary_file() {
        $manifest = $this->generate_basic_manifest();
        
        header('Content-Type: application/octet-stream');
        header('Cache-Control: public, max-age=3600');
        
        echo json_encode($manifest);
        exit;
    }
    
    /**
     * Generate basic manifest
     */
    private function generate_basic_manifest() {
        return array(
            'version' => '1.0',
            'generator' => 'AIOX Publisher Suite Pro v' . AIOX_VERSION,
            'site' => array(
                'name' => get_bloginfo('name'),
                'url' => home_url(),
                'description' => get_bloginfo('description')
            ),
            'generated_at' => current_time('c'),
            'content_count' => wp_count_posts()->publish,
            'endpoints' => array(
                'manifest' => home_url('/.well-known/aio.json'),
                'binary' => home_url('/.well-known/aiox.aiox'),
                'api' => home_url('/wp-json/aiox/v1/')
            )
        );
    }
    
    /**
     * Get manifest via REST API
     */
    public function get_manifest() {
        return rest_ensure_response($this->generate_basic_manifest());
    }
    
    /**
     * Create database tables
     */
    private function create_database_tables() {
        global $wpdb;
        
        $charset_collate = $wpdb->get_charset_collate();
        
        // Queue table
        $table_name = $wpdb->prefix . 'aiox_queue';
        $sql = "CREATE TABLE $table_name (
            id mediumint(9) NOT NULL AUTO_INCREMENT,
            post_id bigint(20) NOT NULL,
            status varchar(20) DEFAULT 'pending',
            priority int(11) DEFAULT 10,
            created_at datetime DEFAULT CURRENT_TIMESTAMP,
            updated_at datetime DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
            PRIMARY KEY (id),
            KEY post_id (post_id),
            KEY status (status)
        ) $charset_collate;";
        
        if (file_exists(ABSPATH . 'wp-admin/includes/upgrade.php')) {
            require_once(ABSPATH . 'wp-admin/includes/upgrade.php');
        }
        dbDelta($sql);
        
        // Capsules table
        $table_name = $wpdb->prefix . 'aiox_capsules';
        $sql = "CREATE TABLE $table_name (
            id varchar(100) NOT NULL,
            title varchar(255) NOT NULL,
            description text,
            type varchar(50) DEFAULT 'post',
            status varchar(20) DEFAULT 'draft',
            metadata longtext,
            content longtext,
            file_path varchar(500),
            file_size bigint(20) DEFAULT 0,
            content_hash varchar(64),
            created_at datetime DEFAULT CURRENT_TIMESTAMP,
            updated_at datetime DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
            author_id bigint(20) DEFAULT 0,
            PRIMARY KEY (id),
            KEY type (type),
            KEY status (status),
            KEY author_id (author_id),
            KEY created_at (created_at)
        ) $charset_collate;";
        
        dbDelta($sql);
    }
    
    /**
     * Create well-known directory
     */
    private function create_well_known_directory() {
        if (!is_dir(AIOX_WELL_KNOWN_PATH)) {
            wp_mkdir_p(AIOX_WELL_KNOWN_PATH);
        }
        
        if (!is_dir(AIOX_WELL_KNOWN_PATH . 'capsules/')) {
            wp_mkdir_p(AIOX_WELL_KNOWN_PATH . 'capsules/');
        }
    }
    
    /**
     * Set default options
     */
    private function set_default_options() {
        $defaults = array(
            'license_key' => '',
            'ai_endpoint' => AIOX_DEFAULT_AI_ENDPOINT,
            'auto_process' => 1,
            'enable_badge' => 1,
            'mode' => 'free',
            'use_simplified_structure' => true,
            'structure_version' => 'simplified',
            // Processing Settings
            'enable_comprehensive_processing' => true,
            'auto_extract_ingredients' => true,
            'auto_classify_metadata' => true,
            'auto_update_manifesto' => true,
            'auto_update_badges' => true,
            'auto_update_metadata' => true,
            'enable_background_processing' => false,
            'processing_delay' => 0,
            'batch_size' => 5
        );
        
        foreach ($defaults as $key => $value) {
            if (get_option('aiox_' . $key) === false) {
                update_option('aiox_' . $key, $value);
            }
        }
    }
    
    /**
     * Plugin uninstallation
     * 
     * Intentionally preserves all data so reinstallation restores everything.
     * Use "Total Reset" in Settings to remove all data before uninstalling.
     */
    public static function uninstall() {
        // Clear cache only - preserve all data for reinstallation
        wp_cache_flush();
        
        // ALL DATA IS PRESERVED:
        // - Database tables (aiox_queue, aiox_capsules, aiox_logs)
        // - Plugin options and settings
        // - Post meta (_aiox_* keys)
        // - .well-known files (manifest, capsules, etc.)
        // - Upload directory files
        //
        // Use "Total Reset" in AIOX Settings to remove all data before uninstalling
    }

    /**
     * Clean up AIOX files
     */
    private function cleanup_files() {
        $well_known_path = ABSPATH . '.well-known/';
        $files_to_delete = array(
            'aio.json',
            'tdmrep.json',
            'aiox-pubkey.pem',
            'aiox.ndjson',
            'aiox.aiox',
            'ai-info.json',
            'ingredients.json',
            'aiox-proof.json',
        );

        // Also check for post-specific AI info files
        $ai_info_files = glob($well_known_path . 'ai-info-*.json');
        if ($ai_info_files) {
            foreach ($ai_info_files as $ai_file) {
                $files_to_delete[] = basename($ai_file);
            }
        }

        $deleted_files = array();
        $failed_files = array();

        foreach ($files_to_delete as $file) {
            $file_path = $well_known_path . $file;
            if (file_exists($file_path)) {
                if (unlink($file_path)) {
                    $deleted_files[] = $file;
                } else {
                    $failed_files[] = $file;
                }
            }
        }

        // Delete capsules directory if it exists
        $capsules_path = $well_known_path . 'capsules/';
        if (is_dir($capsules_path)) {
            if (self::delete_directory_recursive($capsules_path)) {
                $deleted_files[] = 'capsules/';
            } else {
                $failed_files[] = 'capsules/';
            }
        }
        
        // Delete capsules upload directory if it exists
        if (function_exists('wp_upload_dir')) {
            $upload_dir = wp_upload_dir();
            $capsules_upload_path = $upload_dir['basedir'] . '/aiox-capsules/';
            if (is_dir($capsules_upload_path)) {
                if (self::delete_directory_recursive($capsules_upload_path)) {
                    $deleted_files[] = 'uploads/aiox-capsules/ (legacy)';
                } else {
                    $failed_files[] = 'uploads/aiox-capsules/ (legacy)';
                }
            }
        }
        
        // Delete ALL capsules database records on deactivation
        global $wpdb;
        if ($wpdb) {
            $table_name = $wpdb->prefix . 'aiox_capsules';
            if ($wpdb->get_var("SHOW TABLES LIKE '{$table_name}'") == $table_name) {
                $deleted_count = $wpdb->query("DELETE FROM {$table_name}");
                $wpdb->query("DROP TABLE IF EXISTS {$table_name}");
                $deleted_files[] = "aiox_capsules table ({$deleted_count} records)";
            }
        }

        // Log cleanup results
        if (class_exists('AIOX_Logger')) {
            $logger = new AIOX_Logger();
            if (!empty($deleted_files)) {
                $logger->log('info', 'Cleanup: Deleted files: ' . implode(', ', $deleted_files));
            }
            if (!empty($failed_files)) {
                $logger->log('warning', 'Cleanup: Failed to delete files: ' . implode(', ', $failed_files));
            }
        }
    }

    /**
     * Static version of cleanup_files for uninstall hook
     */
    private static function cleanup_files_static() {
        $well_known_path = ABSPATH . '.well-known/';
        $files_to_delete = array(
            'aio.json',
            'tdmrep.json',
            'aiox-pubkey.pem',
            'aiox.ndjson',
            'aiox.aiox',
            'ai-info.json',
            'ingredients.json',
            'aiox-proof.json',
        );

        // Also check for post-specific AI info files
        $ai_info_files = glob($well_known_path . 'ai-info-*.json');
        if ($ai_info_files) {
            foreach ($ai_info_files as $ai_file) {
                $files_to_delete[] = basename($ai_file);
            }
        }

        $deleted_count = 0;
        $failed_count = 0;

        foreach ($files_to_delete as $file) {
            $file_path = $well_known_path . $file;
            if (file_exists($file_path)) {
                if (@unlink($file_path)) {
                    $deleted_count++;
                } else {
                    $failed_count++;
                }
            }
        }

        // Delete capsules directory if it exists
        $capsules_path = $well_known_path . 'capsules/';
        if (is_dir($capsules_path)) {
            if (self::delete_directory_recursive($capsules_path)) {
                $deleted_count++;
            } else {
                $failed_count++;
            }
        }
        
        // Delete capsules upload directory if it exists
        $upload_dir = wp_upload_dir();
        if ($upload_dir && !is_wp_error($upload_dir)) {
            $capsules_upload_path = $upload_dir['basedir'] . '/aiox-capsules/';
            if (is_dir($capsules_upload_path)) {
                if (self::delete_directory_recursive($capsules_upload_path)) {
                    $deleted_count++;
                } else {
                    $failed_count++;
                }
            }
            
            // Delete AIOX upload directory if it exists
            $aiox_upload_path = $upload_dir['basedir'] . '/aiox/';
            if (is_dir($aiox_upload_path)) {
                if (self::delete_directory_recursive($aiox_upload_path)) {
                    $deleted_count++;
                } else {
                    $failed_count++;
                }
            }
        }
    }

    /**
     * Fix ingredients during activation
     */
    private function fix_ingredients_during_activation() {
        try {
            // Only run if ingredients class is available
            if (!class_exists('AIOX_Ingredients')) {
                $ingredients_file = AIOX_PLUGIN_DIR . 'core/class-aiox-ingredients.php';
                if (file_exists($ingredients_file)) {
                    require_once $ingredients_file;
                }
            }
            
            if (!class_exists('AIOX_Ingredients')) {
                return;
            }
            
            // Get posts with ingredients that might be corrupted
            $posts_with_ingredients = get_posts(array(
                'post_type' => array('post', 'page'),
                'post_status' => 'publish',
                'numberposts' => -1,
                'meta_query' => array(
                    array(
                        'key' => '_aiox_ingredients',
                        'compare' => 'EXISTS'
                    )
                )
            ));
            
            $fixed_count = 0;
            
            foreach ($posts_with_ingredients as $post) {
                $raw_data = get_post_meta($post->ID, '_aiox_ingredients', true);
                
                if (is_string($raw_data)) {
                    $decoded = json_decode($raw_data, true);
                    
                    if (json_last_error() !== JSON_ERROR_NONE) {
                        // Create comprehensive ingredients
                        $content = $post->post_content;
                        $title = $post->post_title;
                        
                        $word_count = str_word_count(strip_tags($content));
                        $char_count = strlen(strip_tags($content));
                        $paragraph_count = substr_count($content, '</p>');
                        $heading_count = preg_match_all('/<h[1-6][^>]*>/', $content);
                        $list_count = substr_count($content, '<li>');
                        $link_count = substr_count($content, '<a ');
                        
                        $comprehensive_ingredients = array(
                            array('name' => 'Title', 'value' => $title),
                            array('name' => 'Content Type', 'value' => ucfirst($post->post_type) . ' Page'),
                            array('name' => 'Post Type', 'value' => $post->post_type),
                            array('name' => 'Post Status', 'value' => $post->post_status),
                            array('name' => 'Word Count', 'value' => $word_count),
                            array('name' => 'Character Count', 'value' => $char_count),
                            array('name' => 'Paragraph Count', 'value' => $paragraph_count),
                            array('name' => 'Heading Count', 'value' => $heading_count),
                            array('name' => 'List Items', 'value' => $list_count),
                            array('name' => 'Links', 'value' => $link_count),
                            array('name' => 'Reading Time', 'value' => ceil($word_count / 200) . ' minutes'),
                            array('name' => 'Language', 'value' => 'English'),
                            array('name' => 'Author', 'value' => get_the_author_meta('display_name', $post->post_author)),
                            array('name' => 'Published Date', 'value' => $post->post_date),
                            array('name' => 'Last Modified', 'value' => $post->post_modified),
                        );
                        
                        // Save with proper JSON encoding
                        $json_data = json_encode($comprehensive_ingredients, JSON_UNESCAPED_SLASHES | JSON_UNESCAPED_UNICODE);
                        update_post_meta($post->ID, '_aiox_ingredients', $json_data);
                        update_post_meta($post->ID, '_aiox_ingredients_backup', $json_data);
                        
                        $fixed_count++;
                    }
                }
            }
            
            // Export ingredients file after fixes
            if ($fixed_count > 0 && class_exists('AIOX_Ingredients')) {
                AIOX_Ingredients::export_ingredients_file();
            }
            
            // Log the fix
            
        } catch (Exception $e) {
            error_log('AIOX: Error fixing ingredients during activation: ' . $e->getMessage());
        }
    }

    /**
     * Fix ingredients database - scheduled action
     */
    public function fix_ingredients_database() {
        // Only run if ingredients class is available
        if (!class_exists('AIOX_Ingredients')) {
            return;
        }
        
        // Get posts with ingredients that might be corrupted
        $posts_with_ingredients = get_posts(array(
            'post_type' => array('post', 'page'),
            'post_status' => 'publish',
            'numberposts' => -1,
            'meta_query' => array(
                array(
                    'key' => '_aiox_ingredients',
                    'compare' => 'EXISTS'
                )
            )
        ));
        
        $fixed_count = 0;
        
        foreach ($posts_with_ingredients as $post) {
            $raw_data = get_post_meta($post->ID, '_aiox_ingredients', true);
            
            if (is_string($raw_data)) {
                $decoded = json_decode($raw_data, true);
                
                if (json_last_error() !== JSON_ERROR_NONE) {
                    // Create comprehensive ingredients
                    $content = $post->post_content;
                    $title = $post->post_title;
                    
                    $word_count = str_word_count(strip_tags($content));
                    $char_count = strlen(strip_tags($content));
                    $paragraph_count = substr_count($content, '</p>');
                    $heading_count = preg_match_all('/<h[1-6][^>]*>/', $content);
                    $list_count = substr_count($content, '<li>');
                    $link_count = substr_count($content, '<a ');
                    
                    $comprehensive_ingredients = array(
                        array('name' => 'Title', 'value' => $title),
                        array('name' => 'Content Type', 'value' => ucfirst($post->post_type) . ' Page'),
                        array('name' => 'Post Type', 'value' => $post->post_type),
                        array('name' => 'Post Status', 'value' => $post->post_status),
                        array('name' => 'Word Count', 'value' => $word_count),
                        array('name' => 'Character Count', 'value' => $char_count),
                        array('name' => 'Paragraph Count', 'value' => $paragraph_count),
                        array('name' => 'Heading Count', 'value' => $heading_count),
                        array('name' => 'List Items', 'value' => $list_count),
                        array('name' => 'Links', 'value' => $link_count),
                        array('name' => 'Reading Time', 'value' => ceil($word_count / 200) . ' minutes'),
                        array('name' => 'Language', 'value' => 'English'),
                        array('name' => 'Author', 'value' => get_the_author_meta('display_name', $post->post_author)),
                        array('name' => 'Published Date', 'value' => $post->post_date),
                        array('name' => 'Last Modified', 'value' => $post->post_modified),
                    );
                    
                    // Save with proper JSON encoding
                    $json_data = json_encode($comprehensive_ingredients, JSON_UNESCAPED_SLASHES | JSON_UNESCAPED_UNICODE);
                    update_post_meta($post->ID, '_aiox_ingredients', $json_data);
                    update_post_meta($post->ID, '_aiox_ingredients_backup', $json_data);
                    
                    $fixed_count++;
                }
            }
        }
        
        // Export ingredients file after fixes
        if (class_exists('AIOX_Ingredients') && $fixed_count > 0) {
            AIOX_Ingredients::export_ingredients_file();
        }
        
        // Log the fix
    }

    /**
     * Recursively delete a directory and all its contents
     */
    private static function delete_directory_recursive($dir) {
        if (!is_dir($dir)) {
            return false;
        }

        $files = array_diff(scandir($dir), array('.', '..'));
        foreach ($files as $file) {
            $path = $dir . DIRECTORY_SEPARATOR . $file;
            if (is_dir($path)) {
                self::delete_directory_recursive($path);
            } else {
                @unlink($path);
            }
        }
        return @rmdir($dir);
    }
    
    /**
     * Check if plugin is in pro mode
     */
    private function is_pro_mode() {
        $license_key = get_option('aiox_license_key', '');
        return !empty($license_key);
    }
}

/**
 * Initialize the plugin
 */
function aiox_publisher_suite_pro() {
    return AIOX_Publisher_Suite_Pro::get_instance();
}

// Start the plugin on plugins_loaded to ensure WordPress is fully initialized
add_action('plugins_loaded', 'aiox_publisher_suite_pro', 0);

/**
 * Add action links to plugin page
 */
function aiox_add_action_links($links) {
    $settings_link = '<a href="' . admin_url('admin.php?page=aiox-dashboard') . '">' . __('Dashboard', 'aiox-publisher-suite-pro') . '</a>';
    array_unshift($links, $settings_link);
    return $links;
}
add_filter('plugin_action_links_' . AIOX_PLUGIN_BASENAME, 'aiox_add_action_links');

// Auto version check + migration
function aiox_publisher_check_version() {
    $installed_version = get_option('aiox_publisher_plugin_version');
    if ($installed_version !== AIOX_PLUGIN_VERSION) {
        if (class_exists('AIOX_Publisher_Installer')) {
            AIOX_Publisher_Installer::install();
        }
        update_option('aiox_publisher_plugin_version', AIOX_PLUGIN_VERSION);
    }
}
add_action('plugins_loaded', 'aiox_publisher_check_version', 5);