/**
 * AIOX Publisher Suite Pro - Minimal Admin JavaScript
 */

jQuery(document).ready(function($) {
    
    // Clean up nested error messages to show only the core message
    function cleanErrorMessage(msg) {
        if (!msg) return 'Unknown error';
        
        // Check for any token/usage limit related keywords and return clean message
        var tokenKeywords = ['token limit', 'usage limit', 'insufficient token', 'no purchased tokens', 'monthly token', 'token balance'];
        var msgLower = msg.toLowerCase();
        
        for (var i = 0; i < tokenKeywords.length; i++) {
            if (msgLower.indexOf(tokenKeywords[i]) !== -1) {
                return 'Insufficient token balance for this operation. Please purchase more tokens or wait for the next billing cycle.';
            }
        }
        
        // Remove repeated "Processing failed:" prefixes for non-token errors
        msg = msg.replace(/Processing failed:\s*/gi, '');
        msg = msg.replace(/AI processing failed:\s*/gi, '');
        msg = msg.replace(/AI content processing failed:\s*/gi, '');
        msg = msg.replace(/Please check your API configuration\.\s*/gi, '');
        
        // Clean up any double spaces or periods
        msg = msg.replace(/\s+/g, ' ').replace(/\.+/g, '.').trim();
        
        // If message is now empty, return generic error
        if (!msg || msg === '.') {
            return 'Processing failed. Please try again.';
        }
        
        return msg;
    }
    
    // Generate manifest button
    $('#aiox-generate-manifest').on('click', function(e) {
        e.preventDefault();
        
        var $button = $(this);
        var originalText = $button.text();
        
        $button.text('Generating...').prop('disabled', true);
        
        $.ajax({
            url: aiox_ajax.ajax_url,
            type: 'POST',
            data: {
                action: 'aiox_generate_manifest',
                nonce: aiox_ajax.nonce
            },
            success: function(response) {
                if (response.success) {
                    showNotice('All AIOX files regenerated successfully!', 'success');
                    // Update file status in real-time
                    updateFileStatus();
                } else {
                    showNotice('Failed to regenerate files: ' + (response.data || 'Unknown error'), 'error');
                }
            },
            error: function() {
                showNotice('Failed to generate manifest. Please try again.', 'error');
            },
            complete: function() {
                $button.text(originalText).prop('disabled', false);
            }
        });
    });
    
    // Process queue button
    $('#aiox-process-queue').on('click', function(e) {
        e.preventDefault();
        
        var $button = $(this);
        var originalText = $button.text();
        
        $button.text('Processing...').prop('disabled', true);
        
        $.ajax({
            url: aiox_ajax.ajax_url,
            type: 'POST',
            data: {
                action: 'aiox_process_queue_now',
                nonce: aiox_ajax.nonce
            },
            success: function(response) {
                if (response.success) {
                    showNotice('Queue processed successfully!', 'success');
                    // Reload page to update debug info
                    setTimeout(function() {
                        location.reload();
                    }, 1000);
                } else {
                    showNotice('Failed to process queue: ' + (response.data || 'Unknown error'), 'error');
                }
            },
            error: function() {
                showNotice('Failed to process queue. Please try again.', 'error');
            },
            complete: function() {
                $button.text(originalText).prop('disabled', false);
            }
        });
    });
    

    
    // Create AI info page button
    $('#aiox-create-ai-info-page').on('click', function(e) {
        e.preventDefault();
        
        var $button = $(this);
        var originalText = $button.text();
        
        $button.text('Creating...').prop('disabled', true);
        
        $.ajax({
            url: aiox_ajax.ajax_url,
            type: 'POST',
            data: {
                action: 'aiox_create_ai_info_page',
                nonce: aiox_ajax.nonce
            },
            success: function(response) {
                if (response.success) {
                    showNotice('AI info page created successfully! ' + response.data, 'success');
                    // Reload page to update debug info
                    setTimeout(function() {
                        location.reload();
                    }, 2000);
                } else {
                    showNotice('Failed to create AI info page: ' + (response.data || 'Unknown error'), 'error');
                }
            },
            error: function() {
                showNotice('Failed to create AI info page. Please try again.', 'error');
            },
            complete: function() {
                $button.text(originalText).prop('disabled', false);
            }
        });
    });
    
    // Update file status in real-time
    function updateFileStatus() {
        $.ajax({
            url: ajaxurl,
            type: 'POST',
            data: {
                action: 'aiox_get_file_status',
                nonce: aiox_ajax.nonce
            },
            success: function(response) {
                if (response.success && response.data) {
                    // Update each file status
                    $.each(response.data, function(filename, fileInfo) {
                        var $row = $('.aiox-file-status tr[data-file="' + filename + '"]');
                        if ($row.length) {
                            var statusHtml = fileInfo.exists ? 
                                '<span class="aiox-status-exists">✅ Exists<br><small>' + fileInfo.size + '</small></span>' :
                                '<span class="aiox-status-missing">❌ Missing</span>';
                            
                            var actionHtml = fileInfo.exists ? 
                                '<a href="' + fileInfo.url + '" target="_blank" class="button button-small">View</a>' :
                                '—';
                            
                            $row.find('.aiox-file-status-cell').html(statusHtml);
                            $row.find('.aiox-file-actions-cell').html(actionHtml);
                        }
                    });
                }
            }
        });
    }
    
    // Show admin notice
    function showNotice(message, type) {
        // Remove any existing AIOX notices
        $('.aiox-floating-notice').remove();
        
        var bgColor, borderColor, textColor;
        if (type === 'success') {
            bgColor = '#f0fdf4';
            borderColor = '#22c55e';
            textColor = '#166534';
        } else if (type === 'warning') {
            bgColor = '#fffbeb';
            borderColor = '#f59e0b';
            textColor = '#92400e';
        } else {
            bgColor = '#fef2f2';
            borderColor = '#ef4444';
            textColor = '#991b1b';
        }
        
        var $notice = $('<div class="aiox-floating-notice" style="position: fixed; top: 50px; right: 20px; max-width: 500px; padding: 15px 40px 15px 20px; border-radius: 8px; border-left: 4px solid ' + borderColor + '; background: ' + bgColor + '; z-index: 99999; box-shadow: 0 4px 12px rgba(0,0,0,0.3);"><span style="position: absolute; top: 8px; right: 12px; cursor: pointer; font-size: 20px; color: ' + textColor + ';" class="aiox-notice-close">×</span><p style="margin: 0; color: ' + textColor + '; font-size: 14px; line-height: 1.5;">' + message + '</p></div>');
        
        $('body').append($notice);
        
        // Close on click
        $notice.find('.aiox-notice-close').on('click', function() {
            $notice.fadeOut(function() { $(this).remove(); });
        });
        
        // Auto-dismiss success after 5 seconds, errors stay longer
        var dismissTime = type === 'success' ? 5000 : 10000;
        setTimeout(function() {
            $notice.fadeOut(function() { $(this).remove(); });
        }, dismissTime);
    }
    
    // Handle dismissible notices
    $(document).on('click', '.notice-dismiss', function() {
        $(this).parent().fadeOut();
    });
    
    // Content Management functionality
    function loadContentList(showUnprocessedOnly = false) {
        const container = $('#aiox-content-list-container');
        
        if (container.length === 0) {
            return; // Not on content management page
        }
        
        container.html('<div class="aiox-loading">Loading content list...</div>');
        
        $.ajax({
            url: aiox_ajax.ajax_url,
            type: 'POST',
            data: {
                action: 'aiox_get_content_list',
                nonce: aiox_ajax.nonce,
                show_unprocessed_only: showUnprocessedOnly ? 'true' : 'false'
            },
            success: function(response) {
                if (response.success && response.data.content) {
                    renderContentTable(response.data.content, response.data.count);
                } else {
                    container.html('<div class="notice notice-error"><p>Failed to load content list: ' + (response.data.message || 'Unknown error') + '</p></div>');
                }
            },
            error: function(xhr, status, error) {
                container.html('<div class="notice notice-error"><p>AJAX Error: ' + error + '</p></div>');
            }
        });
    }
    
    function renderContentTable(content, count) {
        const container = $('#aiox-content-list-container');
        
        if (content.length === 0) {
            container.html('<div class="notice notice-info"><p>No content found.</p></div>');
            return;
        }
        
        let html = '<div class="aiox-content-summary" style="margin-bottom: 15px;">';
        html += '<strong>📊 ' + count + ' items found</strong>';
        html += '</div>';
        
        html += '<table class="wp-list-table widefat fixed striped aiox-content-table">';
        html += '<thead>';
        html += '<tr>';
        html += '<td class="manage-column column-cb check-column"><input type="checkbox" id="aiox-select-all" /></td>';
        html += '<th class="manage-column">Title</th>';
        html += '<th class="manage-column">Type</th>';
        html += '<th class="manage-column">Status</th>';
        html += '<th class="manage-column">Ingredients</th>';
        html += '<th class="manage-column">Components</th>';
        html += '<th class="manage-column">Actions</th>';
        html += '</tr>';
        html += '</thead>';
        html += '<tbody>';
        
        content.forEach(function(item) {
            html += '<tr>';
            html += '<th class="check-column"><input type="checkbox" name="post_ids[]" value="' + item.id + '" /></th>';
            html += '<td><strong><a href="' + item.url + '" target="_blank">' + item.title + '</a></strong>';
            html += '<div class="row-actions"><span class="edit"><a href="' + item.edit_url + '">Edit</a></span></div></td>';
            html += '<td>' + item.type + '</td>';
            
            if (item.status === 'processed') {
                html += '<td><span style="color: green;">✅ Processed</span></td>';
            } else {
                html += '<td><span style="color: orange;">⏳ Unprocessed</span></td>';
            }
            
            // Build clickable ingredients display
            let ingredientsDisplay = '';
            if (item.ingredients_count > 0) {
                let ingredientsUrl = window.location.origin + '/.well-known/posts/' + item.id + '.json';
                ingredientsDisplay = '<a href="' + ingredientsUrl + '" target="_blank" title="View ingredients file (' + item.ingredients_count + ' ingredients)" style="text-decoration: none; font-size: 18px; margin-right: 5px;">🧩</a> ';
                ingredientsDisplay += item.ingredients_count + ' items';
            } else {
                ingredientsDisplay = 'None';
            }
            html += '<td>' + ingredientsDisplay + '</td>';
            
            // Build clickable components with file links
            let components = '';
            
            // Capsule icon - show if capsule_id exists
            if (item.capsule_id && item.capsule_id !== '' && item.capsule_id !== '0') {
                let capsuleUrl = window.location.origin + '/.well-known/capsules/' + item.capsule_id + '.json';
                components += '<a href="' + capsuleUrl + '" target="_blank" title="View capsule file" style="text-decoration: none; font-size: 18px; margin-right: 5px;">📦</a>';
            }
            
            // Manifesto/Post data file icon
            if (item.has_manifesto) {
                let manifestoUrl = window.location.origin + '/.well-known/posts/' + item.id + '.json';
                components += '<a href="' + manifestoUrl + '" target="_blank" title="View manifesto file" style="text-decoration: none; font-size: 18px; margin-right: 5px;">📄</a>';
            }
            
            // Metadata icon - show count on hover
            if (item.has_metadata && item.metadata_count > 0) {
                components += '<span title="' + item.metadata_count + ' metadata fields" style="font-size: 18px; margin-right: 5px;">🏷️</span>';
            }
            
            // Badge icon
            if (item.has_badge) {
                components += '<span title="Has badge data" style="font-size: 18px; margin-right: 5px;">🏆</span>';
            }
            
            html += '<td>' + (components || 'None') + '</td>';
            
            if (item.status === 'processed') {
                html += '<td>';
                html += '<button type="button" class="button button-small aiox-reprocess-post" data-post-id="' + item.id + '" style="margin-right: 5px;">Reprocess</button>';
                html += '<button type="button" class="button button-small button-secondary aiox-clear-post" data-post-id="' + item.id + '">Clear AI Data</button>';
                html += '</td>';
            } else {
                html += '<td>';
                html += '<button type="button" class="button button-primary button-small aiox-process-post" data-post-id="' + item.id + '">Process</button>';
                html += '</td>';
            }
            
            html += '</tr>';
        });
        
        html += '</tbody>';
        html += '</table>';
        
        container.html(html);
        
        // Bind events for the new table
        bindContentTableEvents();
    }
    
    function bindContentTableEvents() {
        // Select all checkbox
        $('#aiox-select-all').off('change').on('change', function() {
            $('input[name="post_ids[]"]').prop('checked', $(this).prop('checked'));
        });
        
        // Individual process buttons
        $('.aiox-process-post, .aiox-reprocess-post').off('click').on('click', function() {
            const postId = $(this).data('post-id');
            const button = $(this);
            const originalText = button.text();
            
            button.prop('disabled', true).text('Processing...');
            
            // Trigger comprehensive processing for the post
            $.ajax({
                url: aiox_ajax.ajax_url,
                type: 'POST',
                data: {
                    action: 'aiox_process_all_components',
                    nonce: aiox_ajax.nonce,
                    post_id: postId
                },
                success: function(response) {
                    // Always refresh token balance after processing
                    if (typeof aioxRefreshBalance === 'function') {
                        aioxRefreshBalance();
                    }
                    
                    if (response.success) {
                        button.text('✅ Done').css('color', 'green');
                        showNotice('Processing complete! ' + (response.data.message || ''), 'success');
                        setTimeout(function() {
                            loadContentList($('#aiox-show-unprocessed-only').prop('checked'));
                        }, 1500);
                    } else {
                        button.prop('disabled', false).text(originalText);
                        var errorMsg = response.data && response.data.message ? response.data.message : 'Unknown error';
                        
                        // Clean up nested error messages - extract the core message
                        errorMsg = cleanErrorMessage(errorMsg);
                        
                        // Check for token-related errors
                        if (errorMsg.indexOf('Token') !== -1 || errorMsg.indexOf('token') !== -1 || errorMsg.indexOf('Insufficient') !== -1 || errorMsg.indexOf('limit') !== -1 || errorMsg.indexOf('Usage limit') !== -1) {
                            showNotice('⚠️ ' + errorMsg, 'warning');
                        } else {
                            showNotice('❌ Processing Failed: ' + errorMsg, 'error');
                        }
                    }
                },
                error: function(xhr, status, error) {
                    button.prop('disabled', false).text(originalText);
                    
                    // Try to refresh balance even on error
                    if (typeof aioxRefreshBalance === 'function') {
                        aioxRefreshBalance();
                    }
                    
                    var errorMsg = 'Connection error';
                    if (xhr.responseJSON && xhr.responseJSON.data && xhr.responseJSON.data.message) {
                        errorMsg = xhr.responseJSON.data.message;
                    }
                    showNotice('❌ Request Failed: ' + errorMsg, 'error');
                }
            });
        });
        
        // Clear AI Data buttons
        $('.aiox-clear-post').off('click').on('click', function() {
            const postId = $(this).data('post-id');
            const button = $(this);
            const originalText = button.text();
            
            if (!confirm('Are you sure you want to clear all AI data for this post? This action cannot be undone.')) {
                return;
            }
            
            button.prop('disabled', true).text('Clearing...');
            
            // Trigger clear AI data for the post
            $.ajax({
                url: aiox_ajax.ajax_url,
                type: 'POST',
                data: {
                    action: 'aiox_clear_post_data',
                    nonce: aiox_ajax.nonce,
                    post_id: postId
                },
                success: function(response) {
                    if (response.success) {
                        button.text('✅ Cleared').css('color', 'green');
                        showNotice('AI data cleared successfully!', 'success');
                        setTimeout(function() {
                            loadContentList($('#aiox-show-unprocessed-only').prop('checked'));
                        }, 1000);
                    } else {
                        button.prop('disabled', false).text(originalText);
                        var errorMsg = response.data && response.data.message ? response.data.message : 'Unknown error';
                        showNotice('❌ Clear Failed: ' + errorMsg, 'error');
                    }
                },
                error: function(xhr, status, error) {
                    button.prop('disabled', false).text(originalText);
                    showNotice('❌ Request Failed: Connection error', 'error');
                }
            });
        });
    }
    
    // Event handlers for content management
    $('#aiox-refresh-content').on('click', function() {
        loadContentList($('#aiox-show-unprocessed-only').prop('checked'));
    });
    
    $('#aiox-show-unprocessed-only').on('change', function() {
        loadContentList($(this).prop('checked'));
    });
    
    // Load content list on page load if container exists
    if ($('#aiox-content-list-container').length > 0) {
        loadContentList();
    }
    
});