/**
 * AIOX AI Badge JavaScript
 */

jQuery(document).ready(function($) {
    
    var badge = $('.aiox-badge'); // Use class selector to handle dynamic IDs
    
    if (badge.length === 0) {
        console.log('AIOX: No badge found on page');
        return; // Exit if no badge found
    }
    
    console.log('AIOX: Badge found, initializing...', badge);
    
    var trigger = badge.find('.aiox-badge-trigger');
    var content = badge.find('.aiox-badge-content');
    var closeBtn = badge.find('.aiox-badge-close');
    var retryBtn = badge.find('.aiox-retry-btn');
    
    var isExpanded = false;
    var qaLoaded = false;
    
    // Toggle badge content
    trigger.on('click', function(e) {
        e.preventDefault();
        e.stopPropagation();
        
        console.log('AIOX: Badge clicked, isExpanded:', isExpanded);
        
        if (isExpanded) {
            collapseBadge();
        } else {
            expandBadge();
        }
    });
    
    // Close button
    closeBtn.on('click', function(e) {
        e.preventDefault();
        e.stopPropagation();
        collapseBadge();
    });
    
    // Retry button
    retryBtn.on('click', function(e) {
        e.preventDefault();
        loadQAContent();
    });
    
    // Close on outside click
    $(document).on('click', function(e) {
        if (isExpanded && !badge.is(e.target) && badge.has(e.target).length === 0) {
            collapseBadge();
        }
    });
    
    // Close on escape key
    $(document).on('keydown', function(e) {
        if (e.keyCode === 27 && isExpanded) { // Escape key
            collapseBadge();
        }
    });
    
    function expandBadge() {
        if (isExpanded) return;
        
        isExpanded = true;
        badge.addClass('expanded');
        content.show();
        
        // Track badge view when opened (only once per page load)
        if (!window.aioxBadgeTracked) {
            window.aioxBadgeTracked = true;
            var postId = badge.data('post-id');
            if (postId && typeof aiox_badge !== 'undefined') {
                $.ajax({
                    url: aiox_badge.ajax_url,
                    type: 'POST',
                    data: {
                        action: 'aiox_track_badge_view',
                        post_id: postId
                    }
                });
                console.log('AIOX: Badge view tracked for post', postId);
            }
        }
        
        // Load Q&A content if not already loaded
        if (!qaLoaded) {
            loadQAContent();
        }
        
        // Focus management for accessibility
        setTimeout(function() {
            closeBtn.focus();
        }, 100);
    }
    
    function collapseBadge() {
        if (!isExpanded) return;
        
        isExpanded = false;
        badge.removeClass('expanded');
        content.hide();
        
        // Return focus to trigger
        trigger.focus();
    }
    
    function loadQAContent() {
        var postId = badge.data('post-id');
        
        console.log('AIOX: Loading Q&A content for post ID:', postId);
        
        if (!postId) {
            console.error('AIOX: Invalid post ID');
            showError('Invalid post ID');
            return;
        }
        
        // Show loading state
        showLoading();
        
        $.ajax({
            url: aiox_badge.ajax_url,
            type: 'POST',
            data: {
                action: 'aiox_get_qa',
                nonce: aiox_badge.nonce,
                post_id: postId
            },
            timeout: 15000,
            success: function(response) {
                console.log('AIOX: AJAX response received:', response);
                
                if (response.success && response.data) {
                    displayQAContent(response.data);
                    qaLoaded = true;
                } else {
                    console.error('AIOX: AJAX error response:', response);
                    showError(response.data || aiox_badge.strings.error);
                }
            },
            error: function(xhr, status, error) {
                console.error('AIOX: AJAX request failed:', status, error, xhr);
                
                var errorMsg = aiox_badge.strings.error;
                
                if (status === 'timeout') {
                    errorMsg = 'Request timed out. Please try again.';
                } else if (xhr.responseJSON && xhr.responseJSON.data) {
                    errorMsg = xhr.responseJSON.data;
                }
                
                showError(errorMsg);
            }
        });
    }
    
    function showLoading() {
        badge.find('.aiox-badge-loading').show();
        badge.find('.aiox-badge-qa-list').hide();
        badge.find('.aiox-badge-error').hide();
    }
    
    function showError(message) {
        badge.find('.aiox-badge-loading').hide();
        badge.find('.aiox-badge-qa-list').hide();
        badge.find('.aiox-badge-error').show();
        badge.find('.aiox-badge-error p').text(message);
    }
    
    function displayQAContent(data) {
        badge.find('.aiox-badge-loading').hide();
        badge.find('.aiox-badge-error').hide();
        
        var qaList = badge.find('.aiox-badge-qa-list');
        qaList.empty();
        
        if (!data.qa_pairs || data.qa_pairs.length === 0) {
            showError(aiox_badge.strings.no_content);
            return;
        }
        
        // Add AI enhancement details if available
        if (data.metadata && data.metadata.is_ai_enhanced && data.metadata.processing_details) {
            var aiDetails = $('<div class="aiox-ai-details"></div>');
            
            // Add enhancement badge
            var enhancementBadge = $('<div class="aiox-enhancement-header"></div>');
            enhancementBadge.append('<span class="aiox-ai-badge">✨ AIOX AI Optimization</span>');
            aiDetails.append(enhancementBadge);
            
            // Add summary if available
            if (data.metadata.processing_details.summary) {
                var summarySection = $('<div class="aiox-summary-section"></div>');
                summarySection.append('<h4>AI Summary</h4>');
                summarySection.append('<p>' + data.metadata.processing_details.summary + '</p>');
                aiDetails.append(summarySection);
            }
            
            // Add topics if available
            if (data.metadata.processing_details.topics && data.metadata.processing_details.topics.length > 0) {
                var topicsSection = $('<div class="aiox-topics-section"></div>');
                topicsSection.append('<h4>Key Topics</h4>');
                var topicsList = $('<div class="aiox-topics-list"></div>');
                data.metadata.processing_details.topics.forEach(function(topic) {
                    topicsList.append('<span class="aiox-topic-tag">' + topic + '</span>');
                });
                topicsSection.append(topicsList);
                aiDetails.append(topicsSection);
            }
            
            // Add content analysis if available
            if (data.metadata.processing_details.sentiment || data.metadata.processing_details.content_type) {
                var analysisSection = $('<div class="aiox-analysis-section"></div>');
                analysisSection.append('<h4>Content Analysis</h4>');
                var analysisList = $('<div class="aiox-analysis-list"></div>');
                
                if (data.metadata.processing_details.sentiment) {
                    analysisList.append('<span class="aiox-analysis-item">Sentiment: <strong>' + 
                        data.metadata.processing_details.sentiment.charAt(0).toUpperCase() + 
                        data.metadata.processing_details.sentiment.slice(1) + '</strong></span>');
                }
                
                if (data.metadata.processing_details.content_type) {
                    analysisList.append('<span class="aiox-analysis-item">Type: <strong>' + 
                        data.metadata.processing_details.content_type.charAt(0).toUpperCase() + 
                        data.metadata.processing_details.content_type.slice(1) + '</strong></span>');
                }
                
                analysisSection.append(analysisList);
                aiDetails.append(analysisSection);
            }
            
            qaList.append(aiDetails);
            qaList.append('<hr class="aiox-divider">');
        }
        
        // Add Q&A section header
        var qaHeader = $('<div class="aiox-qa-header"></div>');
        qaHeader.append('<h4>Questions & Answers (' + data.qa_pairs.length + ')</h4>');
        qaList.append(qaHeader);
        
        // Build Q&A list
        data.qa_pairs.forEach(function(qa, index) {
            var qaItem = $('<div class="aiox-qa-item"></div>');
            
            var question = $('<div class="aiox-qa-question"></div>').text(qa.question);
            var answer = $('<div class="aiox-qa-answer"></div>').text(qa.answer);
            
            qaItem.append(question).append(answer);
            
            // Add metadata if available
            if (qa.confidence || qa.source) {
                var meta = $('<div class="aiox-qa-meta"></div>');
                
                if (qa.confidence) {
                    var confidenceLevel = getConfidenceLevel(qa.confidence);
                    var confidenceBadge = $('<span class="aiox-confidence-badge ' + confidenceLevel + '"></span>')
                        .text(Math.round(qa.confidence * 100) + '% confidence');
                    meta.append(confidenceBadge);
                }
                
                if (qa.source) {
                    var source = $('<span class="aiox-source"></span>')
                        .text('Source: ' + formatSource(qa.source));
                    meta.append(source);
                }
                
                qaItem.append(meta);
            }
            
            qaList.append(qaItem);
        });
        
        qaList.show();
        
        // Add scroll behavior for long lists
        if (data.qa_pairs.length > 5) {
            qaList.css('max-height', '300px');
        }
    }
    
    function getConfidenceLevel(confidence) {
        if (confidence >= 0.8) return 'high';
        if (confidence >= 0.6) return 'medium';
        return 'low';
    }
    
    function formatSource(source) {
        var sourceMap = {
            'heading': 'Heading',
            'heading_generated': 'Generated from heading',
            'pattern_qa': 'Content pattern',
            'definition': 'Definition',
            'generated_basic': 'Basic extraction',
            'ai': 'AI analysis'
        };
        
        return sourceMap[source] || source;
    }
    
    // Keyboard navigation
    badge.on('keydown', function(e) {
        if (!isExpanded) return;
        
        switch(e.keyCode) {
            case 9: // Tab
                // Handle tab navigation within badge
                var focusableElements = badge.find('button, [tabindex]:not([tabindex="-1"])');
                var firstElement = focusableElements.first();
                var lastElement = focusableElements.last();
                
                if (e.shiftKey) {
                    if ($(e.target).is(firstElement)) {
                        e.preventDefault();
                        lastElement.focus();
                    }
                } else {
                    if ($(e.target).is(lastElement)) {
                        e.preventDefault();
                        firstElement.focus();
                    }
                }
                break;
                
            case 27: // Escape
                e.preventDefault();
                collapseBadge();
                break;
        }
    });
    
    // Accessibility improvements
    trigger.attr({
        'role': 'button',
        'aria-expanded': 'false',
        'aria-label': 'Show AI-friendly content for this page'
    });
    
    // Update aria-expanded when badge state changes
    badge.on('expanded collapsed', function() {
        trigger.attr('aria-expanded', isExpanded ? 'true' : 'false');
    });
    
    // Trigger custom events
    function triggerEvent(eventName) {
        badge.trigger(eventName);
        $(document).trigger('aiox-badge-' + eventName, [badge]);
    }
    
    // Update event triggers
    var originalExpandBadge = expandBadge;
    var originalCollapseBadge = collapseBadge;
    
    expandBadge = function() {
        originalExpandBadge();
        triggerEvent('expanded');
    };
    
    collapseBadge = function() {
        originalCollapseBadge();
        triggerEvent('collapsed');
    };
    
    // Animation end handling
    content.on('animationend', function() {
        if (isExpanded) {
            triggerEvent('expand-complete');
        }
    });
    
    // Performance optimization: lazy load on first interaction
    var hasInteracted = false;
    trigger.one('mouseenter focus', function() {
        if (!hasInteracted) {
            hasInteracted = true;
            // Preload some resources if needed
        }
    });
    
});