/**
 * AIOX Capsules Admin JavaScript
 * Enhanced functionality for the capsules management interface
 */

(function($) {
    'use strict';
    
    // Global variables
    let currentCapsuleType = '';
    let selectedPosts = [];
    let selectedAuthors = [];
    
    // Initialize when document is ready
    $(document).ready(function() {
        initializeCapsules();
    });
    
    /**
     * Initialize all capsule functionality
     */
    function initializeCapsules() {
        initializeOverviewTab();
        initializeManageTab();
        initializeCreateTab();
        initializeModals();
        initializeAjaxHandlers();
    }
    
    /**
     * Initialize overview tab functionality
     */
    function initializeOverviewTab() {
        // Quick create buttons
        $('.aiox-quick-create').on('click', function(e) {
            e.preventDefault();
            const type = $(this).data('type');
            
            // Switch to create tab and select type
            $('.nav-tab[href*="tab=create"]').click();
            setTimeout(() => {
                selectCapsuleType(type);
            }, 100);
        });
    }
    
    /**
     * Initialize manage tab functionality
     */
    function initializeManageTab() {
        // Search functionality
        $('#aiox-capsules-search').on('input', function() {
            const searchTerm = $(this).val().toLowerCase();
            filterCapsules(searchTerm, $('#aiox-capsules-filter').val());
        });
        
        // Filter functionality
        $('#aiox-capsules-filter').on('change', function() {
            const filterType = $(this).val();
            filterCapsules($('#aiox-capsules-search').val().toLowerCase(), filterType);
        });
        
        // Select all checkbox
        $('#aiox-select-all-capsules').on('change', function() {
            const isChecked = $(this).is(':checked');
            $('.aiox-capsule-checkbox').prop('checked', isChecked);
        });
        
        // Individual checkbox handling
        $(document).on('change', '.aiox-capsule-checkbox', function() {
            updateSelectAllState();
        });
        
        // Bulk actions
        $('#aiox-apply-bulk-action').on('click', function(e) {
            e.preventDefault();
            const action = $('#aiox-bulk-action').val();
            const selectedIds = getSelectedCapsuleIds();
            
            if (!action) {
                showNotice('Please select a bulk action.', 'warning');
                return;
            }
            
            if (selectedIds.length === 0) {
                showNotice('Please select at least one capsule.', 'warning');
                return;
            }
            
            performBulkAction(action, selectedIds);
        });
        
        // Individual action buttons
        $(document).on('click', '.aiox-view-capsule', function(e) {
            e.preventDefault();
            const capsuleId = $(this).data('id');
            viewCapsuleDetails(capsuleId);
        });
        
        $(document).on('click', '.aiox-edit-capsule', function(e) {
            e.preventDefault();
            const capsuleId = $(this).data('id');
            editCapsule(capsuleId);
        });
        
        $(document).on('click', '.aiox-export-capsule', function(e) {
            e.preventDefault();
            const capsuleId = $(this).data('id');
            exportCapsule(capsuleId);
        });
        
        $(document).on('click', '.aiox-delete-capsule', function(e) {
            e.preventDefault();
            const capsuleId = $(this).data('id');
            deleteCapsule(capsuleId);
        });
    }
    
    /**
     * Initialize create tab functionality
     */
    function initializeCreateTab() {
        // Type selection
        $('.aiox-select-create-type').on('click', function(e) {
            e.preventDefault();
            const type = $(this).data('type');
            selectCapsuleType(type);
        });
        
        // Type card selection
        $('.aiox-type-card').on('click', function() {
            const type = $(this).data('type');
            $('.aiox-type-card').removeClass('selected');
            $(this).addClass('selected');
            currentCapsuleType = type;
        });
    }
    
    /**
     * Initialize modal functionality
     */
    function initializeModals() {
        // Close modal handlers
        $('.aiox-modal-close, #aiox-modal-cancel').on('click', function() {
            closeModal();
        });
        
        // Close modal on backdrop click
        $('.aiox-modal').on('click', function(e) {
            if (e.target === this) {
                closeModal();
            }
        });
        
        // Create capsule button in modal
        $('#aiox-modal-create').on('click', function() {
            createCapsuleFromModal();
        });
        
        // Escape key to close modal
        $(document).on('keydown', function(e) {
            if (e.key === 'Escape') {
                closeModal();
            }
        });
    }
    
    /**
     * Initialize AJAX handlers
     */
    function initializeAjaxHandlers() {
        // Set up AJAX defaults
        $.ajaxSetup({
            beforeSend: function(xhr) {
                xhr.setRequestHeader('X-WP-Nonce', aiox_capsules_ajax.nonce);
            }
        });
        
        // Settings save handler
        $('#aiox-save-capsules-settings').on('click', function(e) {
            e.preventDefault();
            
            const $button = $(this);
            const $form = $('#aiox-capsules-settings-form');
            
            // Show loading state
            $button.prop('disabled', true).text('Saving...');
            
            // Collect form data
            const formData = {
                action: 'aiox_save_capsules_settings',
                nonce: aiox_capsules_ajax.nonce,
                auto_generate: $form.find('input[name="auto_generate"]').is(':checked') ? 1 : 0,
                default_status: $form.find('select[name="default_status"]').val(),
                ai_enhancement: $form.find('input[name="ai_enhancement"]').is(':checked') ? 1 : 0,
                export_format: $form.find('select[name="export_format"]').val(),
                retention_days: $form.find('input[name="retention_days"]').val()
            };
            
            $.ajax({
                url: aiox_capsules_ajax.ajax_url,
                type: 'POST',
                data: formData,
                success: function(response) {
                    if (response.success) {
                        showNotice('Settings saved successfully!', 'success');
                    } else {
                        let errorMessage = 'Unknown error occurred';
                        if (typeof response.data === 'string') {
                            errorMessage = response.data;
                        } else if (response.data && response.data.message) {
                            errorMessage = response.data.message;
                        } else if (response.message) {
                            errorMessage = response.message;
                        }
                        showNotice('Error saving settings: ' + errorMessage, 'error');
                    }
                },
                error: function(xhr, status, error) {
                    console.error('AJAX Error:', xhr.responseText);
                    showNotice('Error saving settings: ' + error, 'error');
                },
                complete: function() {
                    $button.prop('disabled', false).text('Save Changes');
                }
            });
        });
    }
    
    /**
     * Select capsule type and show content selection
     */
    function selectCapsuleType(type) {
        currentCapsuleType = type;
        
        // Highlight selected type
        $('.aiox-type-card').removeClass('selected');
        $(`.aiox-type-card[data-type="${type}"]`).addClass('selected');
        
        // Show content selection modal
        showContentSelectionModal(type);
    }
    
    /**
     * Show content selection modal based on type
     */
    function showContentSelectionModal(type) {
        const modal = $('#aiox-content-selection-modal');
        const modalTitle = $('#aiox-modal-title');
        const modalBody = $('#aiox-modal-body');
        
        // Set modal title
        const titles = {
            'post': 'Select Posts for Capsule',
            'collection': 'Select Posts for Collection',
            'author': 'Select Author for Profile',
            'topic': 'Configure Topic-Based Capsule'
        };
        
        modalTitle.text(titles[type] || 'Select Content');
        
        // Load content based on type
        modalBody.html('<div class="aiox-loading"></div>');
        modal.show();
        
        switch (type) {
            case 'post':
            case 'collection':
                loadPostSelection(type === 'collection');
                break;
            case 'author':
                loadAuthorSelection();
                break;
            case 'topic':
                loadTopicConfiguration();
                break;
        }
    }
    
    /**
     * Load post selection interface
     */
    function loadPostSelection(multiple = false) {
        $.ajax({
            url: aiox_capsules_ajax.ajax_url,
            type: 'POST',
            data: {
                action: 'aiox_get_posts_for_selection',
                nonce: aiox_capsules_ajax.nonce
            },
            success: function(response) {
                if (response.success) {
                    renderPostSelection(response.data, multiple);
                } else {
                    $('#aiox-modal-body').html('<p>Error loading posts: ' + response.data + '</p>');
                }
            },
            error: function() {
                $('#aiox-modal-body').html('<p>Error loading posts. Please try again.</p>');
            }
        });
    }
    
    /**
     * Render post selection interface
     */
    function renderPostSelection(posts, multiple) {
        let html = `
            <div class="aiox-form-group">
                <label>Capsule Title</label>
                <input type="text" id="aiox-capsule-title" placeholder="Enter capsule title" />
            </div>
            <div class="aiox-form-group">
                <label>Description (Optional)</label>
                <textarea id="aiox-capsule-description" placeholder="Enter capsule description"></textarea>
            </div>
            <div class="aiox-form-group">
                <label>Select ${multiple ? 'Posts' : 'Post'}</label>
                <div class="aiox-post-selection">
        `;
        
        posts.forEach(post => {
            html += `
                <div class="aiox-post-item">
                    <input type="${multiple ? 'checkbox' : 'radio'}" 
                           name="selected_posts" 
                           value="${post.ID}" 
                           id="post_${post.ID}" />
                    <div class="aiox-post-info">
                        <div class="aiox-post-title">${post.post_title}</div>
                        <div class="aiox-post-meta">
                            ${post.post_type} • ${post.post_date} • ${post.post_status}
                        </div>
                    </div>
                </div>
            `;
        });
        
        html += `
                </div>
            </div>
        `;
        
        $('#aiox-modal-body').html(html);
        
        // Reset selected posts
        selectedPosts = [];
        
        // Handle post selection with event delegation
        $('#aiox-modal-body').off('change', 'input[name="selected_posts"]').on('change', 'input[name="selected_posts"]', function() {
            if (multiple) {
                selectedPosts = $('#aiox-modal-body input[name="selected_posts"]:checked').map(function() {
                    return parseInt($(this).val());
                }).get();
            } else {
                selectedPosts = [parseInt($(this).val())];
            }
            console.log('Selected posts:', selectedPosts); // Debug log
        });
    }
    
    /**
     * Load author selection interface
     */
    function loadAuthorSelection() {
        $.ajax({
            url: aiox_capsules_ajax.ajax_url,
            type: 'POST',
            data: {
                action: 'aiox_get_authors_for_selection',
                nonce: aiox_capsules_ajax.nonce
            },
            success: function(response) {
                if (response.success) {
                    renderAuthorSelection(response.data);
                } else {
                    $('#aiox-modal-body').html('<p>Error loading authors: ' + response.data + '</p>');
                }
            },
            error: function() {
                $('#aiox-modal-body').html('<p>Error loading authors. Please try again.</p>');
            }
        });
    }
    
    /**
     * Render author selection interface
     */
    function renderAuthorSelection(authors) {
        let html = `
            <div class="aiox-form-group">
                <label>Capsule Title</label>
                <input type="text" id="aiox-capsule-title" placeholder="Enter capsule title" />
            </div>
            <div class="aiox-form-group">
                <label>Select Author</label>
                <select id="aiox-author-select">
                    <option value="">Choose an author...</option>
        `;
        
        authors.forEach(author => {
            html += `<option value="${author.ID}">${author.display_name} (${author.user_login})</option>`;
        });
        
        html += `
                </select>
            </div>
        `;
        
        $('#aiox-modal-body').html(html);
        
        // Reset selected authors
        selectedAuthors = [];
        
        // Handle author selection with event delegation
        $('#aiox-modal-body').off('change', '#aiox-author-select').on('change', '#aiox-author-select', function() {
            const authorId = parseInt($(this).val());
            selectedAuthors = authorId ? [authorId] : [];
            console.log('Selected authors:', selectedAuthors); // Debug log
        });
    }
    
    /**
     * Load topic configuration interface
     */
    function loadTopicConfiguration() {
        const html = `
            <div class="aiox-form-group">
                <label>Capsule Title</label>
                <input type="text" id="aiox-capsule-title" placeholder="Enter capsule title" />
            </div>
            <div class="aiox-form-group">
                <label>Keywords/Topics</label>
                <input type="text" id="aiox-topic-keywords" placeholder="Enter keywords separated by commas" />
                <p class="description">Enter keywords or topics to automatically find related content.</p>
            </div>
            <div class="aiox-form-group">
                <label>Maximum Posts</label>
                <input type="number" id="aiox-max-posts" value="10" min="1" max="100" />
            </div>
        `;
        
        $('#aiox-modal-body').html(html);
    }
    
    /**
     * Create capsule from modal data
     */
    function createCapsuleFromModal() {
        console.log('AIOX DEBUG: createCapsuleFromModal called', {
            currentCapsuleType: currentCapsuleType,
            selectedPosts: selectedPosts,
            selectedAuthors: selectedAuthors
        });
        
        const title = $('#aiox-capsule-title').val().trim();
        const description = $('#aiox-capsule-description').val().trim();
        
        console.log('AIOX DEBUG: Form data collected', {
            title: title,
            description: description,
            titleElement: $('#aiox-capsule-title').length,
            descriptionElement: $('#aiox-capsule-description').length
        });
        
        if (!title) {
            console.log('AIOX DEBUG: Validation failed - no title');
            showNotice('Please enter a capsule title.', 'error');
            return;
        }
        
        // Validate based on type
        if ((currentCapsuleType === 'post' || currentCapsuleType === 'collection') && selectedPosts.length === 0) {
            console.log('AIOX DEBUG: Validation failed - no posts selected', {
                type: currentCapsuleType,
                selectedPosts: selectedPosts
            });
            showNotice('Please select at least one post.', 'error');
            return;
        }
        
        if (currentCapsuleType === 'author' && selectedAuthors.length === 0) {
            console.log('AIOX DEBUG: Validation failed - no author selected', {
                type: currentCapsuleType,
                selectedAuthors: selectedAuthors
            });
            showNotice('Please select an author.', 'error');
            return;
        }
        
        if (currentCapsuleType === 'topic' && !$('#aiox-topic-keywords').val().trim()) {
            console.log('AIOX DEBUG: Validation failed - no keywords');
            showNotice('Please enter keywords for the topic capsule.', 'error');
            return;
        }
        
        // Prepare data
        const data = {
            action: 'aiox_create_capsule',
            nonce: aiox_capsules_ajax.nonce,
            type: currentCapsuleType,
            title: title,
            description: description
        };
        
        // Add type-specific data
        switch (currentCapsuleType) {
            case 'post':
            case 'collection':
                data.post_ids = selectedPosts;
                break;
            case 'author':
                data.author_id = selectedAuthors[0];
                break;
            case 'topic':
                data.keywords = $('#aiox-topic-keywords').val().trim();
                data.max_posts = parseInt($('#aiox-max-posts').val()) || 10;
                break;
        }
        
        console.log('AIOX DEBUG: Final data prepared for AJAX', data);
        
        // Show loading state
        $('#aiox-modal-create').prop('disabled', true).text('Creating...');
        
        // Add visual feedback
        showNotice('Creating capsule: ' + title, 'info');
        
        // Create capsule
        $.ajax({
            url: aiox_capsules_ajax.ajax_url,
            type: 'POST',
            data: data,
            beforeSend: function() {
                console.log('AIOX DEBUG: AJAX request starting for createCapsule');
            },
            success: function(response) {
                console.log('AIOX DEBUG: createCapsule success', response);
                
                if (response.success) {
                    showNotice('Capsule created successfully!', 'success');
                    closeModal();
                    
                    // Refresh manage tab if visible
                    if ($('.nav-tab-active[href*="tab=manage"]').length) {
                        console.log('AIOX DEBUG: Reloading page to show new capsule');
                        location.reload();
                    } else {
                        console.log('AIOX DEBUG: Switching to manage tab');
                        // Switch to manage tab
                        $('.nav-tab[href*="tab=manage"]').click();
                    }
                } else {
                    // Handle different error response formats
                    let errorMessage = 'Unknown error occurred';
                    if (typeof response.data === 'string') {
                        errorMessage = response.data;
                    } else if (response.data && response.data.message) {
                        errorMessage = response.data.message;
                    } else if (response.message) {
                        errorMessage = response.message;
                    }
                    console.error('AIOX DEBUG: createCapsule error response', response);
                    showNotice('Error creating capsule: ' + errorMessage, 'error');
                }
            },
            error: function(xhr, status, error) {
                console.error('AIOX DEBUG: createCapsule AJAX error', {
                    status: status,
                    error: error,
                    responseText: xhr.responseText,
                    readyState: xhr.readyState,
                    statusCode: xhr.status
                });
                showNotice('Error creating capsule: ' + error + ' (Status: ' + status + ')', 'error');
            },
            complete: function() {
                console.log('AIOX DEBUG: createCapsule AJAX complete');
                $('#aiox-modal-create').prop('disabled', false).text('Create Capsule');
            }
        });
    }
    
    /**
     * Filter capsules in manage tab
     */
    function filterCapsules(searchTerm, filterType) {
        $('.aiox-capsule-row').each(function() {
            const $row = $(this);
            const title = $row.find('.aiox-capsule-title strong').text().toLowerCase();
            const description = $row.find('.aiox-capsule-description').text().toLowerCase();
            const type = $row.data('type');
            
            const matchesSearch = !searchTerm || title.includes(searchTerm) || description.includes(searchTerm);
            const matchesFilter = !filterType || type === filterType;
            
            $row.toggle(matchesSearch && matchesFilter);
        });
    }
    
    /**
     * Update select all checkbox state
     */
    function updateSelectAllState() {
        const totalCheckboxes = $('.aiox-capsule-checkbox').length;
        const checkedCheckboxes = $('.aiox-capsule-checkbox:checked').length;
        
        $('#aiox-select-all-capsules').prop('indeterminate', checkedCheckboxes > 0 && checkedCheckboxes < totalCheckboxes);
        $('#aiox-select-all-capsules').prop('checked', checkedCheckboxes === totalCheckboxes && totalCheckboxes > 0);
    }
    
    /**
     * Get selected capsule IDs
     */
    function getSelectedCapsuleIds() {
        return $('.aiox-capsule-checkbox:checked').map(function() {
            return $(this).val();
        }).get();
    }
    
    /**
     * Perform bulk action
     */
    function performBulkAction(action, capsuleIds) {
        if (action === 'delete') {
            if (!confirm(`Are you sure you want to delete ${capsuleIds.length} capsule(s)? This action cannot be undone.`)) {
                return;
            }
        }
        
        $.ajax({
            url: aiox_capsules_ajax.ajax_url,
            type: 'POST',
            data: {
                action: 'aiox_bulk_capsule_action',
                nonce: aiox_capsules_ajax.nonce,
                bulk_action: action,
                capsule_ids: capsuleIds
            },
            success: function(response) {
                if (response.success) {
                    showNotice(`Bulk action "${action}" completed successfully.`, 'success');
                    location.reload();
                } else {
                    showNotice('Error performing bulk action: ' + response.data, 'error');
                }
            },
            error: function() {
                showNotice('Error performing bulk action. Please try again.', 'error');
            }
        });
    }
    
    /**
     * View capsule details
     */
    function viewCapsuleDetails(capsuleId) {
        console.log('AIOX DEBUG: viewCapsuleDetails called', {
            capsuleId: capsuleId,
            ajaxUrl: aiox_capsules_ajax.ajax_url,
            nonce: aiox_capsules_ajax.nonce
        });
        
        // Add visual feedback
        showNotice('Loading capsule details for: ' + capsuleId, 'info');
        
        $.ajax({
            url: aiox_capsules_ajax.ajax_url,
            type: 'POST',
            data: {
                action: 'aiox_get_capsule_details',
                nonce: aiox_capsules_ajax.nonce,
                capsule_id: capsuleId
            },
            beforeSend: function() {
                console.log('AIOX DEBUG: AJAX request starting for viewCapsuleDetails');
            },
            success: function(response) {
                console.log('AIOX DEBUG: viewCapsuleDetails success', response);
                
                if (response.success) {
                    showCapsuleDetailsModal(response.data);
                    showNotice('Capsule details loaded successfully', 'success');
                } else {
                    // Handle different error response formats
                    let errorMessage = 'Unknown error occurred';
                    if (typeof response.data === 'string') {
                        errorMessage = response.data;
                    } else if (response.data && response.data.message) {
                        errorMessage = response.data.message;
                    } else if (response.message) {
                        errorMessage = response.message;
                    }
                    console.error('AIOX DEBUG: viewCapsuleDetails error response', response);
                    showNotice('Error loading capsule details: ' + errorMessage, 'error');
                }
            },
            error: function(xhr, status, error) {
                console.error('AIOX DEBUG: viewCapsuleDetails AJAX error', {
                    status: status,
                    error: error,
                    responseText: xhr.responseText,
                    readyState: xhr.readyState,
                    statusCode: xhr.status
                });
                showNotice('Error loading capsule details: ' + error + ' (Status: ' + status + ')', 'error');
            },
            complete: function() {
                console.log('AIOX DEBUG: viewCapsuleDetails AJAX complete');
            }
        });
    }
    
    /**
     * Show capsule details modal
     */
    function showCapsuleDetailsModal(data) {
        console.log('AIOX DEBUG: showCapsuleDetailsModal called with data:', data);
        
        const modal = $('#aiox-content-selection-modal');
        const modalTitle = $('#aiox-modal-title');
        const modalBody = $('#aiox-modal-body');
        const modalFooter = $('.aiox-modal-footer');
        
        // Extract capsule data - handle both direct capsule object and wrapped response
        const capsule = data.capsule || data;
        
        console.log('AIOX DEBUG: Extracted capsule data:', capsule);
        console.log('AIOX DEBUG: Capsule content:', capsule.content);
        console.log('AIOX DEBUG: Capsule metadata:', capsule.metadata);
        
        modalTitle.text('Capsule Details: ' + (capsule.title || 'Untitled Capsule'));
        
        // Build comprehensive details HTML
        let html = `
            <div class="aiox-capsule-details">
                <div class="aiox-detail-section">
                    <h3>Basic Information</h3>
                    <div class="aiox-detail-row">
                        <strong>ID:</strong> ${capsule.id || 'Unknown'}
                    </div>
                    <div class="aiox-detail-row">
                        <strong>Type:</strong> ${capsule.type || 'Unknown'}
                    </div>
                    <div class="aiox-detail-row">
                        <strong>Status:</strong> ${capsule.status || 'Unknown'}
                    </div>
                    <div class="aiox-detail-row">
                        <strong>Created:</strong> ${capsule.created_at || 'Unknown'}
                    </div>
                    <div class="aiox-detail-row">
                        <strong>Description:</strong> ${capsule.description || 'No description'}
                    </div>
                    <div class="aiox-detail-row">
                        <strong>File Size:</strong> ${formatFileSize(capsule.file_size || 0)}
                    </div>
                </div>
        `;
        
        // Add content summary if available
        if (capsule.content_summary) {
            html += `
                <div class="aiox-detail-section">
                    <h3>Content Summary</h3>
                    <div class="aiox-detail-row">
                        <strong>Has Summary:</strong> ${capsule.content_summary.has_summary ? 'Yes' : 'No'}
                    </div>
                    <div class="aiox-detail-row">
                        <strong>Key Points:</strong> ${capsule.content_summary.key_points_count || 0}
                    </div>
                    <div class="aiox-detail-row">
                        <strong>Topics:</strong> ${capsule.content_summary.topics_count || 0}
                    </div>
                    <div class="aiox-detail-row">
                        <strong>Q&A Pairs:</strong> ${capsule.content_summary.qa_pairs_count || 0}
                    </div>
                    <div class="aiox-detail-row">
                        <strong>Entities:</strong> ${capsule.content_summary.entities_count || 0}
                    </div>
                </div>
            `;
        }
        
        // Add rich content details if available
        if (capsule.content && typeof capsule.content === 'object') {
            console.log('AIOX DEBUG: Processing rich content', capsule.content);
            
            html += `<div class="aiox-detail-section"><h3>Rich Content</h3>`;
            
            // Summary - check both direct and processed_content paths
            let summary = capsule.content.summary;
            if (!summary && capsule.content.processed_content && capsule.content.processed_content.summary) {
                summary = capsule.content.processed_content.summary;
            }
            
            if (summary) {
                html += `
                    <div class="aiox-detail-row">
                        <strong>Summary:</strong>
                        <div class="aiox-content-preview">${summary.substring(0, 300)}${summary.length > 300 ? '...' : ''}</div>
                    </div>
                `;
            }
            
            // Topics - check both direct and processed_content paths
            let topics = capsule.content.topics;
            if (!topics && capsule.content.processed_content && capsule.content.processed_content.topics) {
                topics = capsule.content.processed_content.topics;
            }
            
            if (topics && Array.isArray(topics) && topics.length > 0) {
                html += `
                    <div class="aiox-detail-row">
                        <strong>Topics (${topics.length}):</strong>
                        <div class="aiox-tags">${topics.slice(0, 15).map(topic => `<span class="aiox-tag">${topic}</span>`).join('')}</div>
                        ${topics.length > 15 ? `<div class="aiox-qa-more">... and ${topics.length - 15} more topics</div>` : ''}
                    </div>
                `;
            }
            
            // Q&A Pairs - check both direct and processed_content paths
            let qaPairs = capsule.content.qa_pairs;
            if (!qaPairs && capsule.content.processed_content && capsule.content.processed_content.qa_pairs) {
                qaPairs = capsule.content.processed_content.qa_pairs;
            }
            
            if (qaPairs && Array.isArray(qaPairs) && qaPairs.length > 0) {
                html += `
                    <div class="aiox-detail-row">
                        <strong>Q&A Pairs (${qaPairs.length}):</strong>
                        <div class="aiox-qa-preview">
                `;
                qaPairs.slice(0, 3).forEach(qa => {
                    if (qa.question && qa.answer) {
                        html += `
                            <div class="aiox-qa-item">
                                <strong>Q:</strong> ${qa.question}<br>
                                <strong>A:</strong> ${qa.answer}
                            </div>
                        `;
                    }
                });
                if (qaPairs.length > 3) {
                    html += `<div class="aiox-qa-more">... and ${qaPairs.length - 3} more Q&A pairs</div>`;
                }
                html += `</div></div>`;
            }
            
            // Key Points
            let keyPoints = capsule.content.key_points;
            if (!keyPoints && capsule.content.processed_content && capsule.content.processed_content.key_points) {
                keyPoints = capsule.content.processed_content.key_points;
            }
            
            if (keyPoints && Array.isArray(keyPoints) && keyPoints.length > 0) {
                html += `
                    <div class="aiox-detail-row">
                        <strong>Key Points (${keyPoints.length}):</strong>
                        <ul class="aiox-key-points">
                `;
                keyPoints.slice(0, 5).forEach(point => {
                    html += `<li>${point}</li>`;
                });
                if (keyPoints.length > 5) {
                    html += `<li class="aiox-qa-more">... and ${keyPoints.length - 5} more key points</li>`;
                }
                html += `</ul></div>`;
            }
            
            // Raw content preview
            if (capsule.content.raw_content) {
                const rawPreview = capsule.content.raw_content.replace(/<[^>]*>/g, '').substring(0, 200);
                if (rawPreview.trim()) {
                    html += `
                        <div class="aiox-detail-row">
                            <strong>Raw Content Preview:</strong>
                            <div class="aiox-content-preview">${rawPreview}${capsule.content.raw_content.length > 200 ? '...' : ''}</div>
                        </div>
                    `;
                }
            }
            
            html += `</div>`;
        } else {
            console.log('AIOX DEBUG: No rich content available or content is not an object');
        }
        
        // Add metadata if available
        if (capsule.metadata && Object.keys(capsule.metadata).length > 0) {
            html += `
                <div class="aiox-detail-section">
                    <h3>Metadata</h3>
                    <div class="aiox-metadata-preview">
                        <pre>${JSON.stringify(capsule.metadata, null, 2).substring(0, 500)}${JSON.stringify(capsule.metadata, null, 2).length > 500 ? '...' : ''}</pre>
                    </div>
                </div>
            `;
        }
        
        html += `</div>`;
        
        modalBody.html(html);
        
        // Hide create button, show close button
        $('#aiox-modal-create').hide();
        $('#aiox-modal-cancel').text('Close');
        
        modal.show();
    }
    
    /**
     * Edit capsule
     */
    function editCapsule(capsuleId) {
        console.log('AIOX DEBUG: editCapsule called', {capsuleId: capsuleId});
        
        // Add visual feedback
        showNotice('Loading capsule for editing: ' + capsuleId, 'info');
        
        // Load capsule data first
        $.ajax({
            url: aiox_capsules_ajax.ajax_url,
            type: 'POST',
            data: {
                action: 'aiox_get_capsule_details',
                nonce: aiox_capsules_ajax.nonce,
                capsule_id: capsuleId
            },
            success: function(response) {
                if (response.success) {
                    showEditCapsuleModal(response.data.capsule);
                } else {
                    showNotice('Error loading capsule for editing: ' + (response.data || 'Unknown error'), 'error');
                }
            },
            error: function(xhr, status, error) {
                showNotice('Error loading capsule for editing: ' + error, 'error');
            }
        });
    }
    
    /**
     * Show edit capsule modal
     */
    function showEditCapsuleModal(capsule) {
        const modal = $('#aiox-content-selection-modal');
        const modalTitle = $('#aiox-modal-title');
        const modalBody = $('#aiox-modal-body');
        
        modalTitle.text('Edit Capsule: ' + (capsule.title || 'Untitled Capsule'));
        
        const html = `
            <div class="aiox-edit-capsule-form">
                <div class="aiox-form-section">
                    <h3>Basic Information</h3>
                    <div class="aiox-form-row">
                        <label for="edit-capsule-title">Title:</label>
                        <input type="text" id="edit-capsule-title" value="${capsule.title || ''}" class="aiox-form-input" />
                    </div>
                    <div class="aiox-form-row">
                        <label for="edit-capsule-description">Description:</label>
                        <textarea id="edit-capsule-description" class="aiox-form-textarea">${capsule.description || ''}</textarea>
                    </div>
                    <div class="aiox-form-row">
                        <label for="edit-capsule-type">Type:</label>
                        <select id="edit-capsule-type" class="aiox-form-select">
                            <option value="post" ${capsule.type === 'post' ? 'selected' : ''}>Post</option>
                            <option value="collection" ${capsule.type === 'collection' ? 'selected' : ''}>Collection</option>
                            <option value="topic" ${capsule.type === 'topic' ? 'selected' : ''}>Topic</option>
                            <option value="author" ${capsule.type === 'author' ? 'selected' : ''}>Author</option>
                            <option value="custom" ${capsule.type === 'custom' ? 'selected' : ''}>Custom</option>
                        </select>
                    </div>
                    <div class="aiox-form-row">
                        <label for="edit-capsule-status">Status:</label>
                        <select id="edit-capsule-status" class="aiox-form-select">
                            <option value="draft" ${capsule.status === 'draft' ? 'selected' : ''}>Draft</option>
                            <option value="ready" ${capsule.status === 'ready' ? 'selected' : ''}>Ready</option>
                            <option value="published" ${capsule.status === 'published' ? 'selected' : ''}>Published</option>
                            <option value="archived" ${capsule.status === 'archived' ? 'selected' : ''}>Archived</option>
                        </select>
                    </div>
                </div>
                
                <div class="aiox-form-section">
                    <h3>Content Information</h3>
                    <div class="aiox-form-row">
                        <strong>File Size:</strong> ${formatFileSize(capsule.file_size || 0)}
                    </div>
                    <div class="aiox-form-row">
                        <strong>Created:</strong> ${capsule.created_at || 'Unknown'}
                    </div>
                    ${capsule.content_summary ? `
                        <div class="aiox-form-row">
                            <strong>Content Summary:</strong>
                            <ul>
                                <li>Topics: ${capsule.content_summary.topics_count || 0}</li>
                                <li>Q&A Pairs: ${capsule.content_summary.qa_pairs_count || 0}</li>
                                <li>Key Points: ${capsule.content_summary.key_points_count || 0}</li>
                            </ul>
                        </div>
                    ` : ''}
                </div>
                
                <div class="aiox-form-actions">
                    <button type="button" class="button button-primary" id="save-capsule-changes" data-capsule-id="${capsule.id}">
                        Save Changes
                    </button>
                    <button type="button" class="button" id="cancel-capsule-edit">
                        Cancel
                    </button>
                </div>
            </div>
        `;
        
        modalBody.html(html);
        
        // Show save button, hide create button
        $('#aiox-modal-create').hide();
        $('#aiox-modal-cancel').text('Cancel');
        
        // Bind save functionality
        $('#save-capsule-changes').on('click', function() {
            saveCapsuleChanges(capsule.id);
        });
        
        $('#cancel-capsule-edit, #aiox-modal-cancel').on('click', function() {
            closeModal();
        });
        
        modal.show();
    }
    
    /**
     * Save capsule changes
     */
    function saveCapsuleChanges(capsuleId) {
        const title = $('#edit-capsule-title').val();
        const description = $('#edit-capsule-description').val();
        const type = $('#edit-capsule-type').val();
        const status = $('#edit-capsule-status').val();
        
        showNotice('Saving capsule changes...', 'info');
        
        $.ajax({
            url: aiox_capsules_ajax.ajax_url,
            type: 'POST',
            data: {
                action: 'aiox_update_capsule',
                nonce: aiox_capsules_ajax.nonce,
                capsule_id: capsuleId,
                title: title,
                description: description,
                type: type,
                status: status
            },
            success: function(response) {
                if (response.success) {
                    showNotice('Capsule updated successfully!', 'success');
                    closeModal();
                    // Refresh the page to show updated data
                    setTimeout(function() {
                        location.reload();
                    }, 1000);
                } else {
                    showNotice('Error updating capsule: ' + (response.data || 'Unknown error'), 'error');
                }
            },
            error: function(xhr, status, error) {
                showNotice('Error updating capsule: ' + error, 'error');
            }
        });
    }
    
    /**
     * Export capsule
     */
    function exportCapsule(capsuleId) {
        $.ajax({
            url: aiox_capsules_ajax.ajax_url,
            type: 'POST',
            data: {
                action: 'aiox_export_capsule',
                nonce: aiox_capsules_ajax.nonce,
                capsule_id: capsuleId
            },
            success: function(response) {
                if (response.success) {
                    // Create download link
                    const blob = new Blob([response.data], { type: 'application/json' });
                    const url = window.URL.createObjectURL(blob);
                    const a = document.createElement('a');
                    a.href = url;
                    a.download = `capsule_${capsuleId}.json`;
                    document.body.appendChild(a);
                    a.click();
                    document.body.removeChild(a);
                    window.URL.revokeObjectURL(url);
                    
                    showNotice('Capsule exported successfully.', 'success');
                } else {
                    showNotice('Error exporting capsule: ' + response.data, 'error');
                }
            },
            error: function() {
                showNotice('Error exporting capsule. Please try again.', 'error');
            }
        });
    }
    
    /**
     * Delete capsule
     */
    function deleteCapsule(capsuleId) {
        console.log('AIOX DEBUG: deleteCapsule called', {
            capsuleId: capsuleId,
            ajaxUrl: aiox_capsules_ajax.ajax_url,
            nonce: aiox_capsules_ajax.nonce
        });
        
        if (!confirm('Are you sure you want to delete this capsule? This action cannot be undone.')) {
            console.log('AIOX DEBUG: Delete cancelled by user');
            return;
        }
        
        // Add visual feedback
        showNotice('Deleting capsule: ' + capsuleId, 'info');
        
        $.ajax({
            url: aiox_capsules_ajax.ajax_url,
            type: 'POST',
            data: {
                action: 'aiox_delete_capsule',
                nonce: aiox_capsules_ajax.nonce,
                capsule_id: capsuleId
            },
            beforeSend: function() {
                console.log('AIOX DEBUG: AJAX request starting for deleteCapsule');
            },
            success: function(response) {
                console.log('AIOX DEBUG: deleteCapsule success', response);
                
                if (response.success) {
                    showNotice('Capsule deleted successfully.', 'success');
                    
                    // Find and remove the row immediately to prevent reappearing
                    // Try multiple selectors to find the row
                    let $row = $(`tr[data-capsule-id="${capsuleId}"]`);
                    
                    if ($row.length === 0) {
                        // Fallback: find by checkbox value
                        $row = $(`.aiox-capsule-row`).filter(function() {
                            return $(this).find(`.aiox-capsule-checkbox[value="${capsuleId}"]`).length > 0;
                        });
                    }
                    
                    if ($row.length === 0) {
                        // Fallback: find by button data-id
                        $row = $(`.aiox-capsule-row`).filter(function() {
                            return $(this).find(`[data-id="${capsuleId}"]`).length > 0;
                        });
                    }
                    
                    if ($row.length === 0) {
                        // Fallback: find by any element containing the capsule ID
                        $row = $(`tr:contains("${capsuleId}")`);
                    }
                    
                    console.log('AIOX DEBUG: Found row to delete', {
                        capsuleId: capsuleId,
                        rowCount: $row.length,
                        rowHtml: $row.length > 0 ? $row[0].outerHTML.substring(0, 200) + '...' : 'none'
                    });
                    
                    if ($row.length > 0) {
                        // Mark the row as being deleted to prevent double-deletion
                        $row.addClass('aiox-deleting');
                        
                        $row.fadeOut(300, function() {
                            $(this).remove();
                            
                            // Update UI state
                            if (typeof updateSelectAllState === 'function') {
                                updateSelectAllState();
                            }
                            
                            // Check if no capsules left
                            const remainingRows = $('.aiox-capsule-row:not(.aiox-deleting)').length;
                            console.log('AIOX DEBUG: Remaining rows after deletion:', remainingRows);
                            
                            if (remainingRows === 0) {
                                console.log('AIOX DEBUG: No capsules left, showing empty state message');
                                // Show empty state instead of reloading
                                const emptyMessage = `
                                    <tr>
                                        <td colspan="6" style="text-align: center; padding: 40px;">
                                            <p>No capsules found. <a href="?page=aiox-capsules&tab=create">Create your first capsule</a></p>
                                        </td>
                                    </tr>
                                `;
                                $('.aiox-capsules-table tbody').html(emptyMessage);
                            }
                        });
                    } else {
                        console.log('AIOX DEBUG: Row not found with any selector, forcing page reload');
                        // Last resort: reload the page
                        setTimeout(function() {
                            location.reload();
                        }, 1000);
                    }
                } else {
                    // Handle different error response formats
                    let errorMessage = 'Unknown error occurred';
                    if (typeof response.data === 'string') {
                        errorMessage = response.data;
                    } else if (response.data && response.data.message) {
                        errorMessage = response.data.message;
                    } else if (response.message) {
                        errorMessage = response.message;
                    }
                    console.error('AIOX DEBUG: deleteCapsule error response', response);
                    showNotice('Error deleting capsule: ' + errorMessage, 'error');
                }
            },
            error: function(xhr, status, error) {
                console.error('AIOX DEBUG: deleteCapsule AJAX error', {
                    status: status,
                    error: error,
                    responseText: xhr.responseText,
                    readyState: xhr.readyState,
                    statusCode: xhr.status
                });
                showNotice('Error deleting capsule: ' + error + ' (Status: ' + status + ')', 'error');
            },
            complete: function() {
                console.log('AIOX DEBUG: deleteCapsule AJAX complete');
            }
        });
    }
    
    /**
     * Close modal
     */
    function closeModal() {
        $('#aiox-content-selection-modal').hide();
        $('#aiox-modal-create').show().text('Create Capsule');
        $('#aiox-modal-cancel').text('Cancel');
        
        // Reset selections
        selectedPosts = [];
        selectedAuthors = [];
        currentCapsuleType = '';
        $('.aiox-type-card').removeClass('selected');
    }
    
    /**
     * Show notice message
     */
    function showNotice(message, type = 'info') {
        const noticeClass = `aiox-notice aiox-notice-${type}`;
        const notice = $(`<div class="${noticeClass}">${message}</div>`);
        
        // Remove existing notices
        $('.aiox-notice').remove();
        
        // Add new notice
        $('.aiox-tab-content').prepend(notice);
        
        // Auto-hide after 5 seconds
        setTimeout(() => {
            notice.fadeOut(300, function() {
                $(this).remove();
            });
        }, 5000);
        
        // Scroll to top to show notice
        $('.aiox-tab-content').scrollTop(0);
    }
    
    /**
     * Format file size for display
     */
    function formatFileSize(bytes) {
        if (bytes === 0) return '0 Bytes';
        
        const k = 1024;
        const sizes = ['Bytes', 'KB', 'MB', 'GB'];
        const i = Math.floor(Math.log(bytes) / Math.log(k));
        
        return parseFloat((bytes / Math.pow(k, i)).toFixed(2)) + ' ' + sizes[i];
    }
    
})(jQuery);
