/**
 * AIOX Meta Boxes JavaScript
 */

jQuery(document).ready(function($) {
    
    // Auto-classify content button REMOVED
    // Metadata is now automatically extracted during comprehensive AI processing
    // See: class-aiox-comprehensive-processor.php for automatic metadata extraction
    
    // Preview ingredients
    $('#aiox-preview-ingredients').on('click', function() {
        var $button = $(this);
        var originalText = $button.text();
        
        // Get post content - enhanced detection
        var content = '';
        var title = '';
        
        // Try Block Editor first (Gutenberg)
        if (typeof wp !== 'undefined' && wp.data && wp.data.select) {
            try {
                var editorSelect = wp.data.select('core/editor');
                if (editorSelect && typeof editorSelect.getEditedPostContent === 'function') {
                    content = editorSelect.getEditedPostContent() || '';
                    if (typeof editorSelect.getEditedPostAttribute === 'function') {
                        title = editorSelect.getEditedPostAttribute('title') || '';
                    }
                }
            } catch (e) {
                // Silently continue to next method
            }
        }
        
        // Try Classic Editor if Block Editor didn't work
        if (!content && typeof tinymce !== 'undefined') {
            var editor = tinymce.get('content');
            if (editor && typeof editor.getContent === 'function') {
                try {
                    content = editor.getContent() || '';
                } catch (e) {
                    // Silently continue to next method
                }
            }
        }
        
        // Try textarea fallback
        if (!content && $('#content').length) {
            content = $('#content').val() || '';
        }
        
        // Get title if not already retrieved
        if (!title) {
            title = $('#title').val() || $('.editor-post-title__input').val() || '';
        }
        
        if (!content && !title) {
            alert('Please add some content or title before previewing ingredients.');
            return;
        }
        
        // Show loading state
        $button.prop('disabled', true).text(aiox_meta.strings.generating_preview);
        
        // Make AJAX request
        $.ajax({
            url: aiox_meta.ajax_url,
            type: 'POST',
            data: {
                action: 'aiox_preview_ingredients',
                nonce: aiox_meta.nonce,
                content: content,
                title: title,
                post_id: $('#post_ID').val() || 0
            },
            success: function(response) {
                if (response.success && response.data) {
                    // Show ingredients preview in a modal or popup
                    showIngredientsPreview(response.data);
                } else {
                    showMessage('Failed to generate ingredients preview.', 'error');
                }
            },
            error: function() {
                showMessage('Error occurred while generating preview.', 'error');
            },
            complete: function() {
                // Restore button state
                $button.prop('disabled', false).text(originalText);
            }
        });
    });
    
    // Show message function
    function showMessage(message, type) {
        var $message = $('<div class="aiox-message ' + type + '">' + message + '</div>');
        $('.aiox-metadata-container').prepend($message);
        
        // Auto-hide after 5 seconds
        setTimeout(function() {
            $message.fadeOut(function() {
                $(this).remove();
            });
        }, 5000);
    }
    
    // Show ingredients preview
    function showIngredientsPreview(ingredients) {
        var html = '<div id="aiox-ingredients-modal" style="display:none;">';
        html += '<div class="aiox-modal-overlay"></div>';
        html += '<div class="aiox-modal-content">';
        html += '<div class="aiox-modal-header">';
        html += '<h3>Ingredients Preview</h3>';
        html += '<button type="button" class="aiox-modal-close">&times;</button>';
        html += '</div>';
        html += '<div class="aiox-modal-body">';
        
        if (ingredients && ingredients.length > 0) {
            html += '<div class="aiox-ingredients-list">';
            
            // Group ingredients by type
            var groupedIngredients = {};
            $.each(ingredients, function(i, ingredient) {
                if (!groupedIngredients[ingredient.type]) {
                    groupedIngredients[ingredient.type] = [];
                }
                groupedIngredients[ingredient.type].push(ingredient);
            });
            
            // Display grouped ingredients
            $.each(groupedIngredients, function(type, items) {
                html += '<div class="aiox-ingredient-group">';
                html += '<h4>' + type.replace(/_/g, ' ').replace(/\b\w/g, l => l.toUpperCase()) + ' (' + items.length + ')</h4>';
                html += '<div class="aiox-ingredient-items">';
                
                $.each(items, function(j, item) {
                    html += '<div class="aiox-ingredient-item">';
                    
                    if (item.type === 'qa_pair') {
                        try {
                            var qaData = JSON.parse(item.content);
                            html += '<div class="aiox-qa-pair">';
                            html += '<div class="aiox-question"><strong>Q:</strong> ' + escapeHtml(qaData.question) + '</div>';
                            html += '<div class="aiox-answer"><strong>A:</strong> ' + escapeHtml(qaData.answer) + '</div>';
                            html += '</div>';
                        } catch (e) {
                            html += '<div class="aiox-content">' + escapeHtml(item.content.substring(0, 200)) + '...</div>';
                        }
                    } else if (item.type === 'metadata' || item.type === 'link') {
                        html += '<div class="aiox-structured-content">[Structured Data]</div>';
                    } else {
                        var content = item.content.length > 200 ? item.content.substring(0, 200) + '...' : item.content;
                        html += '<div class="aiox-content">' + escapeHtml(content) + '</div>';
                    }
                    
                    html += '<div class="aiox-ingredient-meta">';
                    html += '<span class="aiox-word-count">' + (item.word_count || 0) + ' words</span>';
                    if (item.context && item.context.importance) {
                        html += '<span class="aiox-importance aiox-importance-' + item.context.importance + '">' + item.context.importance + '</span>';
                    }
                    html += '</div>';
                    
                    html += '</div>';
                });
                
                html += '</div>';
                html += '</div>';
            });
            
            html += '</div>';
        } else {
            html += '<p>No ingredients found.</p>';
        }
        
        html += '</div>';
        html += '<div class="aiox-modal-footer">';
        html += '<button type="button" class="button aiox-modal-close">Close</button>';
        html += '</div>';
        html += '</div>';
        html += '</div>';
        
        // Add modal to page
        $('body').append(html);
        
        // Add modal styles
        if (!$('#aiox-modal-styles').length) {
            var styles = `
                <style id="aiox-modal-styles">
                .aiox-modal-overlay {
                    position: fixed;
                    top: 0;
                    left: 0;
                    width: 100%;
                    height: 100%;
                    background: rgba(0,0,0,0.5);
                    z-index: 100000;
                }
                .aiox-modal-content {
                    position: fixed;
                    top: 50%;
                    left: 50%;
                    transform: translate(-50%, -50%);
                    background: white;
                    border-radius: 4px;
                    box-shadow: 0 4px 20px rgba(0,0,0,0.3);
                    z-index: 100001;
                    max-width: 80%;
                    max-height: 80%;
                    overflow: hidden;
                    display: flex;
                    flex-direction: column;
                }
                .aiox-modal-header {
                    padding: 20px;
                    border-bottom: 1px solid #ddd;
                    display: flex;
                    justify-content: space-between;
                    align-items: center;
                }
                .aiox-modal-header h3 {
                    margin: 0;
                }
                .aiox-modal-close {
                    background: none;
                    border: none;
                    font-size: 24px;
                    cursor: pointer;
                    padding: 0;
                    width: 30px;
                    height: 30px;
                    display: flex;
                    align-items: center;
                    justify-content: center;
                }
                .aiox-modal-body {
                    padding: 20px;
                    overflow-y: auto;
                    flex: 1;
                }
                .aiox-modal-footer {
                    padding: 20px;
                    border-top: 1px solid #ddd;
                    text-align: right;
                }
                .aiox-ingredient-group {
                    margin-bottom: 25px;
                }
                .aiox-ingredient-group h4 {
                    margin: 0 0 15px 0;
                    color: #0073aa;
                    border-bottom: 1px solid #eee;
                    padding-bottom: 5px;
                }
                .aiox-ingredient-item {
                    background: #f9f9f9;
                    padding: 15px;
                    margin-bottom: 10px;
                    border-radius: 4px;
                    border-left: 3px solid #0073aa;
                }
                .aiox-qa-pair .aiox-question {
                    margin-bottom: 8px;
                    color: #0073aa;
                }
                .aiox-qa-pair .aiox-answer {
                    color: #333;
                }
                .aiox-content {
                    color: #333;
                    line-height: 1.5;
                }
                .aiox-structured-content {
                    color: #666;
                    font-style: italic;
                }
                .aiox-ingredient-meta {
                    margin-top: 10px;
                    font-size: 12px;
                    color: #666;
                }
                .aiox-word-count {
                    margin-right: 15px;
                }
                .aiox-importance {
                    padding: 2px 6px;
                    border-radius: 3px;
                    font-size: 11px;
                    font-weight: bold;
                    text-transform: uppercase;
                }
                .aiox-importance-high {
                    background: #d4edda;
                    color: #155724;
                }
                .aiox-importance-medium {
                    background: #fff3cd;
                    color: #856404;
                }
                .aiox-importance-low {
                    background: #f8d7da;
                    color: #721c24;
                }
                </style>
            `;
            $('head').append(styles);
        }
        
        // Show modal
        $('#aiox-ingredients-modal').fadeIn();
        
        // Close modal handlers
        $('.aiox-modal-close, .aiox-modal-overlay').on('click', function() {
            $('#aiox-ingredients-modal').fadeOut(function() {
                $(this).remove();
            });
        });
    }
    
    // Escape HTML function
    function escapeHtml(text) {
        var map = {
            '&': '&amp;',
            '<': '&lt;',
            '>': '&gt;',
            '"': '&quot;',
            "'": '&#039;'
        };
        return text.replace(/[&<>"']/g, function(m) { return map[m]; });
    }
    
    // Help icon tooltips
    $('.aiox-help-icon').on('mouseenter', function() {
        var title = $(this).attr('title');
        if (title) {
            $(this).attr('data-title', title).removeAttr('title');
            
            var tooltip = $('<div class="aiox-tooltip">' + title + '</div>');
            $('body').append(tooltip);
            
            var offset = $(this).offset();
            tooltip.css({
                position: 'absolute',
                top: offset.top - tooltip.outerHeight() - 5,
                left: offset.left + ($(this).outerWidth() / 2) - (tooltip.outerWidth() / 2),
                background: '#333',
                color: 'white',
                padding: '5px 10px',
                borderRadius: '4px',
                fontSize: '12px',
                zIndex: 1000,
                whiteSpace: 'nowrap'
            });
        }
    }).on('mouseleave', function() {
        $('.aiox-tooltip').remove();
        var title = $(this).attr('data-title');
        if (title) {
            $(this).attr('title', title).removeAttr('data-title');
        }
    });
});
