/**
 * AIOX Enhanced Modern Analytics JavaScript - COMPLETE FIXED VERSION
 * All functions properly scoped within jQuery wrapper
 */

(function($) {
    'use strict';
    
    // Global variables within this scope
    let currentPeriod = 30;
    let currentTab = 'overview';
    let charts = {};
    
    // Initialize on document ready
    $(document).ready(function() {
        console.log('AIOX Analytics: Initializing...');
        setTimeout(function() {
            initializeAnalytics();
            bindEvents();
            loadAnalyticsData();
        }, 100);
    });
    
    function initializeAnalytics() {
        $('#aiox-analytics-period').val(currentPeriod);
        
        if (typeof Chart === 'undefined') {
            console.error('Chart.js not loaded');
            showError('Chart library failed to load. Please refresh the page.');
            return;
        }
        
        Chart.defaults.font.family = '-apple-system, BlinkMacSystemFont, "Segoe UI", Roboto, sans-serif';
        Chart.defaults.font.size = 12;
        Chart.defaults.color = '#64748b';
        
        console.log('AIOX Analytics: Initialized successfully');
    }
    
    function bindEvents() {
        $('.aiox-analytics-tabs .aiox-tab').on('click', function(e) {
            e.preventDefault();
            switchTab($(this).data('tab'));
        });
        
        $('#aiox-analytics-period').on('change', function() {
            currentPeriod = parseInt($(this).val());
            loadAnalyticsData();
        });
        
        $('#aiox-analytics-refresh').on('click', function() {
            loadAnalyticsData();
        });
        
        $('#aiox-analytics-export').on('click', function() {
            showExportModal();
        });
        
        $('#aiox-analytics-reset').on('click', function() {
            showResetConfirmation();
        });
        
        $('.aiox-modal-close, .aiox-modal-backdrop').on('click', function() {
            hideExportModal();
        });
        
        $('#aiox-export-download').on('click', function() {
            downloadExport();
        });
        
        // Visitor Logs tab handlers
        $('#refresh-visitor-logs').on('click', function() {
            loadVisitorLogs(1);
        });
        
        $('#visitor-type-filter').on('change', function() {
            loadVisitorLogs(1);
        });
    }
    
    function switchTab(tab) {
        if (tab === currentTab) return;
        
        console.log('Switching to tab:', tab);
        
        $('.aiox-analytics-tabs .aiox-tab').removeClass('active');
        $('[data-tab="' + tab + '"]').addClass('active');
        
        $('.aiox-tab-content').removeClass('active');
        $('#tab-' + tab).addClass('active');
        
        currentTab = tab;
        
        if (!hasTabData(tab)) {
            loadTabData(tab);
        }
    }
    
    function hasTabData(tab) {
        switch (tab) {
            case 'overview':
                return $('#human-visits').text() !== '-';
            case 'crawlers':
                return $('#crawlers-table tbody tr').length > 1;
            case 'badges':
                return $('#badge-posts-table tbody tr').length > 1;
            case 'wellknown':
                return $('#wellknown-crawlers-table tbody tr').length > 1;
            case 'content':
                return $('#top-content-table tbody tr').length > 1;
            case 'activity':
                return $('#top-pages-table tbody tr').length > 1;
            case 'visitorlogs':
                return $('#visitor-logs-body tr').length > 1 && !$('#visitor-logs-body .aiox-loading-cell').length;
            default:
                return false;
        }
    }
    
    function loadAnalyticsData() {
        console.log('Loading analytics data for period:', currentPeriod);
        showLoading();
        
        loadTabData('overview')
            .then(function() {
                if (currentTab !== 'overview') {
                    return loadTabData(currentTab);
                }
            })
            .then(function() {
                hideLoading();
                console.log('Analytics data loaded successfully');
            })
            .catch(function(error) {
                hideLoading();
                console.error('Failed to load analytics:', error);
                showError('Failed to load analytics data: ' + error);
            });
    }
    
    function loadTabData(tab) {
        return new Promise(function(resolve, reject) {
            console.log('Loading tab data:', tab);
            
            // Special handling for visitor logs tab
            if (tab === 'visitorlogs') {
                loadVisitorLogs(1);
                resolve();
                return;
            }
            
            $.ajax({
                url: aioxModernAnalytics.ajaxUrl,
                type: 'POST',
                data: {
                    action: 'aiox_modern_analytics_data',
                    nonce: aioxModernAnalytics.nonce,
                    tab: tab,
                    days: currentPeriod
                },
                success: function(response) {
                    console.log('AJAX Response for', tab, ':', response);
                    
                    if (response.success) {
                        updateTabData(tab, response.data);
                        resolve(response.data);
                    } else {
                        console.error('Analytics API Error:', response.data);
                        reject(response.data);
                    }
                },
                error: function(xhr, status, error) {
                    console.error('Analytics AJAX Error:', error, xhr);
                    reject(error);
                }
            });
        });
    }
    
    function updateTabData(tab, data) {
        console.log('Updating tab:', tab, 'with data:', data);
        
        switch (tab) {
            case 'overview':
                updateOverviewData(data);
                break;
            case 'crawlers':
                updateCrawlersData(data);
                break;
            case 'badges':
                updateBadgesData(data);
                break;
            case 'wellknown':
                updateWellknownData(data);
                break;
            case 'content':
                updateContentData(data);
                break;
            case 'activity':
                updateActivityData(data);
                break;
        }
    }
    
    function updateOverviewData(data) {
        // Primary metrics
        $('#human-visits').text(formatNumber(data.human_visits || 0));
        $('#ai-visits').text(formatNumber(data.ai_visits || 0));
        $('#search-visits').text(formatNumber(data.search_visits || 0));
        $('#other-bot-visits').text(formatNumber(data.other_bot_visits || 0));
        $('#unidentified-visits').text(formatNumber(data.unidentified_visits || 0));
        $('#ai-percentage').text((data.ai_percentage || 0) + '%');
        
        // Secondary metrics
        $('#total-visits').text(formatNumber(data.total_visits || 0));
        $('#well-known-requests').text(formatNumber(data.well_known_requests || 0));
        $('#badge-views').text(formatNumber(data.badge_views || 0));
        
        updateDailyActivityChart(data.daily_activity || [], data.is_hourly || false);
        updateTopCrawlersChart(data.top_crawlers || []);
    }
    
    function updateCrawlersData(data) {
        updateCrawlerTypesChart(data.type_distribution || []);
        updateCrawlersTable(data.top_crawlers || []);
    }
    
    function updateBadgesData(data) {
        updateBadgeViewerTypesChart(data.by_type || []);
        updateBadgeDailyChart(data.daily_views || []);
        updateBadgePostsTable(data.top_posts || []);
    }
    
    function updateWellknownData(data) {
        updateWellknownFilesChart(data.top_files || []);
        updateWellknownDailyChart(data.daily_hits || []);
        updateWellknownCrawlersTable(data.by_crawler || []);
    }
    
    function updateContentData(data) {
        updateContentPerformanceChart(data.content_performance || []);
        updateTopContentTable(data.top_posts || []);
    }
    
    function updateActivityData(data) {
        updateHourlyActivityChart(data.hourly_activity || []);
        updateTopPagesTable(data.top_pages || []);
    }
    
    // ==================== CHART FUNCTIONS ====================
    
    function updateDailyActivityChart(data, isHourly) {
        var ctx = document.getElementById('daily-activity-chart');
        if (!ctx || typeof Chart === 'undefined') return;
        
        if (charts.dailyActivity) charts.dailyActivity.destroy();
        if (!data || data.length === 0) return;
        
        // Update chart title based on data type
        var titleText = isHourly ? 'Hourly Activity (Last 24h)' : 'Daily Activity Trends';
        $('#activity-chart-title-text').text(titleText);
        
        // Format labels based on data type (hourly or daily)
        var labels = data.map(function(item) { 
            if (isHourly) {
                // For hourly data, the date field is already in "HH:00" format
                return item.date;
            } else {
                return formatDate(item.date); 
            }
        });
        var humanVisits = data.map(function(item) { return parseInt(item.human_visits || 0); });
        var aiVisits = data.map(function(item) { return parseInt(item.ai_visits || 0); });
        var searchVisits = data.map(function(item) { return parseInt(item.search_visits || 0); });
        var otherBotVisits = data.map(function(item) { return parseInt(item.other_bot_visits || 0); });
        var unidentifiedVisits = data.map(function(item) { return parseInt(item.unidentified_visits || 0); });
        
        charts.dailyActivity = new Chart(ctx, {
            type: 'line',
            data: {
                labels: labels,
                datasets: [
                    {
                        label: 'Browser Visits',
                        data: humanVisits,
                        borderColor: '#2271b1',
                        backgroundColor: 'rgba(34, 113, 177, 0.1)',
                        fill: true,
                        tension: 0.4,
                        borderWidth: 2
                    },
                    {
                        label: 'AI Crawlers',
                        data: aiVisits,
                        borderColor: '#8b5cf6',
                        backgroundColor: 'rgba(139, 92, 246, 0.1)',
                        fill: true,
                        tension: 0.4,
                        borderWidth: 2
                    },
                    {
                        label: 'Search Engines',
                        data: searchVisits,
                        borderColor: '#00a32a',
                        backgroundColor: 'rgba(0, 163, 42, 0.1)',
                        fill: true,
                        tension: 0.4,
                        borderWidth: 2
                    },
                    {
                        label: 'Other Bots',
                        data: otherBotVisits,
                        borderColor: '#dba617',
                        backgroundColor: 'rgba(219, 166, 23, 0.1)',
                        fill: false,
                        tension: 0.4,
                        borderWidth: 2
                    },
                    {
                        label: 'Unidentified',
                        data: unidentifiedVisits,
                        borderColor: '#646970',
                        backgroundColor: 'rgba(100, 105, 112, 0.1)',
                        fill: false,
                        tension: 0.4,
                        borderWidth: 2,
                        borderDash: [5, 5],
                        hidden: true
                    }
                ]
            },
            options: {
                responsive: true,
                maintainAspectRatio: false,
                plugins: {
                    legend: {
                        position: 'top',
                        labels: {
                            usePointStyle: true,
                            padding: 15
                        }
                    },
                    tooltip: {
                        mode: 'index',
                        intersect: false
                    }
                },
                scales: {
                    y: {
                        beginAtZero: true,
                        grid: {
                            color: '#f1f5f9'
                        }
                    },
                    x: {
                        grid: {
                            display: false
                        }
                    }
                }
            }
        });
    }
    
    function updateTopCrawlersChart(data) {
        var ctx = document.getElementById('top-crawlers-chart');
        if (!ctx || typeof Chart === 'undefined') return;
        
        if (charts.topCrawlers) charts.topCrawlers.destroy();
        if (!data || data.length === 0) return;
        
        // Debug: Log the data to console
        console.log('Top Crawlers Data:', data);
        
        // Color palette
        var colorPalette = [
            '#8b5cf6', // Purple
            '#3b82f6', // Blue
            '#10b981', // Green
            '#f59e0b', // Orange
            '#ef4444', // Red
            '#ec4899', // Pink
            '#14b8a6', // Teal
            '#84cc16', // Lime
            '#06b6d4', // Cyan
            '#a855f7', // Violet
            '#f43f5e', // Rose
            '#0ea5e9', // Sky
            '#22c55e', // Green
            '#eab308', // Yellow
            '#6366f1', // Indigo
            '#8b5cf6'  // Purple
        ];
        
        var labels = data.map(function(item) { return item.name || 'Unknown'; });
        var visits = data.map(function(item) { return parseInt(item.visits || 0); });
        
        // Generate colors - use provided color or pick from palette
        var colors = data.map(function(item, index) { 
            if (item.color && item.color !== '#667eea') {
                return item.color;
            }
            return colorPalette[index % colorPalette.length];
        });
        
        // Debug: Log the colors array
        console.log('Chart Colors:', colors);
        
        charts.topCrawlers = new Chart(ctx, {
            type: 'doughnut',
            data: {
                labels: labels,
                datasets: [{
                    data: visits,
                    backgroundColor: colors,
                    borderWidth: 3,
                    borderColor: '#fff'
                }]
            },
            options: {
                responsive: true,
                maintainAspectRatio: false,
                plugins: {
                    legend: {
                        position: 'right',
                        labels: {
                            usePointStyle: true,
                            padding: 15
                        }
                    }
                }
            }
        });
    }
    
    function updateCrawlerTypesChart(data) {
        var ctx = document.getElementById('crawler-types-chart');
        if (!ctx || typeof Chart === 'undefined') return;
        
        if (charts.crawlerTypes) charts.crawlerTypes.destroy();
        if (!data || data.length === 0) return;
        
        var typeColors = {
            'ai': '#8b5cf6',
            'search': '#3b82f6',
            'social': '#ec4899',
            'unknown': '#64748b'
        };
        
        var labels = data.map(function(item) { return capitalizeFirst(item.crawler_type || 'Unknown'); });
        var counts = data.map(function(item) { return parseInt(item.count || 0); });
        var colors = data.map(function(item) { return typeColors[item.crawler_type] || '#64748b'; });
        
        charts.crawlerTypes = new Chart(ctx, {
            type: 'bar',
            data: {
                labels: labels,
                datasets: [{
                    label: 'Visits',
                    data: counts,
                    backgroundColor: colors,
                    borderRadius: 8
                }]
            },
            options: {
                responsive: true,
                maintainAspectRatio: false,
                plugins: {
                    legend: {
                        display: false
                    }
                },
                scales: {
                    y: {
                        beginAtZero: true,
                        grid: {
                            color: '#f1f5f9'
                        }
                    },
                    x: {
                        grid: {
                            display: false
                        }
                    }
                }
            }
        });
    }
    
    function updateBadgeViewerTypesChart(data) {
        var ctx = document.getElementById('badge-viewer-types-chart');
        if (!ctx || typeof Chart === 'undefined') return;
        
        if (charts.badgeViewerTypes) charts.badgeViewerTypes.destroy();
        if (!data || data.length === 0) return;
        
        var labels = data.map(function(item) { return capitalizeFirst(item.viewer_type || 'Unknown'); });
        var views = data.map(function(item) { return parseInt(item.views || 0); });
        var colors = ['#8b5cf6', '#3b82f6', '#10b981', '#f59e0b', '#ec4899'];
        
        charts.badgeViewerTypes = new Chart(ctx, {
            type: 'pie',
            data: {
                labels: labels,
                datasets: [{
                    data: views,
                    backgroundColor: colors,
                    borderWidth: 3,
                    borderColor: '#fff'
                }]
            },
            options: {
                responsive: true,
                maintainAspectRatio: false,
                plugins: {
                    legend: {
                        position: 'bottom',
                        labels: {
                            usePointStyle: true,
                            padding: 15
                        }
                    }
                }
            }
        });
    }
    
    function updateBadgeDailyChart(data) {
        var ctx = document.getElementById('badge-daily-chart');
        if (!ctx || typeof Chart === 'undefined') return;
        
        if (charts.badgeDaily) charts.badgeDaily.destroy();
        if (!data || data.length === 0) return;
        
        var labels = data.map(function(item) { return formatDate(item.date); });
        var views = data.map(function(item) { return parseInt(item.views || 0); });
        
        charts.badgeDaily = new Chart(ctx, {
            type: 'line',
            data: {
                labels: labels,
                datasets: [{
                    label: 'Badge Views',
                    data: views,
                    borderColor: '#ec4899',
                    backgroundColor: 'rgba(236, 72, 153, 0.1)',
                    fill: true,
                    tension: 0.4,
                    borderWidth: 2
                }]
            },
            options: {
                responsive: true,
                maintainAspectRatio: false,
                plugins: {
                    legend: {
                        display: false
                    }
                },
                scales: {
                    y: {
                        beginAtZero: true,
                        grid: {
                            color: '#f1f5f9'
                        }
                    },
                    x: {
                        grid: {
                            display: false
                        }
                    }
                }
            }
        });
    }
    
    function updateWellknownFilesChart(data) {
        var ctx = document.getElementById('wellknown-files-chart');
        if (!ctx || typeof Chart === 'undefined') return;
        
        if (charts.wellknownFiles) charts.wellknownFiles.destroy();
        if (!data || data.length === 0) return;
        
        var labels = data.map(function(item) { return item.file_path.replace('/.well-known/', ''); });
        var hits = data.map(function(item) { return parseInt(item.hits || 0); });
        
        charts.wellknownFiles = new Chart(ctx, {
            type: 'bar',
            data: {
                labels: labels,
                datasets: [{
                    label: 'Total Hits',
                    data: hits,
                    backgroundColor: '#f59e0b',
                    borderRadius: 8
                }]
            },
            options: {
                responsive: true,
                maintainAspectRatio: false,
                indexAxis: 'y',
                plugins: {
                    legend: {
                        display: false
                    }
                },
                scales: {
                    x: {
                        beginAtZero: true,
                        grid: {
                            color: '#f1f5f9'
                        }
                    },
                    y: {
                        grid: {
                            display: false
                        }
                    }
                }
            }
        });
    }
    
    function updateWellknownDailyChart(data) {
        var ctx = document.getElementById('wellknown-daily-chart');
        if (!ctx || typeof Chart === 'undefined') return;
        
        if (charts.wellknownDaily) charts.wellknownDaily.destroy();
        if (!data || data.length === 0) return;
        
        var labels = data.map(function(item) { return formatDate(item.date); });
        var hits = data.map(function(item) { return parseInt(item.hits || 0); });
        
        charts.wellknownDaily = new Chart(ctx, {
            type: 'line',
            data: {
                labels: labels,
                datasets: [{
                    label: 'Well-Known Requests',
                    data: hits,
                    borderColor: '#f59e0b',
                    backgroundColor: 'rgba(245, 158, 11, 0.1)',
                    fill: true,
                    tension: 0.4,
                    borderWidth: 2
                }]
            },
            options: {
                responsive: true,
                maintainAspectRatio: false,
                plugins: {
                    legend: {
                        display: false
                    }
                },
                scales: {
                    y: {
                        beginAtZero: true,
                        grid: {
                            color: '#f1f5f9'
                        }
                    },
                    x: {
                        grid: {
                            display: false
                        }
                    }
                }
            }
        });
    }
    
    function updateContentPerformanceChart(data) {
        var ctx = document.getElementById('content-performance-chart');
        if (!ctx || typeof Chart === 'undefined') return;
        
        if (charts.contentPerformance) charts.contentPerformance.destroy();
        if (!data || data.length === 0) return;
        
        var typeGroups = {};
        data.forEach(function(item) {
            var type = item.post_type || 'page';
            if (!typeGroups[type]) {
                typeGroups[type] = 0;
            }
            typeGroups[type] += parseInt(item.visits || 0);
        });
        
        var labels = Object.keys(typeGroups);
        var visits = Object.values(typeGroups);
        var colors = ['#3b82f6', '#8b5cf6', '#10b981', '#f59e0b', '#ec4899', '#14b8a6'];
        
        charts.contentPerformance = new Chart(ctx, {
            type: 'pie',
            data: {
                labels: labels,
                datasets: [{
                    data: visits,
                    backgroundColor: colors,
                    borderWidth: 3,
                    borderColor: '#fff'
                }]
            },
            options: {
                responsive: true,
                maintainAspectRatio: false,
                plugins: {
                    legend: {
                        position: 'bottom',
                        labels: {
                            usePointStyle: true,
                            padding: 15
                        }
                    }
                }
            }
        });
    }
    
    function updateHourlyActivityChart(data) {
        var ctx = document.getElementById('hourly-activity-chart');
        if (!ctx || typeof Chart === 'undefined') return;
        
        if (charts.hourlyActivity) charts.hourlyActivity.destroy();
        
        var hourlyData = new Array(24).fill(0);
        data.forEach(function(item) {
            var hour = parseInt(item.hour || 0);
            hourlyData[hour] = parseInt(item.visits || 0);
        });
        
        var labels = [];
        for (var i = 0; i < 24; i++) {
            labels.push(i + ':00');
        }
        
        charts.hourlyActivity = new Chart(ctx, {
            type: 'bar',
            data: {
                labels: labels,
                datasets: [{
                    label: 'Visits',
                    data: hourlyData,
                    backgroundColor: '#3b82f6',
                    borderRadius: 8
                }]
            },
            options: {
                responsive: true,
                maintainAspectRatio: false,
                plugins: {
                    legend: {
                        display: false
                    }
                },
                scales: {
                    y: {
                        beginAtZero: true,
                        grid: {
                            color: '#f1f5f9'
                        }
                    },
                    x: {
                        grid: {
                            display: false
                        }
                    }
                }
            }
        });
    }
    
    // ==================== TABLE FUNCTIONS ====================
    
    function updateCrawlersTable(data) {
        var tbody = $('#crawlers-table tbody');
        tbody.empty();
        
        if (!data || data.length === 0) {
            tbody.append('<tr><td colspan="4" class="aiox-loading-cell">No crawler data available</td></tr>');
            return;
        }
        
        data.forEach(function(item) {
            var typeClass = item.type === 'ai' ? 'ai' : (item.type === 'search' ? 'search' : 'unknown');
            var row = $('<tr>')
                .append($('<td>').html('<strong>' + escapeHtml(item.name || 'Unknown') + '</strong>'))
                .append($('<td>').html('<span class="aiox-badge ' + typeClass + '">' + capitalizeFirst(item.type || 'Unknown') + '</span>'))
                .append($('<td>').html('<strong>' + formatNumber(item.visits || 0) + '</strong>'))
                .append($('<td>').text(formatDateTime(item.last_seen || '')));
            tbody.append(row);
        });
    }
    
    function updateBadgePostsTable(data) {
        var tbody = $('#badge-posts-table tbody');
        tbody.empty();
        
        if (!data || data.length === 0) {
            tbody.append('<tr><td colspan="4" class="aiox-loading-cell">No badge view data available</td></tr>');
            return;
        }
        
        data.forEach(function(item) {
            var row = $('<tr>')
                .append($('<td>').html('<strong>' + escapeHtml(item.post_title || 'Untitled') + '</strong>'))
                .append($('<td>').html('<strong>' + formatNumber(item.views || 0) + '</strong>'))
                .append($('<td>').html('<span style="color: #8b5cf6; font-weight: 600;">' + formatNumber(item.ai_views || 0) + '</span>'))
                .append($('<td>').html('<span style="color: #10b981; font-weight: 600;">' + formatNumber(item.human_views || 0) + '</span>'));
            tbody.append(row);
        });
    }
    
    function updateWellknownCrawlersTable(data) {
        var tbody = $('#wellknown-crawlers-table tbody');
        tbody.empty();
        
        if (!data || data.length === 0) {
            tbody.append('<tr><td colspan="3" class="aiox-loading-cell">No well-known file access data available</td></tr>');
            return;
        }
        
        data.forEach(function(item) {
            var typeClass = item.crawler_type === 'ai' ? 'ai' : (item.crawler_type === 'search' ? 'search' : 'unknown');
            var row = $('<tr>')
                .append($('<td>').html('<strong>' + escapeHtml(item.crawler_name || 'Unknown') + '</strong>'))
                .append($('<td>').html('<span class="aiox-badge ' + typeClass + '">' + capitalizeFirst(item.crawler_type || 'Unknown') + '</span>'))
                .append($('<td>').html('<strong>' + formatNumber(item.hits || 0) + '</strong>'));
            tbody.append(row);
        });
    }
    
    function updateTopContentTable(data) {
        var tbody = $('#top-content-table tbody');
        tbody.empty();
        
        if (!data || data.length === 0) {
            tbody.append('<tr><td colspan="4" class="aiox-loading-cell">No content data available</td></tr>');
            return;
        }
        
        data.forEach(function(item) {
            var editUrl = window.location.origin + '/wp-admin/post.php?post=' + item.post_id + '&action=edit';
            var viewUrl = window.location.origin + '/?p=' + item.post_id;
            
            var row = $('<tr>')
                .append($('<td>').html('<strong>' + escapeHtml(item.post_title || 'Untitled') + '</strong>'))
                .append($('<td>').html('<span class="aiox-badge ' + (item.post_type || 'post') + '">' + capitalizeFirst(item.post_type || 'Post') + '</span>'))
                .append($('<td>').html('<strong>' + formatNumber(item.crawler_visits || 0) + '</strong>'))
                .append($('<td>').html(
                    '<a href="' + editUrl + '" target="_blank" class="button button-small">Edit</a> ' +
                    '<a href="' + viewUrl + '" target="_blank" class="button button-small">View</a>'
                ));
            tbody.append(row);
        });
    }
    
    function updateTopPagesTable(data) {
        var tbody = $('#top-pages-table tbody');
        tbody.empty();
        
        if (!data || data.length === 0) {
            tbody.append('<tr><td colspan="3" class="aiox-loading-cell">No page data available</td></tr>');
            return;
        }
        
        data.forEach(function(item) {
            var isWellKnown = item.request_uri && item.request_uri.indexOf('/.well-known/') !== -1;
            var pageType = isWellKnown ? 'Well-Known' : 'Page';
            var badgeClass = isWellKnown ? 'ai' : 'unknown';
            
            var row = $('<tr>')
                .append($('<td>').html('<code>' + escapeHtml(item.request_uri || '/') + '</code>'))
                .append($('<td>').html('<strong>' + formatNumber(item.visits || 0) + '</strong>'))
                .append($('<td>').html('<span class="aiox-badge ' + badgeClass + '">' + pageType + '</span>'));
            tbody.append(row);
        });
    }
    
    // ==================== UTILITY FUNCTIONS ====================
    
    function formatNumber(num) {
        if (num >= 1000000) {
            return (num / 1000000).toFixed(1) + 'M';
        } else if (num >= 1000) {
            return (num / 1000).toFixed(1) + 'K';
        }
        return num.toString();
    }
    
    function formatDate(dateString) {
        if (!dateString) return '';
        var date = new Date(dateString);
        return date.toLocaleDateString('en-US', { month: 'short', day: 'numeric' });
    }
    
    function formatDateTime(dateString) {
        if (!dateString) return 'Never';
        var date = new Date(dateString);
        return date.toLocaleDateString('en-US', { 
            month: 'short', 
            day: 'numeric',
            hour: '2-digit',
            minute: '2-digit'
        });
    }
    
    function capitalizeFirst(str) {
        if (!str) return '';
        return str.charAt(0).toUpperCase() + str.slice(1);
    }
    
    /**
     * Decode HTML entities (handles double/triple encoding)
     */
    function decodeHtml(text) {
        if (!text) return '';
        var txt = document.createElement('textarea');
        txt.innerHTML = text;
        var decoded = txt.value;
        // Decode again to handle double-encoding
        if (decoded !== text) {
            txt.innerHTML = decoded;
            decoded = txt.value;
        }
        return decoded;
    }
    
    /**
     * Escape HTML for safe display (first decodes to handle pre-encoded content)
     */
    function escapeHtml(text) {
        if (!text) return '';
        // First decode any existing HTML entities to avoid double-encoding
        var decoded = decodeHtml(text);
        var div = document.createElement('div');
        div.textContent = decoded;
        return div.innerHTML;
    }
    
    function showLoading() {
        $('#aiox-analytics-loading').fadeIn(200);
    }
    
    function hideLoading() {
        $('#aiox-analytics-loading').fadeOut(200);
    }
    
    function showError(message) {
        console.error('Analytics Error:', message);
        alert(message);
    }
    
    function showExportModal() {
        $('#export-period').val(currentPeriod);
        $('#aiox-export-modal').fadeIn(200);
    }
    
    function hideExportModal() {
        $('#aiox-export-modal').fadeOut(200);
    }
    
    function downloadExport() {
        var format = $('input[name="export-format"]:checked').val();
        var days = parseInt($('#export-period').val());
        
        $('#aiox-export-download').prop('disabled', true).html('<span class="aiox-spinner-small"></span> Generating...');
        
        $.ajax({
            url: aioxModernAnalytics.ajaxUrl,
            type: 'POST',
            data: {
                action: 'aiox_modern_analytics_export',
                nonce: aioxModernAnalytics.nonce,
                format: format,
                days: days
            },
            success: function(response) {
                if (response.success) {
                    var data = atob(response.data.data);
                    var blob = new Blob([data], { type: response.data.mime_type });
                    var url = window.URL.createObjectURL(blob);
                    var a = document.createElement('a');
                    a.href = url;
                    a.download = response.data.filename;
                    document.body.appendChild(a);
                    a.click();
                    document.body.removeChild(a);
                    window.URL.revokeObjectURL(url);
                    
                    hideExportModal();
                } else {
                    alert('Export failed: ' + (response.data.message || 'Unknown error'));
                }
            },
            error: function() {
                alert('Export failed: Network error');
            },
            complete: function() {
                $('#aiox-export-download').prop('disabled', false).html('<span class="dashicons dashicons-download"></span> Download Export');
            }
        });
    }
    
    function showResetConfirmation() {
        if (confirm('⚠️ WARNING: This will permanently delete ALL analytics data.\n\nThis includes:\n• All visitor tracking data\n• Daily statistics\n• Crawler records\n• Badge view history\n\nThis action cannot be undone. Are you sure you want to continue?')) {
            if (confirm('Are you absolutely sure? Type "RESET" to confirm.\n\n(Click OK to proceed or Cancel to abort)')) {
                resetAnalytics();
            }
        }
    }
    
    function resetAnalytics() {
        var $btn = $('#aiox-analytics-reset');
        $btn.prop('disabled', true).html('<span class="dashicons dashicons-update aiox-spin"></span> Resetting...');
        
        $.ajax({
            url: aioxModernAnalytics.ajaxUrl,
            type: 'POST',
            data: {
                action: 'aiox_modern_analytics_reset',
                nonce: aioxModernAnalytics.nonce
            },
            success: function(response) {
                if (response.success) {
                    alert('✓ ' + response.data.message);
                    // Reload the page to show empty state
                    location.reload();
                } else {
                    alert('Reset failed: ' + (response.data.message || 'Unknown error'));
                }
            },
            error: function() {
                alert('Reset failed: Network error');
            },
            complete: function() {
                $btn.prop('disabled', false).html('<span class="dashicons dashicons-trash"></span> Reset');
            }
        });
    }
    
    // Visitor Logs Functions
    let visitorLogsPage = 1;
    
    function loadVisitorLogs(page) {
        page = page || 1;
        visitorLogsPage = page;
        
        var typeFilter = $('#visitor-type-filter').val() || '';
        
        $('#visitor-logs-body').html('<tr><td colspan="6" class="aiox-loading-cell"><div class="aiox-spinner-small"></div> Loading visitor logs...</td></tr>');
        
        $.ajax({
            url: aioxModernAnalytics.ajaxUrl,
            type: 'POST',
            data: {
                action: 'aiox_get_visitor_logs',
                nonce: aioxModernAnalytics.nonce,
                page: page,
                per_page: 50,
                type: typeFilter
            },
            success: function(response) {
                if (response.success) {
                    renderVisitorLogs(response.data);
                } else {
                    $('#visitor-logs-body').html('<tr><td colspan="6" class="aiox-loading-cell">Failed to load logs</td></tr>');
                }
            },
            error: function(xhr, status, error) {
                console.error('Failed to load visitor logs:', error);
                $('#visitor-logs-body').html('<tr><td colspan="6" class="aiox-loading-cell">Error loading logs: ' + error + '</td></tr>');
            }
        });
    }
    
    function renderVisitorLogs(data) {
        var logs = data.logs || [];
        var $tbody = $('#visitor-logs-body');
        
        if (logs.length === 0) {
            $tbody.html('<tr><td colspan="6" class="aiox-loading-cell">No visitor logs found.</td></tr>');
            $('#visitor-logs-pagination').html('');
            return;
        }
        
        var html = '';
        logs.forEach(function(log) {
            // Display "browser" instead of "human" for consistency (case-insensitive)
            var typeLower = (log.type || '').toLowerCase();
            var displayType = (typeLower === 'human') ? 'browser' : log.type;
            var typeClass = 'aiox-type-' + typeLower; // Keep original class for styling
            var typeBadge = '<span class="aiox-visitor-type ' + typeClass + '">' + escapeHtml(displayType) + '</span>';
            
            html += '<tr>';
            html += '<td>' + escapeHtml(log.time) + '</td>';
            html += '<td>' + typeBadge + '</td>';
            html += '<td>' + escapeHtml(log.name) + '</td>';
            html += '<td><code style="font-size: 11px;">' + escapeHtml(log.ip) + '</code></td>';
            html += '<td><span style="font-size: 11px; color: #666; word-break: break-all;">' + escapeHtml(log.user_agent) + '</span></td>';
            html += '<td><a href="' + escapeHtml(log.endpoint) + '" target="_blank" style="font-size: 11px;">' + truncateUrl(log.endpoint, 30) + '</a></td>';
            html += '</tr>';
        });
        
        $tbody.html(html);
        
        // Render pagination
        renderVisitorLogsPagination(data);
    }
    
    function renderVisitorLogsPagination(data) {
        var $pagination = $('#visitor-logs-pagination');
        var totalPages = data.total_pages || 1;
        var currentPage = data.page || 1;
        
        if (totalPages <= 1) {
            $pagination.html('<span style="color: #666;">Showing ' + data.logs.length + ' of ' + data.total + ' entries</span>');
            return;
        }
        
        var html = '<span style="color: #666; margin-right: 15px;">Page ' + currentPage + ' of ' + totalPages + ' (' + data.total + ' total)</span>';
        
        if (currentPage > 1) {
            html += '<button type="button" class="aiox-btn aiox-btn-secondary" onclick="window.aioxLoadVisitorLogs(' + (currentPage - 1) + ')">← Previous</button> ';
        }
        
        if (currentPage < totalPages) {
            html += '<button type="button" class="aiox-btn aiox-btn-secondary" onclick="window.aioxLoadVisitorLogs(' + (currentPage + 1) + ')">Next →</button>';
        }
        
        $pagination.html(html);
    }
    
    function truncateUrl(url, maxLen) {
        if (!url) return '';
        if (url.length <= maxLen) return url;
        return url.substring(0, maxLen) + '...';
    }
    
    function escapeHtml(text) {
        if (!text) return '';
        var div = document.createElement('div');
        div.textContent = text;
        return div.innerHTML;
    }
    
    // Expose loadVisitorLogs for pagination buttons
    window.aioxLoadVisitorLogs = loadVisitorLogs;
    
})(jQuery);