<?php
/**
 * AIOX External AI Client
 * 
 * Handles communication with external AI services for enhanced processing
 */

// Prevent direct access
if (!defined('ABSPATH')) {
    exit;
}

class AIOX_AI_Client {
    
    /**
     * Default AI endpoint
     */
    private $ai_endpoint;
    
    /**
     * API timeout in seconds
     */
    private $timeout = 30;
    
    /**
     * Maximum retry attempts
     */
    private $max_retries = 3;
    
    /**
     * Initialize AI client
     */
    public function __construct() {
        $this->ai_endpoint = get_option('aiox_ai_endpoint', AIOX_DEFAULT_AI_ENDPOINT);
    }
    
    /**
     * Check if external AI is available
     */
    public function is_available() {
        // Check if Pro license is active
        if (!class_exists('AIOX_Licensing') || !AIOX_Licensing::is_pro_feature_available('external_ai')) {
            return false;
        }
        
        // Check if AI endpoint is configured
        if (empty($this->ai_endpoint)) {
            return false;
        }
        
        return true;
    }
    
    /**
     * Process content with external AI
     */
    public function process_content($content, $options = array()) {
        if (!$this->is_available()) {
            throw new Exception(__('External AI processing is not available', 'aiox-publisher-suite-pro'));
        }
        
        $default_options = array(
            'method' => 'ingredients',
            'language' => get_locale(),
            'max_ingredients' => 20,
            'include_metadata' => true
        );
        
        $options = wp_parse_args($options, $default_options);
        
        $request_data = array(
            'content' => $content,
            'options' => $options,
            'site_url' => home_url(),
            'timestamp' => current_time('c')
        );
        
        $response = $this->make_request('process', $request_data);
        
        if (is_wp_error($response)) {
            throw new Exception($response->get_error_message());
        }
        
        return $response;
    }
    
    /**
     * Enhance basic ingredients with AI
     */
    public function enhance_ingredients($basic_ingredients, $content) {
        if (!$this->is_available()) {
            // Return basic ingredients if AI is not available
            return $basic_ingredients;
        }
        
        try {
            $request_data = array(
                'basic_ingredients' => $basic_ingredients,
                'content' => $content,
                'action' => 'enhance',
                'site_url' => home_url(),
                'timestamp' => current_time('c')
            );
            
            $response = $this->make_request('enhance', $request_data);
            
            if (is_wp_error($response)) {
                AIOX_Logger::warning('AI enhancement failed, using basic ingredients', array(
                    'error' => $response->get_error_message()
                ));
                return $basic_ingredients;
            }
            
            // Log successful enhancement
            AIOX_Logger::info('AI enhancement completed', array(
                'basic_count' => count($basic_ingredients),
                'enhanced_count' => isset($response['ingredients']) ? count($response['ingredients']) : 0
            ));
            
            return $response;
            
        } catch (Exception $e) {
            AIOX_Logger::warning('AI enhancement failed, using basic ingredients', array(
                'error' => $e->getMessage()
            ));
            return $basic_ingredients;
        }
    }
    
    /**
     * Make HTTP request to AI service
     */
    private function make_request($endpoint, $data, $retry_count = 0) {
        $url = trailingslashit($this->ai_endpoint) . $endpoint;
        
        $args = array(
            'method' => 'POST',
            'timeout' => $this->timeout,
            'headers' => array(
                'Content-Type' => 'application/json',
                'User-Agent' => 'AIOX-Plugin/' . AIOX_VERSION,
                'X-Site-URL' => home_url()
            ),
            'body' => wp_json_encode($data)
        );
        
        // Add license key if available
        $license_key = get_option('aiox_license_key', '');
        if (!empty($license_key)) {
            $args['headers']['Authorization'] = 'Bearer ' . $license_key;
        }
        
        AIOX_Logger::debug('Making AI request', array(
            'url' => $url,
            'endpoint' => $endpoint,
            'retry_count' => $retry_count
        ));
        
        $response = wp_remote_request($url, $args);
        
        if (is_wp_error($response)) {
            if ($retry_count < $this->max_retries) {
                AIOX_Logger::warning('AI request failed, retrying', array(
                    'error' => $response->get_error_message(),
                    'retry_count' => $retry_count + 1
                ));
                
                // Wait before retry (exponential backoff)
                sleep(pow(2, $retry_count));
                
                return $this->make_request($endpoint, $data, $retry_count + 1);
            }
            
            AIOX_Logger::error('AI request failed after retries', array(
                'error' => $response->get_error_message(),
                'retry_count' => $retry_count
            ));
            
            return $response;
        }
        
        $response_code = wp_remote_retrieve_response_code($response);
        $response_body = wp_remote_retrieve_body($response);
        
        if ($response_code !== 200) {
            $error_message = sprintf(
                __('AI service returned error code %d', 'aiox-publisher-suite-pro'),
                $response_code
            );
            
            AIOX_Logger::error('AI service error', array(
                'response_code' => $response_code,
                'response_body' => $response_body
            ));
            
            return new WP_Error('ai_service_error', $error_message);
        }
        
        $decoded_response = json_decode($response_body, true);
        
        if (json_last_error() !== JSON_ERROR_NONE) {
            AIOX_Logger::error('Invalid JSON response from AI service', array(
                'json_error' => json_last_error_msg(),
                'response_body' => $response_body
            ));
            
            return new WP_Error('invalid_json', __('Invalid response from AI service', 'aiox-publisher-suite-pro'));
        }
        
        AIOX_Logger::info('AI request successful', array(
            'endpoint' => $endpoint,
            'response_size' => strlen($response_body)
        ));
        
        return $decoded_response;
    }
    
    /**
     * Test AI service connection
     */
    public function test_connection() {
        if (!$this->is_available()) {
            return array(
                'success' => false,
                'message' => __('External AI is not available (Pro license required)', 'aiox-publisher-suite-pro')
            );
        }
        
        try {
            $test_data = array(
                'action' => 'test',
                'site_url' => home_url(),
                'timestamp' => current_time('c')
            );
            
            $response = $this->make_request('test', $test_data);
            
            if (is_wp_error($response)) {
                return array(
                    'success' => false,
                    'message' => $response->get_error_message()
                );
            }
            
            return array(
                'success' => true,
                'message' => __('AI service connection successful', 'aiox-publisher-suite-pro'),
                'response' => $response
            );
            
        } catch (Exception $e) {
            return array(
                'success' => false,
                'message' => $e->getMessage()
            );
        }
    }
    
    /**
     * Get AI service status
     */
    public function get_service_status() {
        $status = array(
            'available' => $this->is_available(),
            'endpoint' => $this->ai_endpoint,
            'license_valid' => class_exists('AIOX_Licensing') && AIOX_Licensing::is_license_valid(),
            'pro_feature_available' => class_exists('AIOX_Licensing') && AIOX_Licensing::is_pro_feature_available('external_ai')
        );
        
        if ($status['available']) {
            $connection_test = $this->test_connection();
            $status['connection'] = $connection_test['success'];
            $status['connection_message'] = $connection_test['message'];
        }
        
        return $status;
    }
    
    /**
     * Update AI endpoint
     */
    public function update_endpoint($endpoint) {
        $this->ai_endpoint = esc_url_raw($endpoint);
        update_option('aiox_ai_endpoint', $this->ai_endpoint);
        
        AIOX_Logger::info('AI endpoint updated', array(
            'new_endpoint' => $this->ai_endpoint
        ));
    }
}
