<?php
/**
 * AIOX AI-Only Enforcer
 * 
 * This class enforces the AI-only processing requirement for Version 2.0.1+
 */

if (!defined('ABSPATH')) {
    exit;
}

class AIOX_AI_Only_Enforcer {
    
    /**
     * Initialize the enforcer
     */
    public static function init() {
        // Load the API validator
        require_once AIOX_PLUGIN_DIR . 'core/class-aiox-api-validator.php';
        
        // Force upgrade from basic mode on plugin load
        add_action('plugins_loaded', array(__CLASS__, 'force_upgrade_check'));
        
        // Add filters and actions to enforce AI-only mode
        add_filter('pre_update_option_aiox_ai_provider', array(__CLASS__, 'enforce_ai_provider'), 10, 2);
        add_action('admin_notices', array(__CLASS__, 'show_admin_notices'));
        
        // Prevent content processing without valid API keys
        add_filter('aiox_can_process_content', array(__CLASS__, 'validate_processing_capability'));
    }
    
    /**
     * Force upgrade check on plugin load
     */
    public static function force_upgrade_check() {
        AIOX_API_Validator::force_upgrade_from_basic();
    }
    
    /**
     * Enforce the selection of an AI provider
     */
    public static function enforce_ai_provider($new_value, $old_value) {
        if ($new_value === 'basic') {
            // Prevent switching to basic mode
            add_settings_error(
                'aiox_ai_provider',
                'aiox_ai_provider_error',
                __('Basic extraction is no longer supported. Please select an AI provider (OpenAI or Gemini).', 'aiox-publisher-suite-pro'),
                'error'
            );
            
            // Force to OpenAI as default
            return 'openai';
        }
        return $new_value;
    }
    
    /**
     * Show admin notices for API configuration and upgrades
     */
    public static function show_admin_notices() {
        // Show upgrade notice if needed
        AIOX_API_Validator::show_upgrade_notice();
        
        // Show API key requirement notice on AIOX pages
        $screen = get_current_screen();
        if (strpos($screen->id, 'aiox') === false) {
            return; // Only show on AIOX pages
        }
        
        $validation = AIOX_API_Validator::validate_current_provider();
        
        if (!$validation['valid']) {
            $notice_class = 'notice-error';
            if ($validation['action_required'] === 'select_provider') {
                $notice_class = 'notice-warning';
            }
            
            ?>
            <div class="notice <?php echo esc_attr($notice_class); ?> is-dismissible">
                <p>
                    <strong><?php _e('AIOX Configuration Required:', 'aiox-publisher-suite-pro'); ?></strong>
                    <?php echo esc_html($validation['message']); ?>
                    <a href="<?php echo esc_url(admin_url('admin.php?page=aiox-settings')); ?>" class="button button-primary" style="margin-left: 10px;">
                        <?php _e('Configure Settings', 'aiox-publisher-suite-pro'); ?>
                    </a>
                </p>
            </div>
            <?php
        }
    }
    
    /**
     * Validate processing capability
     */
    public static function validate_processing_capability($can_process) {
        return AIOX_API_Validator::can_process_content();
    }
}

