<?php
/**
 * AIOX AI-Only Processor
 * 
 * Version 2.0.1+ processor that enforces AI-only content processing
 */

if (!defined('ABSPATH')) {
    exit;
}

class AIOX_AI_Only_Processor {
    
    private $ai_processor;
    private $api_validator;
    
    /**
     * Constructor
     */
    public function __construct() {
        require_once AIOX_PLUGIN_DIR . 'core/class-aiox-ai-processor.php';
        require_once AIOX_PLUGIN_DIR . 'core/class-aiox-api-validator.php';
        
        $this->ai_processor = new AIOX_AI_Processor();
        $this->api_validator = new AIOX_API_Validator();
    }
    
    /**
     * Process post content with AI-only enforcement
     */
    public function process_post_content($post_id) {
        // Validate API configuration before processing
        $validation = AIOX_API_Validator::validate_current_provider();
        
        if (!$validation['valid']) {
            throw new Exception($validation['message']);
        }
        
        // Ensure we're not in basic mode
        $provider = get_option('aiox_ai_provider', 'openai');
        if ($provider === 'basic') {
            // Force upgrade and throw error
            AIOX_API_Validator::force_upgrade_from_basic();
            throw new Exception(__('Basic processing is no longer supported. Please configure an AI provider.', 'aiox-publisher-suite-pro'));
        }
        
        AIOX_Logger::info("AI-Only Processor: Processing post {$post_id} with provider: {$provider}");
        
        try {
            // Use the existing AI processor but ensure it's in AI mode
            $result = $this->ai_processor->process_post_content($post_id);
            
            // Verify the result is from AI processing
            if (isset($result['processing_method']) && $result['processing_method'] === 'basic') {
                throw new Exception(__('AI processing failed and basic fallback is disabled.', 'aiox-publisher-suite-pro'));
            }
            
            // Add AI-only metadata
            $result['ai_only_version'] = '2.0.1';
            $result['processing_enforced'] = true;
            $result['fallback_disabled'] = true;
            
            // Save processed data to post meta for badge display
            update_post_meta($post_id, '_aiox_processed_data', $result);
            
            AIOX_Logger::info("AI-Only Processor: Successfully processed post {$post_id}");
            
            return $result;
            
        } catch (Exception $e) {
            AIOX_Logger::error("AI-Only Processor: Failed to process post {$post_id}: " . $e->getMessage());
            
            // No fallback to basic - throw the error
            throw new Exception(
                sprintf(
                    __('AI content processing failed: %s. Please check your API configuration.', 'aiox-publisher-suite-pro'),
                    $e->getMessage()
                )
            );
        }
    }
    
    /**
     * Get processing statistics with AI-only enforcement
     */
    public function get_processing_stats() {
        $stats = $this->ai_processor->get_processing_stats();
        
        // Override basic mode information
        $stats['ai_only_mode'] = true;
        $stats['basic_fallback'] = false;
        $stats['version'] = '2.0.1';
        
        // Add validation status
        $validation = AIOX_API_Validator::validate_current_provider();
        $stats['configuration_valid'] = $validation['valid'];
        $stats['configuration_message'] = $validation['message'];
        
        return $stats;
    }
    
    /**
     * Check if AI processing is available (required in AI-only mode)
     */
    public function is_ai_available() {
        return AIOX_API_Validator::can_process_content();
    }
    
    /**
     * Get current provider (never basic in AI-only mode)
     */
    public function get_provider() {
        $provider = get_option('aiox_ai_provider', 'openai');
        
        // Force upgrade if somehow still basic
        if ($provider === 'basic') {
            AIOX_API_Validator::force_upgrade_from_basic();
            $provider = get_option('aiox_ai_provider', 'openai');
        }
        
        return $provider;
    }
    
    /**
     * Update processing timestamp
     */
    public function update_processing_timestamp() {
        $this->ai_processor->update_processing_timestamp();
        update_option('aiox_last_ai_only_processing', current_time('mysql'));
    }
    
    /**
     * Get supported providers (excludes basic)
     */
    public static function get_supported_providers() {
        return array(
            'openai' => array(
                'name' => 'OpenAI',
                'description' => 'GPT models for advanced content analysis',
                'models' => array(
                    'gpt-4o-mini' => 'GPT-4o Mini (Recommended)',
                    'gpt-4o' => 'GPT-4o (Premium)',
                    'gpt-4-turbo' => 'GPT-4 Turbo',
                    'gpt-3.5-turbo' => 'GPT-3.5 Turbo (Budget)'
                )
            ),
            'gemini' => array(
                'name' => 'Google Gemini',
                'description' => 'Google AI models for content processing',
                'models' => array(
                    'gemini-2.0-flash-lite' => 'Gemini 2.0 Flash Lite (Recommended)',
                    'gemini-2.0-flash-001' => 'Gemini 2.0 Flash',
                    'gemini-2.5-flash' => 'Gemini 2.5 Flash (Latest)',
                    'gemini-2.5-pro' => 'Gemini 2.5 Pro (Premium)',
                    'gemini-1.5-flash' => 'Gemini 1.5 Flash (Retired)',
                    'gemini-1.5-pro' => 'Gemini 1.5 Pro (Retired)'
                )
            )
        );
    }
    
    /**
     * Validate content before processing
     */
    public function validate_content_for_processing($post_id) {
        $post = get_post($post_id);
        
        if (!$post) {
            return array(
                'valid' => false,
                'message' => __('Post not found.', 'aiox-publisher-suite-pro')
            );
        }
        
        // Check API configuration
        $api_validation = AIOX_API_Validator::validate_current_provider();
        if (!$api_validation['valid']) {
            return array(
                'valid' => false,
                'message' => $api_validation['message']
            );
        }
        
        // Check content length
        $content_length = strlen(strip_tags($post->post_content));
        if ($content_length < 100) {
            return array(
                'valid' => false,
                'message' => __('Content is too short for AI processing (minimum 100 characters).', 'aiox-publisher-suite-pro')
            );
        }
        
        if ($content_length > 50000) {
            return array(
                'valid' => false,
                'message' => __('Content is too long for AI processing (maximum 50,000 characters).', 'aiox-publisher-suite-pro')
            );
        }
        
        return array(
            'valid' => true,
            'message' => __('Content is ready for AI processing.', 'aiox-publisher-suite-pro')
        );
    }
    
    /**
     * Get processing requirements for display
     */
    public static function get_processing_requirements() {
        return array(
            'api_key_required' => true,
            'supported_providers' => array_keys(self::get_supported_providers()),
            'minimum_content_length' => 100,
            'maximum_content_length' => 50000,
            'fallback_available' => false,
            'basic_mode_supported' => false
        );
    }
    
    /**
     * Process a post and return results
     */
    public function process_post($post_id) {
        try {
            $content = $this->process_post_content($post_id);
            
            return array(
                'success' => true,
                'data' => $content,
                'message' => 'Post processed successfully with AI'
            );
            
        } catch (Exception $e) {
            return array(
                'success' => false,
                'message' => $e->getMessage()
            );
        }
    }
    
    /**
     * Check if API is configured (required for capsule generation)
     */
    public function is_api_configured() {
        $validation = AIOX_API_Validator::validate_current_provider();
        return $validation['valid'];
    }
}
