<?php
/**
 * AIOX AI Content Processor - Safe Version (Custom Default + Compatibility)
 *
 * Keeps single-call AI workflow, restores helpers, prevents missing-file fatal errors.
 */

if (!defined('ABSPATH')) {
    exit;
}

class AIOX_AI_Processor {
    
    private $ai_client;
    private $provider;
    
    /**
     * Constructor
     */
    public function __construct() {
        $this->provider = function_exists('get_option') ? get_option('aiox_ai_provider', 'custom') : 'custom';
        $this->init_ai_client();
    }
    
    /**
     * Initialize AI client based on provider
     */
    private function init_ai_client() {
        $client_loaded = false;

        try {
            if ($this->provider === 'openai' && file_exists(AIOX_PLUGIN_DIR . 'core/class-aiox-openai-client.php')) {
                require_once AIOX_PLUGIN_DIR . 'core/class-aiox-openai-client.php';
                $this->ai_client = new AIOX_OpenAI_Client();
                $client_loaded = true;
            } elseif ($this->provider === 'gemini' && file_exists(AIOX_PLUGIN_DIR . 'core/class-aiox-gemini-client.php')) {
                require_once AIOX_PLUGIN_DIR . 'core/class-aiox-gemini-client.php';
                $this->ai_client = new AIOX_Gemini_Client();
                $client_loaded = true;
            }
        } catch (Throwable $e) {
            AIOX_Logger::log("AI client init failed for {$this->provider}: " . $e->getMessage(), AIOX_Logger::WARNING);
        }

        // Always fall back to Custom client if others not available
        if (!$client_loaded) {
            if (file_exists(AIOX_PLUGIN_DIR . 'core/class-aiox-custom-client.php')) {
                require_once AIOX_PLUGIN_DIR . 'core/class-aiox-custom-client.php';
                $this->ai_client = new AIOX_Custom_Client();
                $this->provider = 'custom';
                AIOX_Logger::log("Using Custom AI client as fallback", AIOX_Logger::INFO);
            } else {
                AIOX_Logger::log("No AI client found in /core/ directory", AIOX_Logger::ERROR);
                $this->ai_client = null;
            }
        }
    }
    
    /**
     * Process post content
     */
    public function process_post_content($post_id) {
        $post = get_post($post_id);
        if (!$post) {
            throw new Exception("Post not found: {$post_id}");
        }
        
        // Reload provider
        $this->provider = function_exists('get_option') ? get_option('aiox_ai_provider', 'custom') : 'custom';
        $this->init_ai_client();
        
        AIOX_Logger::log("Processing post {$post_id} using {$this->provider}", AIOX_Logger::INFO);
        
        $content = $this->extract_content($post);
        $title   = $post->post_title;
        $excerpt = $post->post_excerpt;
        
        try {
            if (!$this->is_ai_available()) {
                throw new Exception(__('API configuration required. Please configure your AI provider in AIOX Settings.', 'aiox-publisher-suite-pro'));
            }
            
            return $this->process_ai_enhanced_content($content, $title, $excerpt);
            
        } catch (Exception $e) {
            AIOX_Logger::log('AI processing failed: ' . $e->getMessage(), AIOX_Logger::ERROR);
            throw $e; // Re-throw instead of falling back to basic
        }
    }
    
    /**
     * Process with AI (single-call)
     */
    private function process_ai_enhanced_content($content, $title, $excerpt) {
        if (!$this->ai_client) {
            throw new Exception('AI client not available. Check configuration.');
        }

        try {
            AIOX_Logger::log("Processing content with {$this->provider}", AIOX_Logger::INFO);
            $result = $this->ai_client->process_content($content, $title, $excerpt);
            
            // Log response keys for debugging
            AIOX_Logger::log("API Response Keys: " . implode(', ', array_keys($result)), AIOX_Logger::INFO);
            AIOX_Logger::log("Metadata present: " . (isset($result['metadata']) ? 'YES (' . count($result['metadata']) . ' fields)' : 'NO'), AIOX_Logger::INFO);
            
            if (!is_array($result)) {
                $result = array('summary' => 'AI returned invalid response.');
            }
            $result['processing_method'] = $this->provider;
            $this->update_processing_timestamp();
            return $result;
        } catch (Exception $e) {
            throw new Exception('AI processing failed: ' . $e->getMessage());
        }
    }

    /**
     * Basic fallback (DEPRECATED - DO NOT USE)
     * @deprecated since 2.0.2 - All processing now requires API configuration
     */
    private function process_basic_content($content, $title, $excerpt) {
        throw new Exception(__('Basic processing is deprecated. API configuration is required for all processing.', 'aiox-publisher-suite-pro'));
    }
    
    /**
     * Extract content
     */
    /**
 /**
 * Extract clean readable content from a post or page.
 * Compatible with Breakdance, Elementor, Divi, WPBakery, and Gutenberg.
 * Includes a universal rendered-HTML fallback and caching.
 */
private function extract_content($post) {

    // ✅ 1️⃣ CACHE CHECK — return cached version if available
    $cached = get_post_meta($post->ID, '_aiox_extracted_content', true);
    if (!empty($cached)) return $cached;

    // Try normal WP content first
    $content = $post->post_content;

    // If empty or shortcode-heavy, try to render it
    if (empty(trim(strip_tags($content)))) {
        $rendered = apply_filters('the_content', $content);
        if (!empty(trim(strip_tags($rendered)))) {
            $content = $rendered;
        } else {
            // Builder fallback
            $builder_content = '';

            // Elementor fallback
            $elementor_data = get_post_meta($post->ID, '_elementor_data', true);
            if (!empty($elementor_data)) {
                $builder_content .= $this->extract_text_from_json($elementor_data);
            }

            // Breakdance fallback
            $breakdance_data = get_post_meta($post->ID, '_breakdance_data', true);
            if (!empty($breakdance_data)) {
                $builder_content .= $this->extract_text_from_json($breakdance_data);
            }

            // Divi/WPBakery fallback
            $shortcode_render = do_shortcode($post->post_content);
            if (!empty(trim(strip_tags($shortcode_render)))) {
                $builder_content .= ' ' . $shortcode_render;
            }

            // Universal rendered HTML fallback
            if (empty(trim(strip_tags($builder_content)))) {
                $permalink = get_permalink($post->ID);
                if ($permalink) {
                    $response = wp_remote_get($permalink, array('timeout' => 10));
                    if (!is_wp_error($response) && isset($response['body'])) {
                        $html = $response['body'];

                        // Extract readable text from body
                        if (preg_match('/<body[^>]*>(.*?)<\/body>/is', $html, $matches)) {
                            $html_body = $matches[1];
                            $builder_content .= wp_strip_all_tags($html_body);
                        } else {
                            $builder_content .= wp_strip_all_tags($html);
                        }
                    }
                }
            }

            if (!empty(trim($builder_content))) {
                $content = $builder_content;
            }
        }
    }

    // Final cleanup
    $content = strip_shortcodes($content);
    $content = wp_strip_all_tags($content, true);
    $content = preg_replace('/\s+/', ' ', $content);
    $content = trim($content);

    // ✅ 2️⃣ CACHE SAVE — store cleaned content for future use
    update_post_meta($post->ID, '_aiox_extracted_content', $content);

    return $content;
}


/**
 * Extracts human-readable text from complex JSON used by page builders.
 * Handles Breakdance double-encoded structures and Elementor trees.
 */
private function extract_text_from_json($json_string) {
    $decoded = json_decode($json_string, true);
    if (!$decoded) return '';

    // Breakdance often nests its actual content under tree_json_string
    if (isset($decoded['tree_json_string']) && is_string($decoded['tree_json_string'])) {
        $inner = json_decode($decoded['tree_json_string'], true);
        if ($inner) {
            $decoded = $inner; // replace with inner decoded structure
        }
    }

    $text = '';
    $iterator = new RecursiveIteratorIterator(new RecursiveArrayIterator($decoded));
    foreach ($iterator as $key => $value) {
        $lower = strtolower($key);
        if (in_array($lower, ['text', 'content', 'html'])) {
            $text .= ' ' . wp_strip_all_tags($value);
        }
    }

    // Clean excessive whitespace
    $text = preg_replace('/\s+/', ' ', $text);
    return trim($text);
}


// Stop here

    private function generate_basic_summary($sentences) {
        return empty($sentences) ? 'No content available.' : implode(' ', array_slice($sentences, 0, 3));
    }

    private function extract_basic_topics($content, $title) {
        $text = strtolower($title . ' ' . $content);
        $words = str_word_count($text, 1);
        $common = ['the','and','or','but','in','on','at','to','for','of','with','by','is','are','was','were','be','been','have','has','had','do','does','did','will','would','could','should','may','might','can','this','that','these','those','a','an'];
        $words = array_filter($words, fn($w) => strlen($w) > 3 && !in_array($w, $common));
        $counts = array_count_values($words);
        arsort($counts);
        return array_slice(array_keys($counts), 0, 10);
    }

    private function assess_complexity($content) {
        $words = str_word_count($content);
        $sentences = substr_count($content, '.') + substr_count($content, '!') + substr_count($content, '?');
        if ($sentences > 0) {
            $avg = $words / $sentences;
            return $avg > 20 ? 'advanced' : ($avg > 15 ? 'intermediate' : 'beginner');
        }
        return 'intermediate';
    }

    /**
     * Stats + utils
     */
    public function get_processing_stats() {
        $stats = [
            'provider' => $this->provider,
            'ai_available' => $this->ai_client !== null,
            'last_processed' => function_exists('get_option') ? get_option('aiox_last_ai_processing', 'Never') : 'Never'
        ];

        if ($this->ai_client && method_exists($this->ai_client, 'test_connection')) {
            $test = $this->ai_client->test_connection();
            $stats['connection_status'] = $test['success'] ? 'connected' : 'error';
            $stats['connection_message'] = $test['message'];
        } else {
            $stats['connection_status'] = 'not_configured';
            $stats['connection_message'] = 'AI provider not configured';
        }
        return $stats;
    }

    public function update_processing_timestamp() {
        update_option('aiox_last_ai_processing', current_time('mysql'));
    }

    public function is_ai_available() {
        return $this->provider !== 'basic' && $this->ai_client !== null;
    }

    public function get_provider() {
        return $this->provider;
    }
}
// Clear cached extracted content when a post is updated or saved
add_action('save_post', function($post_id) {
    // Prevent accidental recursion
    if (defined('DOING_AUTOSAVE') && DOING_AUTOSAVE) return;
    if (wp_is_post_revision($post_id)) return;
    delete_post_meta($post_id, '_aiox_extracted_content');
});