<?php
/**
 * AIOX API Validator
 * 
 * Validates API keys and ensures AI providers are properly configured
 */

if (!defined('ABSPATH')) {
    exit;
}

class AIOX_API_Validator {
    
    /**
     * Validate current AI provider configuration
     */
    public static function validate_current_provider() {
        return self::validate_custom_config();
    }
    
    /**
     * Validate Custom AI configuration
     */
    private static function validate_custom_config() {
        $api_key = get_option('aiox_custom_api_key', '');
        $endpoint = get_option('aiox_custom_endpoint', '');
        
        if (empty($api_key)) {
            return array(
                'valid' => false,
                'message' => __('Custom API key is required for content processing.', 'aiox-publisher-suite-pro'),
                'action_required' => 'enter_api_key',
                'provider' => 'custom'
            );
        }
        
        if (empty($endpoint)) {
            return array(
                'valid' => false,
                'message' => __('Custom API endpoint is required for content processing.', 'aiox-publisher-suite-pro'),
                'action_required' => 'enter_endpoint',
                'provider' => 'custom'
            );
        }
        
        // Basic URL validation
        if (!filter_var($endpoint, FILTER_VALIDATE_URL)) {
            return array(
                'valid' => false,
                'message' => __('Custom API endpoint must be a valid URL.', 'aiox-publisher-suite-pro'),
                'action_required' => 'check_endpoint',
                'provider' => 'custom'
            );
        }
        
        return array(
            'valid' => true,
            'message' => __('Custom AI configuration is valid.', 'aiox-publisher-suite-pro'),
            'provider' => 'custom'
        );
    }
    
    /**
     * Test API connection for current provider
     */
    public static function test_current_provider_connection() {
        $validation = self::validate_current_provider();
        
        if (!$validation['valid']) {
            return $validation;
        }
        
        try {
            require_once AIOX_PLUGIN_DIR . 'core/class-aiox-custom-client.php';
            $client = new AIOX_Custom_Client();
            return $client->test_connection();
            
        } catch (Exception $e) {
            return array(
                'valid' => false,
                'message' => sprintf(__('Connection test failed: %s', 'aiox-publisher-suite-pro'), $e->getMessage()),
                'action_required' => 'check_api_key',
                'provider' => 'custom'
            );
        }
    }
    
    /**
     * Check if content processing is allowed
     */
    public static function can_process_content() {
        $validation = self::validate_current_provider();
        return $validation['valid'];
    }
    
    /**
     * Get user-friendly error message for content processing
     */
    public static function get_processing_error_message() {
        $validation = self::validate_current_provider();
        
        if ($validation['valid']) {
            return null;
        }
        
        $message = $validation['message'];
        $settings_url = admin_url('admin.php?page=aiox-settings');
        
        return $message . ' ' . sprintf(
            __('<a href="%s">Configure AI settings</a> to enable content processing.', 'aiox-publisher-suite-pro'),
            $settings_url
        );
    }
    
    /**
     * Get validation status for dashboard display
     */
    public static function get_dashboard_status() {
        $validation = self::validate_current_provider();
        
        $status = array(
            'valid' => $validation['valid'],
            'message' => $validation['message'],
            'provider' => isset($validation['provider']) ? $validation['provider'] : 'none',
            'status_class' => $validation['valid'] ? 'success' : 'error',
            'icon' => $validation['valid'] ? '✅' : '❌'
        );
        
        // Add connection test if configuration is valid
        if ($validation['valid']) {
            $connection_test = self::test_current_provider_connection();
            // Check for both 'success' and 'valid' keys for compatibility
            $status['connection_valid'] = $connection_test['success'] ?? $connection_test['valid'] ?? false;
            $status['connection_message'] = $connection_test['connection_message'] ?? $connection_test['message'] ?? 'Unknown status';
            
            if (!$status['connection_valid']) {
                $status['status_class'] = 'warning';
                $status['icon'] = '⚠️';
                $status['message'] .= ' ' . __('Connection test failed.', 'aiox-publisher-suite-pro');
            }
        }
        
        return $status;
    }
    
    /**
     * Force upgrade from basic mode if still set
     */
    public static function force_upgrade_from_basic() {
        $provider = get_option('aiox_ai_provider', 'basic');
        
        if ($provider === 'basic') {
            // Set to OpenAI as default
            update_option('aiox_ai_provider', 'openai');
            
            // Add admin notice
            add_option('aiox_upgrade_notice', array(
                'message' => __('AIOX has been upgraded to AI-only processing. Please configure your OpenAI or Gemini API key.', 'aiox-publisher-suite-pro'),
                'type' => 'warning',
                'dismissible' => true
            ));
            
            return true;
        }
        
        return false;
    }
    
    /**
     * Show upgrade notice if needed
     */
    public static function show_upgrade_notice() {
        $notice = get_option('aiox_upgrade_notice');
        
        if ($notice && is_array($notice)) {
            $class = 'notice notice-' . $notice['type'];
            if ($notice['dismissible']) {
                $class .= ' is-dismissible';
            }
            
            echo '<div class="' . esc_attr($class) . '">';
            echo '<p><strong>' . __('AIOX Upgrade:', 'aiox-publisher-suite-pro') . '</strong> ' . esc_html($notice['message']) . '</p>';
            echo '</div>';
            
            // Remove notice after showing
            delete_option('aiox_upgrade_notice');
        }
    }
}
