<?php
/**
 * AIOX Basic Extractor for Comparison Only
 * 
 * This class provides basic content extraction ONLY for the AI Comparison tool
 * to demonstrate the difference between basic and AI-enhanced processing.
 * 
 * Version 2.0.1+: This is NOT used for actual content processing.
 */

if (!defined('ABSPATH')) {
    exit;
}

class AIOX_Basic_Extractor_Comparison {
    
    /**
     * Extract basic content data for comparison purposes only
     */
    public static function extract_for_comparison($post_id) {
        $post = get_post($post_id);
        if (!$post) {
            throw new Exception('Post not found');
        }
        
        $content = wp_strip_all_tags($post->post_content);
        $title = $post->post_title;
        $excerpt = $post->post_excerpt ?: wp_trim_words($content, 55);
        
        // Basic word and sentence analysis
        $word_count = str_word_count($content);
        $sentences = preg_split('/[.!?]+/', $content, -1, PREG_SPLIT_NO_EMPTY);
        $sentence_count = count($sentences);
        
        // Simple keyword extraction (most frequent words)
        $words = str_word_count(strtolower($content), 1);
        $word_freq = array_count_values($words);
        
        // Remove common stop words
        $stop_words = array('the', 'a', 'an', 'and', 'or', 'but', 'in', 'on', 'at', 'to', 'for', 'of', 'with', 'by', 'is', 'are', 'was', 'were', 'be', 'been', 'have', 'has', 'had', 'do', 'does', 'did', 'will', 'would', 'could', 'should', 'may', 'might', 'must', 'can', 'this', 'that', 'these', 'those', 'i', 'you', 'he', 'she', 'it', 'we', 'they', 'me', 'him', 'her', 'us', 'them');
        
        foreach ($stop_words as $stop_word) {
            unset($word_freq[$stop_word]);
        }
        
        arsort($word_freq);
        $keywords = array_slice(array_keys($word_freq), 0, 10);
        
        // Basic reading time calculation
        $reading_time = max(1, round($word_count / 200));
        
        // Simple content type detection
        $content_type = 'article';
        if (strpos($title, '?') !== false) {
            $content_type = 'question';
        } elseif (preg_match('/\bhow\s+to\b/i', $title)) {
            $content_type = 'tutorial';
        } elseif (preg_match('/\b(review|rating)\b/i', $title)) {
            $content_type = 'review';
        }
        
        // Basic Q&A generation (very simple)
        $qa_pairs = array();
        
        // Generate a basic question about the title
        $qa_pairs[] = array(
            'question' => 'What is this article about?',
            'answer' => 'This article discusses ' . strtolower($title) . '.',
            'category' => 'factual',
            'confidence' => 0.6
        );
        
        // Try to find sentences that could be answers
        foreach (array_slice($sentences, 0, 3) as $sentence) {
            $sentence = trim($sentence);
            if (strlen($sentence) > 20 && strlen($sentence) < 200) {
                $qa_pairs[] = array(
                    'question' => 'Can you tell me more about this topic?',
                    'answer' => $sentence . '.',
                    'category' => 'informational',
                    'confidence' => 0.4
                );
                break;
            }
        }
        
        // Basic summary (first few sentences)
        $summary_sentences = array_slice($sentences, 0, 2);
        $summary = implode('. ', $summary_sentences);
        if (strlen($summary) > 300) {
            $summary = substr($summary, 0, 297) . '...';
        }
        
        return array(
            'processing_method' => 'basic_comparison',
            'timestamp' => current_time('mysql'),
            'post_id' => $post_id,
            'title' => $title,
            'summary' => $summary,
            'excerpt' => $excerpt,
            'word_count' => $word_count,
            'sentence_count' => $sentence_count,
            'reading_time_minutes' => $reading_time,
            'content_type' => $content_type,
            'keywords' => $keywords,
            'key_points' => array_slice($sentences, 0, 3),
            'topics' => array_slice($keywords, 0, 5),
            'entities' => array(
                'people' => array(),
                'organizations' => array(),
                'locations' => array()
            ),
            'sentiment' => 'neutral',
            'complexity_level' => $word_count > 1000 ? 'advanced' : ($word_count > 500 ? 'intermediate' : 'basic'),
            'qa_pairs' => $qa_pairs,
            'metadata' => array(
                'author' => get_the_author_meta('display_name', $post->post_author),
                'published_date' => $post->post_date,
                'modified_date' => $post->post_modified,
                'categories' => wp_get_post_categories($post_id, array('fields' => 'names')),
                'tags' => wp_get_post_tags($post_id, array('fields' => 'names'))
            ),
            'limitations' => array(
                'No AI analysis',
                'Basic keyword extraction only',
                'Simple content type detection',
                'Limited entity recognition',
                'Basic Q&A generation',
                'No sentiment analysis',
                'No advanced summarization'
            ),
            'note' => 'This is a basic extraction for comparison purposes only. Actual content processing uses AI enhancement.'
        );
    }
    
    /**
     * Get comparison statistics
     */
    public static function get_comparison_stats() {
        return array(
            'method' => 'Basic Extraction (Comparison Only)',
            'capabilities' => array(
                'Word counting' => true,
                'Basic keyword extraction' => true,
                'Simple content type detection' => true,
                'Reading time calculation' => true,
                'Basic Q&A generation' => true,
                'AI-powered analysis' => false,
                'Entity recognition' => false,
                'Sentiment analysis' => false,
                'Advanced summarization' => false,
                'Topic modeling' => false,
                'Intent detection' => false,
                'Contextual understanding' => false
            ),
            'accuracy' => 'Low to Medium',
            'processing_time' => 'Fast',
            'api_required' => false,
            'cost' => 'Free',
            'use_case' => 'Comparison demonstration only'
        );
    }
    
    /**
     * Check if this extractor should be used (always false in v2.0.1+)
     */
    public static function should_use_for_processing() {
        return false; // Never use for actual processing in v2.0.1+
    }
    
    /**
     * Get warning message about basic extraction
     */
    public static function get_usage_warning() {
        return 'Basic extraction is only available for comparison purposes. All actual content processing uses AI enhancement in Version 2.0.1+.';
    }
}
