<?php
/**
 * AIOX Capsule Generator Class
 * 
 * Handles automated generation and packaging of content capsules
 * with AI enhancement and optimization.
 */

if (!defined('ABSPATH')) {
    exit;
}

require_once AIOX_PLUGIN_DIR . 'core/class-aiox-capsule-manager.php';
require_once AIOX_PLUGIN_DIR . 'core/class-aiox-ai-only-processor.php';

class AIOX_Capsule_Generator {
    
    /**
     * Capsule manager instance
     */
    private $capsule_manager;
    
    /**
     * AI processor instance
     */
    private $ai_processor;
    
    /**
     * Generation settings
     */
    private $settings;
    
    /**
     * Constructor
     */
    public function __construct() {
        $this->capsule_manager = new AIOX_Capsule_Manager();
        $this->ai_processor = new AIOX_AI_Only_Processor();
        
        $this->settings = array(
            'auto_generate' => function_exists('get_option') ? get_option('aiox_capsules_auto_generate', false) : false,
            'ai_enhance' => function_exists('get_option') ? get_option('aiox_capsules_ai_enhance', true) : true,
            'compression' => function_exists('get_option') ? get_option('aiox_capsules_compression', 'gzip') : 'gzip',
            'max_size' => function_exists('get_option') ? get_option('aiox_capsules_max_size', 10485760) : 10485760, // 10MB
            'batch_size' => function_exists('get_option') ? get_option('aiox_capsules_batch_size', 5) : 5,
            'include_media' => function_exists('get_option') ? get_option('aiox_capsules_include_media', false) : false,
            'include_comments' => function_exists('get_option') ? get_option('aiox_capsules_include_comments', false) : false
        );
    }
    
    /**
     * Generate capsule from single post
     */
    public function generate_post_capsule($post_id, $options = array()) {
        $post = get_post($post_id);
        if (!$post) {
            throw new Exception("Post not found: {$post_id}");
        }
        
        AIOX_Logger::log("Generating capsule for post {$post_id}", AIOX_Logger::INFO);
        
        $start_time = microtime(true);
        
        // Create base capsule
        $capsule = $this->capsule_manager->create_capsule_from_post($post_id);
        
        // Enhance with AI if enabled
        if ($this->settings['ai_enhance']) {
            $this->enhance_capsule_with_ai($capsule, $post);
        }
        
        // Add media if enabled
        if ($this->settings['include_media']) {
            $this->add_media_to_capsule($capsule, $post);
        }
        
        // Add comments if enabled
        if ($this->settings['include_comments']) {
            $this->add_comments_to_capsule($capsule, $post);
        }
        
        // Add SEO and metadata
        $this->add_seo_metadata($capsule, $post);
        
        // Add related content
        $this->add_related_content($capsule, $post);
        
        // Optimize and compress
        $this->optimize_capsule($capsule);
        
        // Update processing metadata
        $processing_time = microtime(true) - $start_time;
        $capsule->set_metadata('processing_time', $processing_time);
        $capsule->set_metadata('generated_at', current_time('mysql'));
        
        $capsule->save();
        
        AIOX_Logger::log("Generated capsule {$capsule->get_id()} in {$processing_time}s", AIOX_Logger::INFO);
        
        return $capsule;
    }
    
    /**
     * Generate collection capsule from multiple posts
     */
    public function generate_collection_capsule($post_ids, $title, $description = '', $options = array()) {
        AIOX_Logger::log("Generating collection capsule with " . count($post_ids) . " posts", AIOX_Logger::INFO);
        
        $start_time = microtime(true);
        
        // Create base collection capsule
        $capsule = $this->capsule_manager->create_collection_capsule($post_ids, $title, $description);
        
        // Enhance collection with AI
        if ($this->settings['ai_enhance']) {
            $this->enhance_collection_with_ai($capsule, $post_ids);
        }
        
        // Add cross-references and relationships
        $this->add_collection_relationships($capsule, $post_ids);
        
        // Generate collection insights
        $this->generate_collection_insights($capsule, $post_ids);
        
        // Optimize and compress
        $this->optimize_capsule($capsule);
        
        // Update processing metadata
        $processing_time = microtime(true) - $start_time;
        $capsule->set_metadata('processing_time', $processing_time);
        $capsule->set_metadata('generated_at', current_time('mysql'));
        
        $capsule->save();
        
        AIOX_Logger::log("Generated collection capsule {$capsule->get_id()} in {$processing_time}s", AIOX_Logger::INFO);
        
        return $capsule;
    }
    
    /**
     * Generate topic-based capsule
     */
    public function generate_topic_capsule($topic, $options = array()) {
        $defaults = array(
            'limit' => 10,
            'include_related' => true,
            'min_relevance' => 0.7
        );
        
        $options = wp_parse_args($options, $defaults);
        
        AIOX_Logger::log("Generating topic capsule for: {$topic}", AIOX_Logger::INFO);
        
        // Find relevant posts
        $post_ids = $this->find_topic_posts($topic, $options);
        
        if (empty($post_ids)) {
            throw new Exception("No relevant posts found for topic: {$topic}");
        }
        
        $title = "Topic Capsule: {$topic}";
        $description = "Comprehensive capsule covering the topic '{$topic}' with " . count($post_ids) . " related posts.";
        
        $capsule = $this->generate_collection_capsule($post_ids, $title, $description, $options);
        $capsule->set_type('topic');
        $capsule->add_tag($topic);
        
        // Add topic-specific enhancements
        $this->enhance_topic_capsule($capsule, $topic);
        
        $capsule->save();
        
        return $capsule;
    }
    
    /**
     * Generate author capsule
     */
    public function generate_author_capsule($author_id, $options = array()) {
        $author = get_userdata($author_id);
        if (!$author) {
            throw new Exception("Author not found: {$author_id}");
        }
        
        AIOX_Logger::log("Generating author capsule for: {$author->display_name}", AIOX_Logger::INFO);
        
        // Get author's posts
        $posts = get_posts(array(
            'author' => $author_id,
            'post_type' => array('post', 'page'),
            'post_status' => 'publish',
            'numberposts' => isset($options['limit']) ? $options['limit'] : 20,
            'orderby' => 'date',
            'order' => 'DESC'
        ));
        
        if (empty($posts)) {
            throw new Exception("No posts found for author: {$author->display_name}");
        }
        
        $post_ids = wp_list_pluck($posts, 'ID');
        
        $title = "Author Profile: {$author->display_name}";
        $description = "Complete author profile capsule for {$author->display_name} including " . count($posts) . " posts.";
        
        $capsule = $this->generate_collection_capsule($post_ids, $title, $description, $options);
        $capsule->set_type('author');
        
        // Add author-specific metadata
        $this->add_author_metadata($capsule, $author);
        
        $capsule->save();
        
        return $capsule;
    }
    
    /**
     * Enhance capsule with AI processing
     */
    private function enhance_capsule_with_ai($capsule, $post) {
        try {
            AIOX_Logger::log("Enhancing capsule with AI: {$capsule->get_id()}", AIOX_Logger::INFO);
            
            // Get or generate AI-enhanced content
            $processed_data = get_post_meta($post->ID, '_aiox_processed_data', true);
            
            if (!$processed_data || !isset($processed_data['processing_method']) || $processed_data['processing_method'] === 'basic') {
                // Process with AI if not already done
                $processed_data = $this->ai_processor->process_post_content($post->ID);
            }
            
            if ($processed_data && $processed_data['processing_method'] !== 'basic') {
                // Update capsule with AI-enhanced data
                $capsule->set_content('ai_insights', array(
                    'processing_method' => $processed_data['processing_method'],
                    'summary' => $processed_data['summary'] ?? '',
                    'key_points' => $processed_data['key_points'] ?? array(),
                    'topics' => $processed_data['topics'] ?? array(),
                    'entities' => $processed_data['entities'] ?? array(),
                    'sentiment' => $processed_data['sentiment'] ?? '',
                    'content_type' => $processed_data['content_type'] ?? '',
                    'readability_score' => $processed_data['readability_score'] ?? 0,
                    'word_count' => $processed_data['word_count'] ?? 0,
                    'reading_time' => $processed_data['reading_time'] ?? 0
                ));
                
                // Enhanced Q&A pairs
                if (isset($processed_data['qa_pairs'])) {
                    $capsule->set_content('qa_pairs', $processed_data['qa_pairs']);
                }
                
                // Update processing metadata
                $capsule->set_metadata('ai_enhanced', true);
                $capsule->set_metadata('ai_provider', $processed_data['processing_method']);
                
                if (isset($processed_data['tokens_used'])) {
                    $ai_processing = $capsule->get_metadata('ai_processing');
                    $ai_processing['tokens_used'] = $processed_data['tokens_used'];
                    $ai_processing['processed_at'] = current_time('mysql');
                    $capsule->set_metadata('ai_processing', $ai_processing);
                }
            }
            
        } catch (Exception $e) {
            AIOX_Logger::error("Failed to enhance capsule with AI: {$capsule->get_id()}", array('error' => $e->getMessage()));
        }
    }
    
    /**
     * Enhance collection with AI
     */
    private function enhance_collection_with_ai($capsule, $post_ids) {
        try {
            AIOX_Logger::log("Enhancing collection capsule with AI: {$capsule->get_id()}", AIOX_Logger::INFO);
            
            // Collect all content for analysis
            $combined_content = '';
            $all_topics = array();
            $all_entities = array();
            
            foreach ($post_ids as $post_id) {
                $post = get_post($post_id);
                if ($post) {
                    $combined_content .= $post->post_title . "\n" . $post->post_content . "\n\n";
                    
                    $processed_data = get_post_meta($post_id, '_aiox_processed_data', true);
                    if ($processed_data) {
                        if (isset($processed_data['topics'])) {
                            $all_topics = array_merge($all_topics, $processed_data['topics']);
                        }
                        if (isset($processed_data['entities'])) {
                            $all_entities = array_merge($all_entities, $processed_data['entities']);
                        }
                    }
                }
            }
            
            // Generate collection-level insights
            $collection_insights = array(
                'common_topics' => array_count_values($all_topics),
                'common_entities' => $this->analyze_common_entities($all_entities),
                'content_themes' => $this->extract_content_themes($combined_content),
                'collection_summary' => $this->generate_ai_collection_summary($combined_content),
                'content_relationships' => $this->analyze_content_relationships($post_ids)
            );
            
            $capsule->set_content('ai_insights', $collection_insights);
            $capsule->set_metadata('ai_enhanced', true);
            
        } catch (Exception $e) {
            AIOX_Logger::error("Failed to enhance collection with AI: {$capsule->get_id()}", array('error' => $e->getMessage()));
        }
    }
    
    /**
     * Add media to capsule
     */
    private function add_media_to_capsule($capsule, $post) {
        $media_items = array();
        
        // Featured image
        $featured_image_id = get_post_thumbnail_id($post->ID);
        if ($featured_image_id) {
            $media_items[] = $this->get_media_info($featured_image_id, 'featured_image');
        }
        
        // Gallery images
        $gallery_images = get_attached_media('image', $post->ID);
        foreach ($gallery_images as $image) {
            $media_items[] = $this->get_media_info($image->ID, 'gallery_image');
        }
        
        // Other attachments
        $attachments = get_attached_media('', $post->ID);
        foreach ($attachments as $attachment) {
            if (!wp_attachment_is_image($attachment->ID)) {
                $media_items[] = $this->get_media_info($attachment->ID, 'attachment');
            }
        }
        
        if (!empty($media_items)) {
            $capsule->set_content('attachments', $media_items);
        }
    }
    
    /**
     * Get media information
     */
    private function get_media_info($attachment_id, $type) {
        $attachment = get_post($attachment_id);
        $metadata = wp_get_attachment_metadata($attachment_id);
        
        return array(
            'id' => $attachment_id,
            'type' => $type,
            'title' => $attachment->post_title,
            'filename' => basename(get_attached_file($attachment_id)),
            'url' => wp_get_attachment_url($attachment_id),
            'mime_type' => $attachment->post_mime_type,
            'file_size' => filesize(get_attached_file($attachment_id)),
            'metadata' => $metadata,
            'alt_text' => get_post_meta($attachment_id, '_wp_attachment_image_alt', true),
            'caption' => $attachment->post_excerpt,
            'description' => $attachment->post_content
        );
    }
    
    /**
     * Add comments to capsule
     */
    private function add_comments_to_capsule($capsule, $post) {
        $comments = get_comments(array(
            'post_id' => $post->ID,
            'status' => 'approve',
            'type' => 'comment'
        ));
        
        $comment_data = array();
        foreach ($comments as $comment) {
            $comment_data[] = array(
                'id' => $comment->comment_ID,
                'author' => $comment->comment_author,
                'author_email' => $comment->comment_author_email,
                'author_url' => $comment->comment_author_url,
                'date' => $comment->comment_date,
                'content' => $comment->comment_content,
                'parent' => $comment->comment_parent,
                'meta' => get_comment_meta($comment->comment_ID)
            );
        }
        
        if (!empty($comment_data)) {
            $capsule->set_content('comments', $comment_data);
        }
    }
    
    /**
     * Add SEO metadata
     */
    private function add_seo_metadata($capsule, $post) {
        $seo_data = array(
            'meta_title' => get_post_meta($post->ID, '_yoast_wpseo_title', true),
            'meta_description' => get_post_meta($post->ID, '_yoast_wpseo_metadesc', true),
            'focus_keyword' => get_post_meta($post->ID, '_yoast_wpseo_focuskw', true),
            'canonical_url' => get_post_meta($post->ID, '_yoast_wpseo_canonical', true),
            'og_title' => get_post_meta($post->ID, '_yoast_wpseo_opengraph-title', true),
            'og_description' => get_post_meta($post->ID, '_yoast_wpseo_opengraph-description', true),
            'twitter_title' => get_post_meta($post->ID, '_yoast_wpseo_twitter-title', true),
            'twitter_description' => get_post_meta($post->ID, '_yoast_wpseo_twitter-description', true)
        );
        
        // Remove empty values
        $seo_data = array_filter($seo_data);
        
        if (!empty($seo_data)) {
            $existing_metadata = $capsule->get_content('metadata');
            $existing_metadata['seo'] = $seo_data;
            $capsule->set_content('metadata', $existing_metadata);
        }
    }
    
    /**
     * Add related content
     */
    private function add_related_content($capsule, $post) {
        // Find related posts by category
        $categories = wp_get_post_categories($post->ID);
        $related_posts = array();
        
        if (!empty($categories)) {
            $related = get_posts(array(
                'category__in' => $categories,
                'post__not_in' => array($post->ID),
                'numberposts' => 5,
                'post_status' => 'publish'
            ));
            
            foreach ($related as $related_post) {
                $related_posts[] = array(
                    'id' => $related_post->ID,
                    'title' => $related_post->post_title,
                    'url' => get_permalink($related_post->ID),
                    'excerpt' => $related_post->post_excerpt,
                    'date' => $related_post->post_date
                );
            }
        }
        
        if (!empty($related_posts)) {
            $capsule->set_content('references', $related_posts);
        }
    }
    
    /**
     * Optimize capsule
     */
    private function optimize_capsule($capsule) {
        // Check file size
        $current_size = $capsule->get_metadata('file_size');
        
        if ($current_size > $this->settings['max_size']) {
            AIOX_Logger::log("Optimizing large capsule: {$capsule->get_id()}", AIOX_Logger::INFO);
            
            // Reduce content if too large
            $this->reduce_capsule_content($capsule);
        }
        
        // Apply compression if enabled
        if ($this->settings['compression'] !== 'none') {
            $this->compress_capsule_content($capsule);
        }
    }
    
    /**
     * Reduce capsule content
     */
    private function reduce_capsule_content($capsule) {
        // Truncate long content
        $content = $capsule->get_content();
        
        if (isset($content['raw_content']) && is_string($content['raw_content'])) {
            if (strlen($content['raw_content']) > 50000) {
                $content['raw_content'] = substr($content['raw_content'], 0, 50000) . '... [truncated]';
            }
        }
        
        // Limit Q&A pairs
        if (isset($content['qa_pairs']) && count($content['qa_pairs']) > 20) {
            $content['qa_pairs'] = array_slice($content['qa_pairs'], 0, 20);
        }
        
        // Limit topics
        if (isset($content['topics']) && count($content['topics']) > 50) {
            $content['topics'] = array_slice($content['topics'], 0, 50);
        }
        
        $capsule->set_content('raw_content', $content['raw_content']);
        $capsule->set_content('qa_pairs', $content['qa_pairs']);
        $capsule->set_content('topics', $content['topics']);
    }
    
    /**
     * Compress capsule content
     */
    private function compress_capsule_content($capsule) {
        // This would implement compression logic
        // For now, just mark as compressed
        $capsule->set_metadata('compression', $this->settings['compression']);
    }
    
    /**
     * Find posts for topic
     */
    private function find_topic_posts($topic, $options) {
        // Search in post content and metadata
        $posts = get_posts(array(
            'post_type' => array('post', 'page'),
            'post_status' => 'publish',
            'numberposts' => $options['limit'] * 2, // Get more to filter
            's' => $topic,
            'meta_query' => array(
                array(
                    'key' => '_aiox_processed_data',
                    'compare' => 'EXISTS'
                )
            )
        ));
        
        $relevant_posts = array();
        
        foreach ($posts as $post) {
            $relevance = $this->calculate_topic_relevance($post, $topic);
            if ($relevance >= $options['min_relevance']) {
                $relevant_posts[] = $post->ID;
            }
            
            if (count($relevant_posts) >= $options['limit']) {
                break;
            }
        }
        
        return $relevant_posts;
    }
    
    /**
     * Calculate topic relevance
     */
    private function calculate_topic_relevance($post, $topic) {
        $content = strtolower($post->post_title . ' ' . $post->post_content);
        $topic_lower = strtolower($topic);
        
        // Simple relevance calculation
        $title_matches = substr_count(strtolower($post->post_title), $topic_lower);
        $content_matches = substr_count($content, $topic_lower);
        
        $total_words = str_word_count($content);
        $relevance = ($title_matches * 3 + $content_matches) / max($total_words / 100, 1);
        
        return min($relevance, 1.0);
    }
    
    /**
     * Analyze common entities
     */
    private function analyze_common_entities($entities) {
        $entity_counts = array();
        
        foreach ($entities as $entity) {
            if (is_array($entity) && isset($entity['text'])) {
                $key = $entity['text'];
                if (!isset($entity_counts[$key])) {
                    $entity_counts[$key] = array(
                        'text' => $entity['text'],
                        'type' => $entity['type'] ?? 'unknown',
                        'count' => 0
                    );
                }
                $entity_counts[$key]['count']++;
            }
        }
        
        // Sort by count
        uasort($entity_counts, function($a, $b) {
            return $b['count'] - $a['count'];
        });
        
        return array_slice($entity_counts, 0, 20);
    }
    
    /**
     * Extract content themes
     */
    private function extract_content_themes($content) {
        // Simple theme extraction based on word frequency
        $words = str_word_count(strtolower($content), 1);
        $word_counts = array_count_values($words);
        
        // Filter out common words
        $stop_words = array('the', 'and', 'or', 'but', 'in', 'on', 'at', 'to', 'for', 'of', 'with', 'by', 'is', 'are', 'was', 'were', 'be', 'been', 'have', 'has', 'had', 'do', 'does', 'did', 'will', 'would', 'could', 'should', 'may', 'might', 'must', 'can', 'this', 'that', 'these', 'those', 'a', 'an');
        
        foreach ($stop_words as $stop_word) {
            unset($word_counts[$stop_word]);
        }
        
        // Filter by minimum length and frequency
        $themes = array();
        foreach ($word_counts as $word => $count) {
            if (strlen($word) > 3 && $count > 2) {
                $themes[$word] = $count;
            }
        }
        
        arsort($themes);
        return array_slice($themes, 0, 10, true);
    }
    
    /**
     * Generate AI collection summary
     */
    private function generate_ai_collection_summary($content) {
        // Truncate content for AI processing
        $truncated_content = substr($content, 0, 5000);
        
        return "Collection summary: " . substr($truncated_content, 0, 500) . "...";
    }
    
    /**
     * Analyze content relationships
     */
    private function analyze_content_relationships($post_ids) {
        $relationships = array();
        
        foreach ($post_ids as $post_id) {
            $post = get_post($post_id);
            if ($post) {
                $categories = wp_get_post_categories($post_id, array('fields' => 'names'));
                $tags = wp_get_post_tags($post_id, array('fields' => 'names'));
                
                $relationships[$post_id] = array(
                    'title' => $post->post_title,
                    'categories' => $categories,
                    'tags' => $tags,
                    'date' => $post->post_date
                );
            }
        }
        
        return $relationships;
    }
    
    /**
     * Add collection relationships
     */
    private function add_collection_relationships($capsule, $post_ids) {
        $relationships = $this->analyze_content_relationships($post_ids);
        
        $existing_content = $capsule->get_content();
        $existing_content['relationships'] = $relationships;
        $capsule->set_content('relationships', $relationships);
    }
    
    /**
     * Generate collection insights
     */
    private function generate_collection_insights($capsule, $post_ids) {
        $insights = array(
            'total_posts' => count($post_ids),
            'date_range' => $this->get_date_range($post_ids),
            'category_distribution' => $this->get_category_distribution($post_ids),
            'tag_distribution' => $this->get_tag_distribution($post_ids),
            'author_distribution' => $this->get_author_distribution($post_ids),
            'content_stats' => $this->get_content_stats($post_ids)
        );
        
        $existing_content = $capsule->get_content();
        $existing_content['insights'] = $insights;
        $capsule->set_content('insights', $insights);
    }
    
    /**
     * Get date range for posts
     */
    private function get_date_range($post_ids) {
        $dates = array();
        
        foreach ($post_ids as $post_id) {
            $post = get_post($post_id);
            if ($post) {
                $dates[] = $post->post_date;
            }
        }
        
        if (empty($dates)) {
            return null;
        }
        
        sort($dates);
        
        return array(
            'earliest' => $dates[0],
            'latest' => end($dates),
            'span_days' => (strtotime(end($dates)) - strtotime($dates[0])) / (60 * 60 * 24)
        );
    }
    
    /**
     * Get category distribution
     */
    private function get_category_distribution($post_ids) {
        $categories = array();
        
        foreach ($post_ids as $post_id) {
            $post_categories = wp_get_post_categories($post_id, array('fields' => 'names'));
            foreach ($post_categories as $category) {
                $categories[$category] = ($categories[$category] ?? 0) + 1;
            }
        }
        
        arsort($categories);
        return $categories;
    }
    
    /**
     * Get tag distribution
     */
    private function get_tag_distribution($post_ids) {
        $tags = array();
        
        foreach ($post_ids as $post_id) {
            $post_tags = wp_get_post_tags($post_id, array('fields' => 'names'));
            foreach ($post_tags as $tag) {
                $tags[$tag] = ($tags[$tag] ?? 0) + 1;
            }
        }
        
        arsort($tags);
        return array_slice($tags, 0, 20, true);
    }
    
    /**
     * Get author distribution
     */
    private function get_author_distribution($post_ids) {
        $authors = array();
        
        foreach ($post_ids as $post_id) {
            $post = get_post($post_id);
            if ($post) {
                $author_name = get_the_author_meta('display_name', $post->post_author);
                $authors[$author_name] = ($authors[$author_name] ?? 0) + 1;
            }
        }
        
        arsort($authors);
        return $authors;
    }
    
    /**
     * Get content statistics
     */
    private function get_content_stats($post_ids) {
        $total_words = 0;
        $total_chars = 0;
        $post_types = array();
        
        foreach ($post_ids as $post_id) {
            $post = get_post($post_id);
            if ($post) {
                $content = strip_tags($post->post_content);
                $total_words += str_word_count($content);
                $total_chars += strlen($content);
                
                $post_types[$post->post_type] = ($post_types[$post->post_type] ?? 0) + 1;
            }
        }
        
        return array(
            'total_words' => $total_words,
            'total_characters' => $total_chars,
            'average_words_per_post' => count($post_ids) > 0 ? round($total_words / count($post_ids)) : 0,
            'post_type_distribution' => $post_types
        );
    }
    
    /**
     * Enhance topic capsule
     */
    private function enhance_topic_capsule($capsule, $topic) {
        // Add topic-specific metadata
        $capsule->set_metadata('primary_topic', $topic);
        
        // Generate topic summary
        $topic_summary = "This capsule focuses on the topic '{$topic}' and contains curated content that provides comprehensive coverage of this subject.";
        $capsule->set_content('topic_summary', $topic_summary);
    }
    
    /**
     * Add author metadata
     */
    private function add_author_metadata($capsule, $author) {
        $author_data = array(
            'id' => $author->ID,
            'display_name' => $author->display_name,
            'user_login' => $author->user_login,
            'user_email' => $author->user_email,
            'user_url' => $author->user_url,
            'description' => $author->description,
            'registered' => $author->user_registered,
            'avatar_url' => get_avatar_url($author->ID),
            'posts_count' => count_user_posts($author->ID),
            'social_profiles' => array(
                'twitter' => get_user_meta($author->ID, 'twitter', true),
                'facebook' => get_user_meta($author->ID, 'facebook', true),
                'linkedin' => get_user_meta($author->ID, 'linkedin', true),
                'website' => $author->user_url
            )
        );
        
        $existing_metadata = $capsule->get_content('metadata');
        $existing_metadata['author_profile'] = $author_data;
        $capsule->set_content('metadata', $existing_metadata);
    }
}
?>
