<?php
/**
 * AIOX Capsule Manager Class
 * 
 * Manages capsule operations, creation, and lifecycle management.
 */

if (!defined('ABSPATH')) {
    exit;
}

require_once AIOX_PLUGIN_DIR . 'core/class-aiox-capsule.php';

class AIOX_Capsule_Manager {
    
    /**
     * Database table name
     */
    private $table_name;
    
    /**
     * Constructor
     */
    public function __construct() {
        global $wpdb;
        
        // Check if WordPress database is available
        if ($wpdb) {
            $this->table_name = $wpdb->prefix . 'aiox_capsules';
            // Ensure database table exists
            $this->create_database_table();
        } else {
            $this->table_name = 'aiox_capsules'; // Fallback table name
        }
    }
    
    /**
     * Create database table for capsules
     */
    private function create_database_table() {
        global $wpdb;
        
        // Check if WordPress database is available
        if (!$wpdb) {
            return;
        }
        
        $charset_collate = $wpdb->get_charset_collate();
        
        $sql = "CREATE TABLE IF NOT EXISTS {$this->table_name} (
            id varchar(100) NOT NULL,
            title varchar(255) NOT NULL,
            description text,
            type varchar(50) DEFAULT 'post',
            status varchar(20) DEFAULT 'draft',
            metadata longtext,
            content longtext,
            file_path varchar(500),
            file_size bigint(20) DEFAULT 0,
            content_hash varchar(64),
            created_at datetime DEFAULT CURRENT_TIMESTAMP,
            updated_at datetime DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
            author_id bigint(20) DEFAULT 0,
            PRIMARY KEY (id),
            KEY type (type),
            KEY status (status),
            KEY author_id (author_id),
            KEY created_at (created_at)
        ) $charset_collate;";
        
        // Only try to create table if WordPress upgrade functions are available
        if (file_exists(ABSPATH . 'wp-admin/includes/upgrade.php')) {
            require_once(ABSPATH . 'wp-admin/includes/upgrade.php');
            if (function_exists('dbDelta')) {
                dbDelta($sql);
            } else {
                // Fallback: try direct SQL execution
                $wpdb->query($sql);
            }
        } else {
            // Fallback: try direct SQL execution
            if ($wpdb && method_exists($wpdb, 'query')) {
                $wpdb->query($sql);
            }
        }
    }
    
    /**
     * Create new capsule
     */
    public function create_capsule($type = 'post', $options = array()) {
        // Handle both old and new parameter formats
        if (is_string($options)) {
            // Old format: create_capsule($type, $title, $description)
            $title = $options;
            $description = func_num_args() > 2 ? func_get_arg(2) : '';
            $options = array(
                'title' => $title,
                'description' => $description
            );
        }
        
        $capsule = new AIOX_Capsule();
        
        // Set basic properties
        $capsule->set_type($type);
        
        // Set metadata from options
        $title = isset($options['title']) ? $options['title'] : 'New Capsule';
        $description = isset($options['description']) ? $options['description'] : '';
        
        $capsule->set_metadata('title', $title)
                ->set_metadata('description', $description)
                ->set_metadata('type', $type)
                ->set_metadata('created_at', current_time('mysql'))
                ->set_metadata('updated_at', current_time('mysql'));
        
        // Handle specific capsule types
        switch ($type) {
            case 'post':
                if (isset($options['post_id'])) {
                    $post = get_post($options['post_id']);
                    if ($post) {
                        $capsule->set_metadata('source_posts', array($options['post_id']));
                        $capsule->set_content('raw_content', $post->post_content);
                        
                        // Get AIOX processed data if available
                        $processed_data = get_post_meta($options['post_id'], '_aiox_processed_data', true);
                        if ($processed_data) {
                            $capsule->set_content('processed_content', $processed_data);
                        }
                    }
                }
                break;
                
            case 'collection':
                if (isset($options['post_ids']) && is_array($options['post_ids'])) {
                    $capsule->set_metadata('source_posts', $options['post_ids']);
                    $content = array();
                    foreach ($options['post_ids'] as $post_id) {
                        $post = get_post($post_id);
                        if ($post) {
                            $content[] = array(
                                'id' => $post_id,
                                'title' => $post->post_title,
                                'content' => $post->post_content
                            );
                        }
                    }
                    $capsule->set_content('collection_content', $content);
                }
                break;
                
            case 'author':
                if (isset($options['author_id'])) {
                    $author = get_userdata($options['author_id']);
                    if ($author) {
                        $capsule->set_metadata('author_id', $options['author_id']);
                        $capsule->set_content('author_data', array(
                            'id' => $author->ID,
                            'name' => $author->display_name,
                            'email' => $author->user_email,
                            'bio' => get_user_meta($author->ID, 'description', true)
                        ));
                    }
                }
                break;
                
            case 'topic':
                if (isset($options['keywords'])) {
                    $capsule->set_metadata('keywords', $options['keywords']);
                    $capsule->set_content('topic_data', array(
                        'keywords' => $options['keywords'],
                        'related_posts' => array() // Could be populated later
                    ));
                }
                break;
        }
        
        // Save the capsule to database after setting metadata
        try {
            $result = $capsule->save();
            
            if (!$result) {
                AIOX_Logger::error("Failed to save capsule during creation", array('type' => $type, 'options' => $options));
                throw new Exception('Failed to save capsule');
            }
            
            AIOX_Logger::log("Capsule created successfully: {$capsule->get_id()}", AIOX_Logger::INFO);
            
            return $capsule;
            
        } catch (Exception $e) {
            AIOX_Logger::error("Exception during capsule creation", array(
                'type' => $type, 
                'options' => $options, 
                'error' => $e->getMessage()
            ));
            throw new Exception('Failed to create capsule: ' . $e->getMessage());
        }
    }
    
    /**
     * Load existing capsule
     */
    public function load_capsule($id) {
        try {
            return new AIOX_Capsule($id);
        } catch (Exception $e) {
            AIOX_Logger::error("Failed to load capsule: {$id}", array('error' => $e->getMessage()));
            return null;
        }
    }
    
    /**
     * Get all capsules
     */
    public function get_all_capsules($args = array()) {
        global $wpdb;
        
        $defaults = array(
            'type' => '',
            'status' => '',
            'limit' => 50,
            'offset' => 0,
            'orderby' => 'updated_at',
            'order' => 'DESC'
        );
        
        $args = wp_parse_args($args, $defaults);
        
        // Check if table exists
        $table_exists = $wpdb->get_var("SHOW TABLES LIKE '{$this->table_name}'") === $this->table_name;
        if (!$table_exists) {
            // If no database table, try to load from files
            return $this->get_capsules_from_files($args);
        }
        
        $where_conditions = array();
        $where_values = array();
        
        if (!empty($args['type'])) {
            $where_conditions[] = "type = %s";
            $where_values[] = $args['type'];
        }
        
        if (!empty($args['status'])) {
            $where_conditions[] = "status = %s";
            $where_values[] = $args['status'];
        }
        
        $where_clause = '';
        if (!empty($where_conditions)) {
            $where_clause = 'WHERE ' . implode(' AND ', $where_conditions);
        }
        
        $order_clause = sprintf(
            'ORDER BY %s %s',
            sanitize_sql_orderby($args['orderby']),
            $args['order'] === 'ASC' ? 'ASC' : 'DESC'
        );
        
        $limit_clause = $wpdb->prepare('LIMIT %d OFFSET %d', $args['limit'], $args['offset']);
        
        $sql = "SELECT * FROM {$this->table_name} {$where_clause} {$order_clause} {$limit_clause}";
        
        if (!empty($where_values)) {
            $sql = $wpdb->prepare($sql, $where_values);
        }
        
        try {
            $results = $wpdb->get_results($sql, ARRAY_A);
            
            $capsules = array();
            foreach ($results as $result) {
                try {
                    // Use capsule_id column from the existing table structure
                    $capsule_id = isset($result['capsule_id']) ? $result['capsule_id'] : $result['id'];
                    $capsules[] = new AIOX_Capsule($capsule_id);
                } catch (Exception $e) {
                    $capsule_id = isset($result['capsule_id']) ? $result['capsule_id'] : $result['id'];
                    AIOX_Logger::error("Failed to load capsule in list: {$capsule_id}", array('error' => $e->getMessage()));
                }
            }
            
            return $capsules;
            
        } catch (Exception $e) {
            AIOX_Logger::error("Database query failed, falling back to file system", array('error' => $e->getMessage()));
            return $this->get_capsules_from_files($args);
        }
    }
    
    /**
     * Get capsules from file system as fallback
     */
    private function get_capsules_from_files($args = array()) {
        $capsules = array();
        
        // Get upload directory
        $upload_dir = function_exists('wp_upload_dir') ? wp_upload_dir() : array('basedir' => AIOX_PLUGIN_DIR . 'wp-content/uploads');
        $capsules_dir = $upload_dir['basedir'] . '/aiox-capsules';
        
        if (!is_dir($capsules_dir)) {
            return $capsules;
        }
        
        $files = glob($capsules_dir . '/*.json');
        
        foreach ($files as $file) {
            $filename = basename($file, '.json');
            try {
                $capsules[] = new AIOX_Capsule($filename);
            } catch (Exception $e) {
                AIOX_Logger::error("Failed to load capsule from file: {$filename}", array('error' => $e->getMessage()));
            }
        }
        
        // Apply basic filtering
        if (!empty($args['type'])) {
            $capsules = array_filter($capsules, function($capsule) use ($args) {
                return $capsule->get_type() === $args['type'];
            });
        }
        
        if (!empty($args['status'])) {
            $capsules = array_filter($capsules, function($capsule) use ($args) {
                return $capsule->get_status() === $args['status'];
            });
        }
        
        // Apply limit
        if ($args['limit'] > 0) {
            $capsules = array_slice($capsules, $args['offset'], $args['limit']);
        }
        
        return $capsules;
    }
    
    /**
     * Get capsules count
     */
    public function get_capsules_count($args = array()) {
        global $wpdb;
        
        $where_conditions = array();
        $where_values = array();
        
        if (!empty($args['type'])) {
            $where_conditions[] = "type = %s";
            $where_values[] = $args['type'];
        }
        
        if (!empty($args['status'])) {
            $where_conditions[] = "status = %s";
            $where_values[] = $args['status'];
        }
        
        $where_clause = '';
        if (!empty($where_conditions)) {
            $where_clause = 'WHERE ' . implode(' AND ', $where_conditions);
        }
        
        $sql = "SELECT COUNT(*) FROM {$this->table_name} {$where_clause}";
        
        if (!empty($where_values)) {
            $sql = $wpdb->prepare($sql, $where_values);
        }
        
        return (int) $wpdb->get_var($sql);
    }
    
    /**
     * Get capsule by post ID
     */
    public function get_capsule_by_post_id($post_id) {
        global $wpdb;
        
        if (!$wpdb) {
            return null;
        }
        
        // Check if table exists
        $table_exists = $wpdb->get_var("SHOW TABLES LIKE '{$this->table_name}'") === $this->table_name;
        if (!$table_exists) {
            // Fallback to file system search
            return $this->get_capsule_by_post_id_from_files($post_id);
        }
        
        // Get all capsules and check their metadata properly
        $sql = "SELECT id, metadata FROM {$this->table_name} ORDER BY created_at DESC";
        $results = $wpdb->get_results($sql, ARRAY_A);
        
        foreach ($results as $result) {
            try {
                $metadata = json_decode($result['metadata'], true);
                
                if (json_last_error() === JSON_ERROR_NONE && is_array($metadata)) {
                    // Check for direct post_id match
                    if (isset($metadata['post_id']) && intval($metadata['post_id']) === intval($post_id)) {
                        return $this->load_capsule($result['id']);
                    }
                    
                    // Check for source_posts array match
                    if (isset($metadata['source_posts']) && is_array($metadata['source_posts'])) {
                        foreach ($metadata['source_posts'] as $source_post_id) {
                            if (intval($source_post_id) === intval($post_id)) {
                                return $this->load_capsule($result['id']);
                            }
                        }
                    }
                }
            } catch (Exception $e) {
                // Skip this capsule if metadata parsing fails
                continue;
            }
        }
        
        // Fallback to file system search if database search fails
        return $this->get_capsule_by_post_id_from_files($post_id);
    }
    
    /**
     * Fallback method to search for capsules by post ID in file system
     */
    private function get_capsule_by_post_id_from_files($post_id) {
        // Get upload directory
        $upload_dir = function_exists('wp_upload_dir') ? wp_upload_dir() : array('basedir' => AIOX_PLUGIN_DIR . 'wp-content/uploads');
        $capsules_dir = $upload_dir['basedir'] . '/aiox-capsules';
        
        if (!is_dir($capsules_dir)) {
            return null;
        }
        
        $files = glob($capsules_dir . '/*.json');
        
        foreach ($files as $file) {
            try {
                $content = file_get_contents($file);
                $data = json_decode($content, true);
                
                if (json_last_error() === JSON_ERROR_NONE && is_array($data)) {
                    $metadata = isset($data['metadata']) ? $data['metadata'] : array();
                    
                    // Check for direct post_id match
                    if (isset($metadata['post_id']) && intval($metadata['post_id']) === intval($post_id)) {
                        $filename = basename($file, '.json');
                        return $this->load_capsule($filename);
                    }
                    
                    // Check for source_posts array match
                    if (isset($metadata['source_posts']) && is_array($metadata['source_posts'])) {
                        foreach ($metadata['source_posts'] as $source_post_id) {
                            if (intval($source_post_id) === intval($post_id)) {
                                $filename = basename($file, '.json');
                                return $this->load_capsule($filename);
                            }
                        }
                    }
                }
            } catch (Exception $e) {
                // Skip this file if parsing fails
                continue;
            }
        }
        
        return null;
    }
    
    /**
     * Create capsule from post
     */
    public function create_capsule_from_post($post_id, $title = '') {
        $post = get_post($post_id);
        if (!$post) {
            throw new Exception("Post not found: {$post_id}");
        }
        
        // Check if a capsule already exists for this post
        $existing_capsule = $this->get_capsule_by_post_id($post_id);
        if ($existing_capsule) {
            AIOX_Logger::log("Capsule already exists for post {$post_id}: {$existing_capsule->get_id()}", AIOX_Logger::INFO);
            return $existing_capsule;
        }
        
        $capsule_title = !empty($title) ? $title : ($post->post_title ?: 'AI Indexing Information') . ' - Capsule';
        
        $capsule = $this->create_capsule('post', $capsule_title, 'Auto-generated capsule from post');
        
        // Add source post
        $capsule->add_source_post($post_id);
        
        // Set basic content
        $capsule->set_content('raw_content', $post->post_content);
        
        // Get existing AIOX processed data if available
        $processed_data = get_post_meta($post_id, '_aiox_processed_data', true);
        if ($processed_data) {
            $capsule->set_content('processed_content', $processed_data);
            
            if (isset($processed_data['summary'])) {
                $capsule->set_content('summary', $processed_data['summary']);
            }
            
            if (isset($processed_data['topics'])) {
                $capsule->set_content('topics', $processed_data['topics']);
            }
            
            if (isset($processed_data['qa_pairs'])) {
                $capsule->set_content('qa_pairs', $processed_data['qa_pairs']);
            }
            
            if (isset($processed_data['entities'])) {
                $capsule->set_content('entities', $processed_data['entities']);
            }
        }
        
        // Add post metadata
        $capsule->set_content('metadata', array(
            'post_id' => $post_id,
            'post_title' => $post->post_title,
            'post_type' => $post->post_type,
            'post_status' => $post->post_status,
            'post_date' => $post->post_date,
            'post_author' => $post->post_author,
            'post_excerpt' => $post->post_excerpt,
            'post_url' => get_permalink($post_id),
            'post_categories' => wp_get_post_categories($post_id, array('fields' => 'names')),
            'post_tags' => wp_get_post_tags($post_id, array('fields' => 'names'))
        ));
        
        // Add WordPress categories and tags to capsule
        $categories = wp_get_post_categories($post_id, array('fields' => 'names'));
        foreach ($categories as $category) {
            $capsule->add_category($category);
        }
        
        $tags = wp_get_post_tags($post_id, array('fields' => 'names'));
        foreach ($tags as $tag) {
            $capsule->add_tag($tag);
        }
        
        $capsule->set_status('ready');
        $capsule->save();
        
        AIOX_Logger::log("Created capsule from post {$post_id}: {$capsule->get_id()}", AIOX_Logger::INFO);
        
        return $capsule;
    }
    
    /**
     * Create collection capsule from multiple posts
     */
    public function create_collection_capsule($post_ids, $title, $description = '') {
        $capsule = $this->create_capsule('collection', $title, $description);
        
        $all_content = array();
        $all_qa_pairs = array();
        $all_topics = array();
        $all_entities = array();
        $collection_metadata = array();
        
        foreach ($post_ids as $post_id) {
            $post = get_post($post_id);
            if (!$post) {
                continue;
            }
            
            $capsule->add_source_post($post_id);
            
            // Collect content
            $all_content[] = array(
                'post_id' => $post_id,
                'title' => $post->post_title,
                'content' => $post->post_content,
                'excerpt' => $post->post_excerpt,
                'url' => get_permalink($post_id)
            );
            
            // Collect processed data
            $processed_data = get_post_meta($post_id, '_aiox_processed_data', true);
            if ($processed_data) {
                if (isset($processed_data['qa_pairs'])) {
                    $all_qa_pairs = array_merge($all_qa_pairs, $processed_data['qa_pairs']);
                }
                
                if (isset($processed_data['topics'])) {
                    $all_topics = array_merge($all_topics, $processed_data['topics']);
                }
                
                if (isset($processed_data['entities'])) {
                    $all_entities = array_merge($all_entities, $processed_data['entities']);
                }
            }
            
            // Collect metadata
            $collection_metadata[] = array(
                'post_id' => $post_id,
                'title' => $post->post_title,
                'type' => $post->post_type,
                'date' => $post->post_date,
                'author' => get_the_author_meta('display_name', $post->post_author),
                'categories' => wp_get_post_categories($post_id, array('fields' => 'names')),
                'tags' => wp_get_post_tags($post_id, array('fields' => 'names'))
            );
        }
        
        // Set collection content
        $capsule->set_content('raw_content', $all_content);
        $capsule->set_content('qa_pairs', $all_qa_pairs);
        $capsule->set_content('topics', array_unique($all_topics));
        $capsule->set_content('entities', array_unique($all_entities, SORT_REGULAR));
        $capsule->set_content('metadata', $collection_metadata);
        
        // Generate collection summary
        $summary = $this->generate_collection_summary($all_content);
        $capsule->set_content('summary', $summary);
        
        $capsule->set_status('ready');
        $capsule->save();
        
        AIOX_Logger::log("Created collection capsule with " . count($post_ids) . " posts: {$capsule->get_id()}", AIOX_Logger::INFO);
        
        return $capsule;
    }
    
    /**
     * Create topic-based capsule
     */
    public function create_topic_capsule($topic, $limit = 10) {
        // Find posts related to the topic
        $posts = get_posts(array(
            'post_type' => array('post', 'page'),
            'post_status' => 'publish',
            'numberposts' => $limit,
            's' => $topic,
            'meta_query' => array(
                array(
                    'key' => '_aiox_processed_data',
                    'compare' => 'EXISTS'
                )
            )
        ));
        
        if (empty($posts)) {
            throw new Exception("No posts found for topic: {$topic}");
        }
        
        $post_ids = wp_list_pluck($posts, 'ID');
        
        $title = "Topic Capsule: {$topic}";
        $description = "Auto-generated capsule for topic '{$topic}' containing " . count($posts) . " related posts.";
        
        $capsule = $this->create_collection_capsule($post_ids, $title, $description);
        $capsule->set_type('topic');
        $capsule->add_tag($topic);
        $capsule->save();
        
        return $capsule;
    }
    
    /**
     * Generate collection summary
     */
    private function generate_collection_summary($content_array) {
        $titles = array();
        $total_words = 0;
        
        foreach ($content_array as $content) {
            $titles[] = $content['title'];
            $total_words += str_word_count(strip_tags($content['content']));
        }
        
        $summary = sprintf(
            "This collection contains %d pieces of content: %s. Total word count: approximately %d words.",
            count($content_array),
            implode(', ', array_slice($titles, 0, 5)) . (count($titles) > 5 ? '...' : ''),
            $total_words
        );
        
        return $summary;
    }
    
    /**
     * Delete capsule
     */
    public function delete_capsule($id) {
        try {
            AIOX_Logger::log("Attempting to delete capsule: {$id}", AIOX_Logger::INFO);
            
            $capsule = $this->load_capsule($id);
            if (!$capsule) {
                AIOX_Logger::error("Capsule not found for deletion: {$id}");
                return false;
            }
            
            $result = $capsule->delete();
            
            if ($result) {
                AIOX_Logger::log("Capsule deleted successfully: {$id}", AIOX_Logger::INFO);
            } else {
                AIOX_Logger::error("Failed to delete capsule: {$id}");
            }
            
            return $result;
            
        } catch (Exception $e) {
            AIOX_Logger::error("Exception during capsule deletion: {$id}", array('error' => $e->getMessage()));
            return false;
        }
    }
    
    /**
     * Bulk delete capsules
     */
    public function bulk_delete_capsules($ids) {
        $deleted = 0;
        foreach ($ids as $id) {
            if ($this->delete_capsule($id)) {
                $deleted++;
            }
        }
        return $deleted;
    }
    
    /**
     * Export capsule
     */
    public function export_capsule($id) {
        $capsule = $this->load_capsule($id);
        if ($capsule) {
            return $capsule->export();
        }
        return null;
    }
    
    /**
     * Import capsule
     */
    public function import_capsule($json_data) {
        $capsule = new AIOX_Capsule();
        return $capsule->import($json_data);
    }
    
    /**
     * Get capsule statistics
     */
    public function get_statistics() {
        global $wpdb;
        
        $stats = array();
        
        // Total capsules
        $stats['total'] = $wpdb->get_var("SELECT COUNT(*) FROM {$this->table_name}");
        
        // By type (with error handling for missing columns)
        $stats['by_type'] = array();
        try {
            $type_results = $wpdb->get_results("SELECT type, COUNT(*) as count FROM {$this->table_name} GROUP BY type", ARRAY_A);
            if ($type_results && !$wpdb->last_error) {
                foreach ($type_results as $result) {
                    $stats['by_type'][$result['type']] = $result['count'];
                }
            }
        } catch (Exception $e) {
            // Column might not exist in older table structure
            $stats['by_type'] = array();
        }
        
        // By status (with error handling for missing columns)
        $stats['by_status'] = array();
        try {
            $status_results = $wpdb->get_results("SELECT status, COUNT(*) as count FROM {$this->table_name} GROUP BY status", ARRAY_A);
            if ($status_results && !$wpdb->last_error) {
                foreach ($status_results as $result) {
                    $stats['by_status'][$result['status']] = $result['count'];
                }
            }
        } catch (Exception $e) {
            // Column might not exist in older table structure
            $stats['by_status'] = array();
        }
        
        // Recent activity
        $stats['recent'] = $wpdb->get_var(
            "SELECT COUNT(*) FROM {$this->table_name} WHERE updated_at >= DATE_SUB(NOW(), INTERVAL 7 DAY)"
        );
        
        return $stats;
    }
    
    /**
     * Search capsules
     */
    public function search_capsules($query, $args = array()) {
        global $wpdb;
        
        $defaults = array(
            'type' => '',
            'status' => '',
            'limit' => 20,
            'offset' => 0
        );
        
        $args = wp_parse_args($args, $defaults);
        
        $where_conditions = array();
        $where_values = array();
        
        // Search in title and metadata
        $where_conditions[] = "(title LIKE %s OR metadata LIKE %s)";
        $search_term = '%' . $wpdb->esc_like($query) . '%';
        $where_values[] = $search_term;
        $where_values[] = $search_term;
        
        if (!empty($args['type'])) {
            $where_conditions[] = "type = %s";
            $where_values[] = $args['type'];
        }
        
        if (!empty($args['status'])) {
            $where_conditions[] = "status = %s";
            $where_values[] = $args['status'];
        }
        
        $where_clause = 'WHERE ' . implode(' AND ', $where_conditions);
        $limit_clause = $wpdb->prepare('LIMIT %d OFFSET %d', $args['limit'], $args['offset']);
        
        $sql = "SELECT * FROM {$this->table_name} {$where_clause} ORDER BY updated_at DESC {$limit_clause}";
        $sql = $wpdb->prepare($sql, $where_values);
        
        $results = $wpdb->get_results($sql, ARRAY_A);
        
        $capsules = array();
        foreach ($results as $result) {
            try {
                $capsules[] = new AIOX_Capsule($result['id']);
            } catch (Exception $e) {
                AIOX_Logger::error("Failed to load capsule in search: {$result['id']}", array('error' => $e->getMessage()));
            }
        }
        
        return $capsules;
    }
    
    /**
     * Clean up old capsules
     */
    public function cleanup_old_capsules($days = 30) {
        global $wpdb;
        
        $cutoff_date = date('Y-m-d H:i:s', strtotime("-{$days} days"));
        
        $old_capsules = $wpdb->get_col(
            $wpdb->prepare(
                "SELECT id FROM {$this->table_name} WHERE status = 'archived' AND updated_at < %s",
                $cutoff_date
            )
        );
        
        $deleted = 0;
        foreach ($old_capsules as $id) {
            if ($this->delete_capsule($id)) {
                $deleted++;
            }
        }
        
        AIOX_Logger::log("Cleaned up {$deleted} old capsules", AIOX_Logger::INFO);
        
        return $deleted;
    }
    
    /**
     * Regenerate capsule from post (replaces existing capsule)
     */
    public function regenerate_capsule_from_post($post_id, $title = '') {
        $post = get_post($post_id);
        if (!$post) {
            throw new Exception("Post not found: {$post_id}");
        }
        
        AIOX_Logger::log("Starting regeneration for post {$post_id}", AIOX_Logger::INFO);
        
        // Check if a capsule already exists for this post
        $existing_capsule = $this->get_capsule_by_post_id($post_id);
        if ($existing_capsule) {
            // Delete the existing capsule first
            $existing_id = $existing_capsule->get_id();
            AIOX_Logger::log("Regenerating capsule for post {$post_id}: found existing capsule {$existing_id}, deleting it", AIOX_Logger::INFO);
            
            // Delete from database and files
            $delete_result = $this->delete_capsule($existing_id);
            if ($delete_result) {
                AIOX_Logger::log("Successfully deleted existing capsule {$existing_id} for post {$post_id}", AIOX_Logger::INFO);
            } else {
                AIOX_Logger::error("Failed to delete existing capsule {$existing_id} for post {$post_id}");
            }
        } else {
            AIOX_Logger::log("No existing capsule found for post {$post_id}, creating new one", AIOX_Logger::INFO);
        }
        
        // Now create a new capsule (this will be a fresh creation)
        $capsule_title = !empty($title) ? $title : ($post->post_title ?: 'AI Indexing Information') . ' - Capsule';
        
        $capsule = $this->create_capsule('post', $capsule_title, 'Auto-generated capsule from post (regenerated)');
        
        // Add source post
        $capsule->add_source_post($post_id);
        
        // Set basic content
        $capsule->set_content('raw_content', $post->post_content);
        
        // Get existing AIOX processed data if available
        $processed_data = get_post_meta($post_id, '_aiox_processed_data', true);
        if ($processed_data) {
            $capsule->set_content('processed_content', $processed_data);
            
            if (isset($processed_data['summary'])) {
                $capsule->set_content('summary', $processed_data['summary']);
            }
            
            if (isset($processed_data['topics'])) {
                $capsule->set_content('topics', $processed_data['topics']);
            }
            
            if (isset($processed_data['qa_pairs'])) {
                $capsule->set_content('qa_pairs', $processed_data['qa_pairs']);
            }
            
            if (isset($processed_data['entities'])) {
                $capsule->set_content('entities', $processed_data['entities']);
            }
        }
        
        // Add post metadata
        $capsule->set_content('metadata', array(
            'post_id' => $post_id,
            'post_title' => $post->post_title,
            'post_type' => $post->post_type,
            'post_status' => $post->post_status,
            'post_date' => $post->post_date,
            'post_author' => $post->post_author,
            'post_excerpt' => $post->post_excerpt,
            'post_url' => get_permalink($post_id),
            'post_categories' => wp_get_post_categories($post_id, array('fields' => 'names')),
            'post_tags' => wp_get_post_tags($post_id, array('fields' => 'names')),
            'regenerated_at' => current_time('mysql')
        ));
        
        // Add WordPress categories and tags to capsule
        $categories = wp_get_post_categories($post_id, array('fields' => 'names'));
        foreach ($categories as $category) {
            $capsule->add_category($category);
        }
        
        $tags = wp_get_post_tags($post_id, array('fields' => 'names'));
        foreach ($tags as $tag) {
            $capsule->add_tag($tag);
        }
        
        $capsule->set_status('ready');
        $capsule->save();
        
        AIOX_Logger::log("Regenerated capsule for post {$post_id}: new capsule {$capsule->get_id()}", AIOX_Logger::INFO);
        
        return $capsule;
    }
    
    /**
     * Regenerate all capsule files from database
     * Used after plugin reinstall when files were deleted but database preserved
     * 
     * @return array Statistics about regeneration
     */
    public function regenerate_all_capsule_files() {
        global $wpdb;
        
        $stats = array(
            'total' => 0,
            'success' => 0,
            'failed' => 0,
            'errors' => array()
        );
        
        // Ensure capsules directory exists
        $well_known_dir = ABSPATH . '.well-known/capsules/';
        if (!file_exists($well_known_dir)) {
            if (function_exists('wp_mkdir_p')) {
                wp_mkdir_p($well_known_dir);
            } else {
                mkdir($well_known_dir, 0755, true);
            }
        }
        
        // Get all capsules from database
        try {
            $results = $wpdb->get_results("SELECT * FROM {$this->table_name}", ARRAY_A);
            
            if (empty($results)) {
                AIOX_Logger::info('No capsules found in database to regenerate');
                return $stats;
            }
            
            $stats['total'] = count($results);
            
            foreach ($results as $result) {
                try {
                    $capsule_id = isset($result['capsule_id']) ? $result['capsule_id'] : $result['id'];
                    
                    // Load capsule from database
                    $capsule = new AIOX_Capsule($capsule_id);
                    
                    // Save will recreate the files
                    $capsule->save();
                    
                    $stats['success']++;
                    
                } catch (Exception $e) {
                    $stats['failed']++;
                    $stats['errors'][] = "Capsule {$capsule_id}: " . $e->getMessage();
                    AIOX_Logger::error("Failed to regenerate capsule file: {$capsule_id}", array('error' => $e->getMessage()));
                }
            }
            
            AIOX_Logger::info("Capsule files regeneration complete", $stats);
            
        } catch (Exception $e) {
            AIOX_Logger::error("Failed to query capsules for regeneration", array('error' => $e->getMessage()));
            $stats['errors'][] = "Database query failed: " . $e->getMessage();
        }
        
        return $stats;
    }
}
?>
