<?php
/**
 * AIOX Capsule Class
 * 
 * Handles individual content capsules - self-contained units of content
 * optimized for AI consumption and processing.
 */

if (!defined('ABSPATH')) {
    exit;
}

class AIOX_Capsule {
    
    /**
     * Capsule ID
     */
    private $id;
    
    /**
     * Database ID (auto-increment)
     */
    private $database_id;
    
    /**
     * Capsule metadata
     */
    private $metadata;
    
    /**
     * Capsule content
     */
    private $content;
    
    /**
     * Capsule type
     */
    private $type;
    
    /**
     * Capsule status
     */
    private $status;
    
    /**
     * Supported capsule types
     */
    const TYPES = array(
        'post' => 'Single Post Capsule',
        'collection' => 'Content Collection Capsule',
        'topic' => 'Topic-based Capsule',
        'author' => 'Author Profile Capsule',
        'category' => 'Category Capsule',
        'custom' => 'Custom Content Capsule'
    );
    
    /**
     * Capsule statuses
     */
    const STATUSES = array(
        'draft' => 'Draft',
        'processing' => 'Processing',
        'ready' => 'Ready',
        'published' => 'Published',
        'archived' => 'Archived',
        'error' => 'Error'
    );
    
    /**
     * Constructor
     */
    public function __construct($id = null) {
        if ($id) {
            $this->id = $id;
            $this->load_capsule($id);
        } else {
            $this->id = $this->generate_id();
            $this->init_empty_capsule();
        }
    }
    
    /**
     * Generate unique capsule ID
     */
    private function generate_id() {
        return 'capsule_' . uniqid() . '_' . time();
    }
    
    /**
     * Initialize empty capsule
     */
    private function init_empty_capsule() {
        $this->metadata = array(
            'id' => $this->id,
            'title' => '',
            'description' => '',
            'type' => 'post',
            'status' => 'draft',
            'created_at' => function_exists('current_time') ? function_exists('current_time') ? current_time('mysql') : date('Y-m-d H:i:s') : date('Y-m-d H:i:s'),
            'updated_at' => function_exists('current_time') ? function_exists('current_time') ? current_time('mysql') : date('Y-m-d H:i:s') : date('Y-m-d H:i:s'),
            'version' => '1.0.0',
            'author' => function_exists('get_current_user_id') ? get_current_user_id() : 0,
            'tags' => array(),
            'categories' => array(),
            'language' => function_exists('get_locale') ? get_locale() : 'en_US',
            'encoding' => 'UTF-8',
            'compression' => 'none',
            'ai_optimized' => true,
            'source_posts' => array(),
            'dependencies' => array(),
            'file_size' => 0,
            'content_hash' => '',
            'ai_processing' => array(
                'provider' => function_exists('get_option') ? get_option('aiox_ai_provider', 'basic') : 'basic',
                'processed_at' => null,
                'processing_time' => 0,
                'tokens_used' => 0,
                'cost' => 0
            )
        );
        
        $this->content = array(
            'raw_content' => '',
            'processed_content' => '',
            'summary' => '',
            'key_points' => array(),
            'topics' => array(),
            'entities' => array(),
            'qa_pairs' => array(),
            'metadata' => array(),
            'attachments' => array(),
            'references' => array(),
            'ai_insights' => array()
        );
        
        $this->type = 'post';
        $this->status = 'draft';
        
        // Save the empty capsule to database
        $this->save();
    }
    
    /**
     * Load existing capsule
     */
    private function load_capsule($id) {
        $capsule_data = $this->get_capsule_data($id);
        
        if ($capsule_data) {
            $this->metadata = is_array($capsule_data['metadata']) ? $capsule_data['metadata'] : array();
            $this->content = is_array($capsule_data['content']) ? $capsule_data['content'] : array();
            
            // Handle type and status from multiple sources
            $this->type = $capsule_data['type'] ?? $this->metadata['type'] ?? 'post';
            $this->status = $capsule_data['status'] ?? $this->metadata['status'] ?? 'draft';
            
            // If we loaded from database and have the actual capsule_id, use that for file operations
            if (isset($capsule_data['database_id']) && isset($capsule_data['capsule_id'])) {
                $this->database_id = $capsule_data['database_id'];
                $this->id = $capsule_data['capsule_id']; // Use the actual capsule ID for file operations
                
                // Also update metadata with database fields if they exist
                if (isset($capsule_data['title'])) $this->metadata['title'] = $capsule_data['title'];
                if (isset($capsule_data['description'])) $this->metadata['description'] = $capsule_data['description'];
                if (isset($capsule_data['created_at'])) $this->metadata['created_at'] = $capsule_data['created_at'];
                if (isset($capsule_data['updated_at'])) $this->metadata['updated_at'] = $capsule_data['updated_at'];
                if (isset($capsule_data['file_size'])) $this->metadata['file_size'] = $capsule_data['file_size'];
                if (isset($capsule_data['content_hash'])) $this->metadata['content_hash'] = $capsule_data['content_hash'];
            }
            
        } else {
            throw new Exception("Capsule not found: {$id}");
        }
    }
    
    /**
     * Get capsule data from storage
     */
    private function get_capsule_data($id) {
        // Try to load from database first
        global $wpdb;
        
        // Check if WordPress database is available
        if (!$wpdb) {
            return null;
        }
        
        $table_name = $wpdb->prefix . 'aiox_capsules';
        
        // Try to find by database ID first (for UI compatibility)
        $result = $wpdb->get_row(
            $wpdb->prepare("SELECT * FROM {$table_name} WHERE id = %s", $id),
            ARRAY_A
        );
        
        // If not found, try by capsule_id (for backward compatibility)
        if (!$result) {
            $result = $wpdb->get_row(
                $wpdb->prepare("SELECT * FROM {$table_name} WHERE capsule_id = %s", $id),
                ARRAY_A
            );
        }
        
        if ($result) {
            // Parse the capsule_data JSON from the existing table structure
            $capsule_data = !empty($result['capsule_data']) ? json_decode($result['capsule_data'], true) : array();
            
            // Handle ID field properly - use capsule_id if available, otherwise use id
            $actual_capsule_id = !empty($result['capsule_id']) ? $result['capsule_id'] : $result['id'];
            
            return array(
                'metadata' => json_decode($result['metadata'], true),
                'content' => json_decode($result['content'], true),
                'database_id' => $result['id'], // Store the database ID for delete operations
                'capsule_id' => $actual_capsule_id, // Store the actual capsule ID for file operations
                'title' => $result['title'],
                'description' => $result['description'],
                'type' => $result['type'],
                'status' => $result['status'],
                'created_at' => $result['created_at'],
                'updated_at' => $result['updated_at'],
                'file_size' => $result['file_size'],
                'content_hash' => $result['content_hash']
            );
        }
        
        // Fallback to file system
        $capsule_file = $this->get_capsule_file_path($id);
        if (file_exists($capsule_file)) {
            $data = json_decode(file_get_contents($capsule_file), true);
            return $data;
        }
        
        return null;
    }
    
    /**
     * Get capsule file path
     */
    private function get_capsule_file_path($id) {
        // Use .well-known directory as per AIOX specification
        $capsules_dir = ABSPATH . '.well-known/';
        
        if (!file_exists($capsules_dir)) {
            if (function_exists('wp_mkdir_p')) {
                wp_mkdir_p($capsules_dir);
            } else {
                mkdir($capsules_dir, 0755, true);
            }
        }
        
        return $capsules_dir . 'capsule_' . $id . '.json';
    }
    
    /**
     * Set capsule metadata
     */
    public function set_metadata($key, $value) {
        $this->metadata[$key] = $value;
        $this->metadata['updated_at'] = function_exists('current_time') ? function_exists('current_time') ? current_time('mysql') : date('Y-m-d H:i:s') : date('Y-m-d H:i:s');
        return $this;
    }
    
    /**
     * Get capsule metadata
     */
    public function get_metadata($key = null) {
        if ($key) {
            return isset($this->metadata[$key]) ? $this->metadata[$key] : null;
        }
        return $this->metadata;
    }
    
    /**
     * Set capsule content
     */
    public function set_content($key, $value) {
        $this->content[$key] = $value;
        $this->metadata['updated_at'] = function_exists('current_time') ? function_exists('current_time') ? current_time('mysql') : date('Y-m-d H:i:s') : date('Y-m-d H:i:s');
        return $this;
    }
    
    /**
     * Get capsule content
     */
    public function get_content($key = null) {
        // If content is empty or not properly loaded, try to load from file
        if (empty($this->content) || (empty($this->content['raw_content']) && empty($this->content['summary']))) {
            $this->load_content_from_file();
        }
        
        if ($key) {
            return isset($this->content[$key]) ? $this->content[$key] : null;
        }
        return $this->content;
    }
    
    /**
     * Load content from file system
     */
    private function load_content_from_file() {
        // Use .well-known directory as per AIOX specification
        $capsules_dir = ABSPATH . '.well-known/';
        
        // Try multiple possible file names with capsule_ prefix
        $possible_files = array(
            'capsule_' . $this->id . '.json',
            'capsule_' . $this->database_id . '.json'
        );
        
        // Also try to find any file that contains the same title or similar pattern
        if (is_dir($capsules_dir)) {
            $files = glob($capsules_dir . 'capsule_*.json');
            foreach ($files as $file) {
                $file_data = json_decode(file_get_contents($file), true);
                if ($file_data && isset($file_data['metadata']['title']) && 
                    $file_data['metadata']['title'] === $this->get_title()) {
                    // Found a file with matching title
                    if (isset($file_data['content']) && !empty($file_data['content']['summary'])) {
                        $this->content = $file_data['content'];
                        return;
                    }
                }
            }
        }
        
        // Try the standard file paths
        foreach ($possible_files as $filename) {
            $capsule_file = $capsules_dir . $filename;
            if (file_exists($capsule_file)) {
                $file_data = json_decode(file_get_contents($capsule_file), true);
                if ($file_data && isset($file_data['content'])) {
                    $this->content = $file_data['content'];
                    return;
                }
            }
        }
        
    }
    
    /**
     * Set capsule type
     */
    public function set_type($type) {
        if (array_key_exists($type, self::TYPES)) {
            $this->type = $type;
            $this->metadata['type'] = $type;
            $this->metadata['updated_at'] = function_exists('current_time') ? function_exists('current_time') ? current_time('mysql') : date('Y-m-d H:i:s') : date('Y-m-d H:i:s');
        }
        return $this;
    }
    

    
    /**
     * Set capsule status
     */
    public function set_status($status) {
        if (array_key_exists($status, self::STATUSES)) {
            $this->status = $status;
            $this->metadata['status'] = $status;
            $this->metadata['updated_at'] = function_exists('current_time') ? function_exists('current_time') ? current_time('mysql') : date('Y-m-d H:i:s') : date('Y-m-d H:i:s');
        }
    }
    
    /**
     * Set capsule title
     */
    public function set_title($title) {
        $this->metadata['title'] = sanitize_text_field($title);
        $this->metadata['updated_at'] = function_exists('current_time') ? current_time('mysql') : date('Y-m-d H:i:s');
    }
    
    /**
     * Set capsule description
     */
    public function set_description($description) {
        $this->metadata['description'] = sanitize_textarea_field($description);
        $this->metadata['updated_at'] = function_exists('current_time') ? current_time('mysql') : date('Y-m-d H:i:s');
    }
    

    

    
    /**
     * Add source post to capsule
     */
    public function add_source_post($post_id) {
        if (!in_array($post_id, $this->metadata['source_posts'])) {
            $this->metadata['source_posts'][] = $post_id;
            $this->metadata['updated_at'] = function_exists('current_time') ? function_exists('current_time') ? current_time('mysql') : date('Y-m-d H:i:s') : date('Y-m-d H:i:s');
        }
        return $this;
    }
    
    /**
     * Remove source post from capsule
     */
    public function remove_source_post($post_id) {
        $key = array_search($post_id, $this->metadata['source_posts']);
        if ($key !== false) {
            unset($this->metadata['source_posts'][$key]);
            $this->metadata['source_posts'] = array_values($this->metadata['source_posts']);
            $this->metadata['updated_at'] = function_exists('current_time') ? function_exists('current_time') ? current_time('mysql') : date('Y-m-d H:i:s') : date('Y-m-d H:i:s');
        }
        return $this;
    }
    
    /**
     * Add tag to capsule
     */
    public function add_tag($tag) {
        if (!in_array($tag, $this->metadata['tags'])) {
            $this->metadata['tags'][] = $tag;
            $this->metadata['updated_at'] = function_exists('current_time') ? function_exists('current_time') ? current_time('mysql') : date('Y-m-d H:i:s') : date('Y-m-d H:i:s');
        }
        return $this;
    }
    
    /**
     * Add category to capsule
     */
    public function add_category($category) {
        if (!in_array($category, $this->metadata['categories'])) {
            $this->metadata['categories'][] = $category;
            $this->metadata['updated_at'] = function_exists('current_time') ? function_exists('current_time') ? current_time('mysql') : date('Y-m-d H:i:s') : date('Y-m-d H:i:s');
        }
        return $this;
    }
    
    /**
     * Save capsule to storage
     */
    public function save() {
        // Update metadata
        $this->metadata['updated_at'] = function_exists('current_time') ? function_exists('current_time') ? current_time('mysql') : date('Y-m-d H:i:s') : date('Y-m-d H:i:s');
        $this->metadata['content_hash'] = $this->generate_content_hash();
        $this->metadata['file_size'] = $this->calculate_file_size();
        
        // Save to database
        $this->save_to_database();
        
        // Save to file system
        $this->save_to_file();
        
        // Publish to well-known directory
        $this->publish_to_well_known();
        
        AIOX_Logger::log("Capsule saved: {$this->id}", AIOX_Logger::INFO);
        
        return $this;
    }
    
    /**
     * Save capsule to database
     */
    private function save_to_database() {
        global $wpdb;
        
        // Check if WordPress database is available
        if (!$wpdb) {
            AIOX_Logger::error('Database not available for capsule save', array('capsule_id' => $this->id));
            return false;
        }
        
        $table_name = $wpdb->prefix . 'aiox_capsules';
        
        // Check if table exists
        $table_exists = $wpdb->get_var("SHOW TABLES LIKE '{$table_name}'") === $table_name;
        if (!$table_exists) {
            AIOX_Logger::error('Capsules table does not exist', array('table_name' => $table_name));
            // Try to create the table
            $this->create_capsules_table();
        }
        
        // Use the new table structure
        $data = array(
            'id' => $this->id,
            'title' => isset($this->metadata['title']) ? $this->metadata['title'] : 'Untitled Capsule',
            'description' => isset($this->metadata['description']) ? $this->metadata['description'] : '',
            'type' => $this->type,
            'status' => $this->status,
            'metadata' => json_encode($this->metadata),
            'content' => json_encode($this->content),
            'file_path' => $this->get_capsule_file_path($this->id),
            'file_size' => $this->calculate_file_size(),
            'content_hash' => $this->generate_content_hash(),
            'author_id' => isset($this->metadata['author']) ? intval($this->metadata['author']) : (function_exists('get_current_user_id') ? get_current_user_id() : 1),
            'created_at' => $this->metadata['created_at'],
            'updated_at' => $this->metadata['updated_at']
        );
        
        try {
            $existing = $wpdb->get_var(
                $wpdb->prepare("SELECT id FROM {$table_name} WHERE id = %s", $this->id)
            );
            
            if ($existing) {
                // Remove created_at from update data
                unset($data['created_at']);
                $result = $wpdb->update($table_name, $data, array('id' => $this->id));
                if ($result === false) {
                    AIOX_Logger::error("Failed to update capsule in database: {$this->id}", array('error' => $wpdb->last_error));
                    throw new Exception("Database update failed: " . $wpdb->last_error);
                }
            } else {
                $result = $wpdb->insert($table_name, $data);
                if ($result === false) {
                    AIOX_Logger::error("Failed to insert capsule into database: {$this->id}", array('error' => $wpdb->last_error));
                    throw new Exception("Database insert failed: " . $wpdb->last_error);
                }
            }
            
            AIOX_Logger::log("Capsule saved to database: {$this->id}", AIOX_Logger::INFO);
            return true;
            
        } catch (Exception $e) {
            AIOX_Logger::error("Database operation failed for capsule: {$this->id}", array('error' => $e->getMessage()));
            // Don't throw exception, just log and continue with file save
            return false;
        }
    }
    
    /**
     * Create capsules table if it doesn't exist
     */
    private function create_capsules_table() {
        global $wpdb;
        
        $table_name = $wpdb->prefix . 'aiox_capsules';
        $charset_collate = $wpdb->get_charset_collate();
        
        $sql = "CREATE TABLE {$table_name} (
            id varchar(255) NOT NULL,
            title text NOT NULL,
            description text,
            type varchar(50) NOT NULL,
            status varchar(20) NOT NULL DEFAULT 'draft',
            metadata longtext,
            content longtext,
            file_path varchar(500),
            file_size bigint DEFAULT 0,
            content_hash varchar(64),
            author_id bigint DEFAULT 0,
            created_at datetime NOT NULL,
            updated_at datetime NOT NULL,
            PRIMARY KEY (id),
            KEY type_status (type, status),
            KEY author_id (author_id),
            KEY created_at (created_at)
        ) {$charset_collate};";
        
        if (!function_exists('dbDelta')) {
            require_once(ABSPATH . 'wp-admin/includes/upgrade.php');
        }
        
        try {
            dbDelta($sql);
            AIOX_Logger::log("Capsules table created successfully", AIOX_Logger::INFO);
        } catch (Exception $e) {
            AIOX_Logger::error("Failed to create capsules table", array('error' => $e->getMessage()));
        }
    }
    
    /**
     * Save capsule to file
     */
    private function save_to_file() {
        $capsule_file = $this->get_capsule_file_path($this->id);
        
        $data = array(
            'metadata' => $this->metadata,
            'content' => $this->content
        );
        
        file_put_contents($capsule_file, json_encode($data, JSON_PRETTY_PRINT));
    }
    
    /**
     * Publish capsule to well-known directory
     */
    private function publish_to_well_known() {
        // Get the well-known capsules directory
        $well_known_dir = ABSPATH . '.well-known/capsules/';
        
        // Create directory if it doesn't exist
        if (!file_exists($well_known_dir)) {
            if (function_exists('wp_mkdir_p')) {
                wp_mkdir_p($well_known_dir);
            } else {
                mkdir($well_known_dir, 0755, true);
            }
        }
        
        // Create the public capsule file
        $public_file = $well_known_dir . $this->id . '.json';
        
        // Create public version of the data (you may want to filter sensitive information)
        $public_data = array(
            'metadata' => $this->metadata,
            'content' => $this->content
        );
        
        // Write the public file
        file_put_contents($public_file, json_encode($public_data, JSON_PRETTY_PRINT));
        
        AIOX_Logger::log("Capsule published to well-known: {$this->id}", AIOX_Logger::INFO);
    }
    
    /**
     * Generate content hash
     */
    private function generate_content_hash() {
        $content_string = json_encode($this->content);
        return hash('sha256', $content_string);
    }
    
    /**
     * Calculate file size
     */
    private function calculate_file_size() {
        $data = array(
            'metadata' => $this->metadata,
            'content' => $this->content
        );
        return strlen(json_encode($data));
    }
    
    /**
     * Delete capsule
     */
    public function delete() {
        global $wpdb;
        
        try {
            $database_deleted = false;
            $file_deleted = false;
            
            // Delete from database
            $table_name = $wpdb->prefix . 'aiox_capsules';
            
            // Check if table exists
            $table_exists = $wpdb->get_var("SHOW TABLES LIKE '{$table_name}'") === $table_name;
            if ($table_exists) {
                // Use database_id if available, otherwise use the main id
                $delete_id = $this->database_id ? $this->database_id : $this->id;
                
                // Try both id and capsule_id fields to ensure deletion
                $result1 = $wpdb->delete($table_name, array('id' => $delete_id), array('%s'));
                $result2 = $wpdb->delete($table_name, array('capsule_id' => $this->id), array('%s'));
                
                $total_deleted = ($result1 !== false ? $result1 : 0) + ($result2 !== false ? $result2 : 0);
                
                if ($result1 === false && $result2 === false) {
                    AIOX_Logger::error("Failed to delete capsule from database: {$this->id}", array('error' => $wpdb->last_error));
                } elseif ($total_deleted === 0) {
                    AIOX_Logger::log("No database record found for capsule: {$this->id}", AIOX_Logger::INFO);
                    $database_deleted = true; // This is OK - capsule might only exist as file
                } else {
                    AIOX_Logger::log("Capsule deleted from database: {$this->id}", AIOX_Logger::INFO);
                    $database_deleted = true;
                }
            } else {
                AIOX_Logger::log("Database table does not exist, skipping database delete for capsule: {$this->id}", AIOX_Logger::INFO);
                $database_deleted = true; // No table means no database record to delete
            }
            
            // Delete file - use the actual capsule ID for file operations
            $capsule_file = $this->get_capsule_file_path($this->id);
            
            if (file_exists($capsule_file)) {
                // Check file permissions before attempting deletion
                $file_perms = fileperms($capsule_file);
                $is_writable = is_writable($capsule_file);
                $is_writable_dir = is_writable(dirname($capsule_file));
                
                
                
                if (unlink($capsule_file)) {
                    AIOX_Logger::log("Capsule file deleted: {$capsule_file}", AIOX_Logger::INFO);
                    $file_deleted = true;
                    
                    // Verify file is actually gone
                    if (file_exists($capsule_file)) {
                        $file_deleted = false;
                    }
                } else {
                    $error = error_get_last();
                    AIOX_Logger::error("Failed to delete capsule file: {$capsule_file}", array('error' => $error));
                }
            } else {
                AIOX_Logger::log("Capsule file does not exist: {$capsule_file}", AIOX_Logger::INFO);
                $file_deleted = true; // No file means nothing to delete
            }
            
            // Return true only if BOTH database and file deletion succeeded (or didn't exist)
            $success = $database_deleted && $file_deleted;
            
            if ($success) {
                AIOX_Logger::log("Capsule deletion completed: {$this->id}", AIOX_Logger::INFO);
            } else {
                AIOX_Logger::error("Capsule deletion failed - Database: " . ($database_deleted ? 'OK' : 'FAILED') . ", File: " . ($file_deleted ? 'OK' : 'FAILED'));
            }
            
            return $success;
            
        } catch (Exception $e) {
            AIOX_Logger::error("Error during capsule deletion: {$this->id}", array('error' => $e->getMessage()));
            return false;
        }
    }
    
    /**
     * Export capsule as JSON
     */
    public function export() {
        return array(
            'id' => $this->id,
            'metadata' => $this->metadata,
            'content' => $this->content,
            'type' => $this->type,
            'status' => $this->status,
            'export_date' => date('Y-m-d H:i:s'),
            'version' => '2.0.0'
        );
    }
    
    /**
     * Import capsule from JSON
     */
    public function import($json_data) {
        $data = json_decode($json_data, true);
        
        if ($data && isset($data['metadata']) && isset($data['content'])) {
            $this->metadata = $data['metadata'];
            $this->content = $data['content'];
            $this->type = $this->metadata['type'];
            $this->status = $this->metadata['status'];
            
            return $this->save();
        }
        
        throw new Exception('Invalid capsule data format');
    }
    

    
    /**
     * Validate capsule data
     */
    public function validate() {
        $errors = array();
        
        // Check required fields
        if (empty($this->metadata['title'])) {
            $errors[] = 'Title is required';
        }
        
        if (empty($this->type) || !array_key_exists($this->type, self::TYPES)) {
            $errors[] = 'Valid type is required';
        }
        
        if (empty($this->status) || !array_key_exists($this->status, self::STATUSES)) {
            $errors[] = 'Valid status is required';
        }
        
        // Check source posts if type requires them
        if (in_array($this->type, array('post', 'collection')) && empty($this->metadata['source_posts'])) {
            $errors[] = 'Source posts are required for this capsule type';
        }
        
        // Log validation errors
        if (!empty($errors)) {
            AIOX_Logger::warning('Capsule validation failed', array(
                'capsule_id' => $this->id,
                'errors' => $errors
            ));
        }
        
        return empty($errors);
    }
    
    /**
     * Get capsule ID
     */
    public function get_id() {
        return $this->id;
    }
    
    /**
     * Get capsule title
     */
    public function get_title() {
        $title = isset($this->metadata['title']) ? $this->metadata['title'] : '';
        return !empty($title) ? $title : 'AI Indexing Information - Capsule';
    }
    
    /**
     * Get capsule type
     */
    public function get_type() {
        return $this->type ?: 'unknown';
    }
    
    /**
     * Get capsule status
     */
    public function get_status() {
        return $this->status ?: 'active';
    }
    
    /**
     * Get capsule description
     */
    public function get_description() {
        return isset($this->metadata['description']) ? $this->metadata['description'] : '';
    }
    
    /**
     * Get created date
     */
    public function get_created_date() {
        $created = isset($this->metadata['created_at']) ? $this->metadata['created_at'] : '';
        if ($created) {
            return date('Y-m-d H:i:s', strtotime($created));
        }
        return 'Unknown';
    }
    
    /**
     * Get capsule data
     */
    public function get_data() {
        return array(
            'id' => $this->id,
            'title' => $this->get_title(),
            'type' => $this->get_type(),
            'status' => $this->get_status(),
            'description' => $this->get_description(),
            'metadata' => $this->metadata,
            'content' => $this->content,
            'created_at' => $this->get_created_date()
        );
    }
    
    /**
     * Get capsule summary
     */
    public function get_summary() {
        return array(
            'id' => $this->id,
            'title' => $this->get_title(),
            'type' => $this->get_type(),
            'status' => $this->get_status(),
            'created_at' => $this->get_created_date(),
            'updated_at' => isset($this->metadata['updated_at']) ? $this->metadata['updated_at'] : '',
            'file_size' => isset($this->metadata['file_size']) ? $this->metadata['file_size'] : 0,
            'source_posts_count' => isset($this->metadata['source_posts']) && is_array($this->metadata['source_posts']) ? count($this->metadata['source_posts']) : 0,
            'qa_pairs_count' => isset($this->content['qa_pairs']) && is_array($this->content['qa_pairs']) ? count($this->content['qa_pairs']) : 0,
            'topics_count' => isset($this->content['topics']) && is_array($this->content['topics']) ? count($this->content['topics']) : 0
        );
    }
}
?>
