<?php
/**
 * AIOX Comprehensive Processor - SERVER-ONLY HARDENED VERSION
 * 
 * This version REQUIRES server-side processing. No client-side fallback.
 * This protects your intellectual property by making the client 100% dependent
 * on your server infrastructure.
 * 
 * SECURITY: All local processing logic has been removed.
 * 
 * @package AIOX_Client
 * @since 2.0.0
 */

if (!defined('ABSPATH')) {
    exit;
}

class AIOX_Comprehensive_Processor {
    
    private $ai_processor;
    private $capsule_manager;
    
    public function __construct() {
        $this->ai_processor = new AIOX_AI_Only_Processor();
        $this->capsule_manager = new AIOX_Capsule_Manager();
    }
    
    /**
     * Process a post completely - SERVER-ONLY MODE
     * 
     * This method now REQUIRES server-generated data.
     * No client-side fallback processing is available.
     */
    public function process_post_completely($post_id) {
        $post = get_post($post_id);
        if (!$post) {
            throw new Exception('Post not found: ' . $post_id);
        }
        
        AIOX_Logger::info("Comprehensive Processor [SERVER-ONLY MODE]: Starting processing for post {$post_id}");
        
        try {
            // SECURITY: Verify server-side processing is enabled
            $server_processing_enabled = get_option('aiox_server_side_processing', true);
            
            if (!$server_processing_enabled) {
                throw new Exception(
                    'SERVER-ONLY MODE: Server-side processing is disabled. ' .
                    'This version requires server processing. No client-side fallback available. ' .
                    'Please enable: update_option("aiox_server_side_processing", true)'
                );
            }
            
            // Step 1: AI Processing (must return server data)
            $ai_result = $this->ai_processor->process_post($post_id);
            
            if (!$ai_result || !isset($ai_result['success']) || !$ai_result['success']) {
                throw new Exception('AI processing failed: ' . (isset($ai_result['message']) ? $ai_result['message'] : 'Unknown error'));
            }
            
            $processed_data = $ai_result['data'];
            
            // SECURITY CHECK: Verify this is server-processed data
            if (!isset($processed_data['server_processed']) || $processed_data['server_processed'] !== true) {
                // This is CRITICAL - without server data, we cannot proceed
                AIOX_Logger::error("SECURITY: Received non-server-processed data. Rejecting.", array(
                    'post_id' => $post_id,
                    'has_server_flag' => isset($processed_data['server_processed']),
                    'server_flag_value' => $processed_data['server_processed'] ?? 'not set'
                ));
                
                throw new Exception(
                    'SERVER-ONLY MODE: Server-generated data required. ' .
                    'The response did not include server-processed ingredients and capsules. ' .
                    'This may indicate: 1) Server plugin not updated, 2) Server processing disabled, ' .
                    '3) API communication error. Please check server configuration.'
                );
            }
            
            AIOX_Logger::info("Comprehensive Processor: ✅ VERIFIED server-side processed data (v2.0)", array('post_id' => $post_id));
            
            // Load server data handler
            if (!class_exists('AIOX_Server_Data_Handler')) {
                require_once AIOX_PLUGIN_DIR . 'core/class-aiox-server-data-handler.php';
            }
            
            // Store all server-generated data (ingredients, capsules, metadata)
            $store_result = AIOX_Server_Data_Handler::store_server_data($post_id, $processed_data);
            
            if (!$store_result['success']) {
                AIOX_Logger::error("Failed to store server data", array(
                    'post_id' => $post_id,
                    'errors' => $store_result['errors']
                ));
                throw new Exception('Failed to store server-generated data: ' . implode(', ', $store_result['errors']));
            }
            
            AIOX_Logger::info("Comprehensive Processor: ✅ Successfully stored server data", array(
                'post_id' => $post_id,
                'stored' => $store_result['stored']
            ));
            
            // Verify critical data was stored
            if (!isset($store_result['stored']['ingredients']) || $store_result['stored']['ingredients'] === 0) {
                AIOX_Logger::warning("No ingredients were stored", array('post_id' => $post_id));
            }
            
            if (!isset($store_result['stored']['capsules']) || $store_result['stored']['capsules'] === 0) {
                AIOX_Logger::warning("No capsules were stored", array('post_id' => $post_id));
            }
            
            // NOTE: File generation is now handled by AIOX_Server_Data_Handler::store_server_data()
            // which calls generate_files_for_post() internally. This generates:
            // - Individual post file: .well-known/posts/{post_id}.json
            // - Main manifest: .well-known/aio.json
            // - Delta feed: .well-known/aiox.ndjson
            // - Pack files: .well-known/packs/
            // - Ingredients index: .well-known/aiox-ingredients.json
            // - TDM-REP: .well-known/tdmrep.json
            AIOX_Logger::info("Comprehensive Processor: All files generated via server_data_handler");
            
            // Mark processing complete
            update_post_meta($post_id, '_aiox_processing_complete', true);
            update_post_meta($post_id, '_aiox_last_processed', current_time('mysql'));
            update_post_meta($post_id, '_aiox_comprehensive_processed', true);
            update_post_meta($post_id, '_aiox_server_only_mode', true);
            
            AIOX_Logger::info("Comprehensive Processor: ✅ COMPLETE - Post {$post_id} processed successfully in SERVER-ONLY mode");
            
            return array(
                'success' => true,
                'post_id' => $post_id,
                'mode' => 'server_only',
                'ingredients_count' => $store_result['stored']['ingredients'] ?? 0,
                'capsules_count' => $store_result['stored']['capsules'] ?? 0,
                'message' => 'Post processed successfully using server-side generation'
            );
            
        } catch (Exception $e) {
            AIOX_Logger::error("Comprehensive Processor [SERVER-ONLY MODE]: Processing failed for post {$post_id}: " . $e->getMessage());
            
            return array(
                'success' => false,
                'post_id' => $post_id,
                'error' => $e->getMessage(),
                'mode' => 'server_only',
                'message' => 'Processing failed: ' . $e->getMessage()
            );
        }
    }
    
    /**
     * Batch process multiple posts - SERVER-ONLY MODE
     */
    public function batch_process_posts($post_ids, $options = array()) {
        if (!is_array($post_ids)) {
            throw new Exception('Post IDs must be an array');
        }
        
        AIOX_Logger::info("Comprehensive Processor [SERVER-ONLY MODE]: Starting batch processing for " . count($post_ids) . " posts");
        
        $results = array(
            'total' => count($post_ids),
            'successful' => 0,
            'failed' => 0,
            'errors' => array()
        );
        
        foreach ($post_ids as $post_id) {
            try {
                $result = $this->process_post_completely($post_id);
                
                if ($result['success']) {
                    $results['successful']++;
                } else {
                    $results['failed']++;
                    $results['errors'][$post_id] = $result['message'] ?? 'Unknown error';
                }
                
                // Small delay to prevent overwhelming the server
                if (isset($options['delay_ms']) && $options['delay_ms'] > 0) {
                    usleep($options['delay_ms'] * 1000);
                }
                
            } catch (Exception $e) {
                $results['failed']++;
                $results['errors'][$post_id] = $e->getMessage();
                AIOX_Logger::error("Batch processing failed for post {$post_id}: " . $e->getMessage());
            }
        }
        
        AIOX_Logger::info("Comprehensive Processor [SERVER-ONLY MODE]: Batch processing complete", array(
            'total' => $results['total'],
            'successful' => $results['successful'],
            'failed' => $results['failed']
        ));
        
        return $results;
    }
    
    /**
     * Get processing status for a post
     */
    public function get_processing_status($post_id) {
        return array(
            'processed' => (bool) get_post_meta($post_id, '_aiox_processed', true),
            'server_processed' => (bool) get_post_meta($post_id, '_aiox_server_processed', true),
            'server_only_mode' => (bool) get_post_meta($post_id, '_aiox_server_only_mode', true),
            'last_processed' => get_post_meta($post_id, '_aiox_last_processed', true),
            'ingredients_count' => (int) get_post_meta($post_id, '_aiox_ingredients_count', true),
            'capsules_count' => (int) get_post_meta($post_id, '_aiox_capsules_count', true),
            'comprehensive_processed' => (bool) get_post_meta($post_id, '_aiox_comprehensive_processed', true)
        );
    }
    
    /**
     * Verify post has server-generated data
     */
    public function verify_server_data($post_id) {
        $status = $this->get_processing_status($post_id);
        
        if (!$status['server_processed']) {
            return array(
                'valid' => false,
                'message' => 'Post was not processed by server'
            );
        }
        
        if ($status['ingredients_count'] === 0) {
            return array(
                'valid' => false,
                'message' => 'No ingredients found'
            );
        }
        
        // Verify ingredients have server marker
        if (class_exists('AIOX_Ingredients')) {
            $ingredients = AIOX_Ingredients::get_post_ingredients($post_id);
            $has_server_marker = false;
            
            foreach ($ingredients as $ingredient) {
                if (isset($ingredient['generated_by']) && $ingredient['generated_by'] === 'server') {
                    $has_server_marker = true;
                    break;
                }
            }
            
            if (!$has_server_marker) {
                return array(
                    'valid' => false,
                    'message' => 'Ingredients do not have server-generated marker'
                );
            }
        }
        
        return array(
            'valid' => true,
            'message' => 'Post has valid server-generated data',
            'ingredients_count' => $status['ingredients_count'],
            'capsules_count' => $status['capsules_count']
        );
    }
    
    /**
     * Delete all AIOX components for a post
     * 
     * Removes ALL server-generated data: ingredients, capsules, metadata, badge, files
     * This is called by the "Clear AI Data" button in Processing Manager
     * 
     * @param int $post_id WordPress post ID
     * @return array Result with success status
     */
    public function delete_post_components($post_id) {
        global $wpdb;
        
        AIOX_Logger::info("Comprehensive Processor: Deleting ALL components for post {$post_id}");
        
        $results = array(
            'success' => true,
            'post_id' => $post_id,
            'deleted' => array(),
            'errors' => array()
        );
        
        try {
            // 1. Delete capsule from database (not just post meta)
            if (class_exists('AIOX_Capsule_Manager')) {
                $capsule_manager = new AIOX_Capsule_Manager();
                $existing_capsule = $capsule_manager->get_capsule_by_post_id($post_id);
                
                if ($existing_capsule) {
                    $capsule_id = $existing_capsule->get_id();
                    
                    // Delete capsule files from BOTH locations
                    $well_known_dir = ABSPATH . '.well-known';
                    
                    // Location 1: Root .well-known directory
                    $capsule_file = $well_known_dir . '/capsule_' . $capsule_id . '.json';
                    if (file_exists($capsule_file)) {
                        @unlink($capsule_file);
                        AIOX_Logger::info("Deleted capsule file from root", array('post_id' => $post_id, 'capsule_id' => $capsule_id));
                    }
                    
                    // Location 2: Capsules subdirectory
                    $public_capsule_file = $well_known_dir . '/capsules/' . $capsule_id . '.json';
                    if (file_exists($public_capsule_file)) {
                        @unlink($public_capsule_file);
                        AIOX_Logger::info("Deleted public capsule file", array('post_id' => $post_id, 'capsule_id' => $capsule_id));
                    }
                    
                    // Delete from database using correct column 'id'
                    $deleted = $wpdb->delete(
                        $wpdb->prefix . 'aiox_capsules',
                        array('id' => $capsule_id),
                        array('%s')
                    );
                    
                    if ($deleted) {
                        $results['deleted']['capsule'] = $capsule_id;
                    }
                }
            }
            
            // 2. Delete ALL _aiox_* post meta using wildcard query
            $wpdb->query(
                $wpdb->prepare(
                    "DELETE FROM {$wpdb->postmeta} WHERE post_id = %d AND meta_key LIKE '_aiox_%%'",
                    $post_id
                )
            );
            $results['deleted']['all_aiox_meta'] = true;
            
            // 3. Delete well-known files for this post
            $well_known_dir = ABSPATH . '.well-known';
            if (is_dir($well_known_dir)) {
                // Delete individual post manifesto file
                $post_file = $well_known_dir . '/aiox-post-' . $post_id . '.json';
                if (file_exists($post_file)) {
                    @unlink($post_file);
                    $results['deleted']['post_manifesto'] = true;
                }
                
                // Delete post-specific files
                $post_json = $well_known_dir . '/posts/' . $post_id . '.json';
                if (file_exists($post_json)) {
                    @unlink($post_json);
                    $results['deleted']['post_json'] = true;
                }
            }
            
            // 4. Clear WordPress cache
            wp_cache_delete($post_id, 'post_meta');
            clean_post_cache($post_id);
            
            // 5. Regenerate master manifesto files (to remove this post from listings)
            if (class_exists('AIOX_Hooks')) {
                $hooks = new AIOX_Hooks();
                $hooks->generate_manifest();
                $hooks->generate_delta_feed();
                $hooks->generate_pack_file();
                $results['deleted']['regenerated_manifesto'] = true;
                $results['deleted']['regenerated_delta'] = true;
                $results['deleted']['regenerated_pack'] = true;
            } else if (class_exists('AIOX_Generator')) {
                // Fallback to old method if Hooks not available
                $generator = new AIOX_Generator();
                $generator->generate_manifesto_file();
                $results['deleted']['regenerated_manifesto'] = true;
            }
            
            // 6. Regenerate ingredients index (to remove this post)
            if (class_exists('AIOX_Ingredients')) {
                AIOX_Ingredients::export_ingredients_file();
                $results['deleted']['regenerated_ingredients'] = true;
            }
            
            AIOX_Logger::info("Comprehensive Processor: Successfully deleted ALL components for post {$post_id}", $results['deleted']);
            
            return $results;
            
        } catch (Exception $e) {
            AIOX_Logger::error("Comprehensive Processor: Failed to delete components for post {$post_id}: " . $e->getMessage());
            $results['success'] = false;
            $results['errors'][] = $e->getMessage();
            return $results;
        }
    }
}