<?php
/**
 * AIOX Comprehensive Synchronization Handler
 * 
 * Handles synchronization of all AIOX components (capsules, ingredients, 
 * manifesto, badges, metadata) with post lifecycle events.
 * 
 * Phase 1: Foundation setup - establishes the synchronization framework
 * without modifying existing functionality.
 */

if (!defined('ABSPATH')) {
    exit;
}

class AIOX_Comprehensive_Sync {
    
    /**
     * Comprehensive processor instance
     */
    private $comprehensive_processor;
    
    /**
     * Sync statistics
     */
    private $sync_stats;
    
    /**
     * Constructor
     */
    public function __construct() {
        AIOX_Logger::info('AIOX Comprehensive Sync: Initializing Phase 1 foundation');
        
        try {
            // Initialize comprehensive processor
            require_once AIOX_PLUGIN_DIR . 'core/class-aiox-comprehensive-processor.php';
            $this->comprehensive_processor = new AIOX_Comprehensive_Processor();
            
            // Initialize sync statistics (load existing or create new)
            $this->sync_stats = get_option('aiox_sync_stats', array(
                'posts_synced' => 0,
                'components_synced' => 0,
                'last_sync' => current_time('mysql'),
                'errors' => array()
            ));
            
            AIOX_Logger::info('AIOX Comprehensive Sync: Phase 1 foundation initialized successfully');
            
        } catch (Exception $e) {
            AIOX_Logger::error('AIOX Comprehensive Sync: Failed to initialize Phase 1 foundation', array(
                'error' => $e->getMessage(),
                'trace' => $e->getTraceAsString()
            ));
        }
    }
    
    /**
     * Synchronize all components for a post
     * This is the main synchronization method that ensures all AIOX components
     * are properly created/updated when a post changes
     */
    public function sync_post_components($post_id, $action = 'update') {
        $post = get_post($post_id);
        if (!$post) {
            throw new Exception("Post not found: {$post_id}");
        }
        
        AIOX_Logger::info("Comprehensive Sync: Starting {$action} sync for post {$post_id}");
        
        try {
            $sync_result = array(
                'post_id' => $post_id,
                'action' => $action,
                'components' => array(),
                'success' => true,
                'errors' => array()
            );
            
            switch ($action) {
                case 'create':
                case 'update':
                    $sync_result = $this->sync_post_update($post_id, $sync_result);
                    break;
                    
                case 'delete':
                    $sync_result = $this->sync_post_deletion($post_id, $sync_result);
                    break;
                    
                default:
                    throw new Exception("Unknown sync action: {$action}");
            }
            
            // Update sync statistics
            $this->update_sync_stats($sync_result);
            
            AIOX_Logger::info("Comprehensive Sync: Completed {$action} sync for post {$post_id}", $sync_result);
            
            return $sync_result;
            
        } catch (Exception $e) {
            AIOX_Logger::error("Comprehensive Sync: Failed {$action} sync for post {$post_id}", array(
                'error' => $e->getMessage(),
                'trace' => $e->getTraceAsString()
            ));
            
            throw $e;
        }
    }
    
    /**
     * Synchronize components when post is created or updated
     */
    private function sync_post_update($post_id, $sync_result) {
        // Check if comprehensive processing is enabled
        $comprehensive_enabled = get_option('aiox_enable_comprehensive_processing', true);
        
        if (!$comprehensive_enabled) {
            AIOX_Logger::debug("Comprehensive Sync: Comprehensive processing disabled, skipping sync for post {$post_id}");
            $sync_result['components']['note'] = 'Comprehensive processing disabled';
            return $sync_result;
        }
        
        try {
            // 1. Sync Capsules (preserve existing functionality)
            $sync_result['components']['capsule'] = $this->sync_capsule($post_id);
            
            // 2. Sync Ingredients
            $sync_result['components']['ingredients'] = $this->sync_ingredients($post_id);
            
            // 3. Sync Metadata
            $sync_result['components']['metadata'] = $this->sync_metadata($post_id);
            
            // 4. Sync Manifesto Files
            $sync_result['components']['manifesto'] = $this->sync_manifesto_files($post_id);
            
            // 5. Sync Badge Data
            $sync_result['components']['badge'] = $this->sync_badge_data($post_id);
            
            // 6. Update processing status
            update_post_meta($post_id, '_aiox_processing_complete', '1');
            update_post_meta($post_id, '_aiox_last_processed', current_time('mysql'));
            update_post_meta($post_id, '_aiox_sync_version', '1.0');
            
            $sync_result['components']['status'] = 'complete';
            
        } catch (Exception $e) {
            $sync_result['success'] = false;
            $sync_result['errors'][] = $e->getMessage();
            
            // Mark as failed but don't throw - allow partial sync
            update_post_meta($post_id, '_aiox_processing_complete', '0');
            update_post_meta($post_id, '_aiox_last_error', $e->getMessage());
        }
        
        return $sync_result;
    }
    
    /**
     * Synchronize components when post is deleted
     */
    private function sync_post_deletion($post_id, $sync_result) {
        try {
            // 1. Delete Capsule
            $sync_result['components']['capsule'] = $this->delete_capsule($post_id);
            
            // 2. Delete Ingredients
            $sync_result['components']['ingredients'] = $this->delete_ingredients($post_id);
            
            // 3. Delete Metadata
            $sync_result['components']['metadata'] = $this->delete_metadata($post_id);
            
            // 4. Update Manifesto Files (remove post)
            $sync_result['components']['manifesto'] = $this->update_manifesto_on_deletion($post_id);
            
            // 5. Clear Badge Data
            $sync_result['components']['badge'] = $this->clear_badge_data($post_id);
            
            // 6. Clean up all post meta
            $this->cleanup_post_meta($post_id);
            
            $sync_result['components']['cleanup'] = 'complete';
            
        } catch (Exception $e) {
            $sync_result['success'] = false;
            $sync_result['errors'][] = $e->getMessage();
        }
        
        return $sync_result;
    }
    
    /**
     * Sync capsule component
     */
    private function sync_capsule($post_id) {
        try {
            // Check if capsule already exists
            $existing_capsule_id = get_post_meta($post_id, '_aiox_capsule_id', true);
            
            if ($existing_capsule_id) {
                // Update existing capsule
                AIOX_Logger::debug("Comprehensive Sync: Updating existing capsule for post {$post_id}");
                return array('action' => 'updated', 'capsule_id' => $existing_capsule_id);
            } else {
                // Create new capsule using existing capsule manager
                require_once AIOX_PLUGIN_DIR . 'core/class-aiox-capsule-manager.php';
                $capsule_manager = new AIOX_Capsule_Manager();
                $capsule = $capsule_manager->create_capsule_from_post($post_id);
                
                if ($capsule) {
                    $capsule_id = method_exists($capsule, 'get_id') ? $capsule->get_id() : 'unknown';
                    update_post_meta($post_id, '_aiox_capsule_id', $capsule_id);
                    
                    AIOX_Logger::debug("Comprehensive Sync: Created new capsule {$capsule_id} for post {$post_id}");
                    return array('action' => 'created', 'capsule_id' => $capsule_id);
                } else {
                    throw new Exception('Failed to create capsule');
                }
            }
            
        } catch (Exception $e) {
            AIOX_Logger::warning("Comprehensive Sync: Capsule sync failed for post {$post_id}: " . $e->getMessage());
            return array('action' => 'failed', 'error' => $e->getMessage());
        }
    }
    
    /**
     * Sync ingredients component
     */
    private function sync_ingredients($post_id) {
        try {
            $post = get_post($post_id);
            if (!$post) {
                throw new Exception('Post not found');
            }
            
            // Extract ingredients
            $ingredients = AIOX_Ingredients::extract_ingredients($post);
            
            // Save ingredients
            AIOX_Ingredients::save_post_ingredients($post_id, $ingredients);
            
            AIOX_Logger::debug("Comprehensive Sync: Extracted " . count($ingredients) . " ingredients for post {$post_id}");
            
            return array(
                'action' => 'extracted',
                'count' => count($ingredients),
                'types' => array_unique(array_column($ingredients, 'type'))
            );
            
        } catch (Exception $e) {
            AIOX_Logger::warning("Comprehensive Sync: Ingredients sync failed for post {$post_id}: " . $e->getMessage());
            return array('action' => 'failed', 'error' => $e->getMessage());
        }
    }
    
    /**
     * Sync metadata component
     */
    private function sync_metadata($post_id) {
        try {
            $post = get_post($post_id);
            if (!$post) {
                throw new Exception('Post not found');
            }
            
            // Generate metadata
            $metadata = AIOX_Metadata::classify_content($post);
            
            // Save metadata
            AIOX_Metadata::save_post_metadata($post_id, $metadata);
            
            AIOX_Logger::debug("Comprehensive Sync: Generated " . count($metadata) . " metadata fields for post {$post_id}");
            
            return array(
                'action' => 'generated',
                'fields' => count($metadata),
                'categories' => array_keys($metadata)
            );
            
        } catch (Exception $e) {
            AIOX_Logger::warning("Comprehensive Sync: Metadata sync failed for post {$post_id}: " . $e->getMessage());
            return array('action' => 'failed', 'error' => $e->getMessage());
        }
    }
    
    /**
     * Sync manifesto files component
     */
    private function sync_manifesto_files($post_id) {
        try {
            $processed_data = get_post_meta($post_id, '_aiox_processed_data', true);
            
            if ($processed_data) {
                // Regenerate all manifesto files using AIOX_Hooks
                if (class_exists('AIOX_Hooks')) {
                    $hooks = new AIOX_Hooks();
                    $hooks->generate_manifest();
                    $hooks->generate_delta_feed();
                    $hooks->generate_pack_file();
                    
                    AIOX_Logger::debug("Comprehensive Sync: Updated all manifesto files for post {$post_id}");
                    
                    return array(
                        'action' => 'updated',
                        'files' => array('aio.json', 'aiox.ndjson', 'posts/' . $post_id . '.json', 'packs/')
                    );
                } else {
                    AIOX_Logger::warning("AIOX_Hooks not available, cannot regenerate manifesto files");
                    return array('action' => 'skipped', 'reason' => 'AIOX_Hooks not available');
                }
            } else {
                AIOX_Logger::debug("Comprehensive Sync: No processed data available for manifesto files for post {$post_id}");
                return array('action' => 'skipped', 'reason' => 'No processed data');
            }
            
        } catch (Exception $e) {
            AIOX_Logger::warning("Comprehensive Sync: Manifesto sync failed for post {$post_id}: " . $e->getMessage());
            return array('action' => 'failed', 'error' => $e->getMessage());
        }
    }
    
    /**
     * Sync badge data component
     */
    private function sync_badge_data($post_id) {
        try {
            $processed_data = get_post_meta($post_id, '_aiox_processed_data', true);
            
            if ($processed_data) {
                // Update badge timestamp
                update_post_meta($post_id, '_aiox_badge_updated', current_time('mysql'));
                
                AIOX_Logger::debug("Comprehensive Sync: Updated badge data for post {$post_id}");
                
                return array('action' => 'updated', 'timestamp' => current_time('mysql'));
            } else {
                return array('action' => 'skipped', 'reason' => 'No processed data');
            }
            
        } catch (Exception $e) {
            AIOX_Logger::warning("Comprehensive Sync: Badge sync failed for post {$post_id}: " . $e->getMessage());
            return array('action' => 'failed', 'error' => $e->getMessage());
        }
    }
    
    /**
     * Delete capsule component
     */
    private function delete_capsule($post_id) {
        try {
            $capsule_id = get_post_meta($post_id, '_aiox_capsule_id', true);
            
            if ($capsule_id) {
                // Use comprehensive processor to delete capsule
                $this->comprehensive_processor->delete_post_components($post_id);
                
                AIOX_Logger::debug("Comprehensive Sync: Deleted capsule {$capsule_id} for post {$post_id}");
                
                return array('action' => 'deleted', 'capsule_id' => $capsule_id);
            } else {
                return array('action' => 'skipped', 'reason' => 'No capsule found');
            }
            
        } catch (Exception $e) {
            AIOX_Logger::warning("Comprehensive Sync: Capsule deletion failed for post {$post_id}: " . $e->getMessage());
            return array('action' => 'failed', 'error' => $e->getMessage());
        }
    }
    
    /**
     * Delete ingredients component
     */
    private function delete_ingredients($post_id) {
        try {
            AIOX_Ingredients::delete_post_ingredients($post_id);
            
            AIOX_Logger::debug("Comprehensive Sync: Deleted ingredients for post {$post_id}");
            
            return array('action' => 'deleted');
            
        } catch (Exception $e) {
            AIOX_Logger::warning("Comprehensive Sync: Ingredients deletion failed for post {$post_id}: " . $e->getMessage());
            return array('action' => 'failed', 'error' => $e->getMessage());
        }
    }
    
    /**
     * Delete metadata component
     */
    private function delete_metadata($post_id) {
        try {
            AIOX_Metadata::delete_post_metadata($post_id);
            
            AIOX_Logger::debug("Comprehensive Sync: Deleted metadata for post {$post_id}");
            
            return array('action' => 'deleted');
            
        } catch (Exception $e) {
            AIOX_Logger::warning("Comprehensive Sync: Metadata deletion failed for post {$post_id}: " . $e->getMessage());
            return array('action' => 'failed', 'error' => $e->getMessage());
        }
    }
    
    /**
     * Update manifesto files on post deletion
     */
    /**
 * Update manifesto files on post deletion
 * NEW: Removes individual post file and updates manifest
 */
private function update_manifesto_on_deletion($post_id) {
    try {
        $well_known_dir = ABSPATH . '.well-known';
        $posts_dir = $well_known_dir . '/posts';
        
        // Delete NEW format individual post file
        $post_file = $posts_dir . '/' . $post_id . '.json';
        if (file_exists($post_file)) {
            $deleted = unlink($post_file);
            if ($deleted) {
                AIOX_Logger::info("Comprehensive Sync: Deleted individual post file: {$post_file}");
            } else {
                AIOX_Logger::warning("Comprehensive Sync: Failed to delete post file: {$post_file}");
            }
        }
        
        // Also delete OLD format file for backward compatibility
        $old_post_file = $well_known_dir . '/aiox-post-' . $post_id . '.json';
        if (file_exists($old_post_file)) {
            unlink($old_post_file);
            AIOX_Logger::info("Comprehensive Sync: Deleted OLD format post file: {$old_post_file}");
        }
        
        // Regenerate all manifest files to update indexes
        if (class_exists('AIOX_Hooks')) {
            $hooks = new AIOX_Hooks();
            $hooks->generate_manifest();
            $hooks->generate_delta_feed();
            $hooks->generate_pack_file();
            AIOX_Logger::info("Comprehensive Sync: Regenerated all manifests after deletion");
        }
        
        // Regenerate ingredients index
        if (class_exists('AIOX_Ingredients')) {
            AIOX_Ingredients::export_ingredients_file();
            AIOX_Logger::info("Comprehensive Sync: Regenerated ingredients index after deletion");
        }
        
        return array(
            'action' => 'updated',
            'removed_files' => array($post_id . '.json', 'aiox-post-' . $post_id . '.json'),
            'manifest_updated' => true
        );
        
    } catch (Exception $e) {
        AIOX_Logger::warning("Comprehensive Sync: Manifesto update on deletion failed for post {$post_id}: " . $e->getMessage());
        return array('action' => 'failed', 'error' => $e->getMessage());
    }
}
    
    /**
     * Clear badge data component
     */
    private function clear_badge_data($post_id) {
        try {
            delete_post_meta($post_id, '_aiox_processed_data');
            delete_post_meta($post_id, '_aiox_badge_updated');
            
            AIOX_Logger::debug("Comprehensive Sync: Cleared badge data for post {$post_id}");
            
            return array('action' => 'cleared');
            
        } catch (Exception $e) {
            AIOX_Logger::warning("Comprehensive Sync: Badge data clearing failed for post {$post_id}: " . $e->getMessage());
            return array('action' => 'failed', 'error' => $e->getMessage());
        }
    }
    
    /**
     * Clean up all AIOX post meta
     */
    private function cleanup_post_meta($post_id) {
        $meta_keys = array(
            '_aiox_processed_data',
            '_aiox_processing_complete',
            '_aiox_last_processed',
            '_aiox_ingredients',
            '_aiox_metadata',
            '_aiox_capsule_id',
            '_aiox_badge_updated',
            '_aiox_sync_version',
            '_aiox_last_error'
        );
        
        foreach ($meta_keys as $key) {
            delete_post_meta($post_id, $key);
        }
        
        AIOX_Logger::debug("Comprehensive Sync: Cleaned up post meta for post {$post_id}");
    }
    
    /**
     * Update synchronization statistics
     */
    private function update_sync_stats($sync_result) {
        if ($sync_result['success']) {
            $this->sync_stats['posts_synced']++;
            $this->sync_stats['components_synced'] += count($sync_result['components']);
        } else {
            $this->sync_stats['errors'] = array_merge($this->sync_stats['errors'], $sync_result['errors']);
        }
        
        $this->sync_stats['last_sync'] = current_time('mysql');
        
        // Store stats in options for persistence
        update_option('aiox_sync_stats', $this->sync_stats);
    }
    
    /**
     * Get synchronization statistics
     */
    public function get_sync_stats() {
        return get_option('aiox_sync_stats', $this->sync_stats);
    }
    
    /**
     * Reset synchronization statistics
     */
    public function reset_sync_stats() {
        $this->sync_stats = array(
            'posts_synced' => 0,
            'components_synced' => 0,
            'last_sync' => current_time('mysql'),
            'errors' => array()
        );
        
        update_option('aiox_sync_stats', $this->sync_stats);
    }
    
    /**
     * Check if comprehensive processing is enabled
     */
    public function is_comprehensive_processing_enabled() {
        return get_option('aiox_enable_comprehensive_processing', true);
    }
    
    /**
     * Enable comprehensive processing
     */
    public function enable_comprehensive_processing() {
        update_option('aiox_enable_comprehensive_processing', true);
        AIOX_Logger::info('Comprehensive Sync: Comprehensive processing enabled');
    }
    
    /**
     * Disable comprehensive processing
     */
    public function disable_comprehensive_processing() {
        update_option('aiox_enable_comprehensive_processing', false);
        AIOX_Logger::info('Comprehensive Sync: Comprehensive processing disabled');
    }
}
