<?php
/**
 * AIOX Cryptographic Functions
 * 
 * Handles Ed25519 key generation, signing, and verification
 */

if (!defined('ABSPATH')) {
    exit;
}

class AIOX_Crypto {
    
    private $keypair_option = 'aiox_ed25519_keypair';
    
    /**
     * Get or create Ed25519 keypair
     */
    public function get_or_create_keypair() {
        // Check if we have a stored keypair
        $stored_keypair = get_option($this->keypair_option);
        
        if ($stored_keypair && isset($stored_keypair['public']) && isset($stored_keypair['private'])) {
            return $stored_keypair;
        }
        
        // Generate new keypair
        return $this->generate_keypair();
    }
    
    /**
     * Generate new Ed25519 keypair
     */
    public function generate_keypair() {
        try {
            // Check if sodium is available
            if (!function_exists('sodium_crypto_sign_keypair')) {
                AIOX_Logger::error('Sodium extension not available for Ed25519 key generation');
                return $this->generate_fallback_keypair();
            }
            
            // Generate Ed25519 keypair
            $keypair = sodium_crypto_sign_keypair();
            $public_key = sodium_crypto_sign_publickey($keypair);
            $private_key = sodium_crypto_sign_secretkey($keypair);
            
            $keypair_data = array(
                'public' => $public_key,
                'private' => $private_key,
                'algorithm' => 'Ed25519',
                'generated_at' => current_time('c')
            );
            
            // Store keypair
            update_option($this->keypair_option, $keypair_data);
            
            AIOX_Logger::info('Ed25519 keypair generated successfully');
            
            return $keypair_data;
            
        } catch (Exception $e) {
            AIOX_Logger::error('Failed to generate Ed25519 keypair', array('error' => $e->getMessage()));
            return $this->generate_fallback_keypair();
        }
    }
    
    /**
     * Generate fallback keypair when sodium is not available
     */
    private function generate_fallback_keypair() {
        // Generate a simple RSA-like keypair as fallback
        $keypair_data = array(
            'public' => $this->generate_random_key(32),
            'private' => $this->generate_random_key(64),
            'algorithm' => 'Fallback',
            'generated_at' => current_time('c'),
            'note' => 'Fallback keypair - sodium extension not available'
        );
        
        // Store keypair
        update_option($this->keypair_option, $keypair_data);
        
        AIOX_Logger::info('Fallback keypair generated (sodium not available)');
        
        return $keypair_data;
    }
    
    /**
     * Generate random key for fallback
     */
    private function generate_random_key($length) {
        if (function_exists('random_bytes')) {
            return random_bytes($length);
        } elseif (function_exists('openssl_random_pseudo_bytes')) {
            return openssl_random_pseudo_bytes($length);
        } else {
            // Last resort - not cryptographically secure
            $key = '';
            for ($i = 0; $i < $length; $i++) {
                $key .= chr(mt_rand(0, 255));
            }
            return $key;
        }
    }
    
    /**
     * Sign data with private key
     */
    public function sign_data($data) {
        $keypair = $this->get_or_create_keypair();
        
        if (!$keypair || !isset($keypair['private'])) {
            return false;
        }
        
        try {
            if ($keypair['algorithm'] === 'Ed25519' && function_exists('sodium_crypto_sign_detached')) {
                return sodium_crypto_sign_detached($data, $keypair['private']);
            } else {
                // Fallback signature
                return hash_hmac('sha256', $data, $keypair['private'], true);
            }
        } catch (Exception $e) {
            AIOX_Logger::error('Failed to sign data', array('error' => $e->getMessage()));
            return false;
        }
    }
    
    /**
     * Verify signature
     */
    public function verify_signature($data, $signature) {
        $keypair = $this->get_or_create_keypair();
        
        if (!$keypair || !isset($keypair['public'])) {
            return false;
        }
        
        try {
            if ($keypair['algorithm'] === 'Ed25519' && function_exists('sodium_crypto_sign_verify_detached')) {
                return sodium_crypto_sign_verify_detached($signature, $data, $keypair['public']);
            } else {
                // Fallback verification
                $expected = hash_hmac('sha256', $data, $keypair['private'], true);
                return hash_equals($expected, $signature);
            }
        } catch (Exception $e) {
            AIOX_Logger::error('Failed to verify signature', array('error' => $e->getMessage()));
            return false;
        }
    }
    
    /**
     * Get public key in PEM format
     */
    public function get_public_key_pem() {
        $keypair = $this->get_or_create_keypair();
        
        if (!$keypair || !isset($keypair['public'])) {
            return false;
        }
        
        $pem_content = "-----BEGIN PUBLIC KEY-----\n";
        $pem_content .= chunk_split(base64_encode($keypair['public']), 64, "\n");
        $pem_content .= "-----END PUBLIC KEY-----\n";
        
        return $pem_content;
    }
    
    /**
     * Get keypair info (without private key)
     */
    public function get_keypair_info() {
        $keypair = $this->get_or_create_keypair();
        
        if (!$keypair) {
            return false;
        }
        
        return array(
            'algorithm' => $keypair['algorithm'],
            'generated_at' => $keypair['generated_at'],
            'public_key_length' => strlen($keypair['public']),
            'has_private_key' => isset($keypair['private']),
            'note' => $keypair['note'] ?? null
        );
    }
    
    /**
     * Regenerate keypair
     */
    public function regenerate_keypair() {
        delete_option($this->keypair_option);
        return $this->generate_keypair();
    }
    
    /**
     * Generate proof file for all AIOX content
     */
    public function generate_proof_file() {
        try {
            $keypair = $this->get_or_create_keypair();
            if (!$keypair) {
                AIOX_Logger::error('Cannot generate proof file: no keypair available');
                return false;
            }
            
            // Get all processed posts
            $processed_posts = get_posts(array(
                'post_type' => array('post', 'page'),
                'post_status' => 'publish',
                'meta_key' => '_aiox_processed',
                'meta_value' => '1',
                'numberposts' => -1
            ));
            
            $proofs = array();
            $site_proof = array(
                'site_url' => home_url(),
                'generated_at' => current_time('c'),
                'algorithm' => $keypair['algorithm'],
                'public_key' => base64_encode($keypair['public']),
                'total_posts' => count($processed_posts),
                'posts' => array()
            );
            
            foreach ($processed_posts as $post) {
                // Get post ingredients for signing
                if (class_exists('AIOX_Ingredients')) {
                    $ingredients = AIOX_Ingredients::get_post_ingredients($post->ID);
                } else {
                    $ingredients = array();
                }
                
                // Create content hash for signing
                $content_data = array(
                    'post_id' => $post->ID,
                    'post_title' => $post->post_title,
                    'post_url' => get_permalink($post->ID),
                    'post_content' => $post->post_content,
                    'published_at' => $post->post_date,
                    'modified_at' => $post->post_modified,
                    'ingredients' => $ingredients
                );
                
                $content_json = json_encode($content_data, JSON_UNESCAPED_SLASHES);
                $content_hash = hash('sha256', $content_json);
                
                // Sign the content hash
                $signature = $this->sign_data($content_hash);
                
                if ($signature) {
                    $post_proof = array(
                        'post_id' => $post->ID,
                        'post_title' => $post->post_title,
                        'post_url' => get_permalink($post->ID),
                        'content_hash' => $content_hash,
                        'signature' => base64_encode($signature),
                        'signed_at' => current_time('c'),
                        'ingredients_count' => count($ingredients)
                    );
                    
                    $proofs[] = $post_proof;
                    $site_proof['posts'][] = array(
                        'post_id' => $post->ID,
                        'title' => $post->post_title,
                        'url' => get_permalink($post->ID),
                        'content_hash' => $content_hash
                    );
                }
            }
            
            // Create site-wide signature
            $site_content = json_encode($site_proof['posts'], JSON_UNESCAPED_SLASHES);
            $site_hash = hash('sha256', $site_content);
            $site_signature = $this->sign_data($site_hash);
            
            // Create the complete proof file structure
            $proof_data = array(
                'version' => '1.0',
                'format' => 'aiox-proof',
                'site_proof' => array_merge($site_proof, array(
                    'site_content_hash' => $site_hash,
                    'site_signature' => $site_signature ? base64_encode($site_signature) : null
                )),
                'post_proofs' => $proofs,
                'verification' => array(
                    'algorithm' => $keypair['algorithm'],
                    'public_key_pem' => $this->get_public_key_pem(),
                    'instructions' => 'Verify signatures using the provided public key and the Ed25519 algorithm'
                )
            );
            
            // Write to .well-known directory
            $well_known_dir = ABSPATH . '.well-known';
            if (!is_dir($well_known_dir)) {
                wp_mkdir_p($well_known_dir);
            }
            
            $proof_file = $well_known_dir . '/aiox-proof.json';
            $json_content = json_encode($proof_data, JSON_PRETTY_PRINT | JSON_UNESCAPED_SLASHES);
            
            if (file_put_contents($proof_file, $json_content) !== false) {
                AIOX_Logger::info('Proof file generated successfully', array(
                    'file' => $proof_file,
                    'posts_count' => count($proofs),
                    'algorithm' => $keypair['algorithm']
                ));
                return true;
            } else {
                AIOX_Logger::error('Failed to write proof file', array('file' => $proof_file));
                return false;
            }
            
        } catch (Exception $e) {
            AIOX_Logger::error('Error generating proof file', array('error' => $e->getMessage()));
            return false;
        }
    }
    
    /**
     * Verify proof file integrity
     */
    public function verify_proof_file() {
        $proof_file = ABSPATH . '.well-known/aiox-proof.json';
        
        if (!file_exists($proof_file)) {
            return array('status' => 'fail', 'message' => 'Proof file does not exist');
        }
        
        try {
            $content = file_get_contents($proof_file);
            $proof_data = json_decode($content, true);
            
            if (json_last_error() !== JSON_ERROR_NONE) {
                return array('status' => 'fail', 'message' => 'Proof file contains invalid JSON');
            }
            
            if (!isset($proof_data['post_proofs']) || !isset($proof_data['site_proof'])) {
                return array('status' => 'fail', 'message' => 'Proof file missing required sections');
            }
            
            $verified_posts = 0;
            $total_posts = count($proof_data['post_proofs']);
            
            // Verify a sample of post signatures (to avoid performance issues)
            $sample_size = min(5, $total_posts);
            $sample_posts = array_slice($proof_data['post_proofs'], 0, $sample_size);
            
            foreach ($sample_posts as $post_proof) {
                if (isset($post_proof['content_hash']) && isset($post_proof['signature'])) {
                    $signature = base64_decode($post_proof['signature']);
                    if ($this->verify_signature($post_proof['content_hash'], $signature)) {
                        $verified_posts++;
                    }
                }
            }
            
            if ($verified_posts === $sample_size) {
                return array(
                    'status' => 'pass', 
                    'message' => 'Proof file verified successfully (' . $total_posts . ' posts, ' . $sample_size . ' signatures verified)'
                );
            } else {
                return array(
                    'status' => 'fail', 
                    'message' => 'Proof verification failed (' . $verified_posts . '/' . $sample_size . ' signatures valid)'
                );
            }
            
        } catch (Exception $e) {
            return array('status' => 'fail', 'message' => 'Error verifying proof file: ' . $e->getMessage());
        }
    }
    
    /**
     * Get proof file data
     */
    public function get_proof_file_data() {
        $proof_file = ABSPATH . '.well-known/aiox-proof.json';
        
        if (!file_exists($proof_file)) {
            return false;
        }
        
        $content = file_get_contents($proof_file);
        return json_decode($content, true);
    }
}
