<?php
/**
 * AIOX Custom AI API Client - SERVER-ONLY HARDENED VERSION 2.0
 * 
 * This version REQUIRES server v2.0 responses with structured data.
 * All prompt building and JSON parsing logic has been REMOVED to protect IP.
 * 
 * SECURITY FEATURES:
 * - No prompt engineering exposed
 * - No AI response parsing logic
 * - No fallback processing
 * - Cannot operate without server-generated data
 * 
 * @package AIOX_Client
 * @since 2.0.0
 * @version 2.0.0-final
 */

if (!defined('ABSPATH')) {
    exit;
}

class AIOX_Custom_Client {
    
    private $api_key;
    private $endpoint;
    
    /**
     * Constructor
     */
    public function __construct($api_key = null, $endpoint = null) {
        $this->api_key = $api_key ?: get_option('aiox_custom_api_key', '');
        $this->endpoint = $endpoint ?: get_option('aiox_custom_endpoint', '');
    }

    /**
     * Process complete content analysis - REQUIRES SERVER v2.0
     * 
     * This method sends RAW data to server. Server builds prompts and processes.
     * All client-side prompt building and parsing logic has been REMOVED.
     * 
     * @param string $content Post content (HTML or text)
     * @param string $title Post title
     * @param string $excerpt Post excerpt
     * @param int $qa_count Number of Q&A pairs to generate (default: 5)
     * @return array Structured data from server
     * @throws Exception If server v2.0 response not received
     */
    public function process_content($content, $title = '', $excerpt = '', $qa_count = 5) {
        if (empty($this->api_key)) {
            throw new Exception('Custom API key not configured');
        }
        
        if (empty($this->endpoint)) {
            throw new Exception('Custom endpoint not configured');
        }
        
        // Apply client-side content length limit (security measure)
        if (strlen($content) > 100000) {
            $first_part = substr($content, 0, 80000);
            $last_part = substr($content, -20000);
            $content = $first_part . "\n\n[... content truncated for processing ...]\n\n" . $last_part;
        }
        
        // Send RAW DATA to server - server builds prompts and processes
        AIOX_Logger::log('Custom Client: Sending raw data format to server', AIOX_Logger::INFO);
        AIOX_Logger::log('Custom Client: Content length: ' . strlen($content) . ' chars', AIOX_Logger::INFO);
        AIOX_Logger::log('Custom Client: Title: ' . (!empty($title) ? 'present' : 'empty'), AIOX_Logger::INFO);
        
        $response = $this->make_api_request(array(
            'content' => $content,
            'title' => $title,
            'excerpt' => $excerpt,
            'options' => array(
                'qa_count' => $qa_count
            )
        ));
        
        AIOX_Logger::log('Custom Client: Received response with keys: ' . implode(', ', array_keys($response)), AIOX_Logger::INFO);
        
        // SECURITY CHECK: Verify server v2.0 response format
        if (isset($response['ingredients']) && isset($response['capsules'])) {
            AIOX_Logger::log('Custom Client: ✅ Received SERVER v2.0 structured data', AIOX_Logger::INFO);
            
            // Verify server-generated markers (optional security check)
            $has_server_marker = false;
            if (isset($response['ingredients']) && is_array($response['ingredients'])) {
                foreach ($response['ingredients'] as $ingredient) {
                    if (isset($ingredient['generated_by']) && $ingredient['generated_by'] === 'server') {
                        $has_server_marker = true;
                        break;
                    }
                }
            }
            
            if (!$has_server_marker && !empty($response['ingredients'])) {
                AIOX_Logger::log('Custom Client: INFO - Response missing server-generated marker', AIOX_Logger::INFO);
            }
            
            // Return server data with processing metadata
            return array(
                'ingredients' => $response['ingredients'],
                'capsules' => $response['capsules'],
                'metadata' => $response['metadata'] ?? array(),
                'files' => $response['files'] ?? array(),
                'processing_stats' => $response['processing_stats'] ?? array(),
                'processing_method' => 'custom',
                'server_processed' => true,
                'server_version' => '2.0'
            );
        }
        
        // CRITICAL: No fallback processing - server v2.0 required
        AIOX_Logger::log('Custom Client: ❌ SECURITY ERROR - Server response missing structured data', AIOX_Logger::ERROR);
        AIOX_Logger::log('Custom Client: Response keys: ' . implode(', ', array_keys($response)), AIOX_Logger::ERROR);
        
        throw new Exception(
            'SERVER-ONLY MODE: Server must return structured data (ingredients, capsules, metadata). ' .
            'Please ensure: 1) Server plugin is updated to v2.0, 2) Server-side processing is enabled, ' .
            '3) Server generators are properly installed. No client-side fallback available in this version.'
        );
    }

    /**
     * Make API request to Custom endpoint
     * 
     * Sends raw data to server endpoint. Server is responsible for all processing.
     * 
     * @param array $data Raw data (content, title, excerpt, options)
     * @return array Response from server
     * @throws Exception On API errors
     */
    private function make_api_request($data) {
        if (!function_exists('wp_remote_post')) {
            throw new Exception('WordPress HTTP functions not available');
        }
        
        // Ensure endpoint URL is correct
        $endpoint_url = rtrim($this->endpoint, '/');
        if (!preg_match('/\/wp-json\/aiox-server\/v1\/ai\/process$/', $endpoint_url)) {
            if (strpos($endpoint_url, '/wp-json') === false) {
                $endpoint_url .= '/wp-json/aiox-server/v1/ai/process';
            }
        }
        
        $site_url = home_url();
        $site_domain = parse_url($site_url, PHP_URL_HOST);
        
        AIOX_Logger::log('Custom Client: Sending request to: ' . $endpoint_url, AIOX_Logger::INFO);
        
        $response = wp_remote_post($endpoint_url, array(
            'timeout' => 120, // Increased timeout for server processing
            'headers' => array(
                'Authorization' => 'Bearer ' . $this->api_key,
                'Content-Type' => 'application/json',
                'User-Agent' => 'WordPress/' . get_bloginfo('version') . '; ' . $site_url,
                'Origin' => $site_url,
                'X-Origin-Domain' => $site_domain,
                'X-API-Key' => $this->api_key,
                'X-Client-Version' => '2.0-hardened'
            ),
            'body' => json_encode($data),
            'sslverify' => true,
            'httpversion' => '1.1'
         ));
        
        if (is_wp_error($response)) {
            $error_message = $response->get_error_message();
            
            AIOX_Logger::log('Custom Client: API request failed - ' . $error_message, AIOX_Logger::ERROR);
            
            if (strpos($error_message, 'timeout') !== false || strpos($error_message, 'timed out') !== false) {
                throw new Exception('Custom API request timed out. Server may be processing a large request. Please try again.');
            }
            if (strpos($error_message, 'cURL error') !== false) {
                throw new Exception('Network connection error: ' . $error_message);
            }
            throw new Exception('Custom API request failed: ' . $error_message);
        }
        
        $status_code = wp_remote_retrieve_response_code($response);
        $body = wp_remote_retrieve_body($response);
        
        AIOX_Logger::log('Custom Client: Response status code: ' . $status_code, AIOX_Logger::INFO);
        
        if ($status_code !== 200) {
            $error_data = json_decode($body, true);
            $error_message = isset($error_data['message']) ? $error_data['message'] : 
                            (isset($error_data['error']['message']) ? $error_data['error']['message'] : 'Unknown error');
            $error_code = isset($error_data['code']) ? $error_data['code'] : '';
            
            AIOX_Logger::log("Custom Client: Error response - Code: {$status_code}, Message: {$error_message}", AIOX_Logger::ERROR);
            
            if ($status_code === 503) {
                throw new Exception("Custom API temporarily unavailable. Please wait and try again.");
            } elseif ($status_code === 429) {
                // Token/rate limit exceeded
                if (strpos($error_message, 'token') !== false || $error_code === 'limit_exceeded') {
                    throw new Exception("⚠️ Token Limit Reached: {$error_message}");
                }
                throw new Exception("Rate limit exceeded. Please wait before trying again.");
            } elseif ($status_code === 402) {
                // Insufficient tokens (payment required)
                throw new Exception("⚠️ Insufficient Tokens: {$error_message}");
            } elseif ($status_code === 401) {
                throw new Exception("Authentication failed. Please check your API key.");
            } elseif ($status_code === 403) {
                throw new Exception("Access denied: {$error_message}");
            } elseif ($status_code === 400) {
                throw new Exception("Request error: {$error_message}");
            }
            
            throw new Exception("API error ({$status_code}): {$error_message}");
        }
        
        $response_data = json_decode($body, true);
        if (json_last_error() !== JSON_ERROR_NONE) {
            AIOX_Logger::log('Custom Client: Invalid JSON response from server', AIOX_Logger::ERROR);
            throw new Exception('Invalid JSON response from Custom API');
        }
        
        return $response_data;
    }

    /**
     * Test connection to custom API endpoint
     * 
     * Validates configuration without making API calls.
     * 
     * @return array Connection status
     */
    public function test_connection() {
        if (empty($this->api_key)) {
            return array(
                'valid' => false,
                'message' => __('Custom API key not configured', 'aiox-publisher-suite-pro'),
                'action_required' => 'configure_api_key',
                'provider' => 'custom'
            );
        }
        
        if (empty($this->endpoint)) {
            return array(
                'valid' => false,
                'message' => __('Custom endpoint not configured', 'aiox-publisher-suite-pro'),
                'action_required' => 'configure_endpoint',
                'provider' => 'custom'
            );
        }
        
        if (!filter_var($this->endpoint, FILTER_VALIDATE_URL)) {
            return array(
                'valid' => false,
                'message' => __('Custom endpoint is not a valid URL', 'aiox-publisher-suite-pro'),
                'action_required' => 'fix_endpoint',
                'provider' => 'custom'
            );
        }
        
        return array(
            'valid' => true,
            'message' => __('Custom API configured (SERVER-ONLY MODE v2.0)', 'aiox-publisher-suite-pro'),
            'provider' => 'custom',
            'mode' => 'server_only',
            'version' => '2.0-hardened',
            'connection_valid' => true,
            'connection_message' => __('Configured - Requires Server v2.0', 'aiox-publisher-suite-pro')
        );
    }
    
    /**
     * @deprecated REMOVED - All Q&A generation is server-side
     */
    public function generate_qa_pairs($content, $title = '', $count = 5) {
        throw new Exception('Client-side Q&A generation removed. Server-side only in v2.0.');
    }

    /**
     * @deprecated REMOVED - All topic extraction is server-side
     */
    public function extract_topics($content, $title = '') {
        throw new Exception('Client-side topic extraction removed. Server-side only in v2.0.');
    }
    
    /**
     * @deprecated REMOVED - All prompt building is server-side
     * 
     * This method previously built prompts on the client side, exposing
     * valuable prompt engineering IP. It has been completely removed
     * in v2.0 to protect intellectual property.
     */
    private function build_combined_prompt($content, $title, $excerpt, $qa_count) {
        throw new Exception('Prompt building is now server-side only. This method should never be called.');
    }
}