<?php
/**
 * AIOX Generator Basic Files Extension
 * 
 * Creates basic AIOX files on activation for Version 2.0.1+
 */

if (!defined('ABSPATH')) {
    exit;
}

class AIOX_Generator_Basic_Files {
    
    /**
     * Create basic AIOX files on activation
     */
    public static function create_basic_aiox_files() {
        $well_known_dir = ABSPATH . '.well-known/';
        
        // Ensure .well-known directory exists
        if (!file_exists($well_known_dir)) {
            wp_mkdir_p($well_known_dir);
        }
        
        // Create basic manifest file
        self::create_basic_manifest($well_known_dir);
        
        // Create basic TDM-REP file
        self::create_basic_tdmrep($well_known_dir);
        
        // Create basic public key
        self::create_basic_pubkey($well_known_dir);
        
        // Create basic delta feed
        self::create_basic_delta($well_known_dir);
        
        // Create basic AIOX pack
        self::create_basic_pack($well_known_dir);
        
        // Create basic proof file
        self::create_basic_proof($well_known_dir);
        
        // Create basic ingredients file
        self::create_basic_ingredients($well_known_dir);
    }
    
    /**
     * Create basic manifest file
     */
    private static function create_basic_manifest($well_known_dir) {
        $manifest = array(
            'version' => '1.0',
            'publisher' => array(
                'name' => get_bloginfo('name'),
                'url' => home_url(),
                'description' => get_bloginfo('description')
            ),
            'endpoints' => array(
                'manifest' => home_url('.well-known/aio.json'),
                'delta' => home_url('.well-known/aiox.ndjson'),
                'tdm_rep' => home_url('.well-known/tdmrep.json'),
                'public_key' => home_url('.well-known/aiox-pubkey.pem')
            ),
            'content_types' => array('post', 'page'),
            'processing_method' => 'ai_enhanced',
            'last_updated' => current_time('c'),
            'total_content' => 0,
            'status' => 'ready',
            'note' => 'Basic AIOX manifest created on activation. Content will be added when processed.'
        );
        
        file_put_contents($well_known_dir . 'aio.json', json_encode($manifest, JSON_PRETTY_PRINT | JSON_UNESCAPED_SLASHES));
    }
    
    /**
     * Create basic TDM-REP file
     */
    private static function create_basic_tdmrep($well_known_dir) {
        $home_url = home_url();
        
        // Get settings - use defaults for initial creation
        $allow_training = get_option('aiox_tdm_training_allowed', false);
        $require_citation = get_option('aiox_tdm_citation_required', true);
        $attribution_name = get_option('aiox_required_attribution', get_bloginfo('name'));
        
        $tdmrep = array(
            'version' => '1.0',
            'policy' => array(
                'default' => $allow_training ? 'allow' : 'deny',
                'training' => (bool) $allow_training,
                'citation' => (bool) $require_citation
            ),
            'rules' => array(
                array(
                    'path' => '/',
                    'permission' => $allow_training ? 'allow' : 'deny',
                    'purpose' => array('training', 'analysis')
                )
            ),
            'contact' => array(
                'email' => get_option('admin_email'),
                'url' => $home_url . '/contact'
            ),
            'attribution' => array(
                'required' => (bool) $require_citation,
                'name' => $attribution_name,
                'url' => $home_url
            ),
            'generated_at' => current_time('c')
        );
        
        file_put_contents($well_known_dir . 'tdmrep.json', json_encode($tdmrep, JSON_PRETTY_PRINT | JSON_UNESCAPED_SLASHES));
    }
    
    /**
     * Create basic public key
     */
    private static function create_basic_pubkey($well_known_dir) {
        // Create a placeholder public key (in production, this would be the actual Ed25519 public key)
        $pubkey_content = "# AIOX Ed25519 Public Key\n";
        $pubkey_content .= "# Generated: " . current_time('c') . "\n";
        $pubkey_content .= "# This is a placeholder - actual key will be generated when crypto system is used\n";
        $pubkey_content .= "# Version: 2.0.1+\n";
        
        file_put_contents($well_known_dir . 'aiox-pubkey.pem', $pubkey_content);
    }
    
    /**
     * Create basic delta feed
     */
    private static function create_basic_delta($well_known_dir) {
        $delta_entry = array(
            'timestamp' => current_time('c'),
            'action' => 'system_initialized',
            'content_type' => 'system',
            'id' => 'aiox_init_' . time(),
            'changes' => array(
                'status' => 'AIOX system initialized',
                'version' => '2.0.1',
                'processing_mode' => 'ai_only'
            ),
            'note' => 'AIOX system ready for manual content processing'
        );
        
        file_put_contents($well_known_dir . 'aiox.ndjson', json_encode($delta_entry) . "\n");
    }
    
    /**
     * Create basic AIOX pack
     */
    private static function create_basic_pack($well_known_dir) {
        $pack_data = array(
            'version' => '2.0.1',
            'created' => current_time('c'),
            'content_count' => 0,
            'processing_method' => 'ai_only',
            'status' => 'initialized',
            'note' => 'AIOX pack file created on activation - content will be added when processed'
        );
        
        // Create a simple binary-like format (JSON for now, can be enhanced later)
        file_put_contents($well_known_dir . 'aiox.aiox', json_encode($pack_data));
    }
    
    /**
     * Create basic proof file
     */
    private static function create_basic_proof($well_known_dir) {
        $proof = array(
            'version' => '1.0',
            'created' => current_time('c'),
            'publisher' => array(
                'name' => get_bloginfo('name'),
                'url' => home_url()
            ),
            'signatures' => array(),
            'verification' => array(
                'method' => 'Ed25519',
                'public_key_url' => home_url('.well-known/aiox-pubkey.pem')
            ),
            'content_count' => 0,
            'status' => 'initialized',
            'note' => 'Proof file created on activation - signatures will be added when content is processed'
        );
        
        file_put_contents($well_known_dir . 'aiox-proof.json', json_encode($proof, JSON_PRETTY_PRINT | JSON_UNESCAPED_SLASHES));
    }
    
    /**
     * Create basic ingredients file
     */
    private static function create_basic_ingredients($well_known_dir) {
        $ingredients = array(
            'version' => '1.0',
            'created' => current_time('c'),
            'publisher' => array(
                'name' => get_bloginfo('name'),
                'url' => home_url()
            ),
            'processing_method' => 'ai_enhanced',
            'total_content' => 0,
            'ingredients_summary' => array(
                'titles' => 0,
                'headings' => 0,
                'paragraphs' => 0,
                'lists' => 0,
                'quotes' => 0,
                'images' => 0,
                'links' => 0,
                'qa_pairs' => 0
            ),
            'content' => array(),
            'status' => 'initialized',
            'note' => 'Ingredients file created on activation - content will be added when processed'
        );
        
        file_put_contents($well_known_dir . 'ingredients.json', json_encode($ingredients, JSON_PRETTY_PRINT | JSON_UNESCAPED_SLASHES));
    }
    
    /**
     * Check if basic files exist
     */
    public static function basic_files_exist() {
        $well_known_dir = ABSPATH . '.well-known/';
        $files = array('aio.json', 'tdmrep.json', 'aiox-pubkey.pem', 'aiox.ndjson', 'aiox.aiox', 'aiox-proof.json', 'ingredients.json');
        
        foreach ($files as $file) {
            if (!file_exists($well_known_dir . $file)) {
                return false;
            }
        }
        
        return true;
    }
    
    /**
     * Get basic files status
     */
    public static function get_basic_files_status() {
        $well_known_dir = ABSPATH . '.well-known/';
        $files = array(
            'aio.json' => 'AIOX Manifest',
            'tdmrep.json' => 'TDM-REP File',
            'aiox-pubkey.pem' => 'Public Key',
            'aiox.ndjson' => 'Delta Feed',
            'aiox.aiox' => 'AIOX Pack',
            'aiox-proof.json' => 'Proof File',
            'ingredients.json' => 'Ingredients File'
        );
        
        $status = array();
        
        foreach ($files as $filename => $description) {
            $filepath = $well_known_dir . $filename;
            $status[$filename] = array(
                'name' => $description,
                'exists' => file_exists($filepath),
                'size' => file_exists($filepath) ? filesize($filepath) : 0,
                'url' => home_url('.well-known/' . $filename),
                'modified' => file_exists($filepath) ? filemtime($filepath) : 0
            );
        }
        
        return $status;
    }
}
