<?php
/**
 * AIOX Generator Class
 * 
 * Handles content processing and Q&A generation
 */

if (!defined('ABSPATH')) {
    exit;
}

class AIOX_Generator {
    
    /**
     * Process a post and generate Q&A content
     */
    public function process_post($post_id) {
        $post = get_post($post_id);
        
        if (!$post || !AIOX_Utils::should_process_post_type($post->post_type)) {
            return false;
        }
        
        AIOX_Logger::info('Starting post processing', array(
            'post_id' => $post_id,
            'post_title' => $post->post_title,
            'post_type' => $post->post_type
        ));
        
        try {
            // Extract content (legacy)
            $content = $this->prepare_content($post);
            
            // Generate enhanced metadata using heuristic classification
            $metadata = AIOX_Metadata::classify_content($post);
            AIOX_Metadata::save_post_metadata($post_id, $metadata);
            
            // Extract ingredients using the new system
            $ingredients = AIOX_Ingredients::extract_ingredients($post);
            AIOX_Ingredients::save_post_ingredients($post_id, $ingredients);
            
            // Generate Q&A pairs (legacy compatibility)
            $qa_pairs = $this->generate_qa_pairs($content, $post_id);
            
            // Save Q&A data (legacy)
            $this->save_qa_data($post_id, $qa_pairs);
            
            // Update processing status
            $this->update_processing_status($post_id, 'completed');
            
            AIOX_Logger::info('Post processing completed with ingredients system', array(
                'post_id' => $post_id,
                'qa_count' => count($qa_pairs),
                'ingredients_count' => count($ingredients),
                'metadata_fields' => count($metadata)
            ));
            
            return true;
            
        } catch (Exception $e) {
            AIOX_Logger::error('Post processing failed', array(
                'post_id' => $post_id,
                'error' => $e->getMessage()
            ));
            
            $this->update_processing_status($post_id, 'failed');
            return false;
        }
    }
    
    /**
     * Prepare content for processing
     */
    private function prepare_content($post) {
        // Get post content
        $content = $post->post_content;
        
        // Apply WordPress filters
        $content = apply_filters('the_content', $content);
        
        // Strip HTML tags but preserve structure
        $content = wp_strip_all_tags($content, true);
        
        // Clean up whitespace
        $content = preg_replace('/\s+/', ' ', $content);
        $content = trim($content);
        
        // Prepare content structure
        $prepared_content = array(
            'title' => $post->post_title,
            'content' => $content,
            'excerpt' => $post->post_excerpt ?: wp_trim_words($content, 55),
            'url' => get_permalink($post->ID),
            'word_count' => str_word_count($content),
            'headings' => $this->extract_headings($post->post_content),
            'meta' => array(
                'post_id' => $post->ID,
                'post_type' => $post->post_type,
                'post_date' => $post->post_date,
                'post_author' => get_the_author_meta('display_name', $post->post_author)
            )
        );
        
        return $prepared_content;
    }
    
    /**
     * Generate Q&A pairs from content
     */
    private function generate_qa_pairs($content, $post_id) {
        if (AIOX_Utils::is_pro_mode()) {
            return $this->generate_qa_with_ai($content, $post_id);
        } else {
            return $this->generate_qa_basic($content, $post_id);
        }
    }
    
    /**
     * Generate Q&A pairs using AI (Pro mode)
     */
    private function generate_qa_with_ai($content, $post_id) {
        $endpoint = AIOX_Utils::get_ai_endpoint();
        $license_key = AIOX_Utils::get_option('license_key');
        
        if (empty($license_key)) {
            AIOX_Logger::warning('AI processing requested but no license key available', array('post_id' => $post_id));
            return $this->generate_qa_basic($content, $post_id);
        }
        
        $request_data = array(
            'action' => 'generate_qa',
            'content' => array(
                'title' => $content['title'],
                'text' => $content['content'],
                'excerpt' => $content['excerpt'],
                'headings' => $content['headings']
            ),
            'options' => array(
                'max_pairs' => 10,
                'min_words_per_answer' => 20,
                'include_context' => true
            )
        );
        
        $response = wp_remote_post($endpoint, array(
            'timeout' => 30,
            'headers' => array(
                'Authorization' => 'Bearer ' . $license_key,
                'Content-Type' => 'application/json',
                'User-Agent' => 'AIOX-Publisher-Suite-Pro/' . AIOX_VERSION
            ),
            'body' => json_encode($request_data)
        ));
        
        if (is_wp_error($response)) {
            AIOX_Logger::error('AI API request failed', array(
                'post_id' => $post_id,
                'error' => $response->get_error_message()
            ));
            return $this->generate_qa_basic($content, $post_id);
        }
        
        $response_code = wp_remote_retrieve_response_code($response);
        $response_body = wp_remote_retrieve_body($response);
        
        if ($response_code !== 200) {
            AIOX_Logger::error('AI API returned error', array(
                'post_id' => $post_id,
                'response_code' => $response_code,
                'response_body' => $response_body
            ));
            return $this->generate_qa_basic($content, $post_id);
        }
        
        $ai_response = json_decode($response_body, true);
        
        if (!$ai_response || !isset($ai_response['qa_pairs'])) {
            AIOX_Logger::error('Invalid AI API response', array(
                'post_id' => $post_id,
                'response' => $response_body
            ));
            return $this->generate_qa_basic($content, $post_id);
        }
        
        AIOX_Logger::info('AI processing successful', array(
            'post_id' => $post_id,
            'qa_count' => count($ai_response['qa_pairs'])
        ));
        
        return $this->format_qa_pairs($ai_response['qa_pairs'], 'ai');
    }
    
    /**
     * Generate Q&A pairs using basic extraction (Free mode)
     */
    private function generate_qa_basic($content, $post_id) {
        $qa_pairs = array();
        
        // Extract Q&A from headings
        $qa_pairs = array_merge($qa_pairs, $this->extract_qa_from_headings($content));
        
        // Extract Q&A from content patterns
        $qa_pairs = array_merge($qa_pairs, $this->extract_qa_from_patterns($content));
        
        // Generate basic Q&A about the content
        $qa_pairs = array_merge($qa_pairs, $this->generate_basic_qa($content));
        
        AIOX_Logger::info('Basic processing completed', array(
            'post_id' => $post_id,
            'qa_count' => count($qa_pairs)
        ));
        
        return $this->format_qa_pairs($qa_pairs, 'basic');
    }
    
    /**
     * Extract headings from content
     */
    private function extract_headings($content) {
        $headings = array();
        
        // Match H1-H6 tags
        preg_match_all('/<h([1-6])[^>]*>(.*?)<\/h[1-6]>/i', $content, $matches, PREG_SET_ORDER);
        
        foreach ($matches as $match) {
            $level = intval($match[1]);
            $text = wp_strip_all_tags($match[2]);
            $text = trim($text);
            
            if (!empty($text)) {
                $headings[] = array(
                    'level' => $level,
                    'text' => $text
                );
            }
        }
        
        return $headings;
    }
    
    /**
     * Extract Q&A from headings
     */
    private function extract_qa_from_headings($content) {
        $qa_pairs = array();
        
        foreach ($content['headings'] as $heading) {
            $text = $heading['text'];
            
            // Check if heading is a question
            if (preg_match('/\?$/', $text)) {
                $qa_pairs[] = array(
                    'question' => $text,
                    'answer' => 'This topic is covered in the article: ' . $content['title'],
                    'source' => 'heading',
                    'confidence' => 0.7
                );
            }
            
            // Generate questions from headings
            if ($heading['level'] <= 3) {
                $qa_pairs[] = array(
                    'question' => 'What does this article say about ' . strtolower($text) . '?',
                    'answer' => 'The article discusses ' . strtolower($text) . ' in detail. ' . $content['excerpt'],
                    'source' => 'heading_generated',
                    'confidence' => 0.6
                );
            }
        }
        
        return $qa_pairs;
    }
    
    /**
     * Extract Q&A from content patterns
     */
    private function extract_qa_from_patterns($content) {
        $qa_pairs = array();
        $text = $content['content'];
        
        // Pattern 1: Question followed by answer
        preg_match_all('/([^.!?]*\?)\s*([^.!?]*[.!])/i', $text, $matches, PREG_SET_ORDER);
        
        foreach ($matches as $match) {
            $question = trim($match[1]);
            $answer = trim($match[2]);
            
            if (strlen($question) > 10 && strlen($answer) > 20) {
                $qa_pairs[] = array(
                    'question' => $question,
                    'answer' => $answer,
                    'source' => 'pattern_qa',
                    'confidence' => 0.8
                );
            }
        }
        
        // Pattern 2: "What is..." definitions
        preg_match_all('/([^.!?]*(?:what is|what are)[^.!?]*\?)\s*([^.!?]*[.!])/i', $text, $matches, PREG_SET_ORDER);
        
        foreach ($matches as $match) {
            $question = trim($match[1]);
            $answer = trim($match[2]);
            
            if (strlen($answer) > 15) {
                $qa_pairs[] = array(
                    'question' => $question,
                    'answer' => $answer,
                    'source' => 'definition',
                    'confidence' => 0.9
                );
            }
        }
        
        return $qa_pairs;
    }
    
    /**
     * Generate basic Q&A about the content
     */
    private function generate_basic_qa($content) {
        $qa_pairs = array();
        
        // Basic questions about the article
        $qa_pairs[] = array(
            'question' => 'What is this article about?',
            'answer' => $content['excerpt'],
            'source' => 'generated_basic',
            'confidence' => 0.9
        );
        
        $qa_pairs[] = array(
            'question' => 'What is the title of this article?',
            'answer' => $content['title'],
            'source' => 'generated_basic',
            'confidence' => 1.0
        );
        
        if (!empty($content['meta']['post_author'])) {
            $qa_pairs[] = array(
                'question' => 'Who wrote this article?',
                'answer' => 'This article was written by ' . $content['meta']['post_author'] . '.',
                'source' => 'generated_basic',
                'confidence' => 1.0
            );
        }
        
        if ($content['word_count'] > 0) {
            $qa_pairs[] = array(
                'question' => 'How long is this article?',
                'answer' => 'This article contains approximately ' . $content['word_count'] . ' words.',
                'source' => 'generated_basic',
                'confidence' => 1.0
            );
        }
        
        return $qa_pairs;
    }
    
    /**
     * Format Q&A pairs
     */
    private function format_qa_pairs($qa_pairs, $method) {
        $formatted = array();
        
        foreach ($qa_pairs as $index => $pair) {
            $formatted[] = array(
                'id' => $index + 1,
                'question' => $this->clean_text($pair['question']),
                'answer' => $this->clean_text($pair['answer']),
                'source' => $pair['source'] ?? $method,
                'confidence' => $pair['confidence'] ?? 0.5,
                'method' => $method,
                'generated_at' => current_time('c')
            );
        }
        
        return $formatted;
    }
    
    /**
     * Clean and normalize text
     */
    private function clean_text($text) {
        // Remove extra whitespace
        $text = preg_replace('/\s+/', ' ', $text);
        
        // Trim
        $text = trim($text);
        
        // Ensure proper punctuation
        if (!empty($text) && !preg_match('/[.!?]$/', $text)) {
            $text .= '.';
        }
        
        return $text;
    }
    
    /**
     * Save Q&A data to database
     */
    private function save_qa_data($post_id, $qa_pairs) {
        // Save as post meta
        update_post_meta($post_id, '_aiox_qa_pairs', $qa_pairs);
        update_post_meta($post_id, '_aiox_processed_at', current_time('mysql'));
        update_post_meta($post_id, '_aiox_processed', '1'); // Mark as processed for manifest inclusion
        update_post_meta($post_id, '_aiox_qa_count', count($qa_pairs));
        update_post_meta($post_id, '_aiox_method', AIOX_Utils::is_pro_mode() ? 'ai' : 'basic');
        
        // Update global statistics
        $total_processed = AIOX_Utils::get_option('total_processed', 0);
        AIOX_Utils::update_option('total_processed', $total_processed + 1);
        
        $total_qa_pairs = AIOX_Utils::get_option('total_qa_pairs', 0);
        AIOX_Utils::update_option('total_qa_pairs', $total_qa_pairs + count($qa_pairs));
        
        // Generate AI info page
        $this->generate_ai_info_page($post_id, $qa_pairs);
    }
    
    /**
     * Update processing status in queue
     */
    private function update_processing_status($post_id, $status) {
        global $wpdb;
        
        $table_name = $wpdb->prefix . 'aiox_queue';
        $wpdb->update(
            $table_name,
            array(
                'status' => $status,
                'updated_at' => current_time('mysql')
            ),
            array('post_id' => $post_id),
            array('%s', '%s'),
            array('%d')
        );
    }
    
    /**
     * Get Q&A data for a post
     */
    public function get_post_qa($post_id) {
        $qa_pairs = get_post_meta($post_id, '_aiox_qa_pairs', true);
        
        if (empty($qa_pairs) || !is_array($qa_pairs)) {
            return array();
        }
        
        return $qa_pairs;
    }
    
    /**
     * Check if post has been processed
     */
    public function is_post_processed($post_id) {
        $processed_at = get_post_meta($post_id, '_aiox_processed_at', true);
        return !empty($processed_at);
    }
    
    /**
     * Generate AI info page for a post
     */
    private function generate_ai_info_page($post_id, $qa_pairs) {
        $post = get_post($post_id);
        if (!$post) {
            AIOX_Logger::error('AI info generation failed: Post not found', array('post_id' => $post_id));
            return false;
        }
        
        // Create well-known directory if it doesn't exist
        $well_known_dir = ABSPATH . '.well-known';
        
        AIOX_Logger::info('AI info generation started', array(
            'post_id' => $post_id,
            'well_known_dir' => $well_known_dir,
            'dir_exists' => file_exists($well_known_dir),
            'abspath' => ABSPATH,
            'abspath_writable' => is_writable(ABSPATH)
        ));
        
        if (!file_exists($well_known_dir)) {
            $mkdir_result = wp_mkdir_p($well_known_dir);
            AIOX_Logger::info('Creating .well-known directory', array(
                'result' => $mkdir_result,
                'dir_exists_after' => file_exists($well_known_dir),
                'permissions' => is_writable(ABSPATH) ? 'writable' : 'not writable'
            ));
            
            if (!$mkdir_result || !file_exists($well_known_dir)) {
                AIOX_Logger::error('Failed to create .well-known directory', array(
                    'path' => $well_known_dir,
                    'parent_writable' => is_writable(ABSPATH),
                    'parent_exists' => file_exists(ABSPATH),
                    'mkdir_result' => $mkdir_result
                ));
                return false;
            }
        }
        
        // Check if directory is writable
        if (!is_writable($well_known_dir)) {
            AIOX_Logger::error('Well-known directory is not writable', array(
                'path' => $well_known_dir,
                'permissions' => file_exists($well_known_dir) ? substr(sprintf('%o', fileperms($well_known_dir)), -4) : 'does not exist'
            ));
            return false;
        }
        
        // Generate AI info data
        $ai_info = array(
            'version' => '1.0',
            'generator' => 'AIOX Publisher Suite Pro',
            'generated_at' => current_time('c'),
            'post' => array(
                'id' => $post_id,
                'title' => $post->post_title,
                'url' => get_permalink($post_id),
                'published' => get_the_date('c', $post_id),
                'modified' => get_the_modified_date('c', $post_id),
                'author' => get_the_author_meta('display_name', $post->post_author),
                'excerpt' => get_the_excerpt($post_id),
                'word_count' => str_word_count(strip_tags($post->post_content))
            ),
            'processing' => array(
                'method' => AIOX_Utils::is_pro_mode() ? 'ai' : 'basic',
                'processed_at' => current_time('c'),
                'qa_count' => count($qa_pairs),
                'confidence_avg' => $this->calculate_average_confidence($qa_pairs)
            ),
            'qa_pairs' => array()
        );
        
        // Add Q&A pairs
        foreach ($qa_pairs as $index => $qa) {
            $ai_info['qa_pairs'][] = array(
                'id' => $index + 1,
                'question' => $qa['question'],
                'answer' => $qa['answer'],
                'confidence' => $qa['confidence'],
                'source' => $qa['source'],
                'context' => isset($qa['context']) ? $qa['context'] : ''
            );
        }
        
        // Add metadata
        $ai_info['metadata'] = array(
            'site_name' => get_bloginfo('name'),
            'site_url' => home_url(),
            'language' => get_locale(),
            'categories' => wp_get_post_categories($post_id, array('fields' => 'names')),
            'tags' => wp_get_post_tags($post_id, array('fields' => 'names')),
            'post_type' => $post->post_type
        );
        
        // Write AI info file
        $ai_info_file = $well_known_dir . '/ai-info.json';
        $json_data = wp_json_encode($ai_info, JSON_PRETTY_PRINT | JSON_UNESCAPED_UNICODE);
        
        AIOX_Logger::info('Attempting to write AI info file', array(
            'file_path' => $ai_info_file,
            'json_length' => strlen($json_data),
            'directory_writable' => is_writable($well_known_dir),
            'file_exists_before' => file_exists($ai_info_file)
        ));
        
        // Clear any previous errors
        error_clear_last();
        
        $success = file_put_contents($ai_info_file, $json_data);
        $last_error = error_get_last();
        
        AIOX_Logger::info('File write attempt completed', array(
            'success' => $success,
            'bytes_written' => $success,
            'file_exists_after' => file_exists($ai_info_file),
            'file_size' => file_exists($ai_info_file) ? filesize($ai_info_file) : 0,
            'last_error' => $last_error
        ));
        
        if ($success && file_exists($ai_info_file)) {
            AIOX_Logger::info('AI info page generated successfully', array(
                'post_id' => $post_id,
                'file_path' => $ai_info_file,
                'file_size' => filesize($ai_info_file)
            ));
            
            // Also create a post-specific AI info file
            $post_ai_info_file = $well_known_dir . '/ai-info-' . $post_id . '.json';
            $post_success = file_put_contents($post_ai_info_file, $json_data);
            
            AIOX_Logger::info('Post-specific AI info file', array(
                'file_path' => $post_ai_info_file,
                'success' => $post_success,
                'file_exists' => file_exists($post_ai_info_file)
            ));
            
            return true;
        } else {
            AIOX_Logger::error('Failed to generate AI info page', array(
                'post_id' => $post_id,
                'file_path' => $ai_info_file,
                'success_result' => $success,
                'file_exists' => file_exists($ai_info_file),
                'directory_writable' => is_writable($well_known_dir),
                'last_error' => $last_error,
                'json_valid' => json_last_error() === JSON_ERROR_NONE
            ));
            
            return false;
        }
    }
    
    /**
     * Calculate average confidence from Q&A pairs
     */
    /**
     * Get count of processed posts
     */
    private function get_processed_posts_count() {
        global $wpdb;
        
        $count = $wpdb->get_var("
            SELECT COUNT(DISTINCT p.ID)
            FROM {$wpdb->posts} p
            INNER JOIN {$wpdb->postmeta} pm ON p.ID = pm.post_id
            WHERE p.post_status = 'publish'
            AND p.post_type IN ('post', 'page')
            AND pm.meta_key = '_aiox_qa_pairs'
            AND pm.meta_value IS NOT NULL
            AND pm.meta_value != ''
        ");
        
        return intval($count);
    }
    
    private function calculate_average_confidence($qa_pairs) {
        if (empty($qa_pairs)) {
            return 0;
        }
        
        $total_confidence = 0;
        foreach ($qa_pairs as $qa) {
            $total_confidence += $qa['confidence'];
        }
        
        return round($total_confidence / count($qa_pairs), 2);
    }
    
    /**
     * Generate master AI info page with all processed posts
     */
    public function generate_master_ai_info_page() {
        global $wpdb;
        
        AIOX_Logger::info('Master AI info generation started');
        
        // Get all processed posts
        $processed_posts = $wpdb->get_results(
            "SELECT post_id FROM {$wpdb->postmeta} WHERE meta_key = '_aiox_processed_at' ORDER BY meta_value DESC",
            ARRAY_A
        );
        
        AIOX_Logger::info('Found processed posts', array(
            'count' => count($processed_posts),
            'query' => "SELECT post_id FROM {$wpdb->postmeta} WHERE meta_key = '_aiox_processed_at'"
        ));
        
        // If no processed posts, try to process some posts first
        if (empty($processed_posts)) {
            AIOX_Logger::info('No processed posts found, attempting to process recent posts');
            
            // Get recent published posts
            $recent_posts = get_posts(array(
                'numberposts' => 3,
                'post_status' => 'publish',
                'post_type' => array('post', 'page')
            ));
            
            AIOX_Logger::info('Found recent posts to process', array('count' => count($recent_posts)));
            
            if (!empty($recent_posts)) {
                foreach ($recent_posts as $post) {
                    AIOX_Logger::info('Processing post for AI info', array('post_id' => $post->ID, 'title' => $post->post_title));
                    $this->process_post($post->ID);
                }
                
                // Re-query for processed posts
                $processed_posts = $wpdb->get_results(
                    "SELECT post_id FROM {$wpdb->postmeta} WHERE meta_key = '_aiox_processed_at' ORDER BY meta_value DESC",
                    ARRAY_A
                );
                
                AIOX_Logger::info('After processing, found posts', array('count' => count($processed_posts)));
            }
            
            if (empty($processed_posts)) {
                AIOX_Logger::error('No posts could be processed for AI info generation');
                return false;
            }
        }
        
        // Create well-known directory if it doesn't exist
        $well_known_dir = ABSPATH . '.well-known';
        if (!file_exists($well_known_dir)) {
            wp_mkdir_p($well_known_dir);
        }
        
        $master_ai_info = array(
            'version' => '1.0',
            'generator' => 'AIOX Publisher Suite Pro',
            'generated_at' => current_time('c'),
            'site' => array(
                'name' => get_bloginfo('name'),
                'url' => home_url(),
                'description' => get_bloginfo('description'),
                'language' => get_locale()
            ),
            'statistics' => $this->get_processing_stats(),
            'posts' => array()
        );
        
        // Add each processed post
        foreach ($processed_posts as $row) {
            $post_id = $row['post_id'];
            $post = get_post($post_id);
            
            if (!$post || $post->post_status !== 'publish') {
                continue;
            }
            
            $qa_pairs = $this->get_post_qa($post_id);
            
            $master_ai_info['posts'][] = array(
                'id' => $post_id,
                'title' => $post->post_title,
                'url' => get_permalink($post_id),
                'published' => get_the_date('c', $post_id),
                'modified' => get_the_modified_date('c', $post_id),
                'qa_count' => count($qa_pairs),
                'ai_info_url' => home_url('/.well-known/ai-info-' . $post_id . '.json')
            );
        }
        
        // Write master AI info file
        $master_file = $well_known_dir . '/ai-info-master.json';
        $json_data = wp_json_encode($master_ai_info, JSON_PRETTY_PRINT | JSON_UNESCAPED_UNICODE);
        
        AIOX_Logger::info('Attempting to write master AI info file', array(
            'file_path' => $master_file,
            'posts_count' => count($master_ai_info['posts']),
            'json_length' => strlen($json_data),
            'directory_writable' => is_writable($well_known_dir),
            'directory_exists' => file_exists($well_known_dir)
        ));
        
        // Clear any previous errors
        error_clear_last();
        
        $success = file_put_contents($master_file, $json_data);
        $last_error = error_get_last();
        
        AIOX_Logger::info('Master file write attempt completed', array(
            'success' => $success,
            'bytes_written' => $success,
            'file_exists_after' => file_exists($master_file),
            'file_size' => file_exists($master_file) ? filesize($master_file) : 0,
            'last_error' => $last_error
        ));
        
        if ($success && file_exists($master_file)) {
            AIOX_Logger::info('Master AI info page generated successfully', array(
                'file_path' => $master_file,
                'posts_count' => count($master_ai_info['posts']),
                'file_size' => filesize($master_file)
            ));
            return true;
        } else {
            AIOX_Logger::error('Failed to generate master AI info page', array(
                'file_path' => $master_file,
                'success_result' => $success,
                'file_exists' => file_exists($master_file),
                'directory_writable' => is_writable($well_known_dir),
                'last_error' => $last_error,
                'json_valid' => json_last_error() === JSON_ERROR_NONE
            ));
            return false;
        }
    }
    
    /**
     * Create AI Info Page (WordPress page, not JSON file)
     */
    public function create_ai_info_page($force = false) {
        $options = AIOX_Utils::get_all_options();
                // AI Info Page is always created - users can manage it manually if needed
        
        // Create the page content
        $title = 'AI Indexing Information';
        $home = home_url();
        $site_name = get_bloginfo('name');
        
        // Get site statistics
        $post_count = wp_count_posts()->publish;
        $page_count = wp_count_posts('page')->publish;
        $processed_count = $this->get_processed_posts_count();
        
        $body = "<p>This website is enabled for AI-friendly indexing using the <strong>AIOX Publisher Suite</strong>.</p>\n\n";
        
        $body .= "<h3>Site Statistics</h3>\n";
        $body .= "<ul>\n";
        $body .= "<li><strong>Published Posts:</strong> " . $post_count . "</li>\n";
        $body .= "<li><strong>Published Pages:</strong> " . $page_count . "</li>\n";
        $body .= "<li><strong>AI-Processed Content:</strong> " . $processed_count . "</li>\n";
        $body .= "</ul>\n\n";
        
        $body .= "<h3>Available Machine-Readable Resources</h3>\n";
        $body .= "<ul>\n";
        $body .= "<li><strong>AIOX Manifest:</strong> <a href=\"{$home}/.well-known/aio.json\" target=\"_blank\">aio.json</a></li>\n";
        $body .= "<li><strong>TDM-REP File:</strong> <a href=\"{$home}/.well-known/tdmrep.json\" target=\"_blank\">tdmrep.json</a></li>\n";
        $body .= "<li><strong>Public Key:</strong> <a href=\"{$home}/.well-known/aiox-pubkey.pem\" target=\"_blank\">aiox-pubkey.pem</a></li>\n";
        $body .= "<li><strong>Delta Feed:</strong> <a href=\"{$home}/.well-known/aiox.ndjson\" target=\"_blank\">aiox.ndjson</a></li>\n";
        $body .= "<li><strong>AIOX Pack:</strong> <a href=\"{$home}/.well-known/aiox.aiox\" target=\"_blank\">aiox.aiox</a></li>\n";
        $body .= "</ul>\n\n";
        
        $body .= "<h3>AI Optimization Features</h3>\n";
        $body .= "<ul>\n";
        $body .= "<li>Structured Q&A content extraction</li>\n";
        $body .= "<li>Cryptographic content verification</li>\n";
        $body .= "<li>Machine-readable metadata</li>\n";
        $body .= "<li>Incremental content updates</li>\n";
        $body .= "<li>Responsible AI practices compliance</li>\n";
        $body .= "</ul>\n\n";
        
        $body .= "<p><em>This site contains structured content optimized for AI systems and follows responsible AI practices.</em></p>\n\n";
        $body .= "<p><strong>Powered by AIOX Publisher Suite Pro</strong></p>";
        
        $post_data = array(
            'post_title' => $title,
            'post_content' => $body,
            'post_status' => 'publish',
            'post_type' => 'page',
            'post_name' => 'ai-indexing-information'
        );
        
        // Check if page already exists
        $post_id = intval(get_option('aiox_aiinfo_post_id', 0));
        $post = $post_id ? get_post($post_id) : null;
        
        AIOX_Logger::info('Creating/updating AI info page', array(
            'existing_post_id' => $post_id,
            'post_exists' => $post ? true : false,
            'post_status' => $post ? $post->post_status : 'none'
        ));
        
        if ($post && $post->post_type === 'page' && $post->post_status !== 'trash') {
            // Update existing page
            $post_data['ID'] = $post_id;
            $result = wp_update_post($post_data);
            
            if ($result && !is_wp_error($result)) {
                AIOX_Logger::info('AI info page updated successfully', array('post_id' => $post_id));
                return $post_id;
            } else {
                AIOX_Logger::error('Failed to update AI info page', array('error' => is_wp_error($result) ? $result->get_error_message() : 'Unknown error'));
                return false;
            }
        } else {
            // Create new page
            $new_id = wp_insert_post($post_data, true);
            
            if (!is_wp_error($new_id)) {
                update_option('aiox_aiinfo_post_id', $new_id);
                AIOX_Logger::info('AI info page created successfully', array('post_id' => $new_id));
                return $new_id;
            } else {
                AIOX_Logger::error('Failed to create AI info page', array('error' => $new_id->get_error_message()));
                return false;
            }
        }
    }
    
    /**
     * Generate a simple AI info file for testing
     */
    public function generate_simple_ai_info() {
        $well_known_dir = ABSPATH . '.well-known';
        
        // Ensure directory exists
        if (!file_exists($well_known_dir)) {
            wp_mkdir_p($well_known_dir);
        }
        
        // Create simple AI info
        $ai_info = array(
            'version' => '1.0',
            'generator' => 'AIOX Publisher Suite Pro',
            'generated_at' => current_time('c'),
            'site' => array(
                'name' => get_bloginfo('name'),
                'url' => home_url(),
                'description' => get_bloginfo('description')
            ),
            'status' => 'Generated successfully',
            'test' => true
        );
        
        $ai_info_file = $well_known_dir . '/ai-info.json';
        $json_data = wp_json_encode($ai_info, JSON_PRETTY_PRINT | JSON_UNESCAPED_UNICODE);
        
        AIOX_Logger::info('Generating simple AI info file', array(
            'file_path' => $ai_info_file,
            'data_length' => strlen($json_data)
        ));
        
        $success = file_put_contents($ai_info_file, $json_data);
        
        if ($success && file_exists($ai_info_file)) {
            AIOX_Logger::info('Simple AI info file created successfully', array(
                'file_size' => filesize($ai_info_file)
            ));
            return true;
        } else {
            AIOX_Logger::error('Failed to create simple AI info file');
            return false;
        }
    }
    
    /**
     * Generate main manifesto file
     * Alias for generate_master_ai_info_page() for compatibility
     */
    public function generate_manifesto_file() {
        return $this->generate_master_ai_info_page();
    }
    
    /**
     * Generate individual post manifesto file
     * Creates a .well-known/aiox-post-{post_id}.json file for a specific post
     */
    public function generate_post_file($post_id) {
        $post = get_post($post_id);
        
        if (!$post) {
            AIOX_Logger::error('Cannot generate post file: post not found', array('post_id' => $post_id));
            return false;
        }
        
        $well_known_dir = ABSPATH . '.well-known';
        $posts_dir = $well_known_dir . '/posts';
        
        // Ensure directories exist
        if (!file_exists($well_known_dir)) {
            if (!wp_mkdir_p($well_known_dir)) {
                AIOX_Logger::error('Failed to create .well-known directory', array('path' => $well_known_dir));
                return false;
            }
        }
        
        if (!file_exists($posts_dir)) {
            if (!wp_mkdir_p($posts_dir)) {
                AIOX_Logger::error('Failed to create posts directory', array('path' => $posts_dir));
                return false;
            }
        }
        
        // Get post ingredients
        $ingredients = array();
        if (class_exists('AIOX_Ingredients')) {
            $ingredients = AIOX_Ingredients::get_post_ingredients($post_id);
        }
        
        // Get capsule data for this post (returns single capsule or null)
        $capsule_data = array();
        if (class_exists('AIOX_Capsule_Manager')) {
            $capsule_manager = new AIOX_Capsule_Manager();
            $capsule = $capsule_manager->get_capsule_by_post_id($post_id);
            
            if ($capsule) {
                // Extract capsule data safely
                $capsule_data = array(
                    'id' => $capsule->get_id(),
                    'status' => $capsule->get_status(),
                    'type' => $capsule->get_type()
                );
                
                // Try to get metadata if method exists
                if (method_exists($capsule, 'get_all_metadata')) {
                    $capsule_data['metadata'] = $capsule->get_all_metadata();
                }
                
                // Try to get content if method exists
                if (method_exists($capsule, 'get_all_content')) {
                    $capsule_data['content'] = $capsule->get_all_content();
                }
            }
        }
        
        // Get metadata
        $metadata = array();
        if (class_exists('AIOX_Metadata')) {
            $metadata = AIOX_Metadata::get_post_metadata($post_id);
        }
        
        // Build post manifesto data
        $post_data = array(
            'version' => '2.0',
            'generator' => 'AIOX Publisher Suite',
            'generated_at' => current_time('c'),
            'post' => array(
                'id' => $post_id,
                'title' => $post->post_title,
                'url' => get_permalink($post_id),
                'published' => get_the_date('c', $post_id),
                'modified' => get_the_modified_date('c', $post_id),
                'type' => $post->post_type,
                'status' => $post->post_status
            ),
            'ingredients' => array(
                'count' => count($ingredients),
                'items' => $ingredients
            ),
            'capsule' => $capsule_data, // Single capsule, not array
            'metadata' => $metadata,
            'processing' => array(
                'server_processed' => (bool) get_post_meta($post_id, '_aiox_server_processed', true),
                'last_processed' => get_post_meta($post_id, '_aiox_last_processed', true),
                'comprehensive_processed' => (bool) get_post_meta($post_id, '_aiox_comprehensive_processed', true)
            )
        );
        
        // Write individual post file to posts directory (matching content_index detail_url)
        $post_file = $posts_dir . '/' . $post_id . '.json';
        $json_data = wp_json_encode($post_data, JSON_PRETTY_PRINT | JSON_UNESCAPED_UNICODE | JSON_UNESCAPED_SLASHES);
        
        AIOX_Logger::info('Generating post manifesto file', array(
            'post_id' => $post_id,
            'file_path' => $post_file,
            'ingredients_count' => count($ingredients),
            'has_capsule' => !empty($capsule_data)
        ));
        
        $success = file_put_contents($post_file, $json_data);
        
        if ($success && file_exists($post_file)) {
            AIOX_Logger::info('Post manifesto file generated successfully', array(
                'post_id' => $post_id,
                'file_path' => $post_file,
                'file_size' => filesize($post_file)
            ));
            return true;
        } else {
            AIOX_Logger::error('Failed to generate post manifesto file', array(
                'post_id' => $post_id,
                'file_path' => $post_file,
                'directory_writable' => is_writable($posts_dir)
            ));
            return false;
        }
    }
    
    /**
     * Get processing statistics
     */
    public function get_processing_stats() {
        return array(
            'total_processed' => AIOX_Utils::get_option('total_processed', 0),
            'total_qa_pairs' => AIOX_Utils::get_option('total_qa_pairs', 0),
            'last_processed' => AIOX_Utils::get_option('last_processed', 'Never'),
            'processing_method' => AIOX_Utils::is_pro_mode() ? 'AI-Powered' : 'Basic Extraction'
        );
    }
}