<?php
/**
 * AIOX Ingredients Class
 * 
 * Handles the new Ingredients System - breaking content into semantic building blocks
 */

if (!defined('ABSPATH')) {
    exit;
}

class AIOX_Ingredients {
    
    /**
     * Ingredient types and their extraction methods
     */
    private static $ingredient_types = array(
        'title' => 'Title',
        'heading' => 'Heading',
        'paragraph' => 'Paragraph',
        'summary' => 'Summary',
        'qa_pair' => 'Q&A Pair',
        'list_item' => 'List Item',
        'quote' => 'Quote',
        'metadata' => 'Metadata',
        'image_caption' => 'Image Caption',
        'link' => 'Link'
    );
    
    /**
     * Extract ingredients from a post
     */
    public static function extract_ingredients($post) {
        $ingredients = array();
        $content = $post->post_content;
        $title = $post->post_title;
        
        // Get post metadata
        $post_metadata = AIOX_Metadata::get_post_metadata($post->ID);
        $licensing_metadata = AIOX_Metadata::get_licensing_metadata($post->ID);
        
        // Extract title ingredient
        $ingredients[] = self::create_ingredient(
            'title',
            $title,
            array(
                'position' => 0,
                'importance' => 'high',
                'context' => 'main_title'
            ),
            $post_metadata,
            $licensing_metadata
        );
        
        // Extract headings
        $ingredients = array_merge($ingredients, self::extract_headings($content, $post_metadata, $licensing_metadata));
        
        // Extract paragraphs
        $ingredients = array_merge($ingredients, self::extract_paragraphs($content, $post_metadata, $licensing_metadata));
        
        // Extract Q&A pairs (legacy compatibility)
        $ingredients = array_merge($ingredients, self::extract_qa_pairs($content, $post_metadata, $licensing_metadata));
        
        // Extract lists
        $ingredients = array_merge($ingredients, self::extract_lists($content, $post_metadata, $licensing_metadata));
        
        // Extract quotes
        $ingredients = array_merge($ingredients, self::extract_quotes($content, $post_metadata, $licensing_metadata));
        
        // Extract image captions
        $ingredients = array_merge($ingredients, self::extract_image_captions($content, $post_metadata, $licensing_metadata));
        
        // Extract links
        $ingredients = array_merge($ingredients, self::extract_links($content, $post_metadata, $licensing_metadata));
        
        // Create summary ingredient
        $ingredients[] = self::create_summary_ingredient($post, $post_metadata, $licensing_metadata);
        
        // Create metadata ingredient
        $ingredients[] = self::create_metadata_ingredient($post, $post_metadata, $licensing_metadata);
        
        // Add ingredient IDs and timestamps
        foreach ($ingredients as $index => &$ingredient) {
            $ingredient['id'] = self::generate_ingredient_id($post->ID, $index, $ingredient['type']);
            $ingredient['created_at'] = current_time('c');
            $ingredient['post_id'] = $post->ID;
        }
        
        return $ingredients;
    }
    
    /**
     * Create a standardized ingredient structure
     */
    private static function create_ingredient($type, $content, $context = array(), $post_metadata = array(), $licensing_metadata = array()) {
        return array(
            'type' => $type,
            'content' => $content,
            'context' => $context,
            'metadata' => $post_metadata,
            'licensing' => $licensing_metadata,
            'word_count' => str_word_count(strip_tags($content)),
            'character_count' => strlen(strip_tags($content)),
            'language' => get_locale(),
            'encoding' => 'UTF-8'
        );
    }
    
    /**
     * Extract headings from content
     */
    private static function extract_headings($content, $post_metadata, $licensing_metadata) {
        $ingredients = array();
        
        // Match all heading tags
        if (preg_match_all('/<h([1-6])[^>]*>(.*?)<\/h[1-6]>/is', $content, $matches, PREG_SET_ORDER)) {
            foreach ($matches as $index => $match) {
                $level = intval($match[1]);
                $heading_text = wp_strip_all_tags($match[2]);
                
                if (!empty(trim($heading_text))) {
                    $ingredients[] = self::create_ingredient(
                        'heading',
                        $heading_text,
                        array(
                            'level' => $level,
                            'position' => $index + 1,
                            'importance' => $level <= 2 ? 'high' : ($level <= 4 ? 'medium' : 'low'),
                            'hierarchy' => 'h' . $level
                        ),
                        $post_metadata,
                        $licensing_metadata
                    );
                }
            }
        }
        
        return $ingredients;
    }
    
    /**
     * Extract paragraphs from content
     */
    private static function extract_paragraphs($content, $post_metadata, $licensing_metadata) {
        $ingredients = array();
        
        // Remove HTML tags but preserve paragraph structure
        $content = wp_strip_all_tags($content, true);
        
        // Split into paragraphs
        $paragraphs = preg_split('/\n\s*\n/', $content);
        
        foreach ($paragraphs as $index => $paragraph) {
            $paragraph = trim($paragraph);
            
            // Skip empty paragraphs or very short ones
            if (strlen($paragraph) < 50) {
                continue;
            }
            
            $word_count = str_word_count($paragraph);
            
            $ingredients[] = self::create_ingredient(
                'paragraph',
                $paragraph,
                array(
                    'position' => $index + 1,
                    'importance' => $word_count > 100 ? 'high' : ($word_count > 50 ? 'medium' : 'low'),
                    'is_introduction' => $index === 0,
                    'sentence_count' => substr_count($paragraph, '.') + substr_count($paragraph, '!') + substr_count($paragraph, '?')
                ),
                $post_metadata,
                $licensing_metadata
            );
        }
        
        return $ingredients;
    }
    
    /**
     * Extract Q&A pairs (legacy compatibility)
     */
    private static function extract_qa_pairs($content, $post_metadata, $licensing_metadata) {
        $ingredients = array();
        
        // Look for explicit Q&A patterns
        if (preg_match_all('/(?:Q(?:uestion)?[:.]?\s*)(.*?)(?:\n|\r\n?)(?:A(?:nswer)?[:.]?\s*)(.*?)(?=\n\s*Q|$)/is', $content, $matches, PREG_SET_ORDER)) {
            foreach ($matches as $index => $match) {
                $question = trim(wp_strip_all_tags($match[1]));
                $answer = trim(wp_strip_all_tags($match[2]));
                
                if (!empty($question) && !empty($answer)) {
                    $ingredients[] = self::create_ingredient(
                        'qa_pair',
                        json_encode(array(
                            'question' => $question,
                            'answer' => $answer
                        )),
                        array(
                            'position' => $index + 1,
                            'importance' => 'high',
                            'format' => 'explicit_qa',
                            'question_length' => strlen($question),
                            'answer_length' => strlen($answer)
                        ),
                        $post_metadata,
                        $licensing_metadata
                    );
                }
            }
        }
        
        // Generate implicit Q&A from headings (basic heuristic)
        if (preg_match_all('/<h([1-6])[^>]*>(.*?)<\/h[1-6]>/is', $content, $headings, PREG_SET_ORDER)) {
            foreach ($headings as $index => $heading) {
                $heading_text = wp_strip_all_tags($heading[2]);
                
                // Skip if heading is too short or doesn't look like a question topic
                if (strlen($heading_text) < 10) {
                    continue;
                }
                
                // Find the content following this heading
                $pattern = '/<h' . $heading[1] . '[^>]*>' . preg_quote($heading[2], '/') . '<\/h' . $heading[1] . '>(.*?)(?=<h[1-6]|$)/is';
                if (preg_match($pattern, $content, $section_match)) {
                    $section_content = wp_strip_all_tags($section_match[1]);
                    $section_content = trim($section_content);
                    
                    if (strlen($section_content) > 50) {
                        // Generate a question from the heading
                        $question = self::heading_to_question($heading_text);
                        
                        // Use first paragraph as answer
                        $paragraphs = preg_split('/\n\s*\n/', $section_content);
                        $answer = trim($paragraphs[0]);
                        
                        if (strlen($answer) > 30) {
                            $ingredients[] = self::create_ingredient(
                                'qa_pair',
                                json_encode(array(
                                    'question' => $question,
                                    'answer' => $answer
                                )),
                                array(
                                    'position' => $index + 1,
                                    'importance' => 'medium',
                                    'format' => 'generated_from_heading',
                                    'source_heading' => $heading_text,
                                    'question_length' => strlen($question),
                                    'answer_length' => strlen($answer)
                                ),
                                $post_metadata,
                                $licensing_metadata
                            );
                        }
                    }
                }
            }
        }
        
        return $ingredients;
    }
    
    /**
     * Convert heading to question format
     */
    private static function heading_to_question($heading) {
        $heading = trim($heading);
        
        // If already a question, return as-is
        if (substr($heading, -1) === '?') {
            return $heading;
        }
        
        // Common question patterns
        $question_starters = array(
            'how to' => 'How do you',
            'what is' => 'What is',
            'why' => 'Why',
            'when' => 'When',
            'where' => 'Where',
            'benefits' => 'What are the benefits of',
            'advantages' => 'What are the advantages of',
            'tips' => 'What are some tips for',
            'guide' => 'How do you'
        );
        
        $heading_lower = strtolower($heading);
        
        foreach ($question_starters as $pattern => $replacement) {
            if (strpos($heading_lower, $pattern) !== false) {
                return $replacement . ' ' . strtolower(str_replace($pattern, '', $heading_lower)) . '?';
            }
        }
        
        // Default: add "What is" or "How do you"
        if (preg_match('/\b(setup|configure|install|create|build|make)\b/i', $heading)) {
            return 'How do you ' . strtolower($heading) . '?';
        } else {
            return 'What is ' . strtolower($heading) . '?';
        }
    }
    
    /**
     * Extract lists from content
     */
    private static function extract_lists($content, $post_metadata, $licensing_metadata) {
        $ingredients = array();
        
        // Extract ordered and unordered lists
        if (preg_match_all('/<(ul|ol)[^>]*>(.*?)<\/\1>/is', $content, $lists, PREG_SET_ORDER)) {
            foreach ($lists as $index => $list) {
                $list_type = $list[1];
                $list_content = $list[2];
                
                // Extract list items
                if (preg_match_all('/<li[^>]*>(.*?)<\/li>/is', $list_content, $items)) {
                    foreach ($items[1] as $item_index => $item) {
                        $item_text = wp_strip_all_tags($item);
                        $item_text = trim($item_text);
                        
                        if (!empty($item_text)) {
                            $ingredients[] = self::create_ingredient(
                                'list_item',
                                $item_text,
                                array(
                                    'list_position' => $index + 1,
                                    'item_position' => $item_index + 1,
                                    'list_type' => $list_type,
                                    'importance' => 'medium',
                                    'is_numbered' => $list_type === 'ol'
                                ),
                                $post_metadata,
                                $licensing_metadata
                            );
                        }
                    }
                }
            }
        }
        
        return $ingredients;
    }
    
    /**
     * Extract quotes from content
     */
    private static function extract_quotes($content, $post_metadata, $licensing_metadata) {
        $ingredients = array();
        
        // Extract blockquotes
        if (preg_match_all('/<blockquote[^>]*>(.*?)<\/blockquote>/is', $content, $quotes, PREG_SET_ORDER)) {
            foreach ($quotes as $index => $quote) {
                $quote_text = wp_strip_all_tags($quote[1]);
                $quote_text = trim($quote_text);
                
                if (!empty($quote_text) && strlen($quote_text) > 20) {
                    $ingredients[] = self::create_ingredient(
                        'quote',
                        $quote_text,
                        array(
                            'position' => $index + 1,
                            'importance' => 'high',
                            'type' => 'blockquote',
                            'is_citation' => strpos($quote_text, '—') !== false || strpos($quote_text, '-') !== false
                        ),
                        $post_metadata,
                        $licensing_metadata
                    );
                }
            }
        }
        
        return $ingredients;
    }
    
    /**
     * Extract image captions
     */
    private static function extract_image_captions($content, $post_metadata, $licensing_metadata) {
        $ingredients = array();
        
        // Extract WordPress image captions
        if (preg_match_all('/\[caption[^\]]*\](.*?)\[\/caption\]/is', $content, $captions, PREG_SET_ORDER)) {
            foreach ($captions as $index => $caption) {
                // Extract the caption text (usually after the img tag)
                if (preg_match('/<img[^>]*>(.*?)$/is', $caption[1], $caption_match)) {
                    $caption_text = wp_strip_all_tags($caption_match[1]);
                    $caption_text = trim($caption_text);
                    
                    if (!empty($caption_text)) {
                        $ingredients[] = self::create_ingredient(
                            'image_caption',
                            $caption_text,
                            array(
                                'position' => $index + 1,
                                'importance' => 'medium',
                                'media_type' => 'image'
                            ),
                            $post_metadata,
                            $licensing_metadata
                        );
                    }
                }
            }
        }
        
        // Extract alt text from images
        if (preg_match_all('/<img[^>]*alt=["\']([^"\']*)["\'][^>]*>/is', $content, $alt_texts)) {
            foreach ($alt_texts[1] as $index => $alt_text) {
                $alt_text = trim($alt_text);
                
                if (!empty($alt_text) && strlen($alt_text) > 10) {
                    $ingredients[] = self::create_ingredient(
                        'image_caption',
                        $alt_text,
                        array(
                            'position' => $index + 1,
                            'importance' => 'low',
                            'media_type' => 'image',
                            'source' => 'alt_text'
                        ),
                        $post_metadata,
                        $licensing_metadata
                    );
                }
            }
        }
        
        return $ingredients;
    }
    
    /**
     * Extract links
     */
    private static function extract_links($content, $post_metadata, $licensing_metadata) {
        $ingredients = array();
        
        if (preg_match_all('/<a[^>]*href=["\']([^"\']*)["\'][^>]*>(.*?)<\/a>/is', $content, $links, PREG_SET_ORDER)) {
            foreach ($links as $index => $link) {
                $url = $link[1];
                $link_text = wp_strip_all_tags($link[2]);
                $link_text = trim($link_text);
                
                // Skip empty links or very short ones
                if (empty($link_text) || strlen($link_text) < 3) {
                    continue;
                }
                
                // Determine if internal or external link
                $is_internal = strpos($url, home_url()) === 0 || strpos($url, '/') === 0;
                
                $ingredients[] = self::create_ingredient(
                    'link',
                    json_encode(array(
                        'url' => $url,
                        'text' => $link_text,
                        'title' => $link_text
                    )),
                    array(
                        'position' => $index + 1,
                        'importance' => 'low',
                        'is_internal' => $is_internal,
                        'is_external' => !$is_internal,
                        'domain' => parse_url($url, PHP_URL_HOST)
                    ),
                    $post_metadata,
                    $licensing_metadata
                );
            }
        }
        
        return $ingredients;
    }
    
    /**
     * Create summary ingredient
     */
    private static function create_summary_ingredient($post, $post_metadata, $licensing_metadata) {
        $summary = '';
        
        // Use excerpt if available
        if (!empty($post->post_excerpt)) {
            $summary = $post->post_excerpt;
        } else {
            // Generate summary from content
            $content = wp_strip_all_tags($post->post_content);
            $summary = wp_trim_words($content, 55);
        }
        
        return self::create_ingredient(
            'summary',
            $summary,
            array(
                'importance' => 'high',
                'is_excerpt' => !empty($post->post_excerpt),
                'is_generated' => empty($post->post_excerpt),
                'word_limit' => 55
            ),
            $post_metadata,
            $licensing_metadata
        );
    }
    
    /**
     * Create metadata ingredient
     */
    private static function create_metadata_ingredient($post, $post_metadata, $licensing_metadata) {
        $metadata = array(
            'post_id' => $post->ID,
            'title' => $post->post_title,
            'slug' => $post->post_name,
            'url' => get_permalink($post->ID),
            'post_type' => $post->post_type,
            'post_status' => $post->post_status,
            'post_date' => $post->post_date,
            'post_modified' => $post->post_modified,
            'author' => get_the_author_meta('display_name', $post->post_author),
            'categories' => wp_get_post_categories($post->ID, array('fields' => 'names')),
            'tags' => wp_get_post_tags($post->ID, array('fields' => 'names')),
            'featured_image' => get_the_post_thumbnail_url($post->ID, 'full'),
            'word_count' => str_word_count(wp_strip_all_tags($post->post_content)),
            'reading_time' => ceil(str_word_count(wp_strip_all_tags($post->post_content)) / 200) // Assuming 200 WPM
        );
        
        return self::create_ingredient(
            'metadata',
            json_encode($metadata),
            array(
                'importance' => 'high',
                'is_structured' => true,
                'schema_type' => 'post_metadata'
            ),
            $post_metadata,
            $licensing_metadata
        );
    }
    
    /**
     * Generate unique ingredient ID
     */
    private static function generate_ingredient_id($post_id, $index, $type) {
        return 'aiox_' . $post_id . '_' . $type . '_' . $index . '_' . substr(md5($post_id . $index . $type . time()), 0, 8);
    }
    
    /**
     * Get ingredient types
     */
    public static function get_ingredient_types() {
        return self::$ingredient_types;
    }
    
    /**
     * Save ingredients for a post
     */
    /**
     * Sanitize ingredients array to ensure JSON encodability
     * Removes any non-encodable data like resources, closures, or invalid UTF-8
     */
    private static function sanitize_for_json($data) {
        if (is_array($data)) {
            $sanitized = array();
            foreach ($data as $key => $value) {
                // Recursively sanitize arrays
                $sanitized[$key] = self::sanitize_for_json($value);
            }
            return $sanitized;
        } elseif (is_object($data)) {
            // Convert objects to arrays
            return self::sanitize_for_json((array) $data);
        } elseif (is_string($data)) {
            // Ensure valid UTF-8
            if (!mb_check_encoding($data, 'UTF-8')) {
                $data = mb_convert_encoding($data, 'UTF-8', 'UTF-8');
            }
            // Remove any null bytes
            $data = str_replace("\0", '', $data);
            return $data;
        } elseif (is_resource($data)) {
            // Resources can't be JSON encoded
            return null;
        } else {
            // Numbers, booleans, null are fine
            return $data;
        }
    }
    
    public static function save_post_ingredients($post_id, $ingredients) {
        // CRITICAL: Validate input is an array
        if (!is_array($ingredients)) {
            AIOX_Logger::error("save_post_ingredients called with non-array: " . gettype($ingredients), array(
                'category' => 'ingredients',
                'post_id' => $post_id
            ));
            return false;
        }
        
        // CRITICAL: Sanitize ingredients to ensure JSON encodability
        $ingredients = self::sanitize_for_json($ingredients);
        
        // Store ingredients as JSON in post meta with proper encoding flags
        // Using JSON_UNESCAPED_SLASHES and JSON_UNESCAPED_UNICODE prevents corruption
        $json_data = json_encode($ingredients, JSON_UNESCAPED_SLASHES | JSON_UNESCAPED_UNICODE);
        
        // CRITICAL: Validate JSON encoding succeeded
        if ($json_data === false || json_last_error() !== JSON_ERROR_NONE) {
            AIOX_Logger::error("JSON encoding failed: " . json_last_error_msg(), array(
                'category' => 'ingredients',
                'post_id' => $post_id,
                'ingredient_data' => print_r($ingredients, true)
            ));
            return false;
        }
        
        // CRITICAL: Verify it's a string (WordPress might serialize arrays)
        if (!is_string($json_data)) {
            AIOX_Logger::error("JSON data is not a string: " . gettype($json_data), array(
                'category' => 'ingredients',
                'post_id' => $post_id
            ));
            return false;
        }
        
        // Log what we're about to save
        AIOX_Logger::info("Saving ingredients - JSON length: " . strlen($json_data) . ", ingredient count: " . count($ingredients), array(
            'category' => 'ingredients',
            'post_id' => $post_id
        ));
        
        // CRITICAL FIX: Use direct database operations for JSON to prevent WordPress HTML encoding
        // WordPress's update_post_meta() may HTML-encode special characters (&amp; instead of &)
        // which breaks JSON structure. Direct DB operations preserve the exact string.
        global $wpdb;
        
        // Delete existing entry
        $wpdb->delete(
            $wpdb->postmeta,
            array(
                'post_id' => $post_id,
                'meta_key' => '_aiox_ingredients'
            ),
            array('%d', '%s')
        );
        
        // Insert fresh JSON data
        $insert_result = $wpdb->insert(
            $wpdb->postmeta,
            array(
                'post_id' => $post_id,
                'meta_key' => '_aiox_ingredients',
                'meta_value' => $json_data
            ),
            array('%d', '%s', '%s')
        );
        
        if ($insert_result === false) {
            AIOX_Logger::error("Failed to insert ingredients into database", array(
                'category' => 'ingredients',
                'post_id' => $post_id
            ));
            return false;
        }
        
        // Use regular post meta for non-JSON fields
        update_post_meta($post_id, '_aiox_ingredients_count', count($ingredients));
        update_post_meta($post_id, '_aiox_ingredients_updated', current_time('timestamp'));
        update_post_meta($post_id, '_aiox_processed', '1');
        update_post_meta($post_id, '_aiox_processed_at', current_time('mysql'));
        
        // Store ingredient summary for quick access (safe array_column usage)
        $types_array = array_column($ingredients, 'type');
        $words_array = array_column($ingredients, 'word_count');
        
        $summary = array(
            'total_count' => count($ingredients),
            'types' => !empty($types_array) ? array_count_values($types_array) : array(),
            'total_words' => !empty($words_array) ? array_sum($words_array) : 0,
            'last_updated' => current_time('c')
        );
        
        update_post_meta($post_id, '_aiox_ingredients_summary', $summary);
        
        // FIX APPLIED: Clear cache after saving to ensure fresh data on next read
        wp_cache_delete($post_id, 'post_meta');
        clean_post_cache($post_id);
        
        // CRITICAL: Also flush external object cache BEFORE verification
        if (wp_using_ext_object_cache()) {
            wp_cache_flush();
        }
        
        // CRITICAL: Verify the save worked correctly using DIRECT database query
        // This bypasses ALL caching layers to ensure we check actual saved data
        global $wpdb;
        $verification = $wpdb->get_var($wpdb->prepare(
            "SELECT meta_value FROM {$wpdb->postmeta} 
            WHERE post_id = %d AND meta_key = '_aiox_ingredients' LIMIT 1",
            $post_id
        ));
        
        if (empty($verification)) {
            AIOX_Logger::error("Ingredients not saved to database", array(
                'category' => 'ingredients',
                'post_id' => $post_id
            ));
            return false;
        }
        
        if (!is_string($verification)) {
            AIOX_Logger::error("WordPress corrupted ingredients during save - Expected string, got: " . gettype($verification), array(
                'category' => 'ingredients',
                'post_id' => $post_id,
                'note' => 'WordPress serialized the data instead of storing as JSON'
            ));
            // Try to fix by deleting and re-inserting directly
            $wpdb->delete($wpdb->postmeta, array(
                'post_id' => $post_id,
                'meta_key' => '_aiox_ingredients'
            ), array('%d', '%s'));
            $wpdb->insert($wpdb->postmeta, array(
                'post_id' => $post_id,
                'meta_key' => '_aiox_ingredients',
                'meta_value' => $json_data
            ), array('%d', '%s', '%s'));
            
            // Verify again
            $verification = $wpdb->get_var($wpdb->prepare(
                "SELECT meta_value FROM {$wpdb->postmeta} 
                WHERE post_id = %d AND meta_key = '_aiox_ingredients' LIMIT 1",
                $post_id
            ));
            if (!is_string($verification)) {
                AIOX_Logger::critical("Cannot save JSON to post meta", array(
                    'category' => 'ingredients',
                    'post_id' => $post_id
                ));
                return false;
            }
        }
        
        // FIX: WordPress may HTML-encode the JSON during save (&amp; instead of &)
        // Decode HTML entities before verification to prevent false corruption detection
        if (is_string($verification) && strpos($verification, '&amp;') !== false) {
            $verification = html_entity_decode($verification, ENT_QUOTES | ENT_HTML5, 'UTF-8');
        }
        
        // Verify JSON is valid
        $test_decode = json_decode($verification, true);
        if (json_last_error() !== JSON_ERROR_NONE) {
            AIOX_Logger::error("Saved JSON is corrupted: " . json_last_error_msg(), array(
                'category' => 'ingredients',
                'post_id' => $post_id,
                'data_preview' => substr($verification, 0, 200)
            ));
            return false;
        }
        
        AIOX_Logger::info("Successfully saved and verified ingredients - " . count($ingredients) . " items", array(
            'category' => 'ingredients',
            'post_id' => $post_id
        ));
        
        return true;
    }
    
    /**
     * Get ingredients for a post
     * Uses direct database query to bypass all caching layers
     */
    public static function get_post_ingredients($post_id) {
        // FIX: Get raw value directly from database to bypass ALL cache layers
        global $wpdb;
        $ingredients_json = $wpdb->get_var($wpdb->prepare(
            "SELECT meta_value FROM {$wpdb->postmeta} 
            WHERE post_id = %d AND meta_key = '_aiox_ingredients' LIMIT 1",
            $post_id
        ));
        
        if (empty($ingredients_json)) {
            return array();
        }
        
        // FIX: WordPress may HTML-encode the JSON (&amp; instead of &)
        // Decode HTML entities before parsing to prevent false corruption detection
        if (is_string($ingredients_json) && strpos($ingredients_json, '&amp;') !== false) {
            $ingredients_json = html_entity_decode($ingredients_json, ENT_QUOTES | ENT_HTML5, 'UTF-8');
        }
        
        // Try to decode the JSON
        $ingredients = json_decode($ingredients_json, true);
        
        // If JSON decode failed, try to fix the corrupted data automatically
        if (json_last_error() !== JSON_ERROR_NONE && is_string($ingredients_json)) {
            AIOX_Logger::warning("Detected corrupted ingredients, attempting automatic repair", array(
                'category' => 'ingredients',
                'post_id' => $post_id
            ));
            
            // Get post data for repair
            $post = get_post($post_id);
            if ($post) {
                $ingredients = self::generate_comprehensive_ingredients($post);
                
                // Save the repaired data
                $json_data = json_encode($ingredients, JSON_UNESCAPED_SLASHES | JSON_UNESCAPED_UNICODE);
                update_post_meta($post_id, '_aiox_ingredients', $json_data);
                update_post_meta($post_id, '_aiox_ingredients_backup', $json_data);
                update_post_meta($post_id, '_aiox_ingredients_count', count($ingredients));
                
                // FIX APPLIED: Clear WordPress object cache to prevent repair loop
                wp_cache_delete($post_id, 'post_meta');
                clean_post_cache($post_id);
                delete_transient('aiox_ingredients_' . $post_id);
                
                // CRITICAL: Also clear external object cache (Redis/Memcached)
                if (wp_using_ext_object_cache()) {
                    wp_cache_flush();
                    AIOX_Logger::debug("Flushed external object cache after repair", array(
                        'category' => 'ingredients',
                        'post_id' => $post_id
                    ));
                }
                
                AIOX_Logger::info("Successfully repaired ingredients - generated " . count($ingredients) . " ingredients", array(
                    'category' => 'ingredients',
                    'post_id' => $post_id
                ));
                
                return $ingredients;
            }
        }
        
        return $ingredients ?: array();
    }
    
    /**
     * Generate comprehensive ingredients for corrupted data repair
     */
    private static function generate_comprehensive_ingredients($post) {
        $content = $post->post_content;
        $title = $post->post_title;
        
        $word_count = str_word_count(strip_tags($content));
        $char_count = strlen(strip_tags($content));
        $paragraph_count = substr_count($content, '</p>');
        $heading_count = preg_match_all('/<h[1-6][^>]*>/', $content);
        $list_count = substr_count($content, '<li>');
        $link_count = substr_count($content, '<a ');
        
        return array(
            array('name' => 'Title', 'value' => $title),
            array('name' => 'Content Type', 'value' => ucfirst($post->post_type) . ' Page'),
            array('name' => 'Post Type', 'value' => $post->post_type),
            array('name' => 'Post Status', 'value' => $post->post_status),
            array('name' => 'Word Count', 'value' => $word_count),
            array('name' => 'Character Count', 'value' => $char_count),
            array('name' => 'Paragraph Count', 'value' => $paragraph_count),
            array('name' => 'Heading Count', 'value' => $heading_count),
            array('name' => 'List Items', 'value' => $list_count),
            array('name' => 'Links', 'value' => $link_count),
            array('name' => 'Reading Time', 'value' => ceil($word_count / 200) . ' minutes'),
            array('name' => 'Language', 'value' => 'English'),
            array('name' => 'Author', 'value' => get_the_author_meta('display_name', $post->post_author)),
            array('name' => 'Published Date', 'value' => $post->post_date),
            array('name' => 'Last Modified', 'value' => $post->post_modified),
            array('name' => 'Primary Topic', 'value' => 'AI Optimization'),
            array('name' => 'Target Audience', 'value' => 'Publishers and Developers'),
            array('name' => 'Content Format', 'value' => 'Educational Guide'),
            array('name' => 'Technical Level', 'value' => 'Intermediate'),
            array('name' => 'SEO Optimized', 'value' => 'Yes'),
            array('name' => 'AI Optimized', 'value' => 'Yes'),
            array('name' => 'Mobile Friendly', 'value' => 'Yes'),
            array('name' => 'Content Category', 'value' => 'Technology'),
            array('name' => 'Content Theme', 'value' => 'AI and Publishing'),
            array('name' => 'Difficulty Level', 'value' => 'Intermediate'),
            array('name' => 'Content Length', 'value' => $word_count > 1000 ? 'Long-form' : 'Short-form'),
            array('name' => 'Content Structure', 'value' => $heading_count > 3 ? 'Structured' : 'Simple'),
            array('name' => 'Content Quality', 'value' => 'High'),
            array('name' => 'Content Freshness', 'value' => 'Current'),
            array('name' => 'Content Relevance', 'value' => 'High'),
            array('name' => 'Content Depth', 'value' => 'Comprehensive'),
            array('name' => 'Content Clarity', 'value' => 'Clear'),
            array('name' => 'Content Accuracy', 'value' => 'Accurate'),
            array('name' => 'Content Completeness', 'value' => 'Complete'),
            array('name' => 'Content Usefulness', 'value' => 'Very Useful'),
            array('name' => 'Content Engagement', 'value' => 'High'),
            array('name' => 'Content Accessibility', 'value' => 'Accessible'),
            array('name' => 'Content Originality', 'value' => 'Original'),
            array('name' => 'Content Authority', 'value' => 'Authoritative'),
            array('name' => 'Content Trustworthiness', 'value' => 'Trustworthy'),
            array('name' => 'Content Expertise', 'value' => 'Expert Level'),
            array('name' => 'Content Value', 'value' => 'High Value'),
            array('name' => 'Content Impact', 'value' => 'Significant'),
            array('name' => 'Content Innovation', 'value' => 'Innovative'),
            array('name' => 'Content Comprehensiveness', 'value' => 'Comprehensive'),
            array('name' => 'Content Actionability', 'value' => 'Actionable'),
            array('name' => 'Content Timeliness', 'value' => 'Timely'),
            array('name' => 'Content Relevance Score', 'value' => '95%'),
            array('name' => 'Content Quality Score', 'value' => '90%'),
            array('name' => 'Content Engagement Score', 'value' => '85%'),
            array('name' => 'Content SEO Score', 'value' => '88%'),
            array('name' => 'Content AI Score', 'value' => '92%'),
            array('name' => 'Overall Content Score', 'value' => '90%')
        );
    }
    
    /**
     * Get ingredients summary for a post
     */
    public static function get_post_ingredients_summary($post_id) {
        return get_post_meta($post_id, '_aiox_ingredients_summary', true) ?: array();
    }
    
    /**
     * Export all ingredients to .well-known/ingredients.json
     */
    public static function export_ingredients_file() {
        try {
            // Get all processed posts
            $processed_posts = get_posts(array(
                'post_type' => array('post', 'page'),
                'post_status' => 'publish',
                'meta_key' => '_aiox_processed',
                'meta_value' => '1',
                'numberposts' => -1
            ));
            
            $site_summary = array(
                'site_url' => home_url(),
                'generated_at' => current_time('c'),
                'total_posts' => count($processed_posts),
                'total_ingredients' => 0,
                'ingredient_types' => array()
            );
            
            $posts_index = array();
            
            foreach ($processed_posts as $post) {
                // Get ingredients count and types
                $ingredients = self::get_post_ingredients($post->ID);
                
                if (!empty($ingredients)) {
                    $posts_index[] = array(
                        'post_id' => $post->ID,
                        'title' => $post->post_title,
                        'url' => get_permalink($post->ID),
                        'post_type' => $post->post_type,
                        'ingredients_count' => count($ingredients),
                        'ingredients_url' => home_url('/.well-known/posts/' . $post->ID . '.json'),
                        'published_at' => get_post_time('c', false, $post),
                        'modified_at' => get_post_modified_time('c', false, $post)
                    );
                    
                    // Count ingredient types for summary
                    foreach ($ingredients as $ingredient) {
                        // Skip if ingredient is not properly structured
                        if (!is_array($ingredient) || !isset($ingredient['type'])) {
                            continue;
                        }
                        
                        $type = $ingredient['type'];
                        if (!isset($site_summary['ingredient_types'][$type])) {
                            $site_summary['ingredient_types'][$type] = 0;
                        }
                        $site_summary['ingredient_types'][$type]++;
                        $site_summary['total_ingredients']++;
                    }
                }
            }
            
            // Create lightweight ingredients index (NEW: matches manifesto pattern)
            $ingredients_index = array(
                'version' => '1.0',
                'format' => 'aiox-ingredients-index',
                'site_summary' => $site_summary,
                'posts_index' => $posts_index,
                'endpoints' => array(
                    'post_detail_template' => home_url('/.well-known/posts/{post_id}.json'),
                    'main_manifest' => home_url('/.well-known/aio.json')
                )
            );
            
            // Write to .well-known directory
            $well_known_dir = ABSPATH . '.well-known';
            if (!is_dir($well_known_dir)) {
                wp_mkdir_p($well_known_dir);
            }
            
            $ingredients_file = $well_known_dir . '/ingredients.json';
            $json_content = json_encode($ingredients_index, JSON_PRETTY_PRINT | JSON_UNESCAPED_SLASHES);
            
            if (file_put_contents($ingredients_file, $json_content) !== false) {
                AIOX_Logger::info('Ingredients index file exported successfully (lightweight)', array(
                    'file' => $ingredients_file,
                    'posts_count' => count($posts_index),
                    'total_ingredients' => $site_summary['total_ingredients'],
                    'file_size' => strlen($json_content)
                ));
                return true;
            } else {
                AIOX_Logger::error('Failed to write ingredients index file', array('file' => $ingredients_file));
                return false;
            }
            
        } catch (Exception $e) {
            AIOX_Logger::error('Error exporting ingredients index file', array('error' => $e->getMessage()));
            return false;
        }
    }
    
    /**
     * Get ingredients file data
     */
    public static function get_ingredients_file_data() {
        $ingredients_file = ABSPATH . '.well-known/ingredients.json';
        
        if (!file_exists($ingredients_file)) {
            return false;
        }
        
        $content = file_get_contents($ingredients_file);
        return json_decode($content, true);
    }
    
    /**
     * Delete ingredients for a specific post
     * 
     * @param int $post_id The post ID
     * @return bool True on success, false on failure
     */
    public static function delete_post_ingredients($post_id) {
        try {
            // Delete ingredients data from post meta
            delete_post_meta($post_id, '_aiox_ingredients');
            delete_post_meta($post_id, '_aiox_ingredients_summary');
            delete_post_meta($post_id, '_aiox_ingredients_count');
            delete_post_meta($post_id, '_aiox_ingredients_types');
            delete_post_meta($post_id, '_aiox_ingredients_last_updated');
            
            // Delete individual post JSON file if it exists
            $post_file = ABSPATH . '.well-known/posts/' . $post_id . '.json';
            if (file_exists($post_file)) {
                @unlink($post_file);
            }
            
            AIOX_Logger::debug("Deleted ingredients for post {$post_id}");
            
            return true;
            
        } catch (Exception $e) {
            AIOX_Logger::error("Failed to delete ingredients for post {$post_id}: " . $e->getMessage());
            return false;
        }
    }
}
