<?php
/**
 * AIOX Licensing System
 * 
 * Handles license validation and Pro feature management
 */

// Prevent direct access
if (!defined('ABSPATH')) {
    exit;
}

class AIOX_Licensing {
    
    /**
     * License validation endpoint
     */
    const LICENSE_SERVER = 'https://aioxsuite.com/api/license';
    
    /**
     * Initialize licensing system
     */
    public function __construct() {
        // Only add hooks if WordPress is loaded
        if (function_exists('add_action')) {
            add_action('admin_init', array($this, 'check_license_status'));
            add_action('wp_ajax_aiox_validate_license', array($this, 'ajax_validate_license'));
        }
    }
    
    /**
     * Validate license key
     */
    public static function validate_license($license_key) {
        if (empty($license_key)) {
            return array(
                'valid' => false,
                'message' => __('License key is required', 'aiox-publisher-suite-pro')
            );
        }
        
        // For now, implement basic validation
        // In production, this would connect to your license server
        if (strlen($license_key) >= 20 && preg_match('/^[A-Z0-9\-]+$/', $license_key)) {
            return array(
                'valid' => true,
                'message' => __('License key is valid', 'aiox-publisher-suite-pro'),
                'features' => array(
                    'external_ai' => true,
                    'advanced_analytics' => true,
                    'priority_support' => true
                )
            );
        }
        
        return array(
            'valid' => false,
            'message' => __('Invalid license key format', 'aiox-publisher-suite-pro')
        );
    }
    
    /**
     * Check if license is valid
     */
    public static function is_license_valid() {
        $license_key = get_option('aiox_license_key', '');
        if (empty($license_key)) {
            return false;
        }
        
        $validation = self::validate_license($license_key);
        return $validation['valid'];
    }
    
    /**
     * Check if Pro feature is available
     */
    public static function is_pro_feature_available($feature) {
        if (!self::is_license_valid()) {
            return false;
        }
        
        $license_key = get_option('aiox_license_key', '');
        $validation = self::validate_license($license_key);
        
        if (!$validation['valid']) {
            return false;
        }
        
        return isset($validation['features'][$feature]) && $validation['features'][$feature];
    }
    
    /**
     * Get license status
     */
    public static function get_license_status() {
        $license_key = get_option('aiox_license_key', '');
        
        if (empty($license_key)) {
            return array(
                'status' => 'inactive',
                'message' => __('No license key entered', 'aiox-publisher-suite-pro')
            );
        }
        
        $validation = self::validate_license($license_key);
        
        if ($validation['valid']) {
            return array(
                'status' => 'active',
                'message' => __('License is active and valid', 'aiox-publisher-suite-pro'),
                'features' => $validation['features']
            );
        } else {
            return array(
                'status' => 'invalid',
                'message' => $validation['message']
            );
        }
    }
    
    /**
     * Check license status periodically
     */
    public function check_license_status() {
        // Check license status once per day
        $last_check = get_option('aiox_license_last_check', 0);
        $current_time = time();
        
        if (($current_time - $last_check) > DAY_IN_SECONDS) {
            $license_key = get_option('aiox_license_key', '');
            
            if (!empty($license_key)) {
                $validation = self::validate_license($license_key);
                update_option('aiox_license_status', $validation);
                update_option('aiox_license_last_check', $current_time);
            }
        }
    }
    
    /**
     * AJAX handler for license validation
     */
    public function ajax_validate_license() {
        check_ajax_referer('aiox_ajax_nonce', 'nonce');
        
        if (!current_user_can('manage_options')) {
            wp_send_json_error(__('Insufficient permissions', 'aiox-publisher-suite-pro'));
        }
        
        $license_key = sanitize_text_field($_POST['license_key']);
        $validation = self::validate_license($license_key);
        
        if ($validation['valid']) {
            update_option('aiox_license_key', $license_key);
            update_option('aiox_license_status', $validation);
            update_option('aiox_license_last_check', time());
            
            wp_send_json_success(array(
                'message' => __('License activated successfully!', 'aiox-publisher-suite-pro'),
                'status' => $validation
            ));
        } else {
            wp_send_json_error($validation['message']);
        }
    }
    
    /**
     * Deactivate license
     */
    public static function deactivate_license() {
        delete_option('aiox_license_key');
        delete_option('aiox_license_status');
        delete_option('aiox_license_last_check');
        
        return array(
            'success' => true,
            'message' => __('License deactivated successfully', 'aiox-publisher-suite-pro')
        );
    }
    
    /**
     * Get Pro upgrade URL
     */
    public static function get_upgrade_url() {
        return 'https://aioxsuite.com/upgrade';
    }
    
    /**
     * Display Pro upgrade notice
     */
    public static function display_upgrade_notice($feature_name) {
        $upgrade_url = self::get_upgrade_url();
        
        echo '<div class="aiox-pro-notice">';
        echo '<h4>' . __('Pro Feature', 'aiox-publisher-suite-pro') . '</h4>';
        echo '<p>' . sprintf(
            __('%s is a Pro feature. Upgrade to unlock advanced AI processing and analytics.', 'aiox-publisher-suite-pro'),
            $feature_name
        ) . '</p>';
        echo '<a href="' . esc_url($upgrade_url) . '" class="aiox-button primary" target="_blank">';
        echo __('Upgrade to Pro', 'aiox-publisher-suite-pro');
        echo '</a>';
        echo '</div>';
    }
}

// Initialize licensing system
new AIOX_Licensing();
