<?php
/**
 * AIOX Logger Class
 * 
 * Handles logging functionality for the AIOX plugin
 */

if (!defined('ABSPATH')) {
    exit;
}

class AIOX_Logger {
    
    /**
     * Log levels
     */
    const DEBUG = 'debug';
    const INFO = 'info';
    const WARNING = 'warning';
    const ERROR = 'error';
    const CRITICAL = 'critical';
    
    /**
     * Log a message with enhanced context
     */
    public static function log($message, $level = self::INFO, $context = array()) {
        global $wpdb;
        
        // Check if logging is enabled
        if (function_exists('get_option') && !get_option('aiox_enable_logging', 1)) {
            return;
        }
        
        // Check log level threshold
        $min_level = function_exists('get_option') ? get_option('aiox_log_level', self::ERROR) : self::ERROR;
        if (!self::should_log($level, $min_level)) {
            return;
        }
        
        // Extract special fields from context
        $category = isset($context['category']) ? $context['category'] : 'general';
        $post_id = isset($context['post_id']) ? intval($context['post_id']) : null;
        unset($context['category'], $context['post_id']); // Remove from context to avoid duplication
        
        // Get user and IP
        $user_id = function_exists('get_current_user_id') ? get_current_user_id() : null;
        $ip_address = self::get_ip_address();
        
        // Prepare context data
        $context_json = !empty($context) ? json_encode($context, JSON_UNESCAPED_SLASHES | JSON_UNESCAPED_UNICODE) : null;
        
        // Insert log entry
        if (!$wpdb) {
            // Skip logging if WordPress database is not available
            return;
        }
        
        $table_name = $wpdb->prefix . 'aiox_logs';
        
        // Check if table exists before inserting
        if ($wpdb->get_var("SHOW TABLES LIKE '{$table_name}'") !== $table_name) {
            // Skip logging if table doesn't exist yet
            return;
        }
        
        $wpdb->insert(
            $table_name,
            array(
                'timestamp' => function_exists('current_time') ? current_time('mysql') : date('Y-m-d H:i:s'),
                'level' => $level,
                'category' => $category,
                'message' => $message,
                'context' => $context_json,
                'post_id' => $post_id,
                'user_id' => $user_id ?: null,
                'ip_address' => $ip_address
            ),
            array('%s', '%s', '%s', '%s', '%s', '%d', '%d', '%s')
        );
        
        // Clean up old logs if needed
        self::cleanup_old_logs();
    }
    
    /**
     * Get IP address
     */
    private static function get_ip_address() {
        if (!empty($_SERVER['HTTP_CLIENT_IP'])) {
            return $_SERVER['HTTP_CLIENT_IP'];
        } elseif (!empty($_SERVER['HTTP_X_FORWARDED_FOR'])) {
            return $_SERVER['HTTP_X_FORWARDED_FOR'];
        } else {
            return $_SERVER['REMOTE_ADDR'] ?? '';
        }
    }
    
    /**
     * Log debug message
     */
    public static function debug($message, $context = array()) {
        self::log($message, self::DEBUG, $context);
    }
    
    /**
     * Log info message
     */
    public static function info($message, $context = array()) {
        self::log($message, self::INFO, $context);
    }
    
    /**
     * Log warning message
     */
    public static function warning($message, $context = array()) {
        self::log($message, self::WARNING, $context);
    }
    
    /**
     * Log error message
     */
    public static function error($message, $context = array()) {
        self::log($message, self::ERROR, $context);
    }
    
    /**
     * Log critical message
     */
    public static function critical($message, $context = array()) {
        self::log($message, self::CRITICAL, $context);
    }
    
    /**
     * Check if message should be logged based on level
     */
    private static function should_log($level, $min_level) {
        $levels = array(
            self::DEBUG => 0,
            self::INFO => 1,
            self::WARNING => 2,
            self::ERROR => 3,
            self::CRITICAL => 4
        );
        
        return isset($levels[$level]) && isset($levels[$min_level]) && 
               $levels[$level] >= $levels[$min_level];
    }
    
    /**
     * Clean up old log entries
     */
    public static function cleanup_old_logs() {
        global $wpdb;
        
        // Only run cleanup occasionally
        if (rand(1, 100) > 5) { // 5% chance
            return;
        }
        
        $table_name = $wpdb->prefix . 'aiox_logs';
        
        // Check if table exists before cleaning
        if ($wpdb->get_var("SHOW TABLES LIKE '{$table_name}'") !== $table_name) {
            return;
        }
        
        $retention_days = function_exists('get_option') ? get_option('aiox_log_retention_days', 30) : 30;
        $cutoff_date = date('Y-m-d H:i:s', strtotime("-{$retention_days} days"));
        
        $wpdb->query($wpdb->prepare(
            "DELETE FROM {$table_name} WHERE timestamp < %s",
            $cutoff_date
        ));
    }
    
    /**
     * Get logs with pagination and filters
     */
    public static function get_logs($args = array()) {
        global $wpdb;
        $table_name = $wpdb->prefix . 'aiox_logs';
        
        // Check if table exists
        if ($wpdb->get_var("SHOW TABLES LIKE '{$table_name}'") !== $table_name) {
            return array('logs' => array(), 'total' => 0, 'pages' => 0, 'current_page' => 1);
        }
        
        // Default arguments
        $defaults = array(
            'page' => 1,
            'per_page' => 50,
            'level' => '',
            'category' => '',
            'post_id' => '',
            'search' => '',
            'date_from' => '',
            'date_to' => '',
            'orderby' => 'timestamp',
            'order' => 'DESC'
        );
        
        $args = wp_parse_args($args, $defaults);
        
        // Build WHERE clause
        $where = array('1=1');
        $where_values = array();
        
        if (!empty($args['level'])) {
            $where[] = 'level = %s';
            $where_values[] = $args['level'];
        }
        
        if (!empty($args['category'])) {
            $where[] = 'category = %s';
            $where_values[] = $args['category'];
        }
        
        if (!empty($args['post_id'])) {
            $where[] = 'post_id = %d';
            $where_values[] = intval($args['post_id']);
        }
        
        if (!empty($args['search'])) {
            $where[] = 'message LIKE %s';
            $where_values[] = '%' . $wpdb->esc_like($args['search']) . '%';
        }
        
        if (!empty($args['date_from'])) {
            $where[] = 'timestamp >= %s';
            $where_values[] = $args['date_from'] . ' 00:00:00';
        }
        
        if (!empty($args['date_to'])) {
            $where[] = 'timestamp <= %s';
            $where_values[] = $args['date_to'] . ' 23:59:59';
        }
        
        $where_clause = implode(' AND ', $where);
        
        // Get total count
        $count_query = "SELECT COUNT(*) FROM $table_name WHERE $where_clause";
        if (!empty($where_values)) {
            $count_query = $wpdb->prepare($count_query, $where_values);
        }
        $total = $wpdb->get_var($count_query);
        
        // Calculate offset
        $offset = ($args['page'] - 1) * $args['per_page'];
        
        // Build ORDER BY
        $allowed_orderby = array('timestamp', 'level', 'category', 'post_id');
        $orderby = in_array($args['orderby'], $allowed_orderby) ? $args['orderby'] : 'timestamp';
        $order = strtoupper($args['order']) === 'ASC' ? 'ASC' : 'DESC';
        
        // Get logs
        $query = "SELECT * FROM $table_name 
                 WHERE $where_clause 
                 ORDER BY $orderby $order 
                 LIMIT %d OFFSET %d";
        
        $where_values[] = intval($args['per_page']);
        $where_values[] = intval($offset);
        
        $logs = $wpdb->get_results($wpdb->prepare($query, $where_values), ARRAY_A);
        
        return array(
            'logs' => $logs,
            'total' => $total,
            'pages' => ceil($total / $args['per_page']),
            'current_page' => $args['page']
        );
    }
    
    /**
     * Get available categories
     */
    public static function get_categories() {
        global $wpdb;
        $table_name = $wpdb->prefix . 'aiox_logs';
        
        // Check if table exists
        if ($wpdb->get_var("SHOW TABLES LIKE '{$table_name}'") !== $table_name) {
            return array();
        }
        
        return $wpdb->get_col("SELECT DISTINCT category FROM $table_name ORDER BY category");
    }
    
    /**
     * Delete specific logs by IDs
     */
    public static function delete_logs($ids) {
        global $wpdb;
        $table_name = $wpdb->prefix . 'aiox_logs';
        
        if (empty($ids)) {
            return false;
        }
        
        $ids = array_map('intval', (array) $ids);
        $placeholders = implode(',', array_fill(0, count($ids), '%d'));
        
        return $wpdb->query($wpdb->prepare(
            "DELETE FROM $table_name WHERE id IN ($placeholders)",
            $ids
        ));
    }
    
    /**
     * Delete logs older than X days
     */
    public static function delete_old_logs($days = 30) {
        global $wpdb;
        $table_name = $wpdb->prefix . 'aiox_logs';
        
        return $wpdb->query($wpdb->prepare(
            "DELETE FROM $table_name WHERE timestamp < DATE_SUB(NOW(), INTERVAL %d DAY)",
            intval($days)
        ));
    }
    
    /**
     * Get recent log entries
     */
    public static function get_recent_logs($limit = 50, $level = null) {
        global $wpdb;
        
        $table_name = $wpdb->prefix . 'aiox_logs';
        
        // Check if table exists
        if ($wpdb->get_var("SHOW TABLES LIKE '{$table_name}'") !== $table_name) {
            return array();
        }
        
        $where_clause = '';
        $params = array();
        
        if ($level) {
            $where_clause = 'WHERE level = %s';
            $params[] = $level;
        }
        
        $params[] = $limit;
        
        return $wpdb->get_results($wpdb->prepare(
            "SELECT * FROM {$table_name} {$where_clause} ORDER BY timestamp DESC LIMIT %d",
            ...$params
        ), ARRAY_A);
    }
    
    /**
     * Get log statistics
     */
    public static function get_log_stats() {
        global $wpdb;
        
        $table_name = $wpdb->prefix . 'aiox_logs';
        
        // Check if table exists
        if ($wpdb->get_var("SHOW TABLES LIKE '{$table_name}'") !== $table_name) {
            return array(
                'total' => 0,
                'debug' => 0,
                'info' => 0,
                'warning' => 0,
                'error' => 0,
                'critical' => 0,
                'recent' => 0,
                'by_level' => array(),
                'recent_errors' => 0,
                'by_category' => array()
            );
        }
        
        $stats = array();
        
        // Total logs
        $stats['total'] = $wpdb->get_var("SELECT COUNT(*) FROM {$table_name}");
        
        // Logs by level
        $levels = array(self::DEBUG, self::INFO, self::WARNING, self::ERROR, self::CRITICAL);
        $stats['by_level'] = array();
        foreach ($levels as $level) {
            $count = $wpdb->get_var($wpdb->prepare(
                "SELECT COUNT(*) FROM {$table_name} WHERE level = %s",
                $level
            ));
            $stats[$level] = $count;
            $stats['by_level'][$level] = $count;
        }
        
        // Recent errors (last 24 hours)
        $stats['recent_errors'] = $wpdb->get_var($wpdb->prepare(
            "SELECT COUNT(*) FROM {$table_name} WHERE level IN ('error', 'critical') AND timestamp >= %s",
            date('Y-m-d H:i:s', strtotime('-24 hours'))
        ));
        
        // Recent logs (last 24 hours)
        $stats['recent'] = $wpdb->get_var($wpdb->prepare(
            "SELECT COUNT(*) FROM {$table_name} WHERE timestamp >= %s",
            date('Y-m-d H:i:s', strtotime('-24 hours'))
        ));
        
        // By category (top 5)
        $stats['by_category'] = $wpdb->get_results(
            "SELECT category, COUNT(*) as count FROM {$table_name} 
            GROUP BY category 
            ORDER BY count DESC 
            LIMIT 5"
        );
        
        return $stats;
    }
    
    /**
     * Get statistics (alias for get_log_stats for compatibility)
     */
    public static function get_statistics() {
        return self::get_log_stats();
    }
    
    /**
     * Clear all logs
     */
    public static function clear_logs() {
        global $wpdb;
        
        $table_name = $wpdb->prefix . 'aiox_logs';
        
        // Check if table exists
        if ($wpdb->get_var("SHOW TABLES LIKE '{$table_name}'") !== $table_name) {
            return array('success' => false, 'error' => 'Logs table does not exist');
        }
        
        // Get count before deletion for verification
        $count_before = $wpdb->get_var("SELECT COUNT(*) FROM {$table_name}");
        
        // Use DELETE instead of TRUNCATE for better compatibility
        $result = $wpdb->query("DELETE FROM {$table_name}");
        
        // Check for database errors
        if ($result === false) {
            $error = $wpdb->last_error ? $wpdb->last_error : 'Unknown database error';
            return array('success' => false, 'error' => $error);
        }
        
        // Verify deletion by checking count after
        $count_after = $wpdb->get_var("SELECT COUNT(*) FROM {$table_name}");
        
        if ($count_after > 0) {
            return array('success' => false, 'error' => "Deletion incomplete. {$count_after} logs remain.");
        }
        
        return array('success' => true, 'deleted' => $count_before);
    }
    
    /**
     * Clear all logs (alias for clear_logs for compatibility)
     */
    public static function clear_all_logs() {
        return self::clear_logs();
    }
}