<?php
/**
 * AIOX Metadata Class
 * 
 * Handles enhanced metadata fields and heuristic classification for the Ingredients System
 */

if (!defined('ABSPATH')) {
    exit;
}

class AIOX_Metadata {
    
    /**
     * Available metadata fields with their options
     */
    private static $metadata_fields = array(
        'intent' => array(
            'label' => 'Content Intent',
            'type' => 'select',
            'options' => array(
                'informational' => 'Informational',
                'persuasive' => 'Persuasive',
                'instructional' => 'Instructional',
                'entertainment' => 'Entertainment',
                'promotional' => 'Promotional',
                'editorial' => 'Editorial',
                'reference' => 'Reference'
            ),
            'default' => 'informational'
        ),
        'tone' => array(
            'label' => 'Content Tone',
            'type' => 'select',
            'options' => array(
                'formal' => 'Formal',
                'casual' => 'Casual',
                'professional' => 'Professional',
                'friendly' => 'Friendly',
                'humorous' => 'Humorous',
                'serious' => 'Serious',
                'conversational' => 'Conversational',
                'academic' => 'Academic'
            ),
            'default' => 'professional'
        ),
        'orientation' => array(
            'label' => 'Narrative Orientation',
            'type' => 'select',
            'options' => array(
                'first_person' => 'First Person',
                'second_person' => 'Second Person',
                'third_person' => 'Third Person',
                'mixed' => 'Mixed'
            ),
            'default' => 'third_person'
        ),
        'domain' => array(
            'label' => 'Subject Domain',
            'type' => 'text',
            'placeholder' => 'e.g., Technology, Health, Business',
            'default' => ''
        ),
        'utility' => array(
            'label' => 'Content Utility',
            'type' => 'select',
            'options' => array(
                'how_to' => 'How-to Guide',
                'reference' => 'Reference Material',
                'summary' => 'Summary',
                'analysis' => 'Analysis',
                'review' => 'Review',
                'tutorial' => 'Tutorial',
                'news' => 'News',
                'opinion' => 'Opinion',
                'case_study' => 'Case Study'
            ),
            'default' => 'reference'
        ),
        'target_audience' => array(
            'label' => 'Target Audience',
            'type' => 'select',
            'options' => array(
                'general' => 'General Public',
                'novice' => 'Beginners/Novice',
                'intermediate' => 'Intermediate',
                'expert' => 'Expert/Professional',
                'academic' => 'Academic',
                'business' => 'Business Professional'
            ),
            'default' => 'general'
        ),
        'narrative_type' => array(
            'label' => 'Narrative Type',
            'type' => 'select',
            'options' => array(
                'article' => 'Article',
                'story' => 'Story',
                'listicle' => 'Listicle',
                'step_by_step' => 'Step-by-step',
                'qa_format' => 'Q&A Format',
                'interview' => 'Interview',
                'case_study' => 'Case Study',
                'comparison' => 'Comparison'
            ),
            'default' => 'article'
        ),
        'media_type' => array(
            'label' => 'Primary Media Type',
            'type' => 'select',
            'options' => array(
                'text' => 'Text',
                'image_heavy' => 'Image Heavy',
                'video' => 'Video',
                'audio' => 'Audio',
                'mixed_media' => 'Mixed Media',
                'interactive' => 'Interactive'
            ),
            'default' => 'text'
        ),
        'geolocation_sensitivity' => array(
            'label' => 'Geographic Relevance',
            'type' => 'select',
            'options' => array(
                'global' => 'Global',
                'regional' => 'Regional',
                'country_specific' => 'Country Specific',
                'local' => 'Local'
            ),
            'default' => 'global'
        ),
        'market_segments' => array(
            'label' => 'Market Segments',
            'type' => 'multiselect',
            'options' => array(
                'b2b' => 'Business to Business',
                'b2c' => 'Business to Consumer',
                'enterprise' => 'Enterprise',
                'small_business' => 'Small Business',
                'startup' => 'Startup',
                'education' => 'Education',
                'healthcare' => 'Healthcare',
                'technology' => 'Technology',
                'finance' => 'Finance',
                'retail' => 'Retail'
            ),
            'default' => array()
        )
    );
    
    /**
     * Get all metadata fields
     */
    public static function get_metadata_fields() {
        return self::$metadata_fields;
    }
    
    /**
     * Get metadata field definition
     */
    public static function get_field_definition($field_name) {
        return isset(self::$metadata_fields[$field_name]) ? self::$metadata_fields[$field_name] : null;
    }
    
    /**
     * Classify content using basic heuristics
     */
    public static function classify_content($post) {
        $content = $post->post_content;
        $title = $post->post_title;
        $categories = wp_get_post_categories($post->ID, array('fields' => 'names'));
        $tags = wp_get_post_tags($post->ID, array('fields' => 'names'));
        
        $metadata = array();
        
        // Classify Intent
        $metadata['intent'] = self::classify_intent($content, $title, $categories, $tags);
        
        // Classify Tone
        $metadata['tone'] = self::classify_tone($content, $title);
        
        // Classify Orientation
        $metadata['orientation'] = self::classify_orientation($content);
        
        // Classify Domain
        $metadata['domain'] = self::classify_domain($categories, $tags);
        
        // Classify Utility
        $metadata['utility'] = self::classify_utility($content, $title);
        
        // Classify Target Audience
        $metadata['target_audience'] = self::classify_target_audience($content, $title);
        
        // Classify Narrative Type
        $metadata['narrative_type'] = self::classify_narrative_type($content, $title);
        
        // Classify Media Type
        $metadata['media_type'] = self::classify_media_type($content, $post);
        
        // Classify Geographic Sensitivity
        $metadata['geolocation_sensitivity'] = self::classify_geolocation($content, $categories, $tags);
        
        // Classify Market Segments
        $metadata['market_segments'] = self::classify_market_segments($categories, $tags, $content);
        
        return $metadata;
    }
    
    /**
     * Classify content intent using heuristics
     */
    private static function classify_intent($content, $title, $categories, $tags) {
        $content_lower = strtolower($content . ' ' . $title);
        
        // Check for instructional keywords
        if (preg_match('/\b(how to|tutorial|guide|step|learn|teach|instruction)\b/i', $content_lower)) {
            return 'instructional';
        }
        
        // Check for promotional keywords
        if (preg_match('/\b(buy|purchase|sale|discount|offer|deal|product|service)\b/i', $content_lower)) {
            return 'promotional';
        }
        
        // Check for persuasive keywords
        if (preg_match('/\b(should|must|need to|important|critical|essential|recommend)\b/i', $content_lower)) {
            return 'persuasive';
        }
        
        // Check for entertainment keywords
        if (preg_match('/\b(story|funny|humor|entertainment|joke|amusing)\b/i', $content_lower)) {
            return 'entertainment';
        }
        
        // Check for editorial keywords
        if (preg_match('/\b(opinion|editorial|commentary|perspective|viewpoint|analysis)\b/i', $content_lower)) {
            return 'editorial';
        }
        
        return 'informational'; // Default
    }
    
    /**
     * Classify content tone using heuristics
     */
    private static function classify_tone($content, $title) {
        $content_lower = strtolower($content . ' ' . $title);
        
        // Check for academic tone
        if (preg_match('/\b(research|study|analysis|methodology|hypothesis|conclusion)\b/i', $content_lower)) {
            return 'academic';
        }
        
        // Check for humorous tone
        if (preg_match('/\b(funny|hilarious|joke|lol|haha|amusing|witty)\b/i', $content_lower)) {
            return 'humorous';
        }
        
        // Check for casual tone
        if (preg_match('/\b(hey|hi|cool|awesome|yeah|gonna|wanna)\b/i', $content_lower)) {
            return 'casual';
        }
        
        // Check for formal tone indicators
        if (preg_match('/\b(furthermore|therefore|consequently|nevertheless|moreover)\b/i', $content_lower)) {
            return 'formal';
        }
        
        return 'professional'; // Default
    }
    
    /**
     * Classify narrative orientation
     */
    private static function classify_orientation($content) {
        $first_person_count = preg_match_all('/\b(I|me|my|mine|we|us|our|ours)\b/i', $content);
        $second_person_count = preg_match_all('/\b(you|your|yours)\b/i', $content);
        
        if ($first_person_count > $second_person_count && $first_person_count > 5) {
            return 'first_person';
        } elseif ($second_person_count > 10) {
            return 'second_person';
        } elseif ($first_person_count > 0 && $second_person_count > 0) {
            return 'mixed';
        }
        
        return 'third_person'; // Default
    }
    
    /**
     * Classify domain from categories and tags
     */
    private static function classify_domain($categories, $tags) {
        $all_terms = array_merge($categories, $tags);
        
        if (!empty($all_terms)) {
            return implode(', ', array_slice($all_terms, 0, 3)); // Take first 3 terms
        }
        
        return 'General';
    }
    
    /**
     * Classify utility type
     */
    private static function classify_utility($content, $title) {
        $content_lower = strtolower($content . ' ' . $title);
        
        if (preg_match('/\b(how to|tutorial|guide|step)\b/i', $content_lower)) {
            return 'how_to';
        }
        
        if (preg_match('/\b(review|rating|stars|recommend)\b/i', $content_lower)) {
            return 'review';
        }
        
        if (preg_match('/\b(news|breaking|update|announcement)\b/i', $content_lower)) {
            return 'news';
        }
        
        if (preg_match('/\b(opinion|think|believe|perspective)\b/i', $content_lower)) {
            return 'opinion';
        }
        
        if (preg_match('/\b(case study|example|real world)\b/i', $content_lower)) {
            return 'case_study';
        }
        
        if (preg_match('/\b(summary|overview|recap)\b/i', $content_lower)) {
            return 'summary';
        }
        
        return 'reference'; // Default
    }
    
    /**
     * Classify target audience
     */
    private static function classify_target_audience($content, $title) {
        $content_lower = strtolower($content . ' ' . $title);
        
        if (preg_match('/\b(beginner|start|basic|introduction|new to)\b/i', $content_lower)) {
            return 'novice';
        }
        
        if (preg_match('/\b(advanced|expert|professional|complex|sophisticated)\b/i', $content_lower)) {
            return 'expert';
        }
        
        if (preg_match('/\b(business|corporate|enterprise|company)\b/i', $content_lower)) {
            return 'business';
        }
        
        if (preg_match('/\b(research|academic|university|study)\b/i', $content_lower)) {
            return 'academic';
        }
        
        return 'general'; // Default
    }
    
    /**
     * Classify narrative type
     */
    private static function classify_narrative_type($content, $title) {
        $content_lower = strtolower($content . ' ' . $title);
        
        // Count list indicators
        $list_count = preg_match_all('/^\s*[\d\-\*\+]\s/m', $content);
        
        if ($list_count > 3) {
            return 'listicle';
        }
        
        if (preg_match('/\b(step 1|first step|then|next|finally)\b/i', $content_lower)) {
            return 'step_by_step';
        }
        
        if (preg_match('/\b(question|answer|q:|a:)\b/i', $content_lower)) {
            return 'qa_format';
        }
        
        if (preg_match('/\b(interview|conversation|spoke with|asked)\b/i', $content_lower)) {
            return 'interview';
        }
        
        if (preg_match('/\b(vs|versus|compared|comparison|difference)\b/i', $content_lower)) {
            return 'comparison';
        }
        
        return 'article'; // Default
    }
    
    /**
     * Classify media type
     */
    private static function classify_media_type($content, $post) {
        // Count images in content
        $image_count = preg_match_all('/<img[^>]*>/i', $content);
        
        // Check for video embeds
        $video_count = preg_match_all('/<iframe[^>]*youtube|vimeo|video/i', $content);
        
        // Check for audio
        $audio_count = preg_match_all('/<audio|podcast|soundcloud/i', $content);
        
        if ($video_count > 0) {
            return 'video';
        }
        
        if ($audio_count > 0) {
            return 'audio';
        }
        
        if ($image_count > 3) {
            return 'image_heavy';
        }
        
        if ($image_count > 0 && ($video_count > 0 || $audio_count > 0)) {
            return 'mixed_media';
        }
        
        return 'text'; // Default
    }
    
    /**
     * Classify geographic sensitivity
     */
    private static function classify_geolocation($content, $categories, $tags) {
        $all_text = strtolower($content . ' ' . implode(' ', $categories) . ' ' . implode(' ', $tags));
        
        // Check for local indicators
        if (preg_match('/\b(local|city|town|neighborhood|nearby|area)\b/i', $all_text)) {
            return 'local';
        }
        
        // Check for country-specific indicators
        if (preg_match('/\b(usa|america|uk|canada|australia|germany|france|japan)\b/i', $all_text)) {
            return 'country_specific';
        }
        
        // Check for regional indicators
        if (preg_match('/\b(europe|asia|africa|america|region|continent)\b/i', $all_text)) {
            return 'regional';
        }
        
        return 'global'; // Default
    }
    
    /**
     * Classify market segments
     */
    private static function classify_market_segments($categories, $tags, $content) {
        $all_text = strtolower($content . ' ' . implode(' ', $categories) . ' ' . implode(' ', $tags));
        $segments = array();
        
        if (preg_match('/\b(business|b2b|enterprise|corporate)\b/i', $all_text)) {
            $segments[] = 'b2b';
        }
        
        if (preg_match('/\b(consumer|customer|b2c|retail)\b/i', $all_text)) {
            $segments[] = 'b2c';
        }
        
        if (preg_match('/\b(startup|entrepreneur|small business)\b/i', $all_text)) {
            $segments[] = 'startup';
        }
        
        if (preg_match('/\b(education|school|university|learning)\b/i', $all_text)) {
            $segments[] = 'education';
        }
        
        if (preg_match('/\b(health|medical|healthcare|doctor)\b/i', $all_text)) {
            $segments[] = 'healthcare';
        }
        
        if (preg_match('/\b(technology|tech|software|digital)\b/i', $all_text)) {
            $segments[] = 'technology';
        }
        
        if (preg_match('/\b(finance|money|investment|banking)\b/i', $all_text)) {
            $segments[] = 'finance';
        }
        
        return $segments;
    }
    
    /**
     * Save metadata for a post
     */
    public static function save_post_metadata($post_id, $metadata) {
        // Save individual fields
        foreach ($metadata as $key => $value) {
            update_post_meta($post_id, '_aiox_' . $key, $value);
        }
        
        // Save complete metadata as JSON for easy retrieval
        update_post_meta($post_id, '_aiox_metadata', json_encode($metadata));
        update_post_meta($post_id, '_aiox_metadata_count', count($metadata));
        
        // Save timestamp
        update_post_meta($post_id, '_aiox_metadata_updated', current_time('timestamp'));
    }
    
    /**
     * Get metadata for a post
     */
    public static function get_post_metadata($post_id) {
        $metadata = array();
        
        foreach (array_keys(self::$metadata_fields) as $field) {
            $value = get_post_meta($post_id, '_aiox_' . $field, true);
            
            if (empty($value)) {
                // Use default value if not set
                $field_def = self::$metadata_fields[$field];
                $metadata[$field] = $field_def['default'];
            } else {
                $metadata[$field] = $value;
            }
        }
        
        return $metadata;
    }
    
    /**
     * Get licensing metadata for a post
     */
    public static function get_licensing_metadata($post_id) {
        return array(
            'training_allowed' => get_post_meta($post_id, '_aiox_training_allowed', true) ?: get_option('aiox_tdm_training_allowed', false),
            'citation_required' => get_post_meta($post_id, '_aiox_citation_required', true) ?: get_option('aiox_tdm_citation_required', true),
            'attribution_string' => get_post_meta($post_id, '_aiox_attribution_string', true) ?: get_option('aiox_attribution_string', get_bloginfo('name')),
            'license_url' => get_post_meta($post_id, '_aiox_license_url', true) ?: '',
            'expiration_date' => get_post_meta($post_id, '_aiox_expiration_date', true) ?: ''
        );
    }
    
    /**
     * Delete metadata for a post
     */
    public static function delete_post_metadata($post_id) {
        // Delete individual fields
        foreach (array_keys(self::$metadata_fields) as $field) {
            delete_post_meta($post_id, '_aiox_' . $field);
        }
        
        // Delete complete metadata JSON
        delete_post_meta($post_id, '_aiox_metadata');
        delete_post_meta($post_id, '_aiox_metadata_count');
        delete_post_meta($post_id, '_aiox_metadata_updated');
        
        // Delete licensing metadata
        delete_post_meta($post_id, '_aiox_training_allowed');
        delete_post_meta($post_id, '_aiox_citation_required');
        delete_post_meta($post_id, '_aiox_attribution_string');
        delete_post_meta($post_id, '_aiox_license_url');
        delete_post_meta($post_id, '_aiox_expiration_date');
    }
}
