<?php
/**
 * AIOX Enhanced Modern Analytics - Core Analytics Engine
 * Comprehensive tracking for AI crawlers, badges, and well-known files
 * 
 * @package AIOX_Publisher_Suite_Pro
 * @version 5.5.0
 * 
 * CHANGELOG v5.5.0:
 * - Added company attribution for AI crawlers
 * - Expanded crawler detection from 5 to 55+ crawlers
 * - Enhanced pattern matching to prevent misclassification
 * - Added referrer-based detection for AI platforms
 * - Backward compatible with existing data
 */

if (!defined('ABSPATH')) {
    exit;
}

class AIOX_Modern_Analytics {

    private static $instance = null;
    private $table_analytics;
    private $table_crawlers;
    private $table_daily;
    private $table_badge_views;
    private $table_wellknown_hits;

    public static function get_instance() {
        if (self::$instance === null) {
            self::$instance = new self();
        }
        return self::$instance;
    }

    private function __construct() {
        global $wpdb;
        $this->table_analytics = $wpdb->prefix . 'aiox_server_analytics';
        $this->table_crawlers  = $wpdb->prefix . 'aiox_server_crawlers';
        $this->table_daily     = $wpdb->prefix . 'aiox_server_daily_stats';
        $this->table_badge_views = $wpdb->prefix . 'aiox_badge_views';
        $this->table_wellknown_hits = $wpdb->prefix . 'aiox_wellknown_hits';

        add_action('init', [__CLASS__, 'maybe_create_tables'], 0);
        add_action('template_redirect', [__CLASS__, 'log_request'], 1);
        
        
        // Badge tracking
        add_action('wp_ajax_nopriv_aiox_track_badge_view', [__CLASS__, 'track_badge_view']);
        add_action('wp_ajax_aiox_track_badge_view', [__CLASS__, 'track_badge_view']);
    }

    /**
     * Create all analytics tables
     * ENHANCED: Added 'company' field to crawlers table for AI company attribution
     * Only runs if tables don't exist yet to avoid expensive operations on every page load
     */
    public static function maybe_create_tables() {
        global $wpdb;
        
        // Quick check: if the main analytics table exists, assume all tables are created
        // This avoids expensive dbDelta calls on every page load
        $table_name = $wpdb->prefix . 'aiox_server_analytics';
        if ($wpdb->get_var("SHOW TABLES LIKE '{$table_name}'") === $table_name) {
            // Tables exist, check for column upgrades only
            self::maybe_add_company_column();
            self::maybe_add_visitor_type_columns();
            return;
        }
        
        $charset_collate = $wpdb->get_charset_collate();

        $sql_analytics = "CREATE TABLE IF NOT EXISTS {$wpdb->prefix}aiox_server_analytics (
            id BIGINT(20) UNSIGNED NOT NULL AUTO_INCREMENT,
            request_time DATETIME NOT NULL DEFAULT CURRENT_TIMESTAMP,
            endpoint VARCHAR(500) NOT NULL,
            request_data LONGTEXT,
            response_data LONGTEXT,
            status_code INT(11) DEFAULT 200,
            user_id BIGINT(20) UNSIGNED DEFAULT 0,
            visitor_type VARCHAR(50) DEFAULT 'unidentified',
            is_aiox_file TINYINT(1) DEFAULT 0,
            PRIMARY KEY (id),
            KEY idx_request_time (request_time),
            KEY idx_endpoint (endpoint(255)),
            KEY idx_visitor_type (visitor_type)
        ) $charset_collate;";

        // ENHANCED: Added 'company' field for AI company tracking
        $sql_crawlers = "CREATE TABLE IF NOT EXISTS {$wpdb->prefix}aiox_server_crawlers (
            id BIGINT(20) UNSIGNED AUTO_INCREMENT PRIMARY KEY,
            crawler_name VARCHAR(255) NOT NULL,
            crawler_type VARCHAR(50) DEFAULT 'unknown',
            company VARCHAR(100) DEFAULT 'Unknown',
            user_agent TEXT,
            ip VARCHAR(100),
            first_seen DATETIME NOT NULL DEFAULT CURRENT_TIMESTAMP,
            last_seen DATETIME NOT NULL DEFAULT CURRENT_TIMESTAMP,
            hits INT(11) DEFAULT 0,
            KEY idx_crawler_name (crawler_name),
            KEY idx_crawler_type (crawler_type),
            KEY idx_company (company),
            KEY idx_last_seen (last_seen)
        ) $charset_collate;";

        $sql_daily = "CREATE TABLE IF NOT EXISTS {$wpdb->prefix}aiox_server_daily_stats (
            id BIGINT(20) UNSIGNED AUTO_INCREMENT PRIMARY KEY,
            date DATE NOT NULL,
            total_visits INT(11) DEFAULT 0,
            total_crawlers INT(11) DEFAULT 0,
            ai_visits INT(11) DEFAULT 0,
            search_visits INT(11) DEFAULT 0,
            well_known_visits INT(11) DEFAULT 0,
            badge_views INT(11) DEFAULT 0,
            human_visits INT(11) DEFAULT 0,
            other_bot_visits INT(11) DEFAULT 0,
            unidentified_visits INT(11) DEFAULT 0,
            UNIQUE KEY idx_date (`date`)
        ) $charset_collate;";
        
        $sql_badge_views = "CREATE TABLE IF NOT EXISTS {$wpdb->prefix}aiox_badge_views (
            id BIGINT(20) UNSIGNED AUTO_INCREMENT PRIMARY KEY,
            post_id BIGINT(20) UNSIGNED NOT NULL,
            view_time DATETIME NOT NULL DEFAULT CURRENT_TIMESTAMP,
            viewer_type VARCHAR(50) DEFAULT 'human',
            user_agent TEXT,
            ip VARCHAR(100),
            referrer TEXT,
            KEY idx_post_id (post_id),
            KEY idx_view_time (view_time),
            KEY idx_viewer_type (viewer_type)
        ) $charset_collate;";
        
        $sql_wellknown_hits = "CREATE TABLE IF NOT EXISTS {$wpdb->prefix}aiox_wellknown_hits (
            id BIGINT(20) UNSIGNED AUTO_INCREMENT PRIMARY KEY,
            file_path VARCHAR(255) NOT NULL,
            hit_time DATETIME NOT NULL DEFAULT CURRENT_TIMESTAMP,
            crawler_name VARCHAR(255),
            crawler_type VARCHAR(50),
            user_agent TEXT,
            ip VARCHAR(100),
            KEY idx_file_path (file_path),
            KEY idx_hit_time (hit_time),
            KEY idx_crawler_type (crawler_type)
        ) $charset_collate;";

        require_once ABSPATH . 'wp-admin/includes/upgrade.php';
        dbDelta($sql_analytics);
        dbDelta($sql_crawlers);
        dbDelta($sql_daily);
        dbDelta($sql_badge_views);
        dbDelta($sql_wellknown_hits);
        
        // Add company column if upgrading from older version
        self::maybe_add_company_column();
        
        // Add visitor_type columns if upgrading
        self::maybe_add_visitor_type_columns();
    }
    
    /**
     * Add company column to crawlers table if it doesn't exist
     */
    private static function maybe_add_company_column() {
        global $wpdb;
        
        // Check if company column exists, add if not (for existing installations)
        $table_name = $wpdb->prefix . 'aiox_server_crawlers';
        
        // Check if table exists first
        if ($wpdb->get_var("SHOW TABLES LIKE '{$table_name}'") !== $table_name) {
            return;
        }
        
        $column_exists = $wpdb->get_results("SHOW COLUMNS FROM `{$table_name}` LIKE 'company'");
        
        if (empty($column_exists)) {
            $wpdb->query("ALTER TABLE `{$table_name}` ADD COLUMN `company` VARCHAR(100) DEFAULT 'Unknown' AFTER `crawler_type`");
            $wpdb->query("ALTER TABLE `{$table_name}` ADD INDEX `idx_company` (`company`)");
        }
    }
    
    /**
     * Add visitor_type and is_aiox_file columns to analytics table if they don't exist
     */
    private static function maybe_add_visitor_type_columns() {
        global $wpdb;
        
        $table_name = $wpdb->prefix . 'aiox_server_analytics';
        
        // Check if table exists first
        if ($wpdb->get_var("SHOW TABLES LIKE '{$table_name}'") !== $table_name) {
            return;
        }
        
        // Check for visitor_type column
        $visitor_type_exists = $wpdb->get_results("SHOW COLUMNS FROM `{$table_name}` LIKE 'visitor_type'");
        if (empty($visitor_type_exists)) {
            $wpdb->query("ALTER TABLE `{$table_name}` ADD COLUMN `visitor_type` VARCHAR(50) DEFAULT 'unidentified' AFTER `user_id`");
            $wpdb->query("ALTER TABLE `{$table_name}` ADD INDEX `idx_visitor_type` (`visitor_type`)");
        }
        
        // Check for is_aiox_file column
        $is_aiox_exists = $wpdb->get_results("SHOW COLUMNS FROM `{$table_name}` LIKE 'is_aiox_file'");
        if (empty($is_aiox_exists)) {
            $wpdb->query("ALTER TABLE `{$table_name}` ADD COLUMN `is_aiox_file` TINYINT(1) DEFAULT 0 AFTER `visitor_type`");
        }
    }

    /**
     * Log every request - IMPROVED VERSION
     */
    public static function log_request() {
        // Don't track admin pages
        if (is_admin()) {
            return;
        }
        
        // Don't track AJAX requests (multiple checks for reliability)
        if (defined('DOING_AJAX') && DOING_AJAX) {
            return;
        }
        
        if (function_exists('wp_doing_ajax') && wp_doing_ajax()) {
            return;
        }
        
        // Don't track cron
        if (defined('DOING_CRON') && DOING_CRON) {
            return;
        }
        
        // Don't track REST API requests from admin
        if (defined('REST_REQUEST') && REST_REQUEST) {
            return;
        }
        
        // Don't track if WordPress is not fully initialized
        if (!did_action('init')) {
            return;
        }
        
        // Get request URI early for additional checks
        $uri = $_SERVER['REQUEST_URI'] ?? '';
        
        // Don't track admin URLs (extra safety)
        if (strpos($uri, '/wp-admin') !== false || 
            strpos($uri, 'admin-ajax.php') !== false ||
            strpos($uri, 'wp-login.php') !== false ||
            strpos($uri, 'wp-cron.php') !== false ||
            strpos($uri, '/wp-json/') !== false) {
            return;
        }
        
        // Don't track WordPress heartbeat
        if (isset($_POST['action']) && $_POST['action'] === 'heartbeat') {
            return;
        }
        
        // Don't track feed requests (these inflate numbers)
        if (function_exists('is_feed') && is_feed()) {
            return;
        }
        
        // Don't track robots.txt and sitemap
        if (strpos($uri, 'robots.txt') !== false || 
            strpos($uri, 'sitemap') !== false ||
            strpos($uri, '.xml') !== false) {
            return;
        }
        
        // Don't track logged-in admin users (they inflate stats during testing/development)
        if (is_user_logged_in() && current_user_can('manage_options')) {
            return;
        }

        global $wpdb;
        
        // Check if analytics table exists before attempting to log
        $table_name = $wpdb->prefix . 'aiox_server_analytics';
        if ($wpdb->get_var("SHOW TABLES LIKE '{$table_name}'") !== $table_name) {
            return; // Table doesn't exist yet, skip logging
        }
        
        $ip = $_SERVER['REMOTE_ADDR'] ?? 'unknown';
        $agent = $_SERVER['HTTP_USER_AGENT'] ?? 'unknown';
        $referrer = $_SERVER['HTTP_REFERER'] ?? '';
        $method = $_SERVER['REQUEST_METHOD'] ?? 'GET';
        $timestamp = current_time('mysql');

        // Detect crawler/visitor type
        $crawler_info = self::detect_crawler($agent, $referrer);
        $visitor_type = $crawler_info['type'];
        
        // Log visitor detection for debugging (only when WP_DEBUG or AIOX_DEBUG_VISITORS is true)
        self::log_visitor_detection($agent, $referrer, $crawler_info, $ip);
        
        $is_ai = $visitor_type === 'ai';
        $is_search = $visitor_type === 'search';
        $is_human = $visitor_type === 'human';
        $is_other_bot = in_array($visitor_type, ['social', 'seo', 'monitoring', 'feed', 'security', 'archive', 'automation', 'infrastructure', 'tool', 'bot']);
        $is_unidentified = $visitor_type === 'unidentified';
        
        // Check if this is an AIOX system file request
        $is_aiox_file = self::is_aiox_system_file($uri);
        
        // Legacy compatibility - also track .well-known for backward compatibility
        $is_well_known = $is_aiox_file;

        // Log to main analytics table
        $inserted = $wpdb->insert(
            $wpdb->prefix . 'aiox_server_analytics',
            [
                'request_time' => $timestamp,
                'endpoint' => esc_url_raw($uri),
                'request_data' => maybe_serialize([
                    'ip' => $ip,
                    'agent' => $agent,
                    'referrer' => $referrer,
                    'method' => $method,
                    'crawler_type' => $crawler_info['type'],
                    'crawler_name' => $crawler_info['name']
                ]),
                'response_data' => '',
                'status_code' => 200,
                'user_id' => get_current_user_id(),
                'visitor_type' => $visitor_type,
                'is_aiox_file' => $is_aiox_file ? 1 : 0
            ],
            ['%s', '%s', '%s', '%s', '%d', '%d', '%s', '%d']
        );

        if ($inserted === false) {
            error_log('AIOX Analytics: Failed to insert analytics record - ' . $wpdb->last_error);
            return;
        }

        // Track well-known file access
        if ($is_well_known) {
            self::log_wellknown_hit($uri, $crawler_info, $ip, $agent, $timestamp);
        }

        // Update daily stats with all visitor types
        self::update_daily_stats($is_ai, $is_search, $is_human, $is_other_bot, $is_unidentified, $is_well_known);

        // Update crawler stats for all identified non-human visitors
        if (!$is_human && !$is_unidentified) {
            self::update_crawler_stats($crawler_info, $ip, $agent, $timestamp);
        }
        
        // Track badge views server-side ONLY for AI crawlers (they don't execute JavaScript)
        // Human visitors will be tracked via JavaScript when they click to open the badge
        if ($is_ai && is_singular() && AIOX_Utils::is_feature_enabled('ai_badge') && get_option('aiox_badge_enabled', true)) {
            $post_id = get_the_ID();
            if ($post_id) {
                // Check if this post has AI-processed content with badge
                $processed_data = get_post_meta($post_id, '_aiox_processed_data', true);
                $is_ai_enhanced = $processed_data && 
                                 isset($processed_data['processing_method']) && 
                                 $processed_data['processing_method'] !== 'basic' &&
                                 isset($processed_data['qa_pairs']) && 
                                 !empty($processed_data['qa_pairs']);
                
                if ($is_ai_enhanced) {
                    // Log badge view as 'ai' type
                    $wpdb->insert(
                        $wpdb->prefix . 'aiox_badge_views',
                        [
                            'post_id' => $post_id,
                            'view_time' => $timestamp,
                            'viewer_type' => 'ai',
                            'user_agent' => $agent,
                            'ip' => $ip,
                            'referrer' => $referrer
                        ],
                        ['%d', '%s', '%s', '%s', '%s', '%s']
                    );
                    
                    // Also update daily stats badge_views counter
                    $today = current_time('Y-m-d');
                    $wpdb->query($wpdb->prepare(
                        "UPDATE {$wpdb->prefix}aiox_server_daily_stats 
                         SET badge_views = badge_views + 1 
                         WHERE date = %s",
                        $today
                    ));
                }
            }
        }
        
    }

    /**
     * Check if URI is an AIOX system file
     * Tracks all AIOX-generated files for AI discovery
     */
    private static function is_aiox_system_file($uri) {
        $uri_lower = strtolower($uri);
        
        // AIOX files in .well-known directory
        $aiox_wellknown_files = [
            '/.well-known/aio.json',           // AIOX Manifest
            '/.well-known/aiox.ndjson',        // Delta Feed
            '/.well-known/aiox.aiox',          // AIOX Pack
            '/.well-known/tdmrep.json',        // TDM-REP File
            '/.well-known/aiox-pubkey.pem',    // Public Key
            '/.well-known/aiox-proof.json',    // Proof File
            '/.well-known/ingredients.json',   // Ingredients
        ];
        
        // Check exact matches first
        foreach ($aiox_wellknown_files as $file) {
            if (strpos($uri_lower, $file) !== false) {
                return true;
            }
        }
        
        // AIOX dynamic files (pattern matches)
        $aiox_patterns = [
            '/.well-known/ai-info-',           // ai-info-{post_id}.json
            '/.well-known/aiox-post-',         // aiox-post-{post_id}.json
            '/.well-known/capsule_',           // capsule_{id}.json
            '/.well-known/capsules/',          // capsules/{id}.json
        ];
        
        foreach ($aiox_patterns as $pattern) {
            if (strpos($uri_lower, $pattern) !== false) {
                return true;
            }
        }
        
        // Root-level AIOX files
        if (preg_match('#^/robots\.txt#i', $uri)) {
            // Only count robots.txt if it contains AIOX rules (we track all for simplicity)
            return true;
        }
        
        return false;
    }
    
    /**
     * Get friendly name for AIOX system file
     */
    private static function get_aiox_file_name($uri) {
        $uri_lower = strtolower($uri);
        
        $file_names = [
            'aio.json' => 'AIOX Manifest',
            'aiox.ndjson' => 'Delta Feed',
            'aiox.aiox' => 'AIOX Pack',
            'tdmrep.json' => 'TDM-REP Policy',
            'aiox-pubkey.pem' => 'Public Key',
            'aiox-proof.json' => 'Proof File',
            'ingredients.json' => 'Ingredients',
            'robots.txt' => 'Robots.txt',
        ];
        
        foreach ($file_names as $file => $name) {
            if (strpos($uri_lower, $file) !== false) {
                return $name;
            }
        }
        
        // Dynamic files
        if (strpos($uri_lower, 'ai-info-') !== false) {
            return 'AI Info (Post)';
        }
        if (strpos($uri_lower, 'aiox-post-') !== false) {
            return 'AIOX Post Data';
        }
        if (strpos($uri_lower, 'capsule') !== false) {
            return 'Capsule';
        }
        
        return 'AIOX File';
    }

    /**
     * Log AIOX system file hits (renamed from log_wellknown_hit)
     */
    private static function log_wellknown_hit($uri, $crawler_info, $ip, $agent, $timestamp) {
        global $wpdb;
        
        $wpdb->insert("{$wpdb->prefix}aiox_wellknown_hits", [
            'file_path' => $uri,
            'hit_time' => $timestamp,
            'crawler_name' => $crawler_info['name'],
            'crawler_type' => $crawler_info['type'],
            'user_agent' => $agent,
            'ip' => $ip
        ], ['%s', '%s', '%s', '%s', '%s', '%s']);
    }

    /**
     * Track badge views (via AJAX) - Called when human clicks to open the badge
     */
    public static function track_badge_view() {
        global $wpdb;
        
        $post_id = intval($_POST['post_id'] ?? 0);
        $agent = $_SERVER['HTTP_USER_AGENT'] ?? '';
        $ip = $_SERVER['REMOTE_ADDR'] ?? 'unknown';
        $referrer = $_SERVER['HTTP_REFERER'] ?? '';
        
        if (!$post_id) {
            wp_send_json_error('Invalid post ID');
        }
        
        $crawler_info = self::detect_crawler($agent, $referrer);
        // If called via AJAX click, it's definitely a human (bots don't click)
        // But we still detect to categorize if it's an AI referral
        $viewer_type = ($crawler_info['type'] === 'ai') ? 'ai' : 'human';
        
        $wpdb->insert("{$wpdb->prefix}aiox_badge_views", [
            'post_id' => $post_id,
            'view_time' => current_time('mysql'),
            'viewer_type' => $viewer_type,
            'user_agent' => $agent,
            'ip' => $ip,
            'referrer' => $referrer
        ], ['%d', '%s', '%s', '%s', '%s', '%s']);
        
        // Update daily stats
        $today = current_time('Y-m-d');
        $wpdb->query($wpdb->prepare(
            "UPDATE {$wpdb->prefix}aiox_server_daily_stats 
             SET badge_views = badge_views + 1 
             WHERE date = %s",
            $today
        ));
        
        wp_send_json_success(['message' => 'Badge view tracked']);
    }

    /**
     * Detect crawler type with comprehensive coverage
     * ENHANCED v5.6.0: Added human browser detection, more bots, monitoring services
     * 
     * @param string $ua User agent string
     * @param string $referrer Referrer URL
     * @return array Crawler info ['name' => string, 'type' => string, 'company' => string]
     */
    private static function detect_crawler($ua, $referrer) {
        $ua = strtolower($ua);
        $ref = strtolower($referrer);

        // Priority-ordered crawler patterns (most specific first)
        $crawlers = [
            // === OpenAI Crawlers ===
            ['pattern' => 'chatgpt agent', 'name' => 'ChatGPT Agent', 'type' => 'ai', 'company' => 'OpenAI'],
            ['pattern' => 'chatgpt-user', 'name' => 'ChatGPT User', 'type' => 'ai', 'company' => 'OpenAI'],
            ['pattern' => 'oai-searchbot', 'name' => 'OpenAI SearchBot', 'type' => 'ai', 'company' => 'OpenAI'],
            ['pattern' => 'gptbot', 'name' => 'OpenAI GPTBot', 'type' => 'ai', 'company' => 'OpenAI'],
            ['pattern' => 'chatgpt', 'name' => 'ChatGPT', 'type' => 'ai', 'company' => 'OpenAI'],
            
            // === Anthropic Crawlers ===
            ['pattern' => 'claude-searchbot', 'name' => 'Claude SearchBot', 'type' => 'ai', 'company' => 'Anthropic'],
            ['pattern' => 'claude-user', 'name' => 'Claude User', 'type' => 'ai', 'company' => 'Anthropic'],
            ['pattern' => 'claude-web', 'name' => 'Claude Web', 'type' => 'ai', 'company' => 'Anthropic'],
            ['pattern' => 'claudebot', 'name' => 'ClaudeBot', 'type' => 'ai', 'company' => 'Anthropic'],
            ['pattern' => 'anthropic-ai', 'name' => 'Anthropic AI', 'type' => 'ai', 'company' => 'Anthropic'],
            
            // === Google AI ===
            ['pattern' => 'google-cloudvertexbot', 'name' => 'Google CloudVertexBot', 'type' => 'ai', 'company' => 'Google'],
            ['pattern' => 'google-extended', 'name' => 'Google Extended', 'type' => 'ai', 'company' => 'Google'],
            ['pattern' => 'geminibot', 'name' => 'Google Gemini', 'type' => 'ai', 'company' => 'Google'],
            ['pattern' => 'gemini-deep-research', 'name' => 'Gemini Deep Research', 'type' => 'ai', 'company' => 'Google'],
            ['pattern' => 'googleother', 'name' => 'GoogleOther', 'type' => 'ai', 'company' => 'Google'],
            
            // === Meta/Facebook AI ===
            ['pattern' => 'meta-webindexer', 'name' => 'Meta WebIndexer', 'type' => 'ai', 'company' => 'Meta'],
            ['pattern' => 'meta-externalfetcher', 'name' => 'Meta ExternalFetcher', 'type' => 'ai', 'company' => 'Meta'],
            ['pattern' => 'meta-externalagent', 'name' => 'Meta ExternalAgent', 'type' => 'ai', 'company' => 'Meta'],
            
            // === Perplexity ===
            ['pattern' => 'perplexity-user', 'name' => 'Perplexity User', 'type' => 'ai', 'company' => 'Perplexity'],
            ['pattern' => 'perplexitybot', 'name' => 'Perplexity Bot', 'type' => 'ai', 'company' => 'Perplexity'],
            ['pattern' => 'perplexity', 'name' => 'Perplexity', 'type' => 'ai', 'company' => 'Perplexity'],
            
            // === Apple ===
            ['pattern' => 'applebot-extended', 'name' => 'Applebot Extended', 'type' => 'ai', 'company' => 'Apple'],
            ['pattern' => 'applebot', 'name' => 'Applebot', 'type' => 'search', 'company' => 'Apple'],
            
            // === Amazon ===
            ['pattern' => 'amazonbot', 'name' => 'Amazonbot', 'type' => 'ai', 'company' => 'Amazon'],
            
            // === ByteDance/TikTok ===
            ['pattern' => 'tiktokspider', 'name' => 'TikTok Spider', 'type' => 'ai', 'company' => 'ByteDance'],
            ['pattern' => 'bytespider', 'name' => 'Bytespider', 'type' => 'ai', 'company' => 'ByteDance'],
            
            // === Cohere ===
            ['pattern' => 'cohere-training-data-crawler', 'name' => 'Cohere Training Crawler', 'type' => 'ai', 'company' => 'Cohere'],
            ['pattern' => 'cohere-ai', 'name' => 'Cohere AI', 'type' => 'ai', 'company' => 'Cohere'],
            ['pattern' => 'cohere', 'name' => 'Cohere', 'type' => 'ai', 'company' => 'Cohere'],
            
            // === Other AI Companies ===
            ['pattern' => 'mistralai-user', 'name' => 'Mistral AI User', 'type' => 'ai', 'company' => 'Mistral AI'],
            ['pattern' => 'youbot', 'name' => 'YouBot', 'type' => 'ai', 'company' => 'You.com'],
            ['pattern' => 'pangubot', 'name' => 'PanguBot', 'type' => 'ai', 'company' => 'Huawei'],
            ['pattern' => 'petalbot', 'name' => 'PetalBot', 'type' => 'ai', 'company' => 'Huawei'],
            ['pattern' => 'duckassistbot', 'name' => 'DuckAssistBot', 'type' => 'ai', 'company' => 'DuckDuckGo'],
            
            // === Data Collection & Training ===
            ['pattern' => 'ccbot', 'name' => 'Common Crawl', 'type' => 'ai', 'company' => 'Common Crawl'],
            ['pattern' => 'omgilibot', 'name' => 'Omgilibot', 'type' => 'ai', 'company' => 'Webz.io'],
            ['pattern' => 'omgili', 'name' => 'Omgili', 'type' => 'ai', 'company' => 'Webz.io'],
            ['pattern' => 'diffbot', 'name' => 'Diffbot', 'type' => 'ai', 'company' => 'Diffbot'],
            ['pattern' => 'imagesiftbot', 'name' => 'ImagesiftBot', 'type' => 'ai', 'company' => 'The Hive'],
            ['pattern' => 'ai2bot-dolma', 'name' => 'AI2Bot Dolma', 'type' => 'ai', 'company' => 'Allen Institute'],
            ['pattern' => 'ai2bot', 'name' => 'AI2Bot', 'type' => 'ai', 'company' => 'Allen Institute'],
            ['pattern' => 'icc-crawler', 'name' => 'ICC Crawler', 'type' => 'ai', 'company' => 'NICT Japan'],
            ['pattern' => 'kangaroo bot', 'name' => 'Kangaroo Bot', 'type' => 'ai', 'company' => 'Kangaroo LLM'],
            ['pattern' => 'friendlycrawler', 'name' => 'FriendlyCrawler', 'type' => 'ai', 'company' => 'Brandwatch'],
            
            // === Traditional Search Engines ===
            ['pattern' => 'googlebot-image', 'name' => 'Google Images', 'type' => 'search', 'company' => 'Google'],
            ['pattern' => 'googlebot-video', 'name' => 'Google Video', 'type' => 'search', 'company' => 'Google'],
            ['pattern' => 'googlebot-news', 'name' => 'Google News', 'type' => 'search', 'company' => 'Google'],
            ['pattern' => 'googlebot', 'name' => 'Google', 'type' => 'search', 'company' => 'Google'],
            ['pattern' => 'google-inspectiontool', 'name' => 'Google Inspection', 'type' => 'search', 'company' => 'Google'],
            ['pattern' => 'adsbot-google', 'name' => 'Google AdsBot', 'type' => 'search', 'company' => 'Google'],
            ['pattern' => 'mediapartners-google', 'name' => 'Google AdSense', 'type' => 'search', 'company' => 'Google'],
            ['pattern' => 'bingbot', 'name' => 'Bing', 'type' => 'search', 'company' => 'Microsoft'],
            ['pattern' => 'msnbot', 'name' => 'MSN Bot', 'type' => 'search', 'company' => 'Microsoft'],
            ['pattern' => 'duckduckbot', 'name' => 'DuckDuckGo', 'type' => 'search', 'company' => 'DuckDuckGo'],
            ['pattern' => 'baiduspider', 'name' => 'Baidu', 'type' => 'search', 'company' => 'Baidu'],
            ['pattern' => 'yandexbot', 'name' => 'Yandex', 'type' => 'search', 'company' => 'Yandex'],
            ['pattern' => 'slurp', 'name' => 'Yahoo', 'type' => 'search', 'company' => 'Yahoo'],
            ['pattern' => 'sogou', 'name' => 'Sogou', 'type' => 'search', 'company' => 'Sogou'],
            ['pattern' => 'exabot', 'name' => 'Exabot', 'type' => 'search', 'company' => 'Exalead'],
            ['pattern' => 'ia_archiver', 'name' => 'Alexa', 'type' => 'search', 'company' => 'Amazon'],
            
            // === Social Media Crawlers ===
            ['pattern' => 'facebookexternalhit', 'name' => 'Facebook', 'type' => 'social', 'company' => 'Meta'],
            ['pattern' => 'facebookbot', 'name' => 'Facebook', 'type' => 'social', 'company' => 'Meta'],
            ['pattern' => 'twitterbot', 'name' => 'Twitter', 'type' => 'social', 'company' => 'Twitter/X'],
            ['pattern' => 'linkedinbot', 'name' => 'LinkedIn', 'type' => 'social', 'company' => 'LinkedIn'],
            ['pattern' => 'pinterestbot', 'name' => 'Pinterest', 'type' => 'social', 'company' => 'Pinterest'],
            ['pattern' => 'slackbot', 'name' => 'Slack', 'type' => 'social', 'company' => 'Salesforce'],
            ['pattern' => 'discordbot', 'name' => 'Discord', 'type' => 'social', 'company' => 'Discord'],
            ['pattern' => 'telegrambot', 'name' => 'Telegram', 'type' => 'social', 'company' => 'Telegram'],
            ['pattern' => 'whatsapp', 'name' => 'WhatsApp', 'type' => 'social', 'company' => 'Meta'],
            ['pattern' => 'snapchat', 'name' => 'Snapchat', 'type' => 'social', 'company' => 'Snap'],
            ['pattern' => 'redditbot', 'name' => 'Reddit', 'type' => 'social', 'company' => 'Reddit'],
            
            // === SEO & Marketing Tools ===
            ['pattern' => 'semrushbot', 'name' => 'SEMrush', 'type' => 'seo', 'company' => 'SEMrush'],
            ['pattern' => 'ahrefsbot', 'name' => 'Ahrefs', 'type' => 'seo', 'company' => 'Ahrefs'],
            ['pattern' => 'mj12bot', 'name' => 'Majestic', 'type' => 'seo', 'company' => 'Majestic'],
            ['pattern' => 'dotbot', 'name' => 'Moz', 'type' => 'seo', 'company' => 'Moz'],
            ['pattern' => 'rogerbot', 'name' => 'Moz Roger', 'type' => 'seo', 'company' => 'Moz'],
            ['pattern' => 'screaming frog', 'name' => 'Screaming Frog', 'type' => 'seo', 'company' => 'Screaming Frog'],
            ['pattern' => 'sistrix', 'name' => 'Sistrix', 'type' => 'seo', 'company' => 'Sistrix'],
            ['pattern' => 'blexbot', 'name' => 'BLEXBot', 'type' => 'seo', 'company' => 'WebMeUp'],
            ['pattern' => 'seokicks', 'name' => 'SEOkicks', 'type' => 'seo', 'company' => 'SEOkicks'],
            ['pattern' => 'serpstatbot', 'name' => 'Serpstat', 'type' => 'seo', 'company' => 'Serpstat'],
            ['pattern' => 'linkdexbot', 'name' => 'Linkdex', 'type' => 'seo', 'company' => 'Linkdex'],
            
            // === Monitoring & Uptime Services ===
            ['pattern' => 'uptimerobot', 'name' => 'UptimeRobot', 'type' => 'monitoring', 'company' => 'UptimeRobot'],
            ['pattern' => 'pingdom', 'name' => 'Pingdom', 'type' => 'monitoring', 'company' => 'SolarWinds'],
            ['pattern' => 'statuscake', 'name' => 'StatusCake', 'type' => 'monitoring', 'company' => 'StatusCake'],
            ['pattern' => 'site24x7', 'name' => 'Site24x7', 'type' => 'monitoring', 'company' => 'Zoho'],
            ['pattern' => 'newrelic', 'name' => 'New Relic', 'type' => 'monitoring', 'company' => 'New Relic'],
            ['pattern' => 'datadoghq', 'name' => 'Datadog', 'type' => 'monitoring', 'company' => 'Datadog'],
            ['pattern' => 'gtmetrix', 'name' => 'GTmetrix', 'type' => 'monitoring', 'company' => 'GTmetrix'],
            ['pattern' => 'pagespeed', 'name' => 'PageSpeed', 'type' => 'monitoring', 'company' => 'Google'],
            ['pattern' => 'webpagetest', 'name' => 'WebPageTest', 'type' => 'monitoring', 'company' => 'WebPageTest'],
            ['pattern' => 'catchpoint', 'name' => 'Catchpoint', 'type' => 'monitoring', 'company' => 'Catchpoint'],
            
            // === Feed Readers & Aggregators ===
            ['pattern' => 'feedly', 'name' => 'Feedly', 'type' => 'feed', 'company' => 'Feedly'],
            ['pattern' => 'feedburner', 'name' => 'FeedBurner', 'type' => 'feed', 'company' => 'Google'],
            ['pattern' => 'feedfetcher', 'name' => 'FeedFetcher', 'type' => 'feed', 'company' => 'Google'],
            ['pattern' => 'feedvalidator', 'name' => 'Feed Validator', 'type' => 'feed', 'company' => 'W3C'],
            ['pattern' => 'feedpress', 'name' => 'FeedPress', 'type' => 'feed', 'company' => 'FeedPress'],
            ['pattern' => 'inoreader', 'name' => 'Inoreader', 'type' => 'feed', 'company' => 'Inoreader'],
            ['pattern' => 'netvibes', 'name' => 'Netvibes', 'type' => 'feed', 'company' => 'Netvibes'],
            ['pattern' => 'newsblur', 'name' => 'NewsBlur', 'type' => 'feed', 'company' => 'NewsBlur'],
            ['pattern' => 'theoldreader', 'name' => 'The Old Reader', 'type' => 'feed', 'company' => 'The Old Reader'],
            
            // === Security & Vulnerability Scanners ===
            ['pattern' => 'nessus', 'name' => 'Nessus', 'type' => 'security', 'company' => 'Tenable'],
            ['pattern' => 'qualys', 'name' => 'Qualys', 'type' => 'security', 'company' => 'Qualys'],
            ['pattern' => 'sucuri', 'name' => 'Sucuri', 'type' => 'security', 'company' => 'GoDaddy'],
            ['pattern' => 'wordfence', 'name' => 'Wordfence', 'type' => 'security', 'company' => 'Defiant'],
            ['pattern' => 'wpscan', 'name' => 'WPScan', 'type' => 'security', 'company' => 'WPScan'],
            ['pattern' => 'nikto', 'name' => 'Nikto', 'type' => 'security', 'company' => 'Nikto'],
            ['pattern' => 'acunetix', 'name' => 'Acunetix', 'type' => 'security', 'company' => 'Invicti'],
            ['pattern' => 'burpsuite', 'name' => 'Burp Suite', 'type' => 'security', 'company' => 'PortSwigger'],
            
            // === Archive & Research ===
            ['pattern' => 'archive.org_bot', 'name' => 'Internet Archive', 'type' => 'archive', 'company' => 'Internet Archive'],
            ['pattern' => 'ia_archiver', 'name' => 'Internet Archive', 'type' => 'archive', 'company' => 'Internet Archive'],
            ['pattern' => 'wayback', 'name' => 'Wayback Machine', 'type' => 'archive', 'company' => 'Internet Archive'],
            
            // === Headless Browsers & Automation ===
            ['pattern' => 'headlesschrome', 'name' => 'Headless Chrome', 'type' => 'automation', 'company' => 'Google'],
            ['pattern' => 'phantomjs', 'name' => 'PhantomJS', 'type' => 'automation', 'company' => 'PhantomJS'],
            ['pattern' => 'selenium', 'name' => 'Selenium', 'type' => 'automation', 'company' => 'Selenium'],
            ['pattern' => 'puppeteer', 'name' => 'Puppeteer', 'type' => 'automation', 'company' => 'Google'],
            ['pattern' => 'playwright', 'name' => 'Playwright', 'type' => 'automation', 'company' => 'Microsoft'],
            
            // === CDN & Infrastructure ===
            ['pattern' => 'cloudflare', 'name' => 'Cloudflare', 'type' => 'infrastructure', 'company' => 'Cloudflare'],
            ['pattern' => 'fastly', 'name' => 'Fastly', 'type' => 'infrastructure', 'company' => 'Fastly'],
            ['pattern' => 'akamai', 'name' => 'Akamai', 'type' => 'infrastructure', 'company' => 'Akamai'],
            
            // === Other Known Bots ===
            ['pattern' => 'wget', 'name' => 'Wget', 'type' => 'tool', 'company' => 'GNU'],
            ['pattern' => 'curl/', 'name' => 'cURL', 'type' => 'tool', 'company' => 'cURL'],
            ['pattern' => 'python-requests', 'name' => 'Python Requests', 'type' => 'tool', 'company' => 'Python'],
            ['pattern' => 'python-urllib', 'name' => 'Python URLlib', 'type' => 'tool', 'company' => 'Python'],
            ['pattern' => 'go-http-client', 'name' => 'Go HTTP Client', 'type' => 'tool', 'company' => 'Go'],
            ['pattern' => 'java/', 'name' => 'Java HTTP Client', 'type' => 'tool', 'company' => 'Oracle'],
            ['pattern' => 'node-fetch', 'name' => 'Node Fetch', 'type' => 'tool', 'company' => 'Node.js'],
            ['pattern' => 'axios/', 'name' => 'Axios', 'type' => 'tool', 'company' => 'Node.js'],
            ['pattern' => 'libwww-perl', 'name' => 'LWP Perl', 'type' => 'tool', 'company' => 'Perl'],
            ['pattern' => 'scrapy', 'name' => 'Scrapy', 'type' => 'tool', 'company' => 'Scrapy'],
            
            // === Generic Bot Patterns ===
            ['pattern' => 'bot/', 'name' => 'Generic Bot', 'type' => 'bot', 'company' => 'Unknown'],
            ['pattern' => 'crawler', 'name' => 'Generic Crawler', 'type' => 'bot', 'company' => 'Unknown'],
            ['pattern' => 'spider', 'name' => 'Generic Spider', 'type' => 'bot', 'company' => 'Unknown'],
            ['pattern' => 'scraper', 'name' => 'Generic Scraper', 'type' => 'bot', 'company' => 'Unknown'],
            ['pattern' => 'robot', 'name' => 'Generic Robot', 'type' => 'bot', 'company' => 'Unknown'],
        ];

        // Check user agent for crawler patterns
        foreach ($crawlers as $crawler) {
            if (strpos($ua, $crawler['pattern']) !== false) {
                return $crawler;
            }
        }
        
        // === ADDITIONAL BOT INDICATORS ===
        // These patterns indicate non-browser traffic even if they have browser-like UAs
        $bot_indicators = [
            // HTTP clients that pretend to be browsers
            'http://', 'https://', '+http', 
            // Common bot terms not caught above
            'fetch', 'check', 'scan', 'monitor', 'index', 'archive',
            'link', 'preview', 'snapshot', 'capture', 'render',
            // Programming language indicators
            'python', 'perl', 'ruby', 'php/', 'java/', 'node',
            // Known bot services
            'pagespeed', 'lighthouse', 'gtmetrix', 'webpagetest',
            'pingdom', 'uptime', 'statuscake', 'newrelic',
            // Headless/Automation
            'headless', 'phantom', 'zombie', 'splash',
            // Preview/Embedding services  
            'embed', 'oembed', 'iframely', 'unfurl',
            // Validation/Testing
            'validator', 'checker', 'tester', 'probe',
            // Generic suspicious
            'compatible;)', // Empty compatibility list
            'compatible; )', 
        ];
        
        foreach ($bot_indicators as $indicator) {
            if (strpos($ua, $indicator) !== false) {
                return [
                    'name' => 'Detected Bot',
                    'type' => 'bot',
                    'company' => 'Unknown',
                    'indicator' => $indicator
                ];
            }
        }
        
        // Check for suspicious short user agents (real browsers have long UAs)
        if (strlen($ua) < 50) {
            return [
                'name' => 'Short UA',
                'type' => 'unidentified',
                'company' => 'Unknown',
                'reason' => 'ua_too_short'
            ];
        }

        // Check referrer for AI platforms
        $ai_referrers = [
            'chat.openai.com' => ['name' => 'ChatGPT User', 'type' => 'ai', 'company' => 'OpenAI'],
            'chatgpt.com' => ['name' => 'ChatGPT User', 'type' => 'ai', 'company' => 'OpenAI'],
            'claude.ai' => ['name' => 'Claude User', 'type' => 'ai', 'company' => 'Anthropic'],
            'perplexity.ai' => ['name' => 'Perplexity User', 'type' => 'ai', 'company' => 'Perplexity'],
            'you.com' => ['name' => 'You.com User', 'type' => 'ai', 'company' => 'You.com'],
            'gemini.google.com' => ['name' => 'Gemini User', 'type' => 'ai', 'company' => 'Google'],
            'bard.google.com' => ['name' => 'Bard User', 'type' => 'ai', 'company' => 'Google'],
            'copilot.microsoft.com' => ['name' => 'Copilot User', 'type' => 'ai', 'company' => 'Microsoft'],
            'bing.com/chat' => ['name' => 'Bing Chat User', 'type' => 'ai', 'company' => 'Microsoft']
        ];

        foreach ($ai_referrers as $domain => $info) {
            if (strpos($ref, $domain) !== false) {
                return $info;
            }
        }
        
        // === REFINED HUMAN DETECTION ===
        // A real human browser must have:
        // 1. Mozilla/5.0 prefix (all modern browsers)
        // 2. A known browser identifier
        // 3. A valid operating system
        
        $has_mozilla = strpos($ua, 'mozilla/5.0') !== false;
        
        // Check for valid OS
        $has_valid_os = (
            strpos($ua, 'windows nt') !== false ||
            strpos($ua, 'macintosh') !== false ||
            strpos($ua, 'mac os x') !== false ||
            strpos($ua, 'linux') !== false ||
            strpos($ua, 'android') !== false ||
            strpos($ua, 'iphone') !== false ||
            strpos($ua, 'ipad') !== false ||
            strpos($ua, 'cros') !== false ||  // Chrome OS
            strpos($ua, 'x11') !== false       // Unix/Linux
        );
        
        // Check for rendering engine (real browsers have these)
        $has_engine = (
            strpos($ua, 'applewebkit/') !== false ||
            strpos($ua, 'gecko/') !== false ||
            strpos($ua, 'trident/') !== false ||  // IE
            strpos($ua, 'presto/') !== false      // Old Opera
        );
        
        // Known browser patterns
        $browser_patterns = [
            'edg/' => 'Edge',           // Check Edge first (contains chrome)
            'opr/' => 'Opera',          // Check Opera (contains chrome)  
            'vivaldi/' => 'Vivaldi',    // Check Vivaldi (contains chrome)
            'brave' => 'Brave',         // Check Brave (contains chrome)
            'chrome/' => 'Chrome',
            'firefox/' => 'Firefox',
            'safari/' => 'Safari',
            'msie ' => 'Internet Explorer',
            'trident/' => 'Internet Explorer',
            'samsungbrowser/' => 'Samsung Browser',
            'ucbrowser/' => 'UC Browser',
            'yabrowser/' => 'Yandex Browser',
        ];
        
        $detected_browser = null;
        foreach ($browser_patterns as $pattern => $browser_name) {
            if (strpos($ua, $pattern) !== false) {
                $detected_browser = $browser_name;
                break;
            }
        }
        
        // For Safari on iOS/Mac without explicit "safari/" - check AppleWebKit + Apple device
        if (!$detected_browser && strpos($ua, 'applewebkit/') !== false) {
            if (strpos($ua, 'iphone') !== false || strpos($ua, 'ipad') !== false || strpos($ua, 'macintosh') !== false) {
                // Make sure it's not Chrome/Firefox on Apple device
                if (strpos($ua, 'chrome/') === false && strpos($ua, 'firefox/') === false && strpos($ua, 'crios/') === false && strpos($ua, 'fxios/') === false) {
                    $detected_browser = 'Safari';
                }
            }
        }
        
        // Classify as human if all conditions met
        if ($has_mozilla && $has_valid_os && $has_engine && $detected_browser) {
            // Determine device type
            $is_mobile = strpos($ua, 'mobile') !== false || strpos($ua, 'android') !== false && strpos($ua, 'mobile') !== false;
            $is_tablet = strpos($ua, 'ipad') !== false || strpos($ua, 'tablet') !== false;
            
            $device_type = 'Desktop';
            if ($is_tablet) $device_type = 'Tablet';
            elseif ($is_mobile || strpos($ua, 'iphone') !== false) $device_type = 'Mobile';
            
            return [
                'name' => $detected_browser,
                'type' => 'human',
                'company' => 'Browser Visitor',
                'device' => $device_type
            ];
        }
        
        // Mobile browsers fallback - for mobile UAs that might not match above
        if (strpos($ua, 'mobile') !== false && $has_mozilla && $has_valid_os) {
            return [
                'name' => 'Mobile Browser',
                'type' => 'human',
                'company' => 'Browser Visitor',
                'device' => 'Mobile'
            ];
        }

        // Unknown/Unidentified - log for analysis
        return [
            'name' => 'Unidentified',
            'type' => 'unidentified',
            'company' => 'Unknown'
        ];
    }
    
    /**
     * Log visitor detection for debugging
     * Creates a log file in wp-content/aiox-logs/
     */
    private static function log_visitor_detection($ua, $referrer, $result, $ip = '') {
        // Only log if WP_DEBUG is enabled or AIOX_DEBUG_VISITORS is defined
        if (!defined('WP_DEBUG') || !WP_DEBUG) {
            if (!defined('AIOX_DEBUG_VISITORS') || !AIOX_DEBUG_VISITORS) {
                return;
            }
        }
        
        $log_dir = WP_CONTENT_DIR . '/aiox-logs';
        if (!file_exists($log_dir)) {
            wp_mkdir_p($log_dir);
        }
        
        $log_file = $log_dir . '/visitor-detection-' . date('Y-m-d') . '.log';
        
        $log_entry = sprintf(
            "[%s] IP: %s | Type: %s | Name: %s | UA: %s | Ref: %s\n",
            current_time('Y-m-d H:i:s'),
            substr($ip, 0, 20),
            $result['type'],
            $result['name'],
            substr($ua, 0, 200),
            substr($referrer, 0, 100)
        );
        
        file_put_contents($log_file, $log_entry, FILE_APPEND | LOCK_EX);
    }

    /**
     * Update daily statistics - ENHANCED VERSION v5.6.0
     * Tracks: AI, Search, Human, Other Bots, Unidentified
     */
    private static function update_daily_stats($is_ai, $is_search, $is_human, $is_other_bot, $is_unidentified, $is_well_known) {
        global $wpdb;
        $today = current_time('Y-m-d');
        $table = $wpdb->prefix . 'aiox_server_daily_stats';
        
        // Ensure new columns exist
        self::maybe_add_visitor_columns();

        // Try to get existing record
        $existing = $wpdb->get_row($wpdb->prepare(
            "SELECT * FROM $table WHERE date = %s",
            $today
        ));
        
        // Calculate total crawlers (non-human, non-unidentified)
        $is_crawler = $is_ai || $is_search || $is_other_bot;

        if ($existing) {
            // Update existing record
            $update_data = [
                'total_visits' => $existing->total_visits + 1,
                'total_crawlers' => $existing->total_crawlers + ($is_crawler ? 1 : 0),
                'ai_visits' => $existing->ai_visits + ($is_ai ? 1 : 0),
                'search_visits' => $existing->search_visits + ($is_search ? 1 : 0),
                'well_known_visits' => $existing->well_known_visits + ($is_well_known ? 1 : 0),
                'human_visits' => (isset($existing->human_visits) ? $existing->human_visits : 0) + ($is_human ? 1 : 0),
                'other_bot_visits' => (isset($existing->other_bot_visits) ? $existing->other_bot_visits : 0) + ($is_other_bot ? 1 : 0),
                'unidentified_visits' => (isset($existing->unidentified_visits) ? $existing->unidentified_visits : 0) + ($is_unidentified ? 1 : 0)
            ];
            
            $result = $wpdb->update(
                $table,
                $update_data,
                ['date' => $today],
                ['%d', '%d', '%d', '%d', '%d', '%d', '%d', '%d'],
                ['%s']
            );
            
            if ($result === false) {
                error_log('AIOX Analytics: Failed to update daily stats - ' . $wpdb->last_error);
            }
        } else {
            // Insert new record
            $result = $wpdb->insert(
                $table,
                [
                    'date' => $today,
                    'total_visits' => 1,
                    'total_crawlers' => $is_crawler ? 1 : 0,
                    'ai_visits' => $is_ai ? 1 : 0,
                    'search_visits' => $is_search ? 1 : 0,
                    'well_known_visits' => $is_well_known ? 1 : 0,
                    'badge_views' => 0,
                    'human_visits' => $is_human ? 1 : 0,
                    'other_bot_visits' => $is_other_bot ? 1 : 0,
                    'unidentified_visits' => $is_unidentified ? 1 : 0
                ],
                ['%s', '%d', '%d', '%d', '%d', '%d', '%d', '%d', '%d', '%d']
            );
            
            if ($result === false) {
                error_log('AIOX Analytics: Failed to insert daily stats - ' . $wpdb->last_error);
            }
        }
    }
    
    /**
     * Add new visitor columns if they don't exist (for upgrades)
     */
    private static function maybe_add_visitor_columns() {
        global $wpdb;
        static $columns_checked = false;
        
        if ($columns_checked) {
            return;
        }
        
        $table = $wpdb->prefix . 'aiox_server_daily_stats';
        
        // Check if table exists
        if ($wpdb->get_var("SHOW TABLES LIKE '{$table}'") !== $table) {
            return;
        }
        
        // Check for human_visits column
        $column_exists = $wpdb->get_results("SHOW COLUMNS FROM `{$table}` LIKE 'human_visits'");
        if (empty($column_exists)) {
            $wpdb->query("ALTER TABLE `{$table}` ADD COLUMN `human_visits` INT(11) DEFAULT 0");
            $wpdb->query("ALTER TABLE `{$table}` ADD COLUMN `other_bot_visits` INT(11) DEFAULT 0");
            $wpdb->query("ALTER TABLE `{$table}` ADD COLUMN `unidentified_visits` INT(11) DEFAULT 0");
        }
        
        $columns_checked = true;
    }

    /**
     * Update crawler statistics
     * ENHANCED v5.5.0: Added company tracking
     */
    private static function update_crawler_stats($crawler_info, $ip, $agent, $timestamp) {
        global $wpdb;
        $table = $wpdb->prefix . 'aiox_server_crawlers';

        // Try to get existing crawler
        $existing = $wpdb->get_row($wpdb->prepare(
            "SELECT * FROM $table WHERE crawler_name = %s",
            $crawler_info['name']
        ));

        if ($existing) {
            // Update existing crawler (including company info)
            $result = $wpdb->update(
                $table,
                [
                    'last_seen' => $timestamp,
                    'hits' => $existing->hits + 1,
                    'crawler_type' => $crawler_info['type'],
                    'company' => $crawler_info['company'] ?? 'Unknown'
                ],
                ['id' => $existing->id],
                ['%s', '%d', '%s', '%s'],
                ['%d']
            );
            
            if ($result === false) {
                error_log('AIOX Analytics: Failed to update crawler - ' . $wpdb->last_error);
            }
        } else {
            // Insert new crawler (including company info)
            $result = $wpdb->insert(
                $table,
                [
                    'crawler_name' => $crawler_info['name'],
                    'crawler_type' => $crawler_info['type'],
                    'company' => $crawler_info['company'] ?? 'Unknown',
                    'user_agent' => $agent,
                    'ip' => $ip,
                    'first_seen' => $timestamp,
                    'last_seen' => $timestamp,
                    'hits' => 1
                ],
                ['%s', '%s', '%s', '%s', '%s', '%s', '%s', '%d']
            );
            
            if ($result === false) {
                error_log('AIOX Analytics: Failed to insert crawler - ' . $wpdb->last_error);
            }
        }
    }

    /**
     * Get comprehensive overview data - ENHANCED v5.6.0
     * Now includes human visits, other bot visits, and unidentified visits
     */
    public function get_overview_data($days = 30) {
        global $wpdb;
        
        // Ensure columns exist before querying
        self::maybe_add_visitor_columns();
        
        // For 24h view (days=1), return hourly data from analytics table
        if ($days == 1) {
            return $this->get_hourly_overview_data();
        }
        
        $results = $wpdb->get_results($wpdb->prepare(
            "SELECT date, total_visits, ai_visits, search_visits, well_known_visits, badge_views,
                    COALESCE(human_visits, 0) as human_visits,
                    COALESCE(other_bot_visits, 0) as other_bot_visits,
                    COALESCE(unidentified_visits, 0) as unidentified_visits
             FROM {$this->table_daily}
             WHERE date >= DATE_SUB(CURDATE(), INTERVAL %d DAY)
             ORDER BY date ASC",
            $days
        ));

        $total_visits = 0;
        $ai_visits = 0;
        $search_visits = 0;
        $well_known_requests = 0;
        $badge_views = 0;
        $human_visits = 0;
        $other_bot_visits = 0;
        $unidentified_visits = 0;
        $daily_activity = [];

        foreach ($results as $row) {
            $day_total = intval($row->total_visits);
            $day_ai = intval($row->ai_visits);
            $day_search = intval($row->search_visits);
            $day_human = intval($row->human_visits);
            $day_other = intval($row->other_bot_visits);
            $day_unidentified = intval($row->unidentified_visits);
            
            // BACKWARD COMPATIBILITY: For days with old data (no categorization),
            // calculate unidentified as the difference
            $day_categorized = $day_ai + $day_search + $day_human + $day_other + $day_unidentified;
            if ($day_categorized < $day_total && $day_human == 0 && $day_other == 0 && $day_unidentified == 0) {
                // This is old data - all uncategorized traffic goes to unidentified
                $day_unidentified = $day_total - $day_ai - $day_search;
            }
            
            $total_visits += $day_total;
            $ai_visits += $day_ai;
            $search_visits += $day_search;
            $well_known_requests += intval($row->well_known_visits);
            $badge_views += intval($row->badge_views);
            $human_visits += $day_human;
            $other_bot_visits += $day_other;
            $unidentified_visits += $day_unidentified;
            
            $daily_activity[] = [
                'date' => $row->date,
                'total_visits' => $day_total,
                'ai_visits' => $day_ai,
                'search_visits' => $day_search,
                'well_known_visits' => intval($row->well_known_visits),
                'badge_views' => intval($row->badge_views),
                'human_visits' => $day_human,
                'other_bot_visits' => $day_other,
                'unidentified_visits' => $day_unidentified
            ];
        }

        // Calculate AI percentage
        $ai_percentage = $total_visits > 0 ? round(($ai_visits / $total_visits) * 100, 1) : 0;
        
        // BACKWARD COMPATIBILITY: If new columns are empty but we have old data,
        // calculate unidentified as the difference (this is historical data before the update)
        $known_visits = $ai_visits + $search_visits + $human_visits + $other_bot_visits + $unidentified_visits;
        if ($known_visits < $total_visits && $human_visits == 0 && $other_bot_visits == 0 && $unidentified_visits == 0) {
            // Historical data - we only know AI and Search, rest is unidentified
            $unidentified_visits = $total_visits - $ai_visits - $search_visits;
            
            // Also update daily_activity for chart consistency
            foreach ($daily_activity as &$day) {
                $day_known = $day['ai_visits'] + $day['search_visits'] + $day['human_visits'] + $day['other_bot_visits'] + $day['unidentified_visits'];
                if ($day_known < $day['total_visits'] && $day['human_visits'] == 0 && $day['other_bot_visits'] == 0 && $day['unidentified_visits'] == 0) {
                    $day['unidentified_visits'] = $day['total_visits'] - $day['ai_visits'] - $day['search_visits'];
                }
            }
            unset($day);
        }
        
        // Get top AI crawlers for overview - query analytics table for accurate period filtering
        $ai_requests = $wpdb->get_results($wpdb->prepare(
            "SELECT request_data
             FROM {$this->table_analytics}
             WHERE request_time >= DATE_SUB(NOW(), INTERVAL %d DAY)
               AND visitor_type = 'ai'
             LIMIT 5000",
            $days
        ));
        
        // Aggregate crawler counts
        $crawler_counts = [];
        foreach ($ai_requests as $row) {
            $data = maybe_unserialize($row->request_data);
            if (is_array($data) && !empty($data['crawler_name'])) {
                $name = $data['crawler_name'];
                if (!isset($crawler_counts[$name])) {
                    $crawler_counts[$name] = 0;
                }
                $crawler_counts[$name]++;
            }
        }
        
        // Sort by count and take top 10
        arsort($crawler_counts);
        $crawler_counts = array_slice($crawler_counts, 0, 10, true);
        
        // Format for chart and add company info
        $top_crawlers = [];
        foreach ($crawler_counts as $name => $visits) {
            $company = $wpdb->get_var($wpdb->prepare(
                "SELECT company FROM {$this->table_crawlers} WHERE crawler_name = %s LIMIT 1",
                $name
            ));
            $top_crawlers[] = [
                'name' => $name,
                'type' => 'ai',
                'company' => $company ?: 'Unknown',
                'visits' => $visits
            ];
        }

        return [
            'total_visits' => $total_visits,
            'ai_visits' => $ai_visits,
            'search_visits' => $search_visits,
            'well_known_requests' => $well_known_requests,
            'badge_views' => $badge_views,
            'human_visits' => $human_visits,
            'other_bot_visits' => $other_bot_visits,
            'unidentified_visits' => $unidentified_visits,
            'ai_percentage' => $ai_percentage,
            'top_crawlers' => $top_crawlers,
            'daily_activity' => $daily_activity
        ];
    }

    /**
     * Get hourly overview data for last 24 hours
     * Returns data grouped by hour for the 24h period view
     */
    private function get_hourly_overview_data() {
        global $wpdb;
        
        // Get hourly breakdown from analytics table for last 24 hours
        // Note: visitor_type and is_aiox_file columns may not exist in older data
        $results = $wpdb->get_results(
            "SELECT 
                DATE_FORMAT(request_time, '%Y-%m-%d %H:00') as hour_slot,
                COUNT(*) as total_visits,
                SUM(CASE WHEN visitor_type = 'ai' THEN 1 ELSE 0 END) as ai_visits,
                SUM(CASE WHEN visitor_type = 'search' THEN 1 ELSE 0 END) as search_visits,
                SUM(CASE WHEN visitor_type = 'human' THEN 1 ELSE 0 END) as human_visits,
                SUM(CASE WHEN visitor_type IN ('social', 'seo', 'monitoring', 'feed', 'security', 'archive', 'automation', 'infrastructure', 'tool', 'bot') THEN 1 ELSE 0 END) as other_bot_visits,
                SUM(CASE WHEN visitor_type = 'unidentified' OR visitor_type IS NULL THEN 1 ELSE 0 END) as unidentified_visits,
                SUM(CASE WHEN is_aiox_file = 1 THEN 1 ELSE 0 END) as aiox_file_visits
             FROM {$this->table_analytics}
             WHERE request_time >= DATE_SUB(NOW(), INTERVAL 24 HOUR)
             GROUP BY hour_slot
             ORDER BY hour_slot ASC"
        );

        $total_visits = 0;
        $ai_visits = 0;
        $search_visits = 0;
        $aiox_file_requests = 0;
        $human_visits = 0;
        $other_bot_visits = 0;
        $unidentified_visits = 0;
        $hourly_activity = [];

        foreach ($results as $row) {
            $hour_total = intval($row->total_visits);
            $hour_ai = intval($row->ai_visits);
            $hour_search = intval($row->search_visits);
            $hour_human = intval($row->human_visits);
            $hour_other = intval($row->other_bot_visits);
            $hour_unidentified = intval($row->unidentified_visits);
            
            $total_visits += $hour_total;
            $ai_visits += $hour_ai;
            $search_visits += $hour_search;
            $aiox_file_requests += intval($row->aiox_file_visits);
            $human_visits += $hour_human;
            $other_bot_visits += $hour_other;
            $unidentified_visits += $hour_unidentified;
            
            // Format hour for display (e.g., "14:00" or "2:00 PM")
            $hour_time = date('H:i', strtotime($row->hour_slot));
            
            $hourly_activity[] = [
                'date' => $hour_time,  // Using 'date' key for chart compatibility
                'total_visits' => $hour_total,
                'ai_visits' => $hour_ai,
                'search_visits' => $hour_search,
                'well_known_visits' => intval($row->aiox_file_visits),
                'badge_views' => 0, // Badge views tracked separately
                'human_visits' => $hour_human,
                'other_bot_visits' => $hour_other,
                'unidentified_visits' => $hour_unidentified
            ];
        }

        // Get badge views for last 24 hours
        $badge_views = intval($wpdb->get_var(
            "SELECT COUNT(*) FROM {$this->table_badges}
             WHERE view_time >= DATE_SUB(NOW(), INTERVAL 24 HOUR)"
        ));

        // Calculate AI percentage
        $ai_percentage = $total_visits > 0 ? round(($ai_visits / $total_visits) * 100, 1) : 0;
        
        // Get top AI crawlers for last 24 hours - query analytics table for accurate counting
        $ai_requests = $wpdb->get_results(
            "SELECT request_data
             FROM {$this->table_analytics}
             WHERE request_time >= DATE_SUB(NOW(), INTERVAL 24 HOUR)
               AND visitor_type = 'ai'
             LIMIT 5000"
        );
        
        // Aggregate crawler counts
        $crawler_counts = [];
        foreach ($ai_requests as $row) {
            $data = maybe_unserialize($row->request_data);
            if (is_array($data) && !empty($data['crawler_name'])) {
                $name = $data['crawler_name'];
                if (!isset($crawler_counts[$name])) {
                    $crawler_counts[$name] = 0;
                }
                $crawler_counts[$name]++;
            }
        }
        
        // Sort by count and take top 10
        arsort($crawler_counts);
        $crawler_counts = array_slice($crawler_counts, 0, 10, true);
        
        // Format for chart and add company info
        $top_crawlers = [];
        foreach ($crawler_counts as $name => $visits) {
            $company = $wpdb->get_var($wpdb->prepare(
                "SELECT company FROM {$this->table_crawlers} WHERE crawler_name = %s LIMIT 1",
                $name
            ));
            $top_crawlers[] = [
                'name' => $name,
                'type' => 'ai',
                'company' => $company ?: 'Unknown',
                'visits' => $visits
            ];
        }

        return [
            'total_visits' => $total_visits,
            'ai_visits' => $ai_visits,
            'search_visits' => $search_visits,
            'well_known_requests' => $aiox_file_requests,
            'badge_views' => $badge_views,
            'human_visits' => $human_visits,
            'other_bot_visits' => $other_bot_visits,
            'unidentified_visits' => $unidentified_visits,
            'ai_percentage' => $ai_percentage,
            'top_crawlers' => $top_crawlers,
            'daily_activity' => $hourly_activity,  // Using same key for chart compatibility
            'is_hourly' => true  // Flag for frontend to know this is hourly data
        ];
    }

    /**
     * Get crawler activity details
     */
    public function get_crawler_activity($days = 30) {
        global $wpdb;
        
        // Top crawlers with company info
        $top_crawlers = $wpdb->get_results($wpdb->prepare(
            "SELECT crawler_name, crawler_type, company, SUM(hits) as visits
             FROM {$this->table_crawlers}
             WHERE last_seen >= DATE_SUB(NOW(), INTERVAL %d DAY)
             GROUP BY crawler_name, crawler_type, company
             ORDER BY visits DESC
             LIMIT 20",
            $days
        ), ARRAY_A);
        
        // Crawler types
        $by_type = $wpdb->get_results($wpdb->prepare(
            "SELECT crawler_type, SUM(hits) as count
             FROM {$this->table_crawlers}
             WHERE last_seen >= DATE_SUB(NOW(), INTERVAL %d DAY)
             GROUP BY crawler_type
             ORDER BY count DESC",
            $days
        ), ARRAY_A);
        
        // Daily crawler activity
        $daily_activity = $wpdb->get_results($wpdb->prepare(
            "SELECT DATE(last_seen) as date, 
                    SUM(CASE WHEN crawler_type = 'ai' THEN hits ELSE 0 END) as ai_hits,
                    SUM(CASE WHEN crawler_type = 'search' THEN hits ELSE 0 END) as search_hits
             FROM {$this->table_crawlers}
             WHERE last_seen >= DATE_SUB(NOW(), INTERVAL %d DAY)
             GROUP BY DATE(last_seen)
             ORDER BY date ASC",
            $days
        ), ARRAY_A);
        
        return [
            'top_crawlers' => $top_crawlers,
            'by_type' => $by_type,
            'daily_activity' => $daily_activity
        ];
    }

    /**
     * Get badge tracking performance
     */
    public function get_badge_performance($days = 30) {
        global $wpdb;
        
        // Badge views by type
        $by_type = $wpdb->get_results($wpdb->prepare(
            "SELECT viewer_type, COUNT(*) as views
             FROM {$this->table_badge_views}
             WHERE view_time >= DATE_SUB(NOW(), INTERVAL %d DAY)
             GROUP BY viewer_type
             ORDER BY views DESC",
            $days
        ), ARRAY_A);
        
        // Top posts with badge views
        $top_posts_raw = $wpdb->get_results($wpdb->prepare(
            "SELECT bv.post_id, p.post_title, 
                    COUNT(*) as views,
                    SUM(CASE WHEN bv.viewer_type = 'ai' THEN 1 ELSE 0 END) as ai_views,
                    SUM(CASE WHEN bv.viewer_type = 'human' THEN 1 ELSE 0 END) as human_views
             FROM {$this->table_badge_views} bv
             LEFT JOIN {$wpdb->posts} p ON bv.post_id = p.ID
             WHERE bv.view_time >= DATE_SUB(NOW(), INTERVAL %d DAY)
             GROUP BY bv.post_id, p.post_title
             ORDER BY views DESC
             LIMIT 20",
            $days
        ), ARRAY_A);
        
        // Decode post titles
        $top_posts = array_map(function($post) {
            $post['post_title'] = html_entity_decode($post['post_title'] ?? '', ENT_QUOTES | ENT_HTML5, 'UTF-8');
            return $post;
        }, $top_posts_raw);
        
        // Daily badge views
        $daily_views = $wpdb->get_results($wpdb->prepare(
            "SELECT DATE(view_time) as date, COUNT(*) as views
             FROM {$this->table_badge_views}
             WHERE view_time >= DATE_SUB(NOW(), INTERVAL %d DAY)
             GROUP BY DATE(view_time)
             ORDER BY date ASC",
            $days
        ), ARRAY_A);
        
        return [
            'by_type' => $by_type,
            'top_posts' => $top_posts,
            'daily_views' => $daily_views
        ];
    }

    /**
     * Get AIOX system files performance (renamed from well-known)
     */
    public function get_wellknown_performance($days = 30) {
        global $wpdb;
        
        // Top AIOX files
        $top_files = $wpdb->get_results($wpdb->prepare(
            "SELECT file_path, COUNT(*) as hits,
                    SUM(CASE WHEN crawler_type = 'ai' THEN 1 ELSE 0 END) as ai_hits,
                    SUM(CASE WHEN crawler_type = 'search' THEN 1 ELSE 0 END) as search_hits
             FROM {$this->table_wellknown_hits}
             WHERE hit_time >= DATE_SUB(NOW(), INTERVAL %d DAY)
             GROUP BY file_path
             ORDER BY hits DESC",
            $days
        ), ARRAY_A);
        
        // Add friendly names to files
        foreach ($top_files as &$file) {
            $file['friendly_name'] = self::get_aiox_file_name($file['file_path']);
        }
        unset($file);
        
        // Crawlers accessing AIOX files
        $by_crawler = $wpdb->get_results($wpdb->prepare(
            "SELECT crawler_name, crawler_type, COUNT(*) as hits
             FROM {$this->table_wellknown_hits}
             WHERE hit_time >= DATE_SUB(NOW(), INTERVAL %d DAY)
             GROUP BY crawler_name, crawler_type
             ORDER BY hits DESC
             LIMIT 20",
            $days
        ), ARRAY_A);
        
        // Daily AIOX file hits
        $daily_hits = $wpdb->get_results($wpdb->prepare(
            "SELECT DATE(hit_time) as date, COUNT(*) as hits
             FROM {$this->table_wellknown_hits}
             WHERE hit_time >= DATE_SUB(NOW(), INTERVAL %d DAY)
             GROUP BY DATE(hit_time)
             ORDER BY date ASC",
            $days
        ), ARRAY_A);
        
        return [
            'top_files' => $top_files,
            'by_crawler' => $by_crawler,
            'daily_hits' => $daily_hits
        ];
    }

    // Additional methods from original file
    public function get_crawler_types($days = 30) {
        global $wpdb;
        
        // Query analytics table for accurate period filtering
        $results = $wpdb->get_results($wpdb->prepare(
            "SELECT visitor_type as crawler_type, COUNT(*) as count
             FROM {$this->table_analytics}
             WHERE request_time >= DATE_SUB(NOW(), INTERVAL %d DAY)
               AND visitor_type IN ('ai', 'search', 'social', 'seo', 'monitoring', 'feed', 'security', 'archive', 'automation', 'infrastructure', 'tool', 'bot')
             GROUP BY visitor_type
             ORDER BY count DESC",
            $days
        ), ARRAY_A);

        return array_map(function($row) {
            return [
                'crawler_type' => $row['crawler_type'],
                'count' => intval($row['count']),
                'crawler_count' => 1 // Individual request count, not distinct crawlers
            ];
        }, $results);
    }

    public function get_top_crawlers($days = 30) {
        global $wpdb;
        
        // Query analytics table for accurate period filtering (all non-human, non-unidentified types)
        $crawler_requests = $wpdb->get_results($wpdb->prepare(
            "SELECT request_data, visitor_type, request_time
             FROM {$this->table_analytics}
             WHERE request_time >= DATE_SUB(NOW(), INTERVAL %d DAY)
               AND visitor_type NOT IN ('human', 'unidentified')
               AND visitor_type IS NOT NULL
             LIMIT 10000",
            $days
        ));
        
        // Aggregate crawler counts with last_seen tracking
        $crawler_data = [];
        foreach ($crawler_requests as $row) {
            $data = maybe_unserialize($row->request_data);
            if (is_array($data) && !empty($data['crawler_name'])) {
                $name = $data['crawler_name'];
                if (!isset($crawler_data[$name])) {
                    $crawler_data[$name] = [
                        'visits' => 0,
                        'type' => $row->visitor_type,
                        'last_seen' => $row->request_time
                    ];
                }
                $crawler_data[$name]['visits']++;
                if ($row->request_time > $crawler_data[$name]['last_seen']) {
                    $crawler_data[$name]['last_seen'] = $row->request_time;
                }
            }
        }
        
        // Sort by visits and take top 50
        uasort($crawler_data, function($a, $b) {
            return $b['visits'] - $a['visits'];
        });
        $crawler_data = array_slice($crawler_data, 0, 50, true);
        
        // Color palette for different crawler types
        $type_colors = [
            'ai' => '#8b5cf6',        // Purple
            'search' => '#3b82f6',    // Blue
            'social' => '#10b981',    // Green
            'monitoring' => '#f59e0b', // Orange
            'seo' => '#ef4444',       // Red
            'unknown' => '#6b7280'    // Gray
        ];
        
        // Additional colors for variety when we have many crawlers
        $extra_colors = [
            '#ec4899', // Pink
            '#14b8a6', // Teal
            '#f97316', // Orange
            '#84cc16', // Lime
            '#06b6d4', // Cyan
            '#a855f7', // Violet
            '#f43f5e', // Rose
            '#0ea5e9', // Sky
            '#22c55e', // Green
            '#eab308'  // Yellow
        ];
        
        // Format results
        $results = [];
        $index = 0;
        foreach ($crawler_data as $name => $info) {
            $type = $info['type'] ?? 'unknown';
            
            // Get company from crawlers table
            $company = $wpdb->get_var($wpdb->prepare(
                "SELECT company FROM {$this->table_crawlers} WHERE crawler_name = %s LIMIT 1",
                $name
            ));
            
            // Try to get color by type first
            if (isset($type_colors[$type])) {
                $color = $type_colors[$type];
            } else {
                // Use extra colors cycling through them
                $color = $extra_colors[$index % count($extra_colors)];
            }
            
            $results[] = [
                'name' => $name,
                'type' => $type,
                'company' => $company ?: 'Unknown',
                'visits' => intval($info['visits']),
                'last_seen' => $info['last_seen'],
                'color' => $color
            ];
            $index++;
        }
        
        return $results;
    }

    public function get_content_performance($days = 30) {
        global $wpdb;
        
        $results = $wpdb->get_results($wpdb->prepare(
            "SELECT endpoint, COUNT(*) as visits
             FROM {$this->table_analytics}
             WHERE request_time >= DATE_SUB(NOW(), INTERVAL %d DAY)
               AND endpoint NOT LIKE '%%.css%%'
               AND endpoint NOT LIKE '%%.js%%'
               AND endpoint NOT LIKE '%%.png%%'
               AND endpoint NOT LIKE '%%.jpg%%'
             GROUP BY endpoint
             ORDER BY visits DESC
             LIMIT 20",
            $days
        ), ARRAY_A);

        $formatted = [];
        foreach ($results as $row) {
            $post_type = 'page';
            
            if (preg_match('/\?p=(\d+)/', $row['endpoint'], $matches)) {
                $post = get_post($matches[1]);
                if ($post) {
                    $post_type = $post->post_type;
                }
            }
            
            $formatted[] = [
                'endpoint' => $row['endpoint'],
                'post_type' => $post_type,
                'visits' => intval($row['visits'])
            ];
        }

        return $formatted;
    }

    public function get_top_pages($days = 30) {
        global $wpdb;
        
        $results = $wpdb->get_results($wpdb->prepare(
            "SELECT endpoint as request_uri, COUNT(*) as visits
             FROM {$this->table_analytics}
             WHERE request_time >= DATE_SUB(NOW(), INTERVAL %d DAY)
             GROUP BY endpoint
             ORDER BY visits DESC
             LIMIT 20",
            $days
        ), ARRAY_A);

        return array_map(function($row) {
            return [
                'request_uri' => $row['request_uri'],
                'visits' => intval($row['visits'])
            ];
        }, $results);
    }

    public function get_hourly_activity($days = 30) {
        global $wpdb;
        
        $results = $wpdb->get_results($wpdb->prepare(
            "SELECT HOUR(request_time) as hour, COUNT(*) as visits
             FROM {$this->table_analytics}
             WHERE request_time >= DATE_SUB(NOW(), INTERVAL %d DAY)
             GROUP BY HOUR(request_time)
             ORDER BY hour ASC",
            $days
        ), ARRAY_A);

        return array_map(function($row) {
            return [
                'hour' => intval($row['hour']),
                'visits' => intval($row['visits'])
            ];
        }, $results);
    }

    public function get_top_posts($days = 30) {
        global $wpdb;
        
        $results = $wpdb->get_results($wpdb->prepare(
            "SELECT endpoint, COUNT(*) as crawler_visits
             FROM {$this->table_analytics}
             WHERE request_time >= DATE_SUB(NOW(), INTERVAL %d DAY)
               AND endpoint LIKE '%%?p=%%'
             GROUP BY endpoint
             ORDER BY crawler_visits DESC
             LIMIT 20",
            $days
        ), ARRAY_A);

        $formatted = [];
        foreach ($results as $row) {
            if (preg_match('/\?p=(\d+)/', $row['endpoint'], $matches)) {
                $post_id = $matches[1];
                $post = get_post($post_id);
                
                if ($post) {
                    $formatted[] = [
                        'post_id' => $post_id,
                        'post_title' => html_entity_decode($post->post_title, ENT_QUOTES | ENT_HTML5, 'UTF-8'),
                        'post_type' => $post->post_type,
                        'crawler_visits' => intval($row['crawler_visits'])
                    ];
                }
            }
        }

        return $formatted;
    }
    
    /**
     * Get crawler activity by company
     * NEW v5.5.0: Company-based analytics
     */
    public function get_crawler_by_company($days = 30) {
        global $wpdb;
        
        $results = $wpdb->get_results($wpdb->prepare(
            "SELECT company, crawler_type, COUNT(*) as crawler_count, SUM(hits) as total_hits
             FROM {$this->table_crawlers}
             WHERE last_seen >= DATE_SUB(NOW(), INTERVAL %d DAY)
             GROUP BY company, crawler_type
             ORDER BY total_hits DESC",
            $days
        ), ARRAY_A);

        return array_map(function($row) {
            return [
                'company' => $row['company'],
                'crawler_type' => $row['crawler_type'],
                'crawler_count' => intval($row['crawler_count']),
                'total_hits' => intval($row['total_hits'])
            ];
        }, $results);
    }

    public static function init() {
        return self::get_instance();
    }
}

// Initialize
AIOX_Modern_Analytics::get_instance();