<?php
/**
 * AIOX Pack Generator
 * 
 * Generates binary AIOX pack files with processed content
 */

if (!defined('ABSPATH')) {
    exit;
}

class AIOX_Pack {
    
    /**
     * Generate AIOX pack files (paginated) with index
     */
    public function generate_pack() {
        $well_known_dir = ABSPATH . '.well-known';
        $packs_dir = $well_known_dir . '/packs';
        
        try {
            // Create packs directory if it doesn't exist
            if (!file_exists($packs_dir)) {
                wp_mkdir_p($packs_dir);
            }
            
            // Get ALL processed posts
            $all_processed_posts = $this->get_all_processed_posts();
            $total_posts = count($all_processed_posts);
            
            if ($total_posts === 0) {
                AIOX_Logger::info('No processed posts to pack');
                return true;
            }
            
            // Paginate posts into chunks of 100
            $posts_per_pack = 100;
            $chunks = array_chunk($all_processed_posts, $posts_per_pack, true);
            $total_packs = count($chunks);
            
            $packs_info = array();
            
            // Generate each pack file
            foreach ($chunks as $pack_number => $posts_chunk) {
                $pack_number_display = $pack_number + 1; // 1-indexed for display
                
                // Create pack data structure
                $pack_data = array(
                    'version' => '1.0',
                    'generator' => 'AIOX Publisher Suite Pro',
                    'pack_number' => $pack_number_display,
                    'total_packs' => $total_packs,
                    'site' => array(
                        'name' => get_bloginfo('name'),
                        'url' => home_url(),
                        'description' => get_bloginfo('description')
                    ),
                    'generated_at' => current_time('c'),
                    'content_count' => count($posts_chunk),
                    'content' => array_values($posts_chunk)
                );
                
                // Convert to JSON and compress
                $json_content = wp_json_encode($pack_data, JSON_UNESCAPED_SLASHES);
                $json_size = strlen($json_content);
                
                // Pack filename
                $pack_file = $packs_dir . '/aiox-pack-' . $pack_number_display . '.aiox';
                
                // Compress if available
                if (function_exists('gzencode')) {
                    $compressed_content = gzencode($json_content, 9);
                    $compressed_size = strlen($compressed_content);
                    file_put_contents($pack_file, $compressed_content);
                } else {
                    // Fallback: uncompressed
                    $compressed_size = $json_size;
                    file_put_contents($pack_file, $json_content);
                }
                
                // Extract post IDs for index
                $post_ids = array_map(function($post) {
                    return $post['id'];
                }, $posts_chunk);
                
                // Track pack info for index
                $packs_info[] = array(
                    'pack_number' => $pack_number_display,
                    'url' => home_url('/.well-known/packs/aiox-pack-' . $pack_number_display . '.aiox'),
                    'post_range' => (($pack_number * $posts_per_pack) + 1) . '-' . (($pack_number * $posts_per_pack) + count($posts_chunk)),
                    'post_ids' => $post_ids,
                    'post_count' => count($posts_chunk),
                    'compressed_size' => $compressed_size,
                    'uncompressed_size' => $json_size,
                    'generated_at' => current_time('c')
                );
                
                AIOX_Logger::info('Generated pack ' . $pack_number_display . '/' . $total_packs, array(
                    'file' => basename($pack_file),
                    'posts' => count($posts_chunk),
                    'compressed_size' => $compressed_size
                ));
            }
            
            // Generate pack index file
            $this->generate_pack_index($packs_info, $total_posts);
            
            // BACKWARD COMPATIBILITY: Also create single pack file as aiox.aiox (first 100 posts)
            if (!empty($chunks)) {
                $legacy_pack_file = $well_known_dir . '/aiox.aiox';
                $first_pack_file = $packs_dir . '/aiox-pack-1.aiox';
                if (file_exists($first_pack_file)) {
                    copy($first_pack_file, $legacy_pack_file);
                    AIOX_Logger::info('Created legacy pack file for backward compatibility', array(
                        'file' => 'aiox.aiox'
                    ));
                }
            }
            
            AIOX_Logger::info('AIOX pack system generated successfully', array(
                'total_packs' => $total_packs,
                'total_posts' => $total_posts,
                'posts_per_pack' => $posts_per_pack
            ));
            
            return true;
            
        } catch (Exception $e) {
            AIOX_Logger::error('Failed to generate AIOX pack system', array('error' => $e->getMessage()));
            return false;
        }
    }
    
    /**
     * Generate pack index file
     */
    private function generate_pack_index($packs_info, $total_posts) {
        $well_known_dir = ABSPATH . '.well-known';
        $index_file = $well_known_dir . '/aiox-pack-index.json';
        
        try {
            $index_data = array(
                'version' => '1.0',
                'format' => 'aiox-pack-index',
                'site_info' => array(
                    'site_url' => home_url(),
                    'site_name' => get_bloginfo('name'),
                    'total_posts' => $total_posts,
                    'total_packs' => count($packs_info),
                    'posts_per_pack' => 100
                ),
                'packs' => $packs_info,
                'endpoints' => array(
                    'pack_template' => home_url('/.well-known/packs/aiox-pack-{number}.aiox'),
                    'individual_posts' => home_url('/.well-known/posts/{post_id}.json'),
                    'main_manifest' => home_url('/.well-known/aio.json')
                ),
                'generated_at' => current_time('c')
            );
            
            $json_content = json_encode($index_data, JSON_PRETTY_PRINT | JSON_UNESCAPED_SLASHES);
            file_put_contents($index_file, $json_content);
            
            AIOX_Logger::info('Pack index generated', array(
                'file' => $index_file,
                'packs' => count($packs_info),
                'size' => strlen($json_content)
            ));
            
            return true;
            
        } catch (Exception $e) {
            AIOX_Logger::error('Failed to generate pack index', array('error' => $e->getMessage()));
            return false;
        }
    }
    
    /**
     * Get all processed posts with ingredients data (no limit)
     */
    private function get_all_processed_posts() {
        // Get ALL posts that have been processed
        $posts = get_posts(array(
            'post_type' => array('post', 'page'),
            'post_status' => 'publish',
            'numberposts' => -1, // Get ALL posts (no limit)
            'meta_query' => array(
                array(
                    'key' => '_aiox_processed',
                    'value' => '1',
                    'compare' => '='
                )
            ),
            'orderby' => 'modified',
            'order' => 'DESC'
        ));
        
        $processed_posts = array();
        
        foreach ($posts as $post) {
            // Get ingredients data using the same method as the working ingredients.json
            $ingredients_data = get_post_meta($post->ID, '_aiox_ingredients', true);
            $ingredients_array = array();
            
            if (!empty($ingredients_data)) {
                $decoded_ingredients = json_decode($ingredients_data, true);
                if (is_array($decoded_ingredients)) {
                    $ingredients_array = $decoded_ingredients;
                }
            }
            
            // Only include posts that have ingredients data
            if (!empty($ingredients_array)) {
                $processed_posts[] = array(
                    'id' => intval($post->ID),
                    'title' => $post->post_title,
                    'url' => get_permalink($post->ID),
                    'excerpt' => $post->post_excerpt ?: wp_trim_words($post->post_content, 30),
                    'published' => $post->post_date,
                    'modified' => $post->post_modified,
                    'ingredients' => $ingredients_array,
                    'ingredients_count' => count($ingredients_array),
                    'word_count' => str_word_count(strip_tags($post->post_content))
                );
            }
        }
        
        return $processed_posts;
    }
    
    /**
     * Check if pack file exists and get info
     */
    public function get_pack_info() {
        $pack_file = ABSPATH . '.well-known/aiox.aiox';
        
        if (!file_exists($pack_file)) {
            return false;
        }
        
        return array(
            'exists' => true,
            'size' => filesize($pack_file),
            'modified' => filemtime($pack_file),
            'url' => home_url('/.well-known/aiox.aiox')
        );
    }
    
    /**
     * Update pack file when content changes
     */
    public function update_pack_on_content_change($post_id) {
        // Only update if post has ingredients data
        $ingredients_data = get_post_meta($post_id, '_aiox_ingredients', true);
        
        if (!empty($ingredients_data)) {
            // Regenerate pack file
            $this->generate_pack();
        }
    }
}
