<?php
/**
 * AIOX Robots.txt Manager
 * 
 * Handles robots.txt integration for AIOX crawler discovery
 * Safely adds AIOX-specific directives without overwriting existing rules
 * 
 * @package AIOX_Publisher_Suite_Pro
 * @version 5.2.0
 */

if (!defined('ABSPATH')) {
    exit;
}

class AIOX_Robots_Manager {
    
    /**
     * AIOX robots.txt marker for safe insertion/removal
     */
    const AIOX_MARKER_START = '# BEGIN AIOX';
    const AIOX_MARKER_END = '# END AIOX';
    
    /**
     * Constructor
     */
    public function __construct() {
        // Hook into robots.txt generation
        add_filter('robots_txt', array($this, 'add_aiox_rules'), 10, 2);
        
        // Add admin menu item for manual management
        add_action('admin_menu', array($this, 'add_admin_menu'), 100);
        
        // AJAX handlers for robots.txt management
        add_action('wp_ajax_aiox_regenerate_robots', array($this, 'ajax_regenerate_robots'));
        add_action('wp_ajax_aiox_remove_robots', array($this, 'ajax_remove_robots'));
    }
    
    /**
     * Add AIOX rules to WordPress robots.txt
     * 
     * @param string $output Existing robots.txt content
     * @param bool $public Whether the site is public
     * @return string Modified robots.txt content
     */
    public function add_aiox_rules($output, $public) {
        // Only add rules if site is public
        if (!$public) {
            return $output;
        }
        
        $home_url = trailingslashit(home_url());
        $site_url = trailingslashit(site_url());
        
        // Build AIOX-specific rules
        $aiox_rules = array();
        
        // Add header
        $aiox_rules[] = '';
        $aiox_rules[] = self::AIOX_MARKER_START;
        $aiox_rules[] = '# AIOX (AI Optimization) Discovery Rules';
        $aiox_rules[] = '# Generated by AIOX Publisher Suite Pro v' . (defined('AIOX_VERSION') ? AIOX_VERSION : '5.2.0');
        $aiox_rules[] = '# Last updated: ' . current_time('Y-m-d H:i:s');
        $aiox_rules[] = '';
        
        // Main AIOX Manifest (like sitemap.xml for AI)
        $aiox_rules[] = '# AIOX Manifest - Primary discovery endpoint for AI systems';
        $aiox_rules[] = 'Sitemap: ' . $home_url . 'aio.json';
        
        // Also reference standard sitemap.xml if it exists
        $aiox_rules[] = 'Sitemap: ' . $home_url . 'sitemap.xml';
        $aiox_rules[] = '';
        
        // Allow all AI crawlers to access AIOX files
        $aiox_rules[] = '# Allow all crawlers to access AIOX files';
        $aiox_rules[] = 'User-agent: *';
        $aiox_rules[] = 'Allow: /aio.json';
        $aiox_rules[] = 'Allow: /.well-known/aio.json';
        $aiox_rules[] = 'Allow: /.well-known/aiox.ndjson';
        $aiox_rules[] = 'Allow: /.well-known/aiox.aiox';
        $aiox_rules[] = 'Allow: /.well-known/tdmrep.json';
        $aiox_rules[] = 'Allow: /.well-known/aiox-pubkey.pem';
        $aiox_rules[] = 'Allow: /.well-known/aiox-proof.json';
        $aiox_rules[] = 'Allow: /.well-known/ingredients.json';
        $aiox_rules[] = 'Allow: /.well-known/capsules/';
        $aiox_rules[] = '';
        
        // Specific rules for known AI crawlers - explicitly allow full access
        $ai_crawlers = $this->get_known_ai_crawlers();
        
        if (!empty($ai_crawlers)) {
            $aiox_rules[] = '# Explicit access rules for AI crawlers';
            
            foreach ($ai_crawlers as $crawler) {
                $aiox_rules[] = '';
                $aiox_rules[] = '# ' . $crawler['name'];
                $aiox_rules[] = 'User-agent: ' . $crawler['user_agent'];
                
                // Add crawl delay if configured
                $crawl_delay = get_option('aiox_robots_crawl_delay', 0);
                if ($crawl_delay > 0) {
                    $aiox_rules[] = 'Crawl-delay: ' . $crawl_delay;
                }
                
                // Explicitly allow full site access for AI crawlers
                $aiox_rules[] = 'Allow: /';
            }
            
            $aiox_rules[] = '';
        }
        
        // Add custom AIOX meta directive
        $aiox_rules[] = '# AIOX Protocol Information';
        $aiox_rules[] = '# AIOX-Version: 1.0';
        $aiox_rules[] = '# AIOX-Manifest: ' . $home_url . 'aio.json';
        $aiox_rules[] = '# AIOX-Endpoint: ' . $home_url . '.well-known/aio.json';
        $aiox_rules[] = '# AIOX-Delta-Feed: ' . $home_url . '.well-known/aiox.ndjson';
        $aiox_rules[] = '# AIOX-Pack: ' . $home_url . '.well-known/aiox.aiox';
        $aiox_rules[] = '';
        
        // Add TDM-REP reference
        $aiox_rules[] = '# Text & Data Mining Rights Expression Protocol';
        $aiox_rules[] = '# TDM-REP: ' . $home_url . '.well-known/tdmrep.json';
        $aiox_rules[] = '';
        
        // Add footer
        $aiox_rules[] = '# For more information about AIOX:';
        $aiox_rules[] = '# Visit: https://aioxsuite.com';
        $aiox_rules[] = '# Documentation: https://docs.aioxsuite.com';
        $aiox_rules[] = self::AIOX_MARKER_END;
        $aiox_rules[] = '';
        
        // Append AIOX rules to existing robots.txt
        $output .= "\n" . implode("\n", $aiox_rules);
        
        // Log the update
        AIOX_Logger::info('AIOX robots.txt rules added via WordPress filter');
        
        return $output;
    }
    
    /**
     * Get list of known AI crawlers
     * 
     * @return array List of AI crawlers with user agents
     */
    private function get_known_ai_crawlers() {
        return array(
            array(
                'name' => 'OpenAI GPTBot',
                'user_agent' => 'GPTBot',
            ),
            array(
                'name' => 'Google Bard/Gemini',
                'user_agent' => 'Google-Extended',
            ),
            array(
                'name' => 'Anthropic Claude',
                'user_agent' => 'anthropic-ai',
            ),
            array(
                'name' => 'Common Crawl',
                'user_agent' => 'CCBot',
            ),
            array(
                'name' => 'Perplexity AI',
                'user_agent' => 'PerplexityBot',
            ),
            array(
                'name' => 'Cohere',
                'user_agent' => 'cohere-ai',
            ),
            array(
                'name' => 'Meta AI',
                'user_agent' => 'FacebookBot',
            ),
            array(
                'name' => 'Diffbot',
                'user_agent' => 'Diffbot',
            ),
            array(
                'name' => 'YouBot',
                'user_agent' => 'YouBot',
            ),
            array(
                'name' => 'Omgilibot',
                'user_agent' => 'omgilibot',
            )
        );
    }
    
    /**
     * Create or update physical robots.txt file
     * 
     * @return bool|WP_Error True on success, WP_Error on failure
     */
    public function create_physical_robots_file() {
        $robots_file = ABSPATH . 'robots.txt';
        
        try {
            // Check if we can write to the root directory
            if (!is_writable(ABSPATH)) {
                return new WP_Error(
                    'aiox_robots_not_writable',
                    'WordPress root directory is not writable. Please check file permissions.'
                );
            }
            
            // Get existing robots.txt content
            $existing_content = '';
            if (file_exists($robots_file)) {
                $existing_content = file_get_contents($robots_file);
                
                // Remove any existing AIOX rules to prevent duplicates
                $existing_content = $this->remove_aiox_rules($existing_content);
            }
            
            // Get WordPress generated robots.txt (without our filter)
            remove_filter('robots_txt', array($this, 'add_aiox_rules'), 10);
            $wp_robots = $this->get_wp_default_robots();
            add_filter('robots_txt', array($this, 'add_aiox_rules'), 10, 2);
            
            // Generate AIOX rules
            $aiox_content = $this->generate_aiox_rules_standalone();
            
            // Combine content
            $final_content = '';
            
            // Use WordPress default if no existing content
            if (empty($existing_content)) {
                $final_content = $wp_robots . "\n";
            } else {
                $final_content = $existing_content . "\n";
            }
            
            // Append AIOX rules
            $final_content .= $aiox_content;
            
            // Write to file
            $result = file_put_contents($robots_file, $final_content);
            
            if ($result === false) {
                return new WP_Error(
                    'aiox_robots_write_failed',
                    'Failed to write robots.txt file.'
                );
            }
            
            AIOX_Logger::info('Physical robots.txt file created/updated successfully', array(
                'file' => $robots_file,
                'size' => $result
            ));
            
            return true;
            
        } catch (Exception $e) {
            AIOX_Logger::error('Failed to create physical robots.txt file', array(
                'error' => $e->getMessage()
            ));
            
            return new WP_Error(
                'aiox_robots_exception',
                'Exception while creating robots.txt: ' . $e->getMessage()
            );
        }
    }
    
    /**
     * Generate standalone AIOX rules
     * 
     * @return string AIOX rules content
     */
    private function generate_aiox_rules_standalone() {
        $output = $this->add_aiox_rules('', true);
        return $output;
    }
    
    /**
     * Get WordPress default robots.txt content
     * 
     * @return string Default robots.txt
     */
    private function get_wp_default_robots() {
        ob_start();
        do_robots();
        $content = ob_get_clean();
        return $content;
    }
    
    /**
     * Remove AIOX rules from existing content
     * 
     * @param string $content Existing robots.txt content
     * @return string Content without AIOX rules
     */
    private function remove_aiox_rules($content) {
        $pattern = '/' . preg_quote(self::AIOX_MARKER_START, '/') . '.*?' . preg_quote(self::AIOX_MARKER_END, '/') . '/s';
        $content = preg_replace($pattern, '', $content);
        
        // Clean up extra blank lines
        $content = preg_replace("/\n{3,}/", "\n\n", $content);
        
        return trim($content);
    }
    
    /**
     * Remove physical robots.txt AIOX rules
     * 
     * @return bool|WP_Error True on success, WP_Error on failure
     */
    public function remove_physical_robots_rules() {
        $robots_file = ABSPATH . 'robots.txt';
        
        if (!file_exists($robots_file)) {
            return true; // Nothing to remove
        }
        
        try {
            $content = file_get_contents($robots_file);
            $new_content = $this->remove_aiox_rules($content);
            
            // If content is now empty, delete the file
            if (empty(trim($new_content))) {
                unlink($robots_file);
                AIOX_Logger::info('Empty robots.txt file removed');
            } else {
                file_put_contents($robots_file, $new_content);
                AIOX_Logger::info('AIOX rules removed from robots.txt');
            }
            
            return true;
            
        } catch (Exception $e) {
            AIOX_Logger::error('Failed to remove AIOX rules from robots.txt', array(
                'error' => $e->getMessage()
            ));
            
            return new WP_Error(
                'aiox_robots_remove_failed',
                'Failed to remove AIOX rules: ' . $e->getMessage()
            );
        }
    }
    
    /**
     * Check if robots.txt has AIOX rules
     * 
     * @return array Status information
     */
    public function check_robots_status() {
        $robots_file = ABSPATH . 'robots.txt';
        
        $status = array(
            'physical_file_exists' => file_exists($robots_file),
            'physical_file_writable' => is_writable(ABSPATH),
            'has_aiox_rules' => false,
            'virtual_filter_active' => has_filter('robots_txt', array($this, 'add_aiox_rules')),
            'file_path' => $robots_file,
            'file_size' => 0,
            'last_modified' => null
        );
        
        if ($status['physical_file_exists']) {
            $content = file_get_contents($robots_file);
            $status['has_aiox_rules'] = (strpos($content, self::AIOX_MARKER_START) !== false);
            $status['file_size'] = filesize($robots_file);
            $status['last_modified'] = date('Y-m-d H:i:s', filemtime($robots_file));
        }
        
        return $status;
    }
    
    /**
     * Add admin menu for robots.txt management
     */
    public function add_admin_menu() {
        // Add as submenu under AIOX dashboard
        add_submenu_page(
            'aiox-dashboard',
            __('AIOX Robots.txt Manager', 'aiox-publisher-suite-pro'),
            __('🤖 Robots.txt', 'aiox-publisher-suite-pro'),
            'manage_options',
            'aiox-robots',
            array($this, 'render_admin_page')
        );
    }
    
    /**
     * Render admin page for robots.txt management
     */
    public function render_admin_page() {
        $status = $this->check_robots_status();
        $robots_url = home_url('robots.txt');
        
        ?>
        <div class="wrap">
            <h1><span class="dashicons dashicons-admin-site-alt3" style="font-size: 30px; width: 30px; height: 30px; margin-right: 10px; color: #2271b1;"></span><?php esc_html_e('Robots.txt Manager', 'aiox-publisher-suite-pro'); ?></h1>
            <p class="description"><?php esc_html_e('Manage AI crawler directives in your robots.txt file', 'aiox-publisher-suite-pro'); ?></p>
            
            <div id="poststuff">
                <div id="post-body" class="metabox-holder columns-2">
                    <div id="post-body-content">
                        
                        <!-- Info Notice -->
                        <div class="notice notice-info inline" style="margin: 20px 0;">
                            <p><strong><?php esc_html_e('About AIOX Robots.txt Integration', 'aiox-publisher-suite-pro'); ?></strong><br>
                            <?php esc_html_e('AIOX adds special directives to your robots.txt file to help AI crawlers discover your AIOX manifest and content. This is similar to how sitemap.xml helps search engines discover your content.', 'aiox-publisher-suite-pro'); ?></p>
                        </div>
                        
                        <!-- Current Status -->
                        <div class="postbox">
                            <h2 class="hndle"><span class="dashicons dashicons-info"></span> <?php esc_html_e('Current Status', 'aiox-publisher-suite-pro'); ?></h2>
                            <div class="inside">
                                <table class="widefat striped">
                                    <tbody>
                                        <tr>
                                            <td><strong><?php esc_html_e('Physical File', 'aiox-publisher-suite-pro'); ?></strong></td>
                                            <td>
                                                <?php if ($status['physical_file_exists']): ?>
                                                    <span style="color: #00a32a;"><span class="dashicons dashicons-yes-alt"></span> <?php esc_html_e('Exists', 'aiox-publisher-suite-pro'); ?></span>
                                                    <br><small class="description"><?php echo esc_html(basename($status['file_path'])); ?> • <?php echo number_format($status['file_size']); ?> bytes</small>
                                                <?php else: ?>
                                                    <span style="color: #dba617;"><span class="dashicons dashicons-warning"></span> <?php esc_html_e('Not found', 'aiox-publisher-suite-pro'); ?></span>
                                                    <br><small class="description"><?php esc_html_e('Using WordPress virtual robots.txt', 'aiox-publisher-suite-pro'); ?></small>
                                                <?php endif; ?>
                                            </td>
                                        </tr>
                                        <tr>
                                            <td><strong><?php esc_html_e('AIOX Rules', 'aiox-publisher-suite-pro'); ?></strong></td>
                                            <td>
                                                <?php if ($status['has_aiox_rules'] || $status['virtual_filter_active']): ?>
                                                    <span style="color: #00a32a;"><span class="dashicons dashicons-yes-alt"></span> <?php esc_html_e('Active', 'aiox-publisher-suite-pro'); ?></span>
                                                <?php else: ?>
                                                    <span style="color: #d63638;"><span class="dashicons dashicons-dismiss"></span> <?php esc_html_e('Not active', 'aiox-publisher-suite-pro'); ?></span>
                                                <?php endif; ?>
                                            </td>
                                        </tr>
                                        <tr>
                                            <td><strong><?php esc_html_e('Directory Writable', 'aiox-publisher-suite-pro'); ?></strong></td>
                                            <td>
                                                <?php if ($status['physical_file_writable']): ?>
                                                    <span style="color: #00a32a;"><span class="dashicons dashicons-yes-alt"></span> <?php esc_html_e('Yes', 'aiox-publisher-suite-pro'); ?></span>
                                                <?php else: ?>
                                                    <span style="color: #d63638;"><span class="dashicons dashicons-warning"></span> <?php esc_html_e('No', 'aiox-publisher-suite-pro'); ?></span>
                                                <?php endif; ?>
                                            </td>
                                        </tr>
                                    </tbody>
                                </table>
                                
                                <p style="margin-top: 15px;">
                                    <a href="<?php echo esc_url($robots_url); ?>" target="_blank" class="button">
                                        <span class="dashicons dashicons-external" style="vertical-align: middle;"></span>
                                        <?php esc_html_e('View Current robots.txt', 'aiox-publisher-suite-pro'); ?>
                                    </a>
                                </p>
                            </div>
                        </div>
                        
                        <!-- Actions -->
                        <div class="postbox">
                            <h2 class="hndle"><span class="dashicons dashicons-admin-tools"></span> <?php esc_html_e('Actions', 'aiox-publisher-suite-pro'); ?></h2>
                            <div class="inside">
                                <?php if ($status['physical_file_writable']): ?>
                                    <p>
                                        <button type="button" class="button button-primary" id="aiox-create-robots">
                                            <span class="dashicons dashicons-update" style="vertical-align: middle;"></span>
                                            <?php esc_html_e('Create/Update Physical robots.txt', 'aiox-publisher-suite-pro'); ?>
                                        </button>
                                        
                                        <?php if ($status['physical_file_exists'] && $status['has_aiox_rules']): ?>
                                            <button type="button" class="button" id="aiox-remove-robots" style="color: #d63638; border-color: #d63638;">
                                                <span class="dashicons dashicons-trash" style="vertical-align: middle;"></span>
                                                <?php esc_html_e('Remove AIOX Rules', 'aiox-publisher-suite-pro'); ?>
                                            </button>
                                        <?php endif; ?>
                                    </p>
                                <?php else: ?>
                                    <div class="notice notice-warning inline">
                                        <p><?php esc_html_e('Cannot create physical robots.txt file due to permission restrictions. AIOX rules are being added dynamically via WordPress filters.', 'aiox-publisher-suite-pro'); ?></p>
                                    </div>
                                <?php endif; ?>
                                
                                <div id="aiox-robots-status"></div>
                            </div>
                        </div>
                        
                        <!-- Settings -->
                        <div class="postbox">
                            <h2 class="hndle"><span class="dashicons dashicons-admin-settings"></span> <?php esc_html_e('Settings', 'aiox-publisher-suite-pro'); ?></h2>
                            <div class="inside">
                                <form method="post" action="options.php">
                                    <?php settings_fields('aiox_robots_settings'); ?>
                                    
                                    <table class="form-table">
                                        <tr>
                                            <th scope="row">
                                                <label for="aiox_robots_crawl_delay"><?php esc_html_e('Crawl Delay (seconds)', 'aiox-publisher-suite-pro'); ?></label>
                                            </th>
                                            <td>
                                                <input type="number" id="aiox_robots_crawl_delay" name="aiox_robots_crawl_delay" value="<?php echo esc_attr(get_option('aiox_robots_crawl_delay', 0)); ?>" min="0" max="60" class="small-text" />
                                                <p class="description"><?php esc_html_e('Minimum time AI crawlers should wait between requests. 0 = no delay. Recommended: 1-5 seconds.', 'aiox-publisher-suite-pro'); ?></p>
                                            </td>
                                        </tr>
                                    </table>
                                    
                                    <?php submit_button(__('Save Settings', 'aiox-publisher-suite-pro')); ?>
                                </form>
                            </div>
                        </div>
                        
                    </div><!-- /post-body-content -->
                    
                    <!-- Sidebar -->
                    <div id="postbox-container-1" class="postbox-container">
                        <div class="postbox">
                            <h2 class="hndle"><span class="dashicons dashicons-editor-code"></span> <?php esc_html_e('AIOX Directives Preview', 'aiox-publisher-suite-pro'); ?></h2>
                            <div class="inside">
                                <p class="description"><?php esc_html_e('These directives are added to your robots.txt:', 'aiox-publisher-suite-pro'); ?></p>
                                <pre style="background: #f6f7f7; padding: 12px; border: 1px solid #dcdcde; border-radius: 4px; font-size: 11px; line-height: 1.6; overflow-x: auto; white-space: pre;">
<span style="color: #666;"># AIOX Sitemap</span>
<span style="color: #2271b1;">Sitemap:</span> <?php echo esc_url(home_url('/aio.json')); ?>

<span style="color: #2271b1;">Sitemap:</span> <?php echo esc_url(home_url('/sitemap.xml')); ?>

<span style="color: #666;"># Allow AIOX files</span>
<span style="color: #2271b1;">User-agent:</span> *
<span style="color: #2271b1;">Allow:</span> /aio.json
<span style="color: #2271b1;">Allow:</span> /.well-known/aio.json
<span style="color: #2271b1;">Allow:</span> /.well-known/aiox.ndjson
<span style="color: #2271b1;">Allow:</span> /.well-known/tdmrep.json
<span style="color: #2271b1;">Allow:</span> /.well-known/capsules/

<span style="color: #666;"># AI Crawlers</span>
<span style="color: #2271b1;">User-agent:</span> GPTBot
<span style="color: #2271b1;">Allow:</span> /

<span style="color: #2271b1;">User-agent:</span> Google-Extended
<span style="color: #2271b1;">Allow:</span> /

<span style="color: #2271b1;">User-agent:</span> anthropic-ai
<span style="color: #2271b1;">Allow:</span> /

<span style="color: #2271b1;">User-agent:</span> PerplexityBot
<span style="color: #2271b1;">Allow:</span> /</pre>
                            </div>
                        </div>
                    </div>
                    
                </div><!-- /post-body -->
            </div><!-- /poststuff -->
        </div>
        
        <style>
        @keyframes spin {
            from { transform: rotate(0deg); }
            to { transform: rotate(360deg); }
        }
        .spin { animation: spin 1s linear infinite; }
        </style>
        
        <script>
        jQuery(document).ready(function($) {
            $('#aiox-create-robots').on('click', function() {
                var $button = $(this);
                var $status = $('#aiox-robots-status');
                
                $button.prop('disabled', true);
                $status.html('<div class="notice notice-info inline"><p><span class="dashicons dashicons-update spin" style="vertical-align: middle;"></span> <?php esc_html_e('Creating/updating robots.txt file...', 'aiox-publisher-suite-pro'); ?></p></div>');
                
                $.post(ajaxurl, {
                    action: 'aiox_regenerate_robots',
                    nonce: '<?php echo wp_create_nonce('aiox_robots_nonce'); ?>'
                }, function(response) {
                    $button.prop('disabled', false);
                    
                    if (response.success) {
                        $status.html('<div class="notice notice-success inline"><p><span class="dashicons dashicons-yes-alt" style="vertical-align: middle;"></span> ' + response.data.message + '</p></div>');
                        setTimeout(function() {
                            location.reload();
                        }, 1500);
                    } else {
                        $status.html('<div class="notice notice-error inline"><p><span class="dashicons dashicons-warning" style="vertical-align: middle;"></span> ' + response.data.message + '</p></div>');
                    }
                });
            });
            
            $('#aiox-remove-robots').on('click', function() {
                if (!confirm('<?php esc_html_e('Are you sure you want to remove AIOX rules from robots.txt?', 'aiox-publisher-suite-pro'); ?>')) {
                    return;
                }
                
                var $button = $(this);
                var $status = $('#aiox-robots-status');
                
                $button.prop('disabled', true);
                $status.html('<div class="notice notice-info inline"><p><span class="dashicons dashicons-update spin" style="vertical-align: middle;"></span> <?php esc_html_e('Removing AIOX rules...', 'aiox-publisher-suite-pro'); ?></p></div>');
                
                $.post(ajaxurl, {
                    action: 'aiox_remove_robots',
                    nonce: '<?php echo wp_create_nonce('aiox_robots_nonce'); ?>'
                }, function(response) {
                    $button.prop('disabled', false);
                    
                    if (response.success) {
                        $status.html('<div class="notice notice-success inline"><p><span class="dashicons dashicons-yes-alt" style="vertical-align: middle;"></span> ' + response.data.message + '</p></div>');
                        setTimeout(function() {
                            location.reload();
                        }, 1500);
                    } else {
                        $status.html('<div class="notice notice-error inline"><p><span class="dashicons dashicons-warning" style="vertical-align: middle;"></span> ' + response.data.message + '</p></div>');
                    }
                });
            });
        });
        </script>
        <?php
    }
    
    /**
     * AJAX handler for regenerating robots.txt
     */
    public function ajax_regenerate_robots() {
        check_ajax_referer('aiox_robots_nonce', 'nonce');
        
        if (!current_user_can('manage_options')) {
            wp_send_json_error(array(
                'message' => __('You do not have permission to perform this action.', 'aiox-publisher-suite-pro')
            ));
        }
        
        $result = $this->create_physical_robots_file();
        
        if (is_wp_error($result)) {
            wp_send_json_error(array(
                'message' => $result->get_error_message()
            ));
        } else {
            wp_send_json_success(array(
                'message' => __('Robots.txt file created/updated successfully!', 'aiox-publisher-suite-pro')
            ));
        }
    }
    
    /**
     * AJAX handler for removing AIOX rules
     */
    public function ajax_remove_robots() {
        check_ajax_referer('aiox_robots_nonce', 'nonce');
        
        if (!current_user_can('manage_options')) {
            wp_send_json_error(array(
                'message' => __('You do not have permission to perform this action.', 'aiox-publisher-suite-pro')
            ));
        }
        
        $result = $this->remove_physical_robots_rules();
        
        if (is_wp_error($result)) {
            wp_send_json_error(array(
                'message' => $result->get_error_message()
            ));
        } else {
            wp_send_json_success(array(
                'message' => __('AIOX rules removed from robots.txt successfully!', 'aiox-publisher-suite-pro')
            ));
        }
    }
}

// Initialize on plugins_loaded
add_action('plugins_loaded', function() {
    if (class_exists('AIOX_Logger')) {
        new AIOX_Robots_Manager();
    }
}, 20);