<?php
/**
 * AIOX Server Data Handler
 * 
 * Handles storage of server-generated ingredients, capsules, and metadata.
 * This class provides a clean interface for saving server-side processed data
 * to the client's WordPress database.
 * 
 * @package AIOX_Client
 * @since 2.0.0
 */

if (!defined('ABSPATH')) {
    exit;
}

class AIOX_Server_Data_Handler {
    
    /**
     * Store complete server response data
     * 
     * This is the main entry point for storing all server-generated data.
     * It orchestrates storage of ingredients, capsules, and metadata.
     * 
     * @param int $post_id WordPress post ID
     * @param array $server_data Complete server response with ingredients/capsules/metadata
     * @return array Result with success status and details
     */
    public static function store_server_data($post_id, $server_data) {
        if (!is_array($server_data)) {
            AIOX_Logger::error('Server data must be an array', array('post_id' => $post_id));
            return array('success' => false, 'message' => 'Invalid server data format');
        }
        
        $results = array(
            'success' => true,
            'post_id' => $post_id,
            'stored' => array(),
            'errors' => array()
        );
        
        try {
            // Store ingredients if present
            if (isset($server_data['ingredients']) && is_array($server_data['ingredients'])) {
                $ingredient_result = self::store_ingredients($post_id, $server_data['ingredients']);
                if ($ingredient_result['success']) {
                    $results['stored']['ingredients'] = $ingredient_result['count'];
                    AIOX_Logger::info("Stored {$ingredient_result['count']} ingredients", array('post_id' => $post_id));
                } else {
                    $results['errors']['ingredients'] = $ingredient_result['message'];
                    AIOX_Logger::error('Failed to store ingredients: ' . $ingredient_result['message'], array('post_id' => $post_id));
                }
            }
            
            // Store capsules if present
            if (isset($server_data['capsules']) && is_array($server_data['capsules'])) {
                $capsule_result = self::store_capsules($post_id, $server_data['capsules']);
                if ($capsule_result['success']) {
                    $results['stored']['capsules'] = $capsule_result['count'];
                    AIOX_Logger::info("Stored {$capsule_result['count']} capsules", array('post_id' => $post_id));
                } else {
                    $results['errors']['capsules'] = $capsule_result['message'];
                    AIOX_Logger::error('Failed to store capsules: ' . $capsule_result['message'], array('post_id' => $post_id));
                }
            }
            
            // Store metadata if present
            if (isset($server_data['metadata']) && is_array($server_data['metadata'])) {
                $metadata_result = self::store_metadata($post_id, $server_data['metadata']);
                if ($metadata_result['success']) {
                    $results['stored']['metadata'] = true;
                    AIOX_Logger::info("Stored metadata", array('post_id' => $post_id));
                } else {
                    $results['errors']['metadata'] = $metadata_result['message'];
                    AIOX_Logger::error('Failed to store metadata: ' . $metadata_result['message'], array('post_id' => $post_id));
                }
            }
            
            // Store processing stats if present
            if (isset($server_data['processing_stats'])) {
                update_post_meta($post_id, '_aiox_server_processing_stats', $server_data['processing_stats']);
                $results['stored']['processing_stats'] = true;
            }
            
            // Mark post as processed with server v2.0
            update_post_meta($post_id, '_aiox_processed', '1');
            update_post_meta($post_id, '_aiox_processed_at', current_time('mysql'));
            update_post_meta($post_id, '_aiox_server_version', '2.0');
            update_post_meta($post_id, '_aiox_server_processed', true);
            
            // Store processed data for badge (required for badge to show)
            $processed_data_for_badge = array(
                'processing_method' => $server_data['processing_method'] ?? 'custom',
                'server_processed' => true,
                'summary' => '',
                'qa_pairs' => array(),
                'metadata' => isset($server_data['metadata']) ? $server_data['metadata'] : array()
            );
            
            // Extract summary from capsules or metadata
            if (isset($server_data['capsules']) && is_array($server_data['capsules'])) {
                foreach ($server_data['capsules'] as $capsule) {
                    if (isset($capsule['type']) && $capsule['type'] === 'summary' && isset($capsule['content'])) {
                        $processed_data_for_badge['summary'] = $capsule['content'];
                        break;
                    }
                }
            }
            
            // Generate Q&A pairs from capsules for badge display
            if (isset($server_data['capsules']) && is_array($server_data['capsules'])) {
                $qa_count = 0;
                
                // Get configurable Q&A limit (default 5, max 10)
                $qa_limit = intval(get_option('aiox_badge_qa_count', 5));
                if ($qa_limit < 1) $qa_limit = 1;
                if ($qa_limit > 10) $qa_limit = 10;
                
                // First, check for qa_pairs capsule with real Q&A and confidence
                foreach ($server_data['capsules'] as $capsule) {
                    if (isset($capsule['type']) && $capsule['type'] === 'qa_pairs') {
                        // qa_details can be in metadata or directly on capsule
                        $qa_details = null;
                        if (isset($capsule['metadata']['qa_details'])) {
                            $qa_details = $capsule['metadata']['qa_details'];
                        } elseif (isset($capsule['qa_details'])) {
                            $qa_details = $capsule['qa_details'];
                        }
                        
                        if ($qa_details && is_array($qa_details)) {
                            // Use the real Q&A pairs with validated confidence from server
                            foreach ($qa_details as $qa_detail) {
                                if ($qa_count >= $qa_limit) break;
                                
                                $processed_data_for_badge['qa_pairs'][] = array(
                                    'id' => 'qa-' . ($qa_count + 1),
                                    'question' => $qa_detail['question'] ?? '',
                                    'answer' => $qa_detail['answer'] ?? '',
                                    'confidence' => isset($qa_detail['confidence']) ? floatval($qa_detail['confidence']) : 0.7,
                                    'source' => 'AIOX AI Optimization',
                                    'source_type' => $qa_detail['source_type'] ?? 'inferred'
                                );
                                $qa_count++;
                            }
                            break; // Found qa_pairs capsule, don't continue
                        }
                    }
                }
                
                // If no qa_details found, fall back to generating Q&A from other capsules
                if ($qa_count === 0) {
                    foreach ($server_data['capsules'] as $capsule) {
                        if ($qa_count >= $qa_limit) break;
                        
                        if (isset($capsule['type']) && isset($capsule['content'])) {
                            $question = '';
                            $answer = $capsule['content'];
                            
                            // Skip if content is empty or just whitespace
                            if (empty(trim($answer))) {
                                continue;
                            }
                            
                            // Decode HTML entities if string
                            if (is_string($answer)) {
                                $answer = html_entity_decode($answer, ENT_QUOTES | ENT_HTML5, 'UTF-8');
                            }
                            
                            // Handle array content (like keywords)
                            if (is_array($answer)) {
                                // Flatten nested arrays and filter
                                $flattened = array();
                                array_walk_recursive($answer, function($item) use (&$flattened) {
                                    if (!empty($item) && is_string($item)) {
                                        $flattened[] = trim($item);
                                    }
                                });
                                
                                if (!empty($flattened)) {
                                    $answer = implode(', ', $flattened);
                                } else {
                                    continue; // Skip if no valid content after flattening
                                }
                            }
                            
                            // Skip if answer is still "Array" or empty after processing
                            if (empty(trim($answer)) || stripos($answer, 'Array') !== false) {
                                continue;
                            }
                            
                            switch ($capsule['type']) {
                                case 'summary':
                                    $question = 'What is this content about?';
                                    break;
                                case 'keywords':
                                    $question = 'What are the main topics and keywords?';
                                    break;
                                case 'key_points':
                                    $question = 'What are the key points?';
                                    break;
                                case 'entities':
                                    $question = 'What entities are mentioned?';
                                    break;
                                case 'topics':
                                    $question = 'What topics are covered?';
                                    break;
                                case 'tldr':
                                    $question = 'Can you give me a quick summary?';
                                    break;
                                case 'qa_pairs':
                                    // Skip this one as it's meta Q&A
                                    continue 2;
                                default:
                                    continue 2; // Skip unknown types
                            }
                            
                            // Use capsule confidence if available, otherwise use type-based defaults
                            $capsule_confidence = isset($capsule['confidence']) ? floatval($capsule['confidence']) : 0.85;
                            
                            $processed_data_for_badge['qa_pairs'][] = array(
                                'id' => 'qa-' . ($qa_count + 1),
                                'question' => $question,
                                'answer' => $answer,
                                'confidence' => $capsule_confidence,
                                'source' => 'AIOX AI Optimization'
                            );
                            $qa_count++;
                        }
                    }
                }
            }
            
            // Set processing method to a friendly name
            if (isset($server_data['processing_method'])) {
                $method_map = array(
                    'custom' => 'gemini',
                    'api' => 'gemini',
                    'server' => 'gemini'
                );
                $processed_data_for_badge['processing_method'] = $method_map[$server_data['processing_method']] ?? 'gemini';
            } else {
                $processed_data_for_badge['processing_method'] = 'gemini';
            }
            
            update_post_meta($post_id, '_aiox_processed_data', $processed_data_for_badge);
            AIOX_Logger::info("Stored processed data for badge with " . count($processed_data_for_badge['qa_pairs']) . " Q&A pairs", array('post_id' => $post_id));
            
            // Generate files after storing data
            self::generate_files_for_post($post_id);
            
            // Set final success status
            $results['success'] = empty($results['errors']);
            $results['message'] = $results['success'] 
                ? 'All data stored successfully'
                : 'Some data failed to store';
            
            return $results;
            
        } catch (Exception $e) {
            AIOX_Logger::error('Exception in store_server_data: ' . $e->getMessage(), array('post_id' => $post_id));
            return array(
                'success' => false,
                'message' => 'Exception: ' . $e->getMessage(),
                'post_id' => $post_id
            );
        }
    }
    
    /**
     * Store ingredients from server
     * 
     * Uses the existing AIOX_Ingredients::save_post_ingredients() method
     * which handles JSON encoding and database storage properly.
     * 
     * @param int $post_id WordPress post ID
     * @param array $ingredients Array of ingredient structures from server
     * @return array Result with success status
     */
    private static function store_ingredients($post_id, $ingredients) {
        if (!is_array($ingredients) || empty($ingredients)) {
            return array('success' => false, 'message' => 'Ingredients array is empty or invalid');
        }
        
        // Validate ingredients structure
        foreach ($ingredients as $ingredient) {
            if (!isset($ingredient['type']) || !isset($ingredient['content'])) {
                return array('success' => false, 'message' => 'Invalid ingredient structure');
            }
        }
        
        // Use existing ingredients save method
        if (class_exists('AIOX_Ingredients')) {
            $save_result = AIOX_Ingredients::save_post_ingredients($post_id, $ingredients);
            
            if ($save_result) {
                return array(
                    'success' => true,
                    'count' => count($ingredients),
                    'message' => 'Ingredients stored successfully'
                );
            } else {
                return array('success' => false, 'message' => 'Failed to save ingredients');
            }
        } else {
            return array('success' => false, 'message' => 'AIOX_Ingredients class not found');
        }
    }
    
    /**
     * Store capsules from server
     * 
     * Creates actual capsule objects in the wp_aiox_capsules database table
     * using the AIOX_Capsule_Manager. This matches the old client behavior.
     * 
     * @param int $post_id WordPress post ID
     * @param array $capsules Array of capsule structures from server
     * @return array Result with success status
     */
    private static function store_capsules($post_id, $capsules) {
        if (!is_array($capsules) || empty($capsules)) {
            return array('success' => false, 'message' => 'Capsules array is empty or invalid');
        }
        
        // Validate capsules structure
        foreach ($capsules as $capsule) {
            if (!isset($capsule['type']) || !isset($capsule['content'])) {
                return array('success' => false, 'message' => 'Invalid capsule structure');
            }
        }
        
        try {
            // Use Capsule Manager to create actual capsule objects
            if (!class_exists('AIOX_Capsule_Manager')) {
                return array('success' => false, 'message' => 'AIOX_Capsule_Manager class not found');
            }
            
            $capsule_manager = new AIOX_Capsule_Manager();
            $created_count = 0;
            $created_capsule_id = null;
            
            // Delete existing capsule for this post first
            $existing_capsule = $capsule_manager->get_capsule_by_post_id($post_id);
            if ($existing_capsule) {
                // Directly delete from database to avoid the capsule_id vs id bug in AIOX_Capsule::delete()
                global $wpdb;
                $capsule_id = $existing_capsule->get_id();
                
                // Delete capsule files if they exist
                if (method_exists($existing_capsule, 'get_file_path')) {
                    $file_path = $existing_capsule->get_file_path();
                    if ($file_path && file_exists($file_path)) {
                        @unlink($file_path);
                    }
                }
                
                // Delete from database using correct column name 'id'
                $deleted = $wpdb->delete(
                    $wpdb->prefix . 'aiox_capsules',
                    array('id' => $capsule_id),
                    array('%s')
                );
                
                if ($deleted) {
                    AIOX_Logger::info("Deleted existing capsule from database", array(
                        'post_id' => $post_id,
                        'capsule_id' => $capsule_id
                    ));
                } else {
                    AIOX_Logger::warning("Failed to delete existing capsule", array(
                        'post_id' => $post_id,
                        'capsule_id' => $capsule_id
                    ));
                }
            }
            
            // Create a single capsule for this post with all server data
            $capsule = $capsule_manager->create_capsule('post', array(
                'post_id' => $post_id,
                'title' => get_the_title($post_id) . ' - AI Capsule',
                'description' => 'Server-generated AI capsule'
            ));
            
            if (!$capsule) {
                return array('success' => false, 'message' => 'Failed to create capsule object');
            }
            
            // Add all capsule content from server
            foreach ($capsules as $capsule_data) {
                $capsule->set_content($capsule_data['type'], $capsule_data['content']);
                
                // Add additional capsule metadata if available
                if (isset($capsule_data['word_count'])) {
                    $capsule->set_metadata($capsule_data['type'] . '_word_count', $capsule_data['word_count']);
                }
                if (isset($capsule_data['confidence'])) {
                    $capsule->set_metadata($capsule_data['type'] . '_confidence', $capsule_data['confidence']);
                }
            }
            
            // Mark as server-generated
            $capsule->set_metadata('generated_by', 'server');
            $capsule->set_metadata('server_version', '2.0');
            $capsule->set_metadata('generated_at', current_time('mysql'));
            
            // Add source post reference
            $capsule->add_source_post($post_id);
            
            // Set capsule status and save
            $capsule->set_status('ready');
            $save_result = $capsule->save();
            
            if ($save_result) {
                $created_capsule_id = $capsule->get_id();
                $created_count = count($capsules);
                
                // Store capsule reference in post meta for quick lookups
                update_post_meta($post_id, '_aiox_capsule_id', $created_capsule_id);
                update_post_meta($post_id, '_aiox_capsules_count', $created_count);
                update_post_meta($post_id, '_aiox_capsules_updated', current_time('mysql'));
                
                // Also store as JSON backup in post meta (for compatibility)
                global $wpdb;
                $json_data = json_encode($capsules, JSON_UNESCAPED_SLASHES | JSON_UNESCAPED_UNICODE);
                $wpdb->delete(
                    $wpdb->postmeta,
                    array(
                        'post_id' => $post_id,
                        'meta_key' => '_aiox_capsules'
                    ),
                    array('%d', '%s')
                );
                
                $wpdb->insert(
                    $wpdb->postmeta,
                    array(
                        'post_id' => $post_id,
                        'meta_key' => '_aiox_capsules',
                        'meta_value' => $json_data
                    ),
                    array('%d', '%s', '%s')
                );
                
                // Store capsule summary
                $types = array_column($capsules, 'type');
                $summary = array(
                    'capsule_id' => $created_capsule_id,
                    'total_count' => count($capsules),
                    'types' => array_count_values($types),
                    'last_updated' => current_time('c'),
                    'server_generated' => true
                );
                update_post_meta($post_id, '_aiox_capsules_summary', $summary);
                
                // Clear cache
                wp_cache_delete($post_id, 'post_meta');
                clean_post_cache($post_id);
                
                AIOX_Logger::info("Created capsule object in database", array(
                    'post_id' => $post_id,
                    'capsule_id' => $created_capsule_id,
                    'types_count' => $created_count
                ));
                
                return array(
                    'success' => true,
                    'count' => $created_count,
                    'capsule_id' => $created_capsule_id,
                    'message' => 'Capsules stored successfully in database'
                );
            } else {
                return array('success' => false, 'message' => 'Failed to save capsule to database');
            }
            
        } catch (Exception $e) {
            AIOX_Logger::error('Exception storing capsules: ' . $e->getMessage(), array('post_id' => $post_id));
            return array('success' => false, 'message' => 'Exception: ' . $e->getMessage());
        }
    }
    
    /**
     * Store metadata from server
     * 
     * Maps server-generated metadata to the client's expected field structure.
     * Server sends structured data (schema_org, open_graph, seo), but client
     * expects specific fields (intent, tone, orientation, etc.).
     * 
     * @param int $post_id WordPress post ID
     * @param array $metadata Metadata structure from server
     * @return array Result with success status
     */
    private static function store_metadata($post_id, $metadata) {
        if (!is_array($metadata)) {
            return array('success' => false, 'message' => 'Metadata must be an array');
        }
        
        try {
            $stored_sections = array();
            
            // FIRST: Store the structured data as-is for compatibility
            // Store Schema.org data
            if (isset($metadata['schema_org']) && is_array($metadata['schema_org'])) {
                update_post_meta($post_id, '_aiox_schema_org', $metadata['schema_org']);
                $stored_sections[] = 'schema_org';
            }
            
            // Store Open Graph data
            if (isset($metadata['open_graph']) && is_array($metadata['open_graph'])) {
                update_post_meta($post_id, '_aiox_open_graph', $metadata['open_graph']);
                $stored_sections[] = 'open_graph';
            }
            
            // Store Twitter Card data
            if (isset($metadata['twitter_card']) && is_array($metadata['twitter_card'])) {
                update_post_meta($post_id, '_aiox_twitter_card', $metadata['twitter_card']);
                $stored_sections[] = 'twitter_card';
            }
            
            // Store SEO metadata
            if (isset($metadata['seo']) && is_array($metadata['seo'])) {
                update_post_meta($post_id, '_aiox_seo_metadata', $metadata['seo']);
                $stored_sections[] = 'seo';
            }
            
            // Store AI metadata
            if (isset($metadata['ai_metadata']) && is_array($metadata['ai_metadata'])) {
                update_post_meta($post_id, '_aiox_ai_metadata', $metadata['ai_metadata']);
                $stored_sections[] = 'ai_metadata';
            }
            
            // SECOND: Map to AIOX_Metadata expected field structure
            // The old client expects fields like: intent, tone, orientation, domain, etc.
            // We need to extract these from the server's structured data
            
            $aiox_metadata = array();
            
            // Map from server metadata to AIOX fields
            if (isset($metadata['content_classification'])) {
                // Server sent content classification data
                $classification = $metadata['content_classification'];
                
                if (isset($classification['intent'])) {
                    $aiox_metadata['intent'] = $classification['intent'];
                }
                if (isset($classification['tone'])) {
                    $aiox_metadata['tone'] = $classification['tone'];
                }
                if (isset($classification['orientation'])) {
                    $aiox_metadata['orientation'] = $classification['orientation'];
                }
                if (isset($classification['domain'])) {
                    $aiox_metadata['domain'] = $classification['domain'];
                }
                if (isset($classification['utility'])) {
                    $aiox_metadata['utility'] = $classification['utility'];
                }
                if (isset($classification['target_audience'])) {
                    $aiox_metadata['target_audience'] = $classification['target_audience'];
                }
                if (isset($classification['narrative_type'])) {
                    $aiox_metadata['narrative_type'] = $classification['narrative_type'];
                }
                if (isset($classification['media_type'])) {
                    $aiox_metadata['media_type'] = $classification['media_type'];
                }
                if (isset($classification['geolocation_sensitivity'])) {
                    $aiox_metadata['geolocation_sensitivity'] = $classification['geolocation_sensitivity'];
                }
                if (isset($classification['market_segments'])) {
                    $aiox_metadata['market_segments'] = $classification['market_segments'];
                }
            }
            
            // If server didn't send content_classification, try to infer from other data
            if (empty($aiox_metadata)) {
                // Set ALL 10 defaults to match old client behavior
                $aiox_metadata = array(
                    'intent' => 'informational',
                    'tone' => 'professional',
                    'orientation' => 'general',
                    'domain' => 'general',
                    'utility' => 'reference',
                    'target_audience' => 'general',
                    'narrative_type' => 'expository',
                    'media_type' => 'text',
                    'geolocation_sensitivity' => 'none',
                    'market_segments' => array('general')
                );
                
                // Try to infer from Schema.org type
                if (isset($metadata['schema_org']['@type'])) {
                    $schema_type = $metadata['schema_org']['@type'];
                    
                    // Map Schema.org types to intent
                    $type_to_intent = array(
                        'Article' => 'informational',
                        'NewsArticle' => 'informational',
                        'BlogPosting' => 'editorial',
                        'HowTo' => 'instructional',
                        'Recipe' => 'instructional',
                        'Review' => 'editorial',
                        'Product' => 'promotional'
                    );
                    
                    if (isset($type_to_intent[$schema_type])) {
                        $aiox_metadata['intent'] = $type_to_intent[$schema_type];
                    }
                }
            }
            
            // Ensure we always have all 10 fields even if server sent partial data
            $required_fields = array(
                'intent' => 'informational',
                'tone' => 'professional',
                'orientation' => 'general',
                'domain' => 'general',
                'utility' => 'reference',
                'target_audience' => 'general',
                'narrative_type' => 'expository',
                'media_type' => 'text',
                'geolocation_sensitivity' => 'none',
                'market_segments' => array('general')
            );
            
            // Fill in any missing fields with defaults
            foreach ($required_fields as $field => $default) {
                if (!isset($aiox_metadata[$field])) {
                    $aiox_metadata[$field] = $default;
                }
            }
            
            // Use AIOX_Metadata::save_post_metadata() to store in expected format
            if (class_exists('AIOX_Metadata') && !empty($aiox_metadata)) {
                AIOX_Metadata::save_post_metadata($post_id, $aiox_metadata);
                $stored_sections[] = 'aiox_metadata_fields';
                
                AIOX_Logger::info("Mapped and stored AIOX metadata fields", array(
                    'post_id' => $post_id,
                    'fields' => array_keys($aiox_metadata)
                ));
            }
            
            // Mark metadata as generated
            update_post_meta($post_id, '_aiox_metadata_generated', time());
            update_post_meta($post_id, '_aiox_metadata_version', '2.0');
            update_post_meta($post_id, '_aiox_server_metadata', true);
            
            return array(
                'success' => true,
                'sections' => $stored_sections,
                'message' => 'Metadata stored and mapped successfully'
            );
            
        } catch (Exception $e) {
            AIOX_Logger::error('Exception storing metadata: ' . $e->getMessage(), array('post_id' => $post_id));
            return array('success' => false, 'message' => 'Exception: ' . $e->getMessage());
        }
    }
    
    /**
     * Sanitize data for JSON encoding
     * 
     * Recursively sanitizes data to prevent JSON encoding issues.
     * Handles WordPress-specific issues like slashes and encoding.
     * 
     * @param mixed $data Data to sanitize
     * @return mixed Sanitized data
     */
    private static function sanitize_for_json($data) {
        if (is_array($data)) {
            $sanitized = array();
            foreach ($data as $key => $value) {
                $clean_key = wp_strip_all_tags($key);
                $sanitized[$clean_key] = self::sanitize_for_json($value);
            }
            return $sanitized;
        } elseif (is_string($data)) {
            // Remove slashes that WordPress might add
            $data = wp_unslash($data);
            // Ensure UTF-8 encoding
            $data = mb_convert_encoding($data, 'UTF-8', 'UTF-8');
            // Remove any invalid UTF-8 characters
            $data = mb_check_encoding($data, 'UTF-8') ? $data : '';
            return $data;
        } elseif (is_object($data)) {
            return self::sanitize_for_json((array) $data);
        } else {
            return $data;
        }
    }
    
    /**
     * Get stored server data for a post
     * 
     * Retrieves all server-generated data for a post.
     * 
     * @param int $post_id WordPress post ID
     * @return array|false Stored data or false if not found
     */
    public static function get_stored_data($post_id) {
        $data = array(
            'post_id' => $post_id,
            'server_processed' => get_post_meta($post_id, '_aiox_server_processed', true),
            'server_version' => get_post_meta($post_id, '_aiox_server_version', true),
            'processed_at' => get_post_meta($post_id, '_aiox_processed_at', true)
        );
        
        // Get ingredients if available
        if (class_exists('AIOX_Ingredients')) {
            $data['ingredients'] = AIOX_Ingredients::get_post_ingredients($post_id);
            $data['ingredients_summary'] = get_post_meta($post_id, '_aiox_ingredients_summary', true);
        }
        
        // Get capsules
        $capsules_json = get_post_meta($post_id, '_aiox_capsules', true);
        if (!empty($capsules_json)) {
            $data['capsules'] = json_decode($capsules_json, true);
            $data['capsules_summary'] = get_post_meta($post_id, '_aiox_capsules_summary', true);
        }
        
        // Get metadata
        $data['metadata'] = array(
            'schema_org' => get_post_meta($post_id, '_aiox_schema_org', true),
            'open_graph' => get_post_meta($post_id, '_aiox_open_graph', true),
            'twitter_card' => get_post_meta($post_id, '_aiox_twitter_card', true),
            'seo' => get_post_meta($post_id, '_aiox_seo_metadata', true),
            'ai_metadata' => get_post_meta($post_id, '_aiox_ai_metadata', true)
        );
        
        // Get processing stats
        $data['processing_stats'] = get_post_meta($post_id, '_aiox_server_processing_stats', true);
        
        return $data;
    }
    
    /**
     * Check if post has server-generated data
     * 
     * @param int $post_id WordPress post ID
     * @return bool True if post has been processed by server v2.0
     */
    public static function has_server_data($post_id) {
        return (bool) get_post_meta($post_id, '_aiox_server_processed', true);
    }
    
    /**
     * Generate all required files for a post
     * Replicates the old client workflow exactly:
     * 1. Individual post file with COMPLETE data structure
     * 2. Sets proper meta flags BEFORE updating main manifest
     * 3. Updates main manifesto (aio.json)
     * 4. Updates delta feed
     * 5. Updates pack file
     * 6. Updates ingredients index
     * 7. Updates TDM-REP
     */
    private static function generate_files_for_post($post_id) {
        try {
            AIOX_Logger::info("Starting file generation for post {$post_id}");
            
            // Create .well-known directory structure
            $well_known_dir = ABSPATH . '.well-known';
            $posts_dir = $well_known_dir . '/posts';
            
            if (!file_exists($well_known_dir)) {
                if (!wp_mkdir_p($well_known_dir)) {
                    AIOX_Logger::error("Failed to create .well-known directory");
                    return false;
                }
            }
            
            if (!file_exists($posts_dir)) {
                if (!wp_mkdir_p($posts_dir)) {
                    AIOX_Logger::error("Failed to create posts directory");
                    return false;
                }
            }
            
            // Get post data
            $post = get_post($post_id);
            if (!$post) {
                AIOX_Logger::error("Post not found", array('post_id' => $post_id));
                return false;
            }
            
            // Get processed data for content section
            $processed_data = get_post_meta($post_id, '_aiox_processed_data', true);
            if (!$processed_data) {
                $processed_data = array();
            }
            
            // Get ingredients
            $ingredients = AIOX_Ingredients::get_post_ingredients($post_id);
            
            // Count ingredient types
            $ingredient_types = array();
            foreach ($ingredients as $ingredient) {
                if (isset($ingredient['type'])) {
                    $type = strtolower($ingredient['type']);
                    $ingredient_types[$type] = isset($ingredient_types[$type]) ? $ingredient_types[$type] + 1 : 1;
                }
            }
            
            // Get capsule data
            $capsule_id = '';
            $capsule_data = array();
            if (class_exists('AIOX_Capsule_Manager')) {
                $capsule_manager = new AIOX_Capsule_Manager();
                $capsule = $capsule_manager->get_capsule_by_post_id($post_id);
                
                if ($capsule) {
                    $capsule_id = $capsule->get_id();
                    $capsule_data = array(
                        'id' => $capsule_id,
                        'status' => $capsule->get_status(),
                        'type' => $capsule->get_type()
                    );
                }
            }
            
            // Get metadata
            $metadata_raw = get_post_meta($post_id, '_aiox_metadata', true);
            $metadata = array();
            if (!empty($metadata_raw)) {
                if (is_string($metadata_raw)) {
                    $decoded = json_decode($metadata_raw, true);
                    if (is_array($decoded)) {
                        $metadata = $decoded;
                    }
                } else if (is_array($metadata_raw)) {
                    $metadata = $metadata_raw;
                }
            }
            
            // Add categories, tags, language to metadata
            $metadata = array_merge($metadata, array(
                'categories' => wp_get_post_categories($post_id, array('fields' => 'names')),
                'tags' => wp_get_post_tags($post_id, array('fields' => 'names')),
                'language' => get_locale()
            ));
            
            // Build COMPLETE post data structure (matching old client exactly)
            $word_count = str_word_count(strip_tags($post->post_content));
            $post_data = array(
                'post_id' => $post_id,
                'title' => $post->post_title,
                'url' => get_permalink($post_id),
                'type' => $post->post_type,
                'status' => $post->post_status,
                'published_at' => get_the_date('c', $post_id),
                'modified_at' => get_the_modified_date('c', $post_id),
                'author' => array(
                    'login' => get_the_author_meta('login', $post->post_author),
                    'display_name' => get_the_author_meta('display_name', $post->post_author),
                    'email' => get_the_author_meta('email', $post->post_author)
                ),
                'excerpt' => get_the_excerpt($post_id),
                'word_count' => $word_count,
                'reading_time' => ceil($word_count / 200),
                'content' => array(
                    'summary' => isset($processed_data['summary']) ? $processed_data['summary'] : '',
                    'key_points' => isset($processed_data['key_points']) ? $processed_data['key_points'] : array(),
                    'topics' => isset($processed_data['topics']) ? $processed_data['topics'] : array(),
                    'qa_pairs' => isset($processed_data['qa_pairs']) ? $processed_data['qa_pairs'] : array()
                ),
                'ingredients' => array(
                    'total_count' => count($ingredients),
                    'types' => $ingredient_types,
                    'items' => $ingredients
                ),
                'metadata' => $metadata,
                'licensing' => array(
                    'training_allowed' => get_post_meta($post_id, '_aiox_training_allowed', true) ?: '0',
                    'citation_required' => get_post_meta($post_id, '_aiox_citation_required', true) ?: '1',
                    'attribution_string' => get_post_meta($post_id, '_aiox_attribution_string', true) ?: get_bloginfo('name'),
                    'license_url' => get_post_meta($post_id, '_aiox_license_url', true) ?: '',
                    'expiration_date' => get_post_meta($post_id, '_aiox_expiration_date', true) ?: ''
                ),
                'verification' => array(
                    'processed_at' => current_time('c'),
                    'processing_method' => 'server_ai_enhanced',
                    'version' => '2.0',
                    'capsule_id' => $capsule_id,
                    'content_hash' => self::generate_content_hash($post_id, $ingredients, $metadata)
                )
            );
            
            // Write individual post file
            $post_file = $posts_dir . '/' . $post_id . '.json';
            $json_data = wp_json_encode($post_data, JSON_PRETTY_PRINT | JSON_UNESCAPED_SLASHES | JSON_UNESCAPED_UNICODE);
            $success = file_put_contents($post_file, $json_data);
            
            if ($success) {
                AIOX_Logger::info("Generated complete post file", array(
                    'post_id' => $post_id,
                    'file' => $post_file,
                    'size' => filesize($post_file)
                ));
            } else {
                AIOX_Logger::error("Failed to write post file", array('post_id' => $post_id, 'file' => $post_file));
            }
            
            // CRITICAL: Set processing flags BEFORE updating main manifest (so it counts the post)
            update_post_meta($post_id, '_aiox_processed', '1');
            update_post_meta($post_id, '_aiox_processed_at', current_time('c'));
            
            // Update main manifest using AIOX_Hooks (matching old client workflow)
            if (class_exists('AIOX_Hooks')) {
                $hooks = new AIOX_Hooks();
                
                // Update main aio.json manifest
                $hooks->generate_manifest();
                AIOX_Logger::info("Updated main AIOX manifest");
                
                // Update delta feed
                $hooks->generate_delta_feed();
                AIOX_Logger::info("Updated delta feed");
                
                // Update pack file
                $hooks->generate_pack_file();
                AIOX_Logger::info("Updated AIOX pack");
            }
            
            // Export/update main ingredients index file
            if (class_exists('AIOX_Ingredients')) {
                AIOX_Ingredients::export_ingredients_file();
                AIOX_Logger::info("Updated main ingredients index");
            }
            
            // Generate/update TDM-REP file
            self::generate_tdm_rep_file();
            
            return true;
            
        } catch (Exception $e) {
            AIOX_Logger::error("File generation failed: " . $e->getMessage(), array('post_id' => $post_id));
            return false;
        }
    }
    
    /**
     * Generate content hash for verification
     */
    private static function generate_content_hash($post_id, $ingredients, $metadata) {
        $post = get_post($post_id);
        
        $content_to_hash = wp_json_encode(array(
            'post_id' => $post_id,
            'title' => $post->post_title,
            'content' => $post->post_content,
            'modified' => $post->post_modified,
            'ingredients' => $ingredients,
            'metadata' => $metadata
        ), JSON_UNESCAPED_SLASHES | JSON_UNESCAPED_UNICODE);
        
        return hash('sha256', $content_to_hash);
    }
    
    /**
     * Generate TDM-REP file
     */
    private static function generate_tdm_rep_file() {
        try {
            $well_known_dir = ABSPATH . '.well-known';
            $tdm_file = $well_known_dir . '/tdmrep.json';
            $home_url = home_url();
            
            // Get settings from correct option names
            $allow_training = get_option('aiox_tdm_training_allowed', false);
            $require_citation = get_option('aiox_tdm_citation_required', true);
            $attribution_name = get_option('aiox_required_attribution', get_bloginfo('name'));
            
            $tdm_data = array(
                'version' => '1.0',
                'policy' => array(
                    'default' => $allow_training ? 'allow' : 'deny',
                    'training' => (bool) $allow_training,
                    'citation' => (bool) $require_citation
                ),
                'rules' => array(
                    array(
                        'path' => '/',
                        'permission' => $allow_training ? 'allow' : 'deny',
                        'purpose' => array('training', 'analysis')
                    )
                ),
                'contact' => array(
                    'url' => $home_url . '/contact',
                    'email' => get_option('admin_email')
                ),
                'attribution' => array(
                    'required' => (bool) $require_citation,
                    'name' => $attribution_name,
                    'url' => $home_url
                ),
                'generated_at' => current_time('c')
            );
            
            $tdm_json = wp_json_encode($tdm_data, JSON_PRETTY_PRINT | JSON_UNESCAPED_SLASHES);
            $bytes_written = file_put_contents($tdm_file, $tdm_json);
            
            if ($bytes_written) {
                AIOX_Logger::info("Generated TDM-REP file", array('file' => $tdm_file, 'size' => $bytes_written));
            }
            
        } catch (Exception $e) {
            AIOX_Logger::error("Failed to generate TDM-REP file: " . $e->getMessage());
        }
    }
    
    /**
     * Clear all server data for a post
     * 
     * @param int $post_id WordPress post ID
     * @return bool Success status
     */
    public static function clear_server_data($post_id) {
        try {
            AIOX_Logger::info("Starting clear_server_data for post {$post_id}");
            
            // Delete individual post file from posts subdirectory (matching old client structure)
            $well_known_dir = ABSPATH . '.well-known';
            $posts_dir = $well_known_dir . '/posts';
            $post_file = $posts_dir . '/' . $post_id . '.json';
            
            if (file_exists($post_file)) {
                unlink($post_file);
                AIOX_Logger::info("Deleted post file", array('post_id' => $post_id, 'file' => $post_file));
            }
            
            // Also check old location for backward compatibility
            $old_post_file = $well_known_dir . '/aiox-post-' . $post_id . '.json';
            if (file_exists($old_post_file)) {
                unlink($old_post_file);
                AIOX_Logger::info("Deleted old format post file", array('post_id' => $post_id));
            }
            
            // Clear ingredients
            delete_post_meta($post_id, '_aiox_ingredients');
            delete_post_meta($post_id, '_aiox_ingredients_count');
            delete_post_meta($post_id, '_aiox_ingredients_summary');
            
            // Delete capsule files and database records
            if (class_exists('AIOX_Capsule_Manager')) {
                $capsule_manager = new AIOX_Capsule_Manager();
                $existing_capsule = $capsule_manager->get_capsule_by_post_id($post_id);
                
                if ($existing_capsule) {
                    global $wpdb;
                    $capsule_id = $existing_capsule->get_id();
                    
                    // Delete capsule file from root .well-known: .well-known/capsule_{id}.json
                    $capsule_file = $well_known_dir . '/capsule_' . $capsule_id . '.json';
                    if (file_exists($capsule_file)) {
                        unlink($capsule_file);
                        AIOX_Logger::info("Deleted capsule file", array('post_id' => $post_id, 'capsule_id' => $capsule_id, 'file' => $capsule_file));
                    } else {
                        AIOX_Logger::warning("Capsule file not found", array('post_id' => $post_id, 'capsule_id' => $capsule_id, 'expected_file' => $capsule_file));
                    }
                    
                    // Also delete from capsules subdirectory: .well-known/capsules/{id}.json
                    $capsules_dir = $well_known_dir . '/capsules';
                    $public_capsule_file = $capsules_dir . '/' . $capsule_id . '.json';
                    if (file_exists($public_capsule_file)) {
                        unlink($public_capsule_file);
                        AIOX_Logger::info("Deleted public capsule file", array('post_id' => $post_id, 'capsule_id' => $capsule_id, 'file' => $public_capsule_file));
                    }
                    
                    // Delete from database using 'id' column (PRIMARY KEY), not post_id
                    $deleted = $wpdb->delete(
                        $wpdb->prefix . 'aiox_capsules',
                        array('id' => $capsule_id),
                        array('%s')
                    );
                    
                    if ($deleted) {
                        AIOX_Logger::info("Deleted capsule from database", array('post_id' => $post_id, 'capsule_id' => $capsule_id));
                    } else {
                        AIOX_Logger::warning("Failed to delete capsule from database", array('post_id' => $post_id));
                    }
                } else {
                    AIOX_Logger::info("No capsule found for post", array('post_id' => $post_id));
                }
            }
            
            // Clear capsule meta
            delete_post_meta($post_id, '_aiox_capsules');
            delete_post_meta($post_id, '_aiox_capsules_count');
            delete_post_meta($post_id, '_aiox_capsules_summary');
            
            // Clear metadata
            delete_post_meta($post_id, '_aiox_schema_org');
            delete_post_meta($post_id, '_aiox_open_graph');
            delete_post_meta($post_id, '_aiox_twitter_card');
            delete_post_meta($post_id, '_aiox_seo_metadata');
            delete_post_meta($post_id, '_aiox_ai_metadata');
            delete_post_meta($post_id, '_aiox_metadata');
            delete_post_meta($post_id, '_aiox_metadata_generated');
            
            // Clear ALL processing markers
            delete_post_meta($post_id, '_aiox_server_processed');
            delete_post_meta($post_id, '_aiox_server_version');
            delete_post_meta($post_id, '_aiox_server_processing_stats');
            delete_post_meta($post_id, '_aiox_processed');
            delete_post_meta($post_id, '_aiox_processed_at');
            delete_post_meta($post_id, '_aiox_processed_data');
            delete_post_meta($post_id, '_aiox_processing_complete');
            delete_post_meta($post_id, '_aiox_last_processed');
            delete_post_meta($post_id, '_aiox_comprehensive_processed');
            
            // Clear cache
            wp_cache_delete($post_id, 'post_meta');
            clean_post_cache($post_id);
            
            AIOX_Logger::info("Cleared all post meta for post {$post_id}");
            
            // CRITICAL: Regenerate main files after deletion (matching old client workflow)
            AIOX_Logger::info("Starting main files regeneration after deletion");
            
            if (class_exists('AIOX_Hooks')) {
                AIOX_Logger::info("AIOX_Hooks class found, calling generate_manifest()");
                $hooks = new AIOX_Hooks();
                $hooks->generate_manifest();
                AIOX_Logger::info("Main manifesto regenerated via AIOX_Hooks");
            } else if (class_exists('AIOX_Generator')) {
                AIOX_Logger::info("AIOX_Generator class found, calling generate_manifesto_file()");
                $generator = new AIOX_Generator();
                $result = $generator->generate_manifesto_file();
                AIOX_Logger::info("Main manifesto regeneration result: " . ($result ? 'SUCCESS' : 'FAILED'));
                
                if (!$result) {
                    AIOX_Logger::error("Failed to regenerate main manifesto after deletion");
                }
            } else {
                AIOX_Logger::error("No manifest generator class found - cannot regenerate main manifesto");
            }
            
            if (class_exists('AIOX_Ingredients')) {
                AIOX_Logger::info("AIOX_Ingredients class found, calling export_ingredients_file()");
                AIOX_Ingredients::export_ingredients_file();
                AIOX_Logger::info("Ingredients index regenerated after deletion");
            } else {
                AIOX_Logger::error("AIOX_Ingredients class not found - cannot regenerate ingredients index");
            }
            
            AIOX_Logger::info("Completed clear_server_data for post {$post_id}");
            
            return true;
            
        } catch (Exception $e) {
            AIOX_Logger::error('Exception in clear_server_data: ' . $e->getMessage(), array('post_id' => $post_id, 'trace' => $e->getTraceAsString()));
            return false;
        }
    }
}