<?php
/**
 * AIOX Utils Class
 * 
 * Utility functions and configuration management
 */

if (!defined('ABSPATH')) {
    exit;
}

class AIOX_Utils {
    
    /**
     * Decode HTML entities properly to avoid double-encoding issues
     * Handles cases where content was stored with already-encoded entities
     * 
     * @param string $text Text to decode
     * @param bool $escape Whether to escape for HTML output (default: true)
     * @return string Properly decoded text
     */
    public static function decode_text($text, $escape = true) {
        if (empty($text) || !is_string($text)) {
            return '';
        }
        
        // Decode HTML entities multiple times to handle double/triple encoding
        $decoded = $text;
        $previous = '';
        $max_iterations = 5; // Prevent infinite loop
        $i = 0;
        
        while ($decoded !== $previous && $i < $max_iterations) {
            $previous = $decoded;
            $decoded = html_entity_decode($decoded, ENT_QUOTES | ENT_HTML5, 'UTF-8');
            $i++;
        }
        
        // Optionally escape for HTML output
        return $escape ? esc_html($decoded) : $decoded;
    }
    
    /**
     * Get plugin option with default value
     */
    public static function get_option($option_name, $default = '') {
        return get_option('aiox_' . $option_name, $default);
    }
    
    /**
     * Update plugin option
     */
    public static function update_option($option_name, $value) {
        return update_option('aiox_' . $option_name, $value);
    }
    
    /**
     * Delete plugin option
     */
    public static function delete_option($option_name) {
        return delete_option('aiox_' . $option_name);
    }
    
    /**
     * Check if pro mode is enabled
     */
    public static function is_pro_mode() {
        $license_key = self::get_option('license_key');
        $mode = self::get_option('mode', 'free');
        return !empty($license_key) && $mode === 'pro';
    }
    
    /**
     * Get AI endpoint URL
     */
    public static function get_ai_endpoint() {
        return self::get_option('ai_endpoint', AIOX_DEFAULT_AI_ENDPOINT);
    }
    
    /**
     * Get all plugin options with defaults
     */
    public static function get_all_options() {
        $defaults = array(
            'license_key' => '',
            'ai_endpoint_url' => 'https://api.openai.com/v1/chat/completions',
            'enable_pro_features' => false,
            'enable_floating_badge' => true,
            'enable_auto_processing' => true,
            'enable_analytics' => true,
            'enable_ai_info_post' => true,
            'log_level' => 'info',
            'max_log_entries' => 1000,
            // TDM Settings - map to actual WordPress option names
            'allow_training' => get_option('aiox_tdm_training_allowed', false),
            'allow_citation' => get_option('aiox_tdm_citation_required', true),
            'required_attribution' => get_option('aiox_required_attribution', get_bloginfo('name')),
            'tdm_default' => get_option('aiox_tdm_default', 'deny'),
            // Processing settings
            'auto_classify_metadata' => get_option('aiox_auto_classify_metadata', true),
            'pack_enabled' => get_option('aiox_pack_enabled', true)
        );
        
        $options = array();
        foreach ($defaults as $key => $default) {
            // For TDM settings, use the already-fetched values
            if (in_array($key, array('allow_training', 'allow_citation', 'required_attribution', 'tdm_default', 'auto_classify_metadata', 'pack_enabled'))) {
                $options[$key] = $default;
            } else {
                $options[$key] = self::get_option($key, $default);
            }
        }
        
        return $options;
    }
    
    /**
     * Get plugin features configuration
     */
    public static function get_features() {
        return self::get_option('features', array(
            'auto_process' => true,
            'ai_badge' => true,
            'schema_markup' => true,
            'inline_json' => true,
            'analytics' => true
        ));
    }
    
    /**
     * Check if a feature is enabled
     */
    public static function is_feature_enabled($feature_name) {
        // Map feature names to their actual option names
        $feature_map = array(
            'ai_badge' => 'aiox_enable_ai_badge',
            'schema_markup' => 'aiox_enable_faqpage_schema',
            'inline_json' => 'aiox_enable_inline_json',
            'auto_process' => 'aiox_enable_auto_processing',
            'analytics' => 'aiox_enable_analytics'
        );
        
        // If we have a direct mapping, use the WordPress option
        if (isset($feature_map[$feature_name])) {
            return get_option($feature_map[$feature_name], true);
        }
        
        // Fallback to legacy features array (for backward compatibility)
        $features = self::get_features();
        return isset($features[$feature_name]) && $features[$feature_name];
    }
    
    /**
     * Get excluded post types
     */
    public static function get_excluded_post_types() {
        $excluded = self::get_option('excluded_post_types', array());
        
        // Always exclude certain post types
        $always_excluded = array(
            'revision',
            'nav_menu_item',
            'custom_css',
            'customize_changeset',
            'oembed_cache',
            'user_request',
            'wp_block'
        );
        
        return array_merge($excluded, $always_excluded);
    }
    
    /**
     * Check if post type should be processed
     */
    public static function should_process_post_type($post_type) {
        $excluded = self::get_excluded_post_types();
        return !in_array($post_type, $excluded);
    }
    
    /**
     * Get system information
     */
    public static function get_system_info() {
        global $wpdb;
        
        return array(
            'wordpress_version' => get_bloginfo('version'),
            'php_version' => PHP_VERSION,
            'mysql_version' => $wpdb->db_version(),
            'plugin_version' => AIOX_VERSION,
            'site_url' => home_url(),
            'admin_email' => get_option('admin_email'),
            'timezone' => get_option('timezone_string') ?: 'UTC',
            'language' => get_locale(),
            'multisite' => is_multisite(),
            'debug_mode' => defined('WP_DEBUG') && WP_DEBUG,
            'memory_limit' => ini_get('memory_limit'),
            'max_execution_time' => ini_get('max_execution_time'),
            'upload_max_filesize' => ini_get('upload_max_filesize'),
            'post_max_size' => ini_get('post_max_size')
        );
    }
    
    /**
     * Check system requirements
     */
    public static function check_system_requirements() {
        $requirements = array();
        
        // WordPress version
        $requirements['wordpress'] = array(
            'name' => 'WordPress Version',
            'required' => '5.0',
            'current' => get_bloginfo('version'),
            'status' => version_compare(get_bloginfo('version'), '5.0', '>=')
        );
        
        // PHP version
        $requirements['php'] = array(
            'name' => 'PHP Version',
            'required' => '7.4',
            'current' => PHP_VERSION,
            'status' => version_compare(PHP_VERSION, '7.4', '>=')
        );
        
        // OpenSSL extension
        $requirements['openssl'] = array(
            'name' => 'OpenSSL Extension',
            'required' => 'Required',
            'current' => extension_loaded('openssl') ? 'Available' : 'Missing',
            'status' => extension_loaded('openssl')
        );
        
        // cURL extension
        $requirements['curl'] = array(
            'name' => 'cURL Extension',
            'required' => 'Required',
            'current' => extension_loaded('curl') ? 'Available' : 'Missing',
            'status' => extension_loaded('curl')
        );
        
        // JSON extension
        $requirements['json'] = array(
            'name' => 'JSON Extension',
            'required' => 'Required',
            'current' => extension_loaded('json') ? 'Available' : 'Missing',
            'status' => extension_loaded('json')
        );
        
        // Well-known directory
        $requirements['wellknown'] = array(
            'name' => 'Well-Known Directory',
            'required' => 'Writable',
            'current' => is_dir(AIOX_WELL_KNOWN_PATH) && is_writable(AIOX_WELL_KNOWN_PATH) ? 'Ready' : 'Not Ready',
            'status' => is_dir(AIOX_WELL_KNOWN_PATH) && is_writable(AIOX_WELL_KNOWN_PATH)
        );
        
        return $requirements;
    }
    
    /**
     * Format file size
     */
    public static function format_file_size($bytes) {
        if ($bytes >= 1073741824) {
            return number_format($bytes / 1073741824, 2) . ' GB';
        } elseif ($bytes >= 1048576) {
            return number_format($bytes / 1048576, 2) . ' MB';
        } elseif ($bytes >= 1024) {
            return number_format($bytes / 1024, 2) . ' KB';
        } else {
            return $bytes . ' B';
        }
    }
    
    /**
     * Generate unique ID
     */
    public static function generate_id($prefix = '') {
        return $prefix . uniqid() . '_' . wp_rand(1000, 9999);
    }
    
    /**
     * Sanitize filename
     */
    public static function sanitize_filename($filename) {
        return sanitize_file_name($filename);
    }
    
    /**
     * Get current user info for logging
     */
    public static function get_current_user_info() {
        $user = wp_get_current_user();
        
        if ($user->ID === 0) {
            return array(
                'user_id' => 0,
                'user_login' => 'guest',
                'user_email' => '',
                'display_name' => 'Guest'
            );
        }
        
        return array(
            'user_id' => $user->ID,
            'user_login' => $user->user_login,
            'user_email' => $user->user_email,
            'display_name' => $user->display_name
        );
    }
    
    /**
     * Validate URL
     */
    public static function is_valid_url($url) {
        return filter_var($url, FILTER_VALIDATE_URL) !== false;
    }
    
    /**
     * Validate email
     */
    public static function is_valid_email($email) {
        return is_email($email) !== false;
    }
    
    /**
     * Get plugin status
     */
    public static function get_plugin_status() {
        $status = array(
            'version' => AIOX_VERSION,
            'mode' => self::is_pro_mode() ? 'pro' : 'free',
            'license_key' => !empty(self::get_option('license_key')),
            'well_known_ready' => is_dir(AIOX_WELL_KNOWN_PATH) && is_writable(AIOX_WELL_KNOWN_PATH),
            'database_ready' => self::check_database_tables(),
            'features_enabled' => count(array_filter(self::get_features())),
            'last_activity' => self::get_option('last_activity', 'Never')
        );
        
        return $status;
    }
    
    /**
     * Check if database tables exist
     */
    private static function check_database_tables() {
        global $wpdb;
        
        $required_tables = array(
            $wpdb->prefix . 'aiox_queue',
            $wpdb->prefix . 'aiox_logs'
        );
        
        foreach ($required_tables as $table) {
            if ($wpdb->get_var("SHOW TABLES LIKE '$table'") !== $table) {
                return false;
            }
        }
        
        return true;
    }
}
