<?php
/**
 * AIOX Auto Processor
 * Handles automatic processing on post save
 */

if (!defined('ABSPATH')) {
    exit;
}

class AIOX_Auto_Processor {
    
    public function __construct() {
        add_action('save_post', array($this, 'maybe_auto_process'), 20, 2);
    }
    
    /**
     * Maybe auto-process post on save
     */
    public function maybe_auto_process($post_id, $post) {
        // Check if auto-processing is enabled
        if (!get_option('aiox_auto_process_on_save', false)) {
            return;
        }
        
        // Skip if this is an autosave
        if (defined('DOING_AUTOSAVE') && DOING_AUTOSAVE) {
            return;
        }
        
        // Skip if this is a revision
        if (wp_is_post_revision($post_id)) {
            return;
        }
        
        // Only process published posts and pages
        if (!in_array($post->post_type, array('post', 'page'))) {
            return;
        }
        
        if ($post->post_status !== 'publish') {
            return;
        }
        
        // Skip if already processed
        if (get_post_meta($post_id, '_aiox_processed_data', true)) {
            return;
        }
        
        // Skip if user doesn't have permission
        if (!current_user_can('edit_post', $post_id)) {
            return;
        }
        
        // Process the post in the background
        $this->process_post_async($post_id);
    }
    
    /**
     * Process post asynchronously
     */
    private function process_post_async($post_id) {
        // Schedule processing to avoid blocking the save operation
        wp_schedule_single_event(time() + 5, 'aiox_auto_process_post', array($post_id));
    }
    
    /**
     * Actually process the post
     */
    public static function process_post_scheduled($post_id) {
        try {
            $comprehensive_processor = new AIOX_Comprehensive_Processor();
            $result = $comprehensive_processor->process_post_completely($post_id);
            
            if ($result && is_array($result) && isset($result['success']) && $result['success']) {
                AIOX_Logger::log('Auto-processed post ' . $post_id . ' completely with all components', AIOX_Logger::INFO);
            } else {
                AIOX_Logger::log('Auto-processing failed for post ' . $post_id . ': ' . (isset($result['message']) ? $result['message'] : 'Unknown error'), AIOX_Logger::WARNING);
            }
            
        } catch (Exception $e) {
            AIOX_Logger::log('Auto-processing error for post ' . $post_id . ': ' . $e->getMessage(), AIOX_Logger::ERROR);
        }
    }
}

// Hook for scheduled processing
add_action('aiox_auto_process_post', array('AIOX_Auto_Processor', 'process_post_scheduled'));
